#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: ex2.c,v 1.14 1998/04/13 18:03:02 bsmith Exp $";
#endif

/*T
    Concepts: Grid vectors^Multigrid
    Routines: GridCreateRectangular1D(); GridCreateGVec(); 
    Routines: ViewerDrawOpenX(); GridDestroy();
    Routines: GridRefine(); GridCreateRestriction(); GridCreateGMat();
    Routines: GVecEvaluateFunctionGalerkin();
    Routines: KSPSetInitialGuessNonzero(); SLESGetPC(); PCSetType();
    Routines: MGSetLevels(); MGSetResidual(); MGSetInterpolate();
    Routines: MGSetRestriction(); MGGetSmoother(); MGSetRhs();
    Routines: MGSetX(); MGSetR(); MGGetCoarseSolve(); MGCheck();
    Routines: SLESSetFromOptions(); SLESSolve(); SLESSetFromOptions();
    Routines: KSPBuildSolution(); 
    
    Comment: Shows how to set up and solve with multigrid a simple 
             PDE using the GVec routines.
T*/

static char help[] = "Solves simple PDE with GVec routines.\n\n";


#include "sles.h"
#include "gvec.h"
#include "mg.h"

extern int SolutionFunction(int,double *,double *,double *,Scalar *,void*);
extern int RHSFunction(int,double *,double *,double *,Scalar *,void*);

#include "src/gvec/examples/tutorials/common1and2.c"

#define MAXGRIDS 10

int main(int argc,char **args)
{
  GVec                   x[MAXGRIDS],b[MAXGRIDS],r[MAXGRIDS],u;  /* discrete grid vectors */
  GMat                   A[MAXGRIDS],R[MAXGRIDS];
  Grid                   grid[MAXGRIDS];
  int                    ierr, n = 10, flag, its,levels = 2,i;
  SLES                   subsles,sles;                           /* linear solver contexts */
  PC                     pc;
  KSP                    ksp;
  Scalar                 zero = 0.0;
  GVecErrorKSPMonitorCtx mctx;
  Draw                   draw;

  PetscInitialize(&argc,&args,0,help);
  ierr = GVecInitialize(); CHKERRA(ierr);

  ierr = OptionsGetInt(PETSC_NULL,"-n",&n,&flag); CHKERRA(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-levels",&levels,&flag); CHKERRA(ierr);
  if (levels > MAXGRIDS) SETERRA(1,0,"Too many levels requested");

  /*
      Construct the grid and the discretization context
  */
  ierr = GridCreateRectangular1D(PETSC_COMM_WORLD,n,0.0,1.0,PETSC_NULL,DISCRETIZATION_LINEAR,
                                 &grid[0]); CHKERRA(ierr);

  /* 
      From them create the grid vector on the coarsest grid 
  */
  ierr = GridCreateGVec(grid[0],&x[0]); CHKERRA(ierr);

  /*
      Create another vector to hold the right hand side and exact solution
  */
  ierr = VecDuplicate(x[0],&b[0]); CHKERRA(ierr);
  ierr = VecDuplicate(x[0],&r[0]); CHKERRA(ierr);


  /*
      Build the vectors and matrices for the finer grids
  */
  for ( i=0; i<levels-1; i++ ) {
    ierr = GridRefine(grid[i],2,&grid[i+1]); CHKERRA(ierr);
    ierr = GridCreateGVec(grid[i+1],&x[i+1]); CHKERRA(ierr);
    ierr = VecDuplicate(x[i+1],&b[i+1]); CHKERRA(ierr);
    ierr = VecDuplicate(x[i+1],&r[i+1]); CHKERRA(ierr);
    ierr = GridCreateRestriction(grid[i+1],grid[i],&R[i+1]); CHKERRA(ierr);
    ierr = GridCreateGMat(grid[i],A+i); CHKERRA(ierr);
    ierr = UserComputeMatrix(A[i]); CHKERRA(ierr);
    /* patch matrix for Dirichlet boundary conditions */
    ierr = UserComputeDirichlet(0,0,A[i],0,0); CHKERRA(ierr);
  }

  /*
      Evaluate the right-hand-side function on the grid. For finite
      differences this corresponds to the right-hand-side vector for 
      the linear system.
  */
  ierr = GVecEvaluateFunctionGalerkin(b[levels-1],RHSFunction,0); CHKERRA(ierr);

  /*
      Create a matrix data structure; then compute the Laplacian stiffness matrix
  */
  ierr = GridCreateGMat(grid[levels-1],A+levels-1); CHKERRA(ierr);
  ierr = UserComputeMatrix(A[levels-1]); CHKERRA(ierr);

  /*
      Zero the solution vector before applying the known values
  */
  ierr = VecSet(&zero,x[levels-1]); CHKERRA(ierr);

  /*
      Apply the Dirichlet boundary conditions to the vectors and matrix
  */
  ierr = UserComputeDirichlet(x[levels-1],b[levels-1],A[levels-1],
                               SolutionFunction,0); CHKERRA(ierr);

  /*
      Create a vector to hold the exact solution
  */
  ierr = VecDuplicate(x[levels-1],&u); CHKERRA(ierr);
  ierr = GVecEvaluateFunction(u,SolutionFunction,0); CHKERRA(ierr);

  /*
     Create the linear solver context
  */
  ierr = SLESCreate(PETSC_COMM_WORLD,&sles); CHKERRA(ierr);
  ierr = SLESSetOperators(sles,A[levels-1],A[levels-1],SAME_NONZERO_PATTERN); CHKERRA(ierr);
  ierr = SLESGetKSP(sles,&ksp); CHKERRA(ierr);
  ierr = KSPSetInitialGuessNonzero(ksp); CHKERRA(ierr);

  /*
      Set a routine to monitor the error at every iteration
  */
  ierr = ViewerDrawGetDraw(VIEWER_DRAWX_WORLD,&draw); CHKERRA(ierr);
  ierr = DrawSetTitle(draw,"Error"); CHKERRA(ierr);
  mctx.error_viewer      = VIEWER_DRAWX_WORLD;
  mctx.norm_error_viewer = VIEWER_STDOUT_WORLD;
  mctx.solution          = u;
  ierr = KSPSetMonitor(ksp,GVecErrorKSPMonitor,&mctx); CHKERRA(ierr);

  /*
      Set up to use multigrid 
  */
  ierr = SLESGetPC(sles,&pc); CHKERRA(ierr);
  ierr = PCSetType(pc,PCMG);  CHKERRA(ierr);
  ierr = MGSetLevels(pc,levels); CHKERRA(ierr);
  for ( i=1; i<levels; i++ ) {
    ierr = MGSetResidual(pc,i,MGDefaultResidual,A[i]); CHKERRA(ierr);
    ierr = MGSetInterpolate(pc,i,R[i]); CHKERRA(ierr);
    ierr = MGSetRestriction(pc,i,R[i]); CHKERRA(ierr);
    ierr = MGGetSmoother(pc,i,&subsles); CHKERRA(ierr);
    ierr = SLESSetOperators(subsles,A[i],A[i],SAME_NONZERO_PATTERN);CHKERRA(ierr);
    ierr = MGSetRhs(pc,i-1,b[i-1]);CHKERRA(ierr);
    ierr = MGSetX(pc,i-1,x[i-1]);CHKERRA(ierr);
    ierr = MGSetR(pc,i-1,r[i-1]);CHKERRA(ierr);
  }
  ierr = MGGetCoarseSolve(pc,&subsles); CHKERRA(ierr);
  ierr = SLESSetOperators(subsles,A[0],A[0],SAME_NONZERO_PATTERN);CHKERRA(ierr);
  ierr = MGSetR(pc,levels-1,r[levels-1]); CHKERRA(ierr);
  ierr = MGCheck(pc);  CHKERRA(ierr);

  /*
     Set various runtime options; then solve the linear system
  */
  ierr = SLESSetFromOptions(sles); CHKERRA(ierr);
  ierr = SLESSolve(sles,b[levels-1],x[levels-1],&its); CHKERRA(ierr);

  /*
     Free work space.  All PETSc objects should be destroyed when they
     are no longer needed.
  */
  for ( i=0; i<levels-1; i++ ) {
    ierr = VecDestroy(x[i]); CHKERRA(ierr);
    ierr = VecDestroy(b[i]); CHKERRA(ierr);
    ierr = VecDestroy(r[i]); CHKERRA(ierr);
    ierr = MatDestroy(A[i]); CHKERRA(ierr);
    ierr = MatDestroy(R[i+1]); CHKERRA(ierr);
    ierr = GridDestroy(grid[i]); CHKERRA(ierr);
  }
  ierr = GridDestroy(grid[levels-1]); CHKERRA(ierr);
  ierr = VecDestroy(u); CHKERRA(ierr);
  ierr = VecDestroy(x[levels-1]); CHKERRA(ierr);
  ierr = VecDestroy(b[levels-1]); CHKERRA(ierr);
  ierr = VecDestroy(r[levels-1]); CHKERRA(ierr);
  ierr = MatDestroy(A[levels-1]); CHKERRA(ierr);
  ierr = SLESDestroy(sles); CHKERRA(ierr);

  PetscFinalize();
  return 0;
}

#include <math.h>

/* -------------------------------------------------------------------- */
/*
   SolutionFunction - Defines the solution to the PDE. It evaluates the 
   solution at any set of points.

   Input Parameters:
.  n - array dimension
.  x,y,z - grid point coordinates (not used here)
.  values - array to hold solution values
.  ctx - optional user-defined context (not used here)

   Output Parameter:
.  values - newly computed solution values
*/
int SolutionFunction(int n,double *x,double *y,double *z,Scalar *values,void *ctx)
{
  int i;

  for ( i=0; i<n; i++ ) values[i] = x[i]*cos(x[i]);
  return 0;
}
/* -------------------------------------------------------------------- */
/*
   RHSFunction - Defines the right-hand-side function for this problem.

   Input Parameters:
.  n - array dimension
.  x,y,z - grid point coordinates (not used here)
.  values - array to hold RHS values
.  ctx - optional user-defined context (not used here)

   Output Parameter:
.  values - newly computed RHS values
*/
int RHSFunction(int n,double *x,double *y,double *z,Scalar *values,void *ctx)
{
  int i;

  for ( i=0; i<n; i++ ) values[i] = -2.0*sin(x[i]) - x[i]*cos(x[i]);
  return 0;
}




