#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: aijpc.c,v 1.50 1998/04/03 23:15:06 bsmith Exp $";
#endif

/*
   Defines a block Jacobi preconditioner for the Seq/MPIAIJ format.
*/
#include <math.h>
#include "src/mat/impls/aij/mpi/mpiaij.h"
#include "src/pc/pcimpl.h"
#include "src/vec/vecimpl.h"
#include "src/pc/impls/bjacobi/bjacobi.h"
#include "sles.h"

typedef struct {
  int m,m_blocks, nzl, nzu;       
  int *blocks;
  int *rlens;
  int *rlensb;
} PC_GSAIJ;

typedef struct {
  Vec      *x,*y;             /* work vectors for solves on each block */
  int      *starts;           /* starting point of each block */
  Mat      *mat,*pmat;        /* submatrices for each block */
  IS       *is;               /* for gathering the submatrices */
  PC_GSAIJ gsdata;            /* additional data for when running with Gauss-Seidel */
} PC_BJacobi_AIJ;

#undef __FUNC__  
#define __FUNC__ "PCDestroy_BJacobi_AIJ"
int PCDestroy_BJacobi_AIJ(PC pc)
{
  PC_BJacobi     *jac = (PC_BJacobi *) pc->data;
  PC_BJacobi_AIJ *bjac = (PC_BJacobi_AIJ *) jac->data;
  int            i,ierr;

  PetscFunctionBegin;
  ierr = MatDestroyMatrices(jac->n_local,&bjac->pmat); CHKERRQ(ierr);
  if (jac->use_true_local) {
    ierr = MatDestroyMatrices(jac->n_local,&bjac->mat); CHKERRQ(ierr);
  }

  for ( i=0; i<jac->n_local; i++ ) {
    ierr = SLESDestroy(jac->sles[i]); CHKERRQ(ierr);
    ierr = VecDestroy(bjac->x[i]); CHKERRQ(ierr);
    ierr = VecDestroy(bjac->y[i]); CHKERRQ(ierr);
    ierr = ISDestroy(bjac->is[i]); CHKERRQ(ierr);
  }
  PetscFree(jac->sles);
  PetscFree(bjac->x);
  PetscFree(bjac->starts);
  PetscFree(bjac->is);
  if (jac->gs) {
    PetscFree(bjac->gsdata.blocks);
    PetscFree(bjac->gsdata.rlens);
    PetscFree(bjac->gsdata.rlensb);
  }
  PetscFree(bjac);
  if (jac->l_lens) PetscFree(jac->l_lens);
  if (jac->g_lens) PetscFree(jac->g_lens);
  PetscFree(jac); 
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "PCSetUpOnBlocks_BJacobi_AIJ"
int PCSetUpOnBlocks_BJacobi_AIJ(PC pc)
{
  PC_BJacobi     *jac = (PC_BJacobi *) pc->data;
  int            ierr,i,n_local = jac->n_local;
  PC_BJacobi_AIJ *bjac = (PC_BJacobi_AIJ *) jac->data;

  PetscFunctionBegin;
  for ( i=0; i<n_local; i++ ) {
    ierr = SLESSetUp(jac->sles[i],bjac->x[i],bjac->y[i]); CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

/*
      Preconditioner for block Jacobi 
*/
#undef __FUNC__  
#define __FUNC__ "PCApply_BJacobi_AIJ"
int PCApply_BJacobi_AIJ(PC pc,Vec x, Vec y)
{
  PC_BJacobi     *jac = (PC_BJacobi *) pc->data;
  int            ierr,its,i,n_local = jac->n_local;
  PC_BJacobi_AIJ *bjac = (PC_BJacobi_AIJ *) jac->data;
  Scalar         *xin,*yin,*lx_array,*ly_array;
  static int     flag = 1,SUBSlesSolve;

  PetscFunctionBegin;
  if (flag) {
    ierr = PLogEventRegister(&SUBSlesSolve,"SubSlesSolve    ","black:");CHKERRQ(ierr);
    flag=0;
  }
  ierr = VecGetArray(x,&xin);CHKERRQ(ierr);
  ierr = VecGetArray(y,&yin);CHKERRQ(ierr);
  for ( i=0; i<n_local; i++ ) {
    /* 
       To avoid copying the subvector from x into a workspace we instead 
       make the workspace vector array point to the subpart of the array of
       the global vector.
    */
    ierr = VecGetArray(bjac->x[i],&lx_array);CHKERRQ(ierr);
    ierr = VecGetArray(bjac->y[i],&ly_array);CHKERRQ(ierr);
    ierr = VecPlaceArray(bjac->x[i],xin+bjac->starts[i]);CHKERRQ(ierr);
    ierr = VecPlaceArray(bjac->y[i],yin+bjac->starts[i]);CHKERRQ(ierr);

    PLogEventBegin(SUBSlesSolve,jac->sles[i],bjac->x[i],bjac->y[i],0);
    ierr = SLESSolve(jac->sles[i],bjac->x[i],bjac->y[i],&its); CHKERRQ(ierr);
    PLogEventEnd(SUBSlesSolve,jac->sles[i],bjac->x[i],bjac->y[i],0);
    
    ierr = VecPlaceArray(bjac->x[i],lx_array);CHKERRQ(ierr);
    ierr = VecPlaceArray(bjac->y[i],ly_array);CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

/*
     Preconditioner for block Gauss-Seidel
*/
#undef __FUNC__  
#define __FUNC__ "PCApply_BGS_AIJ"
int PCApply_BGS_AIJ(PC pc,Vec x, Vec y)
{
  PC_BJacobi     *jac = (PC_BJacobi *) pc->data;
  Mat_SeqAIJ     *A;
  Mat_MPIAIJ     *Ampi;
  int            ierr,its,i,n_local = jac->n_local,j,n,row,shift ,*idx;
  int            max,*ai, *blks, *aj, *st, *lc, *uc, tmp, i1, i2;
  PC_BJacobi_AIJ *bjac = (PC_BJacobi_AIJ *) jac->data;
  Scalar         *xin,*yin,*lx_array,*ly_array,sum,*b,*v,*yins, *aa, v1, v2;
  PC_GSAIJ       *gs = &bjac->gsdata;
  Vec            *lx, *ly;
  SLES           *lsles ;

  PetscFunctionBegin;
  PLogEventBegin(PC_Apply,pc,x,y,0);
  /* access local matrix data (in parallel case this is the diagonal block */
  if (pc->mat->type == MATSEQAIJ) {
    A = (Mat_SeqAIJ*) pc->mat->data;
  } else {
    Ampi = (Mat_MPIAIJ*) pc->mat->data;
    A    = (Mat_SeqAIJ*) Ampi->A->data; 
  }
  shift = A->indexshift;
  ierr = VecGetArray(x,&xin);CHKERRQ(ierr);
  ierr = VecGetArray(y,&yin);CHKERRQ(ierr); yins = yin + shift;

  /*some shorts to ptrs*/
  aa   = A->a + shift;
  aj   = A->j + shift;
  ai   = A->i;
  blks = gs->blocks;
  lc   = gs->rlens;     /* lower count */
  uc   = gs->rlensb;    /* upper count */
  st   = bjac->starts;
  lx   = bjac->x;
  ly   = bjac->y;
  lsles= jac->sles;
  /*for both PCBGS_FORWARD_SWEEP and PCBGS_SYMMETRIC_SWEEP */
  for ( i=0; i<n_local; i++ ) {
    row  = blks[i];
    ierr = VecGetArray(lx[i],&lx_array);CHKERRQ(ierr); 
    b    = xin + st[i];

    /* do rectangular multiply (D+L)'b */
    for ( j=0, max=blks[i+1]-blks[i]; j<max; j++) {
      v   = aa + ai[row];
      idx = aj + ai[row];
      n   = lc[row];
      sum = b[j];
      for(; n>1; n-=2){
        i1 = idx[0];
        i2 = idx[1]; idx+=2;
        v1 = yins[i1];
        v2 = yins[i2];
        sum-= v[0]*v1 + v[1] *v2; v+=2;
      }
      if (n==1) sum -= *v++ * yins[*idx++];
      lx_array[j] = sum;
      row++;
    }
    ierr = VecGetArray(ly[i],&ly_array);CHKERRQ(ierr);
    ierr = VecPlaceArray(ly[i],yin+st[i]);CHKERRQ(ierr);
    ierr = SLESSolve(lsles[i],lx[i],ly[i],&its); CHKERRQ(ierr);
    ierr = VecPlaceArray(ly[i],ly_array); CHKERRQ(ierr);
  }
  PLogFlops(2*gs->nzl);    /* End Forward sweep*/

  if (jac->gstype == PCBGS_SYMMETRIC_SWEEP) {
    for ( i=n_local-1; i>=0; i-- ) {
      row  = blks[i];
      ierr = VecGetArray(lx[i],&lx_array);CHKERRQ(ierr);
      b    = xin + st[i];

      for ( j=0, max=blks[i+1]-blks[i]; j<max; j++) {

        /*do U*/
        tmp =  ai[row] + uc[row];
        v   = aa + tmp;
        idx = aj + tmp;
        n   = ai[row+1] - tmp;
        sum = b[j];
        for(; n>1; n-=2){
          i1 = idx[0];
          i2 = idx[1]; idx+=2;
          v1 = yins[i1];
          v2 = yins[i2];
          sum-= v[0]*v1 + v[1] *v2; v+=2;
        }
        if (n==1) sum -= *v++ * yins[*idx++];

        /*do L*/
        v   = aa + ai[row];
        idx = aj + ai[row];
        n   = lc[row];
        for(; n>1; n-=2){
          i1 = idx[0];
          i2 = idx[1]; idx+=2;
          v1 = yins[i1];
          v2 = yins[i2];
          sum-= v[0]*v1 + v[1] *v2; v+=2;
        }
        if (n==1) sum -= *v++ * yins[*idx++];
        
        lx_array[j] = sum;
        row++;
      }
      
      ierr = VecGetArray(ly[i],&ly_array);CHKERRQ(ierr);
      ierr = VecPlaceArray(ly[i],yin+st[i]);CHKERRQ(ierr);
      ierr = SLESSolve(lsles[i],lx[i],ly[i],&its); CHKERRQ(ierr);
      ierr = VecPlaceArray(ly[i],ly_array);CHKERRQ(ierr);
    }
    PLogFlops(2*(gs->nzl+gs->nzu));
  }
  PLogEventEnd(PC_Apply,pc,x,y,0);

  PetscFunctionReturn(0);
}


extern int PCSetUp_BJacobi_MPIAIJ(PC);
extern int PCSetUp_BJacobi_SeqAIJ(PC);

#undef __FUNC__  
#define __FUNC__ "PCSetUp_BJacobi_AIJ"
int PCSetUp_BJacobi_AIJ(PC pc)
{
  PC_BJacobi          *jac = (PC_BJacobi *) pc->data;
  Mat                 mat = pc->mat, pmat = pc->pmat;
  int                 ierr, m, n_local, N, M, start, i, rank, size, sum, end, i_start;
  int                 i_end, ct1, ct2;
  char                *prefix;
  SLES                sles;
  Vec                 x,y;
  PC_BJacobi_AIJ      *bjac = (PC_BJacobi_AIJ *) jac->data;
  KSP                 subksp;
  PC                  subpc;
  IS                  is;
  MatGetSubMatrixCall scall = MAT_REUSE_MATRIX;

  PetscFunctionBegin;
  MPI_Comm_rank(pc->comm,&rank);
  MPI_Comm_size(pc->comm,&size);
  MatGetLocalSize(pc->pmat,&M,&N);

  /*   local block count  given */
  if (jac->n_local > 0 && jac->n < 0) {
    ierr = MPI_Allreduce(&jac->n_local,&jac->n,1,MPI_INT,MPI_SUM,pc->comm);CHKERRQ(ierr);
    if (jac->l_lens) { /* check that user set these correctly */
      sum = 0;
      for ( i=0; i<jac->n_local; i++ ) {
        if (!jac->l_lens[i]) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"Block size of zero not allowed");
        sum += jac->l_lens[i];
      }
      if (sum != M) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"Local lens sent incorrectly");
    }
  } else if (jac->n > 0 && jac->n_local < 0) {
    /* global blocks given: determine which ones are local */
    if (jac->g_lens) {
      if (size == 1) {
        jac->n_local = jac->n;
        jac->l_lens  = (int *) PetscMalloc(jac->n_local*sizeof(int));CHKPTRQ(jac->l_lens);
        PLogObjectMemory(pc,jac->n_local*sizeof(int));
        PetscMemcpy(jac->l_lens,jac->g_lens,jac->n_local*sizeof(int));
        /* check that user set these correctly */
        sum = 0;
        for ( i=0; i<jac->n_local; i++ ) {
        if (!jac->l_lens[i]) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"Block size of zero not allowed");
          sum += jac->l_lens[i];
        }
        if (sum != M) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"Global lens sent incorrectly");
      } else {
        MatGetOwnershipRange(pc->pmat,&start,&end);
        /* loop over blocks determing first one owned by me */
        sum = 0;
        for ( i=0; i<jac->n+1; i++ ) {
          if (sum == start) { i_start = i; goto start_1;}
          if (i<jac->n) sum += jac->g_lens[i];
        }
        SETERRQ(PETSC_ERR_ARG_SIZ,0,"Block sizes\n\
                   used in PCBJacobiSetTotalBlocks()\n\
                   are not compatible with parallel matrix layout");
 start_1: 
        for ( i=i_start; i<jac->n+1; i++ ) {
          if (sum == end) { i_end = i; goto end_1; }
          if (i<jac->n) sum += jac->g_lens[i];
        }          
        SETERRQ(PETSC_ERR_ARG_SIZ,0,"Block sizes\n\
                      used in PCBJacobiSetTotalBlocks()\n\
                      are not compatible with parallel matrix layout");
 end_1: 
        jac->n_local = i_end - i_start;
        jac->l_lens = (int *) PetscMalloc(jac->n_local*sizeof(int));CHKPTRQ(jac->l_lens); 
        PLogObjectMemory(pc,jac->n_local*sizeof(int));
        PetscMemcpy(jac->l_lens,jac->g_lens+i_start,jac->n_local*sizeof(int));
      }
    } else {
      jac->n_local = jac->n/size + ((jac->n % size) > rank);
      if (!jac->n_local) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"Not enough blocks for processors");
      jac->l_lens  = (int *) PetscMalloc(jac->n_local*sizeof(int));CHKPTRQ(jac->l_lens);
      PLogObjectMemory(pc,jac->n_local*sizeof(int));
      for ( i=0; i<jac->n_local; i++ ) {
        jac->l_lens[i] = M/jac->n_local + ((M % jac->n_local) > i);
        if (!jac->l_lens[i]) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"Too many blocks given");
      }
    }
  } else if (jac->n < 0 && jac->n_local < 0) {
    jac->n         = size;
    jac->n_local   = 1;
    jac->l_lens    = (int *) PetscMalloc(sizeof(int));CHKPTRQ(jac->l_lens);
    PLogObjectMemory(pc,sizeof(int));
    jac->l_lens[0] = M;
  }

  /* special case of one block on the processor */
  if (pmat->type == MATMPIAIJ && jac->n_local == 1 && !pc->modifysubmatrices) {
    ierr = PCSetUp_BJacobi_MPIAIJ(pc);CHKERRQ(ierr);
    PetscFunctionReturn(0);
  } else if (pmat->type == MATSEQAIJ && jac->n_local == 1 && !pc->modifysubmatrices) {
    ierr = PCSetUp_BJacobi_SeqAIJ(pc);CHKERRQ(ierr);
    PetscFunctionReturn(0);
  }

  /* if parallel matrix extract submatrices from diagonal block */
  if (pmat->type == MATMPIAIJ) {
    Mat_MPIAIJ *tmat  = (Mat_MPIAIJ*) mat->data;
    Mat_MPIAIJ *tpmat = (Mat_MPIAIJ*) pmat->data;
    mat   = tmat->A;
    pmat  = tpmat->A;
  }

  n_local = jac->n_local;

  if (jac->use_true_local) {
    if (pc->mat->type != pc->pmat->type) SETERRQ(PETSC_ERR_ARG_INCOMP,0,"Matrices not of same type");
  }

  /* set default direct solver with no Krylov method */
  if (!pc->setupcalled) {
    scall             = MAT_INITIAL_MATRIX;
    pc->destroy       = PCDestroy_BJacobi_AIJ;
    pc->apply         = PCApply_BJacobi_AIJ;
    pc->setuponblocks = PCSetUpOnBlocks_BJacobi_AIJ;

    bjac         = (PC_BJacobi_AIJ *) PetscMalloc(sizeof(PC_BJacobi_AIJ));CHKPTRQ(bjac);
    PLogObjectMemory(pc,sizeof(PC_BJacobi_AIJ));
    jac->sles    = (SLES*) PetscMalloc(n_local*sizeof(SLES)); CHKPTRQ(jac->sles);
    PLogObjectMemory(pc,sizeof(n_local*sizeof(SLES)));
    bjac->x      = (Vec*) PetscMalloc(2*n_local*sizeof(Vec)); CHKPTRQ(bjac->x);
    PLogObjectMemory(pc,sizeof(2*n_local*sizeof(Vec)));
    bjac->y      = bjac->x + n_local;
    bjac->starts = (int*) PetscMalloc(n_local*sizeof(Scalar));CHKPTRQ(bjac->starts);
    PLogObjectMemory(pc,sizeof(n_local*sizeof(Scalar)));
    
    jac->data    = (void *) bjac;
    bjac->is     = (IS *) PetscMalloc(n_local*sizeof(IS)); CHKPTRQ(bjac->is);
    PLogObjectMemory(pc,sizeof(n_local*sizeof(IS)));

    /* 
        construct data structure for block triangular solves 
        when block Gauss-Seidel is being used 
    */
    if (jac->gs) {
      Mat_SeqAIJ  *aij =  (Mat_SeqAIJ  *) pmat->data;
      PC_GSAIJ    *gs;
      int         p, q, r, nz, nz2, shift = aij->indexshift;

      pc->apply    = PCApply_BGS_AIJ; /* user Gauss-Seidel preconditioner */      
      gs           = &bjac->gsdata;
      gs->m        = aij->m;
      gs->m_blocks = n_local;
      gs->blocks   = (int *) PetscMalloc((gs->m_blocks+1)*sizeof(int));CHKPTRQ(gs->blocks);
      gs->rlens    = (int *) PetscMalloc(gs->m*sizeof(int));CHKPTRQ(gs->rlens);
      gs->rlensb   = (int *) PetscMalloc(gs->m*sizeof(int));CHKPTRQ(gs->rlensb);
      PLogObjectMemory(pc,(gs->m_blocks+1)*sizeof(int)+2*gs->m*sizeof(int));
      gs->blocks[0] = 0;
      for (i=1; i<gs->m_blocks; i++) {
	gs->blocks[i] = gs->blocks[i-1] + jac->l_lens[i-1];
      }
      gs->blocks[gs->m_blocks] = gs->m;

      ct1 = 0; ct2 = 0;
      for (p=0; p<(gs->m_blocks); p++) {
	for (q=gs->blocks[p]; q<gs->blocks[p+1]; q++) {
	  nz=0;
	  nz2=0;
	  for (r=aij->i[q]+shift; r<aij->i[q+1]+shift; r++) {
	    if ((aij->j[r]+shift)<gs->blocks[p]) {nz++;}
	    if ((aij->j[r]+shift)<(gs->blocks[p]+jac->l_lens[p])) {nz2++;}
	    else {break;}
	  }
	  gs->rlens[q] = nz;
	  gs->rlensb[q]= nz2;
          ct1         += nz;
          ct2         += nz2;
	}
      }
      gs->nzl = ct1;
      gs->nzu = aij->nz - ct2;
#ifdef USE_PETSC_LOG      
      {
        int ai;
        Scalar *aa, val;
        double norm, sum1;
        
        aa = aij->a+shift;
        for(p=0,sum1=0.0; p< aij->m; ++p){
          ai = aij->i[p];
          for(q = 0; q<gs->rlens[p]; ++q){
            val =  aa[ai+q];
#if defined(USE_PETSC_COMPLEX)
            sum1 += real(conj(val)*(val));
#else
            sum1+= val*val;
#endif
          }
          for(q = ai+gs->rlensb[p]; q < aij->i[p+1]; ++q) {
            val =  aa[q];
#if defined(USE_PETSC_COMPLEX)
            sum1 += real(conj(val)*(val));
#else
            sum1+= val*val;
#endif
          }
        }
        ct1 = gs->nzu + gs->nzl;
        ierr = MatNorm(pmat,NORM_FROBENIUS,&norm); CHKERRQ(ierr);
        PLogInfo(mat,"PCSetUp_BJacobi_AIJ:Percent non-zeros thrown away= %4.2f%%, Norm ratio =  %4.2f%% \n", ct1*100.0/aij->nz, sqrt(sum1)*100/norm);
      }
#endif

    }

    start = 0;
    for ( i=0; i<n_local; i++ ) {
      ierr = SLESCreate(PETSC_COMM_SELF,&sles); CHKERRQ(ierr);
      PLogObjectParent(pc,sles);
      ierr = SLESGetKSP(sles,&subksp); CHKERRQ(ierr);
      ierr = KSPSetType(subksp,KSPPREONLY); CHKERRQ(ierr);
      ierr = SLESGetPC(sles,&subpc); CHKERRQ(ierr);
      ierr = PCSetType(subpc,PCILU); CHKERRQ(ierr);
      ierr = PCGetOptionsPrefix(pc,&prefix); CHKERRQ(ierr);
      ierr = SLESSetOptionsPrefix(sles,prefix); CHKERRQ(ierr);
      ierr = SLESAppendOptionsPrefix(sles,"sub_"); CHKERRQ(ierr);
      ierr = SLESSetFromOptions(sles); CHKERRQ(ierr);

      m = jac->l_lens[i];

      ierr = VecCreateSeq(PETSC_COMM_SELF,m,&x); CHKERRQ(ierr);
      ierr = VecDuplicate(x,&y); CHKERRQ(ierr);
      PLogObjectParent(pc,x);
      PLogObjectParent(pc,y);
      bjac->x[i]      = x;
      bjac->y[i]      = y;
      bjac->starts[i] = start;
      jac->sles[i]    = sles;

      ierr = ISCreateStride(PETSC_COMM_SELF,m,start,1,&is); CHKERRQ(ierr);
      bjac->is[i] = is;
      PLogObjectParent(pc,is);

      start += m;
    }
  } else {
    bjac = (PC_BJacobi_AIJ *) jac->data;
    /* 
       Destroy the blocks from the previous iteration
    */
    if (pc->flag == DIFFERENT_NONZERO_PATTERN) {
      ierr = MatDestroyMatrices(n_local,&bjac->pmat); CHKERRQ(ierr);
      if (jac->use_true_local) {
        ierr = MatDestroyMatrices(n_local,&bjac->mat); CHKERRQ(ierr);
      }
      scall = MAT_INITIAL_MATRIX;
    }
  }

  ierr = MatGetSubMatrices(pmat,n_local,bjac->is,bjac->is,scall,&bjac->pmat);CHKERRQ(ierr);
  if (jac->use_true_local) {
    ierr = MatGetSubMatrices(mat,n_local,bjac->is,bjac->is,scall,&bjac->mat);CHKERRQ(ierr);
  }
  /* Return control to the user so that the submatrices can be modified (e.g., to apply
     different boundary conditions for the submatrices than for the global problem) */
  ierr = PCModifySubMatrices(pc,n_local,bjac->is,bjac->is,bjac->pmat,pc->modifysubmatricesP); CHKERRQ(ierr);
  for ( i=0; i<n_local; i++ ) {
    PLogObjectParent(pc,bjac->pmat[i]);
    if (jac->use_true_local) {
      PLogObjectParent(pc,bjac->mat[i]);
      ierr = SLESSetOperators(jac->sles[i],bjac->mat[i],bjac->pmat[i],pc->flag);CHKERRQ(ierr);
    } else {
      ierr = SLESSetOperators(jac->sles[i],bjac->pmat[i],bjac->pmat[i],pc->flag);CHKERRQ(ierr);
    }
  }

  PetscFunctionReturn(0);
}



