      subroutine dodcfg(nx,ny,x,f,fgrad,task,lambda)
      character*(*) task
      integer nx, ny
      double precision f, lambda
      double precision x(nx*ny), fgrad(nx*ny)
!     **********
!
!     Subroutine dodcfg
!
!     This subroutine computes the function and gradient of the
!     optimal design with composite materials problem.
!
!     The subroutine statement is
!
!       subroutine dodcfg(nx,ny,x,f,fgrad,task,lambda)
!
!     where
!
!       nx is an integer variable.
!         On entry nx is the number of grid points in the first
!            coordinate direction.
!         On exit nx is unchanged.
!
!       ny is an integer variable.
!         On entry ny is the number of grid points in the second
!            coordinate direction.
!         On exit ny is unchanged.
!
!       x is a double precision array of dimension nx*ny.
!         On entry x specifies the vector x if task = 'F', 'G', or 'FG'.
!            Otherwise x need not be specified.
!         On exit x is unchanged if task = 'F', 'G', or 'FG'. Otherwise
!            x is set according to task.
!
!       f is a double precision variable.
!         On entry f need not be specified.
!         On exit f is set to the function evaluated at x if task = 'F'
!            or 'FG'.
!
!       fgrad is a double precision array of dimension nx*ny.
!         On entry fgrad need not be specified.
!         On exit fgrad contains the gradient evaluated at x if
!            task = 'G' or 'FG'.
!
!       task is a character variable.
!         On entry task specifies the action of the subroutine:
!
!            task               action
!            ----               ------
!             'F'     Evaluate the function at x.
!             'G'     Evaluate the gradient at x.
!             'FG'    Evaluate the function and the gradient at x.
!             'XS'    Set x to the standard starting point xs.
!
!         On exit task is unchanged.
!
!       lambda is a double precision variable.
!         On entry lambda is the Lagrange multiplier.
!         On exit lambda is unchanged.
!
!     Subprograms called
!
!       MINPACK-supplied   ...   dodcps
!
!     MINPACK-2 Project. November 1993.
!     Argonne National Laboratory and University of Minnesota.
!     Brett M. Averick.
!
!     **********
      double precision mu1, mu2, one, p5, two, zero
      parameter (zero=0.0d0,p5=0.5d0,one=1.0d0,two=2.0d0)
      parameter (mu1=one,mu2=two)

      logical feval, geval
      integer i, j, k
      double precision area, dpsi, dpsip, dvdx, dvdy, gradv, hx, hxhy
      double precision hy, temp, t1, t2, v, vb, vl, vr, vt

      external dodcps

!     Initialization.

      hx = one/dble(nx+1)
      hy = one/dble(ny+1)
      hxhy = hx*hy
      area = p5*hxhy

!     Compute the break points.

      t1 = sqrt(two*lambda*mu1/mu2)
!
!     t2 = sqrt(two*lambda*mu2/mu1)
!
      t2 = 2.0*sqrt(lambda)

!     Compute the standard starting point if task = 'XS'.

      if (task .eq. 'XS') then
         do 20 j = 1, ny
            temp = dble(min(j,ny-j+1))*hy
            do 10 i = 1, nx
               k = nx*(j-1) + i
               x(k) = -(min(dble(min(i,nx-i+1))*hx,temp))**2
   10       continue
   20    continue

         return

      end if

      if (task .eq. 'F' .or. task .eq. 'FG') then
         feval = .true.
      else
         feval = .false.
      end if
      if (task .eq. 'G' .or. task .eq. 'FG') then
         geval = .true.
      else
         geval = .false.
      end if

!     Evaluate the function if task = 'F', the gradient if task = 'G',
!     or both if task = 'FG'.

      if (feval) f = zero
      if (geval) then
         do 30 k = 1, nx*ny
            fgrad(k) = zero
   30    continue
      end if

!     Computation of the function and the gradient over the lower
!     triangular elements.

      do 50 j = 0, ny
         do 40 i = 0, nx
            k = nx*(j-1) + i
            v = zero
            vr = zero
            vt = zero
            if (j .ge. 1 .and. i .ge. 1) v = x(k)
            if (i .lt. nx .and. j .gt. 0) vr = x(k+1)
            if (i .gt. 0 .and. j .lt. ny) vt = x(k+nx)
            dvdx = (vr-v)/hx
            dvdy = (vt-v)/hy
            gradv = dvdx**2 + dvdy**2
            if (feval) then
               call dodcps(gradv,mu1,mu2,t1,t2,dpsi,0,lambda)
               f = f + dpsi
            end if
            if (geval) then
               call dodcps(gradv,mu1,mu2,t1,t2,dpsip,1,lambda)
               if (i .ge. 1 .and. j .ge. 1)                             &
     &             fgrad(k) = fgrad(k) - two*(dvdx/hx+dvdy/hy)*dpsip
               if (i .lt. nx .and. j .gt. 0)                            &
     &             fgrad(k+1) = fgrad(k+1) + two*(dvdx/hx)*dpsip
               if (i .gt. 0 .and. j .lt. ny)                            &
     &             fgrad(k+nx) = fgrad(k+nx) + two*(dvdy/hy)*dpsip
            end if
   40    continue
   50 continue

!     Computation of the function and the gradient over the upper
!     triangular elements.

      do 70 j = 1, ny + 1
         do 60 i = 1, nx + 1
            k = nx*(j-1) + i
            vb = zero
            vl = zero
            v = zero
            if (i .le. nx .and. j .gt. 1) vb = x(k-nx)
            if (i .gt. 1 .and. j .le. ny) vl = x(k-1)
            if (i .le. nx .and. j .le. ny) v = x(k)
            dvdx = (v-vl)/hx
            dvdy = (v-vb)/hy
            gradv = dvdx**2 + dvdy**2
            if (feval) then
               call dodcps(gradv,mu1,mu2,t1,t2,dpsi,0,lambda)
               f = f + dpsi
            end if
            if (geval) then
               call dodcps(gradv,mu1,mu2,t1,t2,dpsip,1,lambda)
               if (i .le. nx .and. j .gt. 1)                            &
     &             fgrad(k-nx) = fgrad(k-nx) - two*(dvdy/hy)*dpsip
               if (i .gt. 1 .and. j .le. ny)                            &
     &             fgrad(k-1) = fgrad(k-1) - two*(dvdx/hx)*dpsip
               if (i .le. nx .and. j .le. ny)                           &
     &             fgrad(k) = fgrad(k) + two*(dvdx/hx+dvdy/hy)*dpsip
            end if
   60    continue
   70 continue

!     Scale the function.

      if (feval) f = area*f

!     Integrate v over the domain.

      if (feval) then
         temp = zero
         do 80 k = 1, nx*ny
            temp = temp + x(k)
   80    continue
         f = f + hxhy*temp
      end if
      if (geval) then
         do 90 k = 1, nx*ny
            fgrad(k) = area*fgrad(k) + hxhy
   90    continue
      end if

      end
