#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: ex1.c,v 1.46 1997/11/28 16:21:55 bsmith Exp $";
#endif

static char help[] = "Demonstrates use of the SNES package to solve several\n\
unconstrained minimization problems on a single processor.  All of these\n\
examples are taken from the MINPACK-2 test suite, and employ sparse\n\
storage of the Hessian matrices by default.  The command line options are:\n\
  -mx xg, where xg = number of grid points in the 1st coordinate direction\n\
  -my yg, where yg = number of grid points in the 2nd coordinate direction\n\
  -par param, where param is the test problem parameter\n\
  -p problem_number, where the possible problem numbers are:\n\
     1: Elastic-Plastic Torsion (dept)\n\
        param = angle of twist per unit length\n\
     2: Pressure Journal Bearing (dpjb)\n\
        param = eccentricity in (0,1)\n\
     3: Minimal Surface Area (dmsa)\n\
        Note:  param is not an option for this problem.\n\
     4: Optimal Design with Composites (dodc)\n\
        param = Lagrange multiplier\n\
     5: Steady State Combustion (dssc)\n\
        param = nonnegative Frank-Kamenetski parameter\n\
     6: Ginzburg-Landau (1-Dimensional) (dgl1)\n\
        param = temperature in (3.73, 7.32)\n\
        Note:  The option -my is not valid for this 1-dimensional problem.\n\
     7: Ginzburg-Landau (2-Dimensional) (dgl2)\n\
        param = number of vortices\n";

/* ------------------------------------------------------------------- */
/* We thank Brett Averick and Jorge More' for these test problems from */
/* the MINPACK-2 test suite.                                           */ 
/* ------------------------------------------------------------------- */

#if !defined(USE_PETSC_COMPLEX)

#include "petsc.h"
#include "snes.h"
#include "viewer.h"

/* User-defined application context */
   typedef struct {
      int     mx;       /* discretization in x-direction */
      int     my;       /* discretization in y-direction */
      int     ndim;     /* problem dimension */
      int     number;   /* test problem number */
      double  param;    /* test problem parameter */
      double  *work;    /* work space for problems #3 and #7 */
      double *s, *y;    /* work space for computing Hessian */
   } AppCtx;

int FormHessian(SNES,Vec,Mat*,Mat*,MatStructure*,void*);
int FormMinimizationFunction(SNES,Vec,Scalar*,void*);
int FormGradient(SNES,Vec,Vec,void*);
int FormInitialGuess(AppCtx*,Vec);
int UserDestroyWork(AppCtx);
int UserSetProblem(AppCtx*,int*,char**);

#if defined(HAVE_FORTRAN_CAPS)
#define depths_	DEPTHS
#define dpjbhs_	DPJBHS
#define dmsabc_	DMSABC
#define dmsahs_ DMSAHS
#define dodchs_ DODCHS
#define dsschs_ DSSCHS
#define dgl1hs_ DGL1HS
#define dgl2hs_ DGL2HS
#define mpintf_ MPINTF

#elif !defined(HAVE_FORTRAN_UNDERSCORE)
#define depths_	depths
#define dpjbhs_	dpjbhs
#define dmsabc_	dmsabc
#define dmsahs_ dmsahs
#define dodchs_ dodchs
#define dsschs_ dsschs
#define dgl1hs_ dgl1hs
#define dgl2hs_ dgl2hs
#define mpintf_ mpintf
#endif

#if defined(__cplusplus)
extern "C" {
#endif
int mpintf_(int*,int*,int*,int*,double*,double*,double*,double*,double*,
            char*,int);
void depths_(int*,int*,double*,double*);
void dpjbhs_(int*,int*,double*,double*,double*,double*);
void dmsahs_(int*,int*,double*,double*,double*,double*,double*,double*,double*);
void dmsabc_(int*,int*,double*,double*,double*,double*);
void dodchs_(int*,int*,double*,double*,double*,double*);
void dsschs_(int*,int*,double*,double*,double*,double*);
void dgl1hs_(int*,double*,double*,double*,double*);
void dgl2hs_(int*,int*,double*,double*,double*,double*,double*,int*);
#if defined(__cplusplus)
}
#endif

int main(int argc,char **argv)
{
  SNES     snes;                  /* SNES context */
  SNESType method = SNES_UM_TR;  /* nonlinear solution method */
  Vec      x, g;                  /* solution, gradient vectors */
  Mat      H;                     /* Hessian matrix */
  AppCtx   user;                  /* application context */
  int      ierr, its, N, nfails;

  PetscInitialize(&argc,&argv,(char *)0,help);

  /* Create user context and set problem data */
  ierr = UserSetProblem(&user,&argc,argv); CHKERRQ(ierr);
  N = user.ndim;

  /* Set up data structures */
  ierr = VecCreate(PETSC_COMM_SELF,PETSC_DECIDE,N,&x); CHKERRA(ierr);
  ierr = VecDuplicate(x,&g); CHKERRA(ierr);
  ierr = MatCreate(PETSC_COMM_SELF,N,N,&H); CHKERRA(ierr);
  ierr = MatSetOption(H,MAT_SYMMETRIC); CHKERRA(ierr);

  /* Create nonlinear solver */
  ierr = SNESCreate(PETSC_COMM_SELF,SNES_UNCONSTRAINED_MINIMIZATION,&snes); CHKERRA(ierr);
  ierr = SNESSetType(snes,method); CHKERRA(ierr);
  PetscPrintf(PETSC_COMM_SELF,"Problem = %d, param = %g, N = %d\n",user.number,user.param,N);

  /* Set various routines and options */
  ierr = SNESSetMinimizationFunction(snes,FormMinimizationFunction,
         (void *)&user); CHKERRA(ierr);
  ierr = SNESSetGradient(snes,g,FormGradient,(void *)&user); CHKERRA(ierr);
  ierr = SNESSetHessian(snes,H,H,FormHessian,(void *)&user); CHKERRA(ierr);
  ierr = SNESSetFromOptions(snes); CHKERRA(ierr);

  /* Solve minimization problem */
  ierr = FormInitialGuess(&user,x); CHKERRA(ierr);
  ierr = SNESSolve(snes,x,&its);  CHKERRA(ierr); 
  ierr = SNESGetNumberUnsuccessfulSteps(snes,&nfails); CHKERRA(ierr);
  ierr = SNESView(snes,VIEWER_STDOUT_WORLD); CHKERRA(ierr);
  PetscPrintf(PETSC_COMM_SELF,"number of Newton iterations = %d, ",its);
  PetscPrintf(PETSC_COMM_SELF,"number of unsuccessful steps = %d\n\n",nfails);

  /* Free data structures */
  ierr = VecDestroy(x); CHKERRA(ierr);  ierr = VecDestroy(g); CHKERRA(ierr);
  ierr = MatDestroy(H); CHKERRA(ierr);  ierr = UserDestroyWork(user); CHKERRA(ierr);
  ierr = SNESDestroy(snes); CHKERRA(ierr);

  PetscFinalize();
  return 0;
}
/* -------------------------------------------------------------------- */
/*
    Evaluate function f(x) on a single processor
 */

int FormMinimizationFunction(SNES snes,Vec xvec,Scalar *f,void *ptr)
{
  AppCtx *user = (AppCtx *) ptr;
  int    ierr;
  Scalar *x;

  ierr = VecGetArray(xvec,&x); CHKERRQ(ierr);
  ierr = mpintf_( &(user->number), &(user->mx), &(user->my), &user->ndim, 
         x, f, NULL, user->work, &(user->param), "F", 1 ); 
  if (ierr) SETERRQ(1,0,"Error in function evaluation");
  ierr = VecRestoreArray(xvec,&x); CHKERRQ(ierr);
  return 0;
}
/* -------------------------------------------------------------------- */
/*
    Evaluate gradient g(x) on a single processor
 */

int FormGradient(SNES snes,Vec xvec,Vec gvec,void *ptr)
{
  AppCtx *user = (AppCtx *) ptr;
  int    ierr;
  Scalar *x, *g;

  ierr = VecGetArray(xvec,&x); CHKERRQ(ierr);
  ierr = VecGetArray(gvec,&g); CHKERRQ(ierr);
  ierr = mpintf_( &(user->number), &(user->mx), &(user->my), &user->ndim, 
         x, NULL, g, user->work, &(user->param), "G", 1 ); 
  if (ierr) SETERRQ(1,0,"Error in gradient evaluation.");
  ierr = VecRestoreArray(xvec,&x); CHKERRQ(ierr);
  ierr = VecRestoreArray(gvec,&g); CHKERRQ(ierr);
  return 0;
}
/* -------------------------------------------------------------------- */
/*
    Form initial guess for nonlinear solver on a single processor
 */
int FormInitialGuess(AppCtx *user,Vec xvec)
{
  int    ierr;
  Scalar *x;

  ierr = VecGetArray(xvec,&x); CHKERRQ(ierr);
  ierr = mpintf_( &(user->number), &(user->mx), &(user->my), &user->ndim, 
         x, NULL, NULL, user->work, &(user->param), "XS", 2 ); 
  if (ierr) SETERRQ(1,0,"Error in initial guess evaluation.");
  ierr = VecRestoreArray(xvec,&x); CHKERRQ(ierr);
  return 0;
}
/* -------------------------------------------------------------------- */
/*
   Form Hessian matrix
 */
int FormHessian(SNES snes,Vec xvec,Mat *H,Mat *PrecH,MatStructure *flag,
                void *ptr)
{
  AppCtx     *user = (AppCtx *) ptr;
  Scalar     *w, *s, *y, *x;
  double     b = 10.0, param, zero = 0.0, one = 1.0;
  int        i, j, ierr, nx, ny, iparam, ndim;

  ierr = VecGetArray(xvec,&x); CHKERRQ(ierr);
  ierr = MatZeroEntries(*H); CHKERRQ(ierr);
  param = user->param;
  nx    = user->mx; ny = user->my; ndim = user->ndim;
  s     = user->s;
  y     = user->y;
  w     = user->work;
  if (user->number == 7) iparam = (int)param;

  for (j=0; j<ndim; j++) s[j] = zero;
  for (j=0; j<ndim; j++) {   /* loop over columns */
    s[j] = one;
    switch (user->number) {
      case 1:
        depths_( &nx, &ny, s, y );
        break;
      case 2:
        dpjbhs_( &nx, &ny, s, y, &param, &b );
        break;
      case 3:
        dmsabc_( &nx, &ny, w, &w[nx + 2], &w[2*nx + 4], 
                 &w[2*nx + ny + 6] );
        dmsahs_( &nx, &ny, x, s, y, w, &w[nx + 2], &w[2*nx + 4], 
                 &w[2*nx + ny + 6] );
        break;
      case 4:
        dodchs_( &nx, &ny, x, s, y, &param );
        break;
      case 5:
        dsschs_( &nx, &ny, x, s, y, &param ); 
        break;
      case 6:
         dgl1hs_( &ndim, x, s, y, &param );
         break;
      case 7:
         dgl2hs_( &nx, &ny, x, s, y, w, &w[4*(nx+1)*(ny+1)], &iparam ); 
         break;
    }
    s[j] = zero;
    for (i=0; i<ndim; i++) {
      if (y[i] != zero) {
        ierr = MatSetValues(*H,1,&i,1,&j,&y[i],ADD_VALUES);
        CHKERRQ(ierr);
      }
    }
  }
  ierr = MatAssemblyBegin(*H,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);
  ierr = MatAssemblyEnd(*H,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);

  return 0;
}
/* -------------------------------------------------------------------- */
int UserDestroyWork(AppCtx user)
{
  PetscFree(user.s); 
  PetscFree(user.y); 
  if (user.number == 3 || user.number == 7) PetscFree(user.work);
  return 0;
}
/* -------------------------------------------------------------------- */
int UserSetProblem(AppCtx *user,int *argc,char **argv)
{
  int      probnum = 1, ierr, flg;
  double   par;		/* test problem parameter */
  int      mx;		/* discretization of problem in x-direction */
  int      my;		/* discretization of problem in y-direction */

  /* Parse and check input arguments */
   ierr = OptionsGetInt(PETSC_NULL,"-p",&probnum,&flg); CHKERRQ(ierr);
   if (probnum > 7) SETERRQ(1,0,"Invalid problem number");;
   switch (probnum) {
      case 1: 
         par = 5.0;	mx = 10;	my = 10;	break; 
      case 2: 
	par = 1.0e-1;	mx = 10;	my = 10;	break;
      case 3: 
	par = 0.0;	mx = 10;	my = 10;	break;
      case 4: 
	par = 8.0e-3;	mx = 10;	my = 10;	break;
      case 5: 
	par = 1.0;	mx = 10;	my = 10;	break;
      case 6: 
	par = 5.0;	mx = 100;	my = 1;		break;
      case 7: 
	par = 2.0;	mx = 5;		my = 5;		break;
   }
   ierr = OptionsGetDouble(PETSC_NULL,"-par",&par,&flg); CHKERRQ(ierr);
   ierr = OptionsGetInt(PETSC_NULL,"-my",&my,&flg); CHKERRQ(ierr);
   ierr = OptionsGetInt(PETSC_NULL,"-mx",&mx,&flg); CHKERRQ(ierr);
   user->ndim = mx * my;
   if (probnum == 3) {
      user->work = (double *)PetscMalloc( 2*(mx+my+4) * sizeof(double) );
      CHKPTRQ(user->work);
   } else if (probnum == 6) {
      user->ndim = mx;
   } else if (probnum == 7) {
      user->ndim *= 4;
      user->work = (double *)PetscMalloc( 8*(mx+1)*(my+1) * sizeof(double) );
      CHKPTRQ(user->work);
      }
  /* Allocate work space for formation of Hessian */
  user->s = (double*)PetscMalloc(user->ndim*sizeof(double)); CHKPTRQ(user->s);
  user->y = (double*)PetscMalloc(user->ndim*sizeof(double)); CHKPTRQ(user->y);
  user->number = probnum;
  user->param = par;
  user->mx = mx;
  user->my = my;

  return 0;
}
#else
#include <stdio.h>
int main(int argc,char **args)
{
  fprintf(stdout,"This example does not work for complex numbers.\n");
  return 0;
}
#endif
