#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: vrmlops.c,v 1.46 1998/04/27 20:41:13 curfman Exp $";
#endif

/*
    Defines the operations for the VRML Draw implementation.

    Notes on VRML output.
    There are several variants.  By selecting a file name form,
    a separate VRML file is generated for each "page", and a 
    WWWInline is added for the item.

    For multicolored data, need IN THIS ORDER
    Coordinate3 { }
    Material { }
    MaterialBinding { }
    IndexedFaceSet { ...
        materialIndex [ 0, 1, 2, -1, ... ] }
    
    exactly parallel to the coordIndex

    Much of this operates on "meshes".  I really should have a mesh datatype
    with the information (like coordinate/field limits) that the
    graphics routines need.  There is also the need to define sub-arrays,
    as well as coarser sets.

    The APPARENT definition for a quad mesh (Inventor) is

    BaseColor { 
        rgb [ ... ] }
    MaterialBinding { value PER_VERTEX }
    QuadMesh { 
        vertexProperty VertexProperty { 
	vertex [ ... ] 
        normal [ ]
        texCoord [ ]
        orderedRGBA [  ]
        } 
	verticesPerColumn n
        verticesPerRow m
    }
    BaseColor is parallel to vertex entries.
*/

#include "src/draw/impls/vrml/vrmlimpl.h"


#undef __FUNC__  
#define __FUNC__ "Vi_Out_double_VRML"
/*
 * VRML loads more quickly if we output only at limited precision.
 * This trades write time for read time.
 *
 * digits is for eventual extension, with tests to add additional 
 * precision.  If set to > 10, just use %f
 */
void Vi_Out_double_VRML( FILE *fp, double v, int digits )
{
    double absv;
    char   buf[100], *p;
    int    is_fixed = 1;

    if (v == 0.0) {
	fputs( "0", fp );
	return;
    }
    if (v < 0.0) absv = -v; else absv = v;

    if (digits >= 10) {
	fprintf( fp, "%f", v );
	return;
    }

    /* We'd like to use just the exponent of absv, but that isn't easily 
       available.  Instead, we use a simple binary tree, expanded about 1.0
     */
    if (absv < 1.0) {
	if (absv < 0.01) {
	    sprintf( buf, "%.2e", v );
	    is_fixed = 0;
	}
	else if (absv < 0.1) {
	    /* 0.1 > absv >= 0.01 */
	    sprintf( buf, "%.4f", v );
	}
	else {
	    /* 1.0 > absv >= 0.1 */
	    sprintf( buf, "%.3f", v );
	}
	
    }
    else {
	/* absv > 1.0 */
	if (absv >= 100.0) {
	    sprintf( buf, "%.2e", v );
	    is_fixed = 0;
	}
	else if (absv >= 10.0) {
	    sprintf( buf, "%.1f", v );
	}
	else {
	    /* 1.0 <= absv < 10.0 */
	    sprintf( buf, "%.2f", v );
	}
    }

    if (is_fixed) {
	/* Remove any trailing 0's in fixed format only */
	p = buf + PetscStrlen( buf ) - 1;
	while (p > buf && *p == '0') *p-- = '\0';
	/* Remove trailing decimal point */
	if (*p == '.') *p-- = '\0';
	fputs( buf, fp );
    }
    else {
	/* Remove LEADING zeros after e and TRAILING before */
	char buf2[100];
	int  i = 0;
	p = buf;
	while (*p && *p != 'e') buf2[i++] = *p++;
	while (buf2[i-1] == '0' && i > 0) i--;
	buf2[i++] = *p++;
	if (*p == '-') buf2[i++] = *p++;
	while (*p && *p == '0') p++;
	while (*p) buf2[i++] = *p++;
	buf2[i++] = 0;
	fputs( buf2, fp );
    }
}

#undef __FUNC__  
#define __FUNC__ "ViFlushObj"
/* 
 * This routine flushes completed objects
 */
static int ViFlushObj( Draw_VRML *VWin, VRMLObj obj )
{
    int i;

    switch (obj) {
    case VRML_LINE:
	fprintf( VWin->curfp, "]\n }\nIndexLineSet { coordIndex [\n" );
	for (i=0; i<VWin->line_segs-1; i++) {
	    fprintf( VWin->curfp, "%d,", i );
	}
	fprintf( VWin->curfp, "%d ]\n}\n}\n", VWin->line_segs - 1 );
	VWin->line_segs = 0;
	break;
    case VRML_TRIANGLE:
	break;
    case VRML_POINT:
	break;
    case VRML_NONE:
	break;
    default:
	break;
    }
    VWin->last_obj = VRML_NONE;
    PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "ViStartLOD"
/* Generic code for adding simple LOD (level of detail) information
   to the VRML description.

   Input Parameters:
   center - center of field in [x,y,z] coordinates
   sizes - size of field in [x,y,z] coordinates
 */
static int ViStartLOD( Draw_VRML *VWin, double center[3], double sizes[3] )
{
    double far_size;
    int    i;

    far_size = sizes[0];
    for (i=1; i<3; i++) {
	if (sizes[i] > 0 && far_size < sizes[i]) far_size = sizes[i];
    }
    far_size *= VWin->LOD_thresh;
    fprintf( VWin->curfp, "LOD {\n    range [ %f ]\n", far_size );
    PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "ViEndLOD"
static int ViEndLOD( Draw_VRML *VWin, double center[3], double sizes[3] )
{
    fprintf( VWin->curfp, "\
Separator {\n\
   Translation { translation %f %f %f }\n\
   Material { diffuseColor %f %f %f }\n\
    Cube {\n\
         width %f\n\
         height %f\n\
         depth  %f\n\
         }\n  }\n }\n", 
	     center[0], center[1], center[2], 
	     VWin->cur_red, VWin->cur_green, VWin->cur_blue,
	     sizes[0], sizes[1], sizes[2] );
    PetscFunctionReturn(0);
}

/* These are approximate */
static double colorvals[16][3] = { 
    { 1.0, 1.0, 1.0 },    /* White */
    { 0.0, 0.0, 0.0 },    /* Black */
    { 1.0, 0.0, 0.0 },    /* Red */
    { 0.0, 1.0, 0.0 },    /* Green */
    { 0.0, 1.0, 1.0 },    /* Cyan */
    { 0.0, 0.0, 1.0 },    /* Blue */
    { 1.0, 0.0, 1.0 },    /* Magenta */
    { 0.5, 1.0, 0.8 },    /* Aquamarine */
    { 0.15,0.6, 0.15 },   /* Forestgreen */
    { 1.0, 0.5, 0.0 },    /* Orange */
    { 0.9, 0.5, 0.9 },    /* Violet */
    { 0.5, 0.3, 0.3 },    /* Brown */
    { 1.0, 0.6, 0.7 },    /* Pink */
    { 1.0, 0.4, 0.3 },    /* Coral */
    { 0.8, 0.8, 0.8 },    /* Gray */
    { 1.0, 1.0, 0.0 } };  /* Yellow */
/* Set the color values from the DRAW values */
static int ViSetColor( Draw_VRML *VWin, int color )
{
    if (color < 0 || color > 15) {
	SETERRQ(color,0,"Color value out of range ([0,15])");
    }

    VWin->cur_red   = colorvals[color][0];
    VWin->cur_green = colorvals[color][1];
    VWin->cur_blue  = colorvals[color][2];
    
    PetscFunctionReturn(0);
}

/* Other color routines are in vrmlcolor.c */

/* Set the camera position for a sequence of plots 
 *
 * Note that this sets a camera that wants to use a coordinate system
 * familar to people, not graphics weenies.  Unfortunately, some VRML
 * browsers (SGI's, particularly) can't handle this correctly; they want 
 * y to be UP.  I need an option for human or graphics coordinate
 * systems, and if graphics, the coordinate values need to be adjusted 
 * by replacing taking (x,y,z) -> (z,x,y).
 *
 * In VRML 2, the graphics weenie system is MANDATED (-y is DOWN, 
 * no exceptions).  It is still possible to change the orientation of a 
 * Viewpoint (no camera nodes), but you can't choose a human-oriented
 * coordinate system.  Because of this, I'll probably give up on the 
 * human-oriented system and just switch to the graphics-weenie system.
 */
#undef __FUNC__  
#define __FUNC__ "ViSetCamera"
static int ViSetCamera( Draw_VRML *VWin )
{
    /* The position should correspond with the BOP shifts. */
    if (VWin->is_inventor) PetscFunctionReturn(0);
    if (VWin->vrml_version == 1) {
	if (!VWin->is_human_coord) PetscFunctionReturn(0);
	fprintf( VWin->fp, "\
PerspectiveCamera {\n\
    position 0.5 -2.0 0.5\n\
    orientation 1 0 0 1.57079\n\
    focalDistance 5\n\
    heightAngle 0.785398\n\
    }\n" );
    }
    else if (VWin->vrml_version == 2) {
	/* No perspective camera in VRML 2 */
	fprintf( VWin->fp, "\
Viewpoint {\n\
    position 0.5 -2.0 0.5\n\
    fieldOfView 0.785398\n\
    }\n" );
    }
    PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DrawLine_VRML"
static int DrawLine_VRML(Draw Win, double xl, double yl, 
			 double xr, double yr,int cl)
{
  Draw_VRML* VWin = (Draw_VRML*) Win->data;

  if (VWin->last_obj != VRML_LINE) {
      if (VWin->last_obj != VRML_NONE) {
	  fprintf( VWin->curfp, "}\n" );
      }
      VWin->last_obj = VRML_LINE;
      fprintf( VWin->curfp, "Separator {\nMaterial { diffuseColor %f %f %f }\nCoordinate3 {\n point [\n",
	       VWin->cur_red, VWin->cur_green, VWin->cur_blue );
  }
  else {
      if (xl != VWin->last_x || yl != VWin->last_y) {
	  fprintf( VWin->curfp, ",%f %f %f\n", xl, yl, VWin->z );
	  VWin->line_segs ++;
      }
  }
  putc( ',', VWin->curfp );
  Vi_Out_double_VRML( VWin->curfp, xr, PRECISION );
  putc( ' ', VWin->curfp );
  Vi_Out_double_VRML( VWin->curfp, yr, PRECISION );
  putc( ' ', VWin->curfp );
  Vi_Out_double_VRML( VWin->curfp, VWin->z, PRECISION );
  putc( '\n', VWin->curfp );
  VWin->line_segs ++;
  VWin->last_x = xr;
  VWin->last_y = yr;

  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DrawPoint_VRML"
static int DrawPoint_VRML(Draw Win,double x,double  y,int c)
{
#ifdef FOO    
  int     xx,yy;
  Draw_VRML* VWin = (Draw_VRML*) Win->data;

  xx = XTRANS(Win,VWin,x);  yy = YTRANS(Win,VWin,y);
  ViSetColor( VWin, c );
  XDrawPoint( VWin->disp, XiDrawable(VWin), VWin->gc.set,xx, yy);
#endif
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DrawRectangle_VRML"
static int DrawRectangle_VRML(Draw Win, 
			      double xl, double yl, double xr, double yr,
			      int c1, int c2,int c3,int c4)
{

#ifdef FOO
  Draw_VRML* VWin = (Draw_VRML*) Win->data;
  int     x1,y1,w,h, c = (c1 + c2 + c3 + c4)/4;
  ViSetColor( VWin, c );
  x1 = XTRANS(Win,VWin,xl);   w  = XTRANS(Win,VWin,xr) - x1; 
  y1 = YTRANS(Win,VWin,yr);   h  = YTRANS(Win,VWin,yl) - y1;
  if (w <= 0) w = 1; if (h <= 0) h = 1;
  XFillRectangle( VWin->disp, XiDrawable(VWin), VWin->gc.set, x1, y1, w, h);
#endif
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DrawTriangle_VRML"
static int DrawTriangle_VRML(Draw Win, double X1, double Y1, double X2, 
                          double Y2,double X3,double Y3, int c1, int c2,int c3)
{
/*   Draw_VRML* VWin = (Draw_VRML*) Win->data; */

  /* Need code similar to line drawing; note that if drawing CONNECTED
     triangles, this is NOT the routine that we want to use (duplicates
     the vertices) */

  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "ViOutMeshLOD"
static void ViOutMeshLOD( Draw_VRML *VWin, DrawMesh mesh, int use_z,
			  double center[3], double sizes[3] )
{
    /* Get the max size of the box holding the data, and
       an estimate of the "far away" limit */
    center[0] = (mesh->xmax + mesh->xmin)/2.0;
    center[1] = (mesh->ymax + mesh->ymin)/2.0;
    sizes[0]  = mesh->xmax - mesh->xmin;
    sizes[1]  = mesh->ymax - mesh->ymin;
    if ( use_z ) {
	center[2] = (mesh->zmax+mesh->zmin)/2.0;
	sizes[2]  = mesh->zmax - mesh->zmin;
    }
    else {
	center[2] = (mesh->fmax+mesh->fmin)/2.0;
	sizes[2]  = mesh->fmax - mesh->fmin;
    }
    ViStartLOD( VWin, center, sizes );
}
/*
 * This draws (x(i),y(j),f(i+j*nx))
 */
#undef __FUNC__  
#define __FUNC__ "ViDrawTensorSurface_VRML"
int ViDrawTensorSurface_VRML( Draw Win, DrawMesh mesh, int color )
{
    Draw_VRML* VWin = (Draw_VRML*) Win->data;
    int        i, j, nx, ny;
    double     *x, *y, *f, center[3], sizes[3];
    PetscObject   vobj = (PetscObject) Win;

    if (vobj->cookie == DRAW_COOKIE && vobj->type == DRAW_NULLWINDOW) PetscFunctionReturn(0);

    /* Flush any previous object */
    ViFlushObj( VWin, VWin->last_obj );

    /* Set color */
    ViSetColor( VWin, color );

    /* Get parameters */
    nx = mesh->nx;
    ny = mesh->ny;
    x  = mesh->x;
    y  = mesh->y;
    f  = mesh->f;

    /* If we are using LOD AND fname_form, will this be in the BOP part? */
    if (VWin->use_LOD) {
	ViOutMeshLOD( VWin, mesh, 0, center, sizes );
    }
    
    /* Generate a new object, with examples of additional fields */
    fprintf( VWin->curfp, "\
Separator {\n\
    ShapeHints { vertexOrdering COUNTERCLOCKWISE\n\
                 shapeType      UNKNOWN_SHAPE_TYPE\n\
                 faceType       CONVEX\n\
                 creaseAngle    1\n\
               }\n\
#   Rotation { 1 0 0 -0.78 }\n\
    Material { diffuseColor %f %f %f\n\
#               specularColor 0.8 0.2 0.25\n\
#               emissiveColor 0 0 0\n\
#               ambientColor 0.25 0.22 0\n\
#               shininess    1\n\
              }\n\
    Coordinate3 {\n        point [\n", 
	    VWin->cur_red, VWin->cur_green, VWin->cur_blue );

    /* Output the vertices */
    for (j=0; j<ny; j++) {
	for (i=0; i<nx; i++) {
	    if (VWin->is_human_coord) {
		Vi_Out_double_VRML( VWin->curfp, x[i], PRECISION );
		putc( ' ', VWin->curfp );
		Vi_Out_double_VRML( VWin->curfp, y[j], PRECISION );
		putc( ' ', VWin->curfp );
		Vi_Out_double_VRML( VWin->curfp, *f++, PRECISION );
		putc( ',', VWin->curfp );
		putc( '\n', VWin->curfp );
	    }
	    else {
		Vi_Out_double_VRML( VWin->curfp, y[j], PRECISION );
		putc( ' ', VWin->curfp );
		Vi_Out_double_VRML( VWin->curfp, *f++, PRECISION );
		putc( ' ', VWin->curfp );
		Vi_Out_double_VRML( VWin->curfp, x[i], PRECISION );
		putc( ',', VWin->curfp );
		putc( '\n', VWin->curfp );
	    }
	}
    }
    /* Note that trailing commas are allowed (thank heavens!) */
    fprintf( VWin->curfp, "    ]\n   }\n" );

    /* Output the edges */
    fprintf( VWin->curfp, "IndexedFaceSet {\n    coordIndex [ \n" );
    for (j=0; j<ny-1; j++) {
	for (i=0; i<nx-1; i++) {
	    fprintf( VWin->curfp, "%d,%d,%d,%d,-1,\n", 
		     j*nx+i,j*nx+i+1,(j+1)*nx+i+1,(j+1)*nx+i );
	}
    }
    fprintf( VWin->curfp, "]\n}\n" );

    /* Finish the object */
    fprintf( VWin->curfp, "    }\n" );

    if (VWin->use_LOD) {
	ViEndLOD( VWin, center, sizes );
    }
    
    PetscFunctionReturn(0);
}

/*
 * This draws (x(i),y(j),z(i+j*nx)) with contours
 */
#undef __FUNC__  
#define __FUNC__ "ViDrawTensorSurfaceContour_VRML"
int ViDrawTensorSurfaceContour_VRML( Draw Win, DrawMesh mesh, 
				   VRMLGetHue_fcn GetColor, 
				   void *color_context, int ncolor )
{
    Draw_VRML*  VWin = (Draw_VRML*) Win->data;
    int         i, j, nx, ny;
    double      *x, *y, *f, center[3], sizes[3];
    double      *fval, red, green, blue;
    PetscObject vobj = (PetscObject) Win;

    if (vobj->cookie == DRAW_COOKIE && vobj->type == DRAW_NULLWINDOW) PetscFunctionReturn(0);

    /* Flush any previous object */
    ViFlushObj( VWin, VWin->last_obj );

    /* Get parameters */
    nx = mesh->nx;
    ny = mesh->ny;
    x  = mesh->x;
    y  = mesh->y;
    f  = mesh->f;

    /* If we are using LOD AND fname_form, will this be in the BOP part? */
    if (VWin->use_LOD) {
	ViOutMeshLOD( VWin, mesh, 0, center, sizes );
    }
    
    /* Generate a new object, with examples of additional fields */
    if (VWin->is_inventor) {
	PetscFunctionReturn(0);
    }

    fprintf( VWin->curfp, "\
Separator {\n\
    ShapeHints { vertexOrdering COUNTERCLOCKWISE\n\
                 shapeType      UNKNOWN_SHAPE_TYPE\n\
                 faceType       CONVEX\n\
                 creaseAngle    1\n\
               }\n\
    Coordinate3 {\n        point [\n" );

    /* Output the vertices */
    fval = f;
    for (j=0; j<ny; j++) {
	for (i=0; i<nx; i++) {
	    if (VWin->is_human_coord) {
		Vi_Out_double_VRML( VWin->curfp, x[i], PRECISION );
		putc( ' ', VWin->curfp );
		Vi_Out_double_VRML( VWin->curfp, y[j], PRECISION );
		putc( ' ', VWin->curfp );
		Vi_Out_double_VRML( VWin->curfp, *fval++, PRECISION );
		putc( ',', VWin->curfp );
		putc( '\n', VWin->curfp );
	    }
	    else {
		Vi_Out_double_VRML( VWin->curfp, y[j], PRECISION );
		putc( ' ', VWin->curfp );
		Vi_Out_double_VRML( VWin->curfp, *fval++, PRECISION );
		putc( ' ', VWin->curfp );
		Vi_Out_double_VRML( VWin->curfp, x[i], PRECISION );
		putc( ',', VWin->curfp );
		putc( '\n', VWin->curfp );
	    }
	}
    }
    /* Note that trailing commas are allowed (thank heavens!) */
    fprintf( VWin->curfp, "    ]\n   }\n" );

    /* Output the vertex colors */
    fval = f;
    fprintf( VWin->curfp, "Material {\n diffuseColor [ " );
    for (j=0; j<ny; j++) {
	for (i=0; i<nx; i++) {
	    (*GetColor)( *fval++, color_context, ncolor, &red, &green, &blue );
	    putc( ' ', VWin->curfp );
	    Vi_Out_double_VRML( VWin->curfp, red, PRECISION );
	    putc( ' ', VWin->curfp );
	    Vi_Out_double_VRML( VWin->curfp, green, PRECISION );
	    putc( ' ', VWin->curfp );
	    Vi_Out_double_VRML( VWin->curfp, blue, PRECISION );
	    putc( ',', VWin->curfp );
	    putc( '\n', VWin->curfp );
	}
    }
    fprintf( VWin->curfp, "  ]\n }\nMaterialBinding {\n\
value PER_VERTEX_INDEXED\n}\n" );

    /* Output the edges */
    fprintf( VWin->curfp, "IndexedFaceSet {\n    coordIndex [ \n" );
    for (j=0; j<ny-1; j++) {
	for (i=0; i<nx-1; i++) {
	    fprintf( VWin->curfp, "%d,%d,%d,%d,-1,\n", 
		     j*nx+i,j*nx+i+1,(j+1)*nx+i+1,(j+1)*nx+i );
	}
    }
    fprintf( VWin->curfp, "]\n    materialIndex [ \n" );
    for (j=0; j<ny-1; j++) {
	for (i=0; i<nx-1; i++) {
	    fprintf( VWin->curfp, "%d,%d,%d,%d,-1,\n", 
		     j*nx+i,j*nx+i+1,(j+1)*nx+i+1,(j+1)*nx+i );
	}
    }
    fprintf( VWin->curfp, "]\n}\n}\n" );

    if (VWin->use_LOD) {
	ViEndLOD( VWin, center, sizes );
    }
    
    PetscFunctionReturn(0);
}

/* Compute the mesh parameters for a section */
#undef __FUNC__  
#define __FUNC__ "ViSetupMeshVertices"
int ViSetupMeshVertices( Draw_VRML *VWin, DrawMesh mesh, int coord_dim, 
		       int coord_slice )
{
    int         is, ie, js, je, ks, ke, i_incr, j_incr, k_incr;

    i_incr = mesh->i_incr;
    j_incr = mesh->j_incr;
    k_incr = mesh->k_incr;

    switch (coord_dim) {
    case 0:
	/* constant i */
	is = ie = coord_slice;
	js = mesh->sj; je = mesh->ej;
	ks = mesh->sk; ke = mesh->ek;
        mesh->act_nx = (je-js+1) / j_incr;
        mesh->act_ny = (ke-ks+1) / k_incr;
	break;
    case 1:
	/* constant j */
	js = je = coord_slice;
	is = mesh->si; ie = mesh->ei;
	ks = mesh->sk; ke = mesh->ek;
        mesh->act_nx = (ie-is+1) / i_incr;
        mesh->act_ny = (ke-ks+1) / k_incr;
        break;
    case 2:
	/* constant k */
	ks = ke = coord_slice;
	is = mesh->si; ie = mesh->ei;
	js = mesh->sj; je = mesh->ej;
        mesh->act_nx = (ie-is+1) / i_incr;
        mesh->act_ny = (je-js+1) / j_incr;
	break;
    default:
	/* Invalid slice */
	return 1;
    }

    mesh->is = is;
    mesh->ie = ie;
    mesh->js = js;
    mesh->je = je;
    mesh->ks = ks;
    mesh->ke = ke;
    PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "ViOutMeshVertices"
int ViOutMeshVertices( Draw_VRML *VWin, DrawMesh mesh, int coord_dim, 
		       int coord_slice, 
		       double ox, double oy, double oz, int just_coords )
{
    int         i, j, k, nx, ny;
    int         is, ie, js, je, ks, ke, idx, i_incr, j_incr, k_incr;
    double      *x, *y, *z;

    if (!just_coords) 
	fprintf( VWin->curfp, "    Coordinate3 {\n        point [\n" );

    /* Output the vertices */
    ViSetupMeshVertices( VWin, mesh, coord_dim, coord_slice );

    is	   = mesh->is;
    ie	   = mesh->ie;
    i_incr = mesh->i_incr;
    js	   = mesh->js;
    je	   = mesh->je;
    j_incr = mesh->j_incr;
    ks	   = mesh->ks;
    ke	   = mesh->ke;
    k_incr = mesh->k_incr;
    nx     = mesh->nx;
    ny     = mesh->ny;
    x      = mesh->x;
    y      = mesh->y;
    z      = mesh->z;
    for (k=ks; k<=ke; k += k_incr) {
	for (j=js; j<=je; j += j_incr) {
	    for (i=is; i<=ie; i += i_incr) {
		idx = i + nx * (j + ny * k);
		if (VWin->is_human_coord) {
		    Vi_Out_double_VRML( VWin->curfp, x[idx] + ox, PRECISION );
		    putc( ' ', VWin->curfp );
		    Vi_Out_double_VRML( VWin->curfp, y[idx] + oy, PRECISION );
		    putc( ' ', VWin->curfp );
		    Vi_Out_double_VRML( VWin->curfp, z[idx] + oz, PRECISION );
		    putc( ',', VWin->curfp );
		    putc( '\n', VWin->curfp );
		}
		else {
		    Vi_Out_double_VRML( VWin->curfp, y[idx] + oy, PRECISION );
		    putc( ' ', VWin->curfp );
		    Vi_Out_double_VRML( VWin->curfp, z[idx] + oz, PRECISION );
		    putc( ',', VWin->curfp );
		    Vi_Out_double_VRML( VWin->curfp, x[idx] + ox, PRECISION );
		    putc( ' ', VWin->curfp );
		    putc( '\n', VWin->curfp );
		}
	    }
	}
    }
    /* Note that trailing commas are allowed (thank heavens!) */
    if (!just_coords) 
	fprintf( VWin->curfp, "    ]\n   }\n" );
    
    PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "ViOutMeshVertexIndices"
static int ViOutMeshVertexIndices( Draw_VRML *VWin, DrawMesh mesh, 
				   int do_materialindex )
{
    int i, j, act_nx, act_ny;

    act_nx = mesh->act_nx;
    act_ny = mesh->act_ny;

    /* This is relative to the slice generated, so i and j here refer
       not to the original i,j, but to the values chosen for output */
    /* Output the edges */
    if (do_materialindex) 
	fputs( "IndexedFaceSet", VWin->curfp );
    else
	fputs( "IndexedLineSet", VWin->curfp );

    fputs( "{\n    coordIndex [ \n", VWin->curfp );
    for (j=0; j<act_ny-1; j++) {
	for (i=0; i<act_nx-1; i++) {
	    fprintf( VWin->curfp, "%d,%d,%d,%d,-1,\n", 
		     j*act_nx+i,j*act_nx+i+1,(j+1)*act_nx+i+1,(j+1)*act_nx+i );
	}
    }
    fprintf( VWin->curfp, "]\n" );
    if (do_materialindex) {
	fprintf( VWin->curfp, "    materialIndex [ \n" );
	for (j=0; j<act_ny-1; j++) {
	    for (i=0; i<act_nx-1; i++) {
		fprintf( VWin->curfp, "%d,%d,%d,%d,-1,\n", 
		     j*act_nx+i,j*act_nx+i+1,(j+1)*act_nx+i+1,(j+1)*act_nx+i );
	    }
	}
	fputs( "]\n", VWin->curfp );
    }
    fputs( "}\n", VWin->curfp );

    PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "ViDrawTensorMapSurfaceContour_VRML"
/*
 * This draws slices of (x(i,j,k),y(i,j,k),z(i,j,k)) with contours for 
 * value f(i,j,k).
 * coord_dim (0,1,2) selects which coord is held constant with value
 * coord_slice.
 * Coordinates are offset by (ox,oy,oz)
 * transparency: level of transparency (0<=transparency<1, with 1 being most transparent)
 * This draws only the subset between indices i1-i2, j1-j2, k1-k2.
 */
int ViDrawTensorMapSurfaceContour_VRML( Draw Win, DrawMesh mesh, 
				      double ox, double oy, double oz, 
				      int coord_slice, int coord_dim,
				      VRMLGetHue_fcn GetColor, 
				      void *color_context, int ncolor,
				      double transparency )
{
    Draw_VRML*  VWin = (Draw_VRML*) Win->data;
    int         i, j, k, nx, ny;
    int         is, ie, js, je, ks, ke, idx, i_incr, j_incr, k_incr;
    double      center[3], sizes[3];
    double      red, green, blue;
    double      *f;
    PetscObject vobj = (PetscObject) Win;

    if (vobj->cookie == DRAW_COOKIE && vobj->type == DRAW_NULLWINDOW) PetscFunctionReturn(0);
    if (transparency < 0.0 || transparency >= 1.0) transparency = 0.5;

    /* Flush any previous object */
    ViFlushObj( VWin, VWin->last_obj );

    /* If we are using LOD AND fname_form, will this be in the BOP part? */
    if (VWin->use_LOD) {
	ViOutMeshLOD( VWin, mesh, 1, center, sizes );
    }
    
    /* Generate a new object, with examples of additional fields */
    if (VWin->vrml_version == 1) {
	fprintf( VWin->curfp, "\
Separator {\n\
    ShapeHints { vertexOrdering COUNTERCLOCKWISE\n\
                 shapeType      UNKNOWN_SHAPE_TYPE\n\
                 faceType       CONVEX\n\
                 creaseAngle    1\n\
               }\n" );
    }
    else {
	fprintf( VWin->curfp, "\
Group {\n\
    ShapeHints { vertexOrdering COUNTERCLOCKWISE\n\
                 shapeType      UNKNOWN_SHAPE_TYPE\n\
                 faceType       CONVEX\n\
                 creaseAngle    1\n\
               }\n" );
    }

    if (ViOutMeshVertices( VWin, mesh, coord_dim, coord_slice, ox, oy, oz, 
			   0 )) 
	return 1;

    /* Output the vertex colors */
    f	   = mesh->f;
    is	   = mesh->is;
    ie	   = mesh->ie;
    i_incr = mesh->i_incr;
    js	   = mesh->js;
    je	   = mesh->je;
    j_incr = mesh->j_incr;
    ks	   = mesh->ks;
    ke	   = mesh->ke;
    k_incr = mesh->k_incr;
    nx     = mesh->nx;
    ny     = mesh->ny;

    fprintf( VWin->curfp, "Material {\n transparency %f\n diffuseColor [ ",transparency );
    for (k=ks; k<=ke; k += k_incr) {
	for (j=js; j<=je; j += j_incr) {
	    for (i=is; i<=ie; i += i_incr) {
		idx = i + nx * (j + ny * k);
		/* color context has fmin, fmax in it */
		(*GetColor)( f[idx], color_context, ncolor, 
			     &red, &green, &blue );
		putc( ' ', VWin->curfp );
		Vi_Out_double_VRML( VWin->curfp, red, PRECISION );
		putc( ' ', VWin->curfp );
		Vi_Out_double_VRML( VWin->curfp, green, PRECISION );
		putc( ' ', VWin->curfp );
		Vi_Out_double_VRML( VWin->curfp, blue, PRECISION );
		putc( ',', VWin->curfp );
		putc( '\n', VWin->curfp );
	    }
	}
    }
    fprintf( VWin->curfp, "  ]\n }\nMaterialBinding {\n\
value PER_VERTEX_INDEXED\n}\n" );

    ViOutMeshVertexIndices( VWin, mesh, 1 );

    fprintf( VWin->curfp, "}\n" );

    if (VWin->use_LOD) {
	ViEndLOD( VWin, center, sizes );
    }
    
    PetscFunctionReturn(0);
}

/* ------------------------------------------------------------------------ */
#undef __FUNC__  
#define __FUNC__ "ViDrawTensorMapMesh_VRML"
/* This version draws a mesh (no contours) */
/*
 * This draws slices of (x(i,j,k),y(i,j,k),z(i,j,k)).
 * coord_dim (0,1,2) selects which coord is held constant with value
 * coord_slice.
 * Coordinates are offset by (ox,oy,oz)
 */
int ViDrawTensorMapMesh_VRML( Draw Win, DrawMesh mesh, 
				      double ox, double oy, double oz, 
				      int coord_slice, int coord_dim )
{
    Draw_VRML*  VWin = (Draw_VRML*) Win->data;
    double      center[3], sizes[3];
    PetscObject vobj = (PetscObject) Win;

    if (vobj->cookie == DRAW_COOKIE && vobj->type == DRAW_NULLWINDOW) PetscFunctionReturn(0);

    /* Flush any previous object */
    ViFlushObj( VWin, VWin->last_obj );

    /* If we are using LOD AND fname_form, will this be in the BOP part? */
    if (VWin->use_LOD) {
	ViOutMeshLOD( VWin, mesh, 1, center, sizes );
    }
    
    /* Generate a new object, with examples of additional fields */
    fprintf( VWin->curfp, "Separator {\n" );

    if (ViOutMeshVertices( VWin, mesh, coord_dim, coord_slice, ox, oy, oz, 
			   0 )) 
	return 1;

    /* Output the vertex color (set to red for now) */
    fprintf( VWin->curfp, "Material {\n diffuseColor 1.0 0.0 0.0\n }\nMaterialBinding {\n\
value OVERALL\n}\n" );

    /* Output the edges */
    ViOutMeshVertexIndices( VWin, mesh, 0 );

    fprintf( VWin->curfp, "}\n" );

    if (VWin->use_LOD) {
	ViEndLOD( VWin, center, sizes );
    }

    PetscFunctionReturn(0);
}
/* ------------------------------------------------------------------------ */

#undef __FUNC__  
#define __FUNC__ "DrawString_VRML"
static int DrawString_VRML(Draw Win,double x,double  y,int c,char *chrs )
{
#ifdef FOO
  int     xx,yy;
  Draw_VRML* VWin = (Draw_VRML*) Win->data;
  xx = XTRANS(Win,VWin,x);  yy = YTRANS(Win,VWin,y);
  ViSetColor( VWin, c );
  XDrawString( VWin->disp, XiDrawable(VWin), VWin->gc.set,
               xx, yy - VWin->font->font_descent, chrs, PetscStrlen(chrs) );
#endif
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DrawStringGetSize_VRML"
int DrawStringGetSize_VRML(Draw Win,double *x,double  *y)
{

/*
  Draw_VRML* VWin = (Draw_VRML*) Win->data;
  double  w,h;
  w = VWin->font->font_w; h = VWin->font->font_h;
  *x = w*(Win->coor_xr - Win->coor_xl)/(VWin->w)*(Win->port_xr - Win->port_xl);
  *y = h*(Win->coor_yr - Win->coor_yl)/(VWin->h)*(Win->port_yr - Win->port_yl);
  */  
PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DrawStringVertical_VRML"
int DrawStringVertical_VRML(Draw Win,double x,double  y,int c,char *chrs )
{
#ifdef FOO  
  int     xx,yy,n = PetscStrlen(chrs),i;
  Draw_VRML* VWin = (Draw_VRML*) Win->data;
  char    tmp[2];
  double  tw,th;
  tmp[1] = 0;
  ViSetColor( VWin, c );
  DrawStringGetSize_VRML(Win,&tw,&th);
  xx = XTRANS(Win,VWin,x);
  for ( i=0; i<n; i++ ) {
    tmp[0] = chrs[i];
    yy = YTRANS(Win,VWin,y-th*i);
    XDrawString( VWin->disp, XiDrawable(VWin), VWin->gc.set,
                xx, yy - VWin->font->font_descent, tmp, 1 );
  }
#endif
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DrawFlush_VRML" 
static int DrawFlush_VRML(Draw Win )
{
  Draw_VRML* VWin = (Draw_VRML*) Win->data;
  ViFlushObj( VWin, VWin->last_obj );
  fflush( VWin->curfp );
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DrawSynchronizedFlush_VRML"
static int DrawSynchronizedFlush_VRML(Draw Win )
{
  Draw_VRML* VWin = (Draw_VRML*) Win->data;

  ViFlushObj( VWin, VWin->last_obj );
  fflush( VWin->curfp );
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DrawClear_VRML"
static int DrawClear_VRML(Draw Win)
{
#ifdef FOO
  Draw_VRML*  VWin = (Draw_VRML*) Win->data;
/* This requires VRML animation support */
#endif
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DrawSynchronizedClear_VRML"
static int DrawSynchronizedClear_VRML(Draw Win)
{
  int     rank;
  Draw_VRML* VWin = (Draw_VRML*) Win->data;

  MPI_Barrier(Win->comm);
  MPI_Comm_rank(Win->comm,&rank);
  if (!rank) {
    DrawClear_VRML(Win);
    fflush( VWin->curfp );
  }
  MPI_Barrier(Win->comm);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DrawPause_VRML" 
static int DrawPause_VRML(Draw draw)
{
  if (draw->pause > 0) PetscSleep(draw->pause);
  else if (draw->pause < 0) {
      /* Need to get input from VRML */
      /*
    if (button == BUTTON_RIGHT) SETERRQ(1,0,"User request exit");
    if (button == BUTTON_CENTER) draw->pause = 0;
    */
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DrawBOP_VRML" 
static int DrawBOP_VRML( Draw Win )
{
  Draw_VRML* VWin = (Draw_VRML*) Win->data;

  if (VWin->pagenum == 0) {
      /* Special processing for first page: place camera */
      ViSetCamera( VWin );
  }
  if (VWin->fname_form) {
      char fname[1024];
      sprintf( fname, VWin->fname_form, VWin->pagenum );
      VWin->curfp = fopen( fname, "w" );
      if (!VWin->curfp) {
	  SETERRQ(PETSC_ERR_FILE_OPEN,0,"Could not open file" );
      }
      if (VWin->is_inventor) {
	  fprintf( VWin->curfp, "#Inventor V2.1 ascii\n# Page %d\n#\n", 
		   VWin->pagenum );
      }
      else {
	  if (VWin->vrml_version == 1) {
	      fprintf( VWin->curfp, "#VRML V1.0 ascii\n#\n# PETSc DrawVRML\n#\
Page %d\n#\n", VWin->pagenum );
	  }
	  else {
	      fprintf( VWin->curfp, "#VRML V2.0 ascii\n#\n# PETSc DrawVRML\n#\
Page %d\n#\n", VWin->pagenum );
	  }
      }

      /* Generate the reference in the master file */
      sprintf( fname, VWin->wwwname_form, VWin->pagenum );
      fprintf( VWin->fp, "Separator {\n\
  Translation { translation %f %f %f }\n\
  WWWInline { name \"%s\"\n }\n", 
	       0.0, VWin->pagenum * 5.0, 0.0, fname );
      VWin->pagenum++;
  }
  else {
      /* Just translate the page */
      fprintf( VWin->fp, "Separator {\n\
  Translation { translation %f %f %f }\n", 
	       0.0, VWin->pagenum * 5.0 , 0.0 );
      VWin->pagenum++;
  }

  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DrawEOP_VRML"
static int DrawEOP_VRML( Draw Win )
{
  Draw_VRML* VWin = (Draw_VRML*) Win->data;

  if (VWin->fname_form) {
      /* End the reference to this page in the master file */
      fprintf( VWin->fp, "}\n" );
      /* Close out pages file */
      fclose( VWin->curfp );
      VWin->curfp = VWin->fp;
  }
  else {
      /* Just end the translation */
      fprintf( VWin->fp, "}\n" );
  }

  PetscFunctionReturn(0);
}

static struct _DrawOps DvOps = { 0,
                                 DrawFlush_VRML,DrawLine_VRML,0,0,DrawPoint_VRML,0,
                                 DrawString_VRML,DrawStringVertical_VRML,
                                 0,0,
                                 0,DrawClear_VRML,
                                 DrawSynchronizedFlush_VRML,
                                 DrawRectangle_VRML,
                                 DrawTriangle_VRML,
                                 0,
                                 DrawPause_VRML,
                                 DrawSynchronizedClear_VRML,
                                 DrawBOP_VRML, DrawEOP_VRML};

#undef __FUNC__  
#define __FUNC__ "DrawDestroy_VRML"
int DrawDestroy_VRML(Draw ctx)
{
  Draw_VRML *win = (Draw_VRML *) ctx->data;

/*   PetscFree(win->font); */
  PetscFree(win);
  PLogObjectDestroy(ctx);
  PetscHeaderDestroy(ctx);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DrawOpenVRML"
/*@C
   DrawOpenVRML - Opens an VRML viewer for use with the Draw routines.

   Input Parameters:
+  comm - the communicator that will share the viewer
.  fname - the file on which to open, or null for stdout
-  title - the title to put in the title bar

   Output Parameter:
.  ctx - the drawing context.

   Options Database Keys:
+  -novrml - Disables all output
.  -vrmlfile <name> - Specifies name of VRML output file.  Overrides the parameter fname
.  -vrmlnolod - Does not use LOD (level of detail nodes)
.  -vrmllodthresh <value> - Set distance for LOD thresholding
.  -vrmlwww <name> - Generates separate WWW files for each page.  <name> is
    the name of the URL; it MUST contain a %d to indicate the page number.
.  -vrmlwwwform <name> - Specifies the string that names the files used
   by -vrmlwww ; it must contain a %d to indicate the page number.
.  -vrmliv - Uses Inventor form instead of VRML
.  -vrmlgcoord - Uses graphics (z in, y down) rather than human (z up, y in)
   coordinate system.  This is automatic for VRML version 2. 
-  -vrml2 - Uses VRML version 2 (not yet supported)

   Note:
   When finished with the drawing context, it should be destroyed
   with DrawDestroy().

.keywords: draw, open, VRML

.seealso: DrawSynchronizedFlush(), DrawDestroy()
@*/
int DrawOpenVRML(MPI_Comm comm,char* fname,char *title, Draw* inctx)
{
  Draw      ctx;
  Draw_VRML *VWin;
  int       ierr,size,rank,flg;
  char      string[1024];

  ierr = OptionsHasName(PETSC_NULL,"-novrml",&flg); CHKERRQ(ierr);
  if (flg) {
    return DrawOpenNull(comm,inctx);
  }

  *inctx = 0;
  PetscHeaderCreate(ctx,_p_Draw,struct _DrawOps,DRAW_COOKIE,DRAW_VRML,comm,DrawDestroy,0); 
  /*    {ctx = (struct _p_Draw *) malloc( sizeof(struct _p_Draw) );
       CHKPTRQ((ctx));
       PetscMemzero(ctx,sizeof(struct _p_Draw));
       (ctx)->cookie = DRAW_COOKIE;
       (ctx)->type   = DRAW_VRML;
       (ctx)->prefix = 0;
       PetscCommDup_Private(comm,&(ctx)->comm,&(ctx)->tag);} */

  PLogObjectCreate(ctx);
  PetscMemcpy(ctx->ops,&DvOps,sizeof(DvOps));
  ctx->ops->destroy = DrawDestroy_VRML;
  ctx->ops->view    = 0;
  ctx->pause   = 0;
  ctx->coor_xl = 0.0;  ctx->coor_xr = 1.0;
  ctx->coor_yl = 0.0;  ctx->coor_yr = 1.0;
  ctx->port_xl = 0.0;  ctx->port_xr = 1.0;
  ctx->port_yl = 0.0;  ctx->port_yr = 1.0;

  ierr = OptionsGetInt(PETSC_NULL,"-draw_pause",&ctx->pause,&flg);CHKERRQ(ierr);

  /* actually create and open the window */
  VWin         = (Draw_VRML *) PetscMalloc(sizeof(Draw_VRML));CHKPTRQ(VWin);
  PLogObjectMemory(ctx,sizeof(Draw_VRML)+sizeof(struct _p_Draw));
  PetscMemzero(VWin,sizeof(Draw_VRML));
  MPI_Comm_size(comm,&size);
  MPI_Comm_rank(comm,&rank);
  VWin->z	     = 0.0;
  VWin->last_obj     = VRML_NONE;
  VWin->cur_red	     = 1;
  VWin->cur_green    = 1;
  VWin->cur_blue     = 1;
  VWin->line_segs    = 0;
  VWin->fname_form   = 0;
  VWin->wwwname_form = 0;
  VWin->use_LOD	     = 1;
  VWin->pagenum      = 0;
  VWin->LOD_thresh   = 5.0;
  VWin->vrml_version = 1;
  VWin->is_inventor  = 0;
  VWin->is_human_coord = 1;

  /* Define the extension routines */
  VWin->DrawTensorSurface	    = ViDrawTensorSurface_VRML;
  VWin->DrawTensorSurfaceContour    = ViDrawTensorSurfaceContour_VRML;
  VWin->DrawTensorMapSurfaceContour = ViDrawTensorMapSurfaceContour_VRML;
  VWin->DrawTensorMapMesh	    = ViDrawTensorMapMesh_VRML;

  if (rank == 0) {
      /* Output file */
    ierr = OptionsGetString(PETSC_NULL,"-vrmlfile",string,1024,&flg); CHKERRQ(ierr);
    if (flg) {
      fname = string;
    }
    if (!fname)
	VWin->fp = stdout;
    else
	VWin->fp = fopen( fname, "w" );
    if (!VWin->fp) {
	SETERRQ(PETSC_ERR_FILE_OPEN,0,"Could not open file" );
    }

    /* LOD */
    ierr = OptionsHasName( PETSC_NULL, "-vrmlnolod",&flg );CHKERRQ(ierr);
    if (flg) VWin->use_LOD = 0;
    else {
	ierr = OptionsGetDouble( PETSC_NULL, "-vrmllodthresh",
			     &VWin->LOD_thresh,&flg);CHKERRQ(ierr);
	if (flg) VWin->use_LOD = 1;
    }

    /* Inventor */
    ierr = OptionsHasName( PETSC_NULL, "-vrmliv",&flg );CHKERRQ(ierr);
    if (flg) VWin->is_inventor = 1;

    /* Use Human coordinates (not available with VRML 2) */
    ierr = OptionsHasName( PETSC_NULL, "-vrmlgcoord", &flg ); CHKERRQ(ierr);
    if (flg) VWin->is_human_coord = 0;

    ierr = OptionsHasName( PETSC_NULL, "-vrml2", &flg ); CHKERRQ(ierr);
    if (flg) VWin->vrml_version = 2;

    /* WWWinline */
    ierr = OptionsGetString( PETSC_NULL, "-vrmlwww",string,1024,&flg);CHKERRQ(ierr);
    if (flg) {
	VWin->wwwname_form = (char *)PetscMalloc( PetscStrlen(string) + 1 );
	CHKPTRQ(VWin->wwwname_form);
	PetscStrcpy( VWin->wwwname_form, string );
	ierr = OptionsGetString( PETSC_NULL, "-vrmlwwwform",string,1024,&flg);CHKERRQ(ierr);
	if (!flg) {
	    PetscStrcpy( string, "f%d.wrl" );
	}
	VWin->fname_form = (char *)PetscMalloc( PetscStrlen( string ) + 1 );
	CHKPTRQ(VWin->fname_form);
	PetscStrcpy( VWin->fname_form, string );
    }
  }
  else {
    ierr = OptionsGetString(PETSC_NULL,"-vrmlfile",string,1024,&flg); CHKERRQ(ierr);
    if (!fname && flg) {
      fname = string;
    }
  }

  if (VWin->is_inventor) {
      VWin->DrawTensorSurfaceContour	= ViDrawTensorSurfaceContour_IV;
      VWin->DrawTensorMapSurfaceContour	= ViDrawTensorMapSurfaceContour_IV;
      VWin->DrawTensorMapMesh	        = ViDrawTensorMapMesh_IV;
      /* No LOD information in Inventor */
      VWin->use_LOD                     = 0;
  }

  /* Output header */
  if (VWin->is_inventor) {
      fprintf( VWin->fp, "#Inventor V2.1 ascii\n#\n" );
  }
  else {
      fprintf( VWin->fp, "#VRML V1.0 ascii\n#\n# PETSc DrawVRML\n#\n" );
  }

  VWin->curfp = VWin->fp;

  ctx->data    = (void *) VWin;
  *inctx       = ctx;
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "ViewerDrawOpenVRML" 
/*@C
   ViewerDrawOpenVRML - Opens an VRML file for use as a viewer. If you want to 
   do graphics in this window, you must call ViewerDrawGetDraw() and
   perform the graphics on the Draw object.

   Input Parameters:
+  comm - communicator that will share window
.  fname - the filename on which to open, or null for stdout
.  title - the title to put in the title bar
.  x, y - the screen coordinates of the upper left corner of window
-  w, h - the screen width and height in pixels

   Output Parameters:
.  viewer - the viewer

   Options Database Keys:
.  -nox - Disables all x-windows output
.  -display <name> - Specifies name of machine for the X display

.keywords: draw, open, VRML, viewer

.seealso: DrawOpenVRML()
@*/
int ViewerDrawOpenVRML(MPI_Comm comm,char* fname,char *title,Viewer *viewer)
{
  int    ierr;
  Viewer ctx;

  *viewer = 0;
  PetscHeaderCreate(ctx,_p_Viewer,int,VIEWER_COOKIE,DRAW_VIEWER,comm,ViewerDestroy,0);
  PLogObjectCreate(ctx);
  ierr = DrawOpenVRML(comm,fname,title,&ctx->draw); CHKERRQ(ierr);
  PLogObjectParent(ctx,ctx->draw);

  ctx->flush   = ViewerFlush_Draw;
  ctx->destroy = ViewerDestroy_Draw;
  *viewer      = ctx;
  PetscFunctionReturn(0);
}

