#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: ex4.c,v 1.15 1998/03/06 00:20:45 bsmith Exp $";
#endif

/*T
    Concepts: Grid vectors^Multigrid
    Routines: GridCreateRectangular1D(); 
    Routines: ViewerDrawOpenX(); GridDestroy(); 
    Routines: GridRefine(); GridCreateRestriction();
    Routines: GVecEvaluateFunctionGalerkin();
    Routines: MGSetLevels(); MGSetResidual(); MGSetInterpolate();
    Routines: MGSetRestriction(); MGGetSmoother(); MGSetRhs();
    Routines: MGSetX(); MGSetR(); MGGetCoarseSolve(); MGCheck();
    Routines: SNESSetFromOptions(); SNESSolve(); SNESDestroy();
    
    Comment: Solves a simple nonlinear PDE in one dimension with multigrid.
T*/

static char help[] = "Solves a simple nonlinear PDE in one dimension with multigrid.\n\n";


#include "snes.h"
#include "gvec.h"
#include "mg.h"

#define MAXGRIDS 10

extern int SolutionFunction(int,double *,double *,double *,Scalar *,void*);

extern int FormJacobian(SNES,Vec,Mat*,Mat*,MatStructure*,void*);
extern int FormJacobianFD(SNES,Vec,Mat*,Mat*,MatStructure*,void*);
extern int FormFunction(SNES,Vec,Vec,void*);

#include "src/gvec/examples/tutorials/common3and4.c"

typedef struct {
  GVec    x[MAXGRIDS],b[MAXGRIDS],r[MAXGRIDS];  /* discrete grid vectors */
  GMat    A[MAXGRIDS];                          /* Jacobian for nonlinear solver */
  GMat    R[MAXGRIDS];
  Grid    grid[MAXGRIDS];
  int     levels;
} UserCtx;
  
int main(int argc,char **args)
{
  GVec                    u;  /* exact solution */
  UserCtx                 ctx;
  int                     ierr, i, n = 10, flag, its;
  SNES                    snes;                /* nonlinear solver context */
  SLES                    sles,subsles;
  PC                      pc;
  Scalar                  one = 1.0;
  GVecErrorSNESMonitorCtx mctx;
  Draw                    draw;

  PetscInitialize(&argc,&args,0,help);
  ierr = GVecInitialize(); CHKERRA(ierr);

  ierr = OptionsGetInt(PETSC_NULL,"-n",&n,&flag); CHKERRA(ierr);
  ctx.levels = 2;
  ierr = OptionsGetInt(PETSC_NULL,"-levels",&ctx.levels,&flag); CHKERRA(ierr);
  if (ctx.levels > MAXGRIDS) SETERRA(1,0,"Too many levels requested");

  /*
      Construct the coarse grid and the discretization context
  */
  ierr = GridCreateRectangular1D(PETSC_COMM_WORLD,n,0.0,1.0,PETSC_NULL,
                                 DISCRETIZATION_LINEAR,&ctx.grid[0]); CHKERRA(ierr);

  /*
       Create all the finer grids and the interpolation between the grids
  */
  for ( i=1; i<ctx.levels; i++ ) {
    ierr = GridRefine(ctx.grid[i-1],2,&ctx.grid[i]); CHKERRA(ierr);
    ierr = GridCreateRestriction(ctx.grid[i],ctx.grid[i-1],&ctx.R[i]); CHKERRA(ierr);
  }

  /*
      Create the work vectors and matrix for all levels 
  */
  for ( i=0; i<ctx.levels; i++ ) {
    /*   Create the grid vector */
    ierr = GridCreateGVec(ctx.grid[i],&ctx.x[i]); CHKERRA(ierr);

    /*   Create additional vectors to hold the right hand side and exact solution */
    ierr = VecDuplicate(ctx.x[i],&ctx.b[i]); CHKERRA(ierr);
    ierr = VecDuplicate(ctx.x[i],&ctx.r[i]); CHKERRA(ierr);

    /*   Compute the empty Jacobian matrix */
    ierr = GridCreateGMat(ctx.grid[i],&ctx.A[i]); CHKERRA(ierr);
  }

  /*
      Set the initial guess on the finest level.
      Note: One would actually do a nonlinear problem by grid iteration:
      Solve on the coarsest grid, interpolate to next grid for initial guess, etc.
  */
  ierr = VecSet(&one,ctx.x[ctx.levels-1]); CHKERRA(ierr);

  /*
      Create a vector to hold the exact solution
  */
  ierr = VecDuplicate(ctx.x[ctx.levels-1],&u); CHKERRA(ierr);
  ierr = GVecEvaluateFunction(u,SolutionFunction,0); CHKERRA(ierr);

  /*
     Create the nonlinear solver context 
  */
  ierr = SNESCreate(PETSC_COMM_WORLD,SNES_NONLINEAR_EQUATIONS,&snes); CHKERRA(ierr);

  /*
     Set the function for which we wish to find the zero and its Jacobian
  */
  ierr = SNESSetFunction(snes,ctx.b[ctx.levels-1],FormFunction,&ctx);CHKERRA(ierr);
  ierr = OptionsHasName(PETSC_NULL,"-fd",&flag); CHKERRA(ierr);
  if (flag) {
    ierr = SNESSetJacobian(snes,ctx.A[ctx.levels-1],ctx.A[ctx.levels-1],FormJacobianFD,&ctx);
           CHKERRA(ierr);
  } else {
    ierr = SNESSetJacobian(snes,ctx.A[ctx.levels-1],ctx.A[ctx.levels-1],FormJacobian,&ctx);
           CHKERRA(ierr);
  }

  /*
      Monitor the error at every iteration
  */
  ierr = ViewerDrawGetDraw(VIEWER_DRAWX_WORLD,&draw); CHKERRA(ierr);
  ierr = DrawSetTitle(draw,"Error"); CHKERRA(ierr);
  mctx.error_viewer      = VIEWER_DRAWX_WORLD;
  mctx.norm_error_viewer = VIEWER_STDOUT_WORLD;
  mctx.solution          = u;
  ierr = SNESSetMonitor(snes,GVecErrorSNESMonitor,&mctx); CHKERRA(ierr);

  /*
      Set up to use multigrid 
  */
  ierr = SNESGetSLES(snes,&sles); CHKERRA(ierr);
  ierr = SLESGetPC(sles,&pc); CHKERRA(ierr);
  ierr = PCSetType(pc,PCMG);  CHKERRA(ierr);
  ierr = MGSetLevels(pc,ctx.levels); CHKERRA(ierr);
  for ( i=1; i<ctx.levels; i++ ) {
    ierr = MGSetResidual(pc,i,MGDefaultResidual,ctx.A[i]); CHKERRA(ierr);
    ierr = MGSetInterpolate(pc,i,ctx.R[i]); CHKERRA(ierr);
    ierr = MGSetRestriction(pc,i,ctx.R[i]); CHKERRA(ierr);
    ierr = MGGetSmoother(pc,i,&subsles); CHKERRA(ierr);
    ierr = SLESSetOperators(subsles,ctx.A[i],ctx.A[i],SAME_NONZERO_PATTERN); CHKERRA(ierr);
    ierr = MGSetRhs(pc,i-1,ctx.b[i-1]); CHKERRA(ierr);
    ierr = MGSetX(pc,i-1,ctx.x[i-1]); CHKERRA(ierr);
    ierr = MGSetR(pc,i-1,ctx.r[i-1]); CHKERRA(ierr);
  }
  ierr = MGGetCoarseSolve(pc,&subsles); CHKERRA(ierr);
  ierr = SLESSetOperators(subsles,ctx.A[0],ctx.A[0],SAME_NONZERO_PATTERN); CHKERRA(ierr);
  ierr = MGSetR(pc,ctx.levels-1,ctx.r[ctx.levels-1]); CHKERRA(ierr);
  ierr = MGCheck(pc); CHKERRA(ierr);

  /*
     Set various runtime options; then solve the nonlinear system
  */
  ierr = SNESSetFromOptions(snes); CHKERRA(ierr);
  ierr = SNESSolve(snes,ctx.x[ctx.levels-1],&its); CHKERRA(ierr);

  /*
     Free work space.  All PETSc objects should be destroyed when they
     are no longer needed.
  */
  for ( i=0; i<ctx.levels; i++ ) {
    ierr = GridDestroy(ctx.grid[i]);CHKERRA(ierr);
    ierr = VecDestroy(ctx.x[i]); CHKERRA(ierr);
    ierr = VecDestroy(ctx.b[i]); CHKERRA(ierr);
    ierr = VecDestroy(ctx.r[i]); CHKERRA(ierr);
    ierr = MatDestroy(ctx.A[i]); CHKERRA(ierr);
  }
  for ( i=1; i<ctx.levels; i++ ) {
    ierr = MatDestroy(ctx.R[i]); CHKERRA(ierr);
  }
  ierr = VecDestroy(u); CHKERRA(ierr);
  ierr = SNESDestroy(snes); CHKERRA(ierr);

  PetscFinalize();
  return 0;
}

#include <math.h>

/* -------------------------------------------------------------------- */
/*
   SolutionFunction - Defines the solution to the PDE. It evaluates the 
   solution at any set of points.

   Input Parameters:
.  n - array dimension
.  x,y,z - grid point coordinates (not used here)
.  values - array to hold solution values
.  ctx - optional user-defined context (not used here)

   Output Parameter:
.  values - newly computed solution values
*/
int SolutionFunction(int n,double *x,double *y,double *z,Scalar *values,void *ctx)
{
  int i;

  for ( i=0; i<n; i++ ) values[i] = sqrt(x[i]);
  return 0;
}
/* -------------------------------------------------------------------- */
/*
   FormFunction - Evaluates nonlinear function, F(x).

   Input Parameters:
.  snes - the SNES context
.  x - input vector
.  dummy - optional user-defined context, as set by SNESSetFunction()

   Output Parameter:
.  f - function vector
*/
int FormFunction(SNES snes,GVec x,Vec f,void* dummy)
{
  int     ierr;

  ierr = UserApplyOperator(x,f); CHKERRQ(ierr);
  ierr = UserApplyDirichlet(x,f,SolutionFunction,0);
  return 0;
} 
/* -------------------------------------------------------------------- */
/*
   FormJacobian - Evaluates Jacobian matrix, F'(x).

   Input Parameters:
.  snes - the SNES context
.  x - input vector
.  ctx - optional user-defined context, as set by SNESSetJacobian()

   Output Parameters:
.  A - Jacobian matrix
.  B - optionally different preconditioning matrix
.  str - flag indicating matrix structure
*/
int FormJacobian(SNES snes,GVec x,GMat *A,GMat *B,MatStructure* str,void* dummy)
{
  int     ierr,i;
  UserCtx *ctx = (UserCtx *) dummy;
  SLES    sles,mgsles;
  PC      pc;
  PCType  type;
  Scalar  half = .5;

  /* 
     Forms Jacobian on finest level
  */
  ierr = UserComputeMatrix(x,*B); CHKERRQ(ierr);
  ierr = UserComputeDirichlet(x,*B,SolutionFunction,0);CHKERRQ(ierr);
  *str = SAME_NONZERO_PATTERN;

  ierr = SNESGetSLES(snes,&sles); CHKERRQ(ierr);
  ierr = SLESGetPC(sles,&pc); CHKERRQ(ierr);
  ierr = PCGetType(pc,&type); CHKERRQ(ierr);
  if (!PetscStrcmp(type,"mg")) {
    /*
       Form Jacobian on all the coarser levels and set in MG context
    */
    ierr = VecCopy(x,ctx->r[ctx->levels-1]); CHKERRQ(ierr);
    for ( i=ctx->levels-2; i>=0; i-- ) {
      ierr = MatMult(ctx->R[i+1],ctx->r[i+1],ctx->r[i]); CHKERRQ(ierr);
      ierr = VecScale(&half,ctx->r[i]); CHKERRQ(ierr);
      ierr = UserComputeMatrix(ctx->r[i],ctx->A[i]); CHKERRQ(ierr);
      ierr = UserComputeDirichlet(ctx->r[i],ctx->A[i],SolutionFunction,0);CHKERRQ(ierr);
      ierr = MGGetSmootherDown(pc,i,&mgsles); CHKERRQ(ierr);
      ierr = SLESSetOperators(mgsles,ctx->A[i],ctx->A[i],SAME_NONZERO_PATTERN);CHKERRQ(ierr);
    }
  }
  return 0;
}
/* -------------------------------------------------------------------- */
/*
   FormJacobian - Computes the Jacobian matrix with finite differences.

   Input Parameters:
.  snes - the SNES context
.  x - input vector
.  dummy - optional user-defined context, as set by SNESSetJacobian()

   Output Parameters:
.  A - Jacobian matrix
.  B - optionally different preconditioning matrix
.  str - flag indicating matrix structure
*/
int FormJacobianFD(SNES snes,GVec x,GMat *A,GMat *B,MatStructure* str,void* dummy)
{
  int     ierr,i;
  UserCtx *ctx = (UserCtx *) dummy;
  SLES    sles,mgsles;
  PC      pc;
  PCType  type;
  Scalar  half = .5;

  /* 
     Forms Jacobian on finest level
  */
  *str = SAME_NONZERO_PATTERN; 
  ierr = UserComputeMatrix(x,*B); CHKERRQ(ierr);
  ierr = GMatFDColoringApply(*B,x,str,snes); CHKERRQ(ierr);


  ierr = SNESGetSLES(snes,&sles); CHKERRQ(ierr);
  ierr = SLESGetPC(sles,&pc); CHKERRQ(ierr);
  ierr = PCGetType(pc,&type); CHKERRQ(ierr);
  if (!PetscStrcmp(type,"mg")) {
    /*
       Form Jacobian on all the coarser levels and set in MG context
    */
    ierr = VecCopy(x,ctx->r[ctx->levels-1]); CHKERRQ(ierr);
    for ( i=ctx->levels-2; i>=0; i-- ) {
      ierr = MatMult(ctx->R[i+1],ctx->r[i+1],ctx->r[i]); CHKERRQ(ierr);
      ierr = VecScale(&half,ctx->r[i]); CHKERRQ(ierr);
      ierr = GMatFDColoringApply(ctx->A[i],ctx->r[i],str,snes); CHKERRQ(ierr);
      ierr = MGGetSmootherDown(pc,i,&mgsles); CHKERRQ(ierr);
      ierr = SLESSetOperators(mgsles,ctx->A[i],ctx->A[i],SAME_NONZERO_PATTERN);CHKERRQ(ierr);
    }
  }
  return 0;
}
