#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: qua1d.c,v 1.14 1998/04/21 19:47:44 bsmith Exp $";
#endif

/*
   Defines piecewise quadratic function space on a one dimensional 
 grid. Suitable for finite difference or finite element type discretization of a PDE.

       |   Grid points (as given in the GridData datastructure 
       o   additional discretization points

       |       o        |      o      |       o     |

    Thus both linear and quadratic can share the same GridData structure
  but the quadratic actually has additional "grid points" where the "extra"
  degrees of freedom are defined.

*/

#include "src/gvec/impls/rectangular/rectimpl.h"         /*I "gvec.h" I*/
#include "pinclude/pviewer.h"

typedef struct  {
  int            dof;
  double         *x;      /* coordinates of all points; including "interior nodes" not stored
                             in the GridData data-structure; only the parts on this processor */
  DAPeriodicType per;
} Grid_Rectangular_1D_Quadratic;

#undef __FUNC__  
#define __FUNC__ "GridDestroy_Rectangular_1D_Quadratic"
static int GridDestroy_Rectangular_1D_Quadratic(Grid grid)
{
  Grid_Rectangular_1D_Quadratic *s = (Grid_Rectangular_1D_Quadratic*) grid->data;
  int                           ierr;

  PetscFunctionBegin;
  PetscFree(s);
  ierr = GridDataDestroy(grid->griddata); CHKERRQ(ierr);
  if (grid->iscoloring) {
    ierr = ISColoringDestroy(grid->iscoloring); CHKERRQ(ierr);
  }
  if (grid->fdcoloring) {
    ierr = MatFDColoringDestroy(grid->fdcoloring); CHKERRQ(ierr);
  }
  PLogObjectDestroy(grid);
  PetscHeaderDestroy(grid);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridCreateGMat_Rectangular_1D_Quadratic"
static int GridCreateGMat_Rectangular_1D_Quadratic(Grid grid,GMat *gmat)
{
  Grid_Rectangular_1D_Quadratic *s = (Grid_Rectangular_1D_Quadratic *)grid->data;
  GridData_Rectangular          *gridr = (GridData_Rectangular*)grid->griddata->data;
  int                           m,ierr,Nl,*idx;
  MPI_Comm                      comm;
  Vec                           w;

  PetscFunctionBegin;
  ierr = DACreateGlobalVector(gridr->da,&w); CHKERRQ(ierr);
  ierr = VecGetLocalSize(w,&m); CHKERRQ(ierr);
  ierr = PetscObjectGetComm((PetscObject)grid,&comm);CHKERRQ(ierr);
  ierr = MatCreateMPIAIJ(comm,m,m,PETSC_DECIDE,PETSC_DECIDE,5*s->dof,0,0,0,gmat);CHKERRQ(ierr);

  ierr = DAGetGlobalIndices(gridr->da,&Nl,&idx); CHKERRQ(ierr);
  {
    ISLocalToGlobalMapping ltog;
    ierr = ISLocalToGlobalMappingCreate(PETSC_COMM_SELF,Nl,idx,&ltog);CHKERRQ(ierr);
    ierr = MatSetLocalToGlobalMapping(*gmat,ltog); CHKERRQ(ierr);
    ierr = ISLocalToGlobalMappingDestroy(ltog); CHKERRQ(ierr);
  }
  ierr = PetscObjectCompose((PetscObject) *gmat, "grid",(PetscObject) grid);CHKERRQ(ierr);
  
  PetscFunctionReturn(0);
}

/* -------------------------------------------------------------------*/

#undef __FUNC__  
#define __FUNC__ "GridView_Rectangular_1D_Quadratic" 
static int GridView_Rectangular_1D_Quadratic(Grid grid,Viewer viewer)
{
  ViewerType  vtype;
  int         rank,ierr;
  FILE        *fd;

  PetscFunctionBegin;
  MPI_Comm_rank(grid->comm,&rank); if (rank > 0) PetscFunctionReturn(0);
  ierr = ViewerGetType(viewer,&vtype);
  if (vtype == ASCII_FILE_VIEWER || vtype == ASCII_FILES_VIEWER){
    ierr = ViewerASCIIGetPointer(viewer,&fd); CHKERRQ(ierr);
    fprintf(fd,"Grid Object:\n");
    fprintf(fd,"  Rectangular 1D Quadratic discretization\n");
  } else {
    SETERRQ(1,1,"Viewer type not supported by PETSc object");
  }
  ierr = GridDataView(grid->griddata,viewer); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecView_Rectangular_1D_Quadratic_Draw"
static int GVecView_Rectangular_1D_Quadratic_Draw(GVec gvec,Viewer v)
{
  Grid                          grid;
  Grid_Rectangular_1D_Quadratic *gridl;
  GridData                      gridd;
  GridData_Rectangular          *gridr;
  Vec                           xin = gvec;
  int                           i,rank,size,ierr,start,end,n,tag;
  MPI_Status                    status;
  double                        coors[4],ymin,ymax,xmin,xmax,tmp,*xg;
  Scalar                        *array;
  Draw                          draw;
  PetscTruth                    isnull;
  MPI_Comm                      comm;
  DrawAxis                      axis;

  PetscFunctionBegin;
  ierr = ViewerDrawGetDraw(v,&draw);CHKERRQ(ierr);
  ierr = DrawIsNull(draw,&isnull); CHKERRQ(ierr); if (isnull) PetscFunctionReturn(0);
  ierr = DrawCheckResizedWindow(draw);CHKERRQ(ierr);

  ierr = VecGetArray(xin,&array); CHKERRQ(ierr);
  ierr = VecGetLocalSize(xin,&n); CHKERRQ(ierr);

  ierr = GVecGetGrid(gvec,&grid); CHKERRQ(ierr);
  tag       = grid->tag;
  comm      = grid->comm;
  gridl     = (Grid_Rectangular_1D_Quadratic*)grid->data;
  gridd     = grid->griddata;
  gridr     = (GridData_Rectangular *) gridd->data;
  xg        = gridl->x;

  MPI_Comm_size(comm,&size); 

  xmin = 1.e20; xmax = -1.e20;
  for ( i=0; i<n; i++ ) {
#if defined(USE_PETSC_COMPLEX)
    if (real(array[i]) < xmin) xmin = real(array[i]);
    if (real(array[i]) > xmax) xmax = real(array[i]);
#else
    if (array[i] < xmin) xmin = array[i];
    if (array[i] > xmax) xmax = array[i];
#endif
  }
  if (xmin + 1.e-10 > xmax) {
    xmin -= 1.e-5;
    xmax += 1.e-5;
  }
  ierr = MPI_Reduce(&xmin,&ymin,1,MPI_DOUBLE,MPI_MIN,0,comm);CHKERRQ(ierr);
  ierr = MPI_Reduce(&xmax,&ymax,1,MPI_DOUBLE,MPI_MAX,0,comm);CHKERRQ(ierr);
  MPI_Comm_rank(comm,&rank);
  ierr = DrawAxisCreate(draw,&axis); CHKERRQ(ierr);
  PLogObjectParent(draw,axis);
  if (!rank) {
    ierr = DrawClear(draw); CHKERRQ(ierr);
    ierr = DrawFlush(draw);CHKERRQ(ierr);
    xmin = xg[0]; xmax = xg[gridr->m-1];
    ierr = DrawAxisSetLimits(axis,xmin,xmax,ymin,ymax); CHKERRQ(ierr);
    ierr = DrawAxisDraw(axis); CHKERRQ(ierr);
    ierr = DrawGetCoordinates(draw,coors,coors+1,coors+2,coors+3);CHKERRQ(ierr);
  }
  ierr = DrawAxisDestroy(axis);CHKERRQ(ierr);
  ierr = MPI_Bcast(coors,4,MPI_DOUBLE,0,comm);CHKERRQ(ierr);
  if (rank) DrawSetCoordinates(draw,coors[0],coors[1],coors[2],coors[3]);
  /* draw local part of vector */
  ierr = VecGetOwnershipRange(xin,&start,&end);CHKERRQ(ierr);
  if (rank < size-1) { /*send value to right */
    ierr = MPI_Send(&array[n-1],1,MPI_DOUBLE,rank+1,tag,comm);CHKERRQ(ierr);
  }
  for ( i=1; i<n; i++ ) {
#if !defined(USE_PETSC_COMPLEX)
    ierr = DrawLine(draw,xg[i-1+start],array[i-1],xg[i+start],array[i],DRAW_RED);CHKERRQ(ierr);
#else
    ierr = DrawLine(draw,xg[i-1+start],real(array[i-1]),xg[i+start],real(array[i]),DRAW_RED);CHKERRQ(ierr);
#endif
  }
  if (rank) { /* receive value from right */
    ierr = MPI_Recv(&tmp,1,MPI_DOUBLE,rank-1,tag,comm,&status);CHKERRQ(ierr);
#if !defined(USE_PETSC_COMPLEX)
    ierr = DrawLine(draw,xg[start-1],tmp,xg[start],array[0],DRAW_RED);CHKERRQ(ierr);
#else
    ierr = DrawLine(draw,xg[start-1],tmp,xg[start],real(array[0]),DRAW_RED);CHKERRQ(ierr);
#endif
  }
  ierr = DrawSynchronizedFlush(draw); CHKERRQ(ierr);
  ierr = DrawPause(draw); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecView_Rectangular_1D_Quadratic" 
static int GVecView_Rectangular_1D_Quadratic(GVec gvec,Viewer viewer)
{
  Grid        grid;
  ViewerType  vtype;
  int         rank,ierr;

  PetscFunctionBegin;
  ierr = GVecGetGrid(gvec,&grid); CHKERRQ(ierr);
  MPI_Comm_rank(grid->comm,&rank); 
  ierr = ViewerGetType(viewer,&vtype);
  if (vtype == ASCII_FILE_VIEWER || vtype == ASCII_FILES_VIEWER){
    if (rank == 0) {
      ierr = GridView(grid,viewer); CHKERRQ(ierr);
      ierr = ViewerFlush(viewer); CHKERRQ(ierr);
    }
    PetscBarrier(grid);
    ierr = VecView(gvec,viewer); CHKERRQ(ierr);
  } else if (vtype == DRAW_VIEWER) {
    ierr = GVecView_Rectangular_1D_Quadratic_Draw(gvec,viewer);CHKERRQ(ierr);
  } else {
    SETERRQ(1,1,"Viewer type not supported by PETSc object");
  }
  PetscFunctionReturn(0);
}

/* -------------------------------------------------------------------*/

#undef __FUNC__  
#define __FUNC__ "GridRefine_Rectangular_1D_Quadratic"
static int GridRefine_Rectangular_1D_Quadratic(Grid grid,int l,Grid *newgrid)
{
  GridData                      newgriddata;
  int                           ierr;
  Grid_Rectangular_1D_Quadratic *gridr,*newgridr;
  
  PetscFunctionBegin;
  ierr     = GridDataRefine(grid->griddata,l,&newgriddata); CHKERRQ(ierr);
  ierr     = GridDataCreateGrid(newgriddata,DISCRETIZATION_QUADRATIC,newgrid);CHKERRQ(ierr);
  /* remove extra reference to gridd */ 
  ierr     = GridDataDestroy(newgriddata); CHKERRQ(ierr);
  gridr    = (Grid_Rectangular_1D_Quadratic *) grid->data;
  newgridr = (Grid_Rectangular_1D_Quadratic *) (*newgrid)->data;

  PetscMemcpy(newgridr,gridr,sizeof(Grid_Rectangular_1D_Quadratic));
  (*newgrid)->gridparent = grid;
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridCreateGVec_Rectangular_1D_Quadratic"
static int GridCreateGVec_Rectangular_1D_Quadratic(Grid grid,GVec *gvec)
{
  GridData_Rectangular  *gridr = (GridData_Rectangular*)grid->griddata->data;
  Vec                   v;
  int                   ierr;

  PetscFunctionBegin;
  ierr = DACreateGlobalVector(gridr->da,&v); CHKERRQ(ierr);
  ierr = PetscObjectCompose((PetscObject) v, "grid",(PetscObject) grid);CHKERRQ(ierr);

  *gvec = (GVec) v;

  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecGetLocalGVec_Rectangular_1D_Quadratic"
static int GVecGetLocalGVec_Rectangular_1D_Quadratic(GVec g,GVec *gvec)
{
  Grid                  grid;
  GridData              gridd;
  GridData_Rectangular  *gridr;
  Vec                   w,v;
  int                   ierr;

  PetscFunctionBegin;
  ierr  = GVecGetGrid(g,&grid); CHKERRQ(ierr);
  gridd = grid->griddata;
  gridr = (GridData_Rectangular *)gridd->data;

  ierr = DACreateLocalVector(gridr->da,&w); CHKERRQ(ierr);
  ierr = VecDuplicate(w,&v); CHKERRQ(ierr);
  ierr = PetscObjectCompose((PetscObject) v, "grid",(PetscObject) grid); CHKERRQ(ierr);

  *gvec = (GVec) v;

  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecRestoreLocalGVec_Rectangular_1D_Quadratic" 
static int GVecRestoreLocalGVec_Rectangular_1D_Quadratic(GVec g,GVec *gvec)
{
  int ierr;

  PetscFunctionBegin;
  ierr = VecDestroy(*gvec);CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecGlobalToLocal_Rectangular_1D_Quadratic"
static int GVecGlobalToLocal_Rectangular_1D_Quadratic(GVec g,InsertMode mode, GVec l)
{
  Grid                 grid;
  GridData             gridd;
  GridData_Rectangular *gridr;
  int                  ierr;

  PetscFunctionBegin;
  ierr  = GVecGetGrid(g,&grid); CHKERRQ(ierr);
  gridd = grid->griddata;
  gridr = (GridData_Rectangular *)gridd->data;

  ierr = DAGlobalToLocalBegin(gridr->da,g,mode,l); CHKERRQ(ierr);
  ierr = DAGlobalToLocalEnd(gridr->da,g,mode,l); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecLocalToGlobal_Rectangular_1D_Quadratic"
static int GVecLocalToGlobal_Rectangular_1D_Quadratic(GVec l,InsertMode mode, GVec g)
{
  Grid                 grid;
  GridData             gridd;
  GridData_Rectangular *gridr;
  int                  ierr;

  PetscFunctionBegin;
  ierr  = GVecGetGrid(g,&grid); CHKERRQ(ierr);
  gridd = grid->griddata;
  gridr = (GridData_Rectangular *)gridd->data;

  ierr = DALocalToGlobal(gridr->da,l,mode,g); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridSetUp_Rectangular_1D_Quadratic"
int GridSetUp_Rectangular_1D_Quadratic(Grid grid)
{
  int                           ierr,flag,istart,iend,i;
  Grid_Rectangular_1D_Quadratic *s = (Grid_Rectangular_1D_Quadratic *)grid->data;
  GridData_Rectangular          *gridr = (GridData_Rectangular *)grid->griddata->data;
  Vec                           v;

  PetscFunctionBegin;
  ierr = DACreate1d(grid->comm,s->per,2*gridr->m-1,s->dof,1,PETSC_NULL,&gridr->da);CHKERRQ(ierr);
  grid->setupcalled = 1;

  /* 
     set the coordinates for all the local nodes (including quadratic ones that are not
     represented in the GridData data structure.
  */
  ierr = DACreateGlobalVector(gridr->da,&v); CHKERRQ(ierr);
  ierr = VecGetOwnershipRange(v,&istart,&iend); CHKERRQ(ierr);
  s->x = (double *) PetscMalloc((iend-istart)*sizeof(double)); CHKPTRQ(s->x);
  for ( i=istart; i<iend; i++ ) {
    if ( i % 2 ) {
      s->x[i-istart] = gridr->x[i % 2];
    } else {
      s->x[i-istart] = gridr->x[i % 2] + gridr->x[(i % 2) + 1];
    }
  }

  ierr = OptionsHasName(PETSC_NULL,"-grid_view",&flag); CHKERRQ(ierr);
  if (flag) {
    ierr = GridView(grid,0);
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecEvaluateFunction_Rectangular_1D_Quadratic"
static int GVecEvaluateFunction_Rectangular_1D_Quadratic(GVec v,PointFunction f,void* ctx)
{
  Grid                          grid;
  Grid_Rectangular_1D_Quadratic *s;
  int                           ierr,istart,iend;
  Scalar                        *array;

  PetscFunctionBegin;
  ierr  = GVecGetGrid(v,&grid); CHKERRQ(ierr);
  s     = (Grid_Rectangular_1D_Quadratic*) grid->data;
  
  ierr = VecGetOwnershipRange(v,&istart,&iend); CHKERRQ(ierr);
  ierr = VecGetArray(v,&array); CHKERRQ(ierr);
  ierr = (*f)(iend-istart, s->x,0,0,array,ctx);CHKERRQ(ierr);
  ierr = VecRestoreArray(v,&array); CHKERRQ(ierr);
  ierr = VecAssemblyBegin(v);  CHKERRQ(ierr);
  ierr = VecAssemblyEnd(v);  CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecEvaluateFunctionGalerkin_Rectangular_1D_Quadratic"
static int GVecEvaluateFunctionGalerkin_Rectangular_1D_Quadratic(GVec v,PointFunction f,void* ctx)
{
  Grid                          grid;
  Grid_Rectangular_1D_Quadratic *s;
  int                           ierr,istart,iend,xs,M,m,i;
  Scalar                        *array;
  GridData_Rectangular          *gridr;
  double                        *x,h;

  PetscFunctionBegin;
  ierr  = GVecGetGrid(v,&grid); CHKERRQ(ierr);
  s     = (Grid_Rectangular_1D_Quadratic*) grid->data;
  gridr = (GridData_Rectangular *)grid->griddata->data;
  x     = gridr->x;

  ierr = VecGetOwnershipRange(v,&istart,&iend); CHKERRQ(ierr);
  ierr = VecGetArray(v,&array); CHKERRQ(ierr);
  ierr = (*f)(iend-istart, s->x,0,0,array,ctx);CHKERRQ(ierr);
  /*
      Scale the entries of array by the average cell sizes
  */
  ierr = DAGetCorners(gridr->da,&xs,0,0,&m,0,0); CHKERRQ(ierr);
  ierr = DAGetInfo(gridr->da,0,&M,0,0,0,0,0,0,0,0); CHKERRQ(ierr);
  if (xs == 0) {
    /* special case, left end point of grid */
    h         = x[1] - x[0];
    array[0] *= h;
    xs++; m--;
  }
  if (xs + m == M) {
    /* special case, right end point of grid */
    h                  = x[M-1] - x[M-2];
    array[iend-1-istart] *= h;
    m--;
  }
  for ( i=xs; i<xs+m; i++ ) {
    h                = .5*(x[i+1] - x[i-1]);
    array[i-istart] *= h;
  }

  ierr = VecRestoreArray(v,&array); CHKERRQ(ierr);
  ierr = VecAssemblyBegin(v);  CHKERRQ(ierr);
  ierr = VecAssemblyEnd(v);  CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridCreateRestriction_Rectangular_1D_Quadratic"
int GridCreateRestriction_Rectangular_1D_Quadratic(Grid dcf,Grid dcc,GMat *gmat)
{
  GridData             gridc,gridf;
  GridData_Rectangular *rgridc,*rgridf;
  int                  ierr,xs,mc,mf,indices[3],M,i,l_mc,l_mf,m;
  Scalar               values[3];
  MPI_Comm             comm;
  Vec                  vf,vc;

  PetscFunctionBegin;
  /*
     Get the grid data structure
  */
  gridc   = dcc->griddata;
  rgridc  = (GridData_Rectangular *) gridc->data;
  gridf   = dcf->griddata;
  rgridf  = (GridData_Rectangular *) gridf->data;
  comm    = gridc->comm;

  /* also check dof == 1 */
  if (dcf->gridparent != dcc) {
    SETERRQ(PETSC_ERR_SUP,0,"Only supported for nested grids");
  }
  mf = rgridf->m; mc = rgridc->m;
  if (mf != 2*mc - 1) {
    SETERRQ(PETSC_ERR_SUP,0,"Only supported for factor 2 refinement");
  }
  ierr = DACreateGlobalVector(rgridc->da,&vc); CHKERRQ(ierr);
  ierr = DACreateGlobalVector(rgridf->da,&vf); CHKERRQ(ierr);
  ierr = VecGetLocalSize(vf,&l_mf); CHKERRQ(ierr);
  ierr = VecGetLocalSize(vc,&l_mc); CHKERRQ(ierr);

  /*
      Create the appropriate empty sparse matrix.
  */
  ierr = MatCreateMPIAIJ(comm,l_mc,l_mf,PETSC_DECIDE,PETSC_DECIDE,3,0,0,0,gmat);CHKERRQ(ierr);
  ierr = PetscObjectCompose((PetscObject) *gmat, "grid",(PetscObject) dcc);CHKERRQ(ierr);
  ierr = MatSetOption(*gmat,MAT_COLUMNS_SORTED); CHKERRQ(ierr);

  /*
     Loop over local rows generating stencil 
  */
  ierr = DAGetCorners(rgridc->da,&xs,0,0,&m,0,0); CHKERRQ(ierr);
  ierr = DAGetInfo(rgridc->da,0,&M,0,0,0,0,0,0,0,0); CHKERRQ(ierr);
  /*
      Note that values are inserted into the matrix using the 
    global PETSc ordering.
  */
  if (xs == 0) {
    /* special case, left end point of grid */
    i          = 0;
    indices[0] = 0; 
    indices[1] = 1; 
    values[0]  = 1.0;
    values[1]  = .5;
    ierr = MatSetValues(*gmat,1,&i,2,indices,values,INSERT_VALUES);CHKERRQ(ierr);
    xs++; m--;
  }
  if (xs + m == M) {
    /* special case, right end point of grid */
    i          = M-1;
    indices[0] = 2*M-3; 
    indices[1] = 2*M-2; 
    values[0]  = .5;
    values[1]  = 1.0;
    ierr = MatSetValues(*gmat,1,&i,2,indices,values,INSERT_VALUES);CHKERRQ(ierr);
    m--;
  }
  for ( i=xs; i<xs+m; i++ ) {
    indices[0] = 2*i-1; 
    indices[1] = 2*i; 
    indices[2] = 2*i+1;
    values[0]  = .5;
    values[1]  = 1.0;
    values[2]  = .5;
    ierr = MatSetValues(*gmat,1,&i,3,indices,values,INSERT_VALUES);CHKERRQ(ierr);
  }
  ierr = MatAssemblyBegin(*gmat,MAT_FINAL_ASSEMBLY);
  ierr = MatAssemblyEnd(*gmat,MAT_FINAL_ASSEMBLY);
  PetscFunctionReturn(0);
}

extern int GMatFDColoringCreate_Rectangular_1D_Quadratic(GMat);

static struct _GridOps GOps = { GridSetUp_Rectangular_1D_Quadratic,
                              0,
                              GridRefine_Rectangular_1D_Quadratic,
                              0, 
                              GridCreateGVec_Rectangular_1D_Quadratic,
                              GridCreateGMat_Rectangular_1D_Quadratic,
                              GVecGetLocalGVec_Rectangular_1D_Quadratic,
                              GVecRestoreLocalGVec_Rectangular_1D_Quadratic,
                              0,
                              0,
                              GVecGlobalToLocal_Rectangular_1D_Quadratic,
                              GVecLocalToGlobal_Rectangular_1D_Quadratic,
                              0,
                              0,
                              0,
                              GVecView_Rectangular_1D_Quadratic,
                              GridCreateRestriction_Rectangular_1D_Quadratic,
                              GVecEvaluateFunction_Rectangular_1D_Quadratic,
                              GVecEvaluateFunctionGalerkin_Rectangular_1D_Quadratic,
                              GMatFDColoringCreate_Rectangular_1D_Quadratic};


#undef __FUNC__  
#define __FUNC__ "GridCreateRectangular1DQuadratic_Private"
/*
     DCCreateRectangular1DQuadratic_Private - Creates a discretization 
           appropriate for problems on 1 dimensional rectangular
           grids using Quadratic interpolation.

  Input Parameter:
.   grid - the grid that defines the region
.   dof  - number of degrees of freedom per node
.   per - either DA_NONPERIODIC or DA_XPERIODIC

  Output Parameter:
.   newdc - the discretization context

.keywords: Discrezation

.seealso: GridDestroy()
*/
int GridCreateRectangular1DQuadratic_Private(GridData gridd,Grid *newgrid)
{
  Grid_Rectangular_1D_Quadratic *s;
  Grid                          grid;

  PetscFunctionBegin;
  PetscHeaderCreate(grid,_p_Grid,struct _GridOps,GRID_COOKIE,DISCRETIZATION_QUADRATIC,gridd->comm,GridDestroy,GridView);
  PLogObjectCreate(grid);
  PLogObjectMemory(grid,sizeof(struct _p_Grid)+sizeof(Grid_Rectangular_1D_Quadratic));
  s              = PetscNew(Grid_Rectangular_1D_Quadratic); CHKPTRQ(s);
  PetscMemzero(s,sizeof(Grid_Rectangular_1D_Quadratic));
  PetscMemcpy(grid->ops,&GOps,sizeof(struct _GridOps));
  grid->data     = (void *) s;
  grid->ops->destroy  = GridDestroy_Rectangular_1D_Quadratic;
  grid->ops->view     = GridView_Rectangular_1D_Quadratic;
  grid->griddata = gridd; 
  PetscObjectReference((PetscObject)gridd);
  grid->iscoloring = 0;
  grid->fdcoloring = 0;

  s->dof         = 1;
  s->per         = DA_NONPERIODIC;
  *newgrid       = grid;
  PetscFunctionReturn(0);
}














