#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: mpibdpc.c,v 1.34 1998/04/03 23:15:26 bsmith Exp $";
#endif
/*
   Defines a block Jacobi preconditioner for the MPIBDIAG format.   
   Only supports a single block per processor.
*/

#include "src/pc/impls/bjacobi/bjacobi.h"
#include "sles.h"
#include "src/mat/impls/bdiag/mpi/mpibdiag.h"

typedef struct {
  Vec  x,y;            /* solution and rhs vector for block */
  Mat  Amat;           /* linear system matrix */
  Mat  Pmat;           /* preconditioning matrix */
  int  matalloc;       /* flag indicating whether new matrices were allocated */
} PC_BJacobi_MPIBDiag;

#undef __FUNC__  
#define __FUNC__ "PCDestroy_BJacobi_MPIBDiag"
int PCDestroy_BJacobi_MPIBDiag(PC pc)
{
  PC_BJacobi          *jac = (PC_BJacobi *) pc->data;
  PC_BJacobi_MPIBDiag *bjac = (PC_BJacobi_MPIBDiag *) jac->data;
  int                 i,ierr;

  for ( i=0; i<jac->n_local; i++ ) {
    ierr = SLESDestroy(jac->sles[i]); CHKERRQ(ierr);
  }
  PetscFree(jac->sles);
  ierr = VecDestroy(bjac->x); CHKERRQ(ierr);
  ierr = VecDestroy(bjac->y); CHKERRQ(ierr);
  if (bjac->matalloc) {
    if (bjac->Amat && bjac->Amat != bjac->Pmat) {
      ierr = MatDestroy(bjac->Amat); CHKERRQ(ierr);
    }
    if (bjac->Pmat) {
      ierr = MatDestroy(bjac->Pmat); CHKERRQ(ierr);
    }
  }
  PetscFree(bjac); PetscFree(jac); 
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "PCSetUpOnBlocks_BJacobi_MPIBDiag"
int PCSetUpOnBlocks_BJacobi_MPIBDiag(PC pc)
{
  int                 ierr;
  PC_BJacobi          *jac = (PC_BJacobi *) pc->data;
  PC_BJacobi_MPIBDiag *bjac = (PC_BJacobi_MPIBDiag *) jac->data;

  ierr = SLESSetUp(jac->sles[0],bjac->x,bjac->y); CHKERRQ(ierr);  
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "PCApply_BJacobi_MPIBDiag"
int PCApply_BJacobi_MPIBDiag(PC pc,Vec x, Vec y)
{
  int                 ierr,its;
  PC_BJacobi          *jac = (PC_BJacobi *) pc->data;
  PC_BJacobi_MPIBDiag *bjac = (PC_BJacobi_MPIBDiag *) jac->data;
  Scalar              *x_array,*x_true_array, *y_array,*y_true_array;

  /* 
      The VecPlaceArray() is to avoid having to copy the 
    y vector into the bjac->x vector. The reason for 
    the bjac->x vector is that we need a sequential vector
    for the sequential solve.
  */
  ierr = VecGetArray(x,&x_array); CHKERRQ(ierr);
  ierr = VecGetArray(y,&y_array); CHKERRQ(ierr);
  ierr = VecGetArray(bjac->x,&x_true_array); CHKERRQ(ierr);
  ierr = VecGetArray(bjac->y,&y_true_array); CHKERRQ(ierr); 
  ierr = VecPlaceArray(bjac->x,x_array); CHKERRQ(ierr);
  ierr = VecPlaceArray(bjac->y,y_array); CHKERRQ(ierr);
  ierr = SLESSolve(jac->sles[0],bjac->x,bjac->y,&its); CHKERRQ(ierr);
  ierr = VecPlaceArray(bjac->x,x_true_array); CHKERRQ(ierr);
  ierr = VecPlaceArray(bjac->y,y_true_array); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "PCSetUp_BJacobi_MPIBDiag"
int PCSetUp_BJacobi_MPIBDiag(PC pc)
{
  PC_BJacobi          *jac = (PC_BJacobi *) pc->data;
  Mat                 mat = pc->mat, pmat = pc->pmat, pmsub, amsub;
  Mat_MPIBDiag        *pmatin = (Mat_MPIBDiag *) pmat->data, *matin = 0;
  int                 ierr, size = pmatin->size, m, rstart, rend;
  int                 matalloc, lrows, lcols;
  SLES                sles;
  Vec                 x, y;
  IS                  localr, localc;
  KSP                 subksp;
  PC                  subpc;
  PC_BJacobi_MPIBDiag *bjac = 0;
  MatType             type;
  char                *prefix;

  if (jac->use_true_local) {
    ierr = MatGetType(pc->mat,&type,PETSC_NULL); CHKERRQ(ierr);
    if (type != MATMPIBDIAG) SETERRQ(PETSC_ERR_ARG_NOTSAMETYPE,0,"Incompatible mat type");
    matin = (Mat_MPIBDiag *) mat->data;
  }

  /* user set local number of blocks but not global */
  if (jac->n == -1 && jac->n_local >= 0) {
    ierr = MPI_Allreduce(&jac->n_local,&jac->n,1,MPI_INT,MPI_SUM,pc->comm);CHKERRQ(ierr);
  }
  /* user set global, not local */
  if (jac->n_local == -1 && jac->n > 0) {
    if (jac->n == size) jac->n_local = 1;
    else SETERRQ(PETSC_ERR_ARG_SIZ,0,"Must have exactly 1 block per processor\n"); 
  }
  /* user set nothing */
  if (jac->n_local == -1 && jac->n == -1) {
    jac->n       = size;
    jac->n_local = 1;
  }

  if (size != jac->n) {
    SETERRQ(PETSC_ERR_ARG_SIZ,0,"can only do 1 block per processor\n");
  }

  /* Assume that pmatin and matin have same parallel distribution */
  if (size == 1) {
    matalloc = 0;
    pmsub = pmatin->A;
    if (jac->use_true_local) amsub = matin->A; 
    else                     amsub = pmsub;
  } else {
    Mat *tmp;

    if (pc->setupcalled) { 
      /* Destroy the Submatrix used in the previous SOLVE */
      bjac = (PC_BJacobi_MPIBDiag*)(jac->data);
      ierr = MatDestroy(bjac->Pmat); CHKERRQ(ierr);
    }
    ierr = MatGetLocalSize(pmat,&lrows,&lcols); CHKERRQ(ierr);
    ierr = MatGetOwnershipRange(pmat,&rstart,&rend); CHKERRQ(ierr);
    ierr = ISCreateStride(PETSC_COMM_SELF,lrows,rstart,1,&localc); 
    ierr = ISCreateStride(PETSC_COMM_SELF,lrows,0,1,&localr); 
    PLogObjectParent(pc,localc); PLogObjectParent(pc,localr);CHKERRQ(ierr);
    ierr = MatGetSubMatrices(pmatin->A,1,&localr,&localc,MAT_INITIAL_MATRIX,&tmp);CHKERRQ(ierr);
    pmsub = *tmp; PetscFree(tmp);
    PLogObjectParent(pc,pmsub);

    /* Return control to the user so that the submatrices can be modified (e.g., to apply
       different boundary conditions for the submatrices than for the global problem) */
    ierr = PCModifySubMatrices(pc,1,&localr,&localc,&pmsub,pc->modifysubmatricesP); CHKERRQ(ierr);

    if (jac->use_true_local) {

       if (pc->setupcalled) { 
         /* Destroy the Submatrix used in the previous SOLVE */
         bjac = (PC_BJacobi_MPIBDiag*)(jac->data);
         ierr = MatDestroy(bjac->Amat); CHKERRQ(ierr);
       }

       ierr = MatGetSubMatrices(matin->A,1,&localr,&localc,MAT_INITIAL_MATRIX,&tmp);CHKERRQ(ierr);
      amsub = *tmp; PetscFree(tmp);  
      PLogObjectParent(pc,amsub);
    } else {
      amsub = pmsub;
    }
    ierr = ISDestroy(localr); CHKERRQ(ierr);
    ierr = ISDestroy(localc); CHKERRQ(ierr);
    matalloc = 1;
  }

  if (!pc->setupcalled) { 
    ierr = SLESCreate(PETSC_COMM_SELF,&sles); CHKERRQ(ierr);
    PLogObjectParent(pc,sles);
    ierr = SLESGetKSP(sles,&subksp); CHKERRQ(ierr);
    ierr = KSPSetType(subksp,KSPPREONLY); CHKERRQ(ierr);
    ierr = SLESGetPC(sles,&subpc); CHKERRQ(ierr);
    ierr = PCSetType(subpc,PCILU); CHKERRQ(ierr);
    ierr = PCGetOptionsPrefix(pc,&prefix); CHKERRQ(ierr);
    ierr = SLESSetOptionsPrefix(sles,prefix); CHKERRQ(ierr);
    ierr = SLESAppendOptionsPrefix(sles,"sub_"); CHKERRQ(ierr);
    ierr = SLESSetFromOptions(sles); CHKERRQ(ierr);
    /*
      The reason we need to generate this vector is so KSP may
      generate seq vectors for the local solves
    */
    ierr = MatGetSize(pmsub,&m,&m); CHKERRQ(ierr);
    ierr = VecCreateSeq(PETSC_COMM_SELF,m,&x); CHKERRQ(ierr);
    PLogObjectParent(pc,x);
    ierr = VecCreateSeq(PETSC_COMM_SELF,m,&y); CHKERRQ(ierr);
    PLogObjectParent(pc,y);

    pc->destroy       = PCDestroy_BJacobi_MPIBDiag;
    pc->setuponblocks = PCSetUpOnBlocks_BJacobi_MPIBDiag;
    pc->apply         = PCApply_BJacobi_MPIBDiag;
    bjac = (PC_BJacobi_MPIBDiag *) PetscMalloc(sizeof(PC_BJacobi_MPIBDiag));CHKPTRQ(bjac);
    PLogObjectMemory(pc,sizeof(PC_BJacobi_MPIBDiag));
    bjac->x           = x;
    bjac->y           = y;
    bjac->matalloc    = matalloc;
    jac->sles         = (SLES *) PetscMalloc( sizeof(SLES) ); CHKPTRQ(jac->sles);
    PLogObjectMemory(pc,sizeof(SLES));
    jac->sles[0]      = sles;
    jac->data         = (void *) bjac;
  } else {
    sles = jac->sles[0];
  } 
  bjac->Amat        = amsub;   
  bjac->Pmat        = pmsub;

  ierr = SLESSetOperators(sles,amsub,pmsub,pc->flag); CHKERRQ(ierr);

  PetscFunctionReturn(0);
}
