      subroutine dsfdfj(n,x,fvec,fjac,ldfjac,task,eps,nint)
      character*(*) task
      integer n,ldfjac,nint
      double precision eps
      double precision x(n),fvec(n),fjac(ldfjac,n)
!     **********
!
!     Subroutine dsfdfj
!
!     This subroutine computes the function and Jacobian matrix of the 
!     Swirling Flow between Disks problem.  This problem arises in
!     the analysis of the steady flow of fluid between two rotating,
!     infinite coaxial disks.
!
!     The swirling flow problem is modeled by the system of ordinary 
!     differential equations 
!
!                eps*f'''' + f*f''' + g*g' = 0,
!                eps*g'' + f*g' + f'*g = 0,
!
!     with boundary conditions
!
!                f(0) = f'(0) = f(1) = f'(1) = 0
!                g(0) = omega0, g(1) = omega1.
!
!     In this formulation eps is the viscosity of the fluid and the
!     speed of the disks are given by omega1 and omega2. The boundary 
!     problem is discretized by a k-stage collocation scheme to obtain 
!     a system of nonlinear equations. 

!     This version of the swirling flow problem uses omega1 = -1 and 
!     omega2 = 1. Other versions can be obtained by changing the 
!     appropriate parameter statement.
!
!     The subroutine statement is:
!
!       subroutine dsfdfj(n,x,fvec,fjac,ldfjac,task,eps,nint)
!
!     where
!
!       n is an integer variable.
!         On entry n is the number of variables. 
!            For the swirling flow problem n must equal 14*nint.
!         On exit n is unchanged.
!
!       x is a double precision array of dimension n. 
!         On entry x specifies the vector x if task = 'F', 'J', or 'FJ'.  
!            Otherwise x need not be specified.
!         On exit x is unchanged if task = 'F', 'J', or 'FJ'. Otherwise
!            x is set according to task.
!
!       fvec is a double precision array of dimension n. 
!         On entry fvec need not be specified.
!         On exit fvec contains the function evaluated at x if 
!            task = 'F' or 'FJ'.
!
!       fjac is a double precision array of dimension (ldfjac,n).
!         On entry fjac need not be specified.
!         On exit fjac contains the Jacobian matrix evaluated at x if
!            task = 'J' or 'FJ'. 
!
!       ldfjac is an integer variable.
!          On entry ldfjac is the leading dimension of fjac.
!          On exit ldfjac is unchanged.
!
!       task is a character variable.
!         On entry task specifies the action of the subroutine:
!
!            task               action
!            ----               ------
!             'F'     Evaluate the function at x.
!             'J'     Evaluate the Jacobian matrix at x.
!             'FJ'    Evaluate the function and the Jacobian at x.
!             'XS'    Set x to the standard starting point xs.
!
!         On exit task is unchanged.
!
!       eps is a double precision variable.
!         On entry eps is the viscosity of the fluid.  
!         On exit eps is unchanged.
!
!       nint is an integer variable. 
!         On entry nint is the number of subintervals in the k-stage 
!            collocation.
!         On exit nint is unchanged.
!
!     MINPACK-2 Project. October 1992.
!     Argonne National Laboratory and University of Minnesota.
!     Brett M. Averick.
!
!     **********
      integer bc,cpts,fdeg,gdeg,mdeg,dim,npi
      parameter (bc=3,cpts=4,fdeg=4,gdeg=2,mdeg=4,dim=mdeg+cpts-1,
     +           npi=2*cpts+gdeg+fdeg)
      double precision omega1,omega2,zero,one 
      parameter (omega1=-1.0d0,omega2=1.0d0,zero=0.0d0,one=1.0d0)

      integer eqn1,eqn2,i,j,k,m,var1,var2
      double precision h,xt,hm,nf,rhoijh
      double precision rho(cpts),dwf(fdeg+1,cpts+fdeg),
     +     dwg(gdeg+1,cpts+gdeg),rhnfhk(cpts,0:dim,0:dim,0:mdeg),
     +     wg(gdeg+1),wf(fdeg+1)

      data (rho(i), i = 1, cpts)
     +     /0.694318413734436035D-1, 0.330009490251541138D0,
     +      0.669990539550781250D0, 0.930568158626556396D0/

!     Check input arguments for errors.

      if (n .ne. 14*nint) then
         task = 'ERROR: N MUST .EQ. 14*NINT'
         return
      endif

!     Initialization.

      h = one/dble(nint)

!     Compute the standard starting point if task = 'XS'.

      if (task .eq. 'XS') then
         do 10 i = 1, n
            x(i) = zero
   10    continue

!        The standard starting point corresponds to the solution 
!        of the swirling flow problem with infinite viscosity.

         xt = zero
         do 20 i = 1, nint
            var1 = (i - 1)*npi + fdeg + cpts
            x(var1+1) = omega1 + (omega2 - omega1)*xt
            x(var1+2) = omega2 - omega1
            xt = xt + h
   20    continue

         return

      endif

!     Store all possible combinations of rho, h, and n factorial.

      hm = one
      do 60 m = 0, mdeg
         do 50 i = 1, cpts
            rhoijh = hm
            do 40 j = 0, dim
               nf = one
               do 30 k = 0, dim
                  rhnfhk(i,j,k,m) = rhoijh/nf
                  nf = nf*dble(k+1)
   30          continue
               rhoijh = rhoijh*rho(i)
   40       continue
   50    continue
         hm = hm*h
   60 continue

!     Evaluate the function if task = 'F', the Jacobian matrix if 
!     task = 'J', or both if task = 'FJ'.

!     Initialize arrays.

      if (task .eq. 'F' .or. task .eq. 'FJ') then
         do 70 j = 1, n
            fvec(j) = zero
   70    continue
      endif
      if (task .eq. 'J' .or. task .eq. 'FJ') then
         do 90 j = 1, n
            do 80 i = 1, n
               fjac(i,j) = zero
  80        continue
  90     continue
      endif
      do 110 k = 1, cpts + fdeg 
         do 100 j = 1, fdeg + 1
            dwf(j,k) = zero
  100    continue
  110 continue
      do 130 k = 1, cpts + gdeg
         do 120 j = 1, gdeg + 1
            dwg(j,k) = zero
  120    continue
  130 continue
                      
!     Set up the boundary equations at t = 0.
!     f(0) = 0, f'(0) = 0, g(0) = omega1.

      if (task .eq. 'F' .or. task .eq. 'FJ') then
         fvec(1) = x(1)
         fvec(2) = x(2)
         fvec(3) = x(cpts+fdeg+1) - omega1
      endif
      if (task .eq. 'J' .or. task .eq. 'FJ') then
         fjac(1,1) = one
         fjac(2,2) = one
         fjac(3,cpts+fdeg+1) = one
      endif

!     Set up the collocation equations.

      do 230 i = 1, nint
         var1 = (i - 1)*npi
         eqn1 = var1 + bc
         var2 = var1 + cpts + fdeg
         eqn2 = eqn1 + cpts
         do 220 k = 1, cpts
            do 160 m = 1, fdeg + 1
               wf(m) = zero
               do 140 j = m, fdeg
                  wf(m) = wf(m) + rhnfhk(k,j-m,j-m,j-m)*x(var1+j)
                  dwf(m,j) = rhnfhk(k,j-m,j-m,j-m)
  140          continue
               do 150 j = 1, cpts
                  wf(m) = wf(m) + x(var1+fdeg+j)*
     +                              rhnfhk(k,fdeg+j-m,fdeg+j-m,fdeg-m+1)
                  dwf(m,fdeg+j) = rhnfhk(k,fdeg+j-m,fdeg+j-m,fdeg-m+1)
  150          continue
  160       continue
            do 190 m = 1, gdeg + 1
               wg(m) = zero
               do 170 j = m, gdeg
                  wg(m) = wg(m) + rhnfhk(k,j-m,j-m,j-m)*x(var2+j)
                  dwg(m,j) =  rhnfhk(k,j-m,j-m,j-m)
  170          continue
               do 180 j = 1, cpts
                  wg(m) = wg(m) + x(var2+gdeg+j)*
     +                              rhnfhk(k,gdeg+j-m,gdeg+j-m,gdeg-m+1)
                  dwg(m,gdeg+j) =  
     +                              rhnfhk(k,gdeg+j-m,gdeg+j-m,gdeg-m+1)
  180          continue
  190       continue
            if (task .eq. 'F' .or. task .eq. 'FJ') then
               fvec(eqn1+k) = eps*wf(5) + wf(4)*wf(1) + wg(2)*wg(1)
               fvec(eqn2+k) = eps*wg(3) + wf(1)*wg(2) - wf(2)*wg(1)
            endif
            if (task .eq. 'J' .or. task .eq. 'FJ') then
               do 200 j = 1, cpts + fdeg
                  fjac(eqn1+k,var1+j) = eps*dwf(5,j) + dwf(4,j)*wf(1) 
     +                                                  + wf(4)*dwf(1,j)
                  fjac(eqn2+k,var1+j) = dwf(1,j)*wg(2) - dwf(2,j)*wg(1)
  200          continue
               do 210 j = 1, cpts + gdeg
                  fjac(eqn1+k,var2+j) = dwg(2,j)*wg(1) + wg(2)*dwg(1,j)
                  fjac(eqn2+k,var2+j) = eps*dwg(3,j) + wf(1)*dwg(2,j) 
     +                                                  - wf(2)*dwg(1,j)
  210             continue
            endif

  220    continue
  230 continue

!     Set up the continuity equations.

      do 340 i = 1, nint - 1
         var1 = (i - 1)*npi
         eqn1 = var1 + bc + 2*cpts
         var2 = var1 + fdeg + cpts
         eqn2 = eqn1 + fdeg
         do 260 m = 1, fdeg
            wf(m) = zero
            do 240 j = m, fdeg
               wf(m) = wf(m) + rhnfhk(1,0,j-m,j-m)*x(var1+j)
               dwf(m,j) = rhnfhk(1,0,j-m,j-m)
  240       continue
            do 250 j = 1, cpts
               wf(m) = wf(m) + rhnfhk(1,0,fdeg+j-m,fdeg-m+1) 
     +                                                   *x(var1+fdeg+j)
               dwf(m,fdeg+j) = rhnfhk(1,0,fdeg+j-m,fdeg-m+1)
  250       continue
  260    continue
         do 290 m = 1, gdeg
            wg(m) = zero
            do 270 j = m, gdeg
               wg(m) = wg(m) + rhnfhk(1,0,j-m,j-m)*x(var2+j)
               dwg(m,j) = rhnfhk(1,0,j-m,j-m)
  270       continue
            do 280 j = 1, cpts
               wg(m) = wg(m) + rhnfhk(1,0,gdeg+j-m,gdeg-m+1)
     +                                                   *x(var2+gdeg+j)
               dwg(m,gdeg+j) = rhnfhk(1,0,gdeg+j-m,gdeg-m+1)
  280       continue
  290    continue
         do 310 m = 1, fdeg
            if (task .eq. 'F' .or. task .eq. 'FJ') then
               fvec(eqn1+m) = x(var1+npi+m) - wf(m)
            endif
            if (task .eq. 'J' .or. task .eq. 'FJ') then
               fjac(eqn1+m,var1+npi+m) = one
               do 300 j = 1, cpts + fdeg
                  fjac(eqn1+m,var1+j) = -dwf(m,j)
  300          continue
            endif
  310    continue
         do 330 m = 1, gdeg
            if (task .eq. 'F' .or. task .eq. 'FJ') then
               fvec(eqn2+m) = x(var2+npi+m) - wg(m)
            endif
            if (task .eq. 'J' .or. task .eq. 'FJ') then
               fjac(eqn2+m,var2+npi+m) = one
               do 320 j = 1, cpts + gdeg
                  fjac(eqn2+m,var2+j) = -dwg(m,j)
  320          continue
            endif
  330    continue
  340 continue

!     Prepare for setting up the boundary conditions at t = 1.

      var1 = n - npi
      do 370 m = 1, fdeg + 1
         wf(m) = zero
         do 350 j = m, fdeg 
            wf(m) = wf(m) + rhnfhk(1,0,j-m,j-m)*x(var1+j)
            dwf(m,j) = rhnfhk(1,0,j-m,j-m)
  350    continue
         do 360 j = 1, cpts
            wf(m) = wf(m) + rhnfhk(1,0,fdeg+j-m,fdeg-m+1)
     +                                                   *x(var1+fdeg+j)
            dwf(m,fdeg+j) = rhnfhk(1,0,fdeg+j-m,fdeg-m+1)
  360    continue
  370 continue
      var2 = var1 + fdeg + cpts
      do 400 m = 1, gdeg + 1
         wg(m) = zero
         do 380 j = m, gdeg
            wg(m) = wg(m) + rhnfhk(1,0,j-m,j-m)*x(var2+j)
            dwg(m,j) =  rhnfhk(1,0,j-m,j-m)
  380    continue
         do 390 j = 1, cpts
            wg(m) = wg(m) + rhnfhk(1,0,gdeg+j-m,gdeg-m+1)
     +                                                   *x(var2+gdeg+j)
            dwg(m,gdeg+j) = rhnfhk(1,0,gdeg+j-m,gdeg-m+1)
  390    continue
  400 continue

!     Set up the boundary equations at t = 1.
!     f(1) = 0, f'(1) = 0, g(1) = omega2.

      if (task .eq. 'F' .or. task .eq. 'FJ') then
         fvec(n-2) = wf(1) 
         fvec(n-1) = wf(2)
         fvec(n) = wg(1) - omega2
      endif
      if (task .eq. 'J' .or. task .eq. 'FJ') then
         do 410 j = 1, cpts + fdeg
            fjac(n-2,var1+j) = dwf(1,j)
            fjac(n-1,var1+j) = dwf(2,j)
  410    continue
         do 420 j = 1, cpts + gdeg
            fjac(n,var2+j) = dwg(1,j)
  420    continue
      endif

      return

      end
