    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     *  ocpp : Occam pre-processor with comment-decorated output   *
     *  Version 3.002, September 11 1997                               *
     *  (c) M. I. Barlow 1995,96,97 (released to PD)               *
     *                                                             *
     *  THIS FILE WAS CREATED WITH A FOLDING EDITOR:               *
     *                "Look on my works without one and despair"   *
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*{{{  includes */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#ifdef __TURBOC__
  #include <alloc.h>
  #include <time.h>
#else
  #ifdef _ICC
    #include <ctype.h>
    #include <time.h>
  #else
    #include <malloc.h>
    #include <sys/time.h>
 #endif
#endif

/*}}}   */
/*{{{  defines */

/*{{{  version */

#define VERSION       "3.002"
#define VERSION_DATE  "September 11 1997"
#define RELEASE_STATE "(Released to PD)"

/*}}}   */
/*{{{  truth */

#define TRUE  1
#define FALSE 0

/*}}}   */
/*{{{  debugging */

#define DEBUG_LINE     FALSE
#define DEBUG_LIST     FALSE
#define DEBUG_PARSER   FALSE
#define DEBUG_SIMS     FALSE
#define DEBUG_STACK    FALSE
#define DEBUG_SEARCH   FALSE
#define DEBUG_RXPARSE  FALSE
#define DEBUG_RXSETS   FALSE
#define DEBUG_RXCAPTS  FALSE
#define DEBUG_FREE     TRUE  /* left in for safety...                    */
			     /* <turns round *3 & spits over L shoulder> */

/*}}}   */
/*{{{  limits */

#define MAXFNAME   256
#define MAXLINE    512
#define MAXTPNEST   32

#define TABWIDTH     8

#ifdef __TURBOC__
  #define MINCORE   ((unsigned long)4096)
#endif

/*}}}   */
/*{{{  OS dependencies */

#ifdef __MSDOS__
  #define SWITCH '/'
  #define DIRSEP '\\'
  #define PATHSEPS "; \n"
#else
  #define SWITCH '-'
  #define DIRSEP '/'
  #define PATHSEPS ": \n"
#endif

/*}}}   */
/*{{{  default tokens */

#define DFLT_RESV_CMT    "--:::  "
#define DFLT_PRIV_CMT    "--///"
#define DFLT_OPEN_SUBST  "{"
#define DFLT_CLOSE_SUBST "}"
#define DFLT_SPC_TOKEN   "_"
#define DFLT_SRCH_PATH   "ISEARCH"
#define DFLT_COMP_NAME   "oc"
#define DFLT_FILE_EXT    "occ"

/*}}}   */
/*{{{  subst modifiers */

#define TO_UPPER_MOD   '+'
#define TO_LOWER_MOD   '-'
#define BLANK_OUT_MOD  '_'

/*}}}   */
/*{{{  acannonical template token */

#define ACANNONICAL_TPLT "..."

/*}}}   */
/*{{{  regular expression stuff */

#define ANY       '.'
#define LITCHAR   'L'
#define LITCASE   'I'
#define CCL       '['
#define CCLEND    ']'
#define NEGATE    '^'
#define NCCL      '!'
#define CLOSURE   '*'
#define PLUS      '+'
#define QUERY     '?'
#define ESCAPE    '\\'
#define RANGE     '{'
#define RANEND    '}'
#define COMMA     ','
#define OR_SYM    '|'
#define PAREN     '('
#define PAREND    ')'
#define SUBST_IN  'S'
#define SUBST_CI  'C'
#define CAPTURE0  '0'
#define CAPTURE1  '1'
#define CAPTURE2  '2'
#define CAPTURE3  '3'
#define CAPTURE4  '4'
#define CAPTURE5  '5'
#define CAPTURE6  '6'
#define CAPTURE7  '7'
#define CAPTURE8  '8'
#define CAPTURE9  '9'
#define CASEOFF   '<'
#define CASEON    '>'

/*}}}   */

/*}}}   */
/*{{{  typedefs */

/*{{{  MacDefCode */

typedef enum
{
    M_undef,
    M_cmd_line,
    M_env_var,
    M_immed,
    M_param,
    M_retval,
    M_special
}
MacDefCode;

/*}}}   */
/*{{{  Macro */

typedef struct mac_
{
    char        *name;
    char        *value;
    MacDefCode   defcode;
    char        *src;
    long         in_line;
    long         out_line;
    struct mac_ *next;
}
Macro;

/*}}}   */
/*{{{  Operator */

typedef enum /* order is important! */
{
    EoF = 0,

    Null,
    Blank,
    Vanilla,

    Ifdef,
    Ifndef,
    If,
    Ifn,

    Else,
    Endif,

    Define,
    Derive,
    Undef,

    Mdump,
    Memo,
    UseErr,
    Shell,

    Ordef,
    Orndef,
    Or,
    Orn,

    Anddef,
    Andndef,
    And,
    Andn,

    Subst,

    Indent,
    Outdent,

    Defsfrom,
    InterpC,
    Template,
    Return,

    Include,
    Use,
    Import,

    Extract,
    Replace,

    Unknown
}
Operator;

/*}}}   */
/*{{{  Directive */

typedef struct
{
    char     *token;
    int       tok_len;
    Operator  op;
}
Directive;

/*}}}   */
/*{{{  OcType */

typedef enum
{
    Int,
    Byte,
    ByteArray,
    Boolean,
    Real32,
    Real64,
    Int32
}
OcType;

/*}}}   */
/*{{{  ErrorCode */

typedef enum
{
    Ignore  = -1,
    Info    =  0,
    Warn    =  1,
    Error   =  2,
    Severe  =  3,
    Fatal   =  4,
    Unwind  =  5,
    UserDef =  6
}
ErrorCode;

/*}}}   */
/*{{{  SubMod */

typedef enum
{
    NoModifier,
    ToUpper,
    ToLower,
    InitialCap,
    BlankOut
}
SubMod;

/*}}}   */
/*{{{  ParseResult */

typedef enum
{
    ParseOK,
    ParseFAIL,
    ParseRETURN
}
ParseResult;
/*}}}   */
/*{{{  RxpToken */

typedef struct rxptoken
{
    char                              tok;
    union
    {
	char              lchar;
	char             *bitmap;
	char              limit[2];
	int               subst_no;
	struct rxptoken  *or_patn;
    }                                 data;
    struct rxptoken                  *next;
}
RxpToken;

/*}}}   */
/*{{{  RxpError */

typedef enum
{
    RxpOK       = 0,
    AllocFail,
    BadSet,
    BadClosure,
    BadRange,
    BigRange,
    BadGroup,
    BadEscape,
    BadSubstIn,
    Corrupt,
    NeverSee
}
RxpError;

/*}}}   */

/*}}}   */
/*{{{  constants */

/*{{{  Turbo C stack sizing */

#ifdef __TURBOC__
  extern unsigned _stklen = TC_STACK_SIZE;
#endif

/*}}}   */
/*{{{  help text */

const char *help_text =

"\n"
"ocpp : Occam pre-processor : "
#ifdef _ICC
  #ifdef __MSDOS__
  "Transputer (PC)"
  #else
  "Transputer (UNIX)"
  #endif
#else
  #ifdef __MSDOS__
    "PC"
  #else
    "UNIX"
  #endif
#endif

" Version " VERSION ", " VERSION_DATE "\n"
"(c) M.I.Barlow 1995,96,97 " RELEASE_STATE "\n"
"\n"
#ifdef __MSDOS__
  "Usage:\x85ocpp {/option} <filename> | -\x93( - denotes <stdin>)\n"
  "\x8bocpp {/option} /PP <filename> {compiler_opts}\n"
#else
  "Usage:\x85ocpp {-option} <filename> | -\x93( - denotes <stdin>)\n"
  "\x8bocpp {-option} -PP <filename> {compiler_opts}\n"
#endif
"Options:\n"
"\n"
"\x82O {filename}\x86Specify an output file (default is <stdout>).\n"
"\x82" "D {symbol[=val]}\x82" "Define a macro symbol and optional value (case sensitive).\n"
"\x82" "DE {env_var}\x86Specify an environment variable containing macro(s).\n"
"\x82" "DF {filename}\x85Specify a macro definition indirect file.\n"
"\x82U {symbol}\x88Undefine a macro symbol.\n"
"\x82RMC\x8fRemove existing reserved comments then terminate.\n"
"\x82MIP\x8fModify input file in-place (over-rides O).\n"
"\x82" "DBG\x8fList macros to <stderr>, annotate directives in output.\n"
"\x82IRV\x8fIrrevocable; remove lines that would have been commented.\n"
"\x82T2\x90Treat INT as 16-bit when interpreting (default is 32).\n"
"\x82PP {filename}\x85Pre-process, then chain to compiler passing rest of line.\n"
"\n"
"Directives:\x82#DEFINE\x82#IFDEF\x83#ORDEF\x83#ANDDEF\x83#ELSE\x85#MEMO\x83#DEFSFROM\n"
"\x8d#UNDEF\x83#IFNDEF\x82#ORNDEF\x82#ANDNDEF\x82#ENDIF\x84#ERROR\x82#INTERPRET\n"
"\x83#EXTRACT\x82#DERIVE\x82#IF\x86#OR\x86#AND\x86#INDENT\x83#SHELL\x82#TEMPLATE\n"
"\x83#REPLACE\x82#SUBST\x83#IFN\x85#ORN\x85#ANDN\x85#OUTDENT\x82#MDUMP\x82#RETURN\n"
;

/*}}}   */
/*{{{  extra help */

const char *extra_help =

"\n"
"Directive syntax conventions:\n"
"\n"
"\x82<user supplied>\x83[optional]\x83(zero or more)\x83{subject to substitution}\n"
"\n"
"Detailed directive syntax:\n"
"\n"
"\x82#DEFINE <symbol>[=<value>]\x89#UNDEF <symbol>\n"
"\x82#DERIVE {<symbol>[=<value>]}\x87#SUBST {<source text>}\n"
"\x82#IFDEF <symbol>\x94#IFNDEF <symbol>\n"
"\x82#IF <symbol>=<pattern>\x8d#IFN <symbol>=<pattern>\n"
"\x82#ORDEF <symbol>\x94#ORNDEF <symbol>\n"
"\x82#ANDDEF <symbol>\x93#ANDNDEF <symbol>\n"
"\x82#OR <symbol>=<pattern>\x8d#ORN <symbol>=<pattern>\n"
"\x82#AND <symbol>=<pattern>\x8c#ANDN <symbol>=<pattern>\n"
"\x82#ELSE\x9e#ENDIF\n"
"\x82#INDENT\x9c#OUTDENT\n"
"\x82#MEMO {<message text>}\x8d#ERROR {[<code>]<message text>}\n"
"\x82#SHELL {<command line>}\x8c#MDUMP [\"<dump file>\"]\n"
"\x82#DEFSFROM \"<definition file>\"\x86#INTERPRET \"<C header file>\"\n"
"\n"
"\x82#TEMPLATE [<symbol>=]\"<template file>\" [...] ({<symbol>[=<value>]})\n"
"\x82#RETURN {<return value text>}\n"
"\n"
"\x82#EXTRACT <symbol>=<d><regexp><d>[<d><replacement><d>]{<source text>}\n"
"\x82#REPLACE <symbol>=<d><regexp><d>[<d><replacement><d>]{<source text>}\n"
"\n"
"Patterns for the #(IF|OR|AND)[N] directives:\n"
"\n"
"\x84<pattern> ::= <value>\x92Simple exact matching\n"
"\x84<pattern> ::= |<regexp>(|<regexp>)[|]\x82One or more regular expressions\n"
"\n"
"Regular expression elements:\n"
"\n"
"\x84.\x87matches any character\x84\\<ch>\x84matches <ch> exactly\n"
"\x84[abc]\x83matches any in set\x87[^abc]\x83matches complement of set\n"
"\x84*\x87zero or more repeats\x85+\x88one or more repeats\n"
"\x84?\x87zero or one\x8e<ch>\x85" "all others are literal\n"
"\n"
"\x84(<expr0>|<expr1>|...)\x83" "a group; matches one of it\'s sub-expressions\n"
"\x84\\g\x82(where g = [1-9])\x83matches the string that matched the g\'th group\n"
"\x84\\<<expr>\\>\x8e<expr> is matched in a case-insensitive manner\n"
"\n"
"\x84\\{<n>\\}\x87" "exactly <n> repeats\n"
"\x84\\{<n>,\\}\x86<n> or more repeats\n"
"\x84\\{<n>,<m>\\}\x83" "between <n> and <m> repeats\n"
"\n"
"\x84Sub-expression groups are repeatable and nestable. Sets may contain\n"
"\x84ranges, Eg: [0-9] matches any digit. If \']\' is needed it must be\n"
"\x84the first character; \'-\' must be first or last.\n"
"\n"
"Substitution modifiers:\n"
"\n"
"\x84+\x86" "force uppercase\x8b-\x88" "force lowercase\n"
"\x84+-\x85" "force initial capital\x85_\x88" "blank out\n"
"\n"
"\x84<n>\x84just character <n>\x88<n>,\x85" "characters from <n> onward\n"
"\x84,<m>\x83" "characters up to <m>\x86<n>,<m>\x82" "characters from <n> to <m>\n"
"\n"
"Valid C constant mappings during interpretation:\n"
"\n"
"\x84" "decimal integer literal\x86=>\x82INT | INT32 (with T2 option)\n"
"\x84hexadecimal integer literal\x82=>\x82INT | INT32 (with T2 option)\n"
"\x84" "double literal\x8f=>\x82REAL64\n"
"\x84" "float literal\x90=>\x82REAL32\n"
"\x84" "character literal\x8c=>\x82" "BYTE\n"
"\x84string literal\x8f=>\x82[]BYTE\n"
"\x84truth value\x92=>\x82" "BOOL\n"
"\n"
"Internal macros types:\n"
"\n"
"\x83R = readable, W = writeable, RW = both\n"
"\x83T = default TRUE, F = default FALSE, ? = command dependent\n"
"\x82<n> = maximum length of string.\n"
"\n"
"Special internal macros:\n"
"\n"
"\x82NAME\x91TYPE\x85PURPOSE\xa0" "DEFAULT\n"
"\n"
"\xff" "AddCrToLf\x89WT\x86" "affects C string interpretation\n"
"\xff" "CallersName\x86R\x82" "255\x83name of file that called current template\n"
"\xff" "CallLine\x89R\x83" "10\x83line number of call to current template\n"
"\xff" "CloseSubstToken\x83W\x83" "3\x83sets substitution end delimiter\x8a(\"}\")\n"
"\xff" "CompilerName\x86W 255\x83sets compiler name for PP option\x88(\"oc\")\n"
"\xff" "DebugOption\x86RW?\x86state of DBG command line option\n"
"\xff" "DefaultExtn\x87W\x83" "7\x83" "default input file extension\x8b(\"occ\")\n"
"\xff" "Dir\x8eR\x84" "1\x83OS\'s directory separator character\n"
"\xff" "Env??????\x88R\x88value of environment variable named ??????\n"
"\xff" "ExpandFileNames\x83WT\x86" "affects expansion of env vars in filespecs\n"
"\xff" "FoldUpHashIFs\x85W?\x86" "affects folding of #IF[N][DEF] ... #ENDIF\n"
"\xff" "InLineInsert\x86WF\x86" "eliminates auto-generated include files\n"
"\xff" "InputLine\x88R\x83" "10\x83" "current input file line number\n"
"\xff" "InputName\x88R\x82" "255\x83" "current input file or template name\n"
"\xff" "IntIsInt16\x87RW?\x86state of T2 command line option\n"
"\xff" "Irrevocable\x86RW?\x86state of IRV command line option\n"
"\xff" "MacValSpcToken\x84W\x83" "3\x83placeholder for space in macro value\x85(\"_\")\n"
"\xff" "OpenSubstToken\x84W\x83" "3\x83sets substitution start delimiter\x88(\"{\")\n"
"\xff" "Opt\x8eR\x84" "1\x83OS\'s command option character\n"
"\xff" "OutputName\x87R\x82" "255\x83" "current output file name\n"
"\xff" "PrivateComment\x84W\x82" "15\x83sets private comment style\x8b(\"--///\")\n"
"\xff" "Psep\x8dR\x84" "1\x83OS\'s path separator character\n"
"\xff" "RMCafterPP\x88WT\x86remove comments after compilation\n"
"\xff" "ReservedComment\x83W\x82" "15\x83sets reserved comment style\x88(\"--:::\x82\")\n"
"\xff" "RootInputName\x84R\x82" "255\x83original command line input file\n"
"\xff" "SearchPathVar\x85W\x82" "32\x83" "Env var containing search path\x85(\"ISEARCH\")\n"
"\xff" "StartTime\x88R\x84" "8\x83hours (24), minutes and seconds\n"
"\xff" "StdHeader\x89WT\x86" "affects auto-generated include files\n"
"\xff" "StripPrivates\x85WF\x86remove private comments during RMC\n"
"\xff" "SubstAlways\x87WF\x86" "controls substitution on source text lines\n"
"\xff" "TodaysDate\x87R\x83" "11\x83" "day-of-month, month and year\n"
"\xff" "VersionNo\x88R\x84" "5\x83ocpp version number\n"
"\xff" "WriteTabs\x89W?\x86" "affects use of ASCII 9 in output indents\n"
;

/*}}}   */
/*{{{  switch list */

const char *cm_switch[] = /* order is important! */
{
    "O",
    "DE",
    "DF",
    "D",
    "RMC",
    "MIP",
    "DBG",
    "T2",
    "IRV",
    "U",
    "Z",
    "PP",
    (char *)0,
};

/*}}}   */
/*{{{  directive list */

const Directive known[] = /* must be alpha sorted, length first */
{
    { "ANDDEF",        6, Anddef   },
    { "ANDNDEF",       7, Andndef  },
    { "ANDN",          4, Andn     },
    { "AND",           3, And      },
    { "DEFINE",        6, Define   },
    { "DEFSFROM",      8, Defsfrom },
    { "DERIVE",        6, Derive   },
    { "ELSE",          4, Else     },
    { "ENDIF",         5, Endif    },
    { "ERROR",         5, UseErr   },
    { "EXTRACT",       7, Extract  },
    { "IFDEF",         5, Ifdef    },
    { "IFNDEF",        6, Ifndef   },
    { "IFN",           3, Ifn      },
    { "IF",            2, If       },
    { "IMPORT",        6, Import   },
    { "INCLUDE",       7, Include  },
    { "INDENT",        6, Indent   },
    { "INTERPRET",     9, InterpC  },
    { "MDUMP",         5, Mdump    },
    { "MEMO",          4, Memo     },
    { "ORDEF",         5, Ordef    },
    { "ORNDEF",        6, Orndef   },
    { "ORN",           3, Orn      },
    { "OR",            2, Or       },
    { "OUTDENT",       7, Outdent  },
    { "REPLACE",       7, Replace  },
    { "RETURN",        6, Return   },
    { "SUBST",         5, Subst    },
    { "SHELL",         5, Shell    },
    { "TEMPLATE",      8, Template },
    { "UNDEF",         5, Undef    },
    { "USE",           3, Use      },

    { "",              0, Unknown  }
};

const int n_known = sizeof(known) / sizeof(Directive);

/*}}}   */
/*{{{  condition code mask bits */

const int Cond_DEF  = 1;
const int Cond_ETC  = 2;
const int Cond_IF   = 4;

/*}}}   */
/*{{{  occam type names */

const char *typename[] = /* order is important! */
{
    "INT",
    "BYTE",
    "[]BYTE",
    "BOOL",
    "REAL32",
    "REAL64",
    "INT32",
};

/*}}}   */
/*{{{  acanonical template token */

const char acanonical_tplt[] = ACANNONICAL_TPLT;

/*}}}   */
/*{{{  pipe names */

const char in_pipename[] = "<stdin>";
const char out_pipename[] = "<stdout>";

/*}}}   */
/*{{{  regular expression error messages */

const char *regexp_err[] =
{
"",
"Out of memory parsing",
"Malformed character set in",
"Invalid closure (*+?\\{m,n\\}) in",
"Malformed closure range (\\{m,n\\}) in",
"Range value (\\{m,n\\}) too big in",
"Unterminated group ((x|y)) in",
"Invalid escape (\\) at end of",
"Invalid substitution (\\0) in",
};

/*}}}   */

#if DEBUG_PARSER
  /*{{{  line type names */
  
  const char* line_type_name[] =
  {
      "EoF     ",
      "Null    ",
      "Blank   ",
      "Vanilla ",
      "Ifdef   ",
      "Ifndef  ",
      "If      ",
      "Ifn     ",
      "Else    ",
      "Endif   ",
      "Define  ",
      "Derive  ",
      "Undef   ",
      "Mdump   ",
      "Memo    ",
      "UseErr  ",
      "Shell   ",
      "Ordef   ",
      "Orndef  ",
      "Or      ",
      "Orn     ",
      "Anddef  ",
      "Andndef ",
      "And     ",
      "Andn    ",
      "Subst   ",
      "Indent  ",
      "Outdent ",
      "Defsfrom",
      "InterpC ",
      "Template",
      "Return  ",
      "Include ",
      "Use     ",
      "Import  ",
      "Extract ",
      "Replace ",
      "Unknown "
  };
  
  /*}}}   */
#endif

/*}}}   */
/*{{{  globals */

/*{{{  Macro table */

Macro *mac_table_head = (Macro *)0;

/*}}}   */
/*{{{  error count */

int n_err = 0;

/*}}}   */
/*{{{  files */

char temp_file[MAXFNAME] = "\0";

char *base_file = (char *)0;
char *in_file = (char *)0;
char *out_file = (char *)0;

FILE *in = (FILE *)0;
FILE *out = (FILE *)0;

unsigned long line_no = 0L;
unsigned long out_line_no = 0L;

/*}}}   */
/*{{{  level counter */

int rec_level = -1;

/*}}}   */
/*{{{  templates */

char  tplt_retvalue[MAXLINE];
char  tplt_name[MAXFNAME];
char *caller_name;
long  caller_line;
int   tplt_level = 0;
int   tplt_count = 0;

/*}}}   */
/*{{{  miscelaneous flags */

int annotate = FALSE;
int cl_annotate = FALSE;
int int_16 = FALSE;
int cl_int_16 = FALSE;
int irrevocable = FALSE;
int cl_irrevocable = FALSE;
int add_cr_to_lf = TRUE;
int do_fname_expand = TRUE;
int in_line_insert = FALSE;
int use_std_header = TRUE;
int mod_in_place = FALSE;
int always_subst = FALSE;
int write_tabs = 2;
int fold_ifs = 2;
int remove_priv_cmt = FALSE;
int chain_compiler = FALSE;
int clean_after_chain = TRUE;
int compile_arg = 0;

/*}}}   */
/*{{{  token strings */

char resv_cmt[16]        = DFLT_RESV_CMT;
char priv_cmt[16]        = DFLT_PRIV_CMT;
char open_subst[4]       = DFLT_OPEN_SUBST;
char close_subst[4]      = DFLT_CLOSE_SUBST;
char val_sp_esc[4]       = DFLT_SPC_TOKEN;

char search_path_var[33] = DFLT_SRCH_PATH;
char compiler_name[256]  = DFLT_COMP_NAME;
char std_file_ext[8]     = DFLT_FILE_EXT;

int resv_cmt_len         = (sizeof(DFLT_RESV_CMT) / sizeof(char)) - 1;
int priv_cmt_len         = (sizeof(DFLT_PRIV_CMT) / sizeof(char)) - 1;
int open_subst_len       = (sizeof(DFLT_OPEN_SUBST) / sizeof(char)) - 1;
int close_subst_len      = (sizeof(DFLT_CLOSE_SUBST) / sizeof(char)) - 1;
int val_sp_esc_len       = (sizeof(DFLT_SPC_TOKEN) / sizeof(char)) - 1;

/*}}}   */
/*{{{  indentation */

int global_indent = 0;

/*}}}   */
/*{{{  time */

char *start_time;

/*}}}   */
/*{{{  return codes */

int return_code   = 0;
int user_ret_code = 2;

/*}}}   */

/*}}}   */
/*{{{  prototypes */

ParseResult
parse(int            encomment,
      int            force,
      int            pre_indent,
      int           *why,
      unsigned long  backref,
      unsigned long  backref_in,
      char          *depends_on  );

char *
rxp_alt_match( char      *lin,
	       RxpToken  *pat,
	       int       *which,
	       char      *capture[9] );

/*}}}   */
/*{{{  functions */

#ifdef __TURBOC__
  /*{{{  stack_near_full */
  
  int
  stack_near_full(int size)
  {
      #if __COMPACT__ | __LARGE__ | __HUGE__
      int mark;
      return (((unsigned long)(&mark) & (unsigned long)0x0000ffff) < (MINCORE >> size));
      #else
      return (coreleft() < (MINCORE >> size));
      #endif
  }
  
  /*}}}   */
#else
  /*{{{  strupr */
  
  char *
  strupr(char *txt)
  {
      char *cp = txt;
      while (*cp)
      {
	  if (islower(*cp)) *cp &= 0xdf;
	  cp++;
      }
      return (txt);
  }
  
  /*}}}   */
  /*{{{  strnicmp */
  
  int
  strnicmp(const char *s0, const char *s1, size_t len)
  {
      char *p0 = (char *)s0;
      char *p1 = (char *)s1;
  
      while (len && *p0 && *p1)
      {
	  int c0 = islower(*p0) ? *p0 & 0xdf : *p0;
	  int c1 = islower(*p1) ? *p1 & 0xdf : *p1;
  
	  if (c0 != c1)
	      return (c0 - c1);
	  p0++;
	  p1++;
	  len--;
      }
      return (0);
  }
  
  /*}}}   */
#endif

#if DEBUG_LIST
  /*{{{  show macro */
  
  Macro *
  show_macro(Macro *mp)
  {
      printf("+ -------------------------- Macro --\n");
      printf("| at:     %.8lX\n",(unsigned long)mp);
      printf("| name:   %s\n",mp->name);
      if (mp->value)
	  printf("| value:  %s\n",mp->value);
      else
	  printf("| value:  <null>\n");
      printf("| state:  ");
      switch (mp->defcode)
      {
	  case M_undef:
	      printf("M_undef\n");
	      break;
	  case M_cmd_line:
	      printf("M_cmd_line\n");
	      break;
	  case M_env_var:
	      printf("M_env_var\n");
	      break;
	  case M_immed:
	      printf("M_immed\n");
	      break;
	  default:
	      printf("<unknown>\n");
	      break;
      }
      printf("| next:   %.8lX\n",(unsigned long)mp->next);
      if (!(mp->next))
	  printf("+ -----------------------------------\n");
      return (mp->next);
  }
  
  /*}}}   */
  /*{{{  show_macro_list */
  
  void
  show_macro_list (void)
  {
      Macro *mp = mac_table_head;
  
      while (mp)
	  mp = show_macro(mp);
  }
  
  /*}}}   */
#endif

/*{{{  error */

void
error(ErrorCode level, char *file, unsigned long line, char *msg, ... )
{
    if (level != Ignore)
    {
	const char *severity[] = { "Information",
				   "Warning",
				   "Error",
				   "Serious",
				   "Fatal",
				   "Serious",
				   "Serious"     };
	va_list argp;

	if (line)
	    fprintf(stderr,"%s-ocpp-%s(%ld)- ",severity[(int)level],file,line);
	else
	    fprintf(stderr,"%s-ocpp- ",severity[(int)level]);
	va_start(argp,msg);
	vfprintf(stderr,msg,argp);
	va_end(argp);
	fprintf(stderr,"\n");
	fflush(stderr);

	n_err += (level > Warn);
	if (level == Error) return_code = 1;
	if (level == Severe) exit(2);
	if (level == Fatal)  exit(3);
	if (level == UserDef)  exit(user_ret_code);
    }
}

/*}}}   */
/*{{{  mem_error */

void
mem_error(char *msg, ... )
{
     const char head[] = "Insufficient memory to reliably ";
     const int offset = sizeof(head)/sizeof(char);

     char buf[MAXFNAME];
     va_list argp;

     strcpy(buf,head);
     va_start(argp,msg);
     vsprintf(buf+offset-1,msg,argp);
     va_end(argp);
     error(Fatal,in_file,line_no,buf);
}

/*}}}   */
/*{{{  disaster */

void
disaster(int no)
{
    error(Fatal,
	  in_file,
	  line_no,
	  "Ocpp internal state corrupt, please report code <%d>",no);
}

/*}}}   */
/*{{{  FREE */

#if DEBUG_FREE

  void
  do_free(int locus, void **ptr)
  {
      if (*ptr)
      {
	  free(*ptr);
	  *ptr = (void *)0;   /* so we catch 'double free' error*/
      }
      else
	  error(Fatal,
		in_file,
		line_no,
		"2%.2d Ocpp memory de-allocation error, please report",
		locus                                                   );
  }

  #define FREE(loc,ptr)  do_free(loc,(void **)(&(ptr)))

#else

  #define FREE(loc,ptr)  free(ptr)

#endif

/*}}}   */
/*{{{  do_cleanup */

void
do_cleanup(void)
{
    if (temp_file) remove(temp_file);
    if (n_err) fprintf(stderr,
		       "%d error%s found%s\n",
		       n_err,
		       (n_err > 1) ? "s" : "",
		       in ? " in source" : "");
}

/*}}}   */
/*{{{  regular expression stuff */

#if DEBUG_RXPARSE
  /*{{{  dump_rxp */
  
  void dump_rxp(RxpToken *tptr)
  {
      printf("{%p=%c,",tptr,tptr->tok);
      switch (tptr->tok)
      {
	  case ANY:
	      break;
	  case LITCHAR:
	  case LITCASE:
	      printf("\'%c\'",tptr->data.lchar);
	      break;
	  case CCL:
	  case NCCL:
	      printf("%p",tptr->data.bitmap);
	      break;
	  case OR_SYM:
	  case PAREN:
	  case CAPTURE0:
	  case CAPTURE1:
	  case CAPTURE2:
	  case CAPTURE3:
	  case CAPTURE4:
	  case CAPTURE5:
	  case CAPTURE6:
	  case CAPTURE7:
	  case CAPTURE8:
	  case CAPTURE9:
	      printf("%p",tptr->data.or_patn);
	      break;
	  case SUBST_IN:
	  case SUBST_CI:
	      printf("-%d-",tptr->data.subst_no);
	      break;
	  case CLOSURE:
	      printf("[%d,%d]",tptr->data.limit[0],tptr->data.limit[1]);
	      break;
	  default:
	      printf("!!!ERROR!!!");
      }
      printf(",%p}",tptr->next);
  }
  
  /*}}}   */
  /*{{{  dump_rxp_list */
  
  void dump_rxp_list(RxpToken *tptr)
  {
      int i = 32;
      printf("<<");
      while (tptr && i)
      {
	  dump_rxp(tptr);
	  tptr = tptr->next;
	  i--;
      }
      if (i)
	  printf(">>\n");
      else
	  printf("...\n");
  }
  
  /*}}}   */
#endif

/*{{{  rxp_destroy_captures */

void rxp_destroy_captures(char *capture[9])
{
    int i;

    for (i = 0; i < 9; ++i)
	if (capture[i])
	    FREE(24,capture[i]);
}

/*}}}   */
/*{{{  rxp_init_captures */

void rxp_init_captures(char *capture[9])
{
    int i;

    for (i = 0; i < 9; ++i)
	capture[i] = (char *)0;
}

/*}}}   */
/*{{{  rxp_part_match */

int
rxp_part_match ( char      **linp,
		 RxpToken   *pat,
		 char       *capture[9] )
{
    int advance = -1;

    if (**linp)
    {
	switch (pat->tok)
	{
	case LITCHAR:
	    if (**linp == pat->data.lchar)
		advance = 1;
	    break;
	case LITCASE:
	    {
	       int ch = **linp;

	       if (islower(ch))
		   ch &= 0xdf;
	       if (ch == pat->data.lchar)
		   advance = 1;
	    }
	    break;
	case ANY:
	    if (**linp)
		advance = 1;
	    break;
	case CCL:
	    {
		unsigned i = **linp;
		if ((*((pat->data.bitmap) + (i >> 3)) & (1 << (i & 0x07))) != 0)
		    advance = 1;
	    }
	    break;
	case NCCL:
	    {
		unsigned i = **linp;
		if ((*((pat->data.bitmap) + (i >> 3)) & (1 << (i & 0x07))) == 0)
		    advance = 1;
	    }
	    break;
	case CAPTURE0:
	case CAPTURE1:
	case CAPTURE2:
	case CAPTURE3:
	case CAPTURE4:
	case CAPTURE5:
	case CAPTURE6:
	case CAPTURE7:
	case CAPTURE8:
	case CAPTURE9:
	case PAREN:
	    {
		/*{{{  match the sub-expression and perhaps capture it */
		
		char *end, *start = *linp;
		int dummy = -1;
		
		#ifdef __TURBOC__
		  /*{{{  check stack before recursing */
		  
		  if (stack_near_full(3))
		      mem_error("test regular expression");
		  
		  /*}}}   */
		#endif
		
		end = rxp_alt_match( start,
				     pat->data.or_patn,
				     &dummy,
				     capture            );
		if (end)
		{
		    advance = (end - start + 1);
		
		    if ((pat->tok != PAREN) && !capture[pat->tok - CAPTURE0])
		    {
			char *cap_buf = malloc(advance + 3);
		
			if (cap_buf)
			{
			    strncpy(cap_buf + 2,start,advance);
			    *cap_buf = advance & 0xFF;
			    *(cap_buf + 1) = (advance >> 8) & 0xFF;
			    *(cap_buf + advance + 2) = '\0';
			    capture[pat->tok - CAPTURE0] = cap_buf;
			}
			else
			    mem_error("test regular expression");
		    }
		}
		
		/*}}}   */
	    }
	    break;
	case SUBST_IN:
	case SUBST_CI:
	    {
		char *sub = capture[pat->data.subst_no];
		unsigned len;

		if (sub)
		{
		    len = *sub | (*(sub+1) << 8);
		    if ((pat->tok == SUBST_IN) && !strncmp(*linp,sub+2,len))
			advance = len;
		    else if (!strnicmp(*linp,sub+2,len))
			advance = len;
		}
		else
		    advance = 0;
	    }
	    break;
	default:
	    disaster(101);
	}
    }
    if (advance >= 0)       
	*linp += advance;
    return (++advance);
}

/*}}}   */
/*{{{  rxp_hit_match */

char *
rxp_hit_match( char      *lin,
	       RxpToken  *pat,
	       char      *capture[9] )
{
    char  *bocl, *rval, *strstart;

    strstart = lin;
    if (pat == (RxpToken *)0)
    {
	return ((char *)0);
    }

    while (pat)
    {
	#ifdef __TURBOC__
	  /*{{{  check stack before recursing */
	  
	  if (stack_near_full(3))
	      mem_error("test regular expression");
	  
	  /*}}}   */
	#endif

	if ((pat->tok == CLOSURE) && pat->next)
	{
	    /*{{{  seek and count matches */
	    int min = pat->data.limit[0];
	    int max = pat->data.limit[1];
	    int count = 0;
	    
	    pat = pat->next;
	    bocl = lin;
	    while (*lin && rxp_part_match(&lin,pat,capture))
		    count++;
	    if ((pat = pat->next) != (RxpToken *)0)
	    { 
		/*{{{  scan backwards for next element */
		
		while (bocl <= lin)
		{
		    if ((rval = rxp_hit_match(lin,pat,capture)) != (char *)0)
		    {
			if ((count < min) || (max && (count > max)))
			{
			    return ((char *)0);
			}
			else
			    return(rval);
		    }
		    else
		    {
			--count;
			--lin;
		    }
		}
		return ((char *)0);
		
		/*}}}   */
	    }
	    
	    if ((count < min) || (max && (count > max)))
	    {
		return ((char *)0);
	    }
	    /*}}}   */
	}
	else if (rxp_part_match(&lin,pat,capture))
		pat = pat->next;
	else  
	{
		return((char *)0);
	}
    }
    --lin;
    if (strstart > lin)
	return(strstart);
    else
	return(lin);
}

/*}}}   */
/*{{{  rxp_alt_match */

char *
rxp_alt_match( char      *lin,
	       RxpToken  *pat,
	       int       *which,
	       char      *capture[9] )
{
    char *rval;

    if (pat->tok != OR_SYM)
	disaster(104);

    #ifdef __TURBOC__
      /*{{{  check stack before recursing */
      
      if (stack_near_full(3))
	  mem_error("parse regular expression");
      
      /*}}}   */
    #endif

    if ((rval = rxp_hit_match(lin,pat->next,capture)) != (char *)0)
	return (rval);
    else if (pat->data.or_patn)
    {
	if (*which >= 0)
	{
	    (*which)++;
	    rxp_destroy_captures(capture);
	}
	return (rxp_alt_match( lin,
			       pat->data.or_patn,
			       which,
			       capture      ));
    }
    else
	return ((char *)0);
}

/*}}}   */
/*{{{  rxp_escape */

int
rxp_escape(char **s)
{
    if (**s == ESCAPE)
	(*s)++;
    return(**s);
}

/*}}}   */
/*{{{  rxp_destroy_pat */

void rxp_destroy_pat(RxpToken *head)
{
    RxpToken  *old_head;

    while (head)
    {
	switch (head->tok)
	{
	    case CCL:
	    case NCCL:
	       FREE(9,head->data.bitmap);
	       break;
	    case PAREN:
	    case OR_SYM:
	       rxp_destroy_pat(head->data.or_patn);
	       break;
	    default:
	       break;
	}
	old_head = head;
	head = head->next;
	FREE(10,old_head);
    }
}

/*}}}   */
/*{{{  rxp_init_token */

RxpError
rxp_init_token(RxpToken **ntok)
{

    *ntok = malloc(sizeof(RxpToken));
    if (!(*ntok))
	return (AllocFail);
    (*ntok)->data.lchar = '\0';
    (*ntok)->data.bitmap = &((*ntok)->data.lchar);
    (*ntok)->next = (RxpToken *)0;
    return (RxpOK);
}

/*}}}   */
/*{{{  rxp_set_bit */

void
rxp_set_bit(unsigned c, char *map)
{
    map[c >> 3] |= (1 << (c & 0x07));
}

/*}}}   */
/*{{{  rxp_create_pat */

RxpError
rxp_create_pat( char      **arg,
		RxpToken  **pat,
		int        *which,
		const int   toplevel,
		const char  delim,
		int        *capture_no )
{
    /*{{{  declare */
    
    RxpToken *head, *tail, *ntok;
    RxpToken *prev = (RxpToken *)0;
    RxpError err;
    int case_sensitive = TRUE;
    
    /*}}}   */

    /*{{{  initialise */
    
    (*which)++;
    *pat = (RxpToken *)0;
    if ((err = rxp_init_token(&head)) != RxpOK)
	return (err);
    head->tok = OR_SYM;
    head->data.or_patn = (RxpToken *)0;
    tail = head;
    
    /*}}}   */

    while (**arg != delim)
    {
	/*{{{  initialise a new token structure */
	
	if ((err = rxp_init_token(&ntok)) != RxpOK)
	{
	    rxp_destroy_pat(head);
	    return (err);
	}
	
	/*}}}   */

	switch (**arg)
	{
	    case ANY:
		/*{{{  just tag token */
		
		ntok->tok = ANY;
		break;
		
		/*}}}   */
	    case CCL:
		/*{{{  detect negated form and allocate bitmap */
		
		if (*((*arg)+1) == NEGATE)
		{
		    ntok->tok = NCCL;
		    *arg += 2;
		}
		else
		{
		    ntok->tok = CCL;
		    (*arg)++;
		}
		ntok->data.bitmap = (char *)malloc(32);
		if (ntok->data.bitmap == (char *)0)
		{
		    rxp_destroy_pat(head);
		    FREE(11,ntok);
		    return (AllocFail);
		}
		else
		{
		    /*{{{  scan the contents in */
		    
		    char *map = ntok->data.bitmap;
		    char *start = map;
		    int  first, last;
		    
		    for (first = 0; first < 32; first++)
		    {
			*start = '\0';
			start++;
		    }
		    
		    start = *arg;
		    if (**arg == CCLEND)
		    {
			rxp_set_bit(CCLEND,map);
			(*arg)++;
		    }
		    while (**arg  &&  **arg != CCLEND)
		    {
			if (**arg != '-')
			    if (case_sensitive)
				rxp_set_bit(**arg,map);
			    else if (islower(**arg))
			    {
				rxp_set_bit(**arg,map);
				rxp_set_bit((**arg) & 0xdf,map);
			    }
			    else if (isupper(**arg))
			    {
				rxp_set_bit(**arg,map);
				rxp_set_bit((**arg) | 0x20,map);
			    }
			    else
				rxp_set_bit(**arg,map);
			else if ((*arg == start) || (*((*arg)+1) == CCLEND))
			    rxp_set_bit('-',map);
			else
			{       
			    (*arg)++;
			    if (**arg < *((*arg)-2))
			    {       
				first = **arg;
				last = *((*arg)-2);
			    }
			    else
			    {      
				first = *((*arg)-2);
				last = **arg;
			    }
			    while (++first <= last)
				rxp_set_bit(first,map);
			}
			(*arg)++;
		    }
		    
		    if (!**arg)
		    {
			rxp_destroy_pat(head);
			FREE(12,ntok->data.bitmap);
			FREE(13,ntok);
			return (BadSet);
		    }
		    
		    #if DEBUG_RXSETS
		      /*{{{  display it */
		      printf("Set: ");
		      for (first = 0; first < 32; first++)
		      {
			  printf("%.2X",*(map + first));
		      }
		      printf(" = {\n");
		      for (first = 0; first < 256; first++)
		      {
			  if ((*(map + (first >> 3)) & (1 << (first & 0x07))) != 0)
			     printf("%c",first);
		      }
		      printf("\n}\n");
		      /*}}}   */
		    #endif
		    
		    /*}}}   */
		}
		break;
		
		/*}}}   */
	    case CLOSURE:
		/*{{{  save status & type */
		
		ntok->tok = CLOSURE;
		ntok->data.limit[0] = 0;
		ntok->data.limit[1] = 0;
		break;
		
		/*}}}   */
	    case PLUS:
		/*{{{  save status & type */
		
		ntok->tok = CLOSURE;
		ntok->data.limit[0] = 1;
		ntok->data.limit[1] = 0;
		break;
		
		/*}}}   */
	    case QUERY:
		/*{{{  save status & type */
		
		ntok->tok = CLOSURE;
		ntok->data.limit[0] = 0;
		ntok->data.limit[1] = 1;
		break;
		
		/*}}}   */
	    case OR_SYM:
		/*{{{  start a new tree branch */
		
		(*arg)++;
		FREE(14,ntok);
		
		#if DEBUG_RXPARSE
		  printf("------- new branch --------\n");
		#endif
		
		#ifdef __TURBOC__
		  /*{{{  check stack before recursing */
		  
		  if (stack_near_full(3))
		      mem_error("parse regular expression");
		  
		  /*}}}   */
		#endif
		
		{
		    int more_capts = 0;
		
		    err = rxp_create_pat(arg,
					 &(head->data.or_patn),
					 which,
					 toplevel,
					 delim,
					 toplevel ? &more_capts : capture_no );
		}
		if (err)
		    rxp_destroy_pat(head);
		else
		    *pat = head;
		return (err);
		
		/*}}}   */
	    case PAREN:
		/*{{{  spawn a sub-tree */
		
		#if DEBUG_RXPARSE
		  printf("-------- sub tree ---------\n");
		#endif
		
		(*arg)++;
		if (*capture_no < 10)
		{
		    ntok->tok = CAPTURE0 + *capture_no;
		    (*capture_no)++;
		}
		else
		    ntok->tok = PAREN;
		
		#ifdef __TURBOC__
		  /*{{{  check stack before recursing */
		  
		  if (stack_near_full(3))
		      mem_error("parse regular expression");
		  
		  /*}}}   */
		#endif
		
		{
		    int dummy = 0;
		
		    err = rxp_create_pat( arg,
					  &(ntok->data.or_patn),
					  &dummy,
					  FALSE,
					  ')',
					  capture_no );
		}
		if (err)
		{
		    rxp_destroy_pat(head);
		    FREE(15,ntok);
		    return (err);
		}
		
		#if DEBUG_RXPARSE
		  printf("------ end sub tree -------\n");
		#endif
		
		break;
		
		/*}}}   */
	    case ESCAPE:
		/*{{{  check if it's a special one */
		
		if (!*((*arg)+1))
		{
		    rxp_destroy_pat(head);
		    FREE(25,ntok);
		    return (BadEscape);
		}
		else if ((*((*arg)+1) == CASEOFF) ||
			 (*((*arg)+1) == CASEON)  )
		{
		    (*arg)++;
		    case_sensitive = (**arg == CASEON);
		    ntok->tok = CASEOFF;
		    break;
		}
		else if (*((*arg)+1) == RANGE)
		{
		    /*{{{  scan it for it's values */
		    
		    int min = 0, max = 0;
		    
		    ntok->tok = CLOSURE;
		    *arg += 2;
		    if (!isdigit(**arg))
		    {
			rxp_destroy_pat(head);
			FREE(16,ntok);
			return (BadRange);
		    }
		    while (isdigit(**arg))
		    {
			min = (10 * min) + (**arg - '0');
			(*arg)++;
		    }
		    if (min > 255)
		    {
			rxp_destroy_pat(head);
			FREE(17,ntok);
			return (BigRange);
		    }
		    if ((**arg == ESCAPE) && (*((*arg)+1) == RANEND))
		    {
			max = min;
			(*arg)++;
		    }
		    else if (**arg == COMMA)
		    {
			(*arg)++;
			if ((**arg == ESCAPE) && (*((*arg)+1) == RANEND))
			    (*arg)++;
			else if (!isdigit(**arg))
			{
			    rxp_destroy_pat(head);
			    FREE(18,ntok);
			    return (BadRange);
			}
			else
			{
			    while (isdigit(**arg))
			    {
				max = (10 * max) + (**arg - '0');
				(*arg)++;
			    }
			    if (max > 255)
			    {
				rxp_destroy_pat(head);
				FREE(19,ntok);
				return (BigRange);
			    }
			    if (max < min)
			    {
				rxp_destroy_pat(head);
				FREE(20,ntok);
				return (BadRange);
			    }
			    if ((**arg == ESCAPE) && (*((*arg)+1) == RANEND))
			    {
				(*arg)++;
			    }
			    else
			    {
				rxp_destroy_pat(head);
				FREE(21,ntok);
				return (BadRange);
			    }
			}
		    }
		    else
		    {
			rxp_destroy_pat(head);
			FREE(22,ntok);
			return (BadRange);
		    }
		    ntok->data.limit[0] = min;
		    ntok->data.limit[1] = max;
		    
		    /*}}}   */
		    break;
		}
		else if (isdigit(*((*arg)+1)))
		{
		    /*{{{  it's a capture substitution */
		    
		    (*arg)++;
		    ntok->tok = case_sensitive ? SUBST_IN : SUBST_CI;
		    ntok->data.subst_no = **arg - '1';
		    if (ntok->data.subst_no == -1)
		    {
			rxp_destroy_pat(head);
			FREE(26,ntok);
			return (BadSubstIn);
		    }
		    
		    /*}}}   */
		    break;
		}
		
		/*}}}   */
	    default:
		/*{{{  it's a literal */
		
		if (case_sensitive)
		{
		    ntok->tok = LITCHAR;
		    ntok->data.lchar = rxp_escape(arg);
		}
		else
		{
		    int ch = rxp_escape(arg);
		
		    ntok->tok = LITCASE;
		    ntok->data.lchar = islower(ch) ? ch & 0xdf : ch;
		}
		
		/*}}}   */
	}
	/*{{{  link in this token */
	
	if (ntok == (RxpToken *)0)
	{
	    rxp_destroy_pat(head);
	    return(Corrupt);
	}
	else if (ntok->tok == CASEOFF)
	{
	    FREE(26,ntok);
	}
	else if (ntok->tok != CLOSURE)
	{
	    tail->next = ntok;
	    prev = tail;
	    tail = ntok;
	}
	else if ((prev == (RxpToken *)0) || (prev->tok == CLOSURE))
	{
	    rxp_destroy_pat(head);
	    FREE(23,ntok);
	    return (BadClosure);
	}
	else
	{
	    prev->next = ntok;
	    ntok->next = tail;
	    prev = ntok;
	}
	
	/*}}}   */

	/*{{{  DEBUG */
	
	#if DEBUG_RXPARSE
	  dump_rxp_list(head);
	  printf("%c=>(%p,%p,%p)\n",**arg,head,prev,tail);
	#endif
	
	/*}}}   */

	(*arg)++;
    }

    if (!(**arg) && !toplevel)
    {
	rxp_destroy_pat(head);
	return (BadGroup);
    }

    *pat = head;
    return (RxpOK);
}

/*}}}   */
/*{{{  rxp_error */

int
rxp_error(RxpError err,int which_expr)
{
    switch (err)
    {
	case RxpOK:
	    return (FALSE);
	case NeverSee:
	    disaster(102);
	case Corrupt:
	    disaster(103);
	default:
	    error(Error,
		  in_file,
		  line_no,
		  "%s regular expression #%d",
		  regexp_err[(int)err],
		  which_expr                   );
    }
    return (TRUE);
}

/*}}}   */
/*{{{  rxp_do_match */

int
rxp_do_match(char *string,
	     char *pattern,
	     int  *which,
	     char *capture[9] )
{
    /*{{{  declare */
    
    int matched, capture_no = 0;
    char *cp = (char *)0;
    RxpToken *pat = (RxpToken *)0;
    
    /*}}}   */

    /*{{{  initialise */
    
    *which = 1;
    rxp_init_captures(capture);
    matched = FALSE;
    
    #if DEBUG_RXPARSE
      printf("Pattern =>%s<=\n",pattern);
    #endif
    
    /*}}}   */

    if (*pattern == OR_SYM)
	pattern++;
    else
	return(!strcmp(pattern,string)); /* plain match, not a regexp */

    if (!rxp_error(rxp_create_pat( &pattern,
				   &pat,
				   which,
				   TRUE,
				   '\0',
				   &capture_no),*which))
    {
	*which = 1;
	cp = rxp_alt_match(string,pat,which,capture);
    }

    rxp_destroy_pat(pat);

    #if DEBUG_RXCAPTS
      {
	  int i;
	  printf("Capture strings:\n");
	  for (i = 0; i < 9; ++i)
	      if (capture[i])
		  printf("\\%d =>%s<=\n",i+1,&(capture[i][2]));
	      else
		  printf("\\%d\n",i+1);
      }
    #endif

    if (cp && !(*(cp+1)))
	matched = TRUE;

    return(matched);
}

/*}}}   */

/*}}}   */
/*{{{  split_pair */

int
split_pair(char *text, char *symbol, char *value)
{
    char *pair = strchr(text,'=');

    if (pair)
    {
	char *ip = value;
	char *op = value;

	strcpy(value,pair + 1);
	strncpy(symbol,text,pair - text);
	*(symbol + (pair - text)) = '\0';

	while (ip && *ip)
	{
	    if (val_sp_esc_len                        &&
		!strncmp(ip,val_sp_esc,val_sp_esc_len))
	    {
		*op = ' ';
		ip += val_sp_esc_len - 1;
	    }
	    else
	    {
		*op = *ip;
	    }
	    ip++;
	    op++;
	}
	*op = '\0';
    }
    else
    {
	*value = '\0';
	strcpy(symbol,text);
    }
    return (pair != (char *)0);
}

/*}}}   */
/*{{{  set_bool_val */

void
set_bool_val(char *value, int *flag)
{
    if (!strcmp(value,"TRUE"))
	*flag = TRUE;
    else if (!strcmp(value,"FALSE"))
	*flag = FALSE;
    else if (!strcmp(value,"TOGGLE"))
	*flag = !(*flag);
}

/*}}}   */
/*{{{  set_cl_bool_val */

void
set_cl_bool_val(char *value, int *flag, int initval)
{
    if (!strcmp(value,"REVERT"))
	*flag = initval;
    else
	set_bool_val(value,flag);
}

/*}}}   */
/*{{{  write_special_mac */

int
write_special_mac(char *name, char *value)
{
    /*{{{  debugging */
    
    #if DEBUG_SIMS
      printf("Writing special macro Ocpp%s=%s\n",name,value);
    #endif
    
    /*}}}   */
    /*{{{  IntIsInt16 */
    if (!strcmp(name,"IntIsInt16"))
    {
	set_cl_bool_val(value,&int_16,cl_int_16);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  Irrevocable */
    if (!strcmp(name,"Irrevocable"))
    {
	set_cl_bool_val(value,&irrevocable,cl_irrevocable);
	if (irrevocable && (chain_compiler || mod_in_place))
	    error(Severe,
		  in_file,
		  line_no,
		  "Irrevocable processing illegal with %s command option",
		  chain_compiler ? "PP" : "MIP"                           );
	return (TRUE);
    }
    /*}}}   */
    /*{{{  DebugOption */
    if (!strcmp(name,"DebugOption"))
    {
	set_cl_bool_val(value,&annotate,cl_annotate);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  SubstAlways */
    if (!strcmp(name,"SubstAlways"))
    {
	set_bool_val(value,&always_subst);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  CleanupAfterPP */
    if (!strcmp(name,"RMCafterPP"))
    {
	set_bool_val(value,&clean_after_chain);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  AddCrToLf */
    if (!strcmp(name,"AddCrToLf"))
    {
	set_bool_val(value,&add_cr_to_lf);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  StripPrivates */
    if (!strcmp(name,"StripPrivates"))
    {
	set_bool_val(value,&remove_priv_cmt);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  ExpandFileNames */
    if (!strcmp(name,"ExpandFileNames"))
    {
	set_bool_val(value,&do_fname_expand);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  InLineInsert */
    if (!strcmp(name,"InLineInsert") && !chain_compiler)
    {
	set_bool_val(value,&in_line_insert);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  StdHeader */
    if (!strcmp(name,"StdHeader"))
    {
	set_bool_val(value,&use_std_header);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  WriteTabs */
    if (!strcmp(name,"WriteTabs"))
    {
	set_cl_bool_val(value,&write_tabs,2);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  FoldUpHashIFs */
    if (!strcmp(name,"FoldUpHashIFs"))
    {
	set_cl_bool_val(value,&fold_ifs,2);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  ReservedComment */
    if (!strcmp(name,"ReservedComment"))
    {
	value[15] = '\0';
	strcpy(resv_cmt,value);
	resv_cmt_len = strlen(value);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  SearchPathVar */
    if (!strcmp(name,"SearchPathVar"))
    {
	value[32] = '\0';
    
	#ifdef __MSDOS__
	  (void)strupr(value);
	#endif
    
	strcpy(search_path_var,value);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  PrivateComment */
    if (!strcmp(name,"PrivateComment"))
    {
	value[15] = '\0';
	strcpy(priv_cmt,value);
	priv_cmt_len = strlen(value);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  OpenSubstToken */
    if (!strcmp(name,"OpenSubstToken"))
    {
	value[3] = '\0';
	strcpy(open_subst,value);
	open_subst_len = strlen(value);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  CloseSubstToken */
    if (!strcmp(name,"CloseSubstToken"))
    {
	value[3] = '\0';
	strcpy(close_subst,value);
	close_subst_len = strlen(value);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  MacValSpcToken */
    if (!strcmp(name,"MacValSpcToken"))
    {
	value[3] = '\0';
	strcpy(val_sp_esc,value);
	val_sp_esc_len = strlen(value);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  CompilerName */
    if (!strcmp(name,"CompilerName"))
    {
	strcpy(compiler_name,value);
	return (TRUE);
    }
    /*}}}   */
    /*{{{  DefaultExtn */
    if (!strcmp(name,"DefaultExtn"))
    {
	value[8] = '\0';
	if (*value == '.')
	    strcpy(std_file_ext,value + 1);
	else
	    strcpy(std_file_ext,value);
	return (TRUE);
    }
    /*}}}   */

    error(Warn,in_file,line_no,"Undefined special internal macro 'Ocpp%s'",name);
    return (FALSE);
}

/*}}}   */
/*{{{  def_macro */

/* Returns:
	     0  if definition is corrupt
	     1  if a new macro was created
	     2  if an existing one was edited
*/

int
def_macro( char       *text,
	   MacDefCode  code,
	   char       *source,
	   long        in_line,
	   long        out_line )
{
    char symbol[MAXFNAME], value[MAXFNAME] = "";
    Macro **ip = &mac_table_head;
    Macro *mp = mac_table_head;
    int len, miss, pair;

    /*{{{  find what we're being asked to insert */
    
    pair = split_pair(text,symbol,value);
    len = strlen(symbol);
    
    /*}}}   */
    /*{{{  check for our own 'specials' */
    
    if ((!strncmp(symbol,"Ocpp",4))   &&
	(write_special_mac(&(symbol[4]),value)))
	   return (2);
    
    /*}}}   */
    /*{{{  check the basics */
    
    if (len == 0)
    {
	/*{{{  ?que? */
	
	error(Error,in_file,line_no,"Empty macro definition");
	return (FALSE);
	
	/*}}}   */
    }
    
    if ((*symbol == SWITCH)    ||
	(strchr(symbol,DIRSEP)))
    {
	/*{{{  syntax ? */
	
	error(Severe,in_file,line_no,"Malformed macro (%s)",symbol);
	return (FALSE);
	
	/*}}}   */
    }
    
    /*}}}   */

    miss = TRUE;
    while (mp && ((miss = strcmp(symbol,mp->name)) > 0))
    {
	ip = &(mp->next);
	mp = mp->next;
    }

    if (miss)
    {
	Macro *new;
	char *np, *vp;

	/*{{{  try to malloc space for macro and it's name */
	
	if (((np = (char *)malloc(len + 1)) == (char *)0)         ||
	    ((new = (Macro *)malloc(sizeof(Macro))) == (Macro *)0))
	{
	    mem_error("create macro definition (%s)", symbol);
	}
	strcpy(np,symbol);
	
	/*}}}   */
	/*{{{  copy our value if we've got one */
	
	if (pair)
	{
	    /*{{{  try to copy in our value */
	    
	    if ((vp = (char *)malloc(strlen(value) + 1)) == (char *)0)
		mem_error("create macro definition (%s)", symbol);
	    strcpy(vp,value);
	    
	    /*}}}   */
	}
	else
	{
	    /*{{{  null it out */
	    
	    vp = (char *)0;
	    
	    /*}}}   */
	}
	
	/*}}}   */
	/*{{{  patch the data fields and link it */
	
	new->name = np;
	new->value = vp;
	new->defcode = code;
	new->src = source;
	new->in_line = in_line;
	new->out_line = out_line;
	
	new->next = *ip;
	*ip = new;
	
	/*}}}   */
	/*{{{  debugging */
	
	#if DEBUG_LIST
	  printf("LINKED NEW MACRO\n");
	#endif
	
	/*}}}   */
    }
    else
    {
	/*{{{  free it's value if it's got one */
	
	if (mp->value)
	    FREE(0,mp->value);
	
	/*}}}   */
	/*{{{  copy our value if we've got one */
	
	if (pair)
	{
	    /*{{{  try to copy in our value */
	    
	    char *vp;
	    if ((vp = (char *)malloc(strlen(value) + 1)) == (char *)0)
		mem_error("create macro definition (%s)", symbol);
	    else
	    {
		strcpy(vp,value);
		mp->value = vp;
	    }
	    
	    /*}}}   */
	}
	else
	{
	    /*{{{  null it out */
	    
	    mp->value = (char *)0;
	    
	    /*}}}   */
	}
	
	/*}}}   */
	/*{{{  edit the origin data */
	
	mp->defcode = code;
	mp->src = source;
	mp->in_line = in_line;
	mp->out_line = out_line;
	
	/*}}}   */
	/*{{{  debugging */
	
	#if DEBUG_LIST
	  printf("REDEFINED EXISTING MACRO\n");
	#endif
	
	/*}}}   */
    }
    /*{{{  debugging */
    
    #if DEBUG_LIST
      show_macro_list();
    #endif
    
    /*}}}   */

    return ((!miss) + 1);
}

/*}}}   */
/*{{{  get_bool_val */

char *
get_bool_val(int flag)
{
    static char buf[6];
    if (flag)
	strcpy(buf,"TRUE");
    else
	strcpy(buf,"FALSE");

    return (&(buf[0]));
}

/*}}}   */
/*{{{  read_special_mac */

char *
read_special_mac(char *name)
{
    static char buf[MAXFNAME];

    /*{{{  IntIsInt16 */
    if (!strcmp(name,"IntIsInt16"))
    {
	return (get_bool_val(int_16));
    }
    /*}}}   */
    /*{{{  Irrevocable */
    if (!strcmp(name,"Irrevocable"))
    {
	return (get_bool_val(irrevocable));
    }
    /*}}}   */
    /*{{{  DebugOption */
    if (!strcmp(name,"DebugOption"))
    {
	return (get_bool_val(annotate));
    }
    /*}}}   */
    /*{{{  VersionNo */
    if (!strcmp(name,"VersionNo"))
    {
	strcpy(buf,VERSION);
	return (&(buf[0]));
    }
    /*}}}   */
    /*{{{  InputName */
    if (!strcmp(name,"InputName"))
    {
	strcpy(buf,in_file);
	return (&(buf[0]));
    }
    /*}}}   */
    /*{{{  Author */
    if (!strcmp(name,"Author"))
    {
	strcpy(buf,"M.I.Barlow");
	return (&(buf[0]));
    }
    /*}}}   */
    /*{{{  RootInputName */
    if (!strcmp(name,"RootInputName"))
    {
	strcpy(buf,base_file);
	return (&(buf[0]));
    }
    /*}}}   */
    /*{{{  CallersName */
    if (!strcmp(name,"CallersName"))
    {
	if (tplt_level)
	    strcpy(buf,caller_name);
	else
	    buf[0] = '\0';
	return (&(buf[0]));
    }
    /*}}}   */
    /*{{{  CallLine */
    if (!strcmp(name,"CallLine"))
    {
	if (tplt_level)
	    sprintf(buf,"%ld",caller_line);
	else
	    buf[0] = '\0';
	return (&(buf[0]));
    }
    /*}}}   */
    /*{{{  OutputName */
    if (!strcmp(name,"OutputName"))
    {
	if (mod_in_place)
	    strcpy(buf,in_file);
	else
	    strcpy(buf,out_file);
	return (&(buf[0]));
    }
    /*}}}   */
    /*{{{  InputLine */
    if (!strcmp(name,"InputLine"))
    {
	sprintf(buf,"%ld",line_no);
	return (&(buf[0]));
    }
    /*}}}   */
    /*{{{  DogsSay */
    if (!strcmp(name,"DogsSay"))
    {
	strcpy(buf,"Woof!");
	return (&(buf[0]));
    }
    /*}}}   */
    /*{{{  TodaysDate */
    if (!strcmp(name,"TodaysDate"))
    {
	strcpy(buf,&(start_time[4]));
	strcat(buf," ");
	strcat(buf,&(start_time[20]));
	return (&(buf[0]));
    }
    /*}}}   */
    /*{{{  StartTime */
    if (!strcmp(name,"StartTime"))
    {
	strcpy(buf,&(start_time[11]));
	return (&(buf[0]));
    }
    /*}}}   */
    /*{{{  Dir */
    if (!strcmp(name,"Dir"))
    {
	buf[0] = DIRSEP;
	buf[1] = '\0';
	return (&(buf[0]));
    }
    /*}}}   */
    /*{{{  Opt */
    if (!strcmp(name,"Opt"))
    {
	buf[0] = SWITCH;
	buf[1] = '\0';
	return (&(buf[0]));
    }
    /*}}}   */
    /*{{{  Psep */
    if (!strcmp(name,"Psep"))
    {
	strcpy(buf,PATHSEPS);
	buf[1] = '\0';
	return (&(buf[0]));
    }
    /*}}}   */
    /*{{{  Env */
    if (!strncmp(name,"Env",3))
    {
	char *val;
    
	#ifdef __MSDOS__
	  (void)strupr(&(name[3]));
	#endif
    
	val = getenv(&(name[3]));
	if (val)
	    strcpy(buf,val);
	else
	    strcpy(buf,"");
	return (&(buf[0]));
    }
    /*}}}   */

    error(Warn,in_file,line_no,"Undefined special internal macro 'Ocpp%s'",name);
    return ((char *)0);
}

/*}}}   */
/*{{{  macro_defined */

/* IN:    symbol      ptr to name to search for
	  mp          ptr to ptr that will point to it if found
				else be null

   OUT:   TRUE if defined FALSE if not
*/

int
macro_defined(char *symbol, Macro **result)
{
    static Macro special;
    Macro *mp = mac_table_head;
    int miss = TRUE;

    if (!*symbol)
    {
	error(Error,in_file,line_no,"Null macro symbol is invalid");
	return (FALSE);
    }
    if (!strncmp(symbol,"Ocpp",4))
    {
	char *val = read_special_mac(&(symbol[4]));

	if (val)
	{
	    special.name = symbol;
	    special.value = val;
	    special.defcode = M_special;
	    special.src = (char *)0;
	    special.in_line = 0L;
	    special.out_line = 0L;
	    *result = &special;

	    return (TRUE);
	}
    }

    while (mp && ((miss = strcmp(symbol,mp->name)) > 0))
	mp = mp->next;
    if (miss)
      *result = (Macro *)0;
    else
      *result = mp;

    return (!miss && (mp->defcode != M_undef));
}

/*}}}   */
/*{{{  is_def_value */

int
is_def_value(char *symbol, char *value, Macro **result)
{
    int hit;

    if ((hit = macro_defined(symbol,result)) == TRUE)
    {
	if ((*result)->value == (char *)0)
	{
	    error(Warn,
		  in_file,
		  line_no,
		  "Macro '%s' has no associated value", symbol);
	    hit = (*value == '\0');
	}
	else
	    hit = !strcmp(value,(*result)->value);
    }
    return (hit);

}

/*}}}   */
/*{{{  word_after */

int
word_after(char **bp, int count)
{
    *bp += count;
    if (!isspace(**bp))
    {
	error(Error,in_file,line_no,"Malformed directive");
	return (FALSE);
    }
    while (**bp && isspace(**bp)) (*bp)++;
    return (TRUE);
}

/*}}}   */
/*{{{  get_word */

void
get_word(char *get, char *put)
{
    while (*get && !isspace(*get))
    {
	*put = *get;
	put++;
	get++;
    }
    *put = '\0';
}

/*}}}   */
/*{{{  get_line */

Operator
get_line( int   *saw_tabs,
	  int   *saw_fold,
	  int   *indent,
	  char **text,
	  char **mac,
	  int    doing_rmc )
{
    static char bf[MAXLINE], m_name[MAXFNAME];
    Directive *dp;
    char *bp;
    char *pp = m_name;

    *saw_tabs = FALSE;
    *saw_fold = FALSE;
    *text = (char *)0;
    *mac = (char *)0;

    if (!fgets(bf,MAXLINE - 1,in)) return (EoF);

    line_no++;
    /*{{{  debugging */
    
    #if DEBUG_LINE
      fprintf(stderr,"-- Processing line %ld\n",line_no);
    #endif
    
    /*}}}   */
    /*{{{  strip private comments */
    
    bp = strstr(bf,priv_cmt);
    if (bp && (!doing_rmc || remove_priv_cmt))
    {
	while ((bp != bf)        &&
	       isspace(*(bp - 1)))
	  bp--;
	if (bp == bf)
	    return (Null);
	else
	{
	    *bp = '\n';
	    bp++;
	    *bp = '\0';
	}
    }
    
    /*}}}   */
    /*{{{  measure the indentation */
    
    bp = bf;
    *indent = 0;
    while ((*bp == ' ') || (*bp == '\t'))
    {
	if (*bp == ' ') *indent += 1;
	else
	{
	    *saw_tabs = TRUE;
	    *indent = ((*indent / TABWIDTH) + 1) * TABWIDTH;
	}
	bp++;
    }
    /*}}}   */
    /*{{{  strip any reserved comment, or fold marks on #IF / #ENDIF */
    
    *text = bp;
    if ((bp = strstr(*text,resv_cmt)) != (char *)0)
    {
	bp += resv_cmt_len;
	*text = bp;
    }
    else if (((bp = strstr(*text,"--{{{  #IF")) != (char *)0)   ||
	     ((bp = strstr(*text,"--}}}  #ENDIF")) != (char *)0))
    {
	*saw_fold = TRUE;
	bp += 7;
	*text = bp;
    }
    else
    {
	bp = *text;
    }
    
    /*}}}   */
    /*{{{  deal with simple cases */
    
    if (*bp == '\n')
    {
	return (Blank);
    }
    
    if (*bp != '#')
    {
	return (Vanilla);
    }
    
    bp++;
    dp = (Directive *)&(known[0]);
    while (dp->tok_len                      &&
	   strncmp(bp,dp->token,dp->tok_len))
	dp++;
    
    if (!dp->tok_len)
    {
	return (Vanilla);
    }
    
    /*}}}   */
    /*{{{  deal with known directives */
    
    switch (dp->op)
    {
	case Else:
	case Endif:
	    return (dp->op);
	case Indent:
	case Outdent:
	    /*{{{  deal specially with in/outdent */
	    
	    {
		bp += dp->tok_len;
		/*{{{  detect and remove any pre-existing in/outdentation */
		
		if (!strncmp(bp,"ED",2))
		{
		    char *p0 = bp, *p1 = bp + 2;
		
		    while (*p1)
		    {
			*p0 = *p1;
			p0++;
			p1++;
		    }
		    *p0 = '\0';
		    global_indent += (dp->op == Indent) ? -2 : 2;
		}
		
		/*}}}   */
		*mac = bp;
		return (dp->op);
	    }
	    
	    /*}}}   */
	case Ifdef:
	case Ifndef:
	case Ordef:
	case Orndef:
	case Anddef:
	case Andndef:
	case Define:
	case Derive:
	case Undef:
	    if (!word_after(&bp,dp->tok_len))
		return (Vanilla);
	    get_word(bp,pp);
	    *mac = m_name;
	    return (dp->op);
	case If:
	case Ifn:
	case Or:
	case Orn:
	case And:
	case Andn:
	case Subst:
	case Memo:
	case UseErr:
	case Shell:
	case InterpC:
	case Template:
	case Mdump:
	case Defsfrom:
	case Include:
	case Use:
	case Import:
	case Extract:
	case Replace:
	case Return:
	    if (!word_after(&bp,dp->tok_len))
		return (Vanilla);
	    *mac = bp;
	    return (dp->op);
	default:
	    break;
    }
    return (Vanilla);
    
    /*}}}   */
}

/*}}}   */
/*{{{  put_line */

void
put_line ( int   use_tabs,
	   int   indent,
	   char *text,
	   int   encomment,
	   int   enfold     )
{
    if (*text)
    {
	/*{{{  declare */
	
	char bf[MAXLINE] = "";
	int i;
	int n_spc;
	char *cp = bf;
	
	/*}}}   */

	/*{{{  create the indentation */
	
	if (indent < 0)
	{
	    error(Error,
		  in_file,
		  line_no,
		  "Indentation became less than zero (%d spaces)",
		  indent                                           );
	    n_spc = 0;
	    global_indent -= indent;
	}
	else
	  n_spc = indent;
	if ((write_tabs == 1)              ||
	    ((write_tabs == 2) && use_tabs))
	{
	    int n_tab = n_spc / TABWIDTH;
	
	    n_spc %= TABWIDTH;
	    for (i = 0; i < n_tab; ++i)
	    {
		*cp = (char)9;
		cp++;
	    }
	}
	for (i = 0; i < n_spc; ++i)
	{
	    *cp = ' ';
	    cp++;
	}
	
	/*}}}   */
	/*{{{  add text as appropriate */
	
	if (encomment)
	{
	    if (strncmp(text,"--",2))
	    {
		if (enfold > 0)
		    strcpy(cp,"--{{{  ");
		else if (enfold < 0)
		    strcpy(cp,"--}}}  ");
		else
		    strcpy(cp,resv_cmt);
	    }
	}
	else
	    *cp= '\0';
	
	if (encomment && irrevocable)
	{
	    return;
	}
	else
	{
	    fprintf(out,"%s%s",bf,text);
	    out_line_no += 1L;
	}
	
	/*}}}   */
    }
}

/*}}}   */
/*{{{  def_src */

char *
def_src( Macro *mac, int relative )
{
    static char bf[64];
    if (mac)
    {
	switch (mac->defcode)
	{
	    case M_undef:
		if (mac->src == (char *)0)
		    sprintf(bf," on cmd line");
		else if (!strcmp(mac->src,in_file))
		    sprintf(bf,
			    " on line %ld%s",
			    relative ? mac->out_line : mac->in_line,
			    relative ? " above" : ""                );
		else
		    sprintf(bf," in %s, line %ld",mac->src,mac->in_line);
		break;
	    case M_cmd_line:
		strcpy(bf," on cmd line");
		break;
	    case M_env_var:
		sprintf(bf," in var %s",mac->src);
		break;
	    case M_immed:
		if (!strcmp(mac->src,in_file))
		    sprintf(bf,
			    " on line %ld%s",
			    relative ? mac->out_line : mac->in_line,
			    relative ? " above" : ""                );
		else
		    sprintf(bf," in %s, line %ld",mac->src,mac->in_line);
		break;
	    case M_param:
		sprintf(bf," in call at line %ld of %s",mac->in_line,mac->src);
		break;
	    case M_retval:
		sprintf(bf," returned at line %ld of %s",mac->in_line,mac->src);
		break;
	    case M_special:
		strcpy(bf," special internal");
		break;
	    default:
		break;
	}
    }
    else
    {
	bf[0] = '\0';
    }
    return (bf);
}

/*}}}   */
/*{{{  isearch_open */

FILE *
isearch_open(char *fname, char *mode)
{
    static FILE *fp;
    char *sp, *pp;

    if (!fname) return ((FILE *)0);
    fp = fopen(fname,mode);
    if (fp || strchr(fname,DIRSEP)) return (fp);
    sp = getenv(search_path_var);
    if (!sp || (*sp == '\0')) return ((FILE *)0);
    pp = strtok(sp,PATHSEPS);
    while (pp && !fp)
    {
	char name[MAXFNAME];

	sprintf(name,"%s%s",pp,fname);
	/*{{{  debugging */
	
	#if DEBUG_SEARCH
	  printf("SEEKING: >%s<\n",name);
	#endif
	
	/*}}}   */
	fp = fopen(name,mode);
	pp = strtok((char *)0,PATHSEPS);
    }
    return (fp);
}

/*}}}   */
/*{{{  char_xlate */

char *
char_xlate(char **ptr, int add_cr)
{
    static char fout[32];
    if (**ptr == '*')
    {
	(*ptr)++;
	sprintf(fout,"**");
	return (fout);
    }
    else if (**ptr == '\\')
    {
	char c;
	(*ptr)++;
	c = **ptr;
	(*ptr)++;
	switch ((int)c)
	{
	    case (int)'a':
		sprintf(fout,"*#07");
		return (fout);
	    case (int)'b':
		sprintf(fout,"*#08");
		return (fout);
	    case (int)'f':
		sprintf(fout,"*#0C");
		return (fout);
	    case (int)'r':
		sprintf(fout,"*c");
		return (fout);
	    case (int)'v':
		sprintf(fout,"*#0B");
		return (fout);
	    case (int)'n':
		if (add_cr)
		    sprintf(fout,"*c*n");
		else
		    sprintf(fout,"*n");
		return (fout);
	    case (int)'t':
		sprintf(fout,"*t");
		return (fout);
	    case (int)'?':
	    case (int)'\\':
		sprintf(fout,"%c",c);
		return (fout);
	    case (int)'x':
	    case (int)'X':
		/*{{{  scan hex escape */
		
		{
		    char nbf[32] = "";
		    char *cp = nbf;
		    int x, n;
		    while (isxdigit(**ptr))
		    {
			*cp = **ptr;
			(*ptr)++;
			cp++;
		    }
		    *cp = '\0';
		    n = sscanf(nbf,"%x",&x);
		    if (!n || (x > 0xFF)) return ((char *)0);
		    sprintf(fout,"*#%02X",x);
		    return (fout);
		}
		
		/*}}}   */
	    case (int)'0':
	    case (int)'1':
	    case (int)'2':
	    case (int)'3':
	    case (int)'5':
	    case (int)'6':
	    case (int)'7':
		/*{{{  scan octal escape */
		
		{
		    char nbf[32];
		    char *cp = nbf;
		    int x, n;
		    (*ptr)--;
		    while ((**ptr >= '0') && (**ptr <= '7'))
		    {
			*cp = **ptr;
			(*ptr)++;
			cp++;
		    }
		    *cp = '\0';
		    n = sscanf(nbf,"%o",&x);
		    if (!n || (x > 0xFF)) return ((char *)0);
		    sprintf(fout,"*#%02X",x);
		    return (fout);
		}
		
		/*}}}   */
	    default:
		return ((char *)0);
	}

    }
    else
    {
	sprintf(fout,"%c",**ptr);
	(*ptr)++;
	return (fout);
    }
}

/*}}}   */
/*{{{  scan_subst */

char *
scan_subst(char     *in,
	   char     *out,
	   ErrorCode err_syntax,
	   ErrorCode err_undef,
	   int      *did_one)
{
    *did_one = FALSE;
    while (*in && (*in != '\r') && (*in != '\n'))
    {
	if (!strncmp(in,open_subst,open_subst_len))
	{
	    /*{{{  try to read in a name, and maybe a modifier */
	    
	    Macro *mp;
	    char symbol[MAXFNAME];
	    char *sp = symbol;
	    SubMod modifier = NoModifier;
	    int min = 0, max = 0;
	    
	    in += open_subst_len;
	    
	    /*{{{  look for modifiers */
	    
	    if (*in == TO_UPPER_MOD)
	    {
		modifier = ToUpper;
		in++;
		if (*in == TO_LOWER_MOD)
		{
		    modifier = InitialCap;
		    in++;
		}
	    }
	    else if (*in == TO_LOWER_MOD)
	    {
		modifier = ToLower;
		in++;
	    }
	    else if (*in == BLANK_OUT_MOD)
	    {
		modifier = BlankOut;
		in++;
	    }
	    
	    if (isdigit(*in))
	    {
		while (isdigit(*in))
		{
		    min = (10 * min) + (*in - '0');
		    in++;
		}
		if (*in == ',')
		{
		    in++;
		    if (isdigit(*in))
		    {
			while (isdigit(*in))
			{
			    max = (10 * max) + (*in - '0');
			    in++;
			}
		    }
		}
		else
		    max = min;
	    }
	    else if (*in == ',')
	    {
		in++;
		if (isdigit(*in))
		{
		    while (isdigit(*in))
		    {
			max = (10 * max) + (*in - '0');
			in++;
		    }
		}
	    }
	    if (max && (max < min))
	    {
		int tmp = max;
		max = min;
		min = tmp;
	    }
	    
	    while (isspace(*in))
		in++;
	    
	    /*}}}   */
	    
	    while (*in && (strncmp(in,close_subst,close_subst_len)))
	    {
		if (!strncmp(in,open_subst,open_subst_len))
		{
		    /*{{{  wrong ! */
		    
		    error(err_syntax,
			  in_file,
			  line_no,
			  "Encountered \"%s\" inside substitution", open_subst);
		    return ((char *)0);
		    
		    /*}}}   */
		}
		else
		{
		    /*{{{  next please */
		    
		    *sp = *in;
		    sp++;
		    in++;
		    
		    /*}}}   */
		}
	    }
	    
	    if (!*in)
	    {
		/*{{{  wrong ! */
		
		error(err_syntax,
		      in_file,
		      line_no,
		      "End-of-line encountered before \"%s\" in substitution", close_subst);
		return ((char *)0);
		
		/*}}}   */
	    }
	    
	    *sp = '\0';
	    in += close_subst_len;
	    /*{{{  copy it */
	    
	    if (macro_defined(symbol,&mp))
	    {
		char *vp = mp->value;
		int initial = TRUE;
		int count = 0;
	    
		if (!vp) vp = symbol;
		while(*vp)
		{
		    if ((count >= min) && (!max || (count <= max)))
		    {
			if (((modifier == ToUpper) ||
			     (initial && (modifier == InitialCap))) && islower(*vp))
			    *out = *vp & 0xdf;
			else if (((modifier == ToLower) ||
				  (!initial && (modifier == InitialCap))) && isupper(*vp))
			    *out = *vp | 0x20;
			else if (modifier == BlankOut)
			    *out = ' ';
			else
			    *out = *vp;
			out++;
		    }
		    count++;
		    vp++;
		    initial = FALSE;
		}
		*did_one = TRUE;
	    }
	    else
	    {
		error(err_undef,
		      in_file,
		      line_no,
		      "Macro \"%s\" undefined during substitution", symbol);
	    }
	    
	    /*}}}   */
	    
	    /*}}}   */
	}
	else if (!strncmp(in,close_subst,close_subst_len))
	{
	    /*{{{  wrong ! */
	    
	    error(err_syntax,
		  in_file,
		  line_no,
		  "Isolated \"%s\" in substitution", close_subst);
	    return ((char *)0);
	    
	    /*}}}   */
	}
	else
	{
	    /*{{{  next please */
	    
	    *out = *in;
	    out++;
	    in++;
	    
	    /*}}}   */
	}
    }

    *out = '\0';
    return (out);
}

/*}}}   */
/*{{{  write_header */

void
write_header(FILE *out, char *title, char *fname, int indent)
{

    if (!irrevocable || !in_line_insert)
    {
	fprintf(out,"--\n");
	fprintf(out,"--  ");
    }
    else
    {
	int i;
	for (i = 0; i < indent; ++i)
	    fputc(' ',out);
	fprintf(out,"--{{{  ");
    }
    fprintf(out,"%s file \"%s\" by ocpp\n",title,fname);
    if (!irrevocable || !in_line_insert)
    {
	fprintf(out,"--\n");
	fprintf(out,"--  WARNING! This file may be automatically re-generated without\n");
	fprintf(out,"--           prior notice; any manual editing will be overwitten.\n");
	fprintf(out,"--\n");
    }
}

/*}}}   */
/*{{{  expand_fname */

char *
expand_fname(char *unexp)
{
    /*{{{  declare */
    
    static char xpand[MAXFNAME];
    char *xp = xpand;
    char *ux = unexp;
    int did_subst = FALSE;
    
    /*}}}   */

    if (!unexp || !do_fname_expand)
	return (unexp);

    while (*ux)
    {
	/*{{{  copy bytes until the end, or until we see "$(" */
	
	while (*ux && (*ux != '$') && (*(ux + 1) != '('))
	{
	    *xp = *ux;
	    ux++;
	    xp++;
	}
	
	/*}}}   */
	if (*ux)
	{
	    /*{{{  declare */
	    
	    char envar[MAXFNAME];
	    char *en = envar;
	    char *ep = (char *)0;
	    char *sp = ux + 2;
	    
	    /*}}}   */

	    /*{{{  try to get an environment variable that matches */
	    
	    while (*sp && (*sp != ')'))
	    {
		*en = *sp;
		sp++;
		en++;
	    }
	    if (*sp)
	    {
		*en = '\0';
	    
		#ifdef __MSDOS__
		  (void)strupr(envar);
		#endif
	    
		ep = getenv(envar);
		if (!ep)
		{
		    error(Warn,
			  in_file,
			  line_no,
			  "\"$(%s)\" not in environment",
			  envar);
		}
	    }
	    else
	    {
		error(Warn,
		      in_file,
		      line_no,
		      "Incomplete \"$(<env_var>)\" in file specification");
	    }
	    
	    /*}}}   */
	    if (ep)
	    {
		/*{{{  copy it's contents to the expanded pathname */
		
		while (*ep)
		{
		    *xp = *ep;
		    xp++;
		    ep++;
		}
		
		/*}}}   */
	    }
	    /*{{{  skip over the substitution */
	    
	    ux = sp + 1;
	    did_subst = TRUE;
	    
	    /*}}}   */
	}
    }
    *xp = '\0';

    if (did_subst)
	return (xpand);
    else
	return (unexp);
}

/*}}}   */
/*{{{  do_import */

char *
do_import(char *fname, char *old_line, int indent)
{
    /*{{{  declare */
    
    static char retline[MAXLINE];
    char cfile[MAXFNAME];
    char ofile[MAXFNAME];
    char *fin, *fout;
    FILE *in, *itp_out;
    int cf_line = 0;
    int do_insert = (irrevocable && in_line_insert);
    
    /*}}}   */

    /*{{{  try to open input file */
    
    strcpy(cfile,fname);
    fin = expand_fname(strtok(cfile," \t\"\n"));
    in = isearch_open(fin,"rt");
    if (in == (FILE *)0)
    {
	error(Error,
	      in_file,
	      line_no,
	      "Failed to find C import file \"%s\"", fin);
	return ((char *)0);
    }
    
    /*}}}   */
    /*{{{  form output file name and try to open it too */
    
    if (do_insert)
    {
	itp_out = out;
    }
    else
    {
	char *cp;
    
	strcpy(retline,fin);
	fout = strrchr(retline,DIRSEP);
	if (!fout) fout = retline;
	else fout++;
	cp = strrchr(retline,'.');
	if (!cp) cp = fout + strlen(fout);
	strcpy(cp,".inc");
	strcpy(ofile,fout);
	itp_out = fopen(fout,"wt");
	if (itp_out == (FILE *)0)
	{
	    error(Error,
		  in_file,
		  line_no,
		  "Failed to create \"%s\" for interpreted data",fout);
	    fclose(in);
	    return((char *)0);
	}
    }
    if (use_std_header)
	write_header(itp_out,"Definitions interpreted from C language",fin,indent);
    
    /*}}}   */
    /*{{{  process the data */
    
    while (!feof(in))
    {
	char *gp, *wp;
    
	gp = fgets(retline,MAXLINE - 1,in);
	if (gp)
	{
	    cf_line++;
	    wp = strtok(gp," \t\n");
	    if (wp && !strcmp(wp,"#define"))
	    {
		gp = strtok((char *)0," \t\n");
		if (gp && !strchr(gp,'('))
		{
		    char *cp = strtok((char *)0,"\n");
		    if (cp && (*cp != '\"')) wp = strtok(cp," \t");
		    else wp = cp;
		    if (wp)
		    {
			int valid = TRUE;
			OcType type;
			/*{{{  convert underscores to dots in symbol name */
			
			cp = gp;
			while (*cp)
			{
			    if (*cp == '_') *cp = '.';
			    cp++;
			}
			
			/*}}}   */
			/*{{{  strip any parentheses around value */
			
			cp = wp + strlen(wp) - 1;
			if ((*wp == '(') && (*cp == ')'))
			{
			    wp++;
			    while (isspace(*wp)) wp++;
			    cp--;
			    while (isspace(*cp)) cp--;
			    cp++;
			    *cp = '\0';
			}
			
			/*}}}   */
			/*{{{  convert value field if possible */
			
			cp = wp + strlen(wp) - 1;
			while ((cp > wp) && isspace(*cp)) cp--;
			if ((cp > wp) && (*cp == '\\'))
			{
			    valid = FALSE;
			}
			else if ((*wp == '0') && ((*(wp + 1) == 'x') || (*(wp + 1) == 'X')))
			{
			    /*{{{  convert hex value */
			    
			    type = Int;
			    wp++;
			    *wp = '#';
			    cp = wp + 1;
			    (void)strupr(cp);
			    while (isxdigit(*cp)) cp++;
			    if (int_16)
			    {
				int convert = (*cp == 'L');
				*cp = '\0';
				if (!convert)
				{
				    long lv;
				    sscanf(wp + 1,"%lx",&lv);
				    convert = ((lv > 32767L) || (lv < -32768L));
				}
				if (convert)
				{
				    strcpy(cp,"(INT32)");
				    type = Int32;
				}
			    }
			    else *cp = '\0';
			    
			    /*}}}   */
			}
			else if (isdigit(*wp) ||
				(*wp == '-')  ||
				(*wp == '.')  ||
				(*wp == '+'))
			{
			    /*{{{  convert decimal integers but detect floating point */
			    
			    char bf[64];
			    char *bp = bf + 1;
			    type = Int;
			    cp = wp;
			    if (*cp == '.') *bf = '0';
			    else
			    {
				*bf = *wp;
				cp++;
			    }
			    while (isdigit(*cp))
			    {
				*bp = *cp;
				bp++;
				cp++;
			    }
			    if (*cp == '.')
			    {
				type = Real64;
				if (!isdigit(*(bp - 1)))
				{
				    *bp = '0';
				    bp++;
				}
				*bp = '.';
				bp++;
				cp++;
				if (!isdigit(*cp))
				{
				    *bp = '0';
				    bp++;
				}
				else while (isdigit(*cp))
				{
				    *bp = *cp;
				    bp++;
				    cp++;
				}
			    }
			    if ((*cp == 'E') || (*cp == 'e'))
			    {
				if (type == Int)
				{
				    *bp = '.';
				    bp++;
				    *bp = '0';
				    bp++;
				}
				type = Real64;
				*bp = 'E';
				bp++;
				cp++;
				if (isdigit(*cp))
				{
				    *bp = '+';
				    bp++;
				}
				while(isdigit(*cp) || (*cp == '+') || (*cp == '-'))
				{
				    *bp = *cp;
				    bp++;
				    cp++;
				}
			    }
			    *bp = '\0';
			    if (type == Real64)
			    {
				if (*cp == 'F')
				{
				    type = Real32;
				    strcpy(bp,"(REAL32)");
				}
				else strcpy(bp,"(REAL64)");
			    }
			    if (int_16 && (type == Int))
			    {
				int convert = (*cp == 'L');
				if (!convert)
				{
				    long lv;
				    sscanf(bf,"%ld",&lv);
				    convert = ((lv > 32767L) || (lv < -32768L));
				}
				if (convert)
				{
				    strcpy(bp,"(INT32)");
				    type = Int32;
				}
			    }
			    strcpy(wp,bf);
			    
			    /*}}}   */
			}
			else if (*wp == '\'')
			{
			    /*{{{  try to convert single character constant */
			    
			    type = Byte;
			    cp = strrchr(wp + 1,'\'');
			    if (cp)
			    {
				cp = wp + 1;
				cp = char_xlate(&cp,FALSE);
				valid = (cp != (char *)0);
				if (valid) sprintf(wp,"\'%s\'",cp);
			    }
			    else valid = FALSE;
			    
			    /*}}}   */
			}
			else if (*wp == '\"')
			{
			    /*{{{  try to convert constant character string */
			    
			    type = ByteArray;
			    cp = strrchr(wp + 1,'\"');
			    if (cp)
			    {
				cp = wp + 1;
				if (*cp == '\"') *(wp + 2) = '\0';
				else
				{
				    char bf[MAXFNAME] = "";
				    while (valid && (*cp != '\"'))
				    {
					char *tmp = char_xlate(&cp,add_cr_to_lf);
					if (tmp) strcat(bf,tmp);
					else valid = FALSE;
				    }
				    if (valid) sprintf(wp,"\"%s\"",bf);
				}
			    }
			    else valid = FALSE;
			    
			    /*}}}   */
			}
			else
			{
			    strupr(wp);
			    if (strcmp(wp,"TRUE") && strcmp(wp,"FALSE")) valid = FALSE;
			    else type = Boolean;
			}
			
			/*}}}   */
			if (valid)
			{
			    if (irrevocable && in_line_insert)
			    {
				int i;
    
				for (i = 0; i < indent; ++i)
				{
				    fputc(' ',itp_out);
				}
			    }
			    fprintf(itp_out,
				    "VAL %s %s IS %s :",
				    typename[(int)type],
				    gp,
				    wp);
			if (annotate)
			{
			    if (use_std_header)
				fprintf(itp_out,"  --(from line %d)\n",cf_line);
			    else
				fprintf(itp_out,"  --(line %d, file %s)\n",cf_line,fin);
			}
			else
			    fprintf(itp_out,"\n");
			}
		    }
		}
	    }
	}
    }
    
    /*}}}   */
    /*{{{  finish up */
    
    if (do_insert)
    {
	if (use_std_header)
	    strcpy(retline,"--}}}  End of interpreted definitions\n");
	else
	    *retline = '\0';
    }
    else if (irrevocable)
    {
	sprintf(retline,"#INCLUDE \"%s\"\n",ofile);
    }
    else
    {
	sprintf(retline,"#INCLUDE \"%s\" %s%s",ofile,resv_cmt,old_line);
    }
    fclose(in);
    if (!do_insert)
	fclose(itp_out);
    
    /*}}}   */

    return (retline);
}

/*}}}   */
/*{{{  do_template */

void
do_template( char *fname,
	     char *oldline,
	     char *newline,
	     int indent     )
{
    /*{{{  declare */
    
    char tplt_assign[MAXLINE];
    char *tif_buf = (char *)0;
    char *t_out_file = (char *)0;
    Macro *old_table = (Macro *)0;
    FILE *new_in, *new_out;
    char *scan, *t_in_file, *old_in_file, *old_out_file;
    unsigned long old_line;
    unsigned long old_out_line;
    long file_mark;
    int old_g_indent, old_rec_level, acanonical;
    int out_is_stdout = (out == stdout);
    int in_is_stdin = (in == stdin);
    int doing_insert;
    int dm = 0;
    
    /*}}}   */

    /*{{{  debugging */
    
    #if DEBUG_STACK
      fprintf(stderr,"-- Template level %d, SP = %lx\n",tplt_level,(unsigned long)&dm);
    #endif
    
    /*}}}   */
    /*{{{  provisionally build replacement line */
    
    if (tplt_count > 999)
    {
	error(Severe,
	      in_file,
	      line_no,
	      "Ridiculously large number of templates generated!");
    }
    
    sprintf(1 + strrchr(tplt_name,'.'),"%.3d",tplt_count);
    
    if (irrevocable)
    {
	sprintf(newline,
		"#INCLUDE \"%s\"\n",
		tplt_name            );
    }
    else
    {
	sprintf(newline,
		"#INCLUDE \"%s\" %s%s",
		tplt_name,
		resv_cmt,
		oldline  );
    }
    
    /*}}}   */
    /*{{{  look for a leading symbol assignment */
    
    *tplt_assign = '\0';
    if (*fname != '\"')
    {
	char *cp = tplt_assign;
    
	while (*fname && (*fname != '\"') && (*fname != '='))
	{
	    *cp = *fname;
	    cp++;
	    fname++;
	}
	if ((*fname == '=') && (*(fname+1) == '\"'))
	{
	    *cp = '=';
	    cp++;
	    *cp = '\0';
	    fname++;
	}
	else
	    error(Severe,
		  in_file,
		  line_no,
		  "Malformed assignment of template RETURN value");
    }
    
    /*}}}   */
    /*{{{  try to open template file */
    
    if ((tif_buf = (char *)malloc(MAXFNAME)) == (char *)0)
	mem_error("process level %d template",tplt_level);
    
    strcpy(tif_buf,fname);
    scan = strtok(tif_buf," \t\"\n");
    t_in_file = expand_fname(scan);
    if (t_in_file != scan)
    {
	strcpy(tif_buf,t_in_file);
	t_in_file = tif_buf;
    }
    if ((new_in = isearch_open(t_in_file,"rt")) == (FILE *)0)
    {
	error(Severe,
	      in_file,
	      line_no,
	      "Failed to find template file \"%s\"",t_in_file);
    }
    
    /*}}}   */
    /*{{{  check if template is canonical */
    
    (void)strtok(oldline," \t\n");
    (void)strtok((char *)0," \t\n");
    scan = strtok((char *)0," \t\n");
    if ((acanonical = !strcmp(scan,acanonical_tplt)) == TRUE)
    {
	scan = strtok((char *)0," \t\n");
    }
    
    /*}}}   */
    /*{{{  clone the macro table */
    
    old_table = mac_table_head;
    mac_table_head = (Macro *)0;
    if (acanonical)
    {
	Macro *mp = old_table;
	while (mp)
	{
	    if (mp->value)
	    {
		char defn[MAXLINE];
    
		sprintf(defn,"%s=%s",mp->name,mp->value);
		(void)def_macro(defn,
				mp->defcode,
				mp->src,
				mp->in_line,
				mp->out_line );
	    }
	    else
	    {
		(void)def_macro(mp->name,
				mp->defcode,
				mp->src,
				mp->in_line,
				mp->out_line );
	    }
	    mp = mp->next;
	}
    }
    
    /*}}}   */
    /*{{{  add any embedded definitions / derivations */
    
    while (scan)
    {
	char derived[MAXLINE];
	int dm;
    
	(void)scan_subst(scan,derived,Severe,Severe,&dm);
	def_macro(derived,M_param,in_file,line_no,out_line_no);
	scan = strtok((char *)0," \t\n");
    }
    
    /*}}}   */
    /*{{{  try to open output file */
    
    doing_insert = (irrevocable && in_line_insert);
    if (doing_insert)
    {
	if (use_std_header)
	    sprintf(newline,"--}}}  End of \"%s\" derived code\n",t_in_file);
	else
	    *newline = '\0';
	new_out = out;
    }
    else
    {
	if ((t_out_file = (char *)malloc(strlen(tplt_name) + 1)) == (char *)0)
	    mem_error("process level %d template",tplt_level);
    
	strcpy(t_out_file,tplt_name);
	if ((new_out = fopen(t_out_file,"wt")) == (FILE *)0)
	{
	    error(Severe,
		  in_file,
		  line_no,
		  "Failed to create template output file \"%s\"",t_out_file);
	}
	tplt_count++;
    }
    if (use_std_header)
    {
	write_header(new_out,"Code derived from occam template",t_in_file,indent);
    }
    
    /*}}}   */
    /*{{{  save the file context */
    
    old_in_file = in_file;
    caller_name = in_file;
    old_line = line_no;
    old_out_line = out_line_no;
    if (!doing_insert)
    {
	out_line_no = use_std_header ? 6L : 0L;
    }
    else if (use_std_header)
    {
	out_line_no += 1L;
    }
    caller_line = line_no;
    old_rec_level = rec_level;
    in_file = t_in_file;
    if (!in_is_stdin)
    {
	file_mark = ftell(in);
	fclose(in);
    }
    in = new_in;
    line_no = 0;
    rec_level = -1;
    
    if (!doing_insert)
    {
	old_out_file = out_file;
	old_g_indent = global_indent;
	global_indent = 0;
	if (!out_is_stdout)
	    fclose(out);
	out_file = t_out_file;
	out = new_out;
    }
    
    /*}}}   */
    /*{{{  process template */
    
    tplt_level++;
    if (annotate)
    {
	char tmp[16];
    
	sprintf(tmp," level %d",tplt_level);
	error(Info,
	      old_in_file,
	      old_line,
	      "Begin%s template \"%s\" %c> \"%s\"",
	      (tplt_level > 1) ? tmp : "",
	      t_in_file,
	      in_line_insert ? '>' : '-',
	      in_line_insert ? out_file : t_out_file );
    }
    
    #ifdef __TURBOC__
      /*{{{  check stack before recursing */
      
      if (stack_near_full(0))
	  mem_error("parse template");
      
      /*}}}   */
    #endif
    
    switch (parse(FALSE,FALSE,doing_insert ? indent : 0,&dm,0L,0L,(char *)0))
    {
	case ParseFAIL:
	    exit (1);
	case ParseOK:
	case ParseRETURN:
	    break;
    }
    if (annotate)
    {
	char tmp[16];
    
	sprintf(tmp," level %d",tplt_level);
	error(Info,
	      old_in_file,
	      old_line,
	      "End%s template \"%s\" %c> \"%s\"",
	      (tplt_level > 1) ? tmp : "",
	      t_in_file,
	      in_line_insert ? '>' : '-',
	      in_line_insert ? out_file : t_out_file );
    }
    tplt_level--;
    
    /*}}}   */
    /*{{{  restore file context */
    
    fclose(new_in);
    if (in_is_stdin)
    {
	in = stdin;
    }
    else
    {
	in = fopen(old_in_file,"rt");
	fseek(in,file_mark,SEEK_SET);
    }
    in_file = old_in_file;
    line_no = old_line;
    if (!doing_insert)
	out_line_no = old_out_line;
    rec_level = old_rec_level;
    
    if (!doing_insert)
    {
	fclose(new_out);
	if (out_is_stdout)
	    out = stdout;
	else
	{
	    out = fopen(old_out_file,"at");
	    fseek(out,0L,SEEK_END);
	}
	out_file = old_out_file;
	global_indent = old_g_indent;
    }
    
    /*}}}   */
    /*{{{  destroy clone table etc. */
    
    if (t_out_file)
	FREE(1,t_out_file);
    {
	Macro *mp = mac_table_head;
    
	while (mp)
	{
	    Macro *nxp = mp->next;
    
	    FREE(2,mp->name);
	    if (mp->value)
		FREE(3,mp->value);
	    FREE(4,mp);
	    mp = nxp;
	}
    }
    mac_table_head = old_table;
    FREE(5,tif_buf);
    
    /*}}}   */
    /*{{{  perform any assignment */
    
    if (*tplt_assign)
    {
	strncat(tplt_assign,tplt_retvalue,MAXLINE - strlen(tplt_assign));
	tplt_assign[MAXLINE - 1] = '\0';
	def_macro(tplt_assign,M_retval,in_file,line_no,out_line_no);
    }
    
    /*}}}   */
}

/*}}}   */
/*{{{  dump_maclist */

void
dump_maclist(FILE *dmp)
{
    Macro *mp = mac_table_head;
    int maxlen = 0;

    while (mp)
    {
	int len = strlen(mp->name);
	if (mp->value)
	    len += 1 + strlen(mp->value);
	if (len > maxlen)
	    maxlen = len;
	mp = mp->next;
    }
    if (dmp == stderr)
    {
	error(Info,in_file,line_no,"Macro list dump, current definitions:");
    }
    else
    {
	char tmp[MAXFNAME];
	int sav = use_std_header;

	use_std_header = TRUE;
	sprintf(tmp,"Macro list dumped at line %ld of",line_no);
	write_header(dmp,tmp,in_file,0);
	use_std_header = sav;
    }
    mp = mac_table_head;
    while (mp)
    {
	if (mp->defcode != M_undef)
	{
	    char *where = def_src(mp,FALSE);
	    int len = 1 + maxlen - strlen(mp->name);

	    if (dmp != stderr)
		fprintf(dmp,"#DEFINE");
	    fprintf(dmp," %s",mp->name);
	    if (mp->value)
	    {
		fprintf(dmp,"=%s",mp->value);
		len -= 1 + strlen(mp->value);
	    }
	    if (where)
	    {
		if (len > 0)
		{
		    int i;
		    for (i = 0; i < len; ++i)
			fprintf(dmp," ");
		}
		fprintf(dmp,
			" %s(%s)\n",
			(dmp == stderr) ? "" : "--",
			where + 1                    );
	    }
	    else
		fprintf(dmp,"\n");
	}
	mp = mp->next;
    }
}

/*}}}   */
/*{{{  unwind_cond_stack */

void
unwind_cond_stack(int           cond,
		  char         *depends_on,
		  unsigned long backref      )
{
    if (rec_level)
    {
	n_err++;
	fprintf(stderr,
		"%7ld  --  #IF%s%s %s%s\n",
		backref,
		(cond & Cond_DEF) ? "" : "N",
		(cond & Cond_IF) ? "" : "DEF",
		depends_on,
		(cond & Cond_ETC) ? "  etc..." : "" );
    }
}

/*}}}   */
/*{{{  scan_def_file */

int
scan_def_file(char *fname)
{
    FILE *in;
    int opened;

    in = fopen(fname,"rt");
    opened = (in != (FILE *)0);
    if (opened)
    {
	char bf[MAXLINE];

	in_file = fname;
	line_no = 0L;

	/*{{{  process the contents */
	
	while (fgets(bf,MAXLINE - 1,in))
	{
	    char *sym = strtok(bf," \t\n");
	    int is_derive = FALSE;
	
	    line_no++;
	    if ( sym && *sym &&
		 (!strcmp(sym,"#DEFINE")                       ||
		  !strcmp(sym,"#define")                       ||
		  (is_derive = !strcmp(sym,"#DERIVE")) != FALSE) )
	    {
		sym = strtok((char *)0," \t\n");
		if (sym && !strchr(sym,'('))
		{
		    char *cp = strtok((char *)0," \t\n");
		    int use_it = FALSE;
		    if ((cp == (char *)0)     ||
			(!strncmp(cp,"/*",2)) ||
			(!strncmp(cp,"//",2)) ||
			(!strncmp(cp,"--",2)) )  use_it = TRUE;
		    else
		    {
			if ((*cp == '(') && (*(cp + strlen(cp) - 1) == ')'))
			{
			    cp++;
			    *(cp + strlen(cp) - 1) = '\0';
			}
			strupr(cp);
			if (!strcmp(cp,"TRUE")) use_it = TRUE;
			else
			{
			    char *p = cp;
			    use_it = TRUE;
			    while (*p && use_it)
			    {
				use_it = isdigit(*p);
				p++;
			    }
			    if (use_it)
			    {
				int x;
				use_it = sscanf(cp,"%d",&x);
				use_it &= (x != 0);
			    }
			}
		    }
		    if (use_it)
		    {
			char tmp[MAXLINE];
			int result;
			char *mp;
			int  dm;
	
			if (scan_subst(sym,tmp,Warn,Error,&dm))
			    mp = tmp;
			else
			    mp = sym;
			result = def_macro(mp,
					   M_immed,
					   fname,
					   line_no,
					   out_line_no );
			if (annotate)
			{
			    error(Info,
				  in_file,
				  line_no,
				  "Macro %s %s%s",
				  mp,
				  (result == 2) ? "re-" : "",
				  is_derive ? "derived" : "defined");
			}
		    }
		}
	    }
	}
	
	/*}}}   */

	fclose(in);
	in_file = (char *)0;
	in = (FILE *)0;
	line_no = 0L;
    }

    return (opened);
}

/*}}}   */
/*{{{  parse */

ParseResult
parse(int            encomment,
      int            force,
      int            pre_indent,
      int           *why,
      unsigned long  backref,
      unsigned long  backref_in,
      char          *depends_on )
{
    unsigned long last_mod = 0L;
    int saw_else = FALSE;
    int at_top;

    rec_level++;

    /*{{{  debugging */
    
    #if DEBUG_STACK
      fprintf(stderr,"-- Parse level %d, SP = %lx\n",
		     rec_level,
		     (unsigned long)&use_tabs );
    #endif
    
    /*}}}   */
    at_top = rec_level;
    while (TRUE)
    {
	/*{{{  declare */
	
	char *text, *mac;
	int hidden = encomment || force;
	int indent, idt, use_tabs, fold_seen;
	Operator this_one;
	
	/*}}}   */

	this_one = get_line(&use_tabs,&fold_seen,&idt,&text,&mac,FALSE);
	indent = idt + pre_indent + global_indent;

	/*{{{  debugging */
	
	#if DEBUG_PARSER
	  fprintf(out,"%s | ",line_type_name[this_one]);
	#endif
	
	/*}}}   */
	switch (this_one)
	{
	    case EoF:
		/*{{{  it's either an error or we're finished */
		
		*tplt_retvalue = '\0';
		if (rec_level)
		{
		    int multi = (rec_level > 1);
		
		    error(Unwind,
			  in_file,
			  line_no,
			  "End-of-file before %s#ENDIF%s, see line%s:",
			  multi ? "multiple " : "",
			  multi ? "s" : "",
			  multi ? "s" : ""    );
		    n_err--;
		    unwind_cond_stack(*why,depends_on,backref_in);
		    rec_level--;
		    return_code = 2;
		    return (ParseFAIL);
		}
		else
		    return (ParseOK);
		
		/*}}}   */
	    case Null:
		/*{{{  skip it */
		
		break;
		
		/*}}}   */
	    case Blank:
		/*{{{  just print it */
		
		put_line(use_tabs,indent,text,irrevocable && hidden,FALSE);
		break;
		
		/*}}}   */
	    case Vanilla:
		/*{{{  just print it, unless "SubstAlways" is TRUE */
		
		if (always_subst           &&
		    !encomment             &&
		    !force                 &&
		    (strncmp(text,"--",2)) )
		{
		    char buf[MAXLINE];
		    int did_one;
		
		    if (scan_subst(text,buf,Error,Error,&did_one) && did_one)
		    {
			if (irrevocable)
			{
			    strcat(buf,"\n");
			}
			else
			{
			    strcat(buf," ");
			    strcat(buf,resv_cmt);
			    strcat(buf,text);
			}
			put_line(use_tabs,indent,buf,FALSE,FALSE);
		    }
		    else
		    {
			put_line(use_tabs,indent,text,FALSE,FALSE);
		    }
		}
		else
		{
		    put_line(use_tabs,indent,text,hidden,FALSE);
		}
		at_top = FALSE;
		break;
		
		/*}}}   */
	    case Ifdef:
	    case Ifndef:
		/*{{{  find out and recurse if possible */
		
		at_top = FALSE;
		{
		    Macro *macdef;
		    int def = (this_one == Ifdef) ? Cond_DEF : 0;
		    int hit = (def == Cond_DEF) ^ macro_defined(mac,&macdef);
		    int enfold = (fold_ifs == 2) ? fold_seen : fold_ifs;
		    char *depend;
		
		    /*{{{  prepare to recurse */
		    
		    if (((depend = (char *)malloc(strlen(mac) + 1)) == (char *)0) ||
			#ifdef __TURBOC__
			  stack_near_full(0)                                         )
			#else
			  FALSE                                                     )
			#endif
		    {
			mem_error("nest conditionals beyond %d levels",rec_level);
		    }
		    strcpy(depend,mac);
		    
		    /*}}}   */
		
		    if (annotate)
		    {
			/*{{{  build and put annotated line */
			
			char tmp[MAXFNAME];
			
			sprintf(tmp,
				"#IF%sDEF %s  --(%s%s)-<%d>\n",
				(def == Cond_DEF) ? "" : "N",
				depend,
				hit ? "FALSE" : "TRUE",
				def_src(macdef,TRUE),
				rec_level);
			put_line(use_tabs,indent,tmp,1,enfold);
			
			/*}}}   */
		    }
		    else
		    {
			put_line(use_tabs,indent,text,1,enfold);
		    }
		
		    switch (parse(hit,
				  hidden,
				  pre_indent,
				  &def,
				  out_line_no,
				  line_no,
				  depend)            )
		    {
			case ParseOK:
			    FREE(6,depend);
			    break;
			case ParseFAIL:
			    unwind_cond_stack(*why,depends_on,backref_in);
			    rec_level--;
			    return (ParseFAIL);
			case ParseRETURN:
			    FREE(36,depend);
			    return (ParseRETURN);
		    }
		}
		break;
		
		/*}}}   */
	    case If:
	    case Ifn:
		/*{{{  find out and recurse if possible */
		
		at_top = FALSE;
		{
		    char symbol[MAXFNAME] = "="; /* ie: no such macro */
		    char value[MAXLINE];
		    char *captures[9];
		    Macro *macdef;
		    int def = (this_one == If) ? Cond_DEF : 0;
		    int hit;
		    int which = 0;
		    int enfold = (fold_ifs == 2) ? fold_seen : fold_ifs;
		    char *depend, *cp;
		
		    if ((cp = strchr(mac,'=')) == (char *)0)
		    {
			error(Error,
			      in_file,
			      line_no,
			      "#IF%s predicate is not a name=pattern pair",
			      (def == Cond_DEF) ? "" : "N");
		    }
		    else
		    {
			strncpy(symbol,mac,cp - mac);
			symbol[cp - mac + 1] = '\0';
			cp++;
			strcpy(value,cp);
			if ((cp = strstr(value,"--")) != (char *)0)
			    *cp = '\0';
			else
			    cp = value + strlen(value);
			cp--;
		
			/*{{{  strip trailing space */
			
			while (isspace(*cp) && (cp >= value))
			{
			    *cp = '\0';
			    cp--;
			}
			
			/*}}}   */
		    }
		
		    hit = macro_defined(symbol,&macdef);
		    if (hit)
		    {
			if (macdef->value == (char *)0)
			{
			    error(Warn,
				  in_file,
				  line_no,
				  "Macro '%s' has no associated value",symbol);
			    hit = (*value == '\0');
			}
			else
			{
			    hit = rxp_do_match(macdef->value,value,&which,captures);
			    rxp_destroy_captures(captures);
			}
		    }
		
		    hit ^= (def == Cond_DEF);
		
		    /*{{{  prepare to recurse */
		    
		    if (((depend = (char *)malloc(strlen(symbol) +
						  strlen(value) + 2)) == (char *)0) ||
			#ifdef __TURBOC__
			  stack_near_full(0)                                         )
			#else
			  FALSE                                                     )
			#endif
		    {
			mem_error("nest conditionals beyond %d levels",rec_level);
		    }
		    strcpy(depend,symbol);
		    strcat(depend,"=");
		    strcat(depend,value);
		    
		    /*}}}   */
		
		    if (annotate)
		    {
			/*{{{  build and put annotated line */
			
			char tmp[MAXFNAME];
			char exprno[32];
			
			sprintf(exprno,"expr#%d)-(",which);
			sprintf(tmp,
				"#IF%s %s  --(%s%s%s)-<%d>\n",
				(def == Cond_DEF) ? "" : "N",
				depend,
				(which && ((hit != 0)^(def == Cond_DEF))) ? exprno : "",
				hit ? "FALSE" : "TRUE",
				def_src(macdef,TRUE),
				rec_level);
			put_line(use_tabs,indent,tmp,1,enfold);
			
			/*}}}   */
		    }
		    else
		    {
			put_line(use_tabs,indent,text,1,enfold);
		    }
		    def |= Cond_IF;
		
		    switch (parse(hit,
				  hidden,
				  pre_indent,
				  &def,
				  out_line_no,
				  line_no,
				  depend)            )
		    {
			case ParseOK:
			    FREE(7,depend);
			    break;
			case ParseFAIL:
			    unwind_cond_stack(*why,depends_on,backref_in);
			    rec_level--;
			    return (ParseFAIL);
			case ParseRETURN:
			    FREE(37,depend);
			    return (ParseRETURN);
		    }
		}
		break;
		
		/*}}}   */
	    case Else:
		/*{{{  flip state / it's an orphan */
		
		at_top = FALSE;
		if (!rec_level)
		{
		    /*{{{  it's a level 0 orphan */
		    
		    if (annotate)
			put_line(use_tabs,indent,"#ELSE  --(erroneous orphan directive)\n",1,FALSE);
		    else
			put_line(use_tabs,indent,text,1,FALSE);
		    error(Error,in_file,line_no,"#ELSE without previous #IF[N][DEF]");
		    
		    /*}}}   */
		}
		else if (saw_else)
		{
		    /*{{{  it's a multiple else */
		    
		    error(Error,in_file,line_no,"Only one #ELSE allowed in conditional (ignored)");
		    if (annotate)
		    {
			put_line(use_tabs,indent,"#ELSE  --(erroneous duplicate directive)\n",1,FALSE);
		    }
		    else
		    {
			put_line(use_tabs,indent,text,1,FALSE);
		    }
		    
		    /*}}}   */
		}
		else
		{
		    encomment = !encomment;
		    saw_else = TRUE;
		    if (annotate)
		    {
			/*{{{  build annotated line */
			
			char tmp[MAXFNAME];
			char ext[16];
			int etc = *why & Cond_ETC;
			
			sprintf(ext,"-%ld",last_mod);
			sprintf(tmp,"#ELSE  --(IF%s%s %s%s, see line%s %ld%s above)-<%d>\n",
				      (*why & Cond_DEF) ? "N" : "",
				      (*why & Cond_IF) ? "" : "DEF",
				      depends_on,
				      etc ? " etc" : "",
				      etc ? "s" : "",
				      backref,
				      etc ? ext : "",
				      rec_level - 1 );
			put_line(use_tabs,indent,tmp,1,FALSE);
			
			/*}}}   */
		    }
		    else
		    {
			put_line(use_tabs,indent,text,1,FALSE);
		    }
		}
		break;
		
		/*}}}   */
	    case Endif:
		/*{{{  we're done / it's an orphan */
		
		{
		    int enfold = (fold_ifs == 2) ? fold_seen : fold_ifs;
		
		    if (!rec_level)
		    {
			if (annotate)
			    put_line(use_tabs,indent,"#ENDIF  --(erroneous orphan directive)\n",1,-enfold);
			else
			    put_line(use_tabs,indent,text,1,-enfold);
			error(Error,
			      in_file,
			      line_no,
			      "#ENDIF without previous #IF[N][DEF]");
		    }
		    else
		    {
			rec_level--;
			if (annotate)
			{
			    /*{{{  build annotated line */
			    
			    char tmp[MAXFNAME];
			    char ext[16];
			    int etc = *why & Cond_ETC;
			    
			    sprintf(ext,"-%ld",last_mod);
			    sprintf(tmp,"#ENDIF  --(%s%s%s%s, see line%s %ld%s above)-<%d>\n",
					  (*why & Cond_DEF) ? "" : "N",
					  (*why & Cond_IF) ? ((*why & Cond_DEF) ? "" : " ") : "DEF ",
					  depends_on,
					  etc ? " etc" : "",
					  etc ? "s" : "",
					  backref,
					  etc ? ext : "",
					  rec_level  );
			    put_line(use_tabs,indent,tmp,1,-enfold);
			    
			    /*}}}   */
			}
			else
			{
			    put_line(use_tabs,indent,text,1,-enfold);
			}
			return (ParseOK);
		    }
		}
		
		/*}}}   */
	    case Define:
		/*{{{  create or update the definition */
		
		at_top = FALSE;
		if (!hidden)
		{
		    int result;
		
		    result = def_macro(mac,M_immed,in_file,line_no,out_line_no + 1L);
		    if (annotate)
		    {
			error(Info,
			      in_file,
			      line_no,
			      "Macro %s %sdefined",
			      mac,
			      (result == 2) ? "re-" : "" );
		    }
		}
		put_line(use_tabs,indent,text,1,FALSE);
		break;
		
		/*}}}   */
	    case Derive:
		/*{{{  substitute, then create or update */
		
		at_top = FALSE;
		put_line(use_tabs,indent,text,1,FALSE);
		if (!hidden)
		{
		    char tmp[MAXLINE];
		    int result;
		    char *mp;
		    int  dm;
		
		    if (scan_subst(mac,tmp,Warn,Error,&dm))
			mp = tmp;
		    else
			mp = mac;
		
		    /*{{{  link the new macro */
		    
		    result = def_macro(mp,M_immed,in_file,line_no,out_line_no + 1L);
		    if (annotate)
		    {
			error(Info,
			      in_file,
			      line_no,
			      "Macro %s %sderived",
			      mp,
			      (result == 2) ? "re-" : "" );
		    }
		    
		    /*}}}   */
		}
		break;
		
		/*}}}   */
	    case Undef:
		/*{{{  mark any existing definition as undefined */
		
		at_top = FALSE;
		put_line(use_tabs,indent,text,1,FALSE);
		if (!hidden)
		{
		    Macro *macdef;
		    int result;
		
		    if ((result = macro_defined(mac,&macdef)) != 0)
		    {
			macdef->defcode = M_undef;
			macdef->src = in_file;
			macdef->in_line = line_no;
			macdef->out_line = out_line_no;
		    }
		    if (annotate)
		    {
			error(Info,
			      in_file,
			      line_no,
			      "Macro %s %sundefined",
			      mac,
			      (result == FALSE) ? "already " : "" );
		    }
		}
		break;
		
		/*}}}   */
	    case Subst:
		/*{{{  substitution process the line */
		
		at_top = FALSE;
		{
		    char bf[MAXLINE];
		    char *newtext = (char *)0;
		
		    #ifdef __TURBOC__
		      /*{{{  check stack before recursing */
		      
		      if (stack_near_full(0))
			  mem_error("parse subtitutions");
		      
		      /*}}}   */
		    #endif
		
		    if (!hidden)
		    {
			/*{{{  try to write a modified line */
			
			int dm;
			
			if ((newtext = scan_subst(mac,bf,Error,Error,&dm)) != (char *)0)
			{
			    *newtext = ' ';
			    newtext++;
			    *newtext = '\0';
			    if (irrevocable)
			    {
				strcat(bf,"\n");
			    }
			    else
			    {
				strcat(bf,resv_cmt);
				if ((strlen(bf) + strlen(text)) > (MAXLINE - 1))
				{
				    error(Severe,in_file,line_no,"Line too long when substituted");
				}
				else
				    strcat(bf,text);
			    }
			}
			
			
			/*}}}   */
		    }
		    if (newtext)
			put_line(use_tabs,indent,bf,0,FALSE);
		    else
			put_line(use_tabs,indent,text,1,FALSE);
		}
		break;
		
		/*}}}   */
	    case Memo:
		/*{{{  show the message */
		
		at_top = FALSE;
		put_line(use_tabs,indent,text,1,FALSE);
		if (!hidden)
		{
		    const char head[] = "User Memo: ";
		    int dm;
		
		    char tmp[MAXLINE];
		    char *put = &(tmp[sizeof(head) - 1]);
		    char *wipe = put;
		
		    strcpy(tmp,head);
		    if (!scan_subst(mac,put,Info,Ignore,&dm))
			strcpy(wipe,mac);
		    error(Info,in_file,line_no,tmp);
		}
		break;
		
		/*}}}   */
	    case UseErr:
		/*{{{  report the message and blow out */
		
		at_top = FALSE;
		put_line(use_tabs,indent,text,1,FALSE);
		if (!hidden)
		{
		    const char head[] = "User Error: ";
		    int dm;
		
		    char tmp[MAXLINE];
		    char *put = &(tmp[sizeof(head) - 1]);
		    char *wipe = put;
		
		    strcpy(tmp,head);
		    if (!scan_subst(mac,put,Info,Ignore,&dm))
			strcpy(wipe,mac);
		    else
		    {
			int err;
			if (sscanf(wipe,"%d",&err))
			    user_ret_code = err;
		    }
		    error(UserDef,in_file,line_no,tmp);
		}
		break;
		
		/*}}}   */
	    case Shell:
		/*{{{  call the command processor */
		
		at_top = FALSE;
		put_line(use_tabs,indent,text,1,FALSE);
		if (!hidden)
		{
		    char cmd_line[MAXLINE];
		    char *put = cmd_line;
		    int dm;
		
		    if (!scan_subst(mac,put,Warn,Ignore,&dm))
		    {
			strcpy(cmd_line,"Substitution failure: ");
			strcat(cmd_line,text);
			error(Severe,in_file,line_no,cmd_line);
		    }
		    else
		    {
			int retcode;
			char *cmdptr = cmd_line + strspn(cmd_line," \t\n");
		
			if (*cmdptr == '\0')
			    error(Warn,in_file,line_no,"Empty shell command line");
			else
			{
			    if (annotate)
			    {
				char tmp[MAXFNAME];
		
				sprintf( tmp,"Executing: %s",cmdptr);
				error(Info,in_file,line_no,tmp);
			    }
			    if ((retcode = system(cmdptr)) != 0)
			    {
				error(Severe,
				      in_file,
				      line_no,
				      "Shell error %d: %s",retcode,cmdptr);
			    }
			}
		    }
		}
		break;
		
		/*}}}   */
	    case InterpC:
		/*{{{  get help to do it */
		
		at_top = FALSE;
		if (!hidden)
		{
		    char *newtext;
		
		    #ifdef __TURBOC__
		      /*{{{  check stack before recursing */
		      
		      if (stack_near_full(0))
			  mem_error("import from C");
		      
		      /*}}}   */
		    #endif
		    newtext = do_import(mac,text,indent);
		    if (newtext) put_line(use_tabs,indent,newtext,0,FALSE);
		    else put_line(use_tabs,indent,text,1,FALSE);
		}
		else
		    put_line(use_tabs,indent,text,1,FALSE);
		break;
		
		/*}}}   */
	    case Template:
		/*{{{  get help to swap I/O context and recurse again */
		
		at_top = FALSE;
		if (hidden)
		    put_line(use_tabs,indent,text,1,FALSE);
		else
		{
		    char *newtext;
		
		    /*{{{  test for template recursion */
		    
		    if (tplt_level >= MAXTPNEST)
		    {
			error(Severe,
			      in_file,
			      line_no,
			      "Template nesting > %d; suspected infinite recursion",MAXTPNEST);
		    }
		    
		    /*}}}   */
		    /*{{{  prepare to process it */
		    
		    if (((newtext = (char *)malloc(MAXLINE)) == (char *)0) ||
			#ifdef __TURBOC__
			  stack_near_full(0)                                  )
			#else
			  FALSE                                              )
			#endif
		    {
			mem_error("process level %d template",tplt_level + 1);
		    }
		    
		    /*}}}   */
		    /*{{{  then do the deed */
		    
		    do_template(mac,text,newtext,indent);
		    put_line(use_tabs,indent,newtext,0,FALSE);
		    FREE(8,newtext);
		    
		    /*}}}   */
		}
		break;
		
		/*}}}   */
	    case Return:
		/*{{{  exit template processing if appropriate */
		
		put_line(use_tabs,indent,text,1,FALSE);
		if (tplt_level)
		{
		    if (!hidden)
		    {
			int dm;
			char *rv = scan_subst(mac,
					      tplt_retvalue,
					      Error,
					      Error,
					      &dm     );
			if (!rv)
			    *tplt_retvalue = '\0';
		
			return (ParseRETURN);
		    }
		}
		else
		    error(Error,in_file,line_no,"#RETURN is only legal inside Templates");
		break;
		
		/*}}}   */
	    case Mdump:
		/*{{{  dump the current macro list */
		
		put_line(use_tabs,indent,text,1,FALSE);
		if (!hidden)
		{
		    char *fname = expand_fname(strtok(mac," \t\"\n"));
		    if (fname)
		    {
			FILE *dmp;
		
			if ((dmp = fopen(fname,"wt")) == (FILE *)0)
			{
			    error(Warn,in_file,line_no,
				  "Unable to open \"%s\" for macro dump",
				  fname                                   );
			}
			else
			{
			    dump_maclist(dmp);
			    fclose(dmp);
			}
		    }
		    else
		      dump_maclist(stderr);
		}
		break;
		
		/*}}}   */
	    case Defsfrom:
		/*{{{  read the given file */
		
		put_line(use_tabs,indent,text,1,FALSE);
		if (!hidden)
		{
		    char *fname = expand_fname(strtok(mac," \t\"\n"));
		
		    if (fname)
		    {
			long line_sav = line_no;
			char *in_sav = in_file;
			char *fn;
			int scanned_ok;
		
			if ((fn = (char *)malloc(strlen(fname) + 1)) == (char *)0)
			    mem_error("process definition indirect file");
		
			strcpy(fn,fname);
			scanned_ok = scan_def_file(fn);
			line_no = line_sav;
			in_file = in_sav;
			if (!scanned_ok)
			{
			    error(Error,in_file,line_no,
				  "Unable to open \"%s\" for reading definitions",
				  fname                                            );
			}
		    }
		    else
		      error(Error,in_file,line_no,"Missing or corrupt file name");
		}
		break;
		
		/*}}}   */
	    case Ordef:
	    case Orndef:
	    case Anddef:
	    case Andndef:
		/*{{{  check out it's effect */
		
		{
		    int OrNotAnd = (this_one == Ordef) || (this_one == Orndef);
		    int TrueCond = (this_one == Ordef) || (this_one == Anddef);
		
		    if (!at_top)
		    {
			if (annotate)
			{
			    /*{{{  build error message */
			    
			    char tmp[MAXFNAME];
			    
			    sprintf(tmp,"%s%sDEF %s --(erroneous orphan modifier)\n",
					OrNotAnd ? "OR" : "AND",
					TrueCond ? "" : "N",
					mac                                          );
			    put_line(use_tabs,indent,tmp,1,FALSE);
			    
			    /*}}}   */
			}
			else
			{
			    put_line(use_tabs,indent,text,1,FALSE);
			}
			error(Error,in_file,line_no,"Isolated condition modifier");
		    }
		    else
		    {
			Macro *macdef;
			int hit, which = 0;
		
			last_mod = line_no;
		
			hit = TrueCond ^ macro_defined(mac,&macdef);
			*why |= Cond_ETC;
		
			if (annotate)
			{
			    /*{{{  build and put annotated line */
			    
			    {
				char tmp[MAXFNAME];
				char exprno[32];
			    
				sprintf(exprno,"expr#%d)-(",which);
				sprintf(tmp,
					"#%s%sDEF %s  --(%s%s%s)-<%d>\n",
					OrNotAnd ? "OR" : "AND",
					TrueCond ? "" : "N",
					mac,
					(which && ((hit != 0)^(TrueCond != 0))) ? exprno : "",
					hit ? "FALSE" : "TRUE",
					def_src(macdef,TRUE),
					rec_level - 1);
				put_line(use_tabs,indent,tmp,1,FALSE);
			    }
			    
			    /*}}}   */
			}
			else
			{
			    put_line(use_tabs,indent,text,1,FALSE);
			}
		
			if (OrNotAnd)
			    encomment &= hit;
			else
			    encomment |= hit;
		    }
		}
		break;
		
		/*}}}   */
	    case Or:
	    case Orn:
	    case And:
	    case Andn:
		/*{{{  check out it's effect */
		
		{
		    int OrNotAnd = (this_one == Or) || (this_one == Orn);
		    int TrueCond = (this_one == Or) || (this_one == And);
		
		    if (!at_top)
		    {
			if (annotate)
			{
			    /*{{{  build error message */
			    
			    char tmp[MAXFNAME];
			    
			    sprintf(tmp,"%s%s %s --(erroneous orphan modifier)\n",
					OrNotAnd ? "OR" : "AND",
					TrueCond ? "" : "N",
					mac                                          );
			    put_line(use_tabs,indent,tmp,1,FALSE);
			    
			    /*}}}   */
			}
			else
			{
			    put_line(use_tabs,indent,text,1,FALSE);
			}
			error(Error,in_file,line_no,"Isolated condition modifier");
		    }
		    else
		    {
			char symbol[MAXFNAME] = "="; /* ie: no such macro */
			char value[MAXLINE];
			Macro *macdef;
			int hit, which = 0;
			char *captures[9];
			char *cp;
		
			last_mod = line_no;
		
			if ((cp = strchr(mac,'=')) == (char *)0)
			{
			    error(Error,
				  in_file,
				  line_no,
				  "#%s%s predicate is not a name=pattern pair",
				  OrNotAnd ? "OR" : "AND",
				  TrueCond ? "" : "N"      );
			}
			else
			{
			    strncpy(symbol,mac,cp - mac);
			    symbol[cp - mac + 1] = '\0';
			    cp++;
			    strcpy(value,cp);
		
			    if ((cp = strstr(value,"--")) != (char *)0)
				*cp = '\0';
			    else
				cp = value + strlen(value);
			    cp--;
		
			    while (isspace(*cp) && (cp >= value))
			    {
				*cp = '\0';
				cp--;
			    }
			}
		
			hit = macro_defined(symbol,&macdef);
			if (hit)
			{
			    if (macdef->value == (char *)0)
			    {
				error(Warn,
				      in_file,
				      line_no,
				      "Macro '%s' has no associated value",symbol);
				hit = (*value == '\0');
			    }
			    else
			    {
				hit = rxp_do_match(macdef->value,value,&which,captures);
				rxp_destroy_captures(captures);
			    }
			}
		
			hit = TrueCond ^ hit;
			*why |= Cond_ETC;
		
			if (annotate)
			{
			    /*{{{  build and put annotated line */
			    
			    {
				char tmp[MAXFNAME];
				char exprno[32];
			    
				sprintf(exprno,"expr#%d)-(",which);
				sprintf(tmp,
					"#%s%s %s=%s  --(%s%s%s)-<%d>\n",
					OrNotAnd ? "OR" : "AND",
					TrueCond ? "" : "N",
					symbol,
					value,
					(which && ((hit != 0)^(TrueCond != 0))) ? exprno : "",
					hit ? "FALSE" : "TRUE",
					def_src(macdef,TRUE),
					rec_level - 1);
				put_line(use_tabs,indent,tmp,1,FALSE);
			    }
			    
			    /*}}}   */
			}
			else
			{
			    put_line(use_tabs,indent,text,1,FALSE);
			}
		
			if (OrNotAnd)
			    encomment &= hit;
			else
			    encomment |= hit;
		    }
		}
		break;
		
		/*}}}   */
	    case Indent:
	    case Outdent:
		/*{{{  change the global indent level */
		
		{
		    char tmp[MAXLINE];
		    int delta = hidden ? 0 : ((this_one == Indent) ? 2 : -2);
		
		    global_indent += delta;
		    sprintf(tmp,
			    "#%sDENT%s%s",
			    this_one == Indent ? "IN" : "OUT",
			    !hidden ? "ED" : "",
			    text + (this_one == Indent ? 7 : 8));
		    put_line(use_tabs,indent + delta,tmp,1,FALSE);
		}
		break;
		
		/*}}}   */
	    case Extract:
	    case Replace:
		/*{{{  try to perform the operation */
		
		at_top = FALSE;
		put_line(use_tabs,indent,text,1,FALSE);
		if (!hidden)
		{
		    /*{{{  declare */
		    
		    char *opname = (this_one == Extract) ? "EXTRACT" : "REPLACE";
		    char source[MAXLINE];
		    char *regexp, *repl = (char *)0;
		    int which, dummy, ok = TRUE;
		    char *cp2, *cp = strchr(mac,'=');
		    
		    /*}}}   */
		
		    /*{{{  scan the line */
		    
		    if (!cp || (cp == mac))
		    {
			ok = FALSE;
			error(Error,
			      in_file,
			      line_no,
			      "#%s does not start with <symbol>=",
			      opname                              );
		    }
		    else
		    {
			char delim;
		    
			*cp = '\0';
			cp++;
			delim = *cp;
			cp++;
			if (!delim || ((cp2 = strchr(cp,delim)) == (char *)0))
			{
			    ok = FALSE;
			    error(Error,
				  in_file,
				  line_no,
				  "bad regular expression delimiters in #%s",
				  opname                                     );
			}
			else
			{
			    regexp = cp;
			    *cp2 = '\0';
			    cp2++;
			    if (*cp2 == delim)
			    {
				cp2++;
				if (!*cp2 || ((cp = strchr(cp2,delim)) == (char *)0))
				{
				    ok = FALSE;
				    error(Error,
					  in_file,
					  line_no,
					  "bad replacement delimiter in #%s",
					  opname                             );
				}
				else
				{
				    repl = cp2;
				    *cp = '\0';
				    cp++;
				}
			    }
			    else
				cp = cp2;
			}
		    }
		    /*}}}   */
		
		    if (ok && scan_subst(cp,source,Error,Warn,&dummy))
		    {
			RxpToken *patn = (RxpToken *)0;
			int capture_no = 0;
		
			/* at this point we have character pointers:
			      mac =         <symbol>
			      regexp =      <regexp>
			      repl =        <replacement> | NULL
			      source =      <substituted source text>
			*/
			if (!rxp_error(rxp_create_pat( &regexp,
						       &patn,
						       &which,
						       TRUE,
						       '\0',
						       &capture_no),which))
			{
			    if (this_one == Extract)
			    {
				/*{{{  scan for a match and extract into macro */
				
				char buf[MAXLINE];
				char *capture[9];
				
				cp2 = (char *)0;
				cp = source;
				while (*cp && !cp2)
				{
				    /*{{{  search for a match */
				    
				    char *tst = cp;
				    int which = 1;
				    
				    rxp_init_captures(capture);
				    cp2 = rxp_alt_match(tst,patn,&which,capture);
				    if (cp2)
				    {
					cp2++;
					*cp2 = '\0';
				    }
				    else
				    {
					rxp_destroy_captures(capture);
					cp++;
				    }
				    
				    /*}}}   */
				}
				/*{{{  build the definition line */
				strcpy(buf,mac);
				strcat(buf,"=");
				if (cp2)
				{
				    if (repl)
				    {
					/*{{{  do a complex replacement */
					
					int i;
					
					#if DEBUG_RXCAPTS
					  printf("Capture strings:\n");
					  for (i = 0; i < 9; ++i)
					      if (capture[i])
						  printf("\\%d =>%s<=\n",i+1,&(capture[i][2]));
					      else
						  printf("\\%d\n",i+1);
					#endif
					
					cp2 = buf + strlen(buf);
					while (*repl)
					{
					    if ((*repl == '\\') && isdigit(*(repl+1)))
					    {
						i = *(repl+1) - '0';
						repl++;
						if (i)
						{
						    i--;
						    if (capture[i])
						    {
							strcpy(cp2,&(capture[i][2]));
							cp2 += (capture[i][0] + (capture[i][1] << 8));
						    }
						}
						else
						{
						    strcpy(cp2,cp);
						    cp2 += strlen(cp);
						}
					    }
					    else
					    {
						*cp2 = *repl;
						cp2++;
					    }
					    repl++;
					}
					*cp2 = '\0';
					
					/*}}}   */
				    }
				    else
				    {
					strcat(buf,cp); /* it's a simple one */
				    }
				}
				/*}}}   */
				rxp_destroy_captures(capture);
				ok = def_macro(buf,M_immed,in_file,line_no,out_line_no + 1L);
				
				/*}}}   */
			    }
			    else
			    {
				/*{{{  scan for matches and replace them in the macro */
				
				char buf[MAXLINE];
				char *bmax = buf + MAXLINE - 1;
				char *bp;
				char *capture[9];
				
				cp = source;
				strcpy(buf,mac);
				strcat(buf,"=");
				bp = buf + strlen(buf);
				while (*cp && (bp < bmax))
				{
				    char *tst = cp;
				    int which = 1;
				
				    rxp_init_captures(capture);
				    cp2 = rxp_alt_match(tst,patn,&which,capture);
				    if (cp2)
				    {
					if (repl)
					{
					    /*{{{  patch in the replacement */
					    
					    char *rpp = repl;
					    
					    while (*rpp)
					    {
						if ((*rpp == ESCAPE) && isdigit(*(rpp+1)))
						{
						    int i = *(rpp+1) - '1';
					    
						    if (i < 0)
							while (cp <= cp2)
							{
							    *bp = *cp;
							    bp++;
							    cp++;
							}
						    else if (capture[i])
						    {
							strcpy(bp,&(capture[i][2]));
							bp += (capture[i][0] + (capture[i][1] << 8));
						    }
						    rpp++;
						}
						else
						{
						    *bp = *rpp;
						    bp++;
						}
						rpp++;
					    }
					    
					    /*}}}   */
					}
					cp = cp2;
					cp++;
				    }
				    else
				    {
					*bp = *cp;
					bp++;
					cp++;
				    }
				    rxp_destroy_captures(capture);
				}
				
				*bp = '\0';
				ok = def_macro(buf,M_immed,in_file,line_no,out_line_no + 1L);
				
				/*}}}   */
			    }
			}
			rxp_destroy_pat(patn);
		    }
		}
		break;
		
		/*}}}   */
	    case Include:
	    case Use:
	    case Import:
		/*{{{  expand the filename if need be, else just print it */
		
		if (hidden)
		{
		    put_line(use_tabs,indent,text,TRUE,FALSE);
		}
		else
		{
		    char fname[MAXFNAME];
		    char *fn, *xn;
		
		    strcpy(fname,mac);
		    fn = strtok(fname," \t\"\n");
		    xn = expand_fname(fn);
		    if (xn == fn)
		    {
			put_line(use_tabs,indent,text,FALSE,FALSE);
		    }
		    else
		    {
			char tmp[MAXLINE];
		
			sprintf(tmp,"#%s \"%s\" %s%s",
				(this_one == Include) ? "INCLUDE" :
					    ((this_one == Import) ? "IMPORT" : "USE"),
				xn,
				resv_cmt,
				text      );
			put_line(use_tabs,indent,tmp,FALSE,FALSE);
		    }
		}
		
		break;
		
		/*}}}   */
	    default:
		disaster(100);
	}
    }
}

/*}}}   */
/*{{{  remove_comments */

void
remove_comments(void)
{
    Operator this_one;
    int indent, use_tab, folds;
    char *text, *mac;
    while ((this_one = get_line(&use_tab,&folds,&indent,&text,&mac,TRUE)) != EoF)
    {
	if (this_one != Null)
	{
	    char *cp;
	    if (annotate               &&
		(this_one != Vanilla)  &&
		(this_one != Blank)    &&
		(this_one != Extract)  &&
		(this_one != Template) &&
		(this_one != Memo)     &&
		(this_one != Error)    &&
		(this_one != Shell)    &&
		((cp = strstr(text,"  --(")) != (char *)0))
	    {
		*cp = '\n';
		cp++;
		*cp = '\0';
	    }
	    switch (this_one)
	    {
		case If:
		case Ifn:
		case Ifdef:
		case Ifndef:
		    if (fold_ifs == 2)
			put_line( use_tab,
				  indent + global_indent,
				  text,
				  folds,
				  folds  );
		    else
			put_line( use_tab,
				  indent + global_indent,
				  text,
				  fold_ifs,
				  fold_ifs  );
		    break;
		case Endif:
		    if (fold_ifs == 2)
			put_line( use_tab,
				  indent + global_indent,
				  text,
				  folds,
				  -folds  );
		    else
			put_line( use_tab,
				  indent + global_indent,
				  text,
				  fold_ifs,
				  -fold_ifs  );
		    break;
		default:
		    put_line( use_tab,
			      indent + global_indent,
			      text,
			      FALSE,
			      FALSE );
		    break;
	    }
	}
    }
}

/*}}}   */
/*{{{  puts_rle */

void
puts_rle(const unsigned char *str)
{
    while (*str)
    {
	if (*str == 255)
	{
	    printf("  Ocpp");
	}
	else if (*str > 128)
	{
	    int i = *str - 128;
	    while (i > 0)
	    {
		putchar(' ');
		i--;
	    }
	}
	else
	    putchar(*str);
	str++;
    }
}

/*}}}   */

/*}}}   */
/*{{{  main */

int
main(int argc, char *argv[])
{
    /*{{{  declare */
    
    int input_file = FALSE;
    int clean_up = FALSE;
    
    /*}}}   */

    /*{{{  show help if required */
    
    if (argc < 2)
    {
	puts_rle((unsigned char *)help_text);
	return (0);
    }
    
    /*}}}   */
    /*{{{  set up exit trap */
    
    atexit(do_cleanup);
    
    /*}}}   */
    /*{{{  read "ocpp.rc" files */
    
    {
	char buf[MAXFNAME];
	char *bp;
    
	strcpy(buf,argv[0]);
	bp = strrchr(buf,DIRSEP);
	if (bp)
	{
	    strcpy(bp + 1,"ocpp.rc");
	    (void)scan_def_file(buf);
	}
    }
    (void)scan_def_file("ocpp.rc");
    
    /*}}}   */
    /*{{{  parse command line */
    
    {
	int i;
	for (i = 1; i < argc; ++i)
	{
	    char *word = argv[i];
	    if ((*word == '/') || ((*word == '-') && (*(word + 1) != '\0')))
	    {
		int hit = FALSE;
		int opt = 0;
    
		(void)strupr(++word);
		while (!hit             &&
		       (cm_switch[opt]) &&
		       ((hit = !strcmp(cm_switch[opt],word)) == 0)) opt++;
		if (!hit)
		{
		    if ((*word == 'H') || (*word == '?'))
		    {
			puts_rle((unsigned char *)help_text);
			return(0);
		    }
		    error(Error,(char *)0,0,"Unknown option (%s)",word);
		}
		else
		{
		    switch (opt)
		    {
			/*{{{  O output file */
			
			case 0:
			    out_file = argv[++i];
			    break;
			
			/*}}}   */
			/*{{{  DE defs from env_var */
			
			case 1:
			    {
				char *ep;
				#ifdef __MSDOS__
				  (void)strupr(argv[i + 1]);
				#endif
				ep = getenv(argv[++i]);
				if (!ep || (*ep == '\0'))
				{
				    error(Error,(char *)0,0,"Not in environment (%s)",argv[i]);
				}
				else
				{
				    char *sym = strtok(ep," ");
				    while (sym)
				    {
					if (annotate)
					{
					    error(Info,
						  (char *)0,
						  0,
						  "Macro %s defined in environment variable %s",
						  sym,
						  argv[i] );
					}
					def_macro(sym,M_env_var,argv[i],0L,0L);
					sym = strtok((char *)0," ");
				    }
				}
			    }
			    break;
			
			/*}}}   */
			/*{{{  DF defs from file */
			
			case 2:
			    if (!scan_def_file(argv[++i]))
			    {
				error(Error,(char *)0,0,"Failed to find DF file %s",argv[i]);
			    }
			    break;
			
			/*}}}   */
			/*{{{  D def from cmd line */
			
			case 3:
			    {
				if (annotate)
				{
				    error(Info,
					  (char *)0,
					  0,
					  "Macro %s defined on command line",argv[i + 1]);
				}
				i += (def_macro(argv[i + 1],M_cmd_line,(char *)0,0L,0L) != 0);
			    }
			    break;
			
			/*}}}   */
			/*{{{  RMC cleanup mode */
			
			case 4:
			    clean_up = TRUE;
			    break;
			
			/*}}}   */
			/*{{{  MIP modify in place */
			
			case 5:
			    mod_in_place = TRUE;
			    break;
			
			/*}}}   */
			/*{{{  DBG annotate */
			
			case 6:
			    annotate = TRUE;
			    cl_annotate = TRUE;
			    break;
			
			/*}}}   */
			/*{{{  T2 assume int == INT16 */
			
			case 7:
			    int_16 = TRUE;
			    cl_int_16 = TRUE;
			    break;
			
			/*}}}   */
			/*{{{  IRV irrevocable */
			
			case 8:
			    irrevocable = TRUE;
			    cl_irrevocable = TRUE;
			    break;
			
			/*}}}   */
			/*{{{  U undef from cmd line */
			
			case 9:
			    {
				Macro *macdef;
				if (macro_defined(argv[i + 1],&macdef))
				{
				    macdef->defcode = M_undef;
				    macdef->src = (char *)0;
				    macdef->in_line = 0L;
				    macdef->out_line = 0L;
				    if (annotate)
				    {
					error(Info,(char *)0,0,"Macro %s undefined on command line",argv[i + 1]);
				    }
				}
				i++;
			    }
			    break;
			
			/*}}}   */
			/*{{{  Z show extra help */
			
			case 10:
			    puts_rle((unsigned char *)help_text);
			    puts_rle((unsigned char *)extra_help);
			    return (0);
			
			/*}}}   */
			/*{{{  PP chain to compiler */
			
			case 11:
			    if (input_file)
			    {
				error(Severe,
				      (char *)0,
				      0,
				      "Previous input file (%s) before PP option",argv[input_file]);
			    }
			    input_file = ++i;
			    clean_up = FALSE;
			    in_line_insert = FALSE;
			    irrevocable = FALSE;
			    cl_irrevocable = FALSE;
			    mod_in_place = TRUE;
			    chain_compiler = TRUE;
			    compile_arg = i;
			    i = argc;
			    break;
			
			/*}}}   */
		    }
		}
	    }
	    else
	    {
		if (input_file)
		{
		    error(Severe,
			  (char *)0,
			  0,
			  "Second input file specified (%s)",argv[i]);
		}
		else
		{
		    input_file = i;
		}
	    }
	}
	if (irrevocable && (chain_compiler || mod_in_place))
	{
	    error(Severe,
		  (char *)0,
		  0,
		  "Use of both IRV and %s switches is illegal",
		  chain_compiler ? "PP" : "MIP");
	}
    }
    
    /*}}}   */
    /*{{{  try to open input file */
    
    if (!strcmp(argv[input_file],"-"))
    {
	in_file = (char *)in_pipename;
	in = stdin;
	if (chain_compiler)
	    error(Severe,(char *)0,0,"Cannot chain compiler when using <stdin>");
	if (mod_in_place)
	    error(Severe,(char *)0,0,"Cannot modify <stdin> in-place");
    }
    else if (input_file)
    {
	char *buf;
	int add_ext;
	char *lp, *cp;
    
	in_file = argv[input_file];
	lp = in_file + strlen(in_file);
	cp = lp;
	while ((lp != in_file) && (*lp != DIRSEP))
	    lp--;
	while ((cp != lp) && (*cp != '.'))
	    cp--;
	add_ext = mod_in_place            &&
		  std_file_ext            &&
		  (*std_file_ext != '\0') &&
		  (*cp != '.');
	if (add_ext)
	{
	    buf = (char *)malloc(strlen(in_file) + 5);
	    if (!buf)
		error(Fatal,
		      (char *)0,
		      0,
		      "Insufficient memory");
	    strcpy(buf,in_file);
	    strcat(buf,".");
	    strcat(buf,std_file_ext);
	    in = fopen(buf,"rt");
	    in_file = buf;
	}
	else
	    in = fopen(in_file,"rt");
	if (in == (FILE *)0)
	{
	    error(Severe,
		  (char *)0,
		  0,
		  "Failed to find input file %s",
		  in_file                         );
	}
    }
    else
    {
	error(Severe,(char *)0,0,"No input file specified");
    }
    base_file = in_file;
    
    /*}}}   */
    /*{{{  try to open output file */
    
    if (mod_in_place)
    {
	char *tmp;
    
	out = fopen(tmpnam(temp_file),"wt");
	if (out == (FILE *)0)
	{
	    error(Fatal,(char *)0,0,"Failed to open temporary output file");
	}
	tmp = strrchr(argv[input_file],DIRSEP);
	if (!tmp)
	    tmp = &(argv[input_file][0]);
	else
	    tmp++;
	strcpy(tplt_name,tmp);
    }
    else if (out_file)
    {
	char *tmp;
    
	out = fopen(out_file,"wt");
	if (out == (FILE *)0)
	{
	    error(Severe,(char *)0,0,"Failed to open output file %s",out_file);
	}
	tmp = strrchr(out_file,DIRSEP);
	if (!tmp)
	    tmp = out_file;
	else
	    tmp++;
	strcpy(tplt_name,tmp);
    }
    else
    {
	out = stdout;
	out_file = (char *)out_pipename;
	(void)tmpnam(tplt_name);
    }
    if (!strchr(tplt_name,'.'))
	strcat(tplt_name,".");
    
    /*}}}   */
    /*{{{  log the start time */
    
    {
	time_t t = time((time_t *)0);
    
	start_time = ctime(&t);
	start_time[10] = '\0';
	start_time[19] = '\0';
	start_time[24] = '\0';
    }
    
    /*}}}   */
    /*{{{  process the data */
    
    if (clean_up)
    {
	remove_comments();
    }
    else
    {
	int dm = 0;
	(void)parse(FALSE,FALSE,0,&dm,0L,0L,(char *)0);
    }
    
    /*}}}   */
    /*{{{  chain the compiler and maybe tidy up after */
    
    if (chain_compiler && !return_code)
    {
	char buffer[MAXLINE];
	int i;
    
	fclose(in);
	fclose(out);
	remove(in_file);
	rename(temp_file,in_file);
	*temp_file = '\0';
	strcpy(buffer,compiler_name);
	for (i = compile_arg; i < argc; ++i)
	{
	    strcat(buffer," ");
	    strcat(buffer,argv[i]);
	}
	strcat(buffer,"\n");
	return_code = system(buffer);
    
	if (clean_after_chain && !annotate)
	{
	    in = fopen(in_file,"rt");
	    out = fopen(tmpnam(temp_file),"wt");
	    remove_comments();
	}
    }
    
    /*}}}   */
    /*{{{  clean up ready for exit */
    
    if (out != stdout) fclose(out);
    if (in != stdin) fclose(in);
    
    if (mod_in_place && (!return_code || chain_compiler))
    {
	remove(in_file);
	rename(temp_file,in_file);
	*temp_file = '\0';
    }
    
    /*}}}   */

    return (return_code);
}

/*}}}   */
