/*
	sp1tosddf.c
	sp1tosddf converts SP1 trace files to PABLO SDDF format.

	Koji Sekido
	Computer Science Department, Cornell University

	History:
	Nov 26 1993,	Fix unmatch in AIX kernel statistics items.
	Nov 30 1993,	Resplve CPU statistics value.
					Add "trace record header" event.
	Dec 14 1993,	Stream Attribute Packet; just comments.
	March 18 1994   Removed the IBM include file sources.  This
			code must now be run on SP1 machines.
*/
#include	<stdio.h>
#include	<time.h>
#include	<fcntl.h>
#define SDDF   "sp1.ascii"
#include "sys/time.h"

typedef struct _my_ev_hdr {
        long    evh_prev_len;
        long    evh_len;
        long    evh_type;
        long  evh_subtype;
        struct  timestruc_t     evh_time;
        long    evh_proc;
} my_ev_header;


/* The following should be found in /usr/lpp/poe/include/ or some such */
/* It describes the IBM SP1's binary tracefile format */
#include <vvtrtrc.h>


main(argc, argv)
	int	argc;
	char	*argv[];
{
	int	fd;
	FILE	*fp;

	my_ev_header	evh;

	VT_tracefile_hdr	tracefile_hdr;
	VT_mp_bsend_beg	bsend_beg;		/* VT_MP_BSEND 0x01 */
	VT_mp_bsend_end	bsend_end;
	VT_mp_brecv_beg	brecv_beg;		/* VT_MP_BRECV 0x02 */
	VT_mp_brecv_end	brecv_end;
	VT_mp_bsendrecv_beg	bsendrecv_beg;		/* VT_MP_BSENDRECV 0x03 */
	VT_mp_bsendrecv_end	bsendrecv_end;
	VT_mp_send_beg	send_beg;		/* VT_MP_SEND 0x04 */
	VT_mp_send_end	send_end;
	VT_mp_recv_beg	recv_beg;		/* VT_MP_RECV 0x05 */
	VT_mp_recv_end	recv_end;
	VT_mp_status_beg	status_beg;		/* VT_MP_STATUS 0x06 */
	VT_mp_status_end	status_end;
	VT_mp_wait_beg	wait_beg;		/* VT_MP_WAIT 0x07 */
	VT_mp_wait_end	wait_end;
	VT_mp_probe_beg	probe_beg;		/* VT_MP_PROBE 0x08 */
	VT_mp_probe_end	probe_end;
	VT_mp_environ_beg	environ_beg;		/* VT_MP_ENVIRON 0x09 */
	VT_mp_environ_end	environ_end;
	VT_mp_stopall_beg	stopall_beg;		/* VT_MP_STOPALL 0x0a */
	VT_mp_stopall_end	stopall_end;
	VT_mp_taskquery_beg	taskquery_beg;		/* VT_MP_TASK_QUERY 0x0b */
	VT_mp_taskquery_end	taskquery_end;
	VT_mp_taskset_beg	taskset_beg;		/* VT_MP_TASK_SET 0x0c */
	VT_mp_taskset_end	taskset_end;
	/* VT_mp_shuf	shuf;		/* VT_MP_SHUF 0x0d */
	/* VT_mp_unshuf	unshuf;		/* VT_MP_UNSHUF 0x0e */
	/* VT_mp_packsh	packsh;		/* VT_MP_PACK 0x0f */
	/* VT_mp_unpacksh	unpacksh;		/* VT_MP_UNPACK 0x10 */
	VT_mp_bvrecv_beg	bvrecv_beg;		/* VT_MP_BVRECV 0x11 */
	VT_mp_bvrecv_end	bvrecv_end;
	VT_mp_bvsend_beg	bvsend_beg;		/* VT_MP_BVSEND 0x12 */
	VT_mp_bvsend_end	bvsend_end;
	VT_mp_vrecv_beg	vrecv_beg;		/* VT_MP_VRECV 0x13 */
	VT_mp_vrecv_end	vrecv_end;
	VT_mp_vsend_beg	vsend_beg;		/* VT_MP_VSEND 0x14 */
	VT_mp_vsend_end	vsend_end;

	VT_aix_stats	aix_stats;		/* VT_AIX_STATS 0x20 */
	VT_appl_marker	appl_marker;		/* VT_APPL_MARKER 0x31 */
	VT_recv_info	recv_info;		/* VT_RECV_INFO 0x41 */

	int	evhsize;
	int	recordsize;
	int	b;
	int	subtype;
	int	nproc;

	char	dummy[1024];

	evhsize = sizeof(evh);

	switch (argc) {
		case 2:
			if ((fp = fopen(SDDF, "w")) == NULL) {
				fprintf(stderr, "Can't open %s\n", SDDF);
				exit(1);
			}
			break;
		case 3:
			if ((fp = fopen(argv[2], "w")) == NULL) {
				fprintf(stderr, "Can't open %s\n", argv[2]);
				exit(1);
			}
			break;
		default:
			fprintf(stderr,
			"Usage: sp1tosddf <input_filename> [<output_filename>]\n");
			exit(1);
			break;
	}

	if ((fd = open(argv[1], O_RDONLY)) == NULL) {
		fclose(fp);
		fprintf(stderr, "Can't open %s\n", argv[1]);
		exit(1);
	}

	fprintf(fp, "SDDFA\n");

	fprintf(fp, "#999:\n");		/* This is for trace record header.
								"999" means nothing */
	fprintf(fp, "\"trace record header\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"subtype\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_BSEND | BEGIN_REC);	/* 0x01 */
	fprintf(fp, "\"blocking send begin\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"instructionAddress\";\n");
	fprintf(fp, "\tint\t\"destination\";\n");
	fprintf(fp, "\tint\t\"type\";\n");
	fprintf(fp, "\tint\t\"messageLength\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_BSEND);
	fprintf(fp, "\"blocking send end\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"returnCode\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_BRECV | BEGIN_REC);	/* 0x02 */
	fprintf(fp, "\"blocking receive begin\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"instructionAddress\";\n");
	fprintf(fp, "\tint\t\"source\";\n");
	fprintf(fp, "\tint\t\"type\";\n");
	fprintf(fp, "\tint\t\"messageLength\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_BRECV);
	fprintf(fp, "\"blocking receive end\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"numberofBytes\";\n");
	fprintf(fp, "\tint\t\"returnSrc\";\n");
	fprintf(fp, "\tint\t\"returnType\";\n");
	fprintf(fp, "\tint\t\"returnCode\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_BSENDRECV | BEGIN_REC);	/* 0x03 */
	fprintf(fp, "\"blocking sendreceive begin\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"instructionAddress\";\n");
	fprintf(fp, "\tint\t\"destination\";\n");
	fprintf(fp, "\tint\t\"source\";\n");
	fprintf(fp, "\tint\t\"type\";\n");
	fprintf(fp, "\tint\t\"outMessageLength\";\n");
	fprintf(fp, "\tint\t\"inMessageLength\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_BSENDRECV);
	fprintf(fp, "\"blocking sendreceive end\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"numberofBytes\";\n");
	fprintf(fp, "\tint\t\"returnSrc\";\n");
	fprintf(fp, "\tint\t\"returnCode\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_SEND | BEGIN_REC);	/* 0x04 */
	fprintf(fp, "\"non-blocking send begin\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"instructionAddress\";\n");
	fprintf(fp, "\tint\t\"destination\";\n");
	fprintf(fp, "\tint\t\"type\";\n");
	fprintf(fp, "\tint\t\"messageLength\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_SEND);
	fprintf(fp, "\"non-blocking send end\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"messageId\";\n");
	fprintf(fp, "\tint\t\"returnCode\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_RECV | BEGIN_REC);	/* 0x05 */
	fprintf(fp, "\"non-blocking receive begin\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"instructionAddress\";\n");
	fprintf(fp, "\tint\t\"source\";\n");
	fprintf(fp, "\tint\t\"type\";\n");
	fprintf(fp, "\tint\t\"messageLength\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_RECV);
	fprintf(fp, "\"non-blocking receive end\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"messageId\";\n");
	fprintf(fp, "\tint\t\"returnSource\";\n");
	fprintf(fp, "\tint\t\"returnType\";\n");
	fprintf(fp, "\tint\t\"returnCode\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_STATUS | BEGIN_REC);	/* 0x06 */
	fprintf(fp, "\"status begin\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"instructionAddress\";\n");
	fprintf(fp, "\tint\t\"messageId\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_STATUS);
	fprintf(fp, "\"status end\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"returnCode\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_WAIT | BEGIN_REC);	/* 0x07 */
	fprintf(fp, "\"non-blocking wait begin\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"instructionAddress\";\n");
	fprintf(fp, "\tint\t\"messageId\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_WAIT);
	fprintf(fp, "\"non-blocking wait end\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"returnMessageId\";\n");
	fprintf(fp, "\tint\t\"numberofBytes\";\n");
	fprintf(fp, "\tint\t\"returnCode\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_PROBE | BEGIN_REC);	/* 0x08 */
	fprintf(fp, "\"probe begin\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"instructionAddress\";\n");
	fprintf(fp, "\tint\t\"source\";\n");
	fprintf(fp, "\tint\t\"type\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_PROBE);
	fprintf(fp, "\"probe end\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"numberofBytes\";\n");
	fprintf(fp, "\tint\t\"returnSource\";\n");
	fprintf(fp, "\tint\t\"returnType\";\n");
	fprintf(fp, "\tint\t\"returnCode\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_ENVIRON | BEGIN_REC);	/* 0x09 */
	fprintf(fp, "\"environ begin\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"instructionAddress\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_ENVIRON);
	fprintf(fp, "\"environ end\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"numberofTask\";\n");
	fprintf(fp, "\tint\t\"taskId\";\n");
	fprintf(fp, "\tint\t\"returnCode\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_STOPALL | BEGIN_REC);	/* 0x0a */
	fprintf(fp, "\"stopall begin\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"instructionAddress\";\n");
	fprintf(fp, "\tint\t\"errorCode\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_STOPALL);
	fprintf(fp, "\"stopall end\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"returnCode\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_TASK_QUERY | BEGIN_REC);	/* 0x0b */
	fprintf(fp, "\"task query begin\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"instructionAddress\";\n");
	fprintf(fp, "\tint\t\"numberofElements\";\n");
	fprintf(fp, "\tint\t\"queryType\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_TASK_QUERY);
	fprintf(fp, "\"task query end\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"returnCode\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_TASK_SET | BEGIN_REC);	/* 0x0c */
	fprintf(fp, "\"task set begin\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"instructionAddress\";\n");
	fprintf(fp, "\tint\t\"setType\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_TASK_SET);
	fprintf(fp, "\"task set end\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"returnCode\";\n");
	fprintf(fp, "};;\n");

	/* fprintf(fp, "#%d:\n", VT_MP_SHUF);	/* 0x0d */
	/* fprintf(fp, "\"shuf\" {\n"); */
	/* fprintf(fp, "\tint\t\"bufLen\";\n"); */
	/* fprintf(fp, "\tint\t\"shLen\";\n"); */
	/* fprintf(fp, "\tint\t\"next\";\n"); */
	/* fprintf(fp, "\tint\t\"returnCode\";\n"); */
	/* fprintf(fp, "};;\n"); */

	/* fprintf(fp, "#%d:\n", VT_MP_UNSHUF);	/* 0x0e */
	/* fprintf(fp, "\"unshuf\" {\n"); */
	/* fprintf(fp, "\tint\t\"bufLen\";\n"); */
	/* fprintf(fp, "\tint\t\"shLen\";\n"); */
	/* fprintf(fp, "\tint\t\"next\";\n"); */
	/* fprintf(fp, "\tint\t\"returnCode\";\n"); */
	/* fprintf(fp, "};;\n"); */

	/* fprintf(fp, "#%d:\n", VT_MP_PACK);	/* 0x0f */
	/* fprintf(fp, "\"pack end\" {\n"); */
	/* fprintf(fp, "\tint\t\"returnCode\";\n"); */
	/* fprintf(fp, "};;\n"); */

	/* fprintf(fp, "#%d:\n", VT_MP_UNPACK);	/* 0x10 */
	/* fprintf(fp, "\"unpack end\" {\n"); */
	/* fprintf(fp, "\tint\t\"returnCode\";\n"); */
	/* fprintf(fp, "};;\n"); */

	fprintf(fp, "#%d:\n", VT_MP_BVRECV | BEGIN_REC);	/* 0x11 */
	fprintf(fp, "\"bvrecv begin\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"instructionAddress\";\n");
	fprintf(fp, "\tint\t\"blockLength\";\n");
	fprintf(fp, "\tint\t\"offset\";\n");
	fprintf(fp, "\tint\t\"blockNumber\";\n");
	fprintf(fp, "\tint\t\"source\";\n");
	fprintf(fp, "\tint\t\"type\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_BVRECV);
	fprintf(fp, "\"bvrecv end\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"returnSource\";\n");
	fprintf(fp, "\tint\t\"returnType\";\n");
	fprintf(fp, "\tint\t\"numberofBytes\";\n");
	fprintf(fp, "\tint\t\"returnCode\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_BVSEND | BEGIN_REC);	/* 0x12 */
	fprintf(fp, "\"bvsend begin\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"instructionAddress\";\n");
	fprintf(fp, "\tint\t\"blockLength\";\n");
	fprintf(fp, "\tint\t\"offset\";\n");
	fprintf(fp, "\tint\t\"blockNumber\";\n");
	fprintf(fp, "\tint\t\"destination\";\n");
	fprintf(fp, "\tint\t\"type\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_BVSEND);
	fprintf(fp, "\"bvsend end\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"returnCode\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_VRECV | BEGIN_REC);	/* 0x13 */
	fprintf(fp, "\"vrecv begin\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"instructionAddress\";\n");
	fprintf(fp, "\tint\t\"blockLength\";\n");
	fprintf(fp, "\tint\t\"offset\";\n");
	fprintf(fp, "\tint\t\"blockNumber\";\n");
	fprintf(fp, "\tint\t\"source\";\n");
	fprintf(fp, "\tint\t\"type\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_VRECV);
	fprintf(fp, "\"vrecv end\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"returnSource\";\n");
	fprintf(fp, "\tint\t\"returnType\";\n");
	fprintf(fp, "\tint\t\"returnMsgId\";\n");
	fprintf(fp, "\tint\t\"returnCode\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_VSEND | BEGIN_REC);	/* 0x14 */
	fprintf(fp, "\"vsend begin\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"instructionAddress\";\n");
	fprintf(fp, "\tint\t\"blockLength\";\n");
	fprintf(fp, "\tint\t\"offset\";\n");
	fprintf(fp, "\tint\t\"blockNumber\";\n");
	fprintf(fp, "\tint\t\"destination\";\n");
	fprintf(fp, "\tint\t\"type\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_MP_VSEND);
	fprintf(fp, "\"vsend end\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"returnMsgId\";\n");
	fprintf(fp, "\tint\t\"returnCode\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_AIX_STATS);	/* 0x20 */
	fprintf(fp, "\"AIX statistics\" {\n");
/*
	fprintf(fp, "\tint\t\"cpuUtil\";\n");
*/
	fprintf(fp, "\tint\t\"cpuUser\";\n");
	fprintf(fp, "\tint\t\"cpuKernel\";\n");
	fprintf(fp, "\tint\t\"cpuWait\";\n");
	fprintf(fp, "\tint\t\"cpuIdle\";\n");
	fprintf(fp, "\tint\t\"contextSwitch\";\n");
	fprintf(fp, "\tint\t\"systemCalls\";\n");
	fprintf(fp, "\tint\t\"pageFaults\";\n");
	fprintf(fp, "\tint\t\"totalDiskTransfers\";\n");
	fprintf(fp, "\tint\t\"blocksRead\";\n");
	fprintf(fp, "\tint\t\"blocksWritten\";\n");
	fprintf(fp, "\tint\t\"ipPacketsRcvd\";\n");
	fprintf(fp, "\tint\t\"ipPacketsSent\";\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_APPL_MARKER);	/* 0x31 */
	fprintf(fp, "\"appl marker\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"instructionAddress\";\n");
	fprintf(fp, "\tint\t\"light\";\n");
	fprintf(fp, "\tint\t\"color\";\n");
	fprintf(fp, "\tchar\t\"msg\"[];\n");
	fprintf(fp, "};;\n");

	fprintf(fp, "#%d:\n", VT_RECV_INFO);	/* 0x41 */
	fprintf(fp, "\"receive info\" {\n");
	fprintf(fp, "\tint\t\"timeSecond\";\n");
	fprintf(fp, "\tint\t\"timeNanosecond\";\n");
	fprintf(fp, "\tint\t\"process\";\n");
	fprintf(fp, "\tint\t\"returnMessageId\";\n");
	fprintf(fp, "\tint\t\"returnSource\";\n");
	fprintf(fp, "\tint\t\"returnType\";\n");
	fprintf(fp, "};;\n");

	while (b = read(fd, &evh, evhsize)) {
		if (evh.evh_type == evh_type_EUI1) {
			fprintf(fp, "\"trace record header\" {");
			fprintf(fp, " %d, %d, %d, %d",
			evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
			evh.evh_proc, evh.evh_subtype);
			fprintf(fp, "};;\n");
		}
		switch (evh.evh_type) {
			case evh_type_VTHDR:
				recordsize = sizeof(tracefile_hdr);
				read(fd, &tracefile_hdr, recordsize);
				fprintf(fp, "/*\n");
				fprintf(fp, " * program name: %s\n", tracefile_hdr.pgmname);
				fprintf(fp, " * %s", ctime(&(evh.evh_time.tv_sec)));
				fprintf(fp, " * tracefile type: %d\n", tracefile_hdr.type);
				fprintf(fp, " * process number: %d\n", tracefile_hdr.num_procs);
				fprintf(fp, " */ ;;\n");
				nproc = tracefile_hdr.num_procs;
				break;
			case evh_type_AIX_STATS:
				recordsize = sizeof(aix_stats);
				read(fd, &aix_stats, recordsize);
				fprintf(fp, "\"AIX statistics\" {");
				fprintf(fp, " %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d, %d ",
				GET_CPU_USER(aix_stats.cpu_util),
				GET_CPU_KERN(aix_stats.cpu_util),
				GET_CPU_WAIT(aix_stats.cpu_util),
				GET_CPU_IDLE(aix_stats.cpu_util),
				aix_stats.cswitch,
				aix_stats.syscalls, aix_stats.pagefaults,
				aix_stats.total_xfers, aix_stats.blocks_read,
				aix_stats.blocks_written, aix_stats.ip_packets_rcvd,
				aix_stats.ip_packets_sent);
				fprintf(fp, "};;\n");
				break;
			case evh_type_EUI1:
				switch (evh.evh_subtype) {
					case VT_MP_BSEND | BEGIN_REC:		/* 0x01 */
						recordsize = sizeof(bsend_beg);
						read(fd, &bsend_beg, recordsize);
						fprintf(fp, "\"blocking send begin\" {");
						fprintf(fp, " %d, %d, %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						bsend_beg.iadr, bsend_beg.dest, bsend_beg.type,
						bsend_beg.msglen);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_BSEND:
						recordsize = sizeof(bsend_end);
						read(fd, &bsend_end, recordsize);
						fprintf(fp, "\"blocking send end\" {");
						fprintf(fp, " %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						bsend_end.retcode);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_BRECV | BEGIN_REC:		/* 0x02 */
						recordsize = sizeof(brecv_beg);
						read(fd, &brecv_beg, recordsize);
						fprintf(fp, "\"blocking receive begin\" {");
						fprintf(fp, " %d, %d, %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						brecv_beg.iadr, brecv_beg.source == -1 ? nproc :
						brecv_beg.source, brecv_beg.type,
						brecv_beg.msglen);
						fprintf(fp, "};;\n");
						/*
						if (brecv_beg.source == -1)
							printf("Hex: %0x\n", brecv_beg.source);
							perror("NEGATIVE");
						*/
						break;
					case VT_MP_BRECV:
						recordsize = sizeof(brecv_end);
						read(fd, &brecv_end, recordsize);
						fprintf(fp, "\"blocking receive end\" {");
						fprintf(fp, " %d, %d, %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						brecv_end.nbytes, brecv_end.returnsrc == -1 ?
						nproc : brecv_end.returnsrc,
						brecv_end.returntype, brecv_end.retcode);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_BSENDRECV | BEGIN_REC:		/* 0x03 */
						recordsize = sizeof(bsendrecv_beg);
						read(fd, &bsendrecv_beg, recordsize);
						fprintf(fp, "\"blocking sendreceive begin\" {");
						fprintf(fp, " %d, %d, %d, %d, %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						bsendrecv_beg.iadr, bsendrecv_beg.dest,
						bsendrecv_beg.source, bsendrecv_beg.type,
						bsendrecv_beg.outmsglen, bsendrecv_beg.inmsglen);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_BSENDRECV:
						recordsize = sizeof(bsendrecv_end);
						read(fd, &bsendrecv_end, recordsize);
						fprintf(fp, "\"blocking sendreceive end\" {");
						fprintf(fp, " %d, %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						bsendrecv_end.nbytes, bsendrecv_end.returnsrc,
						bsendrecv_end.retcode);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_SEND | BEGIN_REC:		/* 0x04 */
						recordsize = sizeof(send_beg);
						read(fd, &send_beg, recordsize);
						fprintf(fp, "\"non-blocking send begin\" {");
						fprintf(fp, " %d, %d, %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						send_beg.iadr, send_beg.dest, send_beg.type,
						send_beg.msglen);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_SEND:
						recordsize = sizeof(send_end);
						read(fd, &send_end, recordsize);
						fprintf(fp, "\"non-blocking send end\" {");
						fprintf(fp, " %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						send_end.msgid, send_end.retcode);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_RECV | BEGIN_REC:		/* 0x05 */
						recordsize = sizeof(recv_beg);
						read(fd, &recv_beg, recordsize);
						fprintf(fp, "\"non-blocking receive begin\" {");
						fprintf(fp, " %d, %d, %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						recv_beg.iadr, recv_beg.source == -1 ? nproc :
						recv_beg.source, recv_beg.type,
						recv_beg.msglen);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_RECV:
						recordsize = sizeof(recv_end);
						read(fd, &recv_end, recordsize);
						fprintf(fp, "\"non-blocking receive end\" {");
						fprintf(fp, " %d, %d, %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						recv_end.msgid, recv_end.returnsrc == -1 ? nproc
						: recv_end.returnsrc,
						recv_end.returntype, recv_end.retcode);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_STATUS | BEGIN_REC:		/* 0x06 */
						recordsize = sizeof(status_beg);
						read(fd, &status_beg, recordsize);
						fprintf(fp, "\"status begin\" {");
						fprintf(fp, " %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						status_beg.iadr, status_beg.msgid);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_STATUS:
						recordsize = sizeof(status_end);
						read(fd, &recv_end, recordsize);
						fprintf(fp, "\"status end\" {");
						fprintf(fp, " %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						status_end.retcode);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_WAIT | BEGIN_REC:		/* 0x07 */
						recordsize = sizeof(wait_beg);
						read(fd, &wait_beg, recordsize);
						fprintf(fp, "\"non-blocking wait begin\" {");
						fprintf(fp, " %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						wait_beg.iadr, wait_beg.msgid);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_WAIT:
						recordsize = sizeof(wait_end);
						read(fd, &wait_end, recordsize);
						fprintf(fp, "\"non-blocking wait end\" {");
						fprintf(fp, " %d, %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						wait_end.returnmsgid, wait_end.nbytes,
						wait_end.retcode);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_PROBE | BEGIN_REC:		/* 0x08 */
						recordsize = sizeof(probe_beg);
						read(fd, &probe_beg, recordsize);
						fprintf(fp, "\"probe begin\" {");
						fprintf(fp, " %d, %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						probe_beg.iadr, probe_beg.source,
						probe_beg.type);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_PROBE:
						recordsize = sizeof(probe_end);
						read(fd, &probe_end, recordsize);
						fprintf(fp, "\"probe end\" {");
						fprintf(fp, " %d, %d, %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						probe_end.nbytes, probe_end.returnsrc,
						probe_end.returntype, probe_end.retcode);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_ENVIRON | BEGIN_REC:		/* 0x09 */
						recordsize = sizeof(environ_beg);
						read(fd, &environ_beg, recordsize);
						fprintf(fp, "\"environ begin\" {");
						fprintf(fp, " %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						environ_beg.iadr);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_ENVIRON:
						recordsize = sizeof(environ_end);
						read(fd, &environ_end, recordsize);
						fprintf(fp, "\"environ end\" {");
						fprintf(fp, " %d, %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						environ_end.numtask, environ_end.taskid,
						environ_end.retcode);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_STOPALL | BEGIN_REC:		/* 0x0a */
						recordsize = sizeof(stopall_beg);
						read(fd, &stopall_beg, recordsize);
						fprintf(fp, "\"stopall begin\" {");
						fprintf(fp, " %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						stopall_beg.iadr, stopall_beg.errcode);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_STOPALL:
						recordsize = sizeof(stopall_end);
						read(fd, &stopall_end, recordsize);
						fprintf(fp, "\"stopall end\" {");
						fprintf(fp, " %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						stopall_end.retcode);
						break;
					case VT_MP_TASK_QUERY | BEGIN_REC:		/* 0x0b */
						recordsize = sizeof(taskquery_beg);
						read(fd, &taskquery_beg, recordsize);
						fprintf(fp, "\"task query begin\" {");
						fprintf(fp, " %d, %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						taskquery_beg.iadr, taskquery_beg.elem,
						taskquery_beg.qtype);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_TASK_QUERY:
						recordsize = sizeof(taskquery_end);
						read(fd, &taskquery_beg, recordsize);
						fprintf(fp, "\"task query end\" {");
						fprintf(fp, " %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						taskquery_end.retcode);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_TASK_SET | BEGIN_REC:		/* 0x0c */
						recordsize = sizeof(taskset_beg);
						read(fd, &taskset_beg, recordsize);
						fprintf(fp, "\"task set begin\" {");
						fprintf(fp, " %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						taskset_beg.iadr, taskset_beg.stype);
						fprintf(fp, "};;\n");
						break;
					case VT_MP_TASK_SET:
						recordsize = sizeof(taskset_end);
						read(fd, &taskset_beg, recordsize);
						fprintf(fp, "\"task set end\" {");
						fprintf(fp, " %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						taskset_end.retcode);
						fprintf(fp, "};;\n");
						break;
					case VT_APPL_MARKER:
						recordsize = sizeof(appl_marker);
						read(fd, &appl_marker, recordsize);
						fprintf(fp, "\"appl marker\" {");
						fprintf(fp, " %d, %d, %d, %d, %d, %d,\n",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						appl_marker.iadr, appl_marker.light,
						appl_marker.color);
						fprintf(fp, "\t[%d] {\n", MAX_MARKER_STRLEN);
						fprintf(fp, "\t\"%s\"", appl_marker.msg);
						fprintf(fp, "\t}\n");
						fprintf(fp, "};;\n");
						break;
					case VT_RECV_INFO:
						recordsize = sizeof(recv_info);
						read(fd, &recv_info, recordsize);
						fprintf(fp, "\"receive info\" {");
						fprintf(fp, " %d, %d, %d, %d, %d, %d ",
						evh.evh_time.tv_sec, evh.evh_time.tv_nsec,
						evh.evh_proc,
						recv_info.returnmsgid, recv_info.returnsource ==
						-1 ? nproc : recv_info.returnsource,
						recv_info.returntype);
						fprintf(fp, "};;\n");
						break;
					default:
						fprintf(stderr, "Unknown subtype! %d\n",
						evh.evh_subtype);
						exit(2);
						break;
				}
				break;
			default:
				fprintf(stderr, "Unknown type! %d\n", evh.evh_type);
				exit(1);
				break;
		}
		if (b = evh.evh_len - evhsize - recordsize)
			read(fd, dummy, b);
	}
	close(fd);
	fclose(fp);
}
