/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *          (R)
 * The Pablo    Performance Analysis Environment software is NOT in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1991-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * PABLO is a registered trademark of
 * The Board of Trustees of the University of Illinois
 * registered in the U.S. Patent and Trademark Office.
 *
 * Author:  Keith A. Shields (shields@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004,
 * by a grant from the Digital Equipment Corporation External Research
 * Program, and by a collaborative research agreement with the Intel
 * Supercomputer Systems Division.
 *
 */

/*
 *
 */


#include "MenuBar.h"



/**************************************************************************** 
  This function is registered as the callback for the load file
  selection dialog.  If the callback reason is XmCR_OK (ok button
  pressed) then the selected file is loaded into the source code viewer.
  The file selection dialog is destroyed and a boolean to allow only one 
  file selection dialog at a time is reset.
*****************************************************************************/
void fsdLoadSource(button, fsd, call_data)
     Widget button;
     Widget fsd;
     XmFileSelectionBoxCallbackStruct *call_data;
{
  char *fullFileName;

  if (call_data->reason == XmCR_OK) {
    XmStringGetLtoR(call_data->value, XmSTRING_DEFAULT_CHARSET, &fullFileName);

    XtUnmanageChild(fsd);
    loadSourceFile(fullFileName, getOptionCount(), getOptions());
    free(fullFileName);
  } 

  XtDestroyWidget(fsd);
  FSD_Exists = False;
}



/**************************************************************************** 
  This function is registered as the callback to the menu selections that
  lets the user choose to instrument all routine calls in the program.  The 
  linked list of instrumentable constructs is traversed, and all entries 
  that represent calls have their instrumentation type set to the value of
  the callback's client data (None,Count,Trace) depending on which menu 
  selection was made.
*****************************************************************************/
void instrumentAllCalls(button, instrumentation_type, call_data)
     Widget button;
     INSTTYPES instrumentation_type;
     XtPointer call_data;
{
  INSTREC *tmpPtr = instrumentHead;

  while(tmpPtr != NULL) {
    if (tmpPtr->constructType == FunctionCall)
      { tmpPtr->instrumentType = instrumentation_type;
	LineData[tmpPtr->beginLine-1].instrumentationType = 
	  instrumentation_type;
     }
    tmpPtr = tmpPtr->next;
  }
  updateTextSymbols();
  StateChangedSinceSave = True;
}



/**************************************************************************** 
  This function is registered as the callback to the menu selections that
  lets the user choose to instrument all calls in a selected routine. The 
  linked list of instrumentable constructs is traversed, and all entries 
  of routines that represent calls in the selected routine have their 
  instrumentation type set to the value of the callback's client data 
  (None,Count,Trace) depending on which menu selection was made.
*****************************************************************************/
void instrumentAllCallsInSelectedRoutine(button, instrumentation_type,
					 call_data)
     Widget button;
     INSTTYPES instrumentation_type;
     XtPointer call_data;
{
  INSTREC *tmpPtr = instrumentHead;
  XmString invokingXString;

  while (tmpPtr != NULL) {
    invokingXString = XmStringCreateSimple(tmpPtr->invokingFunc);
    if (XmStringCompare(SelectedRoutine, invokingXString)) {
      if (tmpPtr->constructType == FunctionCall) {
	tmpPtr->instrumentType = instrumentation_type;
	LineData[tmpPtr->beginLine-1].instrumentationType = 
	  instrumentation_type;
      }
    }
    tmpPtr = tmpPtr->next;
    XmStringFree(invokingXString);
  }
  updateTextSymbols();
  StateChangedSinceSave = True;
}



/**************************************************************************** 
  This function is registered as the callback to the menu selections that
  lets the user choose to instrument all calls to a selected routine. The 
  linked list of instrumentable constructs is traversed, and all entries 
  of routines that represent calls to the selected routine have their 
  instrumentation type set to the value of the callback's client data 
  (None,Count,Trace) depending on which menu selection was made.
*****************************************************************************/
void instrumentAllCallsToSelectedRoutine(button, instrumentation_type,
					 call_data)
     Widget button;
     INSTTYPES instrumentation_type;
     XtPointer call_data;
{
  INSTREC *tmpPtr = instrumentHead;
  XmString invokedXString;

  while (tmpPtr != NULL) {
    invokedXString = XmStringCreateSimple(tmpPtr->invokedFunc);
    if (XmStringCompare(SelectedRoutine, invokedXString)) {
      if (tmpPtr->constructType == FunctionCall) {
	tmpPtr->instrumentType = instrumentation_type;
	LineData[tmpPtr->beginLine-1].instrumentationType = 
	  instrumentation_type;
      }
    }
    tmpPtr = tmpPtr->next;
    XmStringFree(invokedXString);
  }
  updateTextSymbols();
  StateChangedSinceSave = True;
}



/**************************************************************************** 
  This function is registered as the callback to the menu selections that
  lets the user choose to instrument all outer loops in the program.  The 
  linked list of instrumentable constructs is traversed, and all entries 
  that represent loops have their instrumentation type set to the value of
  the callback's client data (None,Count,Trace) depending on which menu 
  selection was made.
*****************************************************************************/
void instrumentAllLoops(button, instrumentation_type, call_data)
     Widget button;
     INSTTYPES instrumentation_type;
     XtPointer call_data;
{
  INSTREC *tmpPtr = instrumentHead;

  while(tmpPtr != NULL) {
    if (tmpPtr->constructType == OuterLoop){ 
      tmpPtr->instrumentType = instrumentation_type;
      LineData[tmpPtr->beginLine-1].instrumentationType = 
	instrumentation_type;
     }
    tmpPtr = tmpPtr->next;
  }
  updateTextSymbols();
  StateChangedSinceSave = True;
}



/**************************************************************************** 
  This function is registered as the callback to the menu selections that
  lets the user choose to instrument outer loops of a selected routine. The 
  linked list of instrumentable constructs is traversed, and all entries 
  of the selected routine that represent loops have their instrumentation 
  type set to the value of the callback's client data (None,Count,Trace) 
  depending on which menu selection was made.
*****************************************************************************/
void instrumentOuterLoopsOfSelectedRoutine(button, instrumentation_type, 
					   call_data)
     Widget button;
     INSTTYPES instrumentation_type;
     XtPointer call_data;
{
  INSTREC *tmpPtr = instrumentHead;
  XmString invokingXString;

  while (tmpPtr != NULL) {
    invokingXString = XmStringCreateSimple(tmpPtr->invokingFunc);
    if (XmStringCompare(SelectedRoutine, invokingXString)) {
      if (tmpPtr->constructType == OuterLoop) {
	tmpPtr->instrumentType = instrumentation_type;
	LineData[tmpPtr->beginLine-1].instrumentationType = 
	  instrumentation_type;	
      }
    }
    tmpPtr = tmpPtr->next;
    XmStringFree(invokingXString);
  }
  updateTextSymbols();
  StateChangedSinceSave = True;
}



/****************************************************************************
*****************************************************************************/
void makeInstrumentedSource(button, mainWin, call_data)
     Widget button;
     Widget mainWin;
     XtPointer call_data;
{
  char *instFileName, *instString, *configString, *configName, buf[1024];

  instString = strdup("Instrumented_");
  instFileName = (char *) malloc(sizeof(char)*(strlen(SourceFile) +
					       strlen(instString) + 1));

  sprintf(instFileName, "%s%s", instString, SourceFile);

  configString = strdup(".config");
  configName = (char *) malloc(sizeof(char)*(strlen(SourceFile) + 
					     strlen(configString) + 2));

  sprintf(configName,".%s%s", SourceFile, configString);

  generateInstrumentedSourceFile(instFileName);
  saveInstrumentationConfiguration(configName);
  writeEventIDs(".project");
  generateInstrumentationInitialization("InstrumentationInitialization.c");

  StateChangedSinceSave = False;

  sprintf(buf,"Instrumentation saved in %s",instFileName);
  generateInfo(buf);

  free(instFileName);
  free(instString);
  free(configString);
  free(configName);
}



/**************************************************************************** 
  This function is used to instrument a subset of calls in the selected
  routine.  A dialog is popped up and the user is the able to choose a 
  subset of routines that are called by the currently selected routine, and 
  only calls in the currently selected routine found in that subset are 
  instrumented.
*****************************************************************************/
void instrumentSubsetInSelectedRoutine(button, client_data, call_data)
     Widget button;
     XtPointer client_data;
     XtPointer call_data;
{
  INSTREC *tmpPtr = instrumentHead;
  int i, pos;
  char *tmpStr, errMsg[1024];
  XmString tmpXStr;

  while (tmpPtr != NULL) {
    if (tmpPtr->constructType == FunctionCall) {
      if (addUniqueItemToSubsetList(tmpPtr->invokingFunc, tmpPtr->invokedFunc)) {
	ListData = (listItemStruct *) realloc(ListData, sizeof(listItemStruct)*
					      (NumSubsetItems+1));
	ListData[NumSubsetItems].item = strdup(tmpPtr->invokedFunc);
	ListData[NumSubsetItems].instrumentationType =
	  tmpPtr->instrumentType;
	NumSubsetItems++;
      } 
    }
    tmpPtr = tmpPtr->next;
  }

  if (NumSubsetItems == 0) {
    XmStringGetLtoR(SelectedRoutine, XmSTRING_DEFAULT_CHARSET, &tmpStr);
    sprintf(errMsg,"%s does not call any other routines", tmpStr);
    generateError(errMsg);
  } else {
      DialogType = subsetIn;
      XmStringGetLtoR(SelectedRoutine, XmSTRING_DEFAULT_CHARSET, &tmpStr);
      setSubsetDialogValues("Instrument Calls In Selected Routine", tmpStr,
			    "Makes Calls To");
      popupSubsetDialog();
    }
  free(tmpStr);
  StateChangedSinceSave = True;
}



/**************************************************************************** 
  This function is used to instrument a subset of calls to the selected
  routine.  A dialog is popped up and the user is the able to choose a 
  subset of routines that call the currently selected routine, and only 
  calls to the currently selected routine found in that subset are 
  instrumented.
*****************************************************************************/
void instrumentSubsetToSelectedRoutine(button, client_data, call_data)
     Widget button;
     XtPointer client_data;     
     XtPointer call_data;
{
  INSTREC *tmpPtr = instrumentHead;
  int i, pos;
  char *tmpStr, errMsg[1024];
  XmString tmpXStr;

  while (tmpPtr != NULL) {
    if (tmpPtr->constructType == FunctionCall) {
      if (addUniqueItemToSubsetList(tmpPtr->invokedFunc, tmpPtr->invokingFunc)) {
	ListData = (listItemStruct *) realloc(ListData, sizeof(listItemStruct)*
					      (NumSubsetItems+1));
	ListData[NumSubsetItems].item = strdup(tmpPtr->invokingFunc);
	ListData[NumSubsetItems].instrumentationType =
	  tmpPtr->instrumentType;
	NumSubsetItems++;
      }
    }
    tmpPtr = tmpPtr->next;
  }

  if (NumSubsetItems == 0) {
    XmStringGetLtoR(SelectedRoutine, XmSTRING_DEFAULT_CHARSET, &tmpStr);
    sprintf(errMsg,"%s is not called by any other routines", tmpStr);
    generateError(errMsg);
  } else {
      DialogType = subsetTo;
      XmStringGetLtoR(SelectedRoutine, XmSTRING_DEFAULT_CHARSET, &tmpStr);
      setSubsetDialogValues("Instrument Calls To Selected Routine", tmpStr,
			    "Is Called By");

      popupSubsetDialog();
    }
  free(tmpStr);
  StateChangedSinceSave = True;
}



/**************************************************************************** 
*****************************************************************************/
void loadInstrumentationConfiguration(fileName)
     char* fileName;
{
  struct stat filestatus;
  FILE *fp;
  INSTREC tmpRec, *tmpPtr=instrumentHead;
  INSTTYPES *instTypeArray;
  char buf[1024], buf1[1024], buf2[1024];
  int numRecordsRead = 0, numRecordsProcessed = 0;
  Boolean positionsShifted = False;

  if ((fp = fopen(fileName, "r")) != NULL) {

    instTypeArray = (INSTTYPES *) malloc(sizeof(INSTTYPES));

    while (!feof(fp)) {
      if ( fscanf(fp, "%s %s %d %d %d %d \n", buf1, buf2,
		  &(tmpRec.beginLine), &(tmpRec.endLine), 
		  &(tmpRec.constructType), 
		  &(tmpRec.instrumentType)) == 6) {
	tmpRec.previous = NULL;
	tmpRec.next = NULL;
	tmpRec.invokingFunc = strdup(buf1);
	tmpRec.invokedFunc = strdup(buf2);

	if ((tmpPtr != NULL) && (recordsAreSame(tmpPtr, &tmpRec))) {
	  if (! structurePositionsAreSame(tmpPtr, &tmpRec)) {
	    positionsShifted = True;
	  }

	  instTypeArray = (INSTTYPES *) realloc(instTypeArray, 
						sizeof(INSTTYPES)*
						(numRecordsRead+1));
	  instTypeArray[numRecordsRead] = tmpRec.instrumentType;
	  numRecordsRead++;
	  free(tmpRec.invokingFunc);
	  free(tmpRec.invokedFunc);
	  tmpPtr = tmpPtr-> next;
	} else {
	  generateError(
	    "Saved Instrumentation Does Not Match Current Source File"); 
	  free(tmpRec.invokingFunc);
	  free(tmpRec.invokedFunc);
	  return;
	}
      } else {
	generateError("Error Reading Instrumentation File");
	return;
      }
    }

    if (tmpPtr != NULL) {
      generateError(
	"Saved Instrumentation Does Not Match Current Source File"); 
      return;
    } 

    if (positionsShifted) {
      generateInfo("Source File Has Changed Since Instrumentation Was Saved");
    }

    tmpPtr = instrumentHead;
    numRecordsProcessed = 0;

    while (tmpPtr != NULL) {
      tmpPtr->instrumentType = instTypeArray[numRecordsProcessed];
      LineData[tmpPtr->beginLine-1].instrumentationType = 
	instTypeArray[numRecordsProcessed];
      numRecordsProcessed++;
      tmpPtr = tmpPtr->next;
    }
    updateTextSymbols();
    free(instTypeArray);
    fclose(fp);
  }
}



/****************************************************************************
  This function is registered as the callback for the Open option under the
  File menu.  It creates a file selection dialog that the user will use to
  indicate which source file to load.  A global boolean is maintained to 
  allow only one file selction dialog to be created at a time.
*****************************************************************************/
void loadSource(button, mainWin, call_data)
     Widget button;
     Widget mainWin;
     XtPointer call_data;
{
  Widget fsd;
  XmString tmpStr;

  if ( (fsd = makeFSD(mainWin)) != NULL) {
    tmpStr = XmStringCreateSimple("Open Source File");
    XtVaSetValues(fsd, XmNdialogTitle, tmpStr, NULL);
    XmStringFree(tmpStr);
		  
    XtAddCallback(fsd, XmNokCallback, fsdLoadSource, fsd);
    XtAddCallback(fsd, XmNcancelCallback, fsdLoadSource, fsd);
  }
}



/**************************************************************************** 
 This function reads the selected file into the source code viewer.  Checks 
 are performed to ensure that the file exists and is readable.
*****************************************************************************/
void loadSourceFile(fileName, argc, argv)
     char *fileName;
     int  argc;
     char **argv;
{
  struct stat filestatus;
  FILE *fp;
  char *fileText, ch, buf[1024], *postpendString, *configName, *lastSlash;
  int fileSize, sourcePosition, charsRead;
  Boolean parseFailed;
  XmString parseMsgX;
  Widget parseDialog;
  XEvent event;

  if ((fp = fopen(fileName, "r")) == NULL) {
    sprintf(buf,"Can't Open File %s", fileName);
    generateError(buf);
    return;
  }

  if (stat(fileName, &filestatus) == -1) {
    sprintf(buf,"Can't Read Status Of %s", fileName);
    generateError(buf);
    return;
  }

  if ((filestatus.st_mode & S_IFMT) == S_IFDIR) {
    sprintf(buf,"%s is a directory", fileName);
    generateError(buf);
    return;
  }

  /* in case doing a reload, need to clean up from the previous file */
  XmTextSetHighlight(SourceViewer, 0, XmTextGetLastPosition(SourceViewer),
		     XmHIGHLIGHT_NORMAL);

  free(LineData);
  NumSourceLines = 0;
  LineData = (lineInfoStruct *) malloc(sizeof(lineInfoStruct));

  sourcePosition = 0;
  charsRead = 0;
  LineData[0].startCharPos = 0;
  LineData[0].firstInstrumentableStructure = NULL;
  LineData[0].instrumentationType = InstrumentOff;
  /* first line will be selected initially */
  LineData[0].isLineSelected = True;
  CurrentLineNum = 0;

  fileSize = filestatus.st_size;
  fileText = XtMalloc((unsigned) fileSize+1);

  /* read the file in setting up internal data strucutres to track lines */
  fileText[charsRead] = ch = fgetc(fp);
  while (!feof(fp)) {
    if (ch == '\n') {
      NumSourceLines++;
      LineData = 
	(lineInfoStruct *) realloc(LineData, sizeof(lineInfoStruct)*
				   (NumSourceLines+1));
      LineData[NumSourceLines].startCharPos = sourcePosition+1;
      LineData[NumSourceLines].firstInstrumentableStructure = NULL;
      LineData[NumSourceLines].instrumentationType = InstrumentOff;
      LineData[NumSourceLines].isLineSelected = False;
    }
    sourcePosition++;
    charsRead++;
    fileText[charsRead] = ch = fgetc(fp);
  }

  fclose(fp);

  /* first line of file is initially selected */


  fileText[fileSize-1] = '\0';

  XmTextSetString(SourceViewer, fileText);

  readEventIDs(".project");
  parseFailed = False;

  XSynchronize(XtDisplay(SourceViewer), True);

  parseDialog = XmCreateWorkingDialog(SourceViewer, "parseDialog", NULL, 0);
  sprintf(buf,"%s is now being parsed.", fileName);
  parseMsgX = XmStringCreateSimple(buf);
  XtVaSetValues(parseDialog, 
		XmNmessageString, parseMsgX,
		NULL);

  XtUnmanageChild(XmMessageBoxGetChild(parseDialog, XmDIALOG_CANCEL_BUTTON));
  XtUnmanageChild(XmMessageBoxGetChild(parseDialog, XmDIALOG_HELP_BUTTON));
  XtUnmanageChild(XmMessageBoxGetChild(parseDialog, XmDIALOG_OK_BUTTON));
  XtUnmanageChild(XmMessageBoxGetChild(parseDialog, XmDIALOG_SEPARATOR));
  XtManageChild(parseDialog);

  while ( XtAppPending( App ) ) {
    XtAppNextEvent( App, &event );
    XtDispatchEvent( &event );
  }
  XSynchronize(XtDisplay(SourceViewer), False);

  if (parseFile(fileName, argc, argv) != 0) {
    sprintf(buf,"Can't Parse File %s", fileName);
    generateError(buf);
    parseFailed = True;
  }

  XtDestroyWidget(parseDialog);

  setFileNameLabel(fileName);

  if ((lastSlash = strrchr(fileName, '/')) != NULL) {
    SourceFile = strdup(lastSlash+1);
  } else {
    SourceFile = strdup(fileName);
  }

  if (! parseFailed) {
    makeRoutineList();

    postpendString = strdup(".config");
    configName = (char *) malloc(sizeof(char)*(strlen(SourceFile) + 
		       strlen(postpendString)+2));

    sprintf(configName,".%s%s", SourceFile, postpendString);
    loadInstrumentationConfiguration(configName);

    free(postpendString);
    free(configName);
    setupLines();
    updateTextSymbols();
  }

  XtSetSensitive(Global, True);  
  XtSetSensitive(Routine, True);
  XtSetSensitive(TraceButton, True);
  XtSetSensitive(CountButton, True);
  XtSetSensitive(TraceButton, True);
  XtSetSensitive(ClearButton, True);
  XtSetSensitive(InstrumentSource, True);

  StateChangedSinceSave = False;

  XtFree(fileText);
}



/****************************************************************************
*****************************************************************************/
Widget makeFSD(mainWin)
     Widget mainWin;
{
  Widget fsd;
  Arg wargs[1];

  if (!FSD_Exists) {
    XtSetArg(wargs[0], XmNinitialResourcesPersistent, False);
    fsd = XmCreateFileSelectionDialog(mainWin, "fsd", wargs, 1);
    XtManageChild(fsd);
    FSD_Exists = True;
    return fsd;
  }
  return NULL;
}



/****************************************************************************
  This function creates the file menu off of the main menu bar as well as any 
  needed submenus.  Callbacks to perform the menus specified task are 
  registered as well.
*****************************************************************************/
void makeFileMenu(menuBar, mainWin)
     Widget menuBar, mainWin;
{
  Widget fileMenu, loadSrc, editOpt, quit, sep;

 /* File Menu */
  File = 
    XtCreateManagedWidget("File", xmCascadeButtonWidgetClass, menuBar, NULL, 
			  0);
  fileMenu =
    XmCreatePulldownMenu(menuBar, "fileMenu", NULL, 0);
  XtVaSetValues(File, XmNsubMenuId, fileMenu, NULL);

  /* Open File */
  loadSrc = 
    XtCreateManagedWidget("loadSource", xmPushButtonWidgetClass, fileMenu, 
			  NULL, 0);
  XtAddCallback(loadSrc, XmNactivateCallback, loadSource, mainWin);

  /* Edit Options */
  editOpt = 
    XtCreateManagedWidget("editOptions", xmPushButtonWidgetClass, fileMenu, 
			  NULL, 0);
  XtAddCallback(editOpt, XmNactivateCallback, editOptions, NULL);

  /* Instrument Source Code */
  InstrumentSource = 
    XtVaCreateManagedWidget("InstrumentSource", xmPushButtonWidgetClass, 
			    fileMenu, XmNsensitive, False, NULL);
  XtAddCallback(InstrumentSource, XmNactivateCallback, makeInstrumentedSource, 
		mainWin);


  /* Separator */
  sep = 
    XtCreateManagedWidget("sep", xmSeparatorWidgetClass, fileMenu, NULL, 0);


  /* Quit */
  quit = 
    XtCreateManagedWidget("quit", xmPushButtonWidgetClass, fileMenu, NULL, 0);  XtAddCallback(quit, XmNactivateCallback, quitProgram, mainWin);
}



/****************************************************************************
  This function creates the global menu off of the main menu bar as well as 
  any needed submenus.  Callbacks to perform the menus specified task are 
  registered as well.
*****************************************************************************/
void makeGlobalMenu(menuBar)
     Widget menuBar;
{
  Widget globalMenu, callButton, loopButton, bothButton,
         callMenu, loopMenu, bothMenu, traceB, countB, clear, sep;

  /* Global Instrument Menu */
  Global = 
    XtVaCreateManagedWidget("Global", xmCascadeButtonWidgetClass, 
                            menuBar, XmNsensitive, False, NULL);
  globalMenu = 
    XmCreatePulldownMenu(menuBar, "globalMenu", NULL, 0);
  XtVaSetValues(Global, XmNsubMenuId, globalMenu, NULL);


  /* Call SubMenu */
  callButton = 
    XtCreateManagedWidget("callButton", xmCascadeButtonWidgetClass, globalMenu,
                         NULL, 0);
  callMenu = 
    XmCreatePulldownMenu(globalMenu, "callMenu", NULL, 0);
  XtVaSetValues(callButton, XmNsubMenuId, callMenu, NULL);
  traceB = 
    XtCreateManagedWidget("trace", xmPushButtonWidgetClass, callMenu, NULL, 0);
  countB = 
    XtCreateManagedWidget("count", xmPushButtonWidgetClass, callMenu, NULL, 0);
  sep = 
    XtCreateManagedWidget("sep", xmSeparatorWidgetClass, callMenu, NULL, 0);
  clear = 
    XtCreateManagedWidget("clear", xmPushButtonWidgetClass, callMenu, NULL, 0);
  XtAddCallback(traceB, XmNactivateCallback, instrumentAllCalls, InstrumentTrace);
  XtAddCallback(countB, XmNactivateCallback, instrumentAllCalls, InstrumentCount);
  XtAddCallback(clear, XmNactivateCallback, instrumentAllCalls, InstrumentOff);


  /* Loop SubMenu */
  loopButton = 
    XtCreateManagedWidget("loopButton", xmCascadeButtonWidgetClass, globalMenu,
                          NULL, 0);
  loopMenu = 
    XmCreatePulldownMenu(globalMenu, "loopMenu", NULL, 0);
  XtVaSetValues(loopButton, XmNsubMenuId, loopMenu, NULL);
  traceB = 

    XtCreateManagedWidget("trace", xmPushButtonWidgetClass, loopMenu, NULL, 0);
  countB = 
    XtCreateManagedWidget("count", xmPushButtonWidgetClass, loopMenu, NULL, 0);
  sep = 
    XtCreateManagedWidget("sep", xmSeparatorWidgetClass, loopMenu, NULL, 0);
  clear = 
    XtCreateManagedWidget("clear", xmPushButtonWidgetClass, loopMenu, NULL, 0);
  XtAddCallback(traceB, XmNactivateCallback, instrumentAllLoops, InstrumentTrace);
  XtAddCallback(countB, XmNactivateCallback, instrumentAllLoops, InstrumentCount);
  XtAddCallback(clear, XmNactivateCallback, instrumentAllLoops, InstrumentOff);


  /* Both SubMenu */
  bothButton = 
    XtCreateManagedWidget("bothButton", xmCascadeButtonWidgetClass, globalMenu,
                          NULL, 0);
  bothMenu = 
    XmCreatePulldownMenu(globalMenu, "bothMenu", NULL, 0);
  XtVaSetValues(bothButton, XmNsubMenuId, bothMenu, NULL);
  traceB = 
    XtCreateManagedWidget("trace", xmPushButtonWidgetClass, bothMenu, NULL, 0);
  countB = 
    XtCreateManagedWidget("count", xmPushButtonWidgetClass, bothMenu, NULL, 0);
  sep = 
    XtCreateManagedWidget("sep", xmSeparatorWidgetClass, bothMenu, NULL, 0);
  clear = 
    XtCreateManagedWidget("clear", xmPushButtonWidgetClass, bothMenu, NULL, 0);
  XtAddCallback(traceB, XmNactivateCallback, instrumentAllCalls, InstrumentTrace);
  XtAddCallback(traceB, XmNactivateCallback, instrumentAllLoops, InstrumentTrace);
  XtAddCallback(countB, XmNactivateCallback, instrumentAllCalls, InstrumentCount);
  XtAddCallback(countB, XmNactivateCallback, instrumentAllLoops, InstrumentCount);
  XtAddCallback(clear, XmNactivateCallback, instrumentAllCalls, InstrumentOff);
  XtAddCallback(clear, XmNactivateCallback, instrumentAllLoops, InstrumentOff);
}



/****************************************************************************
  This function creates the help menu off of the main menu bar as well as any 
  needed submenus.  Callbacks to perform the menus specified task are 
  registered as well.
*****************************************************************************/
void makeHelpMenu(menuBar, mainWin)
     Widget menuBar, mainWin;
{
  Widget helpMenu, onFile, onGlobal, onRoutine, onRoutineList, onSourceViewer;

 /* Help Menu */
  Help = 
    XtCreateManagedWidget("Help", xmCascadeButtonWidgetClass, menuBar, NULL, 
			  0);
  helpMenu =
    XmCreatePulldownMenu(menuBar, "helpMenu", NULL, 0);
  XtVaSetValues(Help, XmNsubMenuId, helpMenu, NULL);


  /* Help On File */
  onFile = 
    XtCreateManagedWidget("onFile", xmPushButtonWidgetClass, helpMenu, 
			  NULL, 0);
  XtAddCallback(onFile, XmNactivateCallback, generateHelp, FILE_HELP_TEXT);

  /* Help On Global */
  onGlobal = 
    XtCreateManagedWidget("onGlobal", xmPushButtonWidgetClass, helpMenu, 
			  NULL, 0);
  XtAddCallback(onGlobal, XmNactivateCallback, generateHelp, GLOBAL_HELP_TEXT);

  /* Help On Routine */
  onRoutine = 
    XtCreateManagedWidget("onRoutine", xmPushButtonWidgetClass, helpMenu, 
			  NULL, 0);
  XtAddCallback(onRoutine, XmNactivateCallback, generateHelp, 
		ROUTINE_HELP_TEXT);

  /* Help On Routine List */
  onRoutineList = 
    XtCreateManagedWidget("onRoutineList", xmPushButtonWidgetClass, helpMenu, 
			  NULL, 0);
  XtAddCallback(onRoutineList, XmNactivateCallback, generateHelp, 
		ROUTINE_LIST_HELP_TEXT);

  /* Help On Source Code Viewer */
  onSourceViewer = 
    XtCreateManagedWidget("onSourceViewer", xmPushButtonWidgetClass, helpMenu, 
			  NULL, 0);
  XtAddCallback(onSourceViewer, XmNactivateCallback, generateHelp, 
		SOURCE_VIEWER_HELP_TEXT);
}



/****************************************************************************
  This function creates the main menu bar. Functions that handle the creation
  of specific menus off the menu bar are called. Since the top level shell is 
  a main window widget, we can set the new menu bar to that widget's 
  appropriate resource, so placement is handled automatically.
*****************************************************************************/
void makeMenuBar(mainWin)
     Widget mainWin;
{
  Widget menuBar; 

  menuBar = XmCreateMenuBar(mainWin, "menuBar", NULL, 0);

  makeFileMenu(menuBar, mainWin);
  makeGlobalMenu(menuBar);
  makeRoutineMenu(menuBar);
  makeHelpMenu(menuBar, mainWin);

  XtVaSetValues(menuBar, XmNmenuHelpWidget, Help, NULL);
  XtManageChild(menuBar);
  XtVaSetValues(mainWin, XmNmenuBar, menuBar, NULL);
}


/****************************************************************************
  This function creates the routine menu off of the main menu bar as well as 
  any needed submenus.  Callbacks to perform the menus specified task are 
  registered as well.
*****************************************************************************/
void makeRoutineMenu(menuBar)
     Widget menuBar;
{
  Widget routineMenu, callsToButton, callsToMenu, callsInButton, callsInMenu,
         outerLoopsButton, outerLoopsMenu, traceB, countB, clear, subset, sep;

  /* Routine Instrument Menu */
  Routine = 
    XtVaCreateManagedWidget("Routine", xmCascadeButtonWidgetClass, menuBar, 
			    XmNsensitive, False, NULL);
  routineMenu = 
    XmCreatePulldownMenu(menuBar, "routineMenu", NULL, 0);
  XtVaSetValues(Routine, XmNsubMenuId, routineMenu, NULL);


  /* Calls To SubMenu */
  callsToButton = 
    XtCreateManagedWidget("callsToButton", xmCascadeButtonWidgetClass, 
			  routineMenu, NULL, 0);
  callsToMenu = 
    XmCreatePulldownMenu(routineMenu, "callsToMenu", NULL, 0);
  XtVaSetValues(callsToButton, XmNsubMenuId, callsToMenu, NULL);
  traceB = 
    XtCreateManagedWidget("trace", xmPushButtonWidgetClass, callsToMenu, NULL,
			  0);
  countB = 
    XtCreateManagedWidget("count", xmPushButtonWidgetClass, callsToMenu, NULL,
			  0);
  subset = 
    XtCreateManagedWidget("subset", xmPushButtonWidgetClass, callsToMenu, NULL,
			  0);
  sep = 
    XtCreateManagedWidget("sep", xmSeparatorWidgetClass, callsToMenu, NULL, 0);
  clear = 
    XtCreateManagedWidget("clear", xmPushButtonWidgetClass,callsToMenu, NULL,
			  0);
  XtAddCallback(traceB, XmNactivateCallback, 
		instrumentAllCallsToSelectedRoutine, InstrumentTrace);
  XtAddCallback(countB, XmNactivateCallback, 
		instrumentAllCallsToSelectedRoutine, InstrumentCount);
  XtAddCallback(clear, XmNactivateCallback, 
		instrumentAllCallsToSelectedRoutine, InstrumentOff);
  XtAddCallback(subset, XmNactivateCallback, 
		instrumentSubsetToSelectedRoutine, NULL);


  /* Calls In SubMenu */
  callsInButton = 
    XtCreateManagedWidget("callsInButton", xmCascadeButtonWidgetClass, 
			  routineMenu, NULL, 0);
  callsInMenu = 
    XmCreatePulldownMenu(routineMenu, "callsInMenu", NULL, 0);
  XtVaSetValues(callsInButton, XmNsubMenuId, callsInMenu, NULL);
  traceB = 
    XtCreateManagedWidget("trace", xmPushButtonWidgetClass, callsInMenu, NULL,
			  0);
  countB = 
    XtCreateManagedWidget("count", xmPushButtonWidgetClass, callsInMenu, NULL, 
			  0);
  subset = 
    XtCreateManagedWidget("subset", xmPushButtonWidgetClass, callsInMenu, NULL,
			  0);
  sep = 
    XtCreateManagedWidget("sep", xmSeparatorWidgetClass, callsInMenu, NULL, 0);
  clear = 
    XtCreateManagedWidget("clear", xmPushButtonWidgetClass, callsInMenu, NULL,
			  0);
  XtAddCallback(traceB, XmNactivateCallback, 
		instrumentAllCallsInSelectedRoutine, InstrumentTrace);
  XtAddCallback(countB, XmNactivateCallback, 
		instrumentAllCallsInSelectedRoutine, InstrumentCount);
  XtAddCallback(clear, XmNactivateCallback, 
		instrumentAllCallsInSelectedRoutine, InstrumentOff);
  XtAddCallback(subset, XmNactivateCallback, 
		instrumentSubsetInSelectedRoutine, NULL);


  /* Outer Loops SubMenu */

  outerLoopsButton = 
    XtCreateManagedWidget("outerLoopsButton", xmCascadeButtonWidgetClass, 
                          routineMenu, NULL, 0);
  outerLoopsMenu = 
    XmCreatePulldownMenu(routineMenu, "outerLoopsMenu", NULL,0);
  XtVaSetValues(outerLoopsButton, XmNsubMenuId, outerLoopsMenu, NULL);
  traceB =
    XtCreateManagedWidget("trace", xmPushButtonWidgetClass, outerLoopsMenu, 
                          NULL, 0);
  countB =
    XtCreateManagedWidget("count", xmPushButtonWidgetClass, outerLoopsMenu, 
                          NULL, 0);
  sep = 
    XtCreateManagedWidget("sep", xmSeparatorWidgetClass, outerLoopsMenu, NULL,
                          0);
  clear =
    XtCreateManagedWidget("clear", xmPushButtonWidgetClass, outerLoopsMenu, 
                          NULL, 0);
  XtAddCallback(traceB, XmNactivateCallback, 
                instrumentOuterLoopsOfSelectedRoutine, InstrumentTrace);
  XtAddCallback(countB, XmNactivateCallback, 
                instrumentOuterLoopsOfSelectedRoutine, InstrumentCount);
  XtAddCallback(clear, XmNactivateCallback, 
                instrumentOuterLoopsOfSelectedRoutine, InstrumentOff);
}



/**************************************************************************** 
*****************************************************************************/
Boolean recordsAreSame(internalRecord, fileRecord)
     INSTREC *internalRecord;
     INSTREC *fileRecord;
{
  if (fileRecord->constructType == OuterLoop) {
    if ((strcmp(internalRecord->invokingFunc, fileRecord->invokingFunc) == 0)&&
	(internalRecord->constructType == fileRecord->constructType)) {
      return(True);
    } else {
      return(False);
    }
  } else {
    if ((strcmp(internalRecord->invokingFunc, fileRecord->invokingFunc) == 0)&&
	(strcmp(internalRecord->invokedFunc, fileRecord->invokedFunc) == 0) &&
	(internalRecord->constructType == fileRecord->constructType)) {
      return(True);
    } else {
      return(False);
    }
  }
}



/**************************************************************************** 
*****************************************************************************/
void saveInstrumentationConfiguration(fileName)
     char* fileName;
{
  struct stat filestatus;
  FILE *fp;
  INSTREC *tmpPtr;
  char buf[1024];
 
  tmpPtr = instrumentHead;

  if ((fp = fopen(fileName, "w")) == NULL) {
    sprintf(buf,"Can't Open File %s", fileName);
    generateError(buf);
    return;
  }

  if (stat(fileName, &filestatus) == -1) {
    sprintf(buf,"Can't Read Status Of %s", fileName);
    generateError(buf);
    return;
  }

  if ((filestatus.st_mode & S_IFMT) == S_IFDIR) {
    sprintf(buf, "%s is a directory", fileName);
    generateError(buf);
    return;
  }

  while (tmpPtr != NULL) {
    fprintf(fp, "%s %s %d %d %d %d \n", tmpPtr->invokingFunc, 
	    tmpPtr->invokedFunc, tmpPtr->beginLine, tmpPtr->endLine, 
	    tmpPtr->constructType, tmpPtr->instrumentType);
    tmpPtr = tmpPtr->next;
  }

  fclose(fp);
}



/**************************************************************************** 
*****************************************************************************/
void setFileNameLabel(fileName)
     char *fileName;
{
  XmString fileNameX;

  fileNameX = XmStringCreateSimple(fileName);

  XtVaSetValues(FileNameLabel, 
		XmNlabelString, fileNameX,
		NULL);

  XmStringFree(fileNameX);
}



/**************************************************************************** 
*****************************************************************************/
void setupLines()
{

  INSTREC *tmpPtr = instrumentHead;

  while (tmpPtr != NULL) {
    XmTextSetHighlight(SourceViewer, LineData[tmpPtr->beginLine-1].
		       startCharPos, LineData[tmpPtr->beginLine].
		       startCharPos-1, XmHIGHLIGHT_SELECTED);

    /* lines can have multiple instrumentable constructs, but we just track
       of the first one since we can "walk" down the linked list to find
       others on this line */

    if (LineData[tmpPtr->beginLine-1].firstInstrumentableStructure == NULL) {
      LineData[tmpPtr->beginLine-1].firstInstrumentableStructure = tmpPtr;
    }
    tmpPtr = tmpPtr->next;
  }
}



/**************************************************************************** 
*****************************************************************************/
Boolean structurePositionsAreSame(internalRecord, fileRecord)
     INSTREC *internalRecord;
     INSTREC *fileRecord;
{
  if ((internalRecord->beginLine == fileRecord->beginLine) &&
      (internalRecord->endLine == fileRecord->endLine)) {
    return(True);
  } else {
    return(False);
  }
}



/****************************************************************************
*****************************************************************************/
void quitOK(dialog, client_data, call_data)
     Widget dialog;
     XtPointer client_data;
     XtPointer call_data;
{
/*  free linked list and malloc'd array items too **********************/

  free(LineData);
  free(ListData);
  XmStringFree(SelectedRoutine);

  XtCloseDisplay(display); 
  exit(0);
}



/****************************************************************************
*****************************************************************************/
void quitCancel(dialog, client_data, call_data)
     Widget dialog;
     XtPointer client_data;
     XtPointer call_data;
{
  XtDestroyWidget(dialog);
}



/****************************************************************************
  This function is registered as the callback for the Quit.  The user is
  asked to verify quitting  actually wishes to quit before exiting
*****************************************************************************/
void quitProgram(button, client_data, call_data)
     Widget button;
     XtPointer client_data;
     XtPointer call_data;
{
  char buf[256];
  Widget verifyDialog;
  XmString verifyText;

  if (StateChangedSinceSave == True) {

    verifyDialog = 
      XmCreateQuestionDialog(XtParent(button), "verifyDialog", NULL, 0);

    sprintf(buf,"Quit without saving changes?");
    verifyText = XmStringCreateLtoR(buf, XmSTRING_DEFAULT_CHARSET);
    XtVaSetValues(verifyDialog, 
		  XmNmessageString, verifyText, 
		  NULL);

    XmStringFree(verifyText);
    XtUnmanageChild(XmMessageBoxGetChild(verifyDialog, XmDIALOG_HELP_BUTTON));

    XtAddCallback(verifyDialog, XmNokCallback, quitOK, NULL);
    XtAddCallback(verifyDialog, XmNcancelCallback, quitCancel, NULL);
    XtManageChild(verifyDialog);
  } else {
    quitOK(NULL, NULL, NULL);
  }
}



