/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *          (R)
 * The Pablo    Performance Analysis Environment software is NOT in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1991-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * PABLO is a registered trademark of
 * The Board of Trustees of the University of Illinois
 * registered in the U.S. Patent and Trademark Office.
 *
 * Author:  Roger J. Noe (noe@cs.uiuc.edu)
 * Contributing Author:  Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004,
 * by a grant from the Digital Equipment Corporation External Research
 * Program, and by a collaborative research agreement with the Intel
 * Supercomputer Systems Division.
 *
 */

/*
 * Parser.c:
 *	This file contains various pieces of code needed by the Pablo
 *	instrumenting C parser.  Functions found here are called either by
 *	the lexical analyzer function yylex (produced via lex Scanner.l)
 *	or the LALR(1) parser function yyparse (produced via yacc Grammar.y).
 *	Also provided here are some utility functions and top-level
 *	interface functions, by which the instrumenting parser is invoked.
 */

#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <stdio.h>
#include <fcntl.h>

#include "Parser.h"
#include "y.tab.h"


/*	GLOBAL DATA STRUCTURE DEFINITIONS AND INITIALIZATION		    */


#ifdef STANDALONE
char	       *programName;		/* Name of this program		    */
#endif /* STANDALONE */

SYM	       *currentFunc =		/* Current function pointer	    */
			(SYM *) NULL;

CONTEXT	       *currentContext =	/* Current context tree element	    */
			(CONTEXT *) NULL;

TRACETREE      *currentTrace =		/* Traceable elements tree pointer  */
			(TRACETREE *) NULL;

IFILE	       *inputFileList =		/* Linked list of input file names  */
			(IFILE *) NULL;

char	       *currentInputFile =	/* Name of current input file	    */
			(char *) NULL;

					/* CPP output file name		    */
char		cppOutputFile[ L_tmpnam ] =
			"";

unsigned long	inputOffset = 0;	/* Byte offset within cppOutputFile */

unsigned int	inputLineNumber = 1;	/* Line number within input file    */

INSTREC	       *instrumentHead =	/* Head of instrumentables list	    */
			(INSTREC *) NULL;

INSTREC	       *instrumentTail =	/* Tail of instrumentables list	    */
			(INSTREC *) NULL;

FILE	       *outputFile =		/* Instrumented code output file    */
			(FILE *) NULL;

int		parserEventID = 0;	/* Event ID counter for output	    */

PROC	       *procList =		/* Linked list of function names    */
			(PROC *) NULL;

ITER	       *iterList =		/* Linked list of loop data	    */
			(ITER *) NULL;

jmp_buf		fatalError;		/* Fatal error recovery		    */

TRANSLATE	translateTable[] =	/* Function name translation	    */
			{
			    { "main",		"_PabloParser_main_"	},
			    { (char *) NULL,	(char *) NULL		}
			};

ARGFILE	       *argFileList =		/* Argument file names list	    */
			(ARGFILE *) NULL;


/*	INTERFACE FUNCTIONS AND UTILITIES SECTION			    */


#ifdef STANDALONE

/*
 * main:
 *	Driver function for stand-alone version--ordinarily not to be
 *	used.  Usage:  PabloParser [cpp options] filename ...
 */

int
main( argc, argv )

int	argc;
char  **argv;
{
	int	tempArgc, optArgc;
	char  **tempArgv;
	char	fileName[ 256 ];

	/* Save driver program name and skip over it.			    */

	programName = *argv;

	--argc;
	++argv;

	/* Find end of input options (beginning of input file names)	    */

	for ( tempArgc = 0, tempArgv = argv;
	      tempArgc < argc;
	      tempArgc++, tempArgv++ )

		if ( **tempArgv != '-' )

			break;

	if ( argc <= tempArgc ) {

		fprintf( stderr, "usage: %s [cpp options] input-file ...\n",
				programName );
		return 1;
	}

	for ( optArgc = tempArgc;
	      tempArgc < argc;
	      tempArgc++, tempArgv++ ) {

		if ( parseFile( *tempArgv, optArgc, argv ) != 0 ) {

			fprintf( stderr,
				"%s: could not parse input file %s\n",
				programName, *tempArgv );

			continue;
		}

		printf( "Enter name of file for instrumenting %s: ",
								*tempArgv );
		gets( fileName );

		if ( generateInstrumentedSourceFile( fileName ) != 0 )

			fprintf( stderr,
				"%s: could not output instrumentation to %s\n",
				programName, fileName );
	}

	printf( "\nEnter name of file for initialization code: " );

	gets( fileName );

	if ( generateInstrumentationInitialization( fileName ) != 0 )

		fprintf( stderr,
			"%s: could not output initialization code to %s\n",
			programName, fileName );

	printf( "\nEnter name of file for event ID data: " );

	gets( fileName );

	if ( writeEventIDs( fileName ) != 0 ) {

		fprintf( stderr,
			"%s: could not write event ID data to %s\n",
			programName, fileName );

		return 1;
	}

	if ( readEventIDs( fileName ) != 0 ) {

		fprintf( stderr,
			"%s: could not read event ID data from %s\n",
			programName, fileName );

		return 1;
	}

	return 0;
}

#endif /* STANDALONE */


/*
 * parseFile:
 *	This is the top-level interface function for the first pass of
 *	of the instrumenting C parser.  One entire input file is parsed,
 *	building up data structures, including a partial parse tree.
 *	From these data structures, the list of instrumentable constructs
 *	is generated.  This function returns 0 if the parse was successful,
 *	otherwise nonzero.
 */

int
parseFile( inputFileName, inputOptc, inputOptv )

char   *inputFileName;
int	inputOptc;
char  **inputOptv;
{
	ARGFILE	       *argFilePointer;

	/* Check invocation (usage)					    */

	if (( inputFileName == (char *) NULL ) ||
	    ( inputOptc < 0 ) ||
	    (( inputOptc > 0 ) && ( inputOptv == (char **) NULL )))

		return 1;

	/* Reinitialize data structures (if necessary)			    */

	resetParser();

	/* Preprocess the input file.					    */

	if ( setjmp( fatalError ) != 0 ) {
#ifdef STANDALONE
		fprintf( stderr, "%s: fatal error during preprocessFile\n",
			programName );
#endif /* STANDALONE */
		return 1;
	}

	if ( preprocessFile( inputFileName, inputOptc, inputOptv ) != 0 )

		return 1;


	/* Take input from the cpp output file and parse it.		    */

	if (( yyin = fopen( cppOutputFile, "r" ) ) == (FILE *) NULL ) {
#ifdef STANDALONE
		fprintf( stderr, "%s: cannot open %s\n",
				programName, cppOutputFile );
#endif /* STANDALONE */
		(void) unlink( cppOutputFile );
		return 1;
	}

	(void) unlink( cppOutputFile );

	if ( setjmp( fatalError ) != 0 ) {
#ifdef STANDALONE
		fprintf( stderr, "%s: fatal error during yyparse\n",
				programName );
#endif /* STANDALONE */
		resetParser();
		return 1;
	}

	makeInputFile( inputFileName );

	contextBegin();

	if ( yyparse() != 0 ) {

		/* The parse failed, likely due to syntax error.	    */

		resetParser();
		return 1;
	}

	if ( setjmp( fatalError ) != 0 ) {
#ifdef STANDALONE
		fprintf( stderr, "%s: fatal error during parseFile\n",
				programName );
#endif /* STANDALONE */
		resetParser();
		return 1;
	}

	/*
	 * If an ARGFILE structure for this file already exists, free the
	 * ITER list it points to; that has just been invalidated since
	 * the user has re-parsed the input file.  If no ARGFILE structure
	 * exists for this file, one is created.
	 */

	for ( argFilePointer = argFileList;
	      argFilePointer != (ARGFILE *) NULL;
	      argFilePointer = argFilePointer->argNext )

		if ( strcmp( inputFileName, argFilePointer->argFileName )
		     == 0 )

			break;

	if ( argFilePointer == (ARGFILE *) NULL ) {

		argFilePointer = (ARGFILE *) memAlloc( sizeof(ARGFILE) );

		argFilePointer->argFileName = (char *) memAlloc(
						1 + strlen( inputFileName ) );

		(void) strcpy( argFilePointer->argFileName, inputFileName );

		argFilePointer->argNext = argFileList;

		argFileList = argFilePointer;

	} else

		freeIterList( argFilePointer->argIterList );

	argFilePointer->argIterList = iterList;

	iterList = (ITER *) NULL;

	/* Generate the list of instrumentable constructs.		    */

	if ( currentTrace != (TRACETREE *) NULL )

		while ( currentTrace->traceLeft != (TRACETREE *) NULL )

			currentTrace = currentTrace->traceLeft;

	/* Traverse the trace tree top-down, create list for user interface  */

	if ( setjmp( fatalError ) != 0 ) {
#ifdef STANDALONE
		fprintf( stderr,
			"%s: fatal error during generateInstrumentables\n",
				programName );
#endif /* STANDALONE */
		resetParser();
		return 1;
	}

	generateInstrumentables( currentTrace, FALSE );

	return 0;
}


/*
 * preprocessFile:
 *	Run the C preprocessor (cpp) on the input file, using the cpp
 *	options passed in.
 */

int
preprocessFile( inputFileName, inputOptc, inputOptv )

char   *inputFileName;
int	inputOptc;
char  **inputOptv;
{
	unsigned int	cppArgc;
	char	      **cppArgv,
		      **cppArgPointer;
	char	      **inputOptPointer;
	int		i, pid, status;

	/* Create a temporary file name to hold cpp output.		    */

	(void) tmpnam( cppOutputFile );

	/* Determine how many elements to cppArgv array to be passed
	 * to the execv system call.  This could be done with getopt.
	 *
	 * Start with 6 elements for cppArgv:
	 *	1. "cpp" (the cpp program's argv[0])
	 *	2. "-C" option to include C comments
	 *	3. "-DLANGUAGE_C" option for ANSI compatibility
	 *	4. the input file name
	 *	5. the output file name
	 *	6. the final NULL pointer
	 * and add one for every option passed in which will be passed onto
	 * cpp.  These include the following:
	 *	-D arg	To #define arg
	 *	-U arg	To #undef arg
	 *	-I arg	To add arg to the #include path list
	 * The -C option is silently ignored, since it is always passed to
	 * cpp.  The -P option is rejected, as including it would prevent
	 * cpp from including line control information in its output.  Also,
	 * the -M option is rejected, since the output is make dependencies
	 * instead of normal cpp output.  Any other cpp options, without
	 * arguments, are passed on to cpp.
	 */

	cppArgc = 6;

	for ( i = 0, inputOptPointer = inputOptv;
	      i < inputOptc;
	      i++, inputOptPointer++ ) {

	    if ( *inputOptPointer != (char *) NULL ) {

		if ( (*inputOptPointer)[ 0 ] == '-' ) {

		    switch ( (*inputOptPointer)[ 1 ] ) {
		    case 'C':
			/* Silently ignore this option.			    */
			break;
		    case 'D':
		    case 'I':
		    case 'U':
			cppArgc++;
			if (( (*inputOptPointer)[ 2 ] == '\0' ) &&
			    ( ++i < inputOptc )) {
				inputOptPointer++;
				cppArgc++;
			}
			break;
		    case 'M':
		    case 'P':
#ifdef STANDALONE
			fprintf( stderr, "%s: ignored option %s\n",
					programName, *inputOptPointer );
#endif /* STANDALONE */
			break;
		    default:
			cppArgc++;
			break;
		    }

		}
#ifdef STANDALONE
		else {
		    fprintf( stderr, "%s: unrecognized option %s\n",
					programName, *inputOptPointer );
		}
#endif /* STANDALONE */
	    }
	}


	/* Allocate the cppArgv array and initialize it.		    */

	cppArgv = (char **) memAlloc( cppArgc * sizeof(char *) );

	cppArgPointer = cppArgv;

	*cppArgPointer++ = "cpp";

	*cppArgPointer++ = "-C";

	*cppArgPointer++ = "-DLANGUAGE_C";

	for ( i = 0, inputOptPointer = inputOptv;
	      i < inputOptc;
	      i++, inputOptPointer++ ) {

	    if (( *inputOptPointer != (char *) NULL ) &&
		( (*inputOptPointer)[ 0 ] == '-' )) {

		switch ( (*inputOptPointer)[ 1 ] ) {
		case 'D':
		case 'I':
		case 'U':
			*cppArgPointer++ = *inputOptPointer;
			if (( (*inputOptPointer)[ 2 ] == '\0' ) &&
			    ( ++i < inputOptc )) {
				*cppArgPointer++ = *++inputOptPointer;
			}
			break;
		case 'C':
		case 'M':
		case 'P':
			break;
		default:
			*cppArgPointer++ = *inputOptPointer;
			break;
		}

	    }
	}

	*cppArgPointer++ = inputFileName;

	*cppArgPointer++ = cppOutputFile;

	*cppArgPointer = NULL;

#ifdef DEBUG
	printf( "forking:" );

	for ( cppArgPointer = cppArgv;
	     *cppArgPointer != (char *) NULL;
	      cppArgPointer++ )

		printf( " %s", *cppArgPointer );

	putchar( '\n' );
#endif /* DEBUG */

	/* Fork a child process, have it exec cpp with these arguments.	    */

	if ( ( pid = fork() ) == 0 ) {
		(void) execv( "/lib/cpp", cppArgv );
		(void) execv( "/usr/lib/cpp", cppArgv );
		(void) execvp( "cpp", cppArgv );
#ifdef STANDALONE
		fprintf( stderr, "%s: cannot exec cpp\n", programName );
#endif /* STANDALONE */
		exit( 1 );
	}

	memFree( (void *) cppArgv );

	if ( pid < 0 ) {
#ifdef STANDALONE
		fprintf( stderr, "%s: cannot fork\n", programName );
#endif /* STANDALONE */
		return 1;
	}

	/* Wait for the cpp process to finish.				    */

	while ( waitpid( pid, &status, 0 ) != pid )
		/* do nothing */ ;

	if (( WIFEXITED( status ) == 0 ) ||
	    ( WEXITSTATUS( status ) != 0 )) {
#ifdef STANDALONE
		fprintf( stderr, "%s: cpp failed on %s\n",
				programName, inputFileName );
#endif /* STANDALONE */
		(void) unlink( cppOutputFile );
		return 1;
	}

	return 0;
}



/*
 * generateInstrumentedSourceFile:
 *	This is a top-level interface function for the user interface.
 *	It works like a second pass of the parser, using the data passed
 *	in by the user interface, but generating instrumented source code.
 *	Since the output will have a bunch of   # line-number "file-name"
 *	lines in it, output is instead piped through:  sed s/^#[ \t]/#line /
 *	before output to the user's file.
 */

int
generateInstrumentedSourceFile( outputFileName )

char   *outputFileName;
{
	INSTREC	       *instrument;
	unsigned long	eofOffset;
	int		pipe_fd[ 2 ];
	int		pid, status;
	static char	sedName[] = "sed";
	static char	sedCommand[] = "s/^#[ \t]/#line /";

	if (( yyin == (FILE *) NULL ) || ( yyin == stdin ))

		return 1;

	/* Initialize input and output streams.				    */

	rewind( yyin );				/* cppOutputFile	    */

	inputOffset = 0;

	if ( pipe( pipe_fd ) < 0 ) {
#ifdef STANDALONE
		fprintf( stderr, "%s: cannot open pipe\n", programName );
#endif STANDALONE
		return 1;
	}

	if ( ( pid = fork() ) < 0 ) {
#ifdef STANDALONE
		fprintf( stderr, "%s: cannot fork\n", programName );
#endif /* STANDALONE */
		return 1;
	}

	if ( pid == 0 ) {			/* Child process	    */

		(void) close( 0 );		/* Close stdin		    */

		if ( fcntl( pipe_fd[ 0 ], F_DUPFD, 0 ) != 0 ) {
#ifdef STANDALONE
			fprintf( stderr, "%s: cannot dup pipe to stdin\n",
				programName );
#endif /* STANDALONE */
			exit( 1 );
		}

		(void) close( pipe_fd[ 0 ] );
		(void) close( pipe_fd[ 1 ] );

		if ( freopen( outputFileName, "w", stdout ) != stdout ) {
#ifdef STANDALONE
			fprintf( stderr, "%s: cannot open %s as stdout\n",
				programName, outputFileName );
#endif /* STANDALONE */
			exit( 1 );
		}

		(void) execl( "/bin/sed", sedName, sedCommand, (char *)0 );
		(void) execl( "/usr/bin/sed", sedName, sedCommand, (char *)0 );
		(void) execlp( sedName, sedName, sedCommand, (char *)0 );

#ifdef STANDALONE
		fprintf( stderr, "%s: cannot exec %s\n",
			programName, sedName );
#endif /* STANDALONE */

		exit( 1 );
	}


	/* This is the parent process again.				    */

	(void) close( pipe_fd[ 0 ] );

	outputFile = fdopen( pipe_fd[ 1 ], "w" );

	if ( outputFile == (FILE *) NULL ) {
#ifdef STANDALONE
		fprintf( stderr, "%s: could not write to pipe\n",
			programName );
#endif /* STANDALONE */
		return 1;
	}


	/* Note which constructs are to be instrumented.		    */

	for ( instrument = instrumentHead;
	      instrument != (INSTREC *) NULL;
	      instrument = instrument->next )

		instrument->treeElement->traceType =
						instrument->instrumentType;

	/*
	 * Now go through the tree of traceable constructs, top-down,
	 * and output instrumenting code for everything which is to be
	 * instrumented.
	 */

	instrumentCode( currentTrace );


	/* Copy anything left in the cppOutputFile			    */

	(void) fseek( yyin, (long) 0, 2 );

	eofOffset = ftell( yyin );

	(void) fseek( yyin, (long) inputOffset, 0 );

	copyUpTo( eofOffset );

	(void) fclose( outputFile );

	outputFile = (FILE *) NULL;


	/* Wait for the sed process to finish.				    */

	while( waitpid( pid, &status, 0 ) != pid )
		/* do nothing */ ;

	if (( WIFEXITED( status ) == 0 ) ||
	    ( WEXITSTATUS( status ) != 0 )) {
#ifdef STANDALONE
		fprintf( stderr, "%s: %s %s command failed\n",
			programName, sedName, sedCommand );
#endif /* STANDALONE */
		return 1;
	}

	return 0;
}


/*
 * generateInstrumentationInitialization:
 *	This is a top-level interface function for the user interface.
 *	It outputs the initialization function for the instrumentation
 *	to the file name passed as an argument.
 */

int
generateInstrumentationInitialization( initFileName )

char   *initFileName;
{
	FILE	       *initFile;
	PROC	       *procPointer;
	ARGFILE	       *argFilePointer;
	ITER	       *iterPointer;
	int		numProcs = 0;
	int		numIters = 0;
	TRANSLATE      *translatePointer;

	initFile = fopen( initFileName, "w" );

	if ( initFile == (FILE *) NULL ) {
#ifdef STANDALONE
		fprintf( stderr, "%s: could not open %s for writing\n",
			programName, initFileName );
#endif /* STANDALONE */
		return 1;
	}

	fputs( "\n\n/*\n", initFile );
	fputs( " * The main function below was created by the\n", initFile );
	fputs( " * Pablo Performance Analysis Environment's\n", initFile );
	fputs( " * instrumenting parser.  When linked with\n", initFile );
	fputs( " * other code that was instrumented by this\n", initFile );
	fputs( " * parser, and with the Pablo trace capture\n", initFile );
	fputs( " * library and extensions, it will perform\n", initFile );
	fputs( " * trace capture library initialization before\n", initFile );
	fputs( " * invoking the original main function supplied\n", initFile );
	fputs( " * in the input code.\n *\n", initFile );
	fputs( " * Set PabloTraceFlag to zero to deactivate the\n", initFile );
	fputs( " * instrumenting code.  A nonzero value will\n", initFile );
	fputs( " * reactivate the instrumenting code.\n", initFile );
	fputs( " */\n\n", initFile );

	fputs( "int\tPabloTraceFlag = 1;\n\n", initFile );

	fputs( "main(argc, argv, envp)\n", initFile );
	fputs( "int\targc;\nchar\t**argv, **envp;\n", initFile );
	fputs( "{\n\tint\treturnValue;\n\n", initFile );


	/* Define procedure tracing initialization data structures	    */

	for ( procPointer = procList;
	      procPointer != (PROC *) NULL;
	      procPointer = procPointer->procNext )

		numProcs++;

	if ( numProcs > 0 )

		fprintf( initFile, "\tstatic int\tprocEntryIDs[%d] = {\n",
				numProcs );

	for ( procPointer = procList;
	      procPointer != (PROC *) NULL;
	      procPointer = procPointer->procNext )

		fprintf( initFile, "\t\t%d,\t/* %s\t*/\n",
			procPointer->procEventEntry,
			procPointer->procName );

	if ( numProcs > 0 ) {

		fputs( "\t\t};\n", initFile );
		fprintf( initFile, "\tstatic int\tprocExitIDs[%d] = {\n",
				numProcs );
	}

	for ( procPointer = procList;
	      procPointer != (PROC *) NULL;
	      procPointer = procPointer->procNext )

		fprintf( initFile, "\t\t%d,\t/* %s\t*/\n",
			procPointer->procEventExit,
			procPointer->procName );

	if ( numProcs > 0 )

		fputs( "\t\t};\n\n", initFile );


	/* Define iteration tracing initialization data structures	    */

	for ( argFilePointer = argFileList;
	      argFilePointer != (ARGFILE *) NULL;
	      argFilePointer = argFilePointer->argNext )

		for ( iterPointer = argFilePointer->argIterList;
		      iterPointer != (ITER *) NULL;
		      iterPointer = iterPointer->iterNext )

			numIters++;

	if ( numIters > 0 )

		fprintf( initFile, "\tstatic int\titerEntryIDs[%d] = {\n",
				numIters );

	for ( argFilePointer = argFileList;
	      argFilePointer != (ARGFILE *) NULL;
	      argFilePointer = argFilePointer->argNext ) {

		if ( argFilePointer->argIterList != (ITER *) NULL )

			fprintf( initFile, "\t\t\t/* %s\t*/\n",
				argFilePointer->argFileName );

		for ( iterPointer = argFilePointer->argIterList;
		      iterPointer != (ITER *) NULL;
		      iterPointer = iterPointer->iterNext )

			fprintf( initFile, "\t\t%d,\n",
				iterPointer->iterEventEntry );
	}


	if ( numIters > 0 ) {

		fputs( "\t\t};\n", initFile );
		fprintf( initFile, "\tstatic int\titerExitIDs[%d] = {\n",
				numIters );
	}


	for ( argFilePointer = argFileList;
	      argFilePointer != (ARGFILE *) NULL;
	      argFilePointer = argFilePointer->argNext ) {

		if ( argFilePointer->argIterList != (ITER *) NULL )

			fprintf( initFile, "\t\t\t/* %s\t*/\n",
				argFilePointer->argFileName );

		for ( iterPointer = argFilePointer->argIterList;
		      iterPointer != (ITER *) NULL;
		      iterPointer = iterPointer->iterNext )

			fprintf( initFile, "\t\t%d,\n",
				iterPointer->iterEventExit );
	}

	if ( numIters > 0 )

		fputs( "\t\t};\n\n", initFile );

	fputs( "\t/* call setTraceFileName( fileName ) here\t\t\t*/\n",
			initFile );

	/* Perform procedure tracing pre-initialization			    */

	if ( numProcs > 0 )
		fputs( "\tpreInitProcTrace();\n", initFile );

	/* Perform iteration tracing pre-initialization			    */

	if ( numIters > 0 )
		fputs( "\tpreInitLoopTrace();\n", initFile );


	/* Perform procedure tracing full initialization		    */

	if ( numProcs > 0 )

	    fprintf( initFile,
		"\tinitProcTrace( %d, procEntryIDs, procExitIDs );\n",
			numProcs );

	/* Perform iteration tracing full initialization		    */

	if ( numIters > 0 )

	    fprintf( initFile,
		"\tinitLoopTrace( %d, iterEntryIDs, iterExitIDs );\n",
			numIters );

	/* Process command-line argument for dynamic tracing control	    */

	fputs( "\n\tif (argc >= 2) {\n", initFile );
	fputs( "\t\tif (strcmp(argv[1], \"-trace\") == 0) {\n", initFile );
	fputs( "\t\t\tPabloTraceFlag = 1;\n", initFile );
	fputs( "\t\t\targc--;\n", initFile );
	fputs( "\t\t\targv[1] = argv[0];\n", initFile );
	fputs( "\t\t\targv++;\n", initFile );
	fputs( "\t\t} else if (strcmp(argv[1], \"-notrace\") == 0) {\n",
								initFile );
	fputs( "\t\t\tPabloTraceFlag = 0;\n", initFile );
	fputs( "\t\t\targc--;\n", initFile );
	fputs( "\t\t\targv[1] = argv[0];\n", initFile );
	fputs( "\t\t\targv++;\n", initFile );
	fputs( "\t\t}\n", initFile );
	fputs( "\t}\n", initFile );

	/* Invoke user-supplied main function				    */

	for ( translatePointer = translateTable;
	      translatePointer->oldName != (char *) NULL;
	      translatePointer++ )

		if ( strcmp( translatePointer->oldName, "main" ) == 0 )

			break;

	fprintf( initFile, "\n\treturnValue = %s( argc, argv, envp );\n\n",
		translatePointer->newName );

	if (( numProcs > 0 ) || ( numIters > 0 ))

		fputs( "\tendTracing();\n", initFile );

	fputs( "\treturn returnValue;\n}\n", initFile );

	(void) fclose( initFile );

	return 0;
}


/*
 * writeEventIDs:
 *	This is a top-level interface function between the parser and
 *	the user interface.  Invoking it causes the parser to output
 *	all event ID related data that has been accumulated in the
 *	procList, argFileList, and attached iterList data structures
 *	to the named argument file.  The format used is recognized by
 *	readEventIDs.
 */

int
writeEventIDs( eventFileName )

char   *eventFileName;
{
	FILE	       *eventFile;
	PROC	       *procPointer;
	ARGFILE	       *argFilePointer;
	ITER	       *iterPointer;

	eventFile = fopen( eventFileName, "w" );

	if ( eventFile == (FILE *) NULL ) {
#ifdef STANDALONE
		fprintf( stderr, "%s: could not open %s for writing\n",
			programName, eventFileName );
#endif /* STANDALONE */
		return 1;
	}

	fputs( "# This file was created by the Pablo\n", eventFile );
	fputs( "# Analysis Environment's instrumenting parser.\n", eventFile );
	fputs( "# It should not be modified for any reason!\n", eventFile );

	fputs( "PROCEDURES\n", eventFile );

	for( procPointer = procList;
	     procPointer != (PROC *) NULL;
	     procPointer = procPointer->procNext )

		fprintf( eventFile, "entry=%d exit=%d void=%d\tname=%s\n",
			procPointer->procEventEntry,
			procPointer->procEventExit,
			procPointer->procIsVoid,
			procPointer->procName );

	fputs( "ITERATIONS\n", eventFile );

	for ( argFilePointer = argFileList;
	      argFilePointer != (ARGFILE *) NULL;
	      argFilePointer = argFilePointer->argNext ) {

		fprintf( eventFile, "file=%s\n", argFilePointer->argFileName );

		for ( iterPointer = argFilePointer->argIterList;
		      iterPointer != (ITER *) NULL;
		      iterPointer = iterPointer->iterNext )

			fprintf( eventFile, "entry=%d exit=%d\n",
				iterPointer->iterEventEntry,
				iterPointer->iterEventExit );
	}

	fputs( "END\n", eventFile );

	(void) fclose( eventFile );

	return 0;
}


/*
 * readEventIDs:
 *	This is a top-level interface function between the parser and
 *	the user interface.  Invoking it causes the parser to input the
 *	event ID related data contained in the named argument file.  This
 *	file should have been created by a previous call to writeEventIDs,
 *	possibly in an entirely separate session.  As this causes any
 *	event ID data accumulated in the procList, argFileList, and attached
 *	iterList data structures to be destroyed, this function should only
 *	be called when the user is beginning to instrument a new set of
 *	source code files.  Executing this function has the same effect
 *	as instrumenting the same files, in the same order, as appear in
 *	the named file, and then resetting the parser for the next file.
 */

int
readEventIDs( eventFileName )

char   *eventFileName;
{
	char		lineBuffer[ 256 ];
	char		stringBuffer[ 256 ];
	FILE	       *eventFile;
	PROC	       *procPointer;
	ARGFILE	       *argFilePointer;
	ITER	       *iterPointer;
	int		readState;
	int		entryID, exitID;
	int		voidFlag;

	eventFile = fopen( eventFileName, "r" );

	if ( eventFile == (FILE *) NULL ) {
#ifdef STANDALONE
	    fprintf( stderr, "%s: could not open %s for reading\n",
			programName, eventFileName );
#endif /* STANDALONE */
	    return 1;
	}

	cleanupParser();

	if ( setjmp( fatalError ) != 0 ) {
#ifdef STANDALONE
		fprintf( stderr, "%s: fatal error during readEventIDs\n",
			programName );
#endif /* STANDALONE */
		(void) fclose( eventFile );
		cleanupParser();
		return 1;
	}

	readState = 0;

	while ( fgets( lineBuffer, sizeof lineBuffer, eventFile )
							!= (char *) NULL ) {

	    /* Make sure a whole line was read				    */

	    if ( lineBuffer[ strlen( lineBuffer ) - 1 ] != '\n' )

		break;

	    /* Permit comment lines anywhere within file		    */

	    if ( lineBuffer[ 0 ] == '#' )

		continue;


	    switch( readState ) {

	    case 0:			/* Beginning of file		    */
		if ( strcmp( lineBuffer, "PROCEDURES\n" ) == 0 ) {
		    readState++;
		    continue;
		}

		break;

	    case 1:			/* In procedures section	    */
		if ( strcmp( lineBuffer, "ITERATIONS\n" ) == 0 ) {
		    readState++;
		    continue;
		}

		if ( sscanf( lineBuffer, "entry=%d exit=%d void=%d\tname=%s",
			&entryID, &exitID, &voidFlag, stringBuffer ) != 4 )

		    break;

		procPointer = (PROC *) memAlloc( sizeof(PROC) );

		procPointer->procName = (char *)
					memAlloc( 1 + strlen( stringBuffer ) );
		(void) strcpy( procPointer->procName, stringBuffer );
		procPointer->procEventEntry = entryID;
		procPointer->procEventExit = exitID;
		procPointer->procDeclared = FALSE;
		procPointer->procIsVoid = voidFlag;
		procPointer->procNext = procList;
		procList = procPointer;

		if ( entryID >= parserEventID )
		    parserEventID = entryID + 1;

		if ( exitID >= parserEventID )
		    parserEventID = exitID + 1;

		continue;

	    case 2:			/* In iterations section	    */
		if ( strcmp( lineBuffer, "END\n" ) == 0 ) {
		    readState++;
		    break;
		}

		if ( sscanf( lineBuffer, "file=%s", stringBuffer ) == 1 ) {

		    argFilePointer = (ARGFILE *) memAlloc( sizeof(ARGFILE) );
		    argFilePointer->argFileName = (char *)
					memAlloc( 1 + strlen( stringBuffer ) );
		    (void) strcpy( argFilePointer->argFileName, stringBuffer );
		    argFilePointer->argIterList = (ITER *) NULL;
		    argFilePointer->argNext = argFileList;
		    argFileList = argFilePointer;

		    continue;
		}

		if ( sscanf( lineBuffer, "entry=%d exit=%d",
					&entryID, &exitID ) != 2 )
		    break;

		if ( argFileList == (ARGFILE *) NULL )

		    break;

		iterPointer = (ITER *) memAlloc( sizeof(ITER) );

		iterPointer->iterEventEntry = entryID;
		iterPointer->iterEventExit = exitID;
		iterPointer->iterNext = argFileList->argIterList;
		argFileList->argIterList = iterPointer;

		if ( entryID >= parserEventID )
		    parserEventID = entryID + 1;

		if ( exitID >= parserEventID )
		    parserEventID = exitID + 1;

		continue;
	    }

	    /*
	     * Normal execution is to proceed to the top of the while
	     * loop with a continue statement.  Terminate the loop upon
	     * reaching this point (outside the switch).
	     */

	    break;
	}

	(void) fclose( eventFile );

	if ( readState != 3 ) {
#ifdef STANDALONE
	    fprintf( stderr, "%s: file %s not in proper format\n",
			programName, eventFileName );
#endif /* STANDALONE */
	    cleanupParser();
	}

	return (readState == 3) ? 0 : 1;
}


/*
 * cleanupParser:
 *	Completely reinitialize ALL parser data structures.  The result
 *	is that the parser is put in exactly the same state as it is in
 *	when it is first invoked.
 */

void
cleanupParser()
{
	PROC	       *procPointer;
	ARGFILE	       *argFilePointer;

	resetParser();

	while ( procList != (PROC *) NULL ) {

		procPointer = procList->procNext;

		memFree( (void *) procList->procName );

		memFree( (void *) procList );

		procList = procPointer;

	}

	while ( argFileList != (ARGFILE *) NULL ) {

		argFilePointer = argFileList->argNext;

		memFree( (void *) argFileList->argFileName );

		freeIterList( argFileList->argIterList );

		memFree( (void *) argFileList );

		argFileList = argFilePointer;

	}

	parserEventID = 0;

	return;
}


/*
 * generateInstrumentables:
 *	This is a recursive function which performs a pre-order traversal
 *	of a portion of the tree of instrumentable constructs, generating
 *	elements for the list passed to the user interface.
 */

void
generateInstrumentables( tracePointer, eliminateLoops )

TRACETREE      *tracePointer;
int		eliminateLoops;
{
	INSTREC	       *instrument;
	int		isOuterLoop = eliminateLoops;

	if ( tracePointer == (TRACETREE *) NULL )
		return;

	/*
	 * If this is an actual instrumentable construct (i.e. not one of
	 * the trace tree elements allocated for a function definition)
	 * and it's either a FunctionCall or an OuterLoop, then create
	 * a new list element, initialize it, and link it onto the end of
	 * the list passed to the user interface.
	 */

	if (( tracePointer->traceConstruct == FunctionCall ) ||
	    (( tracePointer->traceConstruct == OuterLoop ) &&
	     ( eliminateLoops == FALSE ))) {

		instrument = (INSTREC *) memAlloc( sizeof(INSTREC) );

		instrument->invokingFunc =
				tracePointer->traceInvokingSym->symName;

		if ( tracePointer->traceConstruct == FunctionCall ) {

			instrument->invokedFunc =
				tracePointer->traceInvokedSym->symName;

		} else {

			instrument->invokedFunc = (char *) NULL;
			isOuterLoop = TRUE;
		}

		instrument->beginLine = tracePointer->traceLineStart;
		instrument->endLine = tracePointer->traceLineEnd;
		instrument->constructType = tracePointer->traceConstruct;
#ifdef DEBUG
		instrument->instrumentType = InstrumentTrace;
#else /* DEBUG */
		instrument->instrumentType = InstrumentOff;
#endif /* DEBUG */
		instrument->previous = instrumentTail;
		instrument->next = (INSTREC *) NULL;
		instrument->treeElement = tracePointer;

		if ( instrumentTail == (INSTREC *) NULL )

			instrumentHead = instrument;
		else
			instrumentTail->next = instrument;

		instrumentTail = instrument;
	}

	/* Descend down the traceable constructs subtree.		    */

	generateInstrumentables( tracePointer->traceChild, isOuterLoop );

	/* Recurse on adjacent (sibling) subtrees.			    */

	generateInstrumentables( tracePointer->traceRight, eliminateLoops );

	return;
}


/*
 * instrumentCode:
 *	Recursively descend through the tree of instrumentable constructs,
 *	generating instrumenting code for those constructs which are to be
 *	instrumented, and in the right places.
 */

void
instrumentCode( tracePointer )

TRACETREE      *tracePointer;
{
	PROC   *procPointer;

	if ( tracePointer == (TRACETREE *) NULL )
		return;


	/* Copy cppOutputFile up to the byte where this construct begins.   */

	copyUpTo( tracePointer->traceByteStart );

	/*
	 * Insert instrumenting code which goes at
	 * the beginning of this construct.
	 */

	switch( tracePointer->traceConstruct ) {

	case FunctionCall:

	    /* Beginning of function invocation				    */

	    if ( tracePointer->traceType != InstrumentOff ) {

		fprintf( outputFile,
			 "\n# %u \"PabloParseCode.c\"\n(PabloTraceFlag && ",
				tracePointer->traceLineStart );

		if ( tracePointer->traceType == InstrumentTrace )

		    fprintf( outputFile, "PabloTraceProc(%d, %lu, %u),",
				tracePointer->traceProc->procEventEntry,
				tracePointer->traceByteStart,
				tracePointer->traceLineStart );
		else

		    fprintf( outputFile, "PabloCountProc(%d, %lu, %u),",
				tracePointer->traceProc->procEventEntry,
				tracePointer->traceByteStart,
				tracePointer->traceLineStart );

		if ( tracePointer->traceProc->procIsVoid == FALSE )

		    fprintf( outputFile, " _%s_ReturnValue_ =",
				tracePointer->traceProc->procName );

		fprintf( outputFile, "\n# %u \"%s\"\n",
				tracePointer->traceLineStart,
				tracePointer->traceFileStart );
	    }

	    break;

	case OuterLoop:

	    /* Beginning of loop					    */

	    if ( tracePointer->traceType != InstrumentOff ) {

		fprintf( outputFile, "\n# %u \"PabloParseCode.c\"\n",
				tracePointer->traceLineStart );
		fprintf( outputFile, "{\tif (PabloTraceFlag)\t" );

		if ( tracePointer->traceType == InstrumentTrace )

		    fprintf( outputFile, "PabloTraceLoop(%d, %lu, %u);",
				tracePointer->traceIter->iterEventEntry,
				tracePointer->traceByteStart,
				tracePointer->traceLineStart );
		else

		    fprintf( outputFile, "PabloCountLoop(%d, %lu, %u);",
				tracePointer->traceIter->iterEventEntry,
				tracePointer->traceByteStart,
				tracePointer->traceLineStart );

		fprintf( outputFile, "\n# %u \"%s\"\n",
				tracePointer->traceLineStart,
				tracePointer->traceFileStart );
	    }

	    break;

	case FunctionBody:

	    /* Beginning of function definition				    */

	    fprintf( outputFile, "\n# %u \"PabloParseCode.c\"\n",
			tracePointer->traceLineStart );

	    fputs( "\textern int\tPabloTraceFlag;\n", outputFile );

	    for ( procPointer = procList;
		  procPointer != (PROC *) NULL;
		  procPointer = procPointer->procNext )

		procPointer->procDeclared = FALSE;

	    declareReturnValues( tracePointer->traceChild );

	    fprintf( outputFile, "# %u \"%s\"\n",
			tracePointer->traceLineStart,
			tracePointer->traceFileStart );

	    break;

	case SymbolName:

	    /* Substitute symbol name (e.g. for "main")			    */

	    fputs( tracePointer->traceTranslate, outputFile );

	    /* Skip past original symbol token				    */

	    inputOffset = tracePointer->traceByteEnd;
	    (void) fseek( yyin, (long) inputOffset, 0 );

	    break;
	}


	/* Descend to children (nested constructs)			    */

	instrumentCode( tracePointer->traceChild );


	/* Copy cppOutputFile up through where this construct ends.	    */

	copyUpTo( tracePointer->traceByteEnd );

	/*
	 * Insert instrumenting code which goes
	 * at the end of this construct.
	 */

	switch( tracePointer->traceConstruct ) {

	case FunctionCall:

	    /* End of function invocation				    */

	    if ( tracePointer->traceType != InstrumentOff ) {

		fprintf( outputFile,
			 "\n# %u \"PabloParseCode.c\"\n, PabloTraceFlag && ",
				tracePointer->traceLineEnd );

		if ( tracePointer->traceType == InstrumentTrace )

		    fprintf( outputFile, "PabloTraceProc(%d, %lu, %u)",
				tracePointer->traceProc->procEventExit,
				tracePointer->traceByteEnd,
				tracePointer->traceLineEnd );
		else

		    fprintf( outputFile, "PabloCountProc(%d, %lu, %u)",
				tracePointer->traceProc->procEventExit,
				tracePointer->traceByteEnd,
				tracePointer->traceLineEnd );

		if ( tracePointer->traceProc->procIsVoid == FALSE )

		    fprintf( outputFile, ", _%s_ReturnValue_",
				tracePointer->traceProc->procName );

		fprintf( outputFile, ")\n# %u \"%s\"\n",
				tracePointer->traceLineEnd,
				tracePointer->traceFileEnd );
	    }

	    break;

	case OuterLoop:

	    /* End of non-nested loop					    */

	    if ( tracePointer->traceType != InstrumentOff ) {

		fprintf( outputFile, "\n# %u \"PabloParseCode.c\"\n",
				tracePointer->traceLineEnd );
		fprintf( outputFile, "\tif (PabloTraceFlag)\t" );

		if ( tracePointer->traceType == InstrumentTrace )

		    fprintf( outputFile, "PabloTraceLoop(%d, %lu, %u);",
				tracePointer->traceIter->iterEventExit,
				tracePointer->traceByteEnd,
				tracePointer->traceLineEnd );
		else

		    fprintf( outputFile, "PabloCountLoop(%d, %lu, %u);",
				tracePointer->traceIter->iterEventExit,
				tracePointer->traceByteEnd,
				tracePointer->traceLineEnd );

		fprintf( outputFile, "}\n# %u \"%s\"\n",
				tracePointer->traceLineEnd,
				tracePointer->traceFileEnd );
	    }

	    break;
	}


	/* Do sibling subtrees						    */

	instrumentCode( tracePointer->traceRight );


	return;
}


/*
 * copyUpTo:
 *	Copy cppOutputFile from current position up to the specified one.
 */

void
copyUpTo( finalOffset )

unsigned long	finalOffset;
{
	char		buffer[ 1024 ];
	unsigned int	numBytes;

#ifdef DEBUG
	if ( inputOffset > finalOffset )
		printf( "copyUpTo: offset=%lu, requested=%lu!\n",
			inputOffset, finalOffset );
#endif /* DEBUG */

	while ( inputOffset < finalOffset ) {

		numBytes = (unsigned int) ( finalOffset - inputOffset );

		if ( numBytes > sizeof buffer )
			numBytes = sizeof buffer;

		numBytes = fread( buffer, sizeof(char), numBytes, yyin );

		inputOffset += numBytes;

		(void) fwrite( buffer, sizeof(char), numBytes, outputFile );
	}

	return;
}


/*
 * copyBlock:
 *	Copy of block of bytes from cppOutputFile to the instrumented
 *	code output file.
 */

void
copyBlock( startingOffset, endingOffset )

unsigned long	startingOffset,
		endingOffset;
{
	unsigned long	saveOffset;

	saveOffset = inputOffset;		/* save the current offset  */

	inputOffset = startingOffset;		/* move to new location	    */
	(void) fseek( yyin, (long) inputOffset, 0 );

	copyUpTo( endingOffset );		/* copy bytes		    */

	inputOffset = saveOffset;		/* return to old offset	    */
	(void) fseek( yyin, (long) inputOffset, 0 );

	return;
}


/*
 * declareReturnValues:
 *	This is a recursive function which outputs to the instrumented
 *	code file declarations for the new variables which are to hold
 *	return values from invoked functions.
 */

void
declareReturnValues( tracePointer )

TRACETREE      *tracePointer;
{
	NODE   *nodePointer;

	if ( tracePointer == (TRACETREE *) NULL )
		return;

	if (( tracePointer->traceConstruct == FunctionCall ) &&
	    ( tracePointer->traceType != InstrumentOff ) &&
	    ( tracePointer->traceProc->procIsVoid == FALSE ) &&
	    ( tracePointer->traceProc->procDeclared == FALSE )) {

		putc( '\t', outputFile );


		nodePointer = tracePointer->traceInvokedSym->symDeclSpec;

		if ( nodePointer == (NODE *) NULL )

			fputs( "int ", outputFile );
		else
			duplicateDeclSpec( nodePointer );


		nodePointer = tracePointer->traceInvokedSym->symDeclarator;

		if ( nodePointer == (NODE *) NULL )

			fprintf( outputFile, "_%s_ReturnValue_",
				tracePointer->traceProc->procName );
		else
			duplicateDeclarator( nodePointer );


		putc( ';', outputFile );

		putc( '\n', outputFile );

		tracePointer->traceProc->procDeclared = TRUE;

	}

	declareReturnValues( tracePointer->traceChild );

	declareReturnValues( tracePointer->traceRight );

	return;
}


/*
 * duplicateDeclSpec:
 *	This function duplicates the declaration specifiers for an
 *	invoked function.  Storage class specifiers are skipped, and
 *	struct/union/enum specifiers are shortened where possible.
 */

void
duplicateDeclSpec( nodePointer )

NODE   *nodePointer;
{
	char   *keyWord;

	for ( ;
	      nodePointer != (NODE *) NULL;
	      nodePointer = nodePointer->nodeNext ) {

	    switch( nodePointer->nodeType ) {

	    case StorClassSpec:

		/* Skip over storage class specifiers			    */

		break;

	    case TypeSpecifier:

		switch( nodePointer->nodeToken ) {

		case CHAR:
		    keyWord = "char";
		    break;

		case SHORT:
		    keyWord = "short";
		    break;

		case INT:
		    keyWord = "int";
		    break;

		case LONG:
		    keyWord = "long";
		    break;

		case SIGNED:
		    keyWord = "signed";
		    break;

		case UNSIGNED:
		    keyWord = "unsigned";
		    break;

		case FLOAT:
		    keyWord = "float";
		    break;

		case DOUBLE:
		    keyWord = "double";
		    break;

		case VOID:
		    keyWord = "void";
		    break;

		case TYPEDEF_NAME:
		    keyWord =
			nodePointer->nodeIdentifier->tokenSymbol->symName;
		    break;
#ifdef DEBUG
		default:
		    printf( "unrecognized TypeSpecifier, token=%d!\n",
			   nodePointer->nodeToken );
		    keyWord="?";
		    break;
#endif /* DEBUG */
		}

		fputs( keyWord, outputFile );
		putc( ' ', outputFile );
		break;

	    case StructOrUnion:
	    case EnumSpecifier:

		/*
		 * If the struct/union/enum specifier is tagged with a name,
		 * then the struct declaration list should not be duplicated.
		 * If untagged, it becomes necessary to duplicate the list.
		 */

		switch( nodePointer->nodeToken ) {

		case STRUCT:
		    keyWord = "struct";
		    break;

		case UNION:
		    keyWord = "union";
		    break;

		case ENUM:
		    keyWord = "enum";
		    break;
#ifdef DEBUG
		default:
		    printf(
		    "unrecognized StructOrUnion or EnumSpecifier, token=%d!\n",
			nodePointer->nodeToken );
		    keyWord="?";
		    break;
#endif /* DEBUG */
		}

		if ( nodePointer->nodeIdentifier != (TOKEN *) NULL )

		    fprintf( outputFile, "%s %s ",
			keyWord,
			nodePointer->nodeIdentifier->tokenSymbol->symName );

		else {

		    fprintf( outputFile, "%s { ", keyWord );

		    /*
		     * This is not the ideal way to accomplish this
		     * declaration.  This ought to be a recursive routine
		     * which will then work for the structure declaration
		     * list pointed to by nodePointer->nodeDeclList.
		     * But that would require parsing constant_expr and
		     * other stuff this implementation does not do.  It
		     * ought to, but it doesn't seem worth the effort
		     * just now--how many programmers write stuff like
		     * "... struct { ... } ... foo( ... ) ..."?  Not many,
		     * fortunately.  So for the present we resort to a
		     * hack, copying blocks of text from the declaration.
		     */

		    copyBlock( nodePointer->nodeDeclList->nodeByteStart,
			       nodePointer->nodeDeclList->nodeByteEnd );

		    fputs( " } ", outputFile );
		}

		break;

	    case TypeQualifier:

		switch( nodePointer->nodeToken ) {

		case CONST:
		    keyWord = "const";
		    break;

		case VOLATILE:
		    keyWord = "volatile";
		    break;
#ifdef DEBUG
		default:
		    printf( "unrecognized TypeQualifier, token=%d!\n",
			nodePointer->nodeToken );
		    keyWord="?";
		    break;
#endif /* DEBUG */
		}

		fputs( keyWord, outputFile );
		putc( ' ', outputFile );
		break;
#ifdef DEBUG
	    default:
		printf( "unrecognized nodetype=%d!\n",
			nodePointer->nodeType );
		break;
#endif /* DEBUG */
	    }

	    /* Loop repeats while there are still declaration specifiers    */
	}

	return;
}


/*
 * duplicateDeclarator:
 *	This function duplicates the declarator for an invoked function,
 *	with some important modifications.  The declarator is for a function.
 *	The final "f(...)" direct declarator is transformed into a simple
 *	variable, whose type is the same as that returned by the function.
 *	The name of this variable is that used in instrumentCode(), to hold
 *	the return value of the function.
 */

void
duplicateDeclarator( nodePointer )

NODE   *nodePointer;
{
	char   *keyWord;

	for ( ;
	      nodePointer != (NODE *) NULL;
	      nodePointer = nodePointer->nodeNext ) {

	    switch( nodePointer->nodeType ) {

	    case InitDeclarator:

		/* Skip initializers, which aren't really parsed, anyway    */

		break;

	    case Pointer:

		putc( '*', outputFile );
		break;

	    case TypeQualifier:

		switch( nodePointer->nodeToken ) {

		case CONST:
		    keyWord = "const";
		    break;

		case VOLATILE:
		    keyWord = "volatile";
		    break;
#ifdef DEBUG
		default:
		    printf( "unrecognized TypeQualifier, token=%d!\n",
			nodePointer->nodeToken );
		    keyWord="?";
		    break;
#endif /* DEBUG */
		}

		fputs( keyWord, outputFile );
		putc( ' ', outputFile );
		break;

	    case DirectDeclarator:

		switch( nodePointer->nodeToken ) {

		case 0:
		    /* This is   ( declarator )		Recurse!	    */

		    putc( '(', outputFile );
		    duplicateDeclarator( nodePointer->nodeNext );
		    putc( ')', outputFile );

		    return;

		case IDENTIFIER:
		    fprintf( outputFile, "_%s_ReturnValue_",
			nodePointer->nodeIdentifier->tokenSymbol->symName );
		    break;

		case LB:
		    /*
		     * This is an array, but no array dimension is
		     * duplicated, even if it was supplied originally,
		     * because this implementation does not parse
		     * constant_expr.
		     */

		    duplicateDeclarator( nodePointer->nodeNext );
		    putc( '[', outputFile );
		    putc( ']', outputFile );

		    return;

		case LP:
		    /*
		     * This is a function type.  In all cases we will
		     * ignore the parameter_type_list or identifier_list
		     * in nodePointer->nodeDeclList.  The parentheses
		     * will be duplicated UNLESS the direct declarator
		     * immediately below this NODE is the identifier.
		     */

		    if ( nodePointer->nodeNext->nodeToken != IDENTIFIER ) {

			duplicateDeclarator( nodePointer->nodeNext );
			putc( '(', outputFile );
			putc( ')', outputFile );

			return;
		    }
		    break;
#ifdef DEBUG
		default:
		    printf( "unrecognized DirectDeclarator, token=%d!\n",
			nodePointer->nodeToken );
		    break;
#endif /* DEBUG */
		}

		break;
#ifdef DEBUG
	    default:
		printf( "unrecognized nodetype=%d!\n",
			nodePointer->nodeType );
		break;
#endif /* DEBUG */
	    }

	    /* Loop repeats while there is still more declarator	    */
	}

	return;
}


/*
 * returnsVoid:
 *	This function inspects a declaration, returning TRUE if and only
 *	if it's a declaration for a function of type void.  The declaration
 *	specifiers must be nonempty, the VOID token must be present, and
 *	any combination of storage class specifiers may be present, but
 *	anything else in the declaration specifiers causes this function
 *	to return FALSE.  The declarator must be that of the simple "f(...)"
 *	kind, anything else causes FALSE to be returned.  One known
 *	limitation of this implementation is that if the input code has
 *	"typedef void VOIDTYPE;" and then "VOIDTYPE foo();" then this
 *	code will incorrectly behave as if foo does not have type void.
 *	Similarly, "typedef void VOIDFUNC();" and "VOIDFUNC foo;" will
 *	not be caught.
 */

int
returnsVoid( symPointer )

SYM    *symPointer;
{
	NODE   *nodePointer;
	int	voidPresent = FALSE;

	/* Examine the declaration specifiers.				    */

	for ( nodePointer = symPointer->symDeclSpec;
	      nodePointer != (NODE *) NULL;
	      nodePointer = nodePointer->nodeNext ) {

	    switch( nodePointer->nodeType ) {

	    case StorClassSpec:
	    case TypeQualifier:

		/* Skip over storage class specifiers and type qualifiers   */

		break;

	    case TypeSpecifier:

		if ( nodePointer->nodeToken != VOID )

		    return FALSE;

		voidPresent = TRUE;
		break;

	    default:

		return FALSE;
	    }
	}

	if ( voidPresent == FALSE )

	    return FALSE;

	/* Now examine the declarator.					    */

	nodePointer = symPointer->symDeclarator;

	if ( nodePointer->nodeType == InitDeclarator )

		nodePointer = nodePointer->nodeNext;

	if (( nodePointer->nodeType != DirectDeclarator ) ||
	    ( nodePointer->nodeToken != LP ))

		return FALSE;

	nodePointer = nodePointer->nodeNext;

	if (( nodePointer == (NODE *) NULL ) ||
	    ( nodePointer->nodeType != DirectDeclarator ) ||
	    ( nodePointer->nodeToken != IDENTIFIER ))

		return FALSE;

	return TRUE;
}


/*
 * freeIterList:
 *	Release storage for an invalidated iteration list.  This is done
 *	only when an input file is re-parsed.  The event IDs allocated
 *	in this list will not be used.
 */

void
freeIterList( iterPointer )

ITER   *iterPointer;
{
	ITER   *nextIter;

	while ( iterPointer != (ITER *) NULL ) {

		nextIter = iterPointer->iterNext;
		memFree( (void *) iterPointer );
		iterPointer = nextIter;
	}

	return;
}


/*
 * resetParser:
 *	This function is invoked to clean up data structures built up and
 *	used by parseFile and generateInstrumentedSourceFile.  Allocated
 *	storage is freed with no dangling pointers left to cause problems.
 */

void
resetParser()
{
	IFILE	       *inputFilePointer;

	currentFunc = (SYM *) NULL;


	/*
	 * Find root of context tree (externals) and free entire tree.
	 * The only reason that currentContext might not point to the
	 * root of the tree is that the parse might have failed with
	 * an error, resulting in an incomplete tree.
	 */

	if ( currentContext != (CONTEXT *) NULL ) {

		while ( currentContext->cntxtParent != (CONTEXT *) NULL )

			currentContext = currentContext->cntxtParent;

		freeContextTree( currentContext );

		currentContext = (CONTEXT *) NULL;

	}


	/* Find root of trace tree and free entire tree.		    */

	if ( currentTrace != (TRACETREE *) NULL ) {

		while(( currentTrace->traceLeft != (TRACETREE *) NULL ) ||
		      ( currentTrace->traceParent != (TRACETREE *) NULL )) {

			if ( currentTrace->traceParent != (TRACETREE *) NULL )

				currentTrace = currentTrace->traceParent;
			else
				currentTrace = currentTrace->traceLeft;
		}

		freeTraceTree( currentTrace );

		currentTrace = (TRACETREE *) NULL;

	}


	/* Free list of input files.					    */

	while ( inputFileList != (IFILE *) NULL ) {

		inputFilePointer = inputFileList->inputFileNext;

		memFree( (void *) inputFileList->inputFileName );

		memFree( (void *) inputFileList );

		inputFileList = inputFilePointer;

	}


	/* Free unlinked iterations list.				    */

	if ( iterList != (ITER *) NULL ) {

		freeIterList( iterList );

		iterList = (ITER *) NULL;

	}

	currentInputFile = (char *) NULL;

	cppOutputFile[ 0 ] = '\0';

	inputOffset = 0;

	inputLineNumber = 1;


	/* Free list of instrumentable constructs.			    */

	while ( instrumentHead != (INSTREC *) NULL ) {

		instrumentTail = instrumentHead->next;

		memFree( (void *) instrumentHead );

		instrumentHead = instrumentTail;

	}


	/* Close parser input stream; this deletes the temporary file.	    */

	if (( yyin != stdin ) && ( yyin != (FILE *) NULL )) {

		(void) fclose( yyin );

		yyin = stdin;

	}

	return;
}


/*
 * freeContextTree:
 *	Free allocated storage for a context (sub)tree.
 */

void
freeContextTree( contextPointer )

CONTEXT	       *contextPointer;
{
	if ( contextPointer == (CONTEXT *) NULL )
		return;

	freeContextTree( contextPointer->cntxtOldestChild );
	freeContextTree( contextPointer->cntxtYoungerSibling );
	freeNodes( contextPointer->cntxtNodeHead );
	freeSyms( contextPointer->cntxtSymHead );

	memFree( (void *) contextPointer );

	return;
}


/*
 * freeTraceTree:
 *	Free (sub)tree of traceable constructs.
 */

void
freeTraceTree( tracePointer )

TRACETREE      *tracePointer;
{
	if ( tracePointer == (TRACETREE *) NULL )
		return;

	freeTraceTree( tracePointer->traceChild );
	freeTraceTree( tracePointer->traceRight );

	memFree( (void *) tracePointer );

	return;
}



/*	PARSER ROUTINES SECTION						    */


/*
 * yyerror:
 *	Called by yyparse when an error situation occurs.  The message
 *	argument will almost always be "syntax error", but "yacc stack
 *	overflow" is also a possibility.  Since there is no error
 *	recovery built into the grammar, yyparse will return 1 (failure)
 *	immediately after invoking yyerror.
 */

yyerror( message )

char   *message;
{
	fprintf( stderr, "Error: %s, line %u: %s\n",
		currentInputFile, inputLineNumber, message );

	return;
}


/*
 * makeNode:
 *	This function is called by the parser code (see below) to
 *	create a new node for the partial parse tree.  A pointer to the
 *	allocated and partially initialized NODE structure is returned.
 *	Arguments to makeNode are the nodeType, indicating which non-
 *	terminal rule in the grammar is causing this node to be created,
 *	and tokenType, which narrows down which alternative is causing
 *	the nonterminal to be reduced.
 */

NODE *
makeNode( nodeType, tokenType )

NODETYPE	nodeType;
int		tokenType;
{
	NODE   *nodePointer;

	nodePointer = (NODE *) memAlloc( sizeof(NODE) );

	nodePointer->nodeType = nodeType;
	nodePointer->nodeToken = tokenType;
	nodePointer->nodeNext = (NODE *) NULL;
	nodePointer->nodeList = (NODE *) NULL;
	nodePointer->nodeEndList = (NODE *) NULL;
	nodePointer->nodeDeclList = (NODE *) NULL;
	nodePointer->nodeIdentifier = (TOKEN *) NULL;

	/* The following positional data will usually be changed by	    */
	/* other parser code reducing this nonterminal.			    */

	nodePointer->nodeByteStart = inputOffset;
	nodePointer->nodeByteEnd = inputOffset;
	nodePointer->nodeLineStart = inputLineNumber;
	nodePointer->nodeLineEnd = inputLineNumber;
	nodePointer->nodeFileStart = currentInputFile;
	nodePointer->nodeFileEnd = currentInputFile;

	return nodePointer;
}


/*
 * findToken:
 *	This function is called by the parser code to search a
 *	parse subtree for a specific token type.  The search is
 *	only down one path of the subtree, the nodeNext pointers,
 *	avoiding the nodeList pointers.  The first node found whose
 *	nodeToken matches the tokenType argument ends the search; a
 *	pointer to that node is then returned.  The NULL pointer is
 *	returned if no node matching tokenType is found in the path.
 *	Note that this function does not actually examine any of the
 *	TOKEN structures, only the token types stored in the nodeToken
 *	elements of NODE structures.
 */

NODE *
findToken( tokenType, nodePointer )

int	tokenType;
NODE   *nodePointer;
{
	while ( nodePointer != (NODE *) NULL )

		if ( nodePointer->nodeToken == tokenType )

			return nodePointer;
		else
			nodePointer = nodePointer->nodeNext;

	return (NODE *) NULL;
}


/*
 * contextBegin:
 *	This function is called by parser code at the beginning of
 *	a compound statement or function definition.  It increases
 *	the depth of the parse context tree, adding a new element.
 */

void
contextBegin()
{
	CONTEXT	       *newContext;

	newContext = (CONTEXT *) memAlloc( sizeof(CONTEXT) );

	newContext->cntxtNodeHead = (NODE *) NULL;
	newContext->cntxtSymHead = (SYM *) NULL;
	newContext->cntxtParent = currentContext;
	newContext->cntxtYoungerSibling = (CONTEXT *) NULL;
	newContext->cntxtOldestChild = (CONTEXT *) NULL;
	newContext->cntxtYoungestChild = (CONTEXT *) NULL;

	if ( currentContext == (CONTEXT *) NULL ) {

		/* Externals context; before parsing has begun		    */

		newContext->cntxtOlderSibling = (CONTEXT *) NULL;

	} else {

		newContext->cntxtOlderSibling =
					currentContext->cntxtYoungestChild;

		if ( currentContext->cntxtYoungestChild == (CONTEXT *) NULL )

			currentContext->cntxtOldestChild = newContext;
		else
			currentContext->cntxtYoungestChild->cntxtYoungerSibling
							= newContext;

		currentContext->cntxtYoungestChild = newContext;

	}

	currentContext = newContext;

	return;
}


/*
 * contextEnd:
 *	This function is called by parser code at the end of a compound
 *	statement or function definition.  It decreases the level of
 *	the context tree.
 */

void
contextEnd()
{
	currentContext = currentContext->cntxtParent;

#ifdef DEBUG
	if ( currentContext == (CONTEXT *) NULL ) {

		printf( "contextEnd: NULL CONTEXT!\n" );
		printf( "\toffset=%lu, file=%s line=%u\n",
			inputOffset, currentInputFile, inputLineNumber );
	}
#endif /* DEBUG */

	return;
}


/*
 * freeNodes:
 *	Called by parser code to free up a subtree of parse nodes,
 *	typically because of a language feature not being parsed.
 */

void
freeNodes( nodePointer )

NODE   *nodePointer;
{
	if ( nodePointer == (NODE *) NULL )
		return;

	freeNodes( nodePointer->nodeNext );
	freeNodes( nodePointer->nodeList );
	freeNodes( nodePointer->nodeDeclList );

	freeToken( nodePointer->nodeIdentifier );

	memFree( (void *) nodePointer );

	return;
}


/*
 * freeSyms:
 *	Called by parser code to free up a chain of symbol descriptors,
 *	all at the same context level.
 */

void
freeSyms( symPointer )

SYM    *symPointer;
{
	SYM    *nextSym;

	while ( symPointer != (SYM *) NULL ) {

		nextSym = symPointer->symNext;

		if ( symPointer->symSameName == (SYM *) NULL )

			memFree( (void *) ( symPointer->symName ) );

		memFree( (void *) symPointer );

		symPointer = nextSym;
	}

	return;
}


/*
 * setStartNode:
 *	This function accepts two NODE pointers as arguments.  The
 *	starting point data for the first argument is set to be equal
 *	to the starting point data for the second argument.
 */

void
setStartNode( destNode, srcNode )

NODE   *destNode;
NODE   *srcNode;
{
	destNode->nodeByteStart = srcNode->nodeByteStart;
	destNode->nodeLineStart = srcNode->nodeLineStart;
	destNode->nodeFileStart = srcNode->nodeFileStart;

	return;
}


/*
 * setEndNode:
 *	This function accepts two NODE pointers as arguments.  The
 *	ending point data for the first argument is set to be equal
 *	to the ending point data for the second argument.
 */

void
setEndNode( destNode, srcNode )

NODE   *destNode;
NODE   *srcNode;
{
	destNode->nodeByteEnd = srcNode->nodeByteEnd;
	destNode->nodeLineEnd = srcNode->nodeLineEnd;
	destNode->nodeFileEnd = srcNode->nodeFileEnd;

	return;
}


/*
 * setStartToken:
 *	This function accepts a NODE pointer and a TOKEN pointer as
 *	arguments.  The starting point data for the NODE argument is
 *	set to be equal to the starting point data for the TOKEN argument.
 */

void
setStartToken( destNode, srcToken )

NODE   *destNode;
TOKEN  *srcToken;
{
	destNode->nodeByteStart = srcToken->tokenByteStart;
	destNode->nodeLineStart = srcToken->tokenLine;
	destNode->nodeFileStart = srcToken->tokenInputFile;

	return;
}


/*
 * setEndToken:
 *	This function accepts a NODE pointer and a TOKEN pointer as
 *	arguments.  The end point data for the NODE argument is
 *	set to be equal to the ending point data for the TOKEN argument.
 */

void
setEndToken( destNode, srcToken )

NODE   *destNode;
TOKEN  *srcToken;
{
	destNode->nodeByteEnd = srcToken->tokenByteEnd;
	destNode->nodeLineEnd = srcToken->tokenLine;
	destNode->nodeFileEnd = srcToken->tokenInputFile;

	return;
}


/*
 * findProc:
 *	This function searches the list of called functions for a
 *	matching name.  If found, a pointer to that element is returned.
 *	Otherwise, a new element is created and added to the list.
 *	Event IDs are assigned at this time for function invocations.
 */

PROC *
findProc( symPointer )

SYM    *symPointer;
{
	PROC   *procPointer;

	for ( procPointer = procList;
	      procPointer != (PROC *) NULL;
	      procPointer = procPointer->procNext )

		if ( strcmp( symPointer->symName, procPointer->procName )
		    == 0 )

			return procPointer;

	procPointer = (PROC *) memAlloc( sizeof(PROC) );

	procPointer->procName = (char *)
				memAlloc( 1 + strlen( symPointer->symName ) );

	(void) strcpy( procPointer->procName, symPointer->symName );

	procPointer->procEventEntry = parserEventID++;
	procPointer->procEventExit = parserEventID++;
	procPointer->procDeclared = FALSE;
	procPointer->procIsVoid = returnsVoid( symPointer );
	procPointer->procNext = procList;
	procList = procPointer;

	return procPointer;
}


/*
 * recordFunctionCall:
 *	This function is called by parser code (see postfix_expr)
 *	to note the occurrence of a function call.  The only type of
 *	function calls which cause this function to be invoked are those
 *	in which the call is simply by the function name, e.g. "f()"
 *	(with or without arguments).  An element is added to the tree
 *	of traceable constructs.
 */

void
recordFunctionCall( nodePointer )

NODE   *nodePointer;
{
	TRACETREE      *newTrace;
	NODE	       *identNode;
	TOKEN	       *identToken;

	newTrace = (TRACETREE *) memAlloc( sizeof(TRACETREE) );

	identNode = findToken( IDENTIFIER, nodePointer );
	if ( identNode != (NODE *) NULL ) {

		identToken = identNode->nodeIdentifier;
		if ( identToken != (TOKEN *) NULL ) {

			checkSymbolName( identToken );
			newTrace->traceInvokedSym =
						identToken->tokenSymbol;
		}
#ifdef DEBUG
		else
			printf( "ident NODE with NULL nodeIdentifier!\n" );
#endif /* DEBUG */
	}
#ifdef DEBUG
	else
		printf( "function call with no identifier node!\n" );
#endif /* DEBUG */

	newTrace->traceInvokingSym = currentFunc;
	newTrace->traceByteStart = nodePointer->nodeByteStart;
	newTrace->traceFileStart = nodePointer->nodeFileStart;
	newTrace->traceLineStart = nodePointer->nodeLineStart;
	newTrace->traceByteEnd = nodePointer->nodeByteEnd;
	newTrace->traceFileEnd = nodePointer->nodeFileEnd;
	newTrace->traceLineEnd = nodePointer->nodeLineEnd;
	newTrace->traceConstruct = FunctionCall;
	newTrace->traceProc = findProc( identToken->tokenSymbol );
	newTrace->traceIter = (ITER *) NULL;
	newTrace->traceTranslate = (char *) NULL;

	addTraceTree( newTrace );

	return;
}


/*
 * recordLoop:
 *	This function is called by parser code (see iteration_statement)
 *	to note the occurrence of a loop.  An element is added to the tree
 *	of traceable constructs.  Nested loops are handled at a later
 *	time, after the entire file has been parsed.
 */

void
recordLoop( nodePointer )

NODE   *nodePointer;
{
	TRACETREE      *newTrace;

	newTrace = (TRACETREE *) memAlloc( sizeof(TRACETREE) );

	newTrace->traceInvokingSym = currentFunc;
	newTrace->traceInvokedSym = (SYM *) NULL;
	newTrace->traceByteStart = nodePointer->nodeByteStart;
	newTrace->traceFileStart = nodePointer->nodeFileStart;
	newTrace->traceLineStart = nodePointer->nodeLineStart;
	newTrace->traceByteEnd = nodePointer->nodeByteEnd;
	newTrace->traceFileEnd = nodePointer->nodeFileEnd;
	newTrace->traceLineEnd = nodePointer->nodeLineEnd;
	newTrace->traceConstruct = OuterLoop;
	newTrace->traceProc = (PROC *) NULL;
	newTrace->traceIter = (ITER *) memAlloc( sizeof(ITER) );
	newTrace->traceIter->iterEventEntry = parserEventID++;
	newTrace->traceIter->iterEventExit = parserEventID++;
	newTrace->traceIter->iterNext = iterList;
	iterList = newTrace->traceIter;
	newTrace->traceTranslate = (char *) NULL;

	addTraceTree( newTrace );

	return;
}


/*
 * recordFunctionDefinition:
 *	This function is called by parser code (see external_declaration)
 *	to note the location of a function definition.  The argument is
 *	the NODE pointer for the compound statement which contains the
 *	function body.  A new element is generated for the tree of traceable
 *	constructs, a parent node for all constructs in the function just
 *	parsed.
 */

void
recordFunctionDefinition( nodePointer )

NODE   *nodePointer;
{
	TRACETREE      *newTrace;

	contextEnd();			/* Back to externals context	    */

	newTrace = (TRACETREE *) memAlloc( sizeof(TRACETREE) );

	newTrace->traceInvokingSym = (SYM *) NULL;
	newTrace->traceInvokedSym = currentFunc;
	newTrace->traceByteStart = nodePointer->nodeByteStart + 1;
	newTrace->traceFileStart = nodePointer->nodeFileStart;
	newTrace->traceLineStart = nodePointer->nodeLineStart;
	newTrace->traceByteEnd = nodePointer->nodeByteEnd;
	newTrace->traceFileEnd = nodePointer->nodeFileEnd;
	newTrace->traceLineEnd = nodePointer->nodeLineEnd;
	newTrace->traceConstruct = FunctionBody;
	newTrace->traceProc = (PROC *) NULL;
	newTrace->traceIter = (ITER *) NULL;
	newTrace->traceTranslate = (char *) NULL;

	currentFunc = (SYM *) NULL;
	freeNodes( nodePointer );

	addTraceTree( newTrace );

	return;
}


/*
 * checkSymbolName:
 *	This function is called after a function call has been recognized
 *	and entered into the tree of traceable constructs (see the function
 *	recordFunctionCall) or when a function definition has begun (see
 *	makeFunctionDefinition).  The argument passed is the TOKEN pointer
 *	for the function name symbol.  This function compares the symbol
 *	name with the first member of each element of the translation
 *	table.  If no match is found, this function does nothing.  If
 *	a match is found, then a new element for the traceable constructs
 *	tree is generated.  At instrumentation time, this function name
 *	will be replaced with the corresponding translation name.
 */

void
checkSymbolName( tokenPointer )

TOKEN  *tokenPointer;
{
	TRANSLATE      *translatePointer;
	TRACETREE      *newTrace;

	for ( translatePointer = translateTable;
	      translatePointer->oldName != (char *) NULL;
	      translatePointer++ )

		if ( strcmp( translatePointer->oldName,
			     tokenPointer->tokenSymbol->symName ) == 0 )

			break;

	if ( translatePointer->oldName == (char *) NULL )

		return;

	newTrace = (TRACETREE *) memAlloc( sizeof(TRACETREE) );

	if ( currentFunc == tokenPointer->tokenSymbol )

		newTrace->traceInvokingSym = (SYM *) NULL;
	else
		newTrace->traceInvokingSym = currentFunc;

	newTrace->traceInvokedSym = tokenPointer->tokenSymbol;
	newTrace->traceByteStart = tokenPointer->tokenByteStart;
	newTrace->traceFileStart = tokenPointer->tokenInputFile;
	newTrace->traceLineStart = tokenPointer->tokenLine;
	newTrace->traceByteEnd = tokenPointer->tokenByteEnd;
	newTrace->traceFileEnd = tokenPointer->tokenInputFile;
	newTrace->traceLineEnd = tokenPointer->tokenLine;
	newTrace->traceConstruct = SymbolName;
	newTrace->traceProc = (PROC *) NULL;
	newTrace->traceIter = (ITER *) NULL;
	newTrace->traceTranslate = translatePointer->newName;

	addTraceTree( newTrace );

	return;
}


/*
 * addTraceTree:
 *	Link the argument element into the tree of traceable constructs.
 */

void
addTraceTree( newTrace )

TRACETREE      *newTrace;
{
	newTrace->traceType = InstrumentOff;
	newTrace->traceParent = (TRACETREE *) NULL;
	newTrace->traceLeft = (TRACETREE *) NULL;
	newTrace->traceRight = (TRACETREE *) NULL;
	newTrace->traceChild = (TRACETREE *) NULL;

	/*
	 * Since constructs are recognized in bottom-up order, it is
	 * guaranteed that:
	 *	newTrace->traceByteEnd >= currentTrace->traceByteEnd
	 * Because language constructs do not overlap except to nest, there
	 * are only two possibilities:
	 * 1.	newTrace->traceByteStart > currentTrace->traceByteEnd
	 *   In which case the constructs are disjoint and become siblings.
	 * 2.	newTrace->traceByteStart <= currentTrace->traceByteStart
	 *   The new trace tree element becomes the parent.
	 */

	while (( currentTrace != (TRACETREE *) NULL ) &&
	       ( newTrace->traceByteStart <= currentTrace->traceByteStart ))
	{
		currentTrace->traceParent = newTrace;
		newTrace->traceChild = currentTrace;
		currentTrace = currentTrace->traceLeft;
	}

	if ( currentTrace != (TRACETREE *) NULL ) {

		if ( currentTrace->traceRight != (TRACETREE *) NULL )

			currentTrace->traceRight->traceLeft =
							(TRACETREE *) NULL;

		newTrace->traceLeft = currentTrace;
		currentTrace->traceRight = newTrace;

	}

	currentTrace = newTrace;

	return;
}


/*
 * makePrimaryExpr:
 *	This function is called by the parser code to reduce the
 *	primary_expr nonterminal.  The rule is:
 *		primary_expr
 *			: identifier
 *			| INTEGER_CONSTANT
 *			| CHARACTER_CONSTANT
 *			| FLOATING_CONSTANT
 *			| STRING_LITERAL
 *			| LP expr RP
 *			;
 *	The first argument is the (first) token pointer.  For the final
 *	case, the exprNode and finalToken arguments will point to the
 *	expr and RP, respectively.
 */

NODE *
makePrimaryExpr( primaryExprToken, exprNode, finalToken )

TOKEN  *primaryExprToken;
NODE   *exprNode;
TOKEN  *finalToken;
{
	NODE   *primaryExpr;

	primaryExpr = makeNode( PrimaryExpr, primaryExprToken->tokenType );

	setStartToken( primaryExpr, primaryExprToken );

	if ( finalToken != (TOKEN *) NULL )

		setEndToken( primaryExpr, finalToken );
	else
		setEndToken( primaryExpr, primaryExprToken );

	if ( primaryExprToken->tokenType == IDENTIFIER )

		primaryExpr->nodeIdentifier = primaryExprToken;
	else
		freeToken( primaryExprToken );

	freeNodes( exprNode );
	freeToken( finalToken );

	return primaryExpr;
}


/*
 * makePostfixExpr:
 *	This function is called by the parser code to reduce the
 *	postfix_expr nonterminal.  The rule is:
 *		postfix_expr
 *			: primary_expr
 *			| postfix_expr LB expr RB
 *			| postfix_expr LP RP
 *			| postfix_expr LP argument_expr_list RP
 *			| postfix_expr MEMBER_OP identifier
 *			| postfix_expr PTR_OP identifier
 *			| postfix_expr INC_OP
 *			| postfix_expr DEC_OP
 *			;
 *	The only ones of these we're interested in (for this implementation)
 *	are the function calls.  All others return a NODE with only positional
 *	data.  The first argument is the NODE pointer returned from a
 *	previous reduction of postfix_expr.  The second argument is a LB or
 *	LP delimiter token, or an operator.  The third argument is an expr or
 *	argument_expr_list (which is syntactically equivalent to expr).
 *	The fourth argument is the final RB or RP delimiter or an identifier
 *	(struct/union member).
 */

NODE *
makePostfixExpr( oldPostfixExpr, leftDelim, exprNode, rightDelim )

NODE   *oldPostfixExpr;
TOKEN  *leftDelim;
NODE   *exprNode;
TOKEN  *rightDelim;
{
	NODE   *postfixExpr;

	postfixExpr = makeNode( PostfixExpr, leftDelim->tokenType );

	postfixExpr->nodeNext = oldPostfixExpr;

	setStartNode( postfixExpr, oldPostfixExpr );

	if ( rightDelim != (TOKEN *) NULL )

		setEndToken( postfixExpr, rightDelim );
	else
		setEndToken( postfixExpr, leftDelim );

	freeToken( leftDelim );
	freeNodes( exprNode );
	freeToken( rightDelim );

	return postfixExpr;
}


/*
 * makeDeclaration:
 *	This function is called by the parser code to reduce the
 *	declaration nonterminal.  The rule is:
 *		declaration
 *			: declaration_specifiers SM
 *			| declaration_specifiers init_declarator_list SM
 *			;
 *	The init_declarator_list is passed in as the second argument,
 *	which may be NULL.  If NULL, there are no actual objects, so
 *	this is no more than the declaration of an incomplete type.
 *	If not NULL, then in that NODE list there are one or more
 *	declarators.  For each of those declarators we must apply the
 *	declaration_specifiers returned by the NODE list in the first
 *	argument.  Search down the declarator list until the identifier
 *	is found, then set the entries of the corresponding SYM structure
 *	to point to both the declaration_specifiers and the head of this
 *	declarator list.  More importantly, set the symToken of that
 *	SYM structure to be TYPEDEF_NAME if the TYPEDEF token is encountered
 *	in the declaration_specifiers list.
 */

NODE *
makeDeclaration( declSpec, declList, semiColon )

NODE   *declSpec;
NODE   *declList;
TOKEN  *semiColon;
{
	NODE	       *declaration, *declarator, *identNode;
	TOKEN	       *identToken;
	SYM	       *identSymbol;
	int		symToken;

	if ( declList != (NODE *) NULL )
		declList->nodeEndList = (NODE *) NULL;

	declaration = makeNode( Declaration, 0 );
	declaration->nodeNext = declSpec;
	declaration->nodeDeclList = declList;

	if ( findToken( TYPEDEF, declSpec ) != (NODE *) NULL )
	    symToken = TYPEDEF_NAME;
	else
	    symToken = IDENTIFIER;

	for ( declarator = declList;
	      declarator != (NODE *) NULL;
	      declarator = declarator->nodeList ) {

	    identNode = findToken( IDENTIFIER, declarator );
	    if ( identNode != (NODE *) NULL ) {

		identToken = identNode->nodeIdentifier;
		if ( identToken != (TOKEN *) NULL ) {

		    identSymbol = identToken->tokenSymbol;
		    if ( identSymbol != (SYM *) NULL ) {

			identSymbol->symDeclSpec = declSpec;
			identSymbol->symDeclarator = declarator;
			identSymbol->symToken = symToken;
		    }
#ifdef DEBUG
		    else {
			printf( "symbol token with tokenSymbol=NULL!\n" );
			printf( "\ttoken=%d offset=%lu-%lu file=%s line=%u\n",
				identToken->tokenType,
				identToken->tokenByteStart,
				identToken->tokenByteEnd,
				identToken->tokenInputFile,
				identToken->tokenLine );
		    }
#endif /* DEBUG */
		}
#ifdef DEBUG
		else {
		    printf( "IDENTIFIER node with nodeIdentifier=NULL!\n" );
		    printf( "\tnterm=%d token=%d\n",
			identNode->nodeType, identNode->nodeToken );
		    printf( "\tstart offset=%lu file=%s line=%u\n",
			identNode->nodeByteStart,
			identNode->nodeFileStart,
			identNode->nodeLineStart );
		    printf( "\tend offset=%lu file=%s line=%u\n",
			identNode->nodeByteEnd,
			identNode->nodeFileEnd,
			identNode->nodeLineEnd );
		}
#endif /* DEBUG */
	    }
#ifdef DEBUG
	    else {
		printf( "declarator with no IDENTIFIER node!\n" );
		printf( "\tnterm=%d token=%d\n",
			declarator->nodeType, declarator->nodeToken );
		printf( "\tstart offset=%lu file=%s line=%u\n",
			declarator->nodeByteStart,
			declarator->nodeFileStart,
			declarator->nodeLineStart );
		printf( "\tend offset=%lu file=%s line=%u\n",
			declarator->nodeByteEnd,
			declarator->nodeFileEnd,
			declarator->nodeLineEnd );
	    }
#endif /* DEBUG */
	}

	setStartNode( declaration, declSpec );
	setEndToken( declaration, semiColon );

	freeToken( semiColon );

	return declaration;
}


/*
 * makeInitDeclarator:
 *	This function is called by the parser code to reduce the
 *	init_declarator nonterminal.  The rule is:
 *		init_declarator
 *			: declarator
 *			| declarator ASSIGN_OP initializer
 *			;
 *	The declarator is passed in as the first argument, the assignment
 *	operator token as the second (which may be NULL) and the initializer
 *	as the third argument, which is NULL if and only if the assignment
 *	operator token is NULL.
 */

NODE *
makeInitDeclarator( declarator, assignOper, initializer )

NODE   *declarator;
TOKEN  *assignOper;
NODE   *initializer;
{
	NODE   *initDecl;

	if ( assignOper == (TOKEN *) NULL )

		initDecl = makeNode( InitDeclarator, 0 );
	else
		initDecl = makeNode( InitDeclarator, assignOper->tokenType );

	initDecl->nodeNext = declarator;

	setStartNode( initDecl, declarator );

	if ( initializer != (NODE *) NULL )

		setEndNode( initDecl, initializer );
	else
		setEndNode( initDecl, declarator );

	freeToken( assignOper );
	freeNodes( initializer );

	return initDecl;
}


/*
 * makeStorClassSpec:
 *	This function is called by the parser code to reduce the
 *	storage_class_specifier nonterminal.  The rule is:
 *		storage_class_specifier
 *			: TYPEDEF
 *			| EXTERN
 *			| STATIC
 *			| AUTO
 *			| REGISTER
 *			;
 *	The argument is the token pointer.
 */

NODE *
makeStorClassSpec( storClass )

TOKEN  *storClass;
{
	NODE   *storClassSpec;

	storClassSpec = makeNode( StorClassSpec, storClass->tokenType );

	setStartToken( storClassSpec, storClass );
	setEndToken( storClassSpec, storClass );

	freeToken( storClass );

	return storClassSpec;
}


/*
 * makeTypeSpecifier:
 *	This function is called by the parser code to reduce the
 *	type_specifier nonterminal.  The rule is:
 *		type_specifier
 *			: CHAR
 *			| SHORT
 *			| INT
 *			| LONG
 *			| SIGNED
 *			| UNSIGNED
 *			| FLOAT
 *			| DOUBLE
 *			| VOID
 *			| struct_or_union_specifier
 *			| enum_specifier
 *			| TYPEDEF_NAME
 *			;
 *	The cases for struct_or_union_specifier and enum_specifier
 *	do not result in the creation of a TypeSpecifier node, the
 *	parser does not call this function for those cases.  This
 *	function accepts a single argument, a TOKEN pointer.
 */

NODE *
makeTypeSpecifier( typeSpecToken )

TOKEN  *typeSpecToken;
{
	NODE   *typeSpecifier;

	typeSpecifier = makeNode( TypeSpecifier, typeSpecToken->tokenType );

	setStartToken( typeSpecifier, typeSpecToken );
	setEndToken( typeSpecifier, typeSpecToken );

	if ( typeSpecToken->tokenType == TYPEDEF_NAME )

		typeSpecifier->nodeIdentifier = typeSpecToken;
	else
		freeToken( typeSpecToken );

	return typeSpecifier;
}


/*
 * makeTypeQualifier:
 *	This function is called by the parser code to reduce the
 *	type_qualifier nonterminal.  The rule is:
 *		type_qualifier
 *			: CONST
 *			| VOLATILE
 *			;
 *	The argument is the token pointer.
 */

NODE *
makeTypeQualifier( typeQual )

TOKEN  *typeQual;
{
	NODE   *typeQualifier;

	typeQualifier = makeNode( TypeQualifier, typeQual->tokenType );

	setStartToken( typeQualifier, typeQual );
	setEndToken( typeQualifier, typeQual );

	freeToken( typeQual );

	return typeQualifier;
}


/*
 * makeStructOrUnion:
 *	This function is called by the parser code to reduce the
 *	struct_or_union nonterminal.  The rule is:
 *		struct_or_union
 *			: STRUCT
 *			| UNION
 *			;
 *	The argument is the token pointer.
 */

NODE *
makeStructOrUnion( structUnionToken )

TOKEN  *structUnionToken;
{
	NODE   *structOrUnion;

	structOrUnion = makeNode( StructOrUnion, structUnionToken->tokenType );

	setStartToken( structOrUnion, structUnionToken );
	setEndToken( structOrUnion, structUnionToken );

	freeToken( structUnionToken );

	return structOrUnion;
}


/*
 * makeStructDeclaration:
 *	This function is called by the parser code to reduce the
 *	struct_declaration nonterminal.  The rule is:
 *		struct_declaration
 *			: specifier_qualifier_list struct_declarator_list SM
 *			;
 *	This is similar to the situation at the declaration nonterminal.
 *	The first argument, specifier_qualifier_list, is basically the
 *	same as declaration_specifiers (with no storage class specifiers
 *	allowed) while the second argument, struct_declarator_list, is a
 *	declarator list, allowing COL instead of initializers.  However,
 *	in this situation we don't change any SYM elements, since both
 *	struct/union member names aren't actual objects and there can be
 *	no TYPEDEFs used.
 */

NODE *
makeStructDeclaration( specQualList, structDeclList, semiColon )

NODE   *specQualList;
NODE   *structDeclList;
TOKEN  *semiColon;
{
	NODE   *structDeclaration;

	structDeclList->nodeEndList = (NODE *) NULL;

	structDeclaration = makeNode( StructDeclaration, 0 );
	structDeclaration->nodeNext = specQualList;
	structDeclaration->nodeDeclList = structDeclList;

	setStartNode( structDeclaration, specQualList );
	setEndToken( structDeclaration, semiColon );

	freeToken( semiColon );

	return structDeclaration;
}


/*
 * makeStructDeclarator:
 *	This function is called by the parser code to reduce the
 *	struct_declarator nonterminal.  The rule is:
 *		struct_declarator
 *			: declarator
 *			| COL constant_expr
 *			| declarator COL constant_expr
 *			;
 *	The declarator is passed in as the first argument, the colon
 *	token as the second and the constant_expr as the third argument,
 *	which are both either NULL or non-NULL.  Note that it is
 *	possible, in the case of an unlabeled bit field, for the
 *	declarator argument to be NULL.
 */

NODE *
makeStructDeclarator( declarator, colonToken, constantExpr )

NODE   *declarator;
TOKEN  *colonToken;
NODE   *constantExpr;
{
	NODE   *structDecl;

	if ( colonToken == (TOKEN *) NULL )

		structDecl = makeNode( StructDeclarator, 0 );
	else
		structDecl = makeNode( StructDeclarator,
				       colonToken->tokenType );

	structDecl->nodeNext = declarator;


	if ( declarator != (NODE *) NULL )

		setStartNode( structDecl, declarator );
	else
		setStartToken( structDecl, colonToken );


	if ( constantExpr != (NODE *) NULL )

		setEndNode( structDecl, constantExpr );
	else
		setEndNode( structDecl, declarator );


	freeToken( colonToken );
	freeNodes( constantExpr );

	return structDecl;
}


/*
 * makeEnumSpecifier:
 *	This function is called by the parser code to reduce the
 *	enum_specifier nonterminal.  The rule is:
 *		enum_specifier
 *			: ENUM LC enumerator_list RC
 *			| ENUM identifier LC enumerator_list RC
 *			| ENUM identifier
 *			;
 *	The arguments passed to this function are the ENUM token
 *	pointer, the identifier tag (if present), the left curly
 *	brace token, the enumerator list, and the right curly brace
 *	token.  Note that the last three arguments are either all
 *	NULL or all non-NULL.  Independently, the identifier token
 *	may be NULL.
 */

NODE *
makeEnumSpecifier( enumToken, identToken, leftCurl, enumList, rightCurl )

TOKEN  *enumToken;
TOKEN  *identToken;
TOKEN  *leftCurl;
NODE   *enumList;
TOKEN  *rightCurl;
{
	NODE   *enumSpecifier;

	if ( enumList != (NODE *) NULL )
		enumList->nodeEndList = (NODE *) NULL;

	enumSpecifier = makeNode( EnumSpecifier, enumToken->tokenType );

	enumSpecifier->nodeIdentifier = identToken;
	enumSpecifier->nodeDeclList = enumList;

	setStartToken( enumSpecifier, enumToken );

	if ( rightCurl != (TOKEN *) NULL )

		setEndToken( enumSpecifier, rightCurl );
	else
		setEndToken( enumSpecifier, identToken );

	freeToken( enumToken );
	freeToken( leftCurl );
	freeToken( rightCurl );

	return enumSpecifier;
}


/*
 * makeEnumerator:
 *	This function is called by the parser code to reduce the
 *	enumerator nonterminal.  The rule is:
 *		enumerator
 *			: identifier
 *			| identifier ASSIGN_OP constant_expr
 *			;
 *	The identifier is passed in as the first argument, the assignment
 *	operator token as the second and the constant_expr as the third,
 *	both of which are either NULL or non-NULL.
 */

NODE *
makeEnumerator( identToken, assignOper, constantExpr )

TOKEN  *identToken;
TOKEN  *assignOper;
NODE   *constantExpr;
{
	NODE   *enumerator;

	enumerator = makeNode( Enumerator, identToken->tokenType );

	enumerator->nodeIdentifier = identToken;

	setStartToken( enumerator, identToken );

	if ( constantExpr != (NODE *) NULL )

		setEndNode( enumerator, constantExpr );
	else
		setEndToken( enumerator, identToken );

	freeToken( assignOper );
	freeNodes( constantExpr );

	return enumerator;
}


/*
 * makeDirectDeclarator:
 *	This function is called by the parser code to reduce the
 *	direct_declarator nonterminal.  The rule is:
 *		direct_declarator
 *			: identifier
 *			| LP declarator RP
 *			| direct_declarator LB RB
 *			| direct_declarator LB constant_expr RB
 *			| direct_declarator LP RP
 *			| direct_declarator LP parameter_type_list RP
 *			| direct_declarator LP identifier_list RP
 *			;
 *	The arguments to this function are:
 *
 *	subTypeToken:  a TOKEN pointer indicating which type of rule is
 *		being reduced, equal to identifier, NULL, LB, LB, LP, LP,
 *		or LP, respectively.
 *	leftDelim:  a TOKEN pointer, NULL in the identifier rule, either
 *		LP or LB in the other cases; not necessarily the leftmost
 *		token in the rule!
 *	rightDelim:  a TOKEN pointer, NULL in the identifier rule, either
 *		RP or RB in the other cases; always the rightmost token
 *		when not NULL.
 *	declarator:  a NODE pointer, NULL in the identifier rule, declarator
 *		or direct_declarator in the others.
 *	declList:  a NODE pointer, NULL in all except the last two rules,
 *		in which it is parameter_type_list or identifier_list.
 *
 *	The constant_exprs are not parsed in this implementation.
 */

NODE *
makeDirectDeclarator( subTypeToken, leftDelim, rightDelim,
		      declarator, declList )

TOKEN  *subTypeToken;
TOKEN  *leftDelim;
TOKEN  *rightDelim;
NODE   *declarator;
NODE   *declList;
{
	NODE   *directDeclarator;
	int	tokenType;

	if ( declList != (NODE *) NULL )
		declList->nodeEndList = (NODE *) NULL;

	if ( subTypeToken == (TOKEN *) NULL )

		tokenType = 0;
	else
		tokenType = subTypeToken->tokenType;

	directDeclarator = makeNode( DirectDeclarator, tokenType );

	if ( tokenType == IDENTIFIER )

		directDeclarator->nodeIdentifier = subTypeToken;

	directDeclarator->nodeNext = declarator;	/* possibly NULL    */

	directDeclarator->nodeDeclList = declList;	/* possibly NULL    */

	switch( tokenType ) {

	case IDENTIFIER:
		setStartToken( directDeclarator, subTypeToken );
		setEndToken( directDeclarator, subTypeToken );
		break;

	case 0:
		setStartToken( directDeclarator, leftDelim );
		setEndToken( directDeclarator, rightDelim );
		break;

	default:
		setStartNode( directDeclarator, declarator );
		setEndToken( directDeclarator, rightDelim );
		break;
	}

	freeToken( leftDelim );
	freeToken( rightDelim );

	return directDeclarator;
}


/*
 * makePointer:
 *	This function is called by the parser code to reduce the
 *	pointer nonterminal.  The rule is:
 *		pointer
 *			: ASTERISK
 *			| ASTERISK type_qualifier_list
 *			| ASTERISK pointer
 *			| ASTERISK type_qualifier_list pointer
 *			;
 *	The arguments are the ASTERISK token pointer, and NODE pointers
 *	returned for the type_qualifier_list and/or pointer nonterminals.
 *	Both NODE pointers may be NULL, but if the first of the two is
 *	NULL then the second one will be also.
 */

NODE *
makePointer( pointerToken, firstList, secondList )

TOKEN  *pointerToken;
NODE   *firstList;
NODE   *secondList;
{
	NODE   *pointerNode;

	pointerNode = makeNode( Pointer, pointerToken->tokenType );

	setStartToken( pointerNode, pointerToken );

	if ( firstList == (NODE *) NULL ) {

		pointerNode->nodeEndList = pointerNode;
		setEndToken( pointerNode, pointerToken );

	} else if ( secondList == (NODE *) NULL ) {

		pointerNode->nodeNext = firstList;
		pointerNode->nodeEndList = firstList->nodeEndList;
		firstList->nodeEndList = (NODE *) NULL;
		setEndNode( pointerNode, firstList );

	} else {

		pointerNode->nodeNext = firstList;
		pointerNode->nodeEndList = secondList->nodeEndList;
		secondList->nodeEndList = (NODE *) NULL;
		firstList->nodeEndList->nodeNext = secondList;
		firstList->nodeEndList = (NODE *) NULL;
		setEndNode( pointerNode, secondList );
	}

	freeToken( pointerToken );

	return pointerNode;
}


/*
 * makeIdentifierList:
 *	This function is called by the parser code to reduce the
 *	identifier_list nonterminal.  The rule is:
 *		identifier_list
 *			: identifier
 *			| identifier_list CM identifier
 *			;
 *	The identifier is passed in as the third argument, the identifier_list
 *	node and comma token as the first and second arguments, which are
 *	either both NULL or both not NULL.
 */

NODE *
makeIdentifierList( oldIdentList, commaToken, identToken )

NODE   *oldIdentList;
TOKEN  *commaToken;
TOKEN  *identToken;
{
	NODE   *identifierList;

	identifierList = makeNode( IdentifierList, identToken->tokenType );

	identifierList->nodeIdentifier = identToken;
	setStartToken( identifierList, identToken );
	setEndToken( identifierList, identToken );

	if ( oldIdentList == (NODE *) NULL ) {

		identifierList->nodeEndList = identifierList;

		return identifierList;

	} else {

		freeToken( commaToken );
		oldIdentList->nodeEndList->nodeNext = identifierList;
		oldIdentList->nodeEndList = identifierList;
		setEndNode( oldIdentList, identifierList );

		return oldIdentList;
	}
}


/*
 * makeParameterTypeList:
 *	This function is called by the parser code to reduce the
 *	parameter_type_list nonterminal.  The rule is:
 *		parameter_type_list
 *			: parameter_list
 *			| parameter_list CM ELLIPSIS
 *			;
 *	The parameter_list is passed in as the first argument, the comma
 *	and ellipsis tokens as the second and third arguments, which are
 *	either both NULL or both not NULL.
 */

NODE *
makeParameterTypeList( parameterList, commaToken, ellipsisToken )

NODE   *parameterList;
TOKEN  *commaToken;
TOKEN  *ellipsisToken;
{
	NODE   *parameterTypeList;

	if ( commaToken != (TOKEN *) NULL ) {

		parameterTypeList = makeNode( ParameterTypeList,
					      ellipsisToken->tokenType );
		setStartToken( parameterTypeList, ellipsisToken );
		setEndToken( parameterTypeList, ellipsisToken );

		parameterList->nodeEndList->nodeList = parameterTypeList;
		setEndNode( parameterList, parameterTypeList );

		freeToken( commaToken );
		freeToken( ellipsisToken );
	}

	parameterList->nodeEndList = (NODE *) NULL;

	return parameterList;
}


/*
 * makeParamDeclaration:
 *	This function is called by the parser code to reduce the
 *	parameter_declaration nonterminal.  The rule is:
 *		parameter_declaration
 *			: declaration_specifiers
 *			| declaration_specifiers declarator
 *			| declaration_specifiers abstract_declarator
 *			;
 *	This is much simpler than the declaration nonterminal.  The
 *	declaration_specifiers are passed in as the (non-NULL) first
 *	argument, the declarator or abstract_declarator as the (possibly
 *	NULL) second argument.
 */

NODE *
makeParamDeclaration( declSpec, declarator )

NODE   *declSpec;
NODE   *declarator;
{
	NODE   *paramDeclaration;

	paramDeclaration = makeNode( ParamDeclaration, 0 );
	paramDeclaration->nodeNext = declSpec;
	paramDeclaration->nodeDeclList = declarator;	/* possibly NULL    */

	setStartNode( paramDeclaration, declSpec );

	if ( declarator == (NODE *) NULL )

		setEndNode( paramDeclaration, declSpec );
	else
		setEndNode( paramDeclaration, declarator );

	return paramDeclaration;
}


/*
 * makeDirAbstrDeclarator:
 *	This function is called by the parser code to reduce the
 *	direct_abstract_declarator nonterminal.  The rule is:
 *		direct_abstract_declarator
 *			: LP abstract_declarator RP
 *			| LB RB
 *			| LB constant_expr RB
 *			| direct_abstract_declarator LB RB
 *			| direct_abstract_declarator LB constant_expr RB
 *			| LP RP
 *			| LP parameter_type_list RP
 *			| direct_abstract_declarator LP RP
 *			| direct_abstract_declarator LP parameter_type_list RP
 *			;
 *	The arguments to this function are:
 *
 *	subTypeToken:  a TOKEN pointer indicating which type of rule is
 *		being reduced, equal to NULL in the first case only, LB or LP
 *		in all other cases.
 *	leftDelim:  a TOKEN pointer, either LB or LP; not necessarily the
 *		leftmost token in the rule!
 *	rightDelim:  a TOKEN pointer, either RB or RP; always the rightmost
 *		token.
 *	declarator:  a NODE pointer one of direct_abstract_declarator,
 *		abstract_declarator, or NULL.
 *	paramList:  a NODE pointer, parameter_type_list or NULL.
 *
 *	The constant_exprs are not parsed in this implementation.
 */

NODE *
makeDirAbstrDeclarator( subTypeToken, leftDelim, rightDelim,
			declarator, paramList )

TOKEN  *subTypeToken;
TOKEN  *leftDelim;
TOKEN  *rightDelim;
NODE   *declarator;
NODE   *paramList;
{
	NODE   *dirAbstrDeclarator;
	int	tokenType;

	if ( subTypeToken == (TOKEN *) NULL )

		tokenType = 0;
	else
		tokenType = subTypeToken->tokenType;

	dirAbstrDeclarator = makeNode( DirAbstrDeclarator, tokenType );
	dirAbstrDeclarator->nodeNext = declarator;	/* possibly NULL    */
	dirAbstrDeclarator->nodeList = paramList;	/* possibly NULL    */

	if (( tokenType != 0 ) && ( declarator != (NODE *) NULL ))

		setStartNode( dirAbstrDeclarator, declarator );
	else
		setStartToken( dirAbstrDeclarator, leftDelim );

	setEndToken( dirAbstrDeclarator, rightDelim );

	freeToken( leftDelim );
	freeToken( rightDelim );

	return dirAbstrDeclarator;
}


/*
 * makeCompoundStatement:
 *	This function is called by the parser code to reduce the
 *	compound_statement nonterminal.  The rule is:
 *		compound_statement
 *			: LC RC
 *			| LC compound_statement_body RC
 *			;
 *	The LC and RC delimiter tokens are passed as the first and
 *	second arguments.
 */

NODE *
makeCompoundStatement( leftDelim, rightDelim )

TOKEN  *leftDelim;
TOKEN  *rightDelim;
{
	NODE   *compoundStatement;

	compoundStatement = makeNode( CompoundStatement, 0 );

	setStartToken( compoundStatement, leftDelim );
	setEndToken( compoundStatement, rightDelim );

	freeToken( leftDelim );
	freeToken( rightDelim );

	return compoundStatement;
}


/*
 * makeExpressionStatement:
 *	This function is called by the parser code to reduce the
 *	expression_statement nonterminal.  The rule is:
 *		expression_statement
 *			: SM
 *			| expr SM
 *			;
 *	The first argument is the expression, the second argument is
 *	the semicolon token.
 */

NODE *
makeExpressionStatement( expression, semiColon )

NODE   *expression;
TOKEN  *semiColon;
{
	NODE   *expressionStatement;

	expressionStatement = makeNode( ExpressionStatement, 0 );

	if ( expression != (NODE *) NULL )

		setStartNode( expressionStatement, expression );
	else
		setStartToken( expressionStatement, semiColon );

	setEndToken( expressionStatement, semiColon );

	freeNodes( expression );
	freeToken( semiColon );

	return expressionStatement;
}


/*
 * makeJumpStatement:
 *	This function is called by the parser code to reduce the
 *	jump_statement nonterminal.  The rule is:
 *		jump_statement
 *			: GOTO identifier SM
 *			| CONTINUE SM
 *			| BREAK SM
 *			| RETURN SM
 *			| RETURN expr SM
 *			;
 *	The first argument is the jump keyword token, the second argument
 *	is the semicolon token.
 */

NODE *
makeJumpStatement( jumpToken, semiColon )

TOKEN  *jumpToken;
TOKEN  *semiColon;
{
	NODE   *jumpStatement;

	jumpStatement = makeNode( JumpStatement, 0 );

	setStartToken( jumpStatement, jumpToken );
	setEndToken( jumpStatement, semiColon );

	freeToken( jumpToken );
	freeToken( semiColon );

	return jumpStatement;
}


/*
 * makeFunctionDefinition:
 *	This function is called by the parser code to reduce the
 *	function_definition nonterminal.  The rule is:
 *		function_definition
 *			: declarator function_body
 *			| declaration_specifiers declarator function_body
 *			;
 *	The declaration_specifiers are passed in as the first argument,
 *	which may be NULL.  The declarator is passed in as the second argument.
 *	This works much like the declaration nonterminal in that the symbol
 *	found in the declarator node list must be associated with the
 *	(possibly NULL) tree of declaration_specifiers.  There is also
 *	function-specific work to do, unlike the declaration nonterminal.
 *	Note that this function is called at external context level.
 *	A new context level is begun (see the actions for this nonterminal
 *	in the grammar file) before processing the function_body.  This
 *	way, the function's formal parameters (if any are declared) go
 *	in a subordinate context, and any declarations inside the compound
 *	statement that makes up the body of the function will go in an
 *	even more subordinate context.
 */

void
makeFunctionDefinition( declSpec, declarator )

NODE   *declSpec;
NODE   *declarator;
{
	NODE   *functionDefinition, *identNode;
	TOKEN  *identToken;

	functionDefinition = makeNode( FunctionDefinition, 0 );
	functionDefinition->nodeNext = declSpec;	/* possibly NULL    */
	functionDefinition->nodeDeclList = declarator;

	/* Note that functions appear in the list pointed to by
	   currentContext->cntxtNodeHead in reverse order.		    */

	functionDefinition->nodeList = currentContext->cntxtNodeHead;
	currentContext->cntxtNodeHead = functionDefinition;

	identNode = findToken( IDENTIFIER, declarator );
	if ( identNode != (NODE *) NULL ) {

	    identToken = identNode->nodeIdentifier;
	    if ( identToken != (TOKEN *) NULL ) {

		currentFunc = identToken->tokenSymbol;
		if ( currentFunc != (SYM *) NULL ) {

		    currentFunc->symDeclSpec = declSpec;
		    currentFunc->symDeclarator = declarator;
		}
#ifdef DEBUG
		else {
		    printf( "symbol token with tokenSymbol=NULL!\n" );
		    printf( "\ttoken=%d offset=%lu-%lu file=%s line=%u\n",
			identToken->tokenType,
			identToken->tokenByteStart,
			identToken->tokenByteEnd,
			identToken->tokenInputFile,
			identToken->tokenLine );
		}
#endif /* DEBUG */

		checkSymbolName( identToken );
	    }
#ifdef DEBUG
	    else {
		printf( "IDENTIFIER node with nodeIdentifier=NULL!\n" );
		printf( "\tnterm=%d token=%d\n",
			identNode->nodeType, identNode->nodeToken );
		printf( "\tstart offset=%lu file=%s line=%u\n",
			identNode->nodeByteStart,
			identNode->nodeFileStart,
			identNode->nodeLineStart );
		printf( "\tend offset=%lu file=%s line=%u\n",
			identNode->nodeByteEnd,
			identNode->nodeFileEnd,
			identNode->nodeLineEnd );
	    }
#endif /* DEBUG */
	}
#ifdef DEBUG
	else {
	    printf( "declarator with no IDENTIFIER node!\n" );
	    printf( "\tnterm=%d token=%d\n",
			declarator->nodeType, declarator->nodeToken );
	    printf( "\tstart offset=%lu file=%s line=%u\n",
			declarator->nodeByteStart,
			declarator->nodeFileStart,
			declarator->nodeLineStart );
	    printf( "\tend offset=%lu file=%s line=%u\n",
			declarator->nodeByteEnd,
			declarator->nodeFileEnd,
			declarator->nodeLineEnd );
	}
#endif /* DEBUG */

	if ( declSpec != (NODE *) NULL )

		setStartNode( functionDefinition, declSpec );
	else
		setStartNode( functionDefinition, declarator );

	setEndNode( functionDefinition, declarator );

	contextBegin();

	return;
}



/*	LEXICAL ANALYZER ROUTINES SECTION				    */


/*
 * scanCPPline:
 *	The lexical analyzer has scanned a '#' character at the
 *	beginning of a line.  This is a cpp-output control line.
 *	If it's "# lineno [filename]", change the input file line
 *	number and potentially also the input file name.  Anything
 *	else can be safely ignored up to a newline character.  Then
 *	return control to yylex() as it pursues the next real token.
 */

void
scanCPPline()
{
	char		c, stringBuffer[ 256 ], *bufferPointer;
	unsigned int	lineNumber;

	inputOffset++;			/* Count the '#' already scanned.   */

	/* Skip optional white space after the #.			    */

	do {
		c = yyinput();
		if ( c == 0 )
			return;			/* EOF reached		    */

		inputOffset++;

	} while (( c == ' ' ) || ( c == '\t' ));

	/* Look for an optional cpp directive keyword.			    */

	bufferPointer = stringBuffer;

	while ( isalpha( c ) ) {

		*bufferPointer++ = c;

		c = yyinput();
		if ( c == 0 )
			return;			/* EOF reached		    */

		inputOffset++;
	}

	if ( bufferPointer != stringBuffer ) {

		/* A cpp directive keyword has been scanned.		    */
		/* If anything besides "line", skip remainder of line.	    */

		*bufferPointer = '\0';

		if ( strcmp( stringBuffer, "line" ) != 0 ) {

			while ( c != '\n' ) {

				c = yyinput();
				if ( c == 0 )	/* EOF reached		    */
					return;

				inputOffset++;
			}

			return;
		}
	}

	/* Skip optional white space (only happens after "#[ \t]*line").    */

	while (( c == ' ' ) || ( c == '\t' )) {

		c = yyinput();
		if ( c == 0 )
			return;			/* EOF reached		    */

		inputOffset++;
	}

	/* Look for a decimal integer and translate.			    */

	lineNumber = 0;

	while ( isdigit( c ) ) {

		lineNumber *= 10;
		lineNumber += c - '0';

		c = yyinput();
		if ( c == 0 )
			return;			/* EOF reached		    */

		inputOffset++;
	}

	/* If no line number found, skip remainder of line.		    */

	if ( lineNumber == 0 ) {

		while ( c != '\n' ) {

			c = yyinput();
			if ( c == 0 )
				return;		/* EOF reached		    */

			inputOffset++;
		}

		return;
	}

	/* Skip over white space after line number.			    */

	while (( c == ' ' ) || ( c == '\t' )) {

		c = yyinput();
		if ( c == 0 )
			return;			/* EOF reached		    */

		inputOffset++;
	}

	/* If a '"' character is found, this is the start of a file	    */
	/* name; copy it into a buffer.					    */

	bufferPointer = stringBuffer;

	if ( c == '"' ) {

		do {
			c = yyinput();
			if ( c == 0 )
				return;		/* EOF reached		    */

			inputOffset++;

			*bufferPointer++ = c;

		} while (( c != '"' ) && ( c != '\n' ));

		*--bufferPointer = '\0';
	}


	/* A complete cpp control line with line number (and optionally	    */
	/* a file name) has now been recognized.  Skip remainder of line.   */

	while ( c != '\n' ) {

		c = yyinput();
		if ( c == 0 )
			return;			/* EOF reached		    */

		inputOffset++;
	}

	/* The cpp control line has been read up through the newline.	    */
	/* Now process the control line by changing the current input	    */
	/* file name and current input line number.  This has to be	    */
	/* done by putting the newline back on the input stream because	    */
	/* yylex() will not properly recognize the ^"#" token if it's	    */
	/* the next character on the input stream after the newline and	    */
	/* we eat the newline here, i.e. if two cpp control lines in a	    */
	/* row appear on input.  This is silly, but cpp does that.	    */

	yyunput( '\n' );

	inputOffset--;

	if ( lineNumber != 0 ) {

		/* Set the current line number to be one less than that	    */
		/* specified by the cpp control line.  The newline just	    */
		/* put back on the yylex input buffer will immediately	    */
		/* cause the line number to be incremented.		    */

		inputLineNumber = lineNumber - 1;

		if ( bufferPointer != stringBuffer )

			makeInputFile( stringBuffer );
	}

	return;
}


/*
 * scanComment:
 *	The lexical analyzer has scanned "/*" signalling the start of a
 *	C comment.  Continue reading characters until a '*' is found.
 *	Then read the next character.  If it is a '/' then the entire
 *	comment has been read and control passes back to yylex() for
 *	the next token.  If it is anything but a '/', put the character
 *	back on the input stream and continue the loop.
 */

void
scanComment()
{
	char	c;

	inputOffset++;			/* Count the '/' already scanned.   */
	inputOffset++;			/* Count the '*' already scanned.   */

	do {
		do {
			c = yyinput();
			if ( c == 0 )
				return;		/* EOF reached		    */

			inputOffset++;

			if ( c == '\n' )

				inputLineNumber++;

		} while ( c != '*' );

		c = yyinput();
		if ( c == 0 )
			return;			/* EOF reached		    */

		if ( c != '/' )
			yyunput( c );

	} while ( c != '/' );

	inputOffset++;			/* Count the final '/'.		    */

	return;
}


/*
 * makeInputFile:
 *	This function is called by the lexical analyzer when it has
 *	scanned a cpp control line indicating a change of input file
 *	name and line number or at the beginning of execution with
 *	the name of the original input file as an argument.  The
 *	linked list of input file names will be examined for a match.
 *	If no match is found, a new element is added to the list.
 *	The currentInputFile pointer is updated.
 */

void
makeInputFile( fileName )

char   *fileName;
{
	IFILE		       *filePointer;

	for ( filePointer = inputFileList;
	      filePointer != (IFILE *) NULL;
	      filePointer = filePointer->inputFileNext )

		if ( strcmp( fileName, filePointer->inputFileName ) == 0 ) {

			currentInputFile = filePointer->inputFileName;

			return;
		}

	/* This is a new input file; allocate a list element and link it.    */

	currentInputFile = (char *) memAlloc( 1 + strlen( fileName ) );

	(void) strcpy( currentInputFile, fileName );

	filePointer = (IFILE *) memAlloc( sizeof(IFILE) );

	filePointer->inputFileName = currentInputFile;
	filePointer->inputFileNext = inputFileList;
	inputFileList = filePointer;

	return;
}


/*
 * makeToken:
 *	This function creates a TOKEN structure for a token of type
 *	tokenType.  The structure is initialized and a pointer to the
 *	structure is stored in yylval.token.  This function is invoked
 *	by yylex() when a non-symbol token is scanned.  The token type
 *	value is returned.
 */

int
makeToken( tokenType )

int	tokenType;
{
	TOKEN  *tokenPointer;
	char   *charPoint;
	int	i;

	tokenPointer = (TOKEN *) memAlloc( sizeof(TOKEN) );

	tokenPointer->tokenType = tokenType;
	tokenPointer->tokenInputFile = currentInputFile;
	tokenPointer->tokenLine = inputLineNumber;
	tokenPointer->tokenByteStart = inputOffset;
	tokenPointer->tokenByteEnd = ( inputOffset += yyleng );
	tokenPointer->tokenSymbol = (SYM *) NULL;

	/*
	 * A couple of specific tokens may have newline characters
	 * embedded within them.  Rather than relying on yylineno
	 * (which is incremented by yyinput() and decremented by
	 * yyunput() whenever a newline character is seen), the
	 * inputLineNumber is updated here, AFTER the token structure
	 * has been initialized.
	 */

	if (( tokenType == CHARACTER_CONSTANT )
	   || ( tokenType == STRING_LITERAL )) {

		for ( i = 0, charPoint = yytext;
		      i < yyleng;
		      i++, charPoint++ ) {

			if ( *charPoint == '\n' )

				inputLineNumber++;
		}
	}

	yylval.token = tokenPointer;
	return tokenType;
}


/*
 * makeSymToken:
 *	This function creates a TOKEN structure for a symbol token,
 *	either an IDENTIFIER or TYPEDEF_NAME.  The structure is
 *	initialized and a pointer to the structure is stored in
 *	yylval.token.  This function is invoked by yylex() when a
 *	symbol token is scanned.  The argument passed to this function
 *	is a pointer to the appropriate initialized SYM structure for
 *	this symbol.  The token type value is returned.
 */

int
makeSymToken( tokenSymbol )

SYM    *tokenSymbol;
{
	(void) makeToken( tokenSymbol->symToken );

	yylval.token->tokenSymbol = tokenSymbol;

	return tokenSymbol->symToken;
}


/*
 * makeSymbol:
 *	This function is invoked by yylex() whenever it has scanned
 *	a symbol token (identifier).  This function examines the symbol
 *	tables looking for a match with the symbol name, obeying scope rules.
 *	If the appropriate matching SYM structure is found, the pointer to
 *	that structure is returned.  Otherwise, a new SYM structure is
 *	created, initialized and entered into the symbol tables, and then
 *	the pointer to the new structure is returned.
 */

SYM *
makeSymbol()
{
	CONTEXT	       *cntxt;
	SYM	       *symHead, *symPointer, *samePointer;

	/* Search for symbol in this scope, return it if found.		    */

	cntxt = currentContext;
	symHead = cntxt->cntxtSymHead;

	for ( symPointer = symHead;
	      symPointer != (SYM *) NULL;
	      symPointer = symPointer->symNext )

		if ( strcmp( yytext, symPointer->symName ) == 0 )

			return symPointer;


	/* Not in this scope, create new element for this scope.	    */

	symPointer = (SYM *) memAlloc( sizeof(SYM) );

	/* Start to initialize new SYM element, link into list.		    */

	symPointer->symDeclSpec = (NODE *) NULL;
	symPointer->symDeclarator = (NODE *) NULL;
	symPointer->symNext = symHead;
	cntxt->cntxtSymHead = symHead = symPointer;

	/* Look for outer scope symbol with same name.			    */

	samePointer = (SYM *) NULL;

	while (( cntxt = cntxt->cntxtParent ) != (CONTEXT *) NULL ) {

		symHead = cntxt->cntxtSymHead;

		for ( samePointer = symHead;
		      samePointer != (SYM *) NULL;
		      samePointer = samePointer->symNext )

			if ( strcmp( yytext, samePointer->symName ) == 0 )

				break /* out of this for loop */;

		if ( samePointer != (SYM *) NULL )

			break /* out of this while loop */;
	}

	if ( samePointer != (SYM *) NULL ) {

		symPointer->symName = samePointer->symName;
		symPointer->symToken = samePointer->symToken;
		symPointer->symSameName = samePointer;
		symPointer->symDeclSpec = samePointer->symDeclSpec;
		symPointer->symDeclarator = samePointer->symDeclarator;

		/*
		 * NOTE that if samePointer->symToken == TYPEDEF_NAME then
		 * the same will be true of symPointer.  That is, we're
		 * disallowing redeclaration of typedef names as objects
		 * in an inner scope.  This is a very technical point in
		 * the language, but this use ought to be allowed, provided
		 * that this is in a declaration where at least one type
		 * specifier has been explicitly given.  This is tricky
		 * to implement, to say the least.  Since it's such an
		 * obscure feature of the language, we can let it pass
		 * for now.  Many commercial C compilers do not even
		 * implement this feature of the language.
		 */

	} else {

		symPointer->symName =
				(char *) memAlloc( (unsigned) yyleng + 1 );

		(void) strcpy( symPointer->symName, yytext );
		symPointer->symToken = IDENTIFIER;
		symPointer->symSameName = (SYM *) NULL;
	}

	return symPointer;
}


/*
 * freeToken:
 *	This function releases storage allocated for a TOKEN structure.
 *	This is commonly done for language features not being parsed.
 */

void
freeToken( tokenPointer )

TOKEN  *tokenPointer;
{
	if ( tokenPointer == (TOKEN *) NULL )
		return;

	memFree( (void *) tokenPointer );

	return;
}


/*
 * memAlloc:
 *	Allocate a sequence of bytes.
 */

void *
memAlloc( numBytes )

unsigned int	numBytes;
{
	extern void    *malloc();
	void	       *address;

	address = malloc( numBytes );

	if ( address == (void *) NULL ) {
#ifdef STANDALONE
		fprintf( stderr, "%s: memory allocation failure\n",
			programName );
#endif /* STANDALONE */
		longjmp( fatalError, 1 );
	}

	return address;
}


/*
 * memFree:
 *	Release a sequence of bytes previously allocated.
 */

void
memFree( address )

void   *address;
{
	extern void	free();

	if ( address != (void *) NULL )

		free( address );

	return;
}
