/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *          (R)
 * The Pablo    Performance Analysis Environment software is NOT in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1991-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * PABLO is a registered trademark of
 * The Board of Trustees of the University of Illinois
 * registered in the U.S. Patent and Trademark Office.
 *
 * Author:  Roger J. Noe (noe@cs.uiuc.edu)
 * Contributing Author:  Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004,
 * by a grant from the Digital Equipment Corporation External Research
 * Program, and by a collaborative research agreement with the Intel
 * Supercomputer Systems Division.
 *
 */

/*
 * Parser.h:
 *	This is the primary include file for the Pablo instrumenting
 *	C parser.  Contained within are definitions of data structure
 *	types, declarations of global data structures and declarations
 *	of functions used by the lexical analyzer, the parser itself
 *	(yyparse) and the supporting functions found in Parser.c.
 */

#include <setjmp.h>

#ifndef FALSE
# define		FALSE		0
# define		TRUE		1
#endif /* FALSE */


/*	DATA STRUCTURE TYPE DEFINITIONS SECTION				    */


/*
 * The enumeration type (typedef NODETYPE) for grammar nonterminals
 * which, when reduced, create parse tree nodes.
 */

typedef enum {
	PrimaryExpr,
	PostfixExpr,
	Declaration,
	InitDeclarator,
	StorClassSpec,
	TypeSpecifier,
	TypeQualifier,
	StructOrUnion,
	StructDeclaration,
	StructDeclarator,
	EnumSpecifier,
	Enumerator,
	DirectDeclarator,
	Pointer,
	IdentifierList,
	ParameterTypeList,
	ParamDeclaration,
	DirAbstrDeclarator,
	CompoundStatement,
	ExpressionStatement,
	JumpStatement,
	FunctionDefinition
} NODETYPE;


/*
 * The enumeration type (typedef INSTTYPES) for degree of instrumentation
 * to be performed on instrumentable constructs.
 */

typedef enum {

	InstrumentOff,			/* No instrumentation		    */

	InstrumentCount,		/* Instrument with count events	    */

	InstrumentTrace			/* Instrument with full traces	    */

} INSTTYPES;


/*
 * The enumeration type (typedef CONSTRUCTS) for different kinds of
 * instrumentable constructs.  Only the first two are ever seen outside
 * of the parser code itself (i.e. the last two are internal).
 */

typedef enum {

	FunctionCall,			/* A function invocation	    */

	OuterLoop,			/* A non-nested loop statement	    */

	FunctionBody,			/* For inserting declarations	    */

	SymbolName			/* For changing symbol names	    */

} CONSTRUCTS;

/*
 * The struct node (typedef NODE) data structure defines the contents
 * of a parse tree node.
 */

typedef struct node {

	NODETYPE	nodeType;	/* Nonterminal of creation	    */

	int		nodeToken;	/* Relevant token value		    */

	struct node    *nodeNext;	/* Down in parse tree		    */

	struct node    *nodeList;	/* Across parse tree		    */

	struct node    *nodeEndList;	/* Left recursive list building	    */

	struct node    *nodeDeclList;	/* List of subdeclarations	    */

	struct token   *nodeIdentifier;	/* Identifier token		    */

	unsigned long	nodeByteStart,	/* Byte offset in cppOutputFile	    */
			nodeByteEnd;	/* where node begins and ends	    */

	unsigned int	nodeLineStart,	/* Line number in input file	    */
			nodeLineEnd;	/* where node begins and ends	    */

	char	       *nodeFileStart,	/* Name of input file where	    */
		       *nodeFileEnd;	/* node begins and ends		    */

} NODE;


/*
 * The struct symb (typedef SYM) data structure describes a symbol
 * that was parsed.
 */

typedef struct symb {

	char	       *symName;	/* Symbol name string		    */

	int		symToken;	/* IDENTIFIER or TYPEDEF_NAME	    */

	struct node    *symDeclSpec;	/* Declaration specifiers	    */

	struct node    *symDeclarator;	/* Declarator			    */

	struct symb    *symSameName;	/* Same name symbol, outer scope    */

	struct symb    *symNext;	/* Next symbol in this scope	    */

} SYM;


/*
 * The struct token (typedef TOKEN) data structure describes a token
 * that was parsed.
 */

typedef struct token {

	int		tokenType;	/* Returned by lexical analyzer	    */

	char	       *tokenInputFile;	/* Name of input file		    */

	unsigned int	tokenLine;	/* Line number of input file	    */

	unsigned long	tokenByteStart,	/* Byte offset in cppOutputFile	    */
			tokenByteEnd;	/* where token begins and ends	    */

	struct symb    *tokenSymbol;	/* If token is a symbol		    */

} TOKEN;


/*
 * The struct contextTree (typedef CONTEXT) data structure defines the
 * contents of an element of the parse context tree.
 */

typedef struct contextTree {

	struct node    *cntxtNodeHead;	/* Head of node list		    */

	struct symb    *cntxtSymHead;	/* Head of symbol list		    */

	struct contextTree		/* Parent context pointer	    */
		       *cntxtParent;

	struct contextTree		/* Older sibling context pointer    */
		       *cntxtOlderSibling;

	struct contextTree		/* Younger sibling context pointer  */
		       *cntxtYoungerSibling;

	struct contextTree		/* Oldest child context pointer	    */
		       *cntxtOldestChild;

	struct contextTree		/* Youngest child context pointer   */
		       *cntxtYoungestChild;

} CONTEXT;


/*
 * The struct procedure (typedef PROC) data structure contains the
 * name of a function and the pair of event IDs (entry and exit) assigned
 * to it.  These are maintained in a linked list and apply across multiple
 * files instrumented in a single session.
 */

typedef struct procedure {

	char	       *procName;	/* Name of the function		    */

	int		procEventEntry,	/* Event IDs for entry to and	    */
			procEventExit;	/* exit from the function	    */

	int		procDeclared;	/* Boolean for return value	    */
					/* declaration in current function  */

	int		procIsVoid;	/* Boolean for void type	    */

	struct procedure		/* Linked list pointer		    */
		       *procNext;

} PROC;


/*
 * The struct iteration (typedef ITER) data structure contains the pair of
 * event IDs (entry and exit) assigned to a loop.  These are maintained in
 * a linked list and apply to a single parsed file.
 */

typedef struct iteration {

	int		iterEventEntry,	/* Event IDs for entry to and	    */
			iterEventExit;	/* exit from the loop		    */

	struct iteration		/* Linked list pointer		    */
		       *iterNext;

} ITER;


/*
 * The struct traceTree (typedef TRACETREE) data structure defines the
 * contents of an element of the tree of instrumentable constructs.
 */

typedef struct traceTree {

	struct symb			/* Invoking function symbol	    */
		       *traceInvokingSym;

	struct symb			/* Invoked function symbol	    */
		       *traceInvokedSym;

	unsigned long	traceByteStart,	/* Byte offset in cppOutputFile	    */
			traceByteEnd;	/* where construct begins and ends  */

	unsigned int	traceLineStart,	/* Line number in input file where  */
			traceLineEnd;	/* construct begins and ends	    */

	char	       *traceFileStart,	/* Name of input file where	    */
		       *traceFileEnd;	/* construct begins and ends	    */

	CONSTRUCTS	traceConstruct;	/* Instrumentable construct type    */

	INSTTYPES	traceType;	/* Degree of instrumentation	    */

	PROC	       *traceProc;	/* Specific to function calls	    */

	ITER	       *traceIter;	/* Specific to loops		    */

	char	       *traceTranslate;	/* Name from translation table	    */

	struct traceTree		/* Parent pointer		    */
		       *traceParent;

	struct traceTree		/* Left sibling pointer		    */
		       *traceLeft;

	struct traceTree		/* Right sibling pointer	    */
		       *traceRight;

	struct traceTree		/* First child pointer		    */
		       *traceChild;

} TRACETREE;


/*
 * The struct inputFile (typedef IFILE) data structure contains the name
 * of an input file, either the original input file passed in through
 * the top-level interface, or the name of a file included by cpp.  These
 * file names are maintained in a linked list.
 */

typedef struct inputFile {

	char	       *inputFileName;	/* Name of input file		    */

	struct inputFile		/* Linked list pointer		    */
		       *inputFileNext;

} IFILE;


/*
 * The struct translate (typedef TRANSLATE) data structure contains a
 * pair of character strings.  An array of these structures is compiled
 * into the parser.  When an input file is parsed and a function name
 * is recognized, that token will be compared against the first member
 * of each element of this translation table.  If a match is found, then
 * upon instrumenting the file every instance of that function name is
 * replaced with its pair.
 */

typedef struct translate {

	char	       *oldName;	/* Symbol name to compare with	    */

	char	       *newName;	/* Replacement symbol name	    */

} TRANSLATE;


/*
 * The struct instRec (typedef INSTREC) data structure defines the elements
 * of the doubly linked list shared between the parser code and the user
 * interface.
 */

typedef struct instRec {

	char	       *invokingFunc;	/* Invoking function name	    */

	char	       *invokedFunc;	/* Invoked function name	    */

	unsigned int	beginLine,	/* Line number in input file where  */
			endLine;	/* construct begins and ends.	    */

	CONSTRUCTS	constructType;	/* Type of instrumentable	    */

	INSTTYPES	instrumentType;	/* Type of instrumentation	    */

	struct instRec *previous,	/* Linked list pointers,	    */
		       *next;		/* forward and backward.	    */

	TRACETREE      *treeElement;	/* Pointer into trace tree	    */

} INSTREC;


/*
 * The struct argFile (typedef ARGFILE) data structure contains the
 * name of an input file passed as an argument into parseFile and a
 * pointer to the list of instrumentable iteration statements that
 * were found within it.  This ARGFILE list, the ITER lists pointed
 * to by individual ARGFILE structures, and the PROC list are the
 * only data structures maintained across invocations of parseFile
 * for different argument files.  They are used to generate the
 * instrumentation initialization function.
 */

typedef struct argFile {

	char	       *argFileName;	/* Argument file name		    */

	ITER	       *argIterList;	/* List of loops in file	    */

	struct argFile *argNext;	/* Linked list pointer		    */

} ARGFILE;


/*	GLOBAL DATA STRUCTURE DECLARATIONS SECTION			    */


#ifdef STANDALONE
extern char	       *programName;		/* Name of this program	    */
#endif /* STANDALONE */

extern SYM	       *currentFunc;		/* Current function pointer */

extern CONTEXT	       *currentContext;		/* Current context	    */

extern TRACETREE       *currentTrace;		/* Trace tree pointer	    */

extern IFILE	       *inputFileList;		/* List of input files	    */

extern char	       *currentInputFile;	/* Name of current file	    */

extern char		cppOutputFile[];	/* CPP output file	    */

extern unsigned long	inputOffset;		/* cppOutputFile offset	    */

extern unsigned int	inputLineNumber;	/* Input file line number   */

extern INSTREC	       *instrumentHead,		/* Head and tail pointers   */
		       *instrumentTail;		/* for instrumentable list  */

extern FILE	       *outputFile;		/* Instrumented code	    */

extern int		parserEventID;		/* Event ID counter	    */

extern PROC	       *procList;		/* List of functions	    */

extern ITER	       *iterList;		/* List of loops	    */

extern jmp_buf		fatalError;		/* Fatal error recovery	    */

extern TRANSLATE	translateTable[];	/* Symbol translation table */

extern ARGFILE	       *argFileList;		/* List of argument files   */


/* The following declarations are for lex/yacc-produced data structures	    */
/* which are not declared in y.tab.h:					    */

extern FILE	       *yyin;			/* Input stdio stream	    */

extern int		yyparse();		/* Parser function	    */

extern int		yyinput();		/* Character input	    */

extern int		yyunput();		/* Character unput	    */

extern char		yytext[];		/* Token string		    */

extern int		yyleng;			/* Token length		    */


/*	FUNCTION DECLARATIONS SECTION					    */

extern int	parseFile(),
		preprocessFile(),
		generateInstrumentedSourceFile(),
		generateInstrumentationInitialization(),
		writeEventIDs(),
		readEventIDs(),
		returnsVoid(),
		makeToken(),
		makeSymToken();

extern void	cleanupParser(),
		generateInstrumentables(),
		instrumentCode(),
		copyUpTo(),
		copyBlock(),
		declareReturnValues(),
		duplicateDeclSpec(),
		duplicateDeclarator(),
		freeIterList(),
		resetParser(),
		freeContextTree(),
		freeTraceTree(),
		contextBegin(),
		contextEnd(),
		freeNodes(),
		freeSyms(),
		setStartNode(),
		setEndNode(),
		setStartToken(),
		setEndToken(),
		recordFunctionCall(),
		recordLoop(),
		recordFunctionDefinition(),
		checkSymbolName(),
		addTraceTree(),
		makeFunctionDefinition(),
		scanCPPline(),
		scanComment(),
		makeInputFile(),
		freeToken(),
		memFree();

extern void    *memAlloc();

extern SYM     *makeSymbol();

extern NODE    *makeNode(),
	       *findToken(),
	       *makePrimaryExpr(),
	       *makePostfixExpr(),
	       *makeDeclaration(),
	       *makeInitDeclarator(),
	       *makeStorClassSpec(),
	       *makeTypeSpecifier(),
	       *makeTypeQualifier(),
	       *makeStructOrUnion(),
	       *makeStructDeclaration(),
	       *makeStructDeclarator(),
	       *makeEnumSpecifier(),
	       *makeEnumerator(),
	       *makeDirectDeclarator(),
	       *makePointer(),
	       *makeIdentifierList(),
	       *makeParameterTypeList(),
	       *makeParamDeclaration(),
	       *makeDirAbstrDeclarator(),
	       *makeCompoundStatement(),
	       *makeExpressionStatement(),
	       *makeJumpStatement();

extern PROC    *findProc();
