/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *          (R)
 * The Pablo    Performance Analysis Environment software is NOT in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1991-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * PABLO is a registered trademark of
 * The Board of Trustees of the University of Illinois
 * registered in the U.S. Patent and Trademark Office.
 *
 * Author:  Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Author:  Xingbin Zhang (zhang@cs.uiuc.edu)
 * Contributing Author:  Roger J. Noe (noe@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004,
 * by a grant from the Digital Equipment Corporation External Research
 * Program, and by a collaborative research agreement with the Intel
 * Supercomputer Systems Division.
 *
 */

/*
 * pdeHost.c:
 *	This file contains the host code for an iterative elliptic
 *	PDE solver, implemented for the Intel iPSC/2 and iPSC/860.
 *	See pdeNode.c for the real guts of this application.
 */

#include <math.h>
#include <stdio.h>

#ifdef __PARAGON__
# include <nx.h>
#endif

#define	TRUE	1
#define	FALSE	0
#define	EPSILON 1e-5

#define MAXITERmsg	0	/* Message type: max number of iterations    */
#define CONVITERmsg	1	/* Message type: convergence check frequency */
#define ROWSmsg		2	/* Message type: grid dimension		     */
#define WIDTHmsg	3	/* Message type: partition width	     */
#define STRIPmsg	4	/* Message type: boundary row		     */

#define ALL -1			/* All nodes in the hypercube 		     */
#define NODEPID 0		/* Node pid                                  */
#define HOSTPID 1		/* Host pid                                  */

FILE	*ofile;

int	Rows;			/* number of rows in grid		     */
float	Range;			/* extent in X and Y			     */
int	ConvIter;		/* iterations between checks		     */
int	MaxIter;		/* iteration count for benchmark	     */

float	**OldPartition;		/* partitions for SOR			     */
float	**NewPartition;		/* iterations				     */

extern long	gray();

#ifndef __PARAGON__
extern long 	numnodes();
extern void	setpid();
#endif

main (argc, argv)
int argc;
char **argv;
{

	int	stripWidth;
	int	nextRow;
	int	Iter;

	long	i;
	long	nNodes;
	long	nLongStrips;

#ifdef __PARAGON__
	long	*pid_list;
#endif

	void InitGrid();
	void PrintGrid();

#ifdef __PARAGON__
	nx_initve("", 0, "", &argc, argv );
	setptype( HOSTPID );
	pid_list = (long *) malloc(sizeof(long) * numnodes());
#else
	setpid(HOSTPID);
#endif

	InitGrid ();
	PrintGrid(0);
	fflush(stdout);

#ifdef __PARAGON__
	nx_load((long *) 0, ALL, NODEPID, pid_list, "pdeNode" );
#else
	load("pdeNode", ALL, NODEPID);		/* load node program to all */
#endif

	/*
	 *
	 * Send dimensions of the grid, max iterations, and freqency for
	 * convergence check.
	 *
	 */

	csend( ROWSmsg, &Rows, sizeof(int), ALL, NODEPID );
	csend( MAXITERmsg, &MaxIter, sizeof(int), ALL, NODEPID );
	csend( CONVITERmsg, &ConvIter, sizeof(int), ALL, NODEPID );

	/*
	 *
	 * Distribute the grid into row strips and send the strips to all
	 * nodes.  The strips are distributed according to gray embedding
	 * so that logically adjacent strips are also physically adjacent
	 * and communication of boundary points is always one hop in the
	 * hypercube.
	 *
	 */

	nNodes = numnodes();
	nLongStrips = Rows % nNodes;
	nextRow = 0;

	for(i = 0; i < nNodes; i++) {
	   stripWidth = (i < nLongStrips) ? Rows/nNodes+1 : Rows / nNodes;

	   csend( WIDTHmsg, &stripWidth, sizeof(int), gray(i), NODEPID );

	/*
	 *
	 * Send top and bottom rows to node 0 and gray(nNodes - 1)
	 *
	 */

	   if (i == 0 || i == nNodes - 1)
	      stripWidth++;

	   csend( STRIPmsg, OldPartition[nextRow],
		stripWidth * (Rows + 2) * sizeof(float), gray(i), NODEPID );

	   nextRow += stripWidth;
	}

	/*
	 *
	 * Receive the actual number of iterations from node 0
	 *
	 */

	crecv( nNodes + 1, &Iter, sizeof(int) );
	nextRow = 1;				/* skip the boundary values */

	/*
	 *
	 * Receive computed grid. The node numbers are used as message
	 * types to order the strips
	 *
	 */

	for (i = 0; i < nNodes; i++) {
	   stripWidth = (i < nLongStrips) ? Rows/nNodes+1 : Rows / nNodes;

	   crecv( gray(i), NewPartition[nextRow],
		stripWidth * (Rows + 2) * sizeof(float) );

	   nextRow += stripWidth;
	}

#ifdef __PARAGON__
	nx_waitall();
#else
	waitall( ALL, NODEPID );		/* Wait for nodes to exit */
	killcube(ALL, ALL);
#endif

	OldPartition = NewPartition;
	PrintGrid (Iter);

	printf("Execution complete\n");
}

/************************************************************************/
/*									*/
/*	Procedure:	InitGrid					*/
/*	Function:	Initializes parameters for PDE solver		*/
/*	Author:		Daniel A. Reed					*/
/*	Last Change:	11/9/87						*/
/*									*/
/*	Parameters:	None (globals modified)				*/
/*									*/
/************************************************************************/

void InitGrid() {
	int	i, j;

	if ((ofile = fopen ("Jacobi.out", "w")) == NULL) {
	   fprintf (stderr, "\nCan't open output file: Jacobi.out\n");
	   exit (1);
	}

	printf("Number of Rows: ");
	scanf("%d", &Rows);
	printf("Extent in X and Y: ");
	scanf("%f", &Range);
	printf("Iterations between convergence checks: ");
        scanf("%d", &ConvIter);
	printf("Maximum number of iterations: ");
        scanf("%d", &MaxIter);
/*
Rows = 16;
Range = 1.0;
ConvIter = 1;
MaxIter = 5000;
*/

	fprintf (ofile, "Rows: %d\n", Rows);
	fprintf (ofile, "Extent in X and Y: %f\n", Range);
	fprintf (ofile, "Iterations between convergence checks: %d\n", ConvIter);
	fprintf (ofile, "MaxIter: %d\n\n", MaxIter);
	fflush (ofile);

	OldPartition = (float **) malloc ((Rows + 2) * sizeof (float *));
	NewPartition = (float **) malloc ((Rows + 2) * sizeof (float *));

	OldPartition [0] = (float *)
		malloc ((Rows + 2) * (Rows + 2) * sizeof (float));
	NewPartition [0] = (float *)
		malloc ((Rows + 2) * (Rows + 2) * sizeof (float));

	for (i = 1; i <= Rows + 1; i++) {
	   OldPartition [i] = OldPartition [0] + i * (Rows + 2);
	   NewPartition [i] = NewPartition [0] + i * (Rows + 2);
	}

	for (i = 0; i <= Rows + 1; i++) {
	   for (j = 0; j <= Rows + 1; j++) {
	      OldPartition [i][j] = 0.0;
	      NewPartition [i][j] = 0.0;
	   }
	}

	FillEdges ();

}


/************************************************************************/
/*									*/
/*	Procedure:	FillEdges					*/
/*	Function:	Computes grid boundaries			*/
/*	Author:		Daniel A. Reed					*/
/*	Last Change:	1/7/88						*/
/*									*/
/*	Parameters:	None						*/
/*									*/
/************************************************************************/

FillEdges ()

{
	int	i, j;
	float	h;			/* mesh spacing 		*/
	float	Boundary ();

	h = Range / (float) (Rows + 1);

	for (i = 0; i <= Rows + 1; i++) { 
	   OldPartition [0][i] = Boundary (i * h, 0.0);
	   NewPartition [0][i] = Boundary (i * h, 0.0);
	}

	for (i = 0; i <= Rows + 1; i++) {
	   OldPartition [Rows + 1][i] = Boundary (i * h, Range);
	   NewPartition [Rows + 1][i] = Boundary (i * h, Range);
	}

	for (i = 0; i <= Rows+1; i++) {
	   OldPartition [i][0] = Boundary (0.0, i * h);
	   NewPartition [i][0] = Boundary (0.0, i * h);
	   OldPartition [i][Rows + 1] = Boundary (Range, i * h);
	   NewPartition [i][Rows + 1] = Boundary (Range, i * h);
	}
}


/************************************************************************/
/*									*/
/*	Function:	Boundary					*/
/*	Function:	Calculates boundary grid points			*/
/*	Author:		Daniel A. Reed					*/
/*	Last Change:	1/7/88						*/
/*									*/
/*	Parameters:	x and y grid values				*/
/*									*/
/************************************************************************/

float	Boundary (x, y)

	float	x, y;
{
	double	exp();
	float	z;

	z = x * exp (y);

	return (z);
}

/************************************************************************/
/*									*/
/*	Procedure:	PrintGrid					*/
/*	Function:	Prints current grid				*/
/*	Author:		Daniel A. Reed					*/
/*	Last Change:	11/9/87						*/
/*									*/
/*	Parameters:	Iter - current iteration number			*/
/*									*/
/************************************************************************/

void PrintGrid (Iter)

	int	Iter;
{

	int	i, j;

	   fprintf (ofile, "--------------------------- Grid %d --------------------------\n", Iter);
	   for (i = 0; i <= Rows + 1; i++) {
	      for (j = 0; j <= Rows + 1; j++) {
	         fprintf (ofile, "%7.4f ", OldPartition [i][j]);
	         if ((j + 1) % 8 == 0)
		    fprintf (ofile, "\n");
	      }
	      fprintf (ofile, "\n");
	   }
	   fprintf (ofile, "\n\n");
}
