/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *          (R)
 * The Pablo    Performance Analysis Environment software is NOT in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1991-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * PABLO is a registered trademark of
 * The Board of Trustees of the University of Illinois
 * registered in the U.S. Patent and Trademark Office.
 *
 * Author:  Tara M. Madhyastha (tara@cs.uiuc.edu)
 * Contributing Author:  Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004,
 * by a grant from the Digital Equipment Corporation External Research
 * Program, and by a collaborative research agreement with the Intel
 * Supercomputer Systems Division.
 *
 */

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <values.h>

#include "Assert.h"
#include "SystemDepend.h"
#include "SDDFparam.h"
#include "TraceParam.h"
#include "Trace.h"
#include "IOTrace.h"

/* The SWAVE_IO flag should be turned on to generate SWAVE records for
   use with the VR system. For general use, it should be undefined.
   This of course assumes that the SWAVE module is compiled with the 
   extension library.
*/

#ifdef SWAVE_IO

extern int produceOwnIOOutput;
int bSeekDim=-1;
int bReadDim=-1;

#endif


CLOCK	lastReadBegin = NOSUCHTIME;
CLOCK	lastWriteBegin = NOSUCHTIME;
CLOCK	lastSeekBegin = NOSUCHTIME;
CLOCK	lastOpenBegin = NOSUCHTIME;
CLOCK	lastCloseBegin = NOSUCHTIME;

char *histogramRecordBuffer;     /* big enough for largest histogram */
int  maxHistogramRecordSize;     /* size of largest histogram */


/* IOTraceInit: Extension interface function to initialize I/O
*  tracing. Must be called from application if you expect stuff to
*  work. */

IOTraceInit()
{
  initIOTrace();
}

/* Fortran interface function for IOTraceInit */
iotraceinit_() {   IOTraceInit(); }



/*
 *	preInitIOTrace: 
 */

preinitiotrace_()
{
	preInitIOTrace();
}

preInitIOTrace()
{
	extern int	writeIORecordDescriptors();
	static int	preInitDone = FALSE;

	if ( preInitDone == TRUE )
		return;

	/* Do initialization of extension-specific data structures */

	histogramRecordBuffer = NULL;
	maxHistogramRecordSize = -1;


	/* Give the instrumentation library a pointer to the function	    */
	/* in which we output the specialized record descriptors for	    */
	/* I/O calls. This is system dependent and is in the system
           dependent part of the code */



	setRecordDescriptor( writeIORecordDescriptors );

	preInitDone = TRUE;

	return;
}

/* Helper macro */

#define mallocBlockIfNecessary(fileNumber)   if (blockSDT[fileNumber] == NULL) {\
      blockSDT[fileNumber] = \
	(struct blockSummaryData *) malloc(sizeof(struct blockSummaryData));\
      zeroData(blockSDT[fileNumber]);   }


/* Generate records for read events. Should be self explanatory */

TR_RECORD *
readEventRecord( recordType, eventPointer, timeStamp,
			  dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	static TR_RECORD			traceRecord;
	static struct readRecordData	readRecordHeader;

	/* This structure defines how the send arguments are passed in	    */
	/* as user data, pointed to by dataPointer.			    */

	struct readArgs {
		long	filepointer;
		long	numVariables;
		long	cause;
	}					*readArguments;


	readArguments = (struct readArgs *) dataPointer;

	/* The time stamp stored in the event descriptor will be used	    */
	/* unless one is specified in the timeStamp parameter.		    */

	if ( clockCompare( timeStamp, noSuchClock ) == 0 )
		timeStamp = eventPointer->eventLast;

	if (isReadEndEvent(eventPointer->eventID)) {

	  readRecordHeader.duration = clockToSeconds(
			     clockSubtract(timeStamp,lastReadBegin));
	  if (*((int *) dataPointer) >= 0) {
	    readRecordHeader.numBytes = *((int *) dataPointer);
	  }  else {
	    readRecordHeader.numBytes = 0;
	  }
	    
	  lastReadBegin = noSuchClock;
#ifdef SWAVE_IO
	  SwaveRegisterValue( bReadDim,
			     readRecordHeader.seconds,
			     readRecordHeader.duration );
#endif
	  returnRecord( &traceRecord);

	} else if (isReadBeginEvent(eventPointer->eventID)) {


	  traceRecord.recordLength = sizeof readRecordHeader;
	  
	  traceRecord.recordContents = (char *) &readRecordHeader;

	  /* Load the trace record fields into the allocated buffer	    */
	  readRecordHeader.packetLength = traceRecord.recordLength;
	  readRecordHeader.packetType = PKT_DATA;
	  readRecordHeader.packetTag = FAMILY_READ | recordType;
	  readRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
	  readRecordHeader.timeStamp = timeStamp;
	  readRecordHeader.seconds = clockToSeconds( timeStamp );
	  readRecordHeader.eventID = eventPointer->eventID;
	  readRecordHeader.nodeID = TRgetNode();
	  readRecordHeader.filepointer = readArguments->filepointer;
	  readRecordHeader.numVariables = readArguments->numVariables;
	  readRecordHeader.cause = readArguments->cause;
	  lastReadBegin = timeStamp;

	  return(nullRecordFunction(recordType,eventPointer,timeStamp,
				    dataPointer,dataLength));
	} else {
	  /* error */
	  fprintf(stderr, "readEventRecord: unknown eventID %d\n", 
	    eventPointer->eventID);
	}
	return(nullRecordFunction(recordType,eventPointer,timeStamp,
				  dataPointer,dataLength));
 }


/* Generate records for open events. */

TR_RECORD *
openEventRecord( recordType, eventPointer, timeStamp,
			  dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
  static TR_RECORD			traceRecord;
  static struct openRecordData	openRecordHeader;

  /* This structure defines how the send arguments are passed in	    */
  /* as user data, pointed to by dataPointer.			    */

  struct openArgs {
    int flags;
    int mode;
    char filename[256];
  }					*openArguments;

  char filename[256], *curr;
  int x;

  if ( clockCompare( timeStamp, noSuchClock ) == 0 )
    timeStamp = eventPointer->eventLast;

  if (isOpenEndEvent(eventPointer->eventID)) {
    openRecordHeader.duration = clockToSeconds(
	       clockSubtract(timeStamp,lastOpenBegin));
    lastOpenBegin = noSuchClock;
    openRecordHeader.fileID = *((int *) dataPointer);
    openRecordHeader.packetLength = traceRecord.recordLength;

    return & traceRecord;

  } else if (isOpenBeginEvent(eventPointer->eventID)) {
    openArguments = (struct openArgs *) dataPointer;
    strcpy(filename, openArguments->filename);

    lastOpenBegin = timeStamp;

    traceRecord.recordLength = sizeof openRecordHeader -
      sizeof(openRecordHeader.userData);


    traceRecord.recordContents = (char *) &openRecordHeader;

    /* Load the trace record fields into the allocated buffer */


    openRecordHeader.packetType = PKT_DATA;
    openRecordHeader.packetTag = FAMILY_OPEN | recordType;
    openRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
    openRecordHeader.timeStamp = timeStamp;
    openRecordHeader.seconds = clockToSeconds( timeStamp );
    openRecordHeader.eventID = eventPointer->eventID;
    openRecordHeader.nodeID = TRgetNode();
    openRecordHeader.flags = openArguments->flags;
    openRecordHeader.mode = openArguments->mode;

    /* copy the filename, pad out to an 8 byte boundary in a portable way */

    curr = openRecordHeader.userData;
    x = strlen(filename) + 1;
    x = (x+7)&(~7);
    bcopy(&x, curr, sizeof(int));
    curr+=(sizeof(int));
    strcpy(curr, filename);
    traceRecord.recordLength += (x + 4) ;
  } else {
    ;
    /* error -- can be elaborated here */
  }

  return(nullRecordFunction(recordType,eventPointer,timeStamp,
			    dataPointer,dataLength));	
}



/* Generate event records for activity measured over a file lifetime.
 * This is called in lieu of open/close event record functions when
 * the doFileSummaries call is invoked. See that call for more
 * information.
 */


TR_RECORD * extFileLifetimeEventRecordFunction( recordType, eventPointer, timeStamp, dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
        TR_RECORD	*traceRecord;
	struct fileLifetimeRecordData *fileLifeData;

	/* This structure defines how the send arguments are passed in	    */
	/* as user data, pointed to by dataPointer.			    */

	struct openArgs {
	  int flags;
	  int mode;
	  char data[256];
	}   *openArguments;

	static char filename[256];
	static int mode;
	char *curr;
	int x;
	static int overhead;
	static int fileid;

	if ( clockCompare( timeStamp, noSuchClock ) == 0 )
		timeStamp = eventPointer->eventLast;

	if (isOpenBeginEvent( eventPointer->eventID )) {
	  openArguments = (struct openArgs *) dataPointer;
	  strcpy(filename, openArguments->data);
	  mode =  openArguments->mode;
	  lastOpenBegin = timeStamp;
	}

	/* if it's an open end event, make sure you record the
	   arguments before moving on */

	if (isOpenEndEvent(eventPointer->eventID)) {
	  fileid = *((int *)dataPointer) ;
	  if (fileLifetime[fileid] == NULL) {
	    fileLifetime[fileid] = (TR_RECORD *) malloc(sizeof (TR_RECORD));
	    fileLifetime[fileid]->recordContents = 
	      (char *) malloc(sizeof(struct fileLifetimeRecordData));
	  } else { /* already allocated, zero*/
	    fileLifeData = (struct fileLifetimeRecordData *) fileLifetime[fileid]->recordContents;
	    bzero(fileLifetime[fileid]->recordContents,
		  sizeof(struct fileLifetimeRecordData));
	    bzero(fileLifetime[fileid], sizeof(TR_RECORD));
	    fileLifetime[fileid]->recordContents = (char *) fileLifeData;
	  }
	      
	  traceRecord = fileLifetime[fileid];
	  fileLifeData = (struct fileLifetimeRecordData *)
	                     fileLifetime[fileid]->recordContents;

	  traceRecord->recordLength = sizeof (struct fileLifetimeRecordData) - sizeof(fileLifeData->userData);
	  fileLifeData->overhead = 
	    clockToSeconds(clockSubtract(timeStamp, lastOpenBegin));
	  fileLifeData->clockDimension = sizeof(CLOCK)/sizeof(int);
	  fileLifeData->timeStamp = timeStamp;
	  fileLifeData->seconds = clockToSeconds( timeStamp );
	  fileLifeData->mode = mode;
	  curr = fileLifeData->userData;
	  x = strlen(filename) + 1;
	  x = (x+7)&(~7);
	  bcopy(&x, curr, sizeof(int));
	  curr+=(sizeof(int));
	  strcpy(curr, filename);
	  traceRecord->recordLength += (x + 4) ;
	}
	if (isCloseBeginEvent(eventPointer->eventID)) {
	  fileid = *((int *)dataPointer) ;
	  lastCloseBegin = timeStamp;
	}

	if (isCloseEndEvent(eventPointer->eventID)) {
	  if (fileLifetime[fileid] == NULL) {
	    fileLifetime[fileid] = (TR_RECORD *) malloc(sizeof (TR_RECORD));
	    fileLifetime[fileid]->recordContents = 
	      malloc(sizeof(struct fileLifetimeRecordData));
	  }
	  traceRecord = fileLifetime[fileid];
	  fileLifeData = (struct fileLifetimeRecordData *) fileLifetime[fileid]->recordContents;
	  fileLifeData->overhead += 
	    clockToSeconds(clockSubtract(timeStamp,lastCloseBegin));
	
	  fileLifeData->packetLength = traceRecord->recordLength;
	  fileLifeData->packetType = PKT_DATA;
	  fileLifeData->packetTag = FAMILY_LIFETIME | recordType;
	  fileLifeData->clockDimension = sizeof(CLOCK)/sizeof(int);
	  fileLifeData->eventID = fileLifeID;
	  fileLifeData->fileID = fileid;

	  if (blockSDT[fileid] != NULL) {
	    fileLifeData->readFrequency = blockSDT[fileid]->readData.frequency;
	    fileLifeData->readBytes = blockSDT[fileid]->readData.numBytes;
	    fileLifeData->readTime =  blockSDT[fileid]->readData.totalTime;
	    fileLifeData->writeFrequency = blockSDT[fileid]->writeData.frequency;
	    fileLifeData->writeBytes =  blockSDT[fileid]->writeData.numBytes;
	    fileLifeData->writeTime =  blockSDT[fileid]->writeData.totalTime;
	    fileLifeData->seekFrequency = blockSDT[fileid]->seekData.frequency;
	    fileLifeData->seekBytes = blockSDT[fileid]->seekData.numBytes;
	    fileLifeData->seekTime =  blockSDT[fileid]->seekData.totalTime;
	    zeroData(blockSDT[fileid]);
	  } else {
	    fileLifeData->readFrequency = 0;
	    fileLifeData->readBytes = 0;
	    fileLifeData->writeFrequency = 0;
	    fileLifeData->writeBytes =  0;
	    fileLifeData->seekFrequency = 0;
	    fileLifeData->seekBytes = 0;
	  }

	  fileLifeData->fileLifetime = clockToSeconds(lastCloseBegin) 
	    - fileLifeData->seconds;

	return  traceRecord;
	}
	/* if we haven't returned a real record, return a null record */
	return(nullRecordFunction(recordType,eventPointer,timeStamp,
				  dataPointer,dataLength));
}


/* Generate write event records  */

TR_RECORD *
writeEventRecord( recordType, eventPointer, timeStamp,
		     dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
  static TR_RECORD			traceRecord;
  static struct writeRecordData	   writeRecordHeader;

  /* This structure defines how the receive arguments are passed in   */
  /* as user data, pointed to by dataPointer.			    */

  struct writeArgs {
    long	filepointer;
    long    numVariables;
    long	cause;
  }					*writeArguments;

  writeArguments = (struct writeArgs *) dataPointer;

  if ( clockCompare( timeStamp, noSuchClock ) == 0 )
    timeStamp = eventPointer->eventLast;


  if (isWriteEndEvent(eventPointer->eventID)) {
	  
    writeRecordHeader.duration = clockToSeconds(
				clockSubtract(timeStamp, lastWriteBegin));
    if (*((int *) dataPointer) >= 0) {
      writeRecordHeader.numBytes = *((int *) dataPointer);
    } else {
      writeRecordHeader.numBytes = 0;
    }

    lastWriteBegin = noSuchClock;

    returnRecord( &traceRecord );

  } else if (isWriteBeginEvent(eventPointer->eventID)) {

    traceRecord.recordLength = sizeof writeRecordHeader;
    traceRecord.recordContents = (char *) &writeRecordHeader;

    writeRecordHeader.packetLength = traceRecord.recordLength;
    writeRecordHeader.packetType = PKT_DATA;
    writeRecordHeader.packetTag = FAMILY_WRITE |
      recordType;
    writeRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
    writeRecordHeader.timeStamp = timeStamp;
    writeRecordHeader.seconds = clockToSeconds( timeStamp );
    writeRecordHeader.eventID = eventPointer->eventID;
    writeRecordHeader.nodeID = TRgetNode();
    writeRecordHeader.filepointer = writeArguments->filepointer;
    writeRecordHeader.numVariables = writeArguments->numVariables;
    writeRecordHeader.cause = writeArguments->cause;
    lastWriteBegin = timeStamp;
    return(nullRecordFunction(recordType,eventPointer,timeStamp,
			      dataPointer,dataLength));
  } else {
    fprintf(stderr, "writeEventRecord: unknown eventID %d\n", 
	    eventPointer->eventID);
  }
  return(nullRecordFunction(recordType,eventPointer,timeStamp,
			    dataPointer,dataLength));
}



/* Generate seek event records */

TR_RECORD *
seekEventRecord( recordType, eventPointer, timeStamp,
		     dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	static TR_RECORD			traceRecord;
	static struct seekRecordData	seekRecordHeader;

	/* This structure defines how the receive arguments are passed in   */
	/* as user data, pointed to by dataPointer.			    */

	struct seekArgs {
		long	filepointer;
		long	offset;
		long    ptrname;
	}					*seekArguments;

	seekArguments = (struct seekArgs *) dataPointer;

	if ( clockCompare( timeStamp, noSuchClock ) == 0 )
		timeStamp = eventPointer->eventLast;

	if (isSeekEndEvent(eventPointer->eventID)) {
	  seekRecordHeader.duration = clockToSeconds(
	     clockSubtract(timeStamp, lastSeekBegin));
	  lastSeekBegin = noSuchClock;
#ifdef SWAVE_IO
	  SwaveRegisterValue( bSeekDim,
			     seekRecordHeader.seconds,
			     seekRecordHeader.duration );
#endif

	  returnRecord(&traceRecord);
	  /* NOTREACHED */
	}


	if (isSeekBeginEvent(eventPointer->eventID)) {
	  traceRecord.recordLength = sizeof seekRecordHeader;
	  traceRecord.recordContents = (char *) &seekRecordHeader;

	/* Load the trace record fields into the allocated buffer   */

	  seekRecordHeader.packetLength = traceRecord.recordLength;
	  seekRecordHeader.packetType = PKT_DATA;
	  seekRecordHeader.packetTag = FAMILY_SEEK | recordType;
	  seekRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
	  seekRecordHeader.timeStamp = timeStamp;
	  seekRecordHeader.seconds = clockToSeconds( timeStamp );
	  seekRecordHeader.eventID = eventPointer->eventID;
	  seekRecordHeader.nodeID = TRgetNode();
	  seekRecordHeader.filepointer = seekArguments->filepointer;
	  seekRecordHeader.offset = seekArguments->offset;
	  seekRecordHeader.ptrname = seekArguments->ptrname;
	  lastSeekBegin = timeStamp;
	  return(nullRecordFunction(recordType,eventPointer,timeStamp,
				    dataPointer,dataLength));
	  /* NOTREACHED */
	}
	
	/* should get here only if event ID is unrecognized */

	fprintf(stderr, "seekEventRecord: unknown eventID %d\n", 
		eventPointer->eventID);	  
	return(nullRecordFunction(recordType,eventPointer,timeStamp,
				  dataPointer,dataLength));
}


/* Generate close event records */

TR_RECORD *
closeEventRecord( recordType, eventPointer, timeStamp,
		        dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
  static TR_RECORD			traceRecord;
  static struct closeRecordData	    closeRecordHeader;


  if ( clockCompare( timeStamp, noSuchClock ) == 0 )
    timeStamp = eventPointer->eventLast;


  if (isCloseBeginEvent( eventPointer->eventID )) {
    lastCloseBegin = timeStamp;
    traceRecord.recordLength = sizeof closeRecordHeader;
    traceRecord.recordContents = (char *) &closeRecordHeader;

    closeRecordHeader.packetLength = traceRecord.recordLength;
    closeRecordHeader.packetType = PKT_DATA;
    closeRecordHeader.packetTag = FAMILY_CLOSE | recordType;
    closeRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
    closeRecordHeader.timeStamp = timeStamp;
    closeRecordHeader.seconds = clockToSeconds( timeStamp );
    closeRecordHeader.eventID = eventPointer->eventID;
    closeRecordHeader.nodeID = TRgetNode();
    closeRecordHeader.fileID = *((int *) dataPointer);
  } else if (isCloseEndEvent(eventPointer->eventID)) {
    closeRecordHeader.duration = clockToSeconds(
				clockSubtract(timeStamp,lastCloseBegin));
    return & traceRecord;
  } else {
    fprintf(stderr, "closeEventRecord: unknown eventID %d\n", 
	    eventPointer->eventID);	  
  }
  return(nullRecordFunction(recordType,eventPointer,timeStamp,
			    dataPointer,dataLength));    
  
}

/* Wrappers below for UNIX I/O calls */

int
traceREAD(fd, buf, nbyte)
     int fd;
     char *buf;
     int nbyte;
{
  struct {
    long filepointer;
    long numVariables;
    long cause;
  } readArgs;  

  int ret;

  readArgs.filepointer = fd;
  readArgs.numVariables = 1;
  readArgs.cause = -1;    /* because cause is a high-level concept,
			    fill in bogus value here. We can't make a better 
			    guess. */

  traceEvent(readBeginID, (char *) &readArgs, sizeof readArgs);
  ret = read(fd, buf, nbyte);
  traceEvent(readEndID, (char *) &ret, sizeof(ret));
  return(ret);
}
       

long 
traceFREAD(ptr, size, nitems, stream)
     char *ptr;
     int size;
     int nitems;
     FILE *stream;
{
  struct {
    long filepointer;
    long numVariables;
    long cause;
  } readArgs;  

  int ret, nbytes;

  readArgs.filepointer = fileno(stream);
  readArgs.numVariables = nitems;
  readArgs.cause = -1;

  traceEvent(freadBeginID, (char *) &readArgs, sizeof readArgs);
  ret = fread(ptr, size, nitems, stream);
  nbytes = ret * size;
  traceEvent(freadEndID, (char *) &nbytes, 1);
  return(ret);
}

       

long 
traceGETW(stream)
  FILE *stream;
{
  struct {
    long filepointer;
    long numVariables;
    long cause;
  } readArgs;  

  int ret, nbytes = sizeof(int);

  readArgs.filepointer = fileno(stream);
  readArgs.numVariables = 1;
  readArgs.cause = -1;

  traceEvent(freadBeginID, (char *) &readArgs, sizeof readArgs);
  ret = getw(stream);
  traceEvent(freadEndID, (char *) &nbytes, 0);
  return(ret);
}
       

long 
traceFGETC(stream)
  FILE *stream;
{
  struct {
    long filepointer;
    long numVariables;
    long cause;
  } readArgs;  

  int ret, nbytes = sizeof(char);

  readArgs.filepointer = fileno(stream);
  readArgs.numVariables = 1;
  readArgs.cause = -1;

  traceEvent(freadBeginID, (char *) &readArgs, sizeof readArgs);
  ret = fgetc(stream);
  traceEvent(freadEndID, (char *) &nbytes, 0);
  return(ret);
}
       

char *
traceGETS(s)
  char *s;
{
  struct {
    long filepointer;
    long numVariables;
    long cause;
  } readArgs;  

  char *ret;
  int nbytes;

  readArgs.filepointer = fileno(stdin);
  readArgs.numVariables = 1;
  readArgs.cause = -1;

  traceEvent(freadBeginID, (char *) &readArgs, sizeof readArgs);
  ret = gets(s);
  nbytes = strlen(s) + 1;
  traceEvent(freadEndID, (char *) &nbytes, sizeof(int));
  return(ret);
}
       

char *
traceFGETS(s, n, stream)
  char *s;
  int n;
  FILE *stream;
{
  struct {
    long filepointer;
    long numVariables;
    long cause;
  } readArgs;  

  char *ret;
  int nbytes;

  readArgs.filepointer = fileno(stream);
  readArgs.numVariables = 1;
  readArgs.cause = -1;

  traceEvent(freadBeginID, (char *) &readArgs, sizeof readArgs);
  ret = fgets(s, n, stream);
  nbytes = strlen(s);
  traceEvent(freadEndID, (char *) &nbytes, 0);
  return(ret);
}

int traceWRITE(fd, buf, nbyte)
     int fd;
     char *buf;
     int nbyte;
{
  struct {
    long filepointer;
    long numVariables;
    long cause;
  } writeArgs;  

  int ret, nbytes;

  writeArgs.filepointer = fd;
  writeArgs.numVariables = 0;
  writeArgs.cause = -1;

  traceEvent(writeBeginID, (char *) &writeArgs, sizeof writeArgs);
  ret = write(fd, buf, nbyte);
  nbytes = ret;
  traceEvent(writeEndID, (char *) &nbytes, sizeof(int));
  return(ret);
}  

long 
traceFWRITE(ptr, size, nitems, stream)
     char *ptr;
     int size;
     int nitems;
     FILE *stream;
{
  struct {
    long filepointer;
    long numVariables;
    long cause;
  } writeArgs;  

  int ret, nbytes;

  writeArgs.filepointer = fileno(stream);
  writeArgs.numVariables = nitems;
  writeArgs.cause = -1;

  traceEvent(fwriteBeginID, (char *) &writeArgs, sizeof writeArgs);
  ret = fwrite(ptr, size, nitems, stream);
  nbytes = ret * size;
  traceEvent(fwriteEndID, (char *) &nbytes, sizeof(int));
  return(ret);
}

       

long 
tracePUTW(w, stream)
  int w;
  FILE *stream;
{
  struct {
    long filepointer;
    long numVariables;
    long cause;
  } writeArgs;  

  int ret, nbytes = sizeof(int);

  writeArgs.filepointer = fileno(stream);
  writeArgs.numVariables = 1;
  writeArgs.cause = -1;

  traceEvent(fwriteBeginID, (char *) &writeArgs, sizeof writeArgs);
  ret = putw(w, stream);
  traceEvent(fwriteEndID, (char *) &nbytes, sizeof(int));
  return(ret);
}
       

long 
traceFPUTC(c, stream)
  int c;
  FILE *stream;
{
  struct {
    long filepointer;
    long numVariables;
    long cause;
  } writeArgs;  

  int ret, nbytes = sizeof(char);

  writeArgs.filepointer = fileno(stream);
  writeArgs.numVariables = 1;
  writeArgs.cause = -1;

  traceEvent(fwriteBeginID, (char *) &writeArgs, sizeof writeArgs);
  ret = fputc(c, stream);
  traceEvent(fwriteEndID, (char *) &nbytes, sizeof(int));
  return(ret);
}
       

tracePUTS(s)
  char *s;
{
  struct {
    long filepointer;
    long numVariables;
    long cause;
  } writeArgs;  

  int nbytes = strlen(s) + 1;

  writeArgs.filepointer = fileno(stdout);
  writeArgs.numVariables = 1;
  writeArgs.cause = -1;

  traceEvent(fwriteBeginID, (char *) &writeArgs, sizeof writeArgs);
  puts(s);
  traceEvent(fwriteEndID, (char *) &nbytes, sizeof(int));
  return;
}
       

traceFPUTS(s, stream)
  char *s;
  FILE *stream;
{
  struct {
    long filepointer;
    long numVariables;
    long cause;
  } writeArgs;  

  int nbytes = strlen(s);

  writeArgs.filepointer = fileno(stream);
  writeArgs.numVariables = 1;
  writeArgs.cause = -1;

  traceEvent(fwriteBeginID, (char *) &writeArgs, sizeof writeArgs);
  fputs(s, stream);
  traceEvent(fwriteEndID, (char *) &nbytes, sizeof(int));
  return;
}



long 
traceFSEEK(stream, offset, ptrname)
     FILE *stream;
     long offset;
     int ptrname;
{
  struct {
    long filepointer;
    long offset;
    long ptrname;
  } seekArgs;  

  int ret;

  seekArgs.filepointer = fileno(stream);
  seekArgs.offset = offset;
  seekArgs.ptrname = ptrname;

  traceEvent(fseekBeginID, (char *) &seekArgs, sizeof seekArgs);
  ret = fseek(stream,offset,ptrname);
  traceEvent(fseekEndID, (char *) 0, 0);
  return(ret);
}


traceREWIND(stream)
  FILE *stream;
{
  struct {
    long filepointer;
    long offset;
    long ptrname;
  } seekArgs;  

  seekArgs.filepointer = fileno(stream);
  seekArgs.offset = 0L;
  seekArgs.ptrname = SEEK_SET;

  traceEvent(fseekBeginID, (char *) &seekArgs, sizeof seekArgs);
  rewind(stream);
  traceEvent(fseekEndID, (char *) 0, 0);
  return;
}



long 
traceLSEEK(fd, offset, whence)
     int fd;
     off_t offset;
     int whence;
{
  struct {
    long filepointer;
    long offset;
    long ptrname;
  } seekArgs;  

  off_t ret;

  seekArgs.filepointer = fd;
  seekArgs.offset = offset;
  seekArgs.ptrname = whence;

  traceEvent(lseekBeginID, (char *) &seekArgs, sizeof seekArgs);
  ret = lseek(fd,offset,whence);
  traceEvent(lseekEndID, (char *) 0, 0);
  return(ret);
}

traceRewindBegin(stream)
  FILE *stream;
{
  struct {
    long filepointer;
    long offset;
    long ptrname;
  } seekArgs;  

  seekArgs.filepointer = fileno(stream);
  seekArgs.offset = 0L;
  seekArgs.ptrname = SEEK_SET;

  traceEvent(fseekBeginID, (char *) &seekArgs, sizeof seekArgs);
  return;
}


traceRewindEnd()
{
  traceEvent(fseekEndID, (char *) 0, 0);
  return;
}


long 
tracereadbegin_(filepointer,  numVariables,  cause)
long *filepointer;
long *numVariables;
long *cause;
{
  return traceReadBegin( *filepointer, *numVariables, *cause );
}

long 
traceReadBegin (filepointer, numVariables, cause )
long filepointer;
long numVariables;
long cause;
{

  struct {
    long filepointer;
    long numVariables;
    long cause;
  } readArgs;

  readArgs.filepointer = filepointer;
  readArgs.numVariables = numVariables;
  readArgs.cause = cause;
  
  traceEvent(readBeginID, (char *) &readArgs, sizeof readArgs);
  return(1);
}


long 
tracewritebegin_(filepointer, numVariables, cause)
long *filepointer;
long *numVariables;
long *cause;
{
  return traceWriteBegin( *filepointer, *numVariables, *cause );
}

long 
traceWriteBegin (filepointer, numVariables, cause )
long filepointer;
long numVariables;
long cause;
{

  struct {
    long filepointer;
    long numVariables;
    long cause;
  } writeArgs;

  writeArgs.filepointer = filepointer;
  writeArgs.numVariables = numVariables;
  writeArgs.cause = cause;
  
  traceEvent(writeBeginID, (char *) &writeArgs, sizeof writeArgs);
  return(1);
}


long 
tracereadend_(nbytes)
long *nbytes;
{
  return traceReadEnd(*nbytes);
}

long 
traceReadEnd (nbytes)
     int nbytes;
{
  traceEvent(readEndID, (char *) &nbytes, 0);
  return(1);
}


long 
tracewriteend_(nbytes)
long *nbytes;     
{
  return traceWriteEnd(*nbytes);
}

long 
traceWriteEnd (nbytes)
     int nbytes;
{
  traceEvent(writeEndID, (char *) &nbytes, 0);
  return(1);
}

setTimeWindowSize ( windowSize )
     double windowSize;
{
  if (windowSize > 0  ) {
    statisticalWindowData.windowSize = windowSize;
  } else { 
    /* error */
    fprintf(stderr, "warning: setTimeWindowSize: attempt to set negative time window\n");
  }
}

/* force a dump of histograms, depending on the argument flags. This
 * is a user invoked call. */

outputHistograms(flags)
     int flags;
{
  if (RS_HIST(flags)) {
    outputHistogram(&readHistogram, getClock(), readSizeHistogramID,
		    RECORD_TRACE);
  }

  if (WS_HIST(flags)) {
    outputHistogram(&writeHistogram, getClock(), writeSizeHistogramID,
		    RECORD_TRACE);
  }
}

/* Actually output a histogram. This is called from somewhere in the
   extension */

outputHistogram(h, timeStamp, recordID, recordType)
     struct histogramData *h;
     CLOCK timeStamp;
     int recordType;
{
  static TR_RECORD           traceRecord;
  static char                *recordBuffer = (char *) 0;
  static struct histogramRecordData    histogramSummaryData; 


  traceRecord.recordLength = sizeof histogramSummaryData + 
    h->numbins * sizeof(int);
  traceRecord.recordContents = (char *) histogramRecordBuffer;
  histogramSummaryData.packetLength = traceRecord.recordLength;
  histogramSummaryData.packetType = PKT_DATA;
  histogramSummaryData.packetTag = FAMILY_HISTOGRAM | recordType;
  histogramSummaryData.clockDimension = sizeof(CLOCK)/sizeof(int);
  histogramSummaryData.timeStamp = timeStamp;
  histogramSummaryData.seconds = clockToSeconds( timeStamp );
  histogramSummaryData.eventID = recordID;
  histogramSummaryData.nodeID  = TRgetNode();
  histogramSummaryData.histSize = h->numbins;
  bcopy(histogramSummaryData, histogramRecordBuffer, sizeof(histogramSummaryData));
  bcopy((char *) h->data, (char *) (histogramRecordBuffer + sizeof(histogramSummaryData)), 
	h->numbins * sizeof(int));


  putBytes(traceRecord.recordContents,
	   (unsigned) traceRecord.recordLength);

  resetHistogram(h, clockToSeconds (timeStamp) );    
}
    
    
    

/* Generate a histogram event record. Calls outputHistogram to do the
   dirty work */

TR_RECORD *
histogramEventRecordFunction(   recordType, eventPointer, timeStamp,
				    dataPointer, dataLength )
     int		recordType;
     TR_EVENT	        *eventPointer;
     CLOCK		timeStamp;
     char		*dataPointer;
     unsigned int	dataLength;
{
  static TR_RECORD		traceRecord;
  static char			*recordBuffer = (char *) 0;
  static int			bufferLength = 0;
  struct histogramRecordData      *histogramSummaryData;
  struct stat buf;
  double ts;
  double lastHistTimeStamp;
  int nbytes;

  /* The time stamp stored in the event descriptor will be used	    */
  /* unless one is specified in the timeStamp parameter.		    */

  if ( clockCompare( timeStamp, noSuchClock ) == 0 )
    timeStamp = eventPointer->eventLast;

  ts = clockToSeconds(timeStamp);
  lastHistTimeStamp = ts;


  if (OUTPUT_HISTOGRAM(readHistogram,ts)) {
    outputHistogram(&readHistogram, timeStamp, readSizeHistogramID,
		    recordType);
  }

  if (OUTPUT_HISTOGRAM(writeHistogram,ts)) {
    outputHistogram(&writeHistogram, timeStamp, writeSizeHistogramID,
		    recordType);    
  }

  if (isReadEndEvent(eventPointer->eventID)) {
    nbytes = *((int *) dataPointer);
    
    histogramInsert(nbytes, &readHistogram);
  } else { 
    if (isWriteEndEvent(eventPointer->eventID)) {
      nbytes = *((int *) dataPointer);
      histogramInsert(nbytes, &writeHistogram);
      
    } else {
/*      fprintf(stderr, "histogramEventRecord: invalid event ID\n", 
	      eventPointer->eventID); */
    }
  }
  /* return a null record to the above */
  return(nullRecordFunction(recordType,eventPointer,timeStamp,
			    dataPointer,dataLength));
}

/* Reset the histogram after it is output. This is an internal
   extension call */

resetHistogram(h, ts)
     struct histogramData *h;
     double ts;
{
  int i;

  while ((h->start+h->windowSize) < ts) {
    h->start += h->windowSize;
  }
  for(i=0; i < h->numbins; i++) {
    h->data[i] = 0;
  }
}

/* Insert a value n into histogram h; used internally in extension */

histogramInsert(n, h)
     int n;
     struct histogramData *h;
{
  int i;

  if (n < h->lo) {
    h->data[0]++;
    return;
  }
  if (n > h->hi) {
    h->data[h->numbins]++;
    return;
  }
  h->data[(int) (n/h->binwidth)]++;
}
  
/* blockSummaryEventRecordFunction is the guts of almost all
 * summarization. It keeps summaries of activity within a certain
 * blocksize. Set this blocksize to MAXINT, and you can control when
 * you reset those summaries yourself (for example, at certain time
 * intervals or upon file close). 
 */

TR_RECORD * blockSummaryEventRecordFunction( recordType, eventPointer,
					    timeStamp, 
					    dataPointer, dataLength ) 

     int recordType; 
     TR_EVENT *eventPointer; 
     CLOCK timeStamp; 
     char *dataPointer; 
     unsigned int *dataLength;

{
  static TR_RECORD		traceRecord;
  static char			*recordBuffer = (char *) 0;
  static int			bufferLength = 0;
  struct blockSummaryRecordData      *blockSummaryData;
  struct stat buf;

  struct readwriteArgs {
    long filepointer;
    long numVariables;
    long cause;
  };

  struct seekArgs {
    long	filepointer;
    long	offset;
    long    ptrname;
  } *seekBeginData;


  struct readwriteArgs *readBeginData, *writeBeginData;

  static CLOCK lastReadBegin;
  static CLOCK lastWriteBegin;
  static CLOCK lastSeekBegin;
  static int outputRecord = 0;

  static int fileNumber; /* static so that the next time this is called,
			    the file number will be "remembered" */
  static struct blockSummaryData *blockData;

  static struct spaceSummaryRecordData summaryData;
  static int fileClosedFlag;

  int nbytes;

  /* The time stamp stored in the event descriptor will be used	    */
  /* unless one is specified in the timeStamp parameter.		    */

  if ( clockCompare( timeStamp, noSuchClock ) == 0 )
    timeStamp = eventPointer->eventLast;


  /* we call putBytes here directly because these kinds of functions
     can only return one record */

  /*this is weird... output record only if outputRecord flag is set
    (we crossed some spatial boundary) or if the file by a certain ID was
    closed. However, this function is used, inappropriately, as a
    workhorse for the file lifetime function by setting the
    GlobalBlockSize to 0 so statistics are accumulated forever. Until this
    is separated out into workhorse function and spatial summary function,
    we include the following kludge 
    */

  if (outputRecord || (fileClosedFlag > 0 && GlobalBlockSize < MAXINT)) {
    /* output record */

    traceRecord.recordLength = sizeof summaryData;
    traceRecord.recordContents = (char *) &summaryData;
    summaryData.packetLength = traceRecord.recordLength;
    summaryData.packetType = PKT_DATA;
    summaryData.packetTag = FAMILY_SPACE_SUMMARY | recordType;
    summaryData.clockDimension = sizeof(CLOCK)/sizeof(int);
    summaryData.timeStamp = timeStamp;
    summaryData.seconds = clockToSeconds(timeStamp);
    summaryData.eventID = spaceSummaryID;
    summaryData.closed = fileClosedFlag > 0 ? 1 : 0;
    summaryData.nodeID = TRgetNode();
    summaryData.fileID = fileNumber;
    summaryData.readCount = blockData->readData.frequency;
    summaryData.readBytes = blockData->readData.numBytes;
    summaryData.readTime =  blockData->readData.totalTime;
    summaryData.writeCount = blockData->writeData.frequency;
    summaryData.writeBytes = blockData->writeData.numBytes;
    summaryData.writeTime =  blockData->writeData.totalTime;
    summaryData.seekCount = blockData->seekData.frequency;
    summaryData.seekBytes = blockData->seekData.numBytes;
    summaryData.seekTime =  blockData->seekData.totalTime;
    summaryData.firstByteAccessed = blockData->firstByte;
    summaryData.lastByteAccessed = blockData->lastByte;

    putBytes(traceRecord.recordContents,
	     (unsigned) traceRecord.recordLength);
      
    zeroData(blockData);
    
    fileClosedFlag = 0;
    outputRecord = 0;
  } 



  if (isReadBeginEvent(eventPointer->eventID)) {

    fileNumber = * ((long *) dataPointer);
    readBeginData = (struct readwriteArgs *) dataPointer;

    mallocBlockIfNecessary(fileNumber);

    /* make updates to right file data */
    blockData = blockSDT[fileNumber];
    lastReadBegin = timeStamp;
    
    if (blockData->bytePosition <  blockData->firstByte ) {
      blockData->firstByte = blockData->bytePosition;
    }
    if (blockData->bytePosition > blockData->lastByte ) {
      blockData->lastByte = blockData->bytePosition;
    }      



    blockData->readData.frequency++;
    blockData->readData.numVariables += readBeginData->numVariables;

  } else if (isReadEndEvent(eventPointer->eventID)) {
    nbytes = *((int *) dataPointer);
      
    if (nbytes < 0) { nbytes = 0; } /*be sure to ignore -1 responses */
      
    if ((int) (blockData->bytePosition / GlobalBlockSize) != 
	(int) ((blockData->bytePosition + nbytes) /
	       GlobalBlockSize)) { /* cross block boundary */
      /* output data */
      outputRecord = 1;
    } 

    blockData->readData.numBytes += nbytes;
    blockData->bytePosition += nbytes;

    if (blockData->bytePosition <  blockData->firstByte ) {
      blockData->firstByte = blockData->bytePosition;
    }
    if (blockData->bytePosition > blockData->lastByte ) {
      blockData->lastByte = blockData->bytePosition;
    }      

    blockData->readData.totalTime += 
      clockToSeconds(clockSubtract(timeStamp,lastReadBegin));
    lastReadBegin = noSuchClock;
  } else if (isWriteBeginEvent(eventPointer->eventID)) {

    fileNumber = * ((long *) dataPointer);
    writeBeginData = (struct readwriteArgs *) dataPointer;

    mallocBlockIfNecessary(fileNumber);
    
    /* make updates to block summary data */
    blockData = blockSDT[fileNumber];

    if (blockData->bytePosition <  blockData->firstByte ) {
      blockData->firstByte = blockData->bytePosition;
    }
    if (blockData->bytePosition > blockData->lastByte ) {
      blockData->lastByte = blockData->bytePosition;
    }      

    lastWriteBegin = timeStamp;
    blockData->writeData.frequency++;
    blockData->writeData.numVariables += writeBeginData->numVariables;
  } else if (isWriteEndEvent(eventPointer->eventID)) {
    nbytes = *((int *) dataPointer);
    if (nbytes < 0) { nbytes = 0; } /*be sure to ignore -1 responses */
    if ((int) (blockData->bytePosition / GlobalBlockSize) != 
	(int) ((blockData->bytePosition + nbytes) /
	       GlobalBlockSize)) { /* cross block boundary */
      /* set flag to output data */
      outputRecord = 1;
    }

    blockData->writeData.totalTime += 
      clockToSeconds(clockSubtract(timeStamp,lastWriteBegin));

    blockData->bytePosition += nbytes;
    blockData->writeData.numBytes += nbytes;

    if (blockData->bytePosition <  blockData->firstByte ) {
      blockData->firstByte = blockData->bytePosition;
    }
    if (blockData->bytePosition > blockData->lastByte ) {
      blockData->lastByte = blockData->bytePosition;
    }      

    lastWriteBegin = noSuchClock;
  } else if (isSeekBeginEvent(eventPointer->eventID)) {
    lastSeekBegin = timeStamp;
    fileNumber = * ((long *) dataPointer);
    seekBeginData = (struct seekArgs *) dataPointer;

    mallocBlockIfNecessary(fileNumber);

    /* make updates to block summary data */
    blockData = blockSDT[fileNumber];
    
    switch(seekBeginData->ptrname)  {
    case SEEK_CUR:
      if ((int) (blockData->bytePosition / GlobalBlockSize) != 
	  (int) ((blockData->bytePosition + seekBeginData->offset) /
		 GlobalBlockSize)) { /* cross block boundary */
	blockData->bytePosition += seekBeginData->offset;
	/* flag to output data */
	outputRecord = 1;
	blockData->bytePosition += seekBeginData->offset;
	blockData->seekData.frequency++;
	blockData->seekData.numBytes += seekBeginData->offset;

      } else { /* didn't cross block boundary */
	blockData->bytePosition += seekBeginData->offset;
	blockData->seekData.frequency++;
	blockData->seekData.numBytes += seekBeginData->offset;
      }
      break;
    case SEEK_SET:
      if ((int) (blockData->bytePosition / GlobalBlockSize) != 
	  (int) ( seekBeginData->offset / GlobalBlockSize)) { 
	/* cross block boundary */
	/* flag to output data */
	outputRecord = 1;
      }
      blockData->seekData.numBytes += 
	abs(seekBeginData->offset - blockData->bytePosition);
      blockData->bytePosition = seekBeginData->offset;
      blockData->seekData.frequency++;
    break;
    case SEEK_END:
      fstat(fileNumber, &buf);
      if ((int) (blockData->bytePosition / GlobalBlockSize) !=
	   (int) ((buf.st_size + seekBeginData->offset)/ GlobalBlockSize)) {
	outputRecord = 1;
      }
      blockData->seekData.numBytes += 
	abs((buf.st_size + seekBeginData->offset) - blockData->bytePosition);
      blockData->bytePosition = buf.st_size + seekBeginData->offset;
      blockData->seekData.frequency++;
      break;
    }
    if (blockData->bytePosition <  blockData->firstByte ) {
      blockData->firstByte = blockData->bytePosition;
    }
    if (blockData->bytePosition > blockData->lastByte ) {
      blockData->lastByte = blockData->bytePosition;
    }      
  } else if (isSeekEndEvent(eventPointer->eventID)) {
    blockData->seekData.totalTime += 
      clockToSeconds(clockSubtract(timeStamp,lastSeekBegin));
    lastSeekBegin = noSuchClock;
  } else if (isCloseBeginEvent(eventPointer->eventID)) {
    fileNumber = * ((long *) dataPointer);
    fileClosedFlag = fileNumber; /* should be positive */
  }
  /* return a null record to the above */
  return(nullRecordFunction(recordType,eventPointer,timeStamp,
			    dataPointer,dataLength));
  
}

/* helper macro */
#define outputTimeRecord 	summaryData.packetLength = traceRecord.recordLength;\
	summaryData.packetType = PKT_DATA;\
	summaryData.packetTag = FAMILY_TIME_SUMMARY | recordType;\
        summaryData.clockDimension = sizeof(CLOCK)/sizeof(int);\
        summaryData.timeStamp = timeStamp;\
        summaryData.seconds = clockToSeconds( timeStamp );\
	summaryData.start =  statisticalWindowData.start;\
	summaryData.eventID = timeSummaryID;\
	summaryData.closed = 0;\
	summaryData.fileID = fileid;\
        summaryData.nodeID = TRgetNode();\
	summaryData.readFrequency = blockSDT[fileid]->readData.frequency;\
        summaryData.readBytes =  blockSDT[fileid]->readData.numBytes;\
	summaryData.readTime =blockSDT[fileid]->readData.totalTime;\
        summaryData.writeFrequency = blockSDT[fileid]->writeData.frequency;\
        summaryData.writeBytes = blockSDT[fileid]->writeData.numBytes;\
        summaryData.writeTime =blockSDT[fileid]->writeData.totalTime;\
        summaryData.seekFrequency =blockSDT[fileid]->seekData.frequency;\
        summaryData.seekBytes = blockSDT[fileid]->seekData.numBytes;\
        summaryData.seekTime =blockSDT[fileid]->seekData.totalTime;\
	summaryData.firstByteAccessed = blockSDT[fileid]->firstByte;\
	summaryData.lastByteAccessed = blockSDT[fileid]->lastByte;


/* Keep time statistics */

TR_RECORD *  
timeStatisticsEventRecordFunction(  recordType, eventPointer, timeStamp,
				    dataPointer, dataLength )
     int		recordType;
     TR_EVENT	        *eventPointer;
     CLOCK		timeStamp;
     char		*dataPointer;
     unsigned int	dataLength;

{
  static TR_RECORD		traceRecord;
  static char			*recordBuffer = (char *) 0;
  static int			bufferLength = 0;
  struct recordDataTrace		*recordHeaderTrace;
  struct recordDataCount		*recordHeaderCount;


  int fileid;
  static struct timeLifeRecordData summaryData;
  double ts;

  if ( clockCompare( timeStamp, noSuchClock ) == 0 )
    timeStamp = eventPointer->eventLast;

  ts = clockToSeconds(timeStamp);

  if (ts > (statisticalWindowData.start + statisticalWindowData.windowSize)) {
    
    /* output a record , reset summary data */

    /* there is a problem here. if you index (as we do) on the file
       descriptor then it's possible to record activity that occurs on the
       same file descriptor, but several different files. */

    /* also, this is not accurate -- what if a read/write/seek spans
       a time boundary?   You will dump a record with an updated 
       frequency, var count, etc. but  a possibly invalid byte position */

    for(fileid=0; fileid < MAX_OPEN_FILES; fileid++) {
      if (blockSDT[fileid] != NULL && (EVENT_FREQUENCY(fileid) > 1)) {
	/* output record, putting bytes */

	traceRecord.recordContents = (char *) &summaryData;
	traceRecord.recordLength = sizeof summaryData;
	
	outputTimeRecord;
	putBytes(traceRecord.recordContents,
		 (unsigned) traceRecord.recordLength);

	/* reset count */
	zeroData(blockSDT[fileid]);
      }
    }
    while ((statisticalWindowData.start + 
	    statisticalWindowData.windowSize) < ts) {
      statisticalWindowData.start += statisticalWindowData.windowSize;
    }
  }

  statisticalWindowData.finish = clockToSeconds(timeStamp);

  if(!isOpenCloseEvent(eventPointer->eventID)) {

    blockSummaryEventRecordFunction(recordType, eventPointer, timeStamp,
				    dataPointer, dataLength );
    /* if we hit a file close in operation, be sure to dump out the
       time record for what we have */

  } else if (isCloseBeginEvent(eventPointer->eventID)){
    fileid = *((int *) dataPointer);

    if (blockSDT[fileid] != NULL && (EVENT_FREQUENCY(fileid) > 1)) {
      traceRecord.recordContents = (char *) &summaryData;
      traceRecord.recordLength = sizeof summaryData;

      outputTimeRecord;
      summaryData.closed = 1;  /* mark this record as being due to a 
				file close event */
      putBytes(traceRecord.recordContents,
	       (unsigned) traceRecord.recordLength);
      zeroData(blockSDT[fileid]);
    }
  }


  /* return a null record */
  return(nullRecordFunction(recordType,eventPointer,timeStamp,
			    dataPointer,dataLength));
  
}


/* zeroData is a helper function that zeros blocks managed by the
   block summary record function. Bzero didn't seem to be working, for
   some odd reason, and I got very frustrated and just did it this
   way.
*/

zeroData(blockData)
  struct blockSummaryData *blockData;     
{
    blockData->readData.frequency = 0;
    blockData->readData.numBytes = 0;
    blockData->readData.numVariables = 0;
    blockData->readData.totalTime = 0;

    blockData->writeData.frequency = 0;
    blockData->writeData.numBytes = 0;
    blockData->writeData.numVariables = 0;
    blockData->writeData.totalTime = 0;

    blockData->seekData.frequency = 0;
    blockData->seekData.numBytes = 0;
    blockData->seekData.numVariables = 0;
    blockData->seekData.totalTime = 0;

    blockData->firstByte = MAXINT;
    blockData->lastByte = 0;
    blockData->lowTime = MAXDOUBLE;
    blockData->hiTime = 0.0;

  }

/* histogram access functions */

void initializeHistograms(lo, hi, numberbins, windowSize, flags)
     int lo, hi, numberbins;
     double windowSize;
     int flags;
{
  /* this should take flags to toggle different kinds of histograms */
  /* numberbins must divide the range equally */
  /* no support as of yet for unequally sized bins */

  if (((hi - lo) % numberbins) != 0) {
    fprintf(stderr, "initializeHistograms: improper bin specification\n");
    return;
  }

  if (RS_HIST(flags)) {
    readHistogram.data = (int *)  malloc((numberbins + 2) * sizeof(int));
    readHistogram.lo = lo;
    readHistogram.hi = hi;
    readHistogram.numbins = numberbins + 2;
    readHistogram.active = 1;
    readHistogram.windowSize = windowSize;
    readHistogram.binwidth = (hi - lo) /numberbins;
  }
  if (WS_HIST(flags)) {

    writeHistogram.data = (int *)  malloc((numberbins + 2) * sizeof(int));
    writeHistogram.lo = lo;
    writeHistogram.hi = hi;
    writeHistogram.numbins = numberbins + 2;
    writeHistogram.active = 1;
    writeHistogram.windowSize = windowSize;
    writeHistogram.binwidth = (hi - lo) /numberbins;
  }

  if ((numberbins+2) > maxHistogramRecordSize) {
    free(histogramRecordBuffer);
  /* allocate space for output histogram records */
    histogramRecordBuffer = (char *) malloc(sizeof(struct histogramRecordData)+
					    (numberbins + 2)* sizeof(int));
  }

  setEventRecordFunction( readBeginID, histogramEventRecordFunction );
  setEventRecordFunction( freadBeginID, histogramEventRecordFunction );
  setEventRecordFunction( writeBeginID, histogramEventRecordFunction);
  setEventRecordFunction( fwriteBeginID, histogramEventRecordFunction );

  setEventRecordFunction(freadEndID, histogramEventRecordFunction);  
  setEventRecordFunction(readEndID, histogramEventRecordFunction);  
  setEventRecordFunction(fwriteEndID, histogramEventRecordFunction);  
  setEventRecordFunction(writeEndID, histogramEventRecordFunction);  
  setEventRecordFunction(fopenBeginID, nullRecordFunction);
  setEventRecordFunction(fopenEndID, nullRecordFunction);
  setEventRecordFunction(fcloseBeginID, nullRecordFunction);
  setEventRecordFunction(fcloseEndID, nullRecordFunction);

  setEventRecordFunction(fseekBeginID, histogramEventRecordFunction);
  setEventRecordFunction(fseekEndID, histogramEventRecordFunction);  
  setEventRecordFunction(lseekBeginID, histogramEventRecordFunction);
  setEventRecordFunction(lseekEndID, histogramEventRecordFunction);  

  setEventRecordFunction(openBeginID, histogramEventRecordFunction);
  setEventRecordFunction(openEndID, histogramEventRecordFunction);
  setEventRecordFunction(closeBeginID, histogramEventRecordFunction);
  setEventRecordFunction(closeEndID, histogramEventRecordFunction);


}


/* doTimeSummaryStatistics: initialize time summarization for a
   particular windowsize. */

void doTimeSummaryStatistics(windowSize)
     float windowSize;
{
  int i;

  setTimeWindowSize(windowSize);
  GlobalBlockSize = MAXINT;
  statisticalWindowData.start = 0;

  for(i=0; i < MAX_OPEN_FILES; i++) {
    blockSDT[i] = NULL;
  }

  /* Note here that you might have to change this function if you add
     a new event id for, say, a new kind of read call. Note also that
     other architectures have their own kinds of reads that are not,
     er, portable. I've handled this by making a device dependent call
     to correctly set the event record function for the nonportable
     event IDs. For UNIX, these are probably empty. This approach
     might be too cumbersome in the long run. */

  setEventRecordFunction(freadBeginID, timeStatisticsEventRecordFunction);
  setEventRecordFunction(freadEndID, timeStatisticsEventRecordFunction);  
  setEventRecordFunction(fseekBeginID, timeStatisticsEventRecordFunction);
  setEventRecordFunction(fseekEndID, timeStatisticsEventRecordFunction);  
  setEventRecordFunction(fwriteBeginID, timeStatisticsEventRecordFunction);
  setEventRecordFunction(fwriteEndID, timeStatisticsEventRecordFunction);  

  setEventRecordFunction(fopenBeginID, timeStatisticsEventRecordFunction);
  setEventRecordFunction(fopenEndID, timeStatisticsEventRecordFunction);
  setEventRecordFunction(fcloseBeginID, timeStatisticsEventRecordFunction);
  setEventRecordFunction(fcloseEndID, timeStatisticsEventRecordFunction);

  setEventRecordFunction(readBeginID, timeStatisticsEventRecordFunction);
  setEventRecordFunction(readEndID, timeStatisticsEventRecordFunction);  
  setEventRecordFunction(lseekBeginID, timeStatisticsEventRecordFunction);
  setEventRecordFunction(lseekEndID, timeStatisticsEventRecordFunction);  
  setEventRecordFunction(writeBeginID, timeStatisticsEventRecordFunction);
  setEventRecordFunction(writeEndID, timeStatisticsEventRecordFunction);  

  setEventRecordFunction(openBeginID, timeStatisticsEventRecordFunction);
  setEventRecordFunction(openEndID, timeStatisticsEventRecordFunction);
  setEventRecordFunction(closeBeginID, timeStatisticsEventRecordFunction);
  setEventRecordFunction(closeEndID, timeStatisticsEventRecordFunction);

  doTimeSummariesSystemDependent();
  

}

/* Fortran interface function for doTimeSummaryStatistics */

dotimesummarystatistics_(windowsize)
     float *windowsize;
{
  doTimeSummaryStatistics(*windowsize);
}


/* setBlockSummarySize: Extension interface function to set the block
   summary size to blocksize. 
*/

void setBlockSummarySize(blocksize)
     int blocksize;
{
  GlobalBlockSize = blocksize; /* GlobalBlockSize is a global variable */
}

/* Fortran interface function for setBlockSummarySize */

setblockSummarysize_(blocksize)
     int *blocksize;
{
  setBlockSummarySize(*blocksize);
}

/* doBlockSummaries: Extension interface function to do block
   summarization over specified blocksize
*/


void doBlockSummaries(blocksize)
     int blocksize;
{
  int i;

  setBlockSummarySize(blocksize);

  /* We index the block Summary Data Table by file id. Perhaps not the
     wisest thing to do, because you might want to collect information
     across file opens and closes, that might have different file ids
     assigned to them. 
   */

  /* MAX_OPEN_FILES is defined in a system file somewhere */

  for(i=0; i < MAX_OPEN_FILES; i++) {
    blockSDT[i] = NULL;
  }

  
  /* Note that when you add an event ID, below might have to
     change. System dependent stuff is taken care of in its own
     function call, below */

  setEventRecordFunction(freadBeginID, blockSummaryEventRecordFunction);
  setEventRecordFunction(freadEndID, blockSummaryEventRecordFunction);  
  setEventRecordFunction(fseekBeginID, blockSummaryEventRecordFunction);
  setEventRecordFunction(fseekEndID, blockSummaryEventRecordFunction);  
  setEventRecordFunction(fwriteBeginID, blockSummaryEventRecordFunction);
  setEventRecordFunction(fwriteEndID, blockSummaryEventRecordFunction);  
  setEventRecordFunction(fcloseBeginID, blockSummaryEventRecordFunction);
  setEventRecordFunction(fcloseEndID, blockSummaryEventRecordFunction);  
  setEventRecordFunction(fopenBeginID, blockSummaryEventRecordFunction);
  setEventRecordFunction(fopenEndID, blockSummaryEventRecordFunction);  

  setEventRecordFunction(readBeginID, blockSummaryEventRecordFunction);
  setEventRecordFunction(readEndID, blockSummaryEventRecordFunction);  
  setEventRecordFunction(lseekBeginID, blockSummaryEventRecordFunction);
  setEventRecordFunction(lseekEndID, blockSummaryEventRecordFunction);  
  setEventRecordFunction(writeBeginID, blockSummaryEventRecordFunction);
  setEventRecordFunction(writeEndID, blockSummaryEventRecordFunction);  
  setEventRecordFunction(openBeginID, blockSummaryEventRecordFunction);
  setEventRecordFunction(openEndID, blockSummaryEventRecordFunction);
  setEventRecordFunction(closeBeginID, blockSummaryEventRecordFunction);
  setEventRecordFunction(closeEndID, blockSummaryEventRecordFunction);

  doBlockSummariesSystemDependent();
}

/* Fortran interface function for doBlockSummaries */
doblocksummaries_(blocksize) 
     int *blocksize;
{
  doBlockSummaries(*blocksize);
}
  

/* doFileSummaries: Extension interface function to enable
   summarization of I/O activity by file.
*/

void doFileSummaries()
{
  int i;

  /* Note that reads and writes pass through
     blockSummaryEventRecordFunction. To trick it into keeping
     statistics over the lifetime of the file, set GlobalBlockSize to
     MAXINT. 
   */

  GlobalBlockSize = MAXINT; 

  /* Note that again, we index on open file id, perhaps a bad idea */

  for(i=0; i < MAX_OPEN_FILES; i++) {
    blockSDT[i] = NULL;
  }  
  
  /*Be sure to update this and other like functions when you add an
    event id */

  setEventRecordFunction(freadBeginID, blockSummaryEventRecordFunction);
  setEventRecordFunction(freadEndID, blockSummaryEventRecordFunction);  
  setEventRecordFunction(fseekBeginID, blockSummaryEventRecordFunction);
  setEventRecordFunction(fseekEndID, blockSummaryEventRecordFunction);  
  setEventRecordFunction(fwriteBeginID, blockSummaryEventRecordFunction);
  setEventRecordFunction(fwriteEndID, blockSummaryEventRecordFunction);  
  setEventRecordFunction(fopenBeginID, extFileLifetimeEventRecordFunction);
  setEventRecordFunction(fopenEndID, extFileLifetimeEventRecordFunction);
  setEventRecordFunction(fcloseBeginID, extFileLifetimeEventRecordFunction);
  setEventRecordFunction(fcloseEndID, extFileLifetimeEventRecordFunction);

  setEventRecordFunction(readBeginID, blockSummaryEventRecordFunction);
  setEventRecordFunction(readEndID, blockSummaryEventRecordFunction);  
  setEventRecordFunction(lseekBeginID, blockSummaryEventRecordFunction);
  setEventRecordFunction(lseekEndID, blockSummaryEventRecordFunction);  
  setEventRecordFunction(writeBeginID, blockSummaryEventRecordFunction);
  setEventRecordFunction(writeEndID, blockSummaryEventRecordFunction);  
  setEventRecordFunction(openBeginID, extFileLifetimeEventRecordFunction);
  setEventRecordFunction(openEndID, extFileLifetimeEventRecordFunction);
  setEventRecordFunction(closeBeginID, extFileLifetimeEventRecordFunction);
  setEventRecordFunction(closeEndID, extFileLifetimeEventRecordFunction);

  doFileSummariesSystemDependent();

}

/* Fortran interface function for doFileSummaries */

dofilesummaries_() { doFileSummaries(); };



traceopenbegin_(path, flags)
     char *path; 
     int *flags; 
{
  struct {
    int flags;
    int mode;
    char data[256];
  } openArgs;

  strcpy(openArgs.data,path);
  openArgs.flags = *flags;
  openArgs.mode = -1;
  
  traceEvent(openBeginID, &openArgs, sizeof(openArgs));
}

traceopenend_(fd)
     int *fd;
{
  traceEvent(openEndID, (char *) fd, sizeof(fd));
}

int trace2OPEN(path, flags) 
  char *path;
  int flags;
{
  struct {
    int flags;
    int mode;
    char data[256];
  } openArgs;

  int fd;

  strcpy(openArgs.data,path);
  openArgs.flags = flags;
  openArgs.mode = -1;

  traceEvent(openBeginID, &openArgs, sizeof(openArgs));
  fd = open(path,flags);
  traceEvent(openEndID, (char *) &fd, sizeof(fd));
  return(fd);
}

int trace3OPEN(path, flags, mode) 
  char *path;
  int flags;
  mode_t mode;     
{
  struct {
    int flags;
    int mode;
    char data[256];
  } openArgs;

  int fd;

  strcpy(openArgs.data,path);
  openArgs.flags = flags;
  openArgs.mode = mode;

  traceEvent(openBeginID, &openArgs, sizeof(openArgs));
  fd = open(path,flags,mode);
  traceEvent(openEndID, (char *) &fd, sizeof(fd));
  return(fd);
}

    

int traceCREAT(path, mode)
  char *path;
  mode_t mode;
{
  struct {
    int flags;
    int mode;
    char data[256];
  } openArgs;

  int fd;

  strcpy(openArgs.data,path);
  openArgs.flags = O_WRONLY | O_CREAT | O_TRUNC;
  openArgs.mode = mode;

  traceEvent(openBeginID, &openArgs, sizeof(openArgs));
  fd = creat(path, mode);
  traceEvent(openEndID, (char *) &fd, sizeof(fd));
  return(fd);
}



FILE *traceFOPEN(filename, type)
     char *filename, *type;
{
  char data[260];

  FILE *fp;
  int fd;
  int flags = 0;

  struct {
    int flags;
    int mode;
    char data[256];
  } openArgs;


  strcpy(openArgs.data, filename);
  if (strlen(type) == 1) {
    switch (type[0]) {
    case 'r':
      flags = flags | O_RDONLY;
      break;
    case 'w':
      flags = flags | O_WRONLY;
      break;
    case 'a':
      flags = flags | O_APPEND;
      break;
    }
  } else {
    switch(type[0]) {
    case 'r':
      flags = O_RDWR;
      break;
    case 'w':
      flags = O_TRUNC | O_CREAT | O_WRONLY;
      break;
    case 'a':
      flags = O_CREAT | O_APPEND;
      break;    
    }
  }
  openArgs.flags = flags;
  openArgs.mode= 0;

  traceEvent(fopenBeginID, &openArgs, sizeof(openArgs));
  fp = fopen(filename,type);
  if (fp != NULL) {
    fd = fileno(fp);
  } else {
    fd = -1;
  }

  traceEvent(fopenEndID, (char *) &fd, sizeof(int));   

  return(fp);
}


int traceFCLOSE(stream)
     FILE *stream;
{

  int fd, ret;
  fd = fileno(stream);

  traceEvent(fcloseBeginID, (char *) &fd, sizeof(int));
  ret = fclose(stream);
  traceEvent(fcloseEndID, (char *) 0, 0);
  return(ret);
}


int traceCLOSE(fd)
     int fd;
{
  int ret;
  traceEvent(closeBeginID, (char *) &fd, sizeof(fd));
  ret = close(fd);
  traceEvent(closeEndID, (char *) 0, 0);
  return(ret);
}

traceclosebegin_(fd)
     int *fd;
{
  traceEvent(closeBeginID, (char *) fd, sizeof(*fd));
}

tracecloseend_()
{
  traceEvent(closeEndID, (char *) 0, 0);
}

