/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *          (R)
 * The Pablo    Performance Analysis Environment software is NOT in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1991-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * PABLO is a registered trademark of
 * The Board of Trustees of the University of Illinois
 * registered in the U.S. Patent and Trademark Office.
 *
 * Author:  Roger J. Noe (noe@cs.uiuc.edu)
 * Contributing Author:  Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004,
 * by a grant from the Digital Equipment Corporation External Research
 * Program, and by a collaborative research agreement with the Intel
 * Supercomputer Systems Division.
 *
 */

/*
 * ProcTrace.c
 *	This file contains extensions to the Pablo instrumentation library
 *	which facilitate procedure entry and exit tracing.  This is done by
 *	defining new families of trace records, procedure entry and exit.  The
 *	procedure entry trace record is like the "ordinary" user event trace
 *	record defined by the instrumentation library except that there is no
 *	field for user data.  The procedure exit trace record replaces the
 *	user data field with a procedure duration field.  The function which
 *	generates the procedure exit records does so by a simple mapping from
 *	procedure exit event IDs to the corresponding procedure entry event
 *	IDs.
 */

#include "Assert.h"
#include "SystemDepend.h"
#include "SDDFparam.h"
#include "TraceParam.h"
#include "Trace.h"

/* Define the masks for the procedure entry/exit event families,	    */
/* similarly to FAMILY_EXTERNAL and internal event families in		    */
/* the include file TraceParam.h.					    */

#define FAMILY_PROCENTRY	0140		/* procedure entry	    */
#define FAMILY_PROCEXIT 	0150		/* procedure exit	    */


/* Number of procedures being traced by this code			    */
static int	numberProcedures = 0;

/* Each procedure being traced has associated with it a distinct pair of    */
/* entry and exit event IDs.  This code maintains a vector of such	    */
/* matchings which permits the ready identification of an event ID as	    */
/* being either an entry event or an exit event and for which procedure.    */

typedef struct procEventMatch {

	int			entryID;	/* procedure entry event ID */

	int			exitID;		/* procedure exit event ID  */

} PROC_EVENTS;

static PROC_EVENTS	*procEvents =		/* array of event ID pairs  */
			(PROC_EVENTS *) 0;


/* For each procedure being traced this code maintains a stack of	    */
/* procedure entry times.  Each procedure entry event causes the	    */
/* corresponding procedure's stack to be pushed, each procedure exit	    */
/* event causes the corresponding procedure's stack to be popped, and	    */
/* from the difference in time between entry and exit the procedure	    */
/* duration may be calculated in a very straightforward subtraction.  The   */
/* above procedure entry-exit event ID matching is used to map events to    */
/* their corresponding procedures.  In addition, the cumulative total of    */
/* these procedure durations is maintained for all traced subprocedures	    */
/* of each traced procedure.  That is, when a traced procedure exits,	    */
/* it increases this duration sum for its most immediate traced ancestor    */
/* procedure.  By subtracting this subprocedure duration sum from the	    */
/* traced procedure's inclusive duration, we arrive at the exclusive	    */
/* duration of the procedure.						    */

typedef struct procEntryTime {

	CLOCK			entryTime;	/* when procedure entered   */

	CLOCK			subProcTime;	/* subprocedure duration    */

	struct procEntryTime	*nextTime;	/* stack pointer down	    */

	struct procEntryTime	*ancestorProc;	/* traced ancestor	    */

} PROC_ENTRY;

static PROC_ENTRY	**procEntryStack =	/* array of pointers to	    */
			(PROC_ENTRY **) 0;	/* stack top elements	    */

static PROC_ENTRY	*activeProc =		/* most recent procedure    */
			(PROC_ENTRY *) 0;	/* entry structure	    */


/* Define data structure types for procedure entry and exit trace records,  */
/* similarly to record data structures in Trace.h			    */

/* FAMILY_PROCENTRY family Record Data packets:				    */

struct procEntryTraceRecordData {

	int	packetLength;		/* bytes in packet		    */

	int	packetType;		/* == PKT_DATA			    */

	int	packetTag;		/* FAMILY_PROCENTRY | RECORD_TRACE  */

	int	clockDimension;		/* number of ints in a CLOCK	    */

	CLOCK	timeStamp;		/* time record generated	    */

	double	seconds;		/* floating-point timestamp	    */

	int	eventID;		/* ID of corresponding event	    */

	int	nodeNumber;		/* occurred on which node	    */

	int	procIndex;		/* index into procedures table	    */

	long	sourceByte;		/* source code byte offset in file  */

	int	sourceLine;		/* source code line number in file  */
};

struct procEntryCountRecordData {

	int	packetLength;		/* bytes in packet		    */

	int	packetType;		/* == PKT_DATA			    */

	int	packetTag;		/* FAMILY_PROCENTRY | RECORD_COUNT  */

	int	clockDimension;		/* number of ints in a CLOCK	    */

	CLOCK	timeStamp;		/* time record generated	    */

	double	seconds;		/* floating-point timestamp	    */

	int	eventID;		/* ID of corresponding event	    */

	int	nodeNumber;		/* occurred on which node	    */

	int	procIndex;		/* index into procedures table	    */

	long	sourceByte;		/* source code byte offset in file  */

	int	sourceLine;		/* source code line number in file  */

	long	eventCount;		/* count of event occurrences	    */
};


/* FAMILY_PROCEXIT family Record Data packets:				    */

struct procExitTraceRecordData {

	int	packetLength;		/* bytes in packet		    */

	int	packetType;		/* == PKT_DATA			    */

	int	packetTag;		/* FAMILY_PROCEXIT | RECORD_TRACE   */

	int	clockDimension;		/* number of ints in a CLOCK	    */

	CLOCK	timeStamp;		/* time record generated	    */

	double	seconds;		/* floating-point timestamp	    */

	int	eventID;		/* ID of corresponding event	    */

	int	nodeNumber;		/* occurred on which node	    */

	int	procIndex;		/* index into procedures table	    */

	long	sourceByte;		/* source code byte offset in file  */

	int	sourceLine;		/* source code line number in file  */

	int	inclDurationDim;	/* number of ints in a CLOCK	    */

	CLOCK	inclusiveDuration;	/* total duration of procedure	    */

	double	inclusiveSeconds;	/* floating-point incl duration	    */

	double	exclusiveSeconds;	/* floating-point excl duration	    */

	int	exclDurationDim;	/* number of ints in a CLOCK	    */

	CLOCK	exclusiveDuration;	/* and its exclusive duration	    */
};

struct procExitCountRecordData {

	int	packetLength;		/* bytes in packet		    */

	int	packetType;		/* == PKT_DATA			    */

	int	packetTag;		/* FAMILY_PROCEXIT | RECORD_COUNT   */

	int	clockDimension;		/* number of ints in a CLOCK	    */

	CLOCK	timeStamp;		/* time record generated	    */

	double	seconds;		/* floating-point timestamp	    */

	int	eventID;		/* ID of corresponding event	    */

	int	nodeNumber;		/* occurred on which node	    */

	int	procIndex;		/* index into procedures table	    */

	long	sourceByte;		/* source code byte offset in file  */

	int	sourceLine;		/* source code line number in file  */

	int	inclDurationDim;	/* number of ints in a CLOCK	    */

	CLOCK	inclusiveDuration;	/* total duration of procedure	    */

	double	inclusiveSeconds;	/* floating-point incl duration	    */

	double	exclusiveSeconds;	/* floating-point excl duration	    */

	int	exclDurationDim;	/* number of ints in a CLOCK	    */

	CLOCK	exclusiveDuration;	/* and its exclusive duration	    */

	long	eventCount;		/* count of event occurrences	    */
};


/* Define data structures used to contain source code location data for	    */
/* Pablo instrumenting parser-generated code.				    */

static long	procByteOffset = -1;	/* source code byte offset	    */

static int	procLineNumber = -1;	/* source code line number	    */



/*
 * SUBROUTINE INITPROCTRACE (C function initProcTrace):
 *	This function initializes data structures specific to
 *	the procedure entry/exit tracing extensions of the Pablo
 *	instrumentation library.  The argument numProcs specifies
 *	how many procedures are to be traced.  The arguments procEntryID
 *	and procExitID, respectively, are vectors specifying the event IDs
 *	to be used as entry and exit events for each of the procedures.
 *	No event ID may be used as both an entry and an exit event.
 *	Further, there must be a one-to-one mapping between entry and
 *	exit event IDs.  The order of appearance of the event IDs in
 *	the two lists dictates which ones are paired as well as what
 *	procedure index is associated with each pair.
 *
 *	Sample Fortran invocation (using 27 as number of procedures):
 *		INTEGER ENT(27),EXT(27)
 *		DATA ENT/list of 27 procedure entry event IDs/
 *		DATA EXT/list of 27 procedure exit event IDs/
 *		CALL INITPROCTRACE(27,ENT,EXT)
 *	Sample C invocation:
 *		int entries[27] = { list of 27 entry IDs };
 *		int exits[27] = { list of exit IDs };
 *		initProcTrace( 27, entries, exits );
 */

initproctrace_( numProcs, procEntryID, procExitID )

int	*numProcs, *procEntryID, *procExitID;
{
	return initProcTrace( *numProcs, procEntryID, procExitID );
}


initProcTrace( numProcs, procEntryID, procExitID )

int	numProcs, *procEntryID, *procExitID;
{
	extern TR_RECORD	*procEventRecord();
	int			procIndex;

	if (( numProcs <= 0 ) ||
	    ( procEntryID == (int *) 0 ) ||
	    ( procExitID == (int *) 0 ))
		return FAILURE;

	/* Allocate space to store a copy of the procedure entry-exit	    */
	/* event ID matchings and also the procedure entry stacks.	    */

	procEvents = (PROC_EVENTS *) TRgetBuffer(
					    numProcs * sizeof(PROC_EVENTS) );

	if ( procEvents == (PROC_EVENTS *) 0 )
		TRfailure( "cannot allocate procedure events matching" );

	procEntryStack = (PROC_ENTRY **) TRgetBuffer(
					    numProcs * sizeof(PROC_ENTRY *) );

	if ( procEntryStack == (PROC_ENTRY **) 0 )
		TRfailure( "cannot allocate procedure entry stack pointers" );

	/* Perform procedure tracing extension pre-initialization, if	    */
	/* not already done.						    */

	preInitProcTrace();

	/* Initialize the procedure events matching from the arguments	    */
	/* passed.  Initialize the procedure entry stack pointers.	    */
	/* Configure the trace record-generating function for these events. */

	for ( procIndex = 0; procIndex < numProcs; procIndex++ ) {

		procEvents[ procIndex ].entryID = procEntryID[ procIndex ];
		procEvents[ procIndex ].exitID = procExitID[ procIndex ];

		procEntryStack[ procIndex ] = (PROC_ENTRY *) 0;

		setEventRecordFunction( procEntryID[ procIndex ],
				        procEventRecord );
		setEventRecordFunction( procExitID[ procIndex ],
				        procEventRecord );
	}

	numberProcedures = numProcs;

#ifdef DEBUG
	fprintf( debugFile, "initProcTrace done\n" );
	fflush( debugFile );
#endif /* DEBUG */

	return SUCCESS;
}


/*
 *	preInitProcTrace:
 *	   This function calls the trace library interface function
 *	   setRecordDescriptor, which records the address of the
 *	   procedure that generates the record descriptors for the
 *	   procedure trace event families.  It is automatically
 *	   invoked by initProcTrace.  However, the user may elect to
 *	   explicitly call preInitProcTrace before initProcTrace does.
 *	   Since initProcTrace calls the setEventRecordFunction interface
 *	   function of the trace library, that will cause the trace
 *	   library to perform its basic initialization.  One of the
 *	   effects of trace library basic initialization is to output
 *	   record descriptors for all event families defined up to
 *	   that point, by invoking all of the functions passed as
 *	   arguments to setRecordDescriptor, and then to output the
 *	   predefined, built-in record descriptors for internal event
 *	   families.  If no user-defined record descriptors beyond
 *	   those needed by this procedure tracing extension are
 *	   to be used, then preInitProcTrace need not be invoked.
 *	   However, if there are other such record descriptors (such
 *	   as an application also using the message tracing extension)
 *	   and if the user desires all these record descriptors to be
 *	   output before the trace library's internal event family
 *	   record descriptors, then all the required setRecordDescriptor
 *	   interface calls must be done before the trace library basic
 *	   initialization is performed.  preInitProcTrace may be used
 *	   for this very purpose.
 */

preinitproctrace_()
{
	preInitProcTrace();
}

preInitProcTrace()
{
	extern int	writeProcRecordDescriptors();
	static int	preInitDone = FALSE;

	if ( preInitDone == TRUE )
		return;

	/* Give the instrumentation library a pointer to the function	    */
	/* in which we output the specialized record descriptors for	    */
	/* procedure entry/exit.					    */

	setRecordDescriptor( writeProcRecordDescriptors );

	preInitDone = TRUE;
}


/*
 *	writeProcRecordDescriptors:
 *	   This function generates the record descriptors for the
 *	   procedure entry/exit event families.  It will be invoked
 *	   by the instrumentation library initialization routines.
 *	   Patterned after instrumentation library internal function
 *	   writeRecordDescriptors.
 */

writeProcRecordDescriptors()
{
	TR_RECORD	*recordPointer, *procSDDFdescriptor();

#ifdef DEBUG
	fprintf( debugFile, "writeProcRecordDescriptors\n" );
	fflush( debugFile );
#endif /* DEBUG */

	recordPointer = procSDDFdescriptor( FAMILY_PROCENTRY, RECORD_TRACE,
		"Procedure Entry Trace", "Procedure Entry Trace Record" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = procSDDFdescriptor( FAMILY_PROCEXIT, RECORD_TRACE,
		"Procedure Exit Trace", "Procedure Exit Trace Record" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = procSDDFdescriptor( FAMILY_PROCENTRY, RECORD_COUNT,
		"Procedure Entry Count", "Procedure Entry Count Record" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = procSDDFdescriptor( FAMILY_PROCEXIT, RECORD_COUNT,
		"Procedure Exit Count", "Procedure Exit Count Record" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	return;
}


/*
 *	procSDDFdescriptor:
 *	   Generate a SDDF binary format record descriptor for the
 *	   full trace class of events in either the procedure entry
 *	   or procedure exit family of events.  Patterned after the
 *	   instrumentation library internal function sddfDescriptor.
 */

TR_RECORD *
procSDDFdescriptor( recordFamily, recordType, recordName, recordDescription )

int	recordFamily;
int	recordType;
char	*recordName;
char	*recordDescription;
{
	static TR_RECORD	traceRecord;
	static char		recordBuffer[ 1024 ];
	char			*recordPointer;

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( recordFamily == FAMILY_PROCENTRY ) ||
	       ( recordFamily == FAMILY_PROCEXIT )
	      );

	Assert(
	       ( recordType == RECORD_TRACE ) ||
	       ( recordType == RECORD_COUNT )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( recordFamily == FAMILY_PROCENTRY ) || \
	       ( recordFamily == FAMILY_PROCEXIT ) \
	      );

	Assert( \
	       ( recordType == RECORD_TRACE ) || \
	       ( recordType == RECORD_COUNT ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

	/* Allow space at the beginning of the record for the packet	    */
	/* length, which will be computed after the packet is complete.	    */

	recordPointer = recordBuffer;
	sddfWriteInteger( &recordPointer, 0 );

	/* Write the record type descriptor.				    */

	sddfWriteInteger( &recordPointer, PKT_DESCRIPTOR );

	/* Write the record tag descriptor.				    */

	sddfWriteInteger( &recordPointer, ( recordFamily | recordType ));

	/* Write the record name.					    */

	sddfWriteString( &recordPointer, recordName );

	/* Write the record attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the record attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "description" );

	sddfWriteString( &recordPointer, recordDescription );

	/* Write the record field count.				    */

	switch (( recordFamily | recordType )) {

	case FAMILY_PROCENTRY | RECORD_TRACE:
		sddfWriteInteger( &recordPointer, 7 );
		break;

	case FAMILY_PROCENTRY | RECORD_COUNT:
		sddfWriteInteger( &recordPointer, 8 );
		break;

	case FAMILY_PROCEXIT | RECORD_TRACE:
		sddfWriteInteger( &recordPointer, 11 );
		break;

	case FAMILY_PROCEXIT | RECORD_COUNT:
		sddfWriteInteger( &recordPointer, 12 );
		break;
	}


	/* FIELD 1:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Timestamp" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Time" );

	sddfWriteString( &recordPointer, "Timestamp" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 1 );


	/* FIELD 2:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Seconds" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Seconds" );

	sddfWriteString( &recordPointer, "Floating Point Timestamp" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, DOUBLE );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 3:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Event Identifier" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "ID" );

	sddfWriteString( &recordPointer, "Event ID" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 4:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Processor Number" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Node" );

	sddfWriteString( &recordPointer, "Processor number" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 5:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Procedure Index" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Procedure" );

	sddfWriteString( &recordPointer, "Procedure Index" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 6:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Source Byte" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Byte" );

	sddfWriteString( &recordPointer, "Source Byte Offset" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 7:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Source Line" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Line" );

	sddfWriteString( &recordPointer, "Source Line Number" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 8:							    */

	if ( recordFamily == FAMILY_PROCEXIT ) {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Inclusive Duration" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "Inclusive Duration" );

		sddfWriteString( &recordPointer, "Inclusive Procedure Duration" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, INTEGER );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 1 );

	}


	/* FIELD 9:							    */

	if ( recordFamily == FAMILY_PROCEXIT ) {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Inclusive Seconds" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "Inclusive Seconds" );

		sddfWriteString( &recordPointer, "Floating Point Inclusive Duration" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, DOUBLE );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 0 );

	}


	/* FIELD 10:							    */

	if ( recordFamily == FAMILY_PROCEXIT ) {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Exclusive Seconds" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "Exclusive Seconds" );

		sddfWriteString( &recordPointer, "Floating Point Exclusive Duration" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, DOUBLE );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 0 );

	}


	/* FIELD 11:							    */

	if ( recordFamily == FAMILY_PROCEXIT ) {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Exclusive Duration" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "Exclusive Duration" );

		sddfWriteString( &recordPointer, "Exclusive Procedure Duration" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, INTEGER );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 1 );

	}


	/* FIELD 12 (8 for FAMILY_PROCENTRY):				    */

	if ( recordType == RECORD_COUNT ) {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Event Count" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "Count" );

		sddfWriteString( &recordPointer, "Event Count" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, INTEGER );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 0 );

	}


	/* The entire record descriptor packet has been written.	    */

	traceRecord.recordLength = recordPointer - recordBuffer;
	traceRecord.recordContents = recordBuffer;

	/* Finally, copy the completed packet length at the beginning	    */
	/* of the record itself.					    */

	recordPointer = recordBuffer;
	sddfWriteInteger( &recordPointer, traceRecord.recordLength );

	return & traceRecord;
}


/*
 *	procEventRecord:
 *	   This function generates trace records for events which are
 *	   to produce procedure entry or exit event family trace records.
 *	   Patterned after the instrumentation library internal functions
 *	   externalEventRecord and sddfRecord.
 */

TR_RECORD *
procEventRecord( recordType, eventPointer, timeStamp,
		dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	static TR_RECORD		traceRecord;
	static char			*recordBuffer = (char *) 0;
	static int			bufferLength = 0;
	struct procEntryTraceRecordData	*entryTraceRecordHeader;
	struct procEntryCountRecordData	*entryCountRecordHeader;
	struct procExitTraceRecordData	*exitTraceRecordHeader;
	struct procExitCountRecordData	*exitCountRecordHeader;
	int				procIndex;
	int				recordFamily;

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( recordType == RECORD_TRACE ) ||
	       ( recordType == RECORD_COUNT )
	      );

	Assert(
	       ( eventPointer != (TR_EVENT *) 0 ) &&
	       ( dataLength == 0 )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( recordType == RECORD_TRACE ) || \ 
	       ( recordType == RECORD_COUNT ) \
	      );

	Assert( \
	       ( eventPointer != (TR_EVENT *) 0 ) && \
	       ( dataLength == 0 ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

#ifdef DEBUG
	fprintf( debugFile, "procEventRecord\n" );
	fflush( debugFile );
#endif /* DEBUG */

	/* Determine whether this is a procedure entry or a procedure	    */
	/* exit family event by lookup in the procedure event ID matching.  */

	procIndex = findProcEvent( eventPointer->eventID );

	if ( procIndex < 0 )
		return nullRecordFunction( recordType, eventPointer,
					timeStamp, dataPointer, dataLength );

	if ( procEvents[ procIndex ].entryID == eventPointer->eventID )
		recordFamily = FAMILY_PROCENTRY;
	else
		recordFamily = FAMILY_PROCEXIT;

	/* The time stamp stored in the event descriptor will be used	    */
	/* unless one is specified in the timeStamp parameter.		    */

	if ( clockCompare( timeStamp, noSuchClock ) == 0 )
		timeStamp = eventPointer->eventLast;

	/* Determine how many bytes of storage will be needed for the	    */
	/* contents of the trace record.				    */

	switch (( recordFamily | recordType )) {

	case FAMILY_PROCENTRY | RECORD_TRACE:
		traceRecord.recordLength = sizeof *entryTraceRecordHeader;
		break;

	case FAMILY_PROCENTRY | RECORD_COUNT:
		traceRecord.recordLength = sizeof *entryCountRecordHeader;
		break;

	case FAMILY_PROCEXIT | RECORD_TRACE:
		traceRecord.recordLength = sizeof *exitTraceRecordHeader;
		break;

	case FAMILY_PROCEXIT | RECORD_COUNT:
		traceRecord.recordLength = sizeof *exitCountRecordHeader;
		break;
	}

	/* If there is a previously-allocated buffer and its size will	    */
	/* hold this record, re-use the buffer.  Otherwise, deallocate	    */
	/* the buffer (if allocated) and allocate a bigger one.		    */

	if ( bufferLength < traceRecord.recordLength ) {

		if ( recordBuffer != (char *) 0 )
			TRfreeBuffer( recordBuffer );

		recordBuffer = TRgetBuffer( traceRecord.recordLength );

		if ( recordBuffer == (char *) 0 )
			TRfailure(
			     "cannot allocate storage for trace record" );

		bufferLength = traceRecord.recordLength;
	}

	traceRecord.recordContents = recordBuffer;

	/* Load the trace record fields into the allocated buffer	    */

	switch (( recordFamily | recordType )) {

	case FAMILY_PROCENTRY | RECORD_TRACE:
		entryTraceRecordHeader = (struct procEntryTraceRecordData *)
							recordBuffer;
		entryTraceRecordHeader->packetLength =
				traceRecord.recordLength;
		entryTraceRecordHeader->packetType = PKT_DATA;
		entryTraceRecordHeader->packetTag = recordFamily | recordType;
		entryTraceRecordHeader->clockDimension = sizeof(CLOCK)/sizeof(int);
		entryTraceRecordHeader->timeStamp = timeStamp;
		entryTraceRecordHeader->seconds = clockToSeconds( timeStamp );
		entryTraceRecordHeader->eventID = eventPointer->eventID;
		entryTraceRecordHeader->nodeNumber = TRgetNode();
		entryTraceRecordHeader->procIndex = procIndex;
		entryTraceRecordHeader->sourceByte = procByteOffset;
		entryTraceRecordHeader->sourceLine = procLineNumber;

		pushProcEntry( procIndex, timeStamp );
		break;

	case FAMILY_PROCENTRY | RECORD_COUNT:
		entryCountRecordHeader = (struct procEntryCountRecordData *)
							recordBuffer;
		entryCountRecordHeader->packetLength =
				traceRecord.recordLength;
		entryCountRecordHeader->packetType = PKT_DATA;
		entryCountRecordHeader->packetTag = recordFamily | recordType;
		entryCountRecordHeader->clockDimension = sizeof(CLOCK)/sizeof(int);
		entryCountRecordHeader->timeStamp = timeStamp;
		entryCountRecordHeader->seconds = clockToSeconds( timeStamp );
		entryCountRecordHeader->eventID = eventPointer->eventID;
		entryCountRecordHeader->nodeNumber = TRgetNode();
		entryCountRecordHeader->procIndex = procIndex;
		entryCountRecordHeader->sourceByte = procByteOffset;
		entryCountRecordHeader->sourceLine = procLineNumber;
		entryCountRecordHeader->eventCount = eventPointer->eventCount;

		pushProcEntry( procIndex, timeStamp );
		break;

	case FAMILY_PROCEXIT | RECORD_TRACE:
		exitTraceRecordHeader = (struct procExitTraceRecordData *)
							recordBuffer;
		exitTraceRecordHeader->packetLength =
				traceRecord.recordLength;
		exitTraceRecordHeader->packetType = PKT_DATA;
		exitTraceRecordHeader->packetTag = recordFamily | recordType;
		exitTraceRecordHeader->clockDimension = sizeof(CLOCK)/sizeof(int);
		exitTraceRecordHeader->timeStamp = timeStamp;
		exitTraceRecordHeader->seconds = clockToSeconds( timeStamp );
		exitTraceRecordHeader->eventID = eventPointer->eventID;
		exitTraceRecordHeader->nodeNumber = TRgetNode();
		exitTraceRecordHeader->procIndex = procIndex;
		exitTraceRecordHeader->sourceByte = procByteOffset;
		exitTraceRecordHeader->sourceLine = procLineNumber;
		exitTraceRecordHeader->inclDurationDim = sizeof(CLOCK)/sizeof(int);
		exitTraceRecordHeader->exclDurationDim = sizeof(CLOCK)/sizeof(int);

		/* The popProcEntry function fills in the inclusiveDuration */
		/* and the exclusiveDuration fields of the record.	    */

		popProcEntry( procIndex, timeStamp,
			    & exitTraceRecordHeader->inclusiveDuration,
			    & exitTraceRecordHeader->exclusiveDuration );

		exitTraceRecordHeader->inclusiveSeconds =
		    clockToSeconds( exitTraceRecordHeader->inclusiveDuration );
		exitTraceRecordHeader->exclusiveSeconds =
		    clockToSeconds( exitTraceRecordHeader->exclusiveDuration );

		break;

	case FAMILY_PROCEXIT | RECORD_COUNT:
		exitCountRecordHeader = (struct procExitCountRecordData *)
							recordBuffer;
		exitCountRecordHeader->packetLength =
				traceRecord.recordLength;
		exitCountRecordHeader->packetType = PKT_DATA;
		exitCountRecordHeader->packetTag = recordFamily | recordType;
		exitCountRecordHeader->clockDimension = sizeof(CLOCK)/sizeof(int);
		exitCountRecordHeader->timeStamp = timeStamp;
		exitCountRecordHeader->seconds = clockToSeconds( timeStamp );
		exitCountRecordHeader->eventID = eventPointer->eventID;
		exitCountRecordHeader->nodeNumber = TRgetNode();
		exitCountRecordHeader->procIndex = procIndex;
		exitCountRecordHeader->sourceByte = procByteOffset;
		exitCountRecordHeader->sourceLine = procLineNumber;
		exitCountRecordHeader->eventCount = eventPointer->eventCount;
		exitCountRecordHeader->inclDurationDim = sizeof(CLOCK)/sizeof(int);
		exitCountRecordHeader->exclDurationDim = sizeof(CLOCK)/sizeof(int);

		/* The popProcEntry function fills in the inclusiveDuration */
		/* and the exclusiveDuration fields of the record.	    */

		popProcEntry( procIndex, timeStamp,
			    & exitCountRecordHeader->inclusiveDuration,
			    & exitCountRecordHeader->exclusiveDuration );

		exitCountRecordHeader->inclusiveSeconds =
		    clockToSeconds( exitCountRecordHeader->inclusiveDuration );
		exitCountRecordHeader->exclusiveSeconds =
		    clockToSeconds( exitCountRecordHeader->exclusiveDuration );

		break;
	}

	return & traceRecord;
}


/*
 *	findProcEvent:
 *	   Search the procedure entry/exit event ID matching data
 *	   structure for an event ID (either entry or exit) which is
 *	   the same as the argument eventID.  If found, return the
 *	   index from that table, which will be between 0 and
 *	   numberProcedures - 1, inclusive.  If not found, return -1;
 */

findProcEvent( eventID )

int	eventID;
{
	int	procIndex;

	for ( procIndex = 0; procIndex < numberProcedures; procIndex++ ) {
		if ( procEvents[ procIndex ].entryID == eventID ||
		     procEvents[ procIndex ].exitID == eventID )
			return procIndex;
	}

	return -1;
}


/*
 *	pushProcEntry:
 *	   Push a procedure entry time onto the corresponding stacks.
 */

pushProcEntry( procIndex, entryTime )

int	procIndex;
CLOCK	entryTime;
{
	PROC_ENTRY	*timeElement;

	Assert(( procIndex >= 0 ) && ( procIndex < numberProcedures ));

	/* Allocate a new procedure entry element and push it onto the	    */
	/* corresponding procedure's stack and the activeProc stack.	    */

	timeElement = (PROC_ENTRY *) TRgetBuffer( sizeof(PROC_ENTRY) );
	if ( timeElement == (PROC_ENTRY *) 0 )
		TRfailure( "unable to allocate procedure entry element" );

	timeElement->entryTime = entryTime;
	timeElement->subProcTime = zeroClock;
	timeElement->nextTime = procEntryStack[ procIndex ];
	procEntryStack[ procIndex ] = timeElement;
	timeElement->ancestorProc = activeProc;
	activeProc = timeElement;

	return;
}


/*
 *	popProcEntry:
 *	   Pop a procedure entry time off the corresponding stacks,
 *	   compute and store the procedure's inclusive and exclusive
 *	   durations.
 */

popProcEntry( procIndex, exitTime, inclusiveDuration, exclusiveDuration )

int	procIndex;
CLOCK	exitTime;
CLOCK	*inclusiveDuration;
CLOCK	*exclusiveDuration;
{
	PROC_ENTRY	*timeElement;
	CLOCK		duration;

	Assert(( procIndex >= 0 ) && ( procIndex < numberProcedures ));

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( inclusiveDuration != (CLOCK *) 0 ) &&
	       ( exclusiveDuration != (CLOCK *) 0 )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( inclusiveDuration != (CLOCK *) 0 ) && \
	       ( exclusiveDuration != (CLOCK *) 0 ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

	timeElement = procEntryStack[ procIndex ];

	/* If corresponding procedure's stack is empty, this is an error.   */

	if ( timeElement == (PROC_ENTRY *) 0 ) {
		*inclusiveDuration = noSuchClock;
		*exclusiveDuration = noSuchClock;
		return;
	}

	/* If this procedure is not the active procedure, the user has	    */
	/* made an error, probably by failing to issue the procedure	    */
	/* exit event for one or more subprocedures.  Unfortunately,	    */
	/* this user error is more difficult to recover from.  Simply	    */
	/* deleting the indicated procedure entry structure would lead	    */
	/* to dangling pointers.  Deleting all subprocedure entry event	    */
	/* structures is reasonable, but could lead to unexpected	    */
	/* behavior, especially if the user's only error was to get	    */
	/* confused about which event IDs correspond to which procedure	    */
	/* entries and exits.  Better to inform the user he has made a	    */
	/* big mistake which must be corrected.				    */

	if ( timeElement != activeProc )
		TRfailure( "Improperly nested procedure entry/exit events" );

	/* The active procedure now becomes the ancestor of the one being   */
	/* popped.  This effectively pops the current procedure off of	    */
	/* the activeProc stack.  Now pop the top entry time off of the	    */
	/* corresponding procedure's stack.				    */

	activeProc = timeElement->ancestorProc;

	procEntryStack[ procIndex ] = timeElement->nextTime;

	/* Compute the difference between this procedure's entry and	    */
	/* exit times; this is its inclusive duration.			    */

	duration = clockSubtract( exitTime, timeElement->entryTime );

	*inclusiveDuration = duration;

	/* This procedure's exclusive duration is less than its		    */
	/* inclusive duration by an amount equal to the sum of its	    */
	/* subprocedure's inclusive durations.				    */

	*exclusiveDuration = clockSubtract( duration,
					    timeElement->subProcTime );

	/* Likewise, add this procedure's inclusive duration to its	    */
	/* ancestor's sum of ITS subprocedure's inclusive durations.	    */

	if ( activeProc != (PROC_ENTRY *) 0 )
		activeProc->subProcTime = clockAdd( activeProc->subProcTime,
						    duration );

	/* Finally, deallocate this structure and return.		    */

	TRfreeBuffer( (char *) timeElement );

	return;
}


/*
 *	PabloTraceProc:
 *	   Interface function called by code instrumented with parser
 *	   to generate procedure trace record with location data.
 */

PabloTraceProc( eventID, byteOffset, lineNumber )

int		eventID;
unsigned long	byteOffset;
unsigned int	lineNumber;
{
	int	returnValue;

	procByteOffset = (long) byteOffset;
	procLineNumber = (int) lineNumber;

	returnValue = traceEvent( eventID, (char *) 0, 0 );

	procByteOffset = -1;
	procLineNumber = -1;

	return returnValue;
}


/*
 *	PabloCountProc:
 *	   Interface function called by code instrumented with parser
 *	   to generate procedure count record with location data.
 */

PabloCountProc( eventID, byteOffset, lineNumber )

int		eventID;
unsigned long	byteOffset;
unsigned int	lineNumber;
{
	int	returnValue;

	procByteOffset = (long) byteOffset;
	procLineNumber = (int) lineNumber;

	returnValue = countEvent( eventID );

	procByteOffset = -1;
	procLineNumber = -1;

	return returnValue;
}
