/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *          (R)
 * The Pablo    Performance Analysis Environment software is NOT in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1991-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * PABLO is a registered trademark of
 * The Board of Trustees of the University of Illinois
 * registered in the U.S. Patent and Trademark Office.
 *
 * Author:  Tara M. Madhyastha (tara@cs.uiuc.edu)
 * Contributing Author:  Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004,
 * by a grant from the Digital Equipment Corporation External Research
 * Program, and by a collaborative research agreement with the Intel
 * Supercomputer Systems Division.
 *
 */

#include <sys/uio.h>

#include "Assert.h"
#include "SystemDepend.h"
#include "SDDFparam.h"
#include "TraceParam.h"
#include "Trace.h"
#include "IOTrace.h"
#include "IntelIOTrace.h"


#ifdef SWAVE_IO
int produceOwnIOOutput = 1;  /* produce both I/O and SWAVE records */

int bWaitDim=-1;
int nReadDim=-1;

#endif SWAVE_IO


CLOCK   lastIOModeBegin = NOSUCHTIME;
CLOCK   lastiowaitBegin = NOSUCHTIME;
CLOCK   lastForcedFlushBegin = NOSUCHTIME;
CLOCK   lastLsizeBegin = NOSUCHTIME;

extern CLOCK lastReadBegin;
extern CLOCK lastWriteBegin;

initIOTrace()
{
	extern TR_RECORD	*readEventRecord();
	extern TR_RECORD	*asynchReadEventRecord();
	extern TR_RECORD	*writeEventRecord();
	extern TR_RECORD	*asynchWriteEventRecord();
	extern TR_RECORD	*seekEventRecord();
	extern TR_RECORD	*ioModeEventRecord();
	extern TR_RECORD        *iowaitEventRecord();
	extern TR_RECORD        *forcedFlushEventRecord();
	extern TR_RECORD        *lsizeEventRecord();
	extern TR_RECORD        *openEventRecord();
	extern TR_RECORD        *closeEventRecord();

	
	/* Perform message-tracing extension pre-initialization, if not	    */
	/* already done.						    */

	preInitIOTrace();

	/* Configure the trace record-generating function for all the
           IO events */

	setEventRecordFunction( readBeginID, readEventRecord );
	setEventRecordFunction( freadBeginID, readEventRecord );
	setEventRecordFunction( creadBeginID, readEventRecord );
	setEventRecordFunction( creadvBeginID, readEventRecord );
	setEventRecordFunction( ireadBeginID, asynchReadEventRecord );
	setEventRecordFunction( ireadvBeginID, asynchReadEventRecord );
	setEventRecordFunction( writeBeginID, writeEventRecord );
	setEventRecordFunction( fwriteBeginID, writeEventRecord );
	setEventRecordFunction( cwriteBeginID, writeEventRecord );
	setEventRecordFunction( cwritevBeginID, writeEventRecord );
	setEventRecordFunction( iwriteBeginID, asynchWriteEventRecord );
	setEventRecordFunction( iwritevBeginID, asynchWriteEventRecord );

	setEventRecordFunction( fseekBeginID, seekEventRecord );
	setEventRecordFunction( lseekBeginID, seekEventRecord );
	setEventRecordFunction( setIOModeBeginID, ioModeEventRecord );
	setEventRecordFunction( ioModeBeginID, ioModeEventRecord );
	setEventRecordFunction( iowaitBeginID, iowaitEventRecord);
	

	setEventRecordFunction( readEndID, readEventRecord );
	setEventRecordFunction( freadEndID, readEventRecord );
	setEventRecordFunction( creadEndID, readEventRecord );
	setEventRecordFunction( creadvEndID, readEventRecord );
	setEventRecordFunction( ireadEndID, asynchReadEventRecord );
	setEventRecordFunction( ireadvEndID, asynchReadEventRecord );
	setEventRecordFunction( writeEndID, writeEventRecord );
	setEventRecordFunction( fwriteEndID, writeEventRecord );
	setEventRecordFunction( cwriteEndID, writeEventRecord );
	setEventRecordFunction( cwritevEndID, writeEventRecord );
	setEventRecordFunction( iwriteEndID, asynchWriteEventRecord );
	setEventRecordFunction( iwritevEndID, asynchWriteEventRecord );

	setEventRecordFunction( fseekEndID, seekEventRecord );
	setEventRecordFunction( lseekEndID, seekEventRecord );
	setEventRecordFunction( setIOModeEndID, ioModeEventRecord );
	setEventRecordFunction( ioModeEndID, ioModeEventRecord );
	setEventRecordFunction( iowaitEndID, iowaitEventRecord);

	setEventRecordFunction( forcedFlushBeginID, forcedFlushEventRecord);
	setEventRecordFunction( forcedFlushEndID, forcedFlushEventRecord);

	setEventRecordFunction( lsizeBeginID, lsizeEventRecord );
	setEventRecordFunction( lsizeEndID, lsizeEventRecord );

	setEventRecordFunction( gopenBeginID, openEventRecord );
	setEventRecordFunction( gopenEndID, openEventRecord );

	setEventRecordFunction( closeBeginID, closeEventRecord );
	setEventRecordFunction( closeEndID, closeEventRecord );

	setEventRecordFunction( iodoneBeginID, iowaitEventRecord);
	setEventRecordFunction( iodoneEndID, iowaitEventRecord);
	

#ifdef DEBUG
	fprintf( debugFile, "ioTrace done\n" );
	fflush( debugFile );
#endif /* DEBUG */


#ifdef SWAVE_IO
        bSeekDim = SwaveRegisterDimension( "Seek Duration" );
        bReadDim = SwaveRegisterDimension( "Blocking Read Duration" );
        nReadDim = SwaveRegisterDimension( "Nonblocking Read Duration");
        bWaitDim = SwaveRegisterDimension( "I/O Wait Duration" );
#endif SWAVE_IO


	return SUCCESS;
}


/*
 *	writeIORecordDescriptors:
 *	   This function generates the record descriptors for the
 *	   message send/receive event families.  It will be invoked
 *	   by the instrumentation library initialization routines.
 *	   Patterned after instrumentation library internal function
 *	   writeRecordDescriptors.
 */

writeIORecordDescriptors()
{
	TR_RECORD	*recordPointer, *ioSDDFdescriptor();

#ifdef DEBUG
	fprintf( debugFile, "writeIORecordDescriptors\n" );
	fflush( debugFile );
#endif /* DEBUG */

#define putbytes(FAMILY, TYPE, name1, name2)  \
	recordPointer = ioSDDFdescriptor( FAMILY, \
					    TYPE,\
					    name1,\
					    name2);\
	putBytes( recordPointer->recordContents,\
		  (unsigned) recordPointer->recordLength );

	putbytes(FAMILY_READ, RECORD_TRACE, "Read", "IO Read");
	putbytes(FAMILY_ASYNCH_READ, RECORD_TRACE, "Asynch Read", "Asynch IO Read");
	putbytes(FAMILY_WRITE, RECORD_TRACE, "Write", "IO Write");
	putbytes(FAMILY_ASYNCH_WRITE, RECORD_TRACE, "Asynch Write", "Asynch IO Write");
	putbytes(FAMILY_SEEK, RECORD_TRACE, "Seek", "IO Seek");
	putbytes(FAMILY_OPEN, RECORD_TRACE, "Open", "IO Open");  
	putbytes(FAMILY_CLOSE, RECORD_TRACE, "Close", "IO File Close"); 
	putbytes(FAMILY_SPACE_SUMMARY, RECORD_TRACE, "Block Summary", "IO Block Summary");   
	putbytes(FAMILY_TIME_SUMMARY, RECORD_TRACE, "Time Summary", "IO Time Window Summary");   
	putbytes(FAMILY_LIFETIME, RECORD_TRACE, "File Lifetime Summary",  
		 "IO File lifetime summary "); 
	putbytes(FAMILY_HISTOGRAM, RECORD_TRACE, "Histogram Data", 
		 "IO Histogram Data"); 

	putbytes(FAMILY_IOMODE, RECORD_TRACE, "IOMode Manipulation", "IOMode Manipulation");
        putbytes(FAMILY_IOWAIT, RECORD_TRACE, "IO Wait", "IO Wait");
	putbytes(FAMILY_FLUSH, RECORD_TRACE, "IO Flush", "IO Flush");
	putbytes(FAMILY_LSIZE, RECORD_TRACE, "Lsize", "Lsize");
	return;
}


/*
 *	ioSDDFdescriptor:
 *	   Generate a SDDF binary format record descriptor for the
 *	   full trace class of events in any of the IO read or write
 *	   families of events.  Patterned after the
 *	   instrumentation library internal function sddfDescriptor.
 */

TR_RECORD *
ioSDDFdescriptor( recordFamily, recordType, recordName, recordDescription )

int	recordFamily;
int	recordType;
char	*recordName;
char	*recordDescription;
{
	static TR_RECORD	traceRecord;
	static char		recordBuffer[ 2048 ];
	char			*recordPointer;
	char			*stringPointer;
	char s[256];   /* temporary string buffer */


	Assert( recordType == RECORD_TRACE );

	/* Allow space at the beginning of the record for the packet	    */
	/* length, which will be computed after the packet is complete.	    */

	recordPointer = recordBuffer;
	sddfWriteInteger( &recordPointer, 0 );

	/* Write the record type descriptor.				    */

	sddfWriteInteger( &recordPointer, PKT_DESCRIPTOR );

	/* Write the record tag descriptor.				    */

	sddfWriteInteger( &recordPointer, ( recordFamily | recordType ));

	/* Write the record name.					    */

	sddfWriteString( &recordPointer, recordName );

	/* Write the record attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the record attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "description" );

	sddfWriteString( &recordPointer, recordDescription );

	/* Write the record field count.				    */

	switch ( recordFamily ) {
	case FAMILY_OPEN:
	  sddfWriteInteger( &recordPointer, OPEN_NUM_FIELDS);
	  break; 
	case FAMILY_READ:
	  sddfWriteInteger( &recordPointer, READ_NUM_FIELDS);
	  break;	  
	case FAMILY_WRITE:
	  sddfWriteInteger( &recordPointer, WRITE_NUM_FIELDS);
	  break;
	case FAMILY_ASYNCH_READ:
	  sddfWriteInteger( &recordPointer, ASYNCH_READ_NUM_FIELDS);	  
	  break;
	case FAMILY_ASYNCH_WRITE:
	  sddfWriteInteger( &recordPointer, ASYNCH_WRITE_NUM_FIELDS);	  
	  break;
	case FAMILY_SEEK:
	  sddfWriteInteger( &recordPointer, SEEK_NUM_FIELDS);
	  break;

	case FAMILY_CLOSE:
	  sddfWriteInteger( &recordPointer, CLOSE_NUM_FIELDS);
	  break;

	case FAMILY_SPACE_SUMMARY:
	  sddfWriteInteger( &recordPointer, SPACE_SUMMARY_NUM_FIELDS);
	  break;	  
	case FAMILY_TIME_SUMMARY:
	  sddfWriteInteger( &recordPointer, TIME_SUMMARY_NUM_FIELDS);
	  break;	  
	case FAMILY_LIFETIME:
	  sddfWriteInteger( &recordPointer, LIFETIME_NUM_FIELDS);
	  break;
	case FAMILY_HISTOGRAM:
	  sddfWriteInteger( &recordPointer, HISTOGRAM_NUM_FIELDS);
	  break;

	case FAMILY_IOMODE:
	  sddfWriteInteger( &recordPointer, IOMODE_NUM_FIELDS );
	  break;	  	  

        case FAMILY_IOWAIT:
	  sddfWriteInteger( &recordPointer, IOWAIT_NUM_FIELDS );
	  break;	  	  

	case FAMILY_FLUSH:
	  sddfWriteInteger( &recordPointer, FORCEFLUSH_NUM_FIELDS );
	  break;

	case FAMILY_LSIZE:
	  sddfWriteInteger( &recordPointer, LSIZE_NUM_FIELDS );
	  break;
	}

#define dofield(name,attributeCount,attString1,attString2,type,dimension) \
	sddfWriteString( &recordPointer, name ); \
	sddfWriteInteger( &recordPointer, attributeCount ); \
	sddfWriteString( &recordPointer, attString1 ); \
	sddfWriteString( &recordPointer, attString2 ); \
	sddfWriteInteger( &recordPointer, type ); \
	sddfWriteInteger( &recordPointer, dimension ); 

	switch (recordFamily) {
	case FAMILY_READ:
	case FAMILY_WRITE:
	  dofield("Timestamp",1,"Timestamp","Timestamp",INTEGER,1);
	  dofield("Seconds",1,"Seconds","Floating Point Timestamp",DOUBLE,0);
	  dofield("Duration",1,"Duration","Floating Point Duration",DOUBLE,0);
	  dofield("Event Identifier",1,"ID","Event ID", INTEGER,0);
	  dofield("Node Identifier",1,"Node","Node ID", INTEGER,0);	  
	  dofield("File ID",1,"File ID","File identifier (file descriptor or unit number) to which all fields in this record pertain", INTEGER,0);
	  dofield("Number Bytes",1,"Bytecount","Number Bytes",INTEGER,0);
	  dofield("Number Variables",1,"Varcount","Number Variables",INTEGER,0);
	  dofield("Cause",1,"Cause","IO Cause",INTEGER,0);
	  break;
	case FAMILY_ASYNCH_READ:
	case FAMILY_ASYNCH_WRITE:
	  dofield("Timestamp",1,"Timestamp","Timestamp",INTEGER,1);
	  dofield("Seconds",1,"Seconds","Floating Point Timestamp",DOUBLE,0);
	  dofield("Duration",1,"Duration","Floating Point Duration",DOUBLE,0);
	  dofield("Event Identifier",1,"ID","Event ID", INTEGER,0);
	  dofield("Node ID",1,"Node ID","Node (processor) identifier", INTEGER,0);	  
	  dofield("File ID",1,"File ID","File identifier (file descriptor or unit number) to which all fields in this record pertain", INTEGER,0);

	  dofield("Number Bytes",1,"Number Bytes","Count of bytes involved in the read/write operation",INTEGER,0);
	  dofield("Number Variables",1,"Number Variables","Number Variables involved in the read/write operation",INTEGER,0);
	  dofield("Cause",1,"Cause","User defined cause of IO -- -1 if none is specified, in the default case",INTEGER,0);
	  dofield("IO ID",1,"IO ID","I/O Identifier returned by the asnychronous call",INTEGER,0);

	  break;

	case FAMILY_SEEK:
	  dofield("Timestamp",1,"Timestamp","Timestamp",INTEGER,1);
	  dofield("Seconds",1,"Seconds","Floating Point Timestamp",DOUBLE,0);
	  dofield("Duration",1,"Duration","Floating Point Duration",DOUBLE,0);
	  dofield("Event Identifier",1,"ID","Event ID", INTEGER,0);
	  dofield("Node Identifier",1,"Node","Node ID", INTEGER,0);	  
	  dofield("File ID",1,"File ID","File identifier (file descriptor or unit number) to which all fields in this record pertain", INTEGER,0);
	  dofield("Offset",1,"Offset","Offset",INTEGER,0);
	  dofield("Ptrname",1,"Ptrname","Ptrname",INTEGER,0);
	  break;
	case FAMILY_SPACE_SUMMARY:	
	  dofield("Timestamp",1,"Timestamp","Timestamp",INTEGER,1);
	  dofield("Seconds",1,"Seconds","Floating Point Timestamp",DOUBLE,0);
	  dofield("Event ID",1,"Event ID","Event Identifier", INTEGER,0);
	  dofield("Closed",1,"Closed","Flag is true if this record was generated due to a file close, implying file ID will be reused",INTEGER,0);

	  dofield("Node ID",1,"Node ID","Processor(node) identifier", INTEGER,0);

	  dofield("File ID",1,"File ID","File identifier (file descriptor or unit number) to which all fields in this record pertain", INTEGER,0);

	  dofield("Read Count",1,"Read Count","Number of Read calls made in this time period",INTEGER,0);

	  dofield("Read Number Bytes",1,"Read Number Bytes","Total number of bytes read in this space interval",INTEGER,0);

	  dofield("Read Time",1,"Read Time","Total time spent in reads within this space interval",FLOAT,0);
	  dofield("Write Count",1,"Write Count","Number of Write calls made within this space interval",INTEGER,0);
	  dofield("Write Number Bytes",1,"Write Number Bytes","Total number of bytes written in this space interval",INTEGER,0);
	  dofield("Write Time",1,"Write Time","Total time spent in writes within this space interval",FLOAT,0);
	  dofield("Seek Count",1,"Seek Count","Number of Seeks made within this space interval",INTEGER,0);
	  dofield("Seek Number Bytes",1,"Seek Number Bytes","Total distance, in bytes, seeked within this space interval",INTEGER,0);
	  dofield("Seek Time",1,"Seek Time","Total time spent seeking within this space interval",FLOAT,0);
	  dofield("First Byte",1,"First Byte","First byte accessed within this space interval",INTEGER,0);
	  dofield("Last Byte",1,"Last Byte","Last byte accessed within this space interval",INTEGER,0);
	  break;
	case FAMILY_LIFETIME:
	  dofield("Timestamp",1,"Timestamp","Timestamp",INTEGER,1);
	  dofield("Seconds",1,"Seconds","Floating Point Timestamp",DOUBLE,0);
	  dofield("Event ID",1,"Event ID","Event identifier", INTEGER,0);
	  dofield("File ID",1,"File ID","File identifier (file descriptor or unit number) to which all fields in this record pertain", INTEGER,0);
	  dofield("Node Identifier",1,"Node","Node ID", INTEGER,0);	  
	  dofield("Dummy Int",1,"Dummy","Dummy", INTEGER,0);
	  dofield("File Lifetime",1,"File Lifetime","Time between end of file open and begin of file close events",DOUBLE,0);
	  dofield("Overhead",1,"Overhead","Time spent in file open and file close",DOUBLE,0);
	  dofield("Read Count",1,"Read Count","Total number of reads during file lifetime",INTEGER,0);
	  dofield("Read Number Bytes",1,"Read Number Bytes","Total number of bytes read during file lifetime",INTEGER,0);
	  dofield("Read Time",1,"Read Time","Total time spent reading during file lifetime",DOUBLE,0);
	  dofield("Write Count",1,"Write Count","Total number of writes during file lifetime",INTEGER,0);
	  dofield("Write Number Bytes",1,"Write Number Bytes","Total number of bytes written during file lifetime",INTEGER,0);
	  dofield("Write Time",1,"Write Time","Total time spent writing during file lifetime",DOUBLE,0);
	  dofield("Seek Count",1,"Seek Count","Total number of seeks during file lifetime",INTEGER,0);
	  dofield("Seek Number Bytes",1,"Seek Number Bytes","Total distance, in bytes, seeked during file lifetime",INTEGER,0);
	  dofield("Seek Time",1,"Seek Time","Total time spent seeking during file lifetime",DOUBLE,0);
	  dofield("Mode",1,"Mode","Mode in which file was opened",INTEGER,0);
	  dofield("File Name",1,"File Name","File name, as specified upon open", CHARACTER,1);
	  break;
	case FAMILY_TIME_SUMMARY:
	  dofield("Timestamp",1,"Timestamp","Timestamp",INTEGER,1);
	  dofield("Seconds",1,"Seconds","Floating Point Timestamp",DOUBLE,0);
	  dofield("Start",1,"Start","Time window supposedly began",DOUBLE,0);
	  dofield("Closed",1,"Closed","Flag is true if this record was generated due to a file close, implying file ID will be reused",INTEGER,0);
	  dofield("Event ID",1,"Event ID","Event Identifier", INTEGER,0);
	  dofield("Node ID",1,"Node ID","Processor/node Identifier", INTEGER,0);

	  dofield("File ID",1,"File ID","File identifier (file descriptor or unit number) to which all fields in this record pertain", INTEGER,0);

	  dofield("Read Count",1,"Read Count","Number of Read calls made in this time period",INTEGER,0);

	  dofield("Read Number Bytes",1,"Read Number Bytes","Number of bytes read in this time period",INTEGER,0);

	  dofield("Read Time",1,"Read Time","Total time spent reading in this time period",FLOAT,0);

	  dofield("Write Count",1,"Write Count","Number of write calls made in this time period",INTEGER,0);

	  dofield("Write Number Bytes",1,"Write Number Bytes","Number bytes written in this time period",INTEGER,0);

	  dofield("Write Time",1,"Write Time","Total time spent writing in this time period",FLOAT,0);

	  dofield("Seek Count",1,"Seek Count","Number of seeks made in this time period",INTEGER,0);

	  dofield("Seek Number Bytes",1,"Seek Number Bytes","Total distance (in bytes) seeked in this time period",INTEGER,0);

	  dofield("Seek Time",1,"Seek Time","Total time spent seeking during this time period",FLOAT,0);

	  dofield("First Byte",1,"First Byte","First byte accessed in this time period",INTEGER,0);

	  dofield("Last Byte",1,"Last Byte","Last byte accessed in this time period",INTEGER,0);

	  break;
	case FAMILY_OPEN:
	  dofield("Timestamp",1,"Timestamp","Timestamp",INTEGER,1);
	  dofield("Seconds",1,"Seconds","Floating Point Timestamp",DOUBLE,0);
	  dofield("Total IO Time",1,"Total Time","IO Time",DOUBLE,0);
	  sprintf(s, "Event identifier (%d = open, %d = fopen, %d = gopen)", 
		  openBeginID,
		  fopenBeginID, gopenBeginID);
	  dofield("Event ID",1,"Event ID", s , INTEGER,0);
	  dofield("Node ID",1,"Node ID","Node Identifier", INTEGER,0);	  
	  dofield("File ID",1,"File ID","File identifier (file descriptor or unit number) to which all fields in this record pertain", INTEGER,0);	  
	  dofield("Flags",1,"Flags","Flags supplied at file open",INTEGER,0);  
	  dofield("Mode",1,"Mode","Mode supplied at file open",INTEGER,0);  
	  dofield("File Name",1,"File Name","Name of file supplied at file open", CHARACTER,1);
	  break; 
	case FAMILY_HISTOGRAM:
	  dofield("Timestamp",1,"Timestamp","Timestamp",INTEGER,1);
	  dofield("Seconds",1,"Seconds","Floating Point Timestamp",DOUBLE,0); 
	  dofield("Event Identifier",1,"ID","Event ID", INTEGER,0);
	  dofield("Node Identifier",1,"Node","Node ID", INTEGER,0);	  
	  dofield("Histogram Data",1,"Histogram", "Histogram", INTEGER,1);
	  break;
	case FAMILY_CLOSE:
	  dofield("Timestamp",1,"Timestamp","Timestamp",INTEGER,1);
	  dofield("Seconds",1,"Seconds","Floating Point Timestamp",DOUBLE,0);
	  dofield("Total IO Time",1,"Total Time","IO Time",DOUBLE,0);
	  sprintf(s, "Event identifier (%d = close, %d = fclose)",
		  closeBeginID, fcloseBeginID);
	  dofield("Event ID",1,"Event ID",s, INTEGER,0);
	  dofield("Node Identifier",1,"Node","Node ID", INTEGER,0);	  
	  dofield("File Identifier",1,"File ID","File ID", INTEGER,0);
	  break; 
	case FAMILY_IOMODE:
	  dofield("Timestamp",1,"Timestamp","Timestamp",INTEGER,1);
	  dofield("Seconds",1,"Seconds","Floating Point Timestamp",DOUBLE,0);
	  sprintf(s, "Event Identifier (setiomode = %d, iomode = %d)", 
		  setIOModeBeginID, ioModeBeginID);
	  dofield("Event ID",1,"Event ID",s, INTEGER,0);
	  dofield("Node ID",1,"Node ID","Node(processor) identifier", INTEGER,0);
	  dofield("File ID",1,"File ID","File identifier (file descriptor or unit number)",INTEGER,0);
	  dofield("IO Mode",1,"IO Mode","IO Mode",INTEGER,0);
	  dofield("Duration",1,"Duration","Total time required by call",DOUBLE,0);
	  break;
	case FAMILY_FLUSH:
          dofield("Timestamp",1,"Timestamp","Timestamp",INTEGER,1);
          dofield("Seconds",1,"Seconds","Floating Point Timestamp",DOUBLE,0);
	  dofield("Duration",1,"Duration","Total time required by call",FLOAT,0);
          dofield("Event Identifier",1,"ID","Event ID", INTEGER,0);
          dofield("Node ID",1,"Node ID","Node(processor) identifer", INTEGER,0);
	  dofield("File ID",1,"File ID","File identifier (file descriptor or unit number)",INTEGER,0);
	  break;
       case FAMILY_IOWAIT:
          dofield("Timestamp",1,"Timestamp","Timestamp",INTEGER,1);
          dofield("Seconds",1,"Seconds","Floating Point Timestamp",DOUBLE,0);
	  sprintf(s, "Event Identifier (iowait = %d, iodone = %d)", 
		  iowaitBeginID, iodoneBeginID);
          dofield("Event ID",1,"Event ID",s, INTEGER,0);
          dofield("Processor Number",1,"Proc ID","Proc ID", INTEGER,0);
          dofield("IO Identifier",1,"IO ID","IO ID",INTEGER,0);
          dofield("Result",1,"Result","Value returned, if this record is generated by iodone",INTEGER,0);
          dofield("Total IO Time",1,"Total Time","IO Time",FLOAT,0);
          break;
	case FAMILY_LSIZE:
	  dofield("Timestamp",1,"Timestamp","Timestamp",INTEGER,1);
	  dofield("Seconds",1,"Seconds","Floating Point Timestamp",DOUBLE,0);
	  dofield("Duration",1,"Duration","Floating Point Duration",DOUBLE,0);
	  dofield("Event ID",1,"Event ID","Event Identifier", INTEGER,0);
	  dofield("Node ID",1,"Node ID","Node Identifier", INTEGER,0);	  
	  dofield("File ID",1,"File ID","File identifier (file descriptor or unit number)", INTEGER, 0);
	  dofield("Offset",1,"Offset","Offset",INTEGER,0);
	  dofield("Whence",1,"Whence","Whence",INTEGER,0);
	  dofield("Size",1,"Size","Size",INTEGER,0);
	  break;
	}

	/* The entire record descriptor packet has been written.	    */

	traceRecord.recordLength = recordPointer - recordBuffer;
	traceRecord.recordContents = recordBuffer;

	/* Finally, copy the completed packet length at the beginning	    */
	/* of the record itself.					    */

	recordPointer = recordBuffer;
	sddfWriteInteger( &recordPointer, traceRecord.recordLength );

	return & traceRecord;
}



int isIOModeBegin(eventID)
     int eventID;
{
  return (eventID == ioModeBeginID);
}

int isIOModeEnd(eventID)
     int eventID;
{
  return (eventID == ioModeEndID);
}

int isSetIOModeBegin(eventID)
     int eventID;
{
  return(eventID == setIOModeBeginID);
}

int isSetIOModeEnd(eventID)
     int eventID;
{
  return(eventID == setIOModeEndID);
}



TR_RECORD *
ioModeEventRecord( recordType, eventPointer, timeStamp,
			  dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	static TR_RECORD			traceRecord;
	static struct ioModeRecordData	ioModeRecordHeader;

	/* This structure defines how the send arguments are passed in	    */
	/* as user data, pointed to by dataPointer.			    */

	struct Args {
		long	filepointer;
		long	mode;
	}					*modeArgs;


	/* The time stamp stored in the event descriptor will be used	    */
	/* unless one is specified in the timeStamp parameter.		    */

	if ( clockCompare( timeStamp, noSuchClock ) == 0 )
		timeStamp = eventPointer->eventLast;


	switch(eventPointer->eventID) {
	case setIOModeBeginID:
	  modeArgs = (struct Args *) dataPointer;
	  lastIOModeBegin = timeStamp;
	  ioModeRecordHeader.timeStamp = timeStamp;
	  ioModeRecordHeader.seconds = clockToSeconds( timeStamp );
	  ioModeRecordHeader.fileID = modeArgs->filepointer;
	  ioModeRecordHeader.ioMode = modeArgs->mode;
	  ioModeRecordHeader.eventID = eventPointer->eventID;
	  break;
	case ioModeBeginID:
	  lastIOModeBegin = timeStamp;
	  ioModeRecordHeader.timeStamp = timeStamp;
	  ioModeRecordHeader.seconds = clockToSeconds( timeStamp );
	  ioModeRecordHeader.fileID = *((long *) dataPointer);
	  ioModeRecordHeader.eventID = eventPointer->eventID;
	  break;
	case setIOModeEndID:
	case ioModeEndID:
	  if (eventPointer->eventID == ioModeEndID) {
	    ioModeRecordHeader.ioMode = (*(long *) dataPointer);
	  }
	  ioModeRecordHeader.nodeID = TRgetNode();
	  ioModeRecordHeader.totalTime =  clockToSeconds(clockSubtract(
					       timeStamp, lastIOModeBegin));
	  traceRecord.recordLength = sizeof ioModeRecordHeader;
	  traceRecord.recordContents = (char *) &ioModeRecordHeader;
	  ioModeRecordHeader.packetLength = traceRecord.recordLength;
	  ioModeRecordHeader.packetType = PKT_DATA;
	  ioModeRecordHeader.packetTag = FAMILY_IOMODE | recordType;
	  ioModeRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
	  lastIOModeBegin = noSuchClock;

	  returnRecord(&traceRecord);

	  /* NOTREACHED */
	  break;
	default:
	  fprintf(stderr, "ioModeEventRecord: unknown eventID %d\n", 
		  eventPointer->eventID);
	  break;
	}
	return nullRecordFunction();
}


TR_RECORD *
iowaitEventRecord( recordType, eventPointer, timeStamp,
                          dataPointer, dataLength )

int             recordType;
TR_EVENT        *eventPointer;
CLOCK           timeStamp;
char            *dataPointer;
unsigned int    dataLength;
{
  static TR_RECORD                        traceRecord;
  static struct iowaitRecordData     iowaitRecordHeader;
  CLOCK duration;

  /* The time stamp stored in the event descriptor will be used       */
  /* unless one is specified in the timeStamp parameter.              */
  
  if ( clockCompare( timeStamp, noSuchClock ) == 0 )
    timeStamp = eventPointer->eventLast;

  switch (eventPointer->eventID) {
  case iowaitBeginID:
  case iodoneBeginID:
    lastiowaitBegin = timeStamp;
    iowaitRecordHeader.timeStamp = timeStamp;
    iowaitRecordHeader.seconds = clockToSeconds( timeStamp );
    iowaitRecordHeader.eventID = eventPointer->eventID;
    iowaitRecordHeader.ioID = *((int *) dataPointer);
    break;
	  
  case iowaitEndID:
  case iodoneEndID:
    traceRecord.recordLength = sizeof iowaitRecordHeader;
    traceRecord.recordContents = (char *) &iowaitRecordHeader;
    iowaitRecordHeader.packetLength = traceRecord.recordLength;
    iowaitRecordHeader.packetType = PKT_DATA;
    iowaitRecordHeader.packetTag = FAMILY_IOWAIT | recordType;
    iowaitRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
    iowaitRecordHeader.nodeID = TRgetNode();
    iowaitRecordHeader.result = *((int *) dataPointer);
    duration =  (clockCompare( lastiowaitBegin, noSuchClock ) == 0 )
      ? noSuchClock : clockSubtract( timeStamp, lastiowaitBegin ) ; 
    iowaitRecordHeader.duration = clockToSeconds(duration);
#ifdef SWAVE_IO
    SwaveRegisterValue(bWaitDim,
		       iowaitRecordHeader.seconds,
		       iowaitRecordHeader.totalTime);
#endif

    lastiowaitBegin = noSuchClock;

    
    returnRecord( &traceRecord);
    /* NOTREACHED */
    break;
  default:
    fprintf(stderr, "ioWaitEventRecord: unknown eventID %d\n", 
	    eventPointer->eventID);
    break;
  }
  return nullRecordFunction();
}


TR_RECORD *
asynchWriteEventRecord( recordType, eventPointer, timeStamp,
		     dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
  static TR_RECORD			traceRecord;
  static struct asynchWriteRecordData	   writeRecordHeader;

  struct intelRWBeginArgs *writeArguments;
  struct intelAsynchRWEndArgs *writeEndArguments;

  writeArguments = (struct intelRWBeginArgs *) dataPointer;

  if ( clockCompare( timeStamp, noSuchClock ) == 0 )
    timeStamp = eventPointer->eventLast;


  if (eventPointer->eventID == iwriteEndID ||
      eventPointer->eventID == iwritevEndID ){

    writeEndArguments = (struct intelAsynchRWEndArgs *) dataPointer;
    writeRecordHeader.duration = clockToSeconds(
				clockSubtract(timeStamp, lastWriteBegin));
    writeRecordHeader.numBytes = writeEndArguments->numBytes;
    writeRecordHeader.ioID = writeEndArguments->ioID;

    lastWriteBegin = noSuchClock;

    return & traceRecord;

  } else if (eventPointer->eventID == iwriteBeginID ||
	     eventPointer->eventID == iwritevBeginID ) {

    traceRecord.recordLength = sizeof writeRecordHeader;
    traceRecord.recordContents = (char *) &writeRecordHeader;

    writeRecordHeader.packetLength = traceRecord.recordLength;
    writeRecordHeader.packetType = PKT_DATA;
    writeRecordHeader.packetTag = FAMILY_ASYNCH_WRITE | recordType;
    writeRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
    writeRecordHeader.timeStamp = timeStamp;
    writeRecordHeader.seconds = clockToSeconds( timeStamp );
    writeRecordHeader.eventID = eventPointer->eventID;
    writeRecordHeader.nodeID = TRgetNode();
    writeRecordHeader.filepointer = writeArguments->filepointer;
    writeRecordHeader.numVariables = writeArguments->numVariables;
    writeRecordHeader.cause = writeArguments->cause;
    lastWriteBegin = timeStamp;
    return(nullRecordFunction(recordType,eventPointer,timeStamp,
			      dataPointer,dataLength));
  } else {
    fprintf(stderr, "asynchWriteEventRecord: unknown eventID %d\n", 
	    eventPointer->eventID);
  }
  return(nullRecordFunction(recordType,eventPointer,timeStamp,
			    dataPointer,dataLength));
}



TR_RECORD *
forcedFlushEventRecord( recordType, eventPointer, timeStamp,
		     dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;

unsigned int	dataLength;
{
  static TR_RECORD			   traceRecord;
  static struct forcedFlushRecordData	   forcedFlushRecordHeader;

  if ( clockCompare( timeStamp, noSuchClock ) == 0 )
    timeStamp = eventPointer->eventLast;

  if (eventPointer->eventID == forcedFlushEndID) {

    forcedFlushRecordHeader.duration = clockToSeconds(
				clockSubtract(timeStamp, 
					      lastForcedFlushBegin));
    forcedFlushRecordHeader.filepointer = *((int *) dataPointer);
    lastForcedFlushBegin = noSuchClock;
    return & traceRecord;

  } else if (eventPointer->eventID == forcedFlushBeginID ) {

    traceRecord.recordLength = sizeof forcedFlushRecordHeader;
    traceRecord.recordContents = (char *) &forcedFlushRecordHeader;

    forcedFlushRecordHeader.packetLength = traceRecord.recordLength;
    forcedFlushRecordHeader.packetType = PKT_DATA;
    forcedFlushRecordHeader.packetTag = FAMILY_FLUSH | recordType;
    forcedFlushRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
    forcedFlushRecordHeader.timeStamp = timeStamp;
    forcedFlushRecordHeader.seconds = clockToSeconds( timeStamp );
    forcedFlushRecordHeader.eventID = eventPointer->eventID;
    forcedFlushRecordHeader.nodeID = TRgetNode();
    lastForcedFlushBegin = timeStamp;
    return(nullRecordFunction(recordType,eventPointer,timeStamp,
			      dataPointer,dataLength));
  } else {
    fprintf(stderr, "forcedFlushEventRecord: unknown eventID %d\n", 
	    eventPointer->eventID);
  }
  return(nullRecordFunction(recordType,eventPointer,timeStamp,
			    dataPointer,dataLength));
}




TR_RECORD *
lsizeEventRecord( recordType, eventPointer, timeStamp,
		     dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;

unsigned int	dataLength;
{
  static TR_RECORD			   traceRecord;
  static struct lsizeRecordData	   lsizeRecordHeader;

  struct intelLsizeBeginArgs   *lsizeArgs;


  if ( clockCompare( timeStamp, noSuchClock ) == 0 )
    timeStamp = eventPointer->eventLast;

  if (eventPointer->eventID == lsizeEndID) {

    lsizeRecordHeader.duration = clockToSeconds(
				clockSubtract(timeStamp, 
					      lastLsizeBegin));
    lsizeRecordHeader.size = *((int *) dataPointer);


    lastLsizeBegin = noSuchClock;
    return & traceRecord;

  } else if (eventPointer->eventID == lsizeBeginID ) {

    traceRecord.recordLength = sizeof lsizeRecordHeader;
    traceRecord.recordContents = (char *) &lsizeRecordHeader;

    lsizeArgs = (struct intelLsizeBeginArgs *)dataPointer;

    lsizeRecordHeader.filepointer = lsizeArgs->fileID;
    lsizeRecordHeader.offset = lsizeArgs->offset;
    lsizeRecordHeader.whence = lsizeArgs->whence;

    lsizeRecordHeader.packetLength = traceRecord.recordLength;
    lsizeRecordHeader.packetType = PKT_DATA;
    lsizeRecordHeader.packetTag = FAMILY_LSIZE | recordType;
    lsizeRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
    lsizeRecordHeader.timeStamp = timeStamp;
    lsizeRecordHeader.seconds = clockToSeconds( timeStamp );
    lsizeRecordHeader.eventID = eventPointer->eventID;
    lsizeRecordHeader.nodeID = TRgetNode();
    lastLsizeBegin = timeStamp;

    return(nullRecordFunction(recordType,eventPointer,timeStamp,
			      dataPointer,dataLength));
  } else {
    fprintf(stderr, "lsizeEventRecord: unknown eventID %d\n", 
	    eventPointer->eventID);
  }
  return(nullRecordFunction(recordType,eventPointer,timeStamp,
			    dataPointer,dataLength));
}


TR_RECORD *
asynchReadEventRecord( recordType, eventPointer, timeStamp,
			  dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
  static TR_RECORD			traceRecord;
  static struct asynchReadRecordData	readRecordHeader;

  struct intelRWBeginArgs *readArguments;
  struct intelAsynchRWEndArgs *readEndArguments;

  readArguments = (struct intelRWBeginArgs *) dataPointer;

  /* The time stamp stored in the event descriptor will be used	    */
  /* unless one is specified in the timeStamp parameter.		    */

  if ( clockCompare( timeStamp, noSuchClock ) == 0 )
    timeStamp = eventPointer->eventLast;

  if (eventPointer->eventID == ireadEndID ||
      eventPointer->eventID == ireadvEndID ) {

    readRecordHeader.duration = clockToSeconds(
			       clockSubtract(timeStamp,lastReadBegin));
    readEndArguments = (struct intelAsynchRWEndArgs *) dataPointer;
    readRecordHeader.numBytes = readEndArguments->numBytes;
    readRecordHeader.ioID = readEndArguments->ioID;
#ifdef SWAVE_IO
    SwaveRegisterValue(nReadDim,
		       readRecordHeader.seconds,
		       readRecordHeader.duration );
#endif

    lastReadBegin = noSuchClock;

    returnRecord(&traceRecord);

  } else if (eventPointer->eventID == ireadBeginID ||
	     eventPointer->eventID == ireadvBeginID )  {
    lastReadBegin = timeStamp;
    traceRecord.recordLength = sizeof readRecordHeader;
    traceRecord.recordContents = (char *) &readRecordHeader;

    /* Load the trace record fields into the allocated buffer	    */
    readRecordHeader.packetLength = traceRecord.recordLength;
    readRecordHeader.packetType = PKT_DATA;
    readRecordHeader.packetTag = FAMILY_ASYNCH_READ | recordType;
    readRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
    readRecordHeader.timeStamp = timeStamp;
    readRecordHeader.seconds = clockToSeconds( timeStamp );
    readRecordHeader.eventID = eventPointer->eventID;
    readRecordHeader.nodeID = TRgetNode();
    readRecordHeader.filepointer = readArguments->filepointer;
    readRecordHeader.numVariables = readArguments->numVariables;
    readRecordHeader.cause = readArguments->cause;

    return(nullRecordFunction(recordType,eventPointer,timeStamp,
			      dataPointer,dataLength));
  } else {
    fprintf(stderr, "asynchReadEventRecord: unknown eventID %d\n", 
	    eventPointer->eventID);
  }
  return(nullRecordFunction(recordType,eventPointer,timeStamp,
			    dataPointer,dataLength));
}



/* the C traceLSEEK is in the portable part of the library */

#ifdef FORTRAN_LIB

long tracelseek_(fd, offset, whence)
     int *fd, *offset, *whence;
{
  struct {
    long filepointer;
    long offset;
    long ptrname;
  } seekArgs;  

  int ret;

  seekArgs.filepointer = *fd;
  seekArgs.offset = *offset;
  seekArgs.ptrname = *whence;

  traceEvent(lseekBeginID, (char *) &seekArgs, sizeof seekArgs);
  ret = lseek_(fd,offset,whence);
  traceEvent(lseekEndID, (char *) 0, 0);
  return(ret);
}
#endif

void traceCREAD(fildes, buffer, nbytes)
     int fildes;
     char *buffer;
     unsigned int nbytes;
{
  struct intelRWBeginArgs readArgs;

  readArgs.filepointer = fildes;
  readArgs.numVariables = 1;
  readArgs.cause = -1;

  traceEvent(creadBeginID, (char *) &readArgs, sizeof(readArgs));
  cread(fildes, buffer, nbytes);
  traceEvent(creadEndID, (char *) &nbytes, sizeof(nbytes));
}

/* put this somewhere else -- does not link correctly */
#ifdef FORTRAN_LIB
void traceforflush_(unit)
     int *unit;
{
  traceEvent(forcedFlushBeginID, (char *) unit, sizeof(*unit));
  forflush_(unit);
  traceEvent(forcedFlushEndID, (char *) 0, 0);  
}

void tracecread_(fildes, buffer, nbytes)
     int *fildes;
     char *buffer;
     int *nbytes;
{
  struct intelRWBeginArgs readArgs;

  readArgs.filepointer = *fildes;
  readArgs.numVariables = 1;
  readArgs.cause = -1;

  traceEvent(creadBeginID, (char *) &readArgs, sizeof(readArgs));
  cread_(fildes, buffer, nbytes);
  traceEvent(creadEndID, (char *) nbytes, sizeof (nbytes) );
}
#endif

void traceCREADV(fildes, iov, iovcnt)
     int fildes;
     struct iovec *iov;
     int iovcnt;
{
  struct intelRWBeginArgs readArgs;
  int nbytes;

  readArgs.filepointer = fildes;
  readArgs.numVariables = iovcnt;
  readArgs.cause = -1;

  traceEvent(creadvBeginID, (char *) &readArgs, sizeof(readArgs));
  creadv(fildes, iov, iovcnt);
  nbytes = iov->iov_len * iovcnt; /* fudge the length because creadv 
				     does not return the number of bytes
				     read or written -- if an error occurs
				     it prints an error and terminates the
				     calling process */

  traceEvent(creadvEndID, (char *) &nbytes, sizeof(nbytes));
}


#ifdef FORTRAN_LIB
void tracecreadv_(fildes, iov, iovcnt)
     int *fildes;
     struct iovec *iov;
     int *iovcnt;
{
  struct intelRWBeginArgs readArgs;

  int nbytes;

  readArgs.filepointer = *fildes;
  readArgs.numVariables = *iovcnt;
  readArgs.cause = -1;

  traceEvent(creadvBeginID, (char *) &readArgs, sizeof(readArgs));
  creadv_(fildes, iov, iovcnt);
  nbytes = iov->iov_len * *iovcnt; /* fudge the length because creadv 
				     does not return the number of bytes
				     read or written -- if an error occurs
				     it prints an error and terminates the
				     calling process */

  traceEvent(creadvEndID, (char *) &nbytes, sizeof(nbytes));
}
#endif

void traceCWRITE(fildes, buffer, nbytes)
     int fildes;
     char *buffer;
     unsigned int nbytes;
{
  struct intelRWBeginArgs writeArgs;

  writeArgs.filepointer = fildes;
  writeArgs.numVariables = 1;
  writeArgs.cause = -1;

  traceEvent(cwriteBeginID, (char *) &writeArgs, sizeof(writeArgs));
  cwrite(fildes, buffer, nbytes);
  traceEvent(cwriteEndID, (char *) &nbytes, sizeof(nbytes));
}

#ifdef FORTRAN_LIB
void tracecwrite_(fildes, buffer, nbytes)
     int *fildes;
     char *buffer;
     int *nbytes;
{
  struct intelRWBeginArgs writeArgs;

  writeArgs.filepointer = *fildes;
  writeArgs.numVariables = 1;
  writeArgs.cause = -1;

  traceEvent(cwriteBeginID, (char *) &writeArgs, sizeof(writeArgs));
  cwrite_(fildes, buffer, nbytes);
  traceEvent(cwriteEndID, (char *) nbytes, sizeof(nbytes));

}
#endif


void traceCWRITEV(fildes, iov, iovcnt)
     int fildes;
     struct iovec *iov;
     int iovcnt;
{
  struct intelRWBeginArgs writeArgs;
  int nbytes;

  writeArgs.filepointer = fildes;
  writeArgs.numVariables = iovcnt;
  writeArgs.cause = -1;

  nbytes = iov->iov_len * iovcnt;

  traceEvent(cwritevBeginID, (char *) &writeArgs, sizeof(writeArgs));
  cwritev(fildes, iov, iovcnt);
  traceEvent(cwritevEndID, (char *) &nbytes, sizeof(nbytes));
}

#ifdef FORTRAN_LIB
void tracecwritev_(fildes, iov, iovcnt)
     int *fildes;
     struct iovec *iov;
     int *iovcnt;
{
  int nbytes;

  struct intelRWBeginArgs writeArgs;

  nbytes = iov->iov_len * *iovcnt;  

  writeArgs.filepointer = *fildes;
  writeArgs.numVariables = *iovcnt;
  writeArgs.cause = -1;

  traceEvent(cwritevBeginID, (char *) &writeArgs, sizeof(writeArgs));
  cwritev_(fildes, iov, iovcnt);
  traceEvent(cwritevEndID, (char *) &nbytes, sizeof(nbytes));

}
#endif

long traceIREAD(fildes, buffer, nbytes)
     int fildes;
     char *buffer;
     unsigned int nbytes;
{

  struct intelRWBeginArgs readArgs;
  struct intelAsynchRWEndArgs readEndArgs;


  readArgs.filepointer = fildes;
  readArgs.numVariables = 1;
  readArgs.cause = -1;

  readEndArgs.numBytes = nbytes;


  traceEvent(ireadBeginID, (char *) &readArgs, sizeof(readArgs));
  readEndArgs.ioID = iread(fildes, buffer, nbytes);
  traceEvent(ireadEndID, (char *) &readEndArgs, sizeof(readEndArgs));
  return(readEndArgs.ioID);
}

#ifdef FORTRAN_LIB
long traceiread_(fildes, buffer, nbytes)
     int *fildes;
     char *buffer;
     int *nbytes;
{
  struct intelRWBeginArgs readArgs;
  struct intelAsynchRWEndArgs readEndArgs;

  readArgs.filepointer = *fildes;
  readArgs.numVariables = 1;
  readArgs.cause = -1;

  readEndArgs.numBytes = *nbytes;

  traceEvent(ireadBeginID, (char *) &readArgs, sizeof(readArgs));
  readEndArgs.ioID = iread_(fildes, buffer, nbytes);
  traceEvent(ireadEndID, (char *) &readEndArgs, sizeof(readEndArgs));
  return(readEndArgs.ioID);

}
#endif

long traceIREADV(fildes, iov, iovcnt)
     int fildes;
     struct iovec *iov;
     int iovcnt;
{
  struct intelRWBeginArgs readArgs;
  struct intelAsynchRWEndArgs readEndArgs;

  readArgs.filepointer = fildes;
  readArgs.numVariables = iovcnt;
  readArgs.cause = -1;

  readEndArgs.numBytes = iov->iov_len * iovcnt;


  traceEvent(ireadBeginID, (char *) &readArgs, sizeof(readArgs));
  readEndArgs.ioID = ireadv(fildes, iov, iovcnt);
  traceEvent(ireadEndID, (char *) &readEndArgs, sizeof(readEndArgs));
  return(readEndArgs.ioID);
}

#ifdef FORTRAN_LIB
long traceireadv_(fildes, iov, iovcnt)
     int *fildes;
     struct iovec *iov;
     int *iovcnt;
{
  struct intelRWBeginArgs readArgs;
  struct intelAsynchRWEndArgs readEndArgs;

  readArgs.filepointer = *fildes;
  readArgs.numVariables = *iovcnt;
  readArgs.cause = -1;
  readEndArgs.numBytes = iov->iov_len * *iovcnt;

  traceEvent(ireadvBeginID, (char *) &readArgs, sizeof(readArgs));
  readEndArgs.ioID = iread_(fildes, iov, iovcnt);
  traceEvent(ireadvEndID, (char *) &readEndArgs, sizeof(readEndArgs));
  return(readEndArgs.ioID);
}
#endif


long traceIWRITE(fildes, buffer, nbytes)
     int fildes;
     char *buffer;
     unsigned int nbytes;
{
  long ret;

  struct intelRWBeginArgs writeArgs;
  struct intelAsynchRWEndArgs writeEndArgs;

  writeArgs.filepointer = fildes;
  writeArgs.numVariables = 1;
  writeArgs.cause = -1;

  writeEndArgs.numBytes = nbytes;

  traceEvent(iwriteBeginID, (char *) &writeArgs, sizeof(writeArgs));
  writeEndArgs.ioID = iwrite(fildes, buffer, nbytes);
  traceEvent(iwriteEndID, (char *) &writeEndArgs, sizeof(writeEndArgs));
  return(writeEndArgs.ioID);
}

#ifdef FORTRAN_LIB

long traceiwrite_(fildes, buffer, nbytes)
     int *fildes;
     char *buffer;
     int *nbytes;
{

  struct intelRWBeginArgs writeArgs;
  struct intelAsynchRWEndArgs writeEndArgs;

  writeArgs.filepointer = *fildes;
  writeArgs.numVariables = 1;
  writeArgs.cause = -1;

  writeEndArgs.numBytes = *nbytes;

  traceEvent(iwriteBeginID, (char *) &writeArgs, sizeof(writeArgs));
  writeEndArgs.ioID = iwrite_(fildes, buffer, nbytes);
  traceEvent(iwriteEndID, (char *) &writeEndArgs, sizeof(writeEndArgs));
  return(writeEndArgs.ioID);

}
#endif


long traceIWRITEV(fildes, iov, iovcount)
     int fildes;
     struct iovec *iov;
     int iovcount;
{
  struct intelRWBeginArgs writeArgs;
  struct intelAsynchRWEndArgs writeEndArgs;

  writeArgs.filepointer = fildes;
  writeArgs.numVariables = iovcount;
  writeArgs.cause = -1;

  writeEndArgs.numBytes = iov->iov_len * iovcount;

  traceEvent(iwritevBeginID, (char *) &writeArgs, sizeof(writeArgs));
  writeEndArgs.ioID  = iwritev(fildes, iov, iovcount);
  traceEvent(iwritevEndID, (char *) &writeEndArgs, sizeof(writeEndArgs));
  return(writeEndArgs.ioID);
}

#ifdef FORTRAN_LIB

long traceiwritev_(fildes, iov, iovcount)
     int *fildes;
     struct iovec *iov;
     int *iovcount;
{
  struct intelRWBeginArgs writeArgs;
  struct intelAsynchRWEndArgs writeEndArgs;

  writeArgs.filepointer = *fildes;
  writeArgs.numVariables = *iovcount;
  writeArgs.cause = -1;

  writeEndArgs.numBytes = iov->iov_len * *iovcount;

  traceEvent(iwritevBeginID, (char *) &writeArgs, sizeof(writeArgs));
  writeEndArgs.ioID = iwritev_(fildes, iov, iovcount);
  traceEvent(iwritevEndID, (char *) &writeEndArgs, sizeof(writeEndArgs));
  return(writeEndArgs.ioID);

}
#endif

long traceSETIOMODE(fildes,iomode)
     int fildes;
     int iomode;
{
  struct {
    long filepointer;
    long iomode;
  } setiomodeArgs;

  setiomodeArgs.filepointer = fildes;
  setiomodeArgs.iomode = iomode;


  traceEvent(setIOModeBeginID,(char *) &setiomodeArgs, sizeof(setiomodeArgs));
  setiomode(fildes,iomode);
  traceEvent(setIOModeEndID, (char *) 0, 0);
}


long traceIOMODE(fildes)
     int fildes;
{
  long ret;

  traceEvent(ioModeBeginID,(char *) &fildes, sizeof(int));
  ret = iomode(fildes);
  traceEvent(ioModeEndID, (char *) &ret, sizeof(ret));
  return(ret);
}


int traceLSIZE(fileID, offset, whence)
     int fileID;
     off_t offset;
     int whence;
{
  long ret;
  
  struct intelLsizeBeginArgs args;
  args.fileID = fileID;
  args.offset = offset;
  args.whence = whence;

  traceEvent(lsizeBeginID,(char *) &args, sizeof(args));
  ret = lsize(fileID, offset, whence);
  traceEvent(lsizeEndID, (char *) &ret, sizeof(ret));
  return(ret);
}

#ifdef FORTRAN_LIB

int tracelsize_(fileID, offset, whence)
     int *fileID;
     int *offset;
     int *whence;
{
  long ret;
  
  struct intelLsizeBeginArgs args;
  args.fileID = *fileID;
  args.offset = *offset;
  args.whence = *whence;

  traceEvent(lsizeBeginID,(char *) &args, sizeof(args));
  ret = lsize_(fileID, offset, whence);
  traceEvent(lsizeEndID, (char *) &ret, sizeof(ret));
  return(ret);
}





long tracesetiomode_(fildes, iomode)
     int *fildes;
     int *iomode;
{
  struct {
    long filepointer;
    long iomode;
  } setiomodeArgs;

  setiomodeArgs.filepointer = *fildes;
  setiomodeArgs.iomode = *iomode;


  traceEvent(setIOModeBeginID,(char *) &setiomodeArgs, sizeof(setiomodeArgs));
  setiomode_(fildes,iomode);
  traceEvent(setIOModeEndID, (char *) 0, 0);

}

long traceiomode_(fildes)
     int *fildes;
{
  long ret;

  traceEvent(ioModeBeginID,(char *) fildes, sizeof(fildes));
  ret = iomode_(fildes);
  traceEvent(ioModeEndID, (char *) &ret, sizeof(ret));
  return(ret);
}
#endif

/*
int trace3GOPEN(path, oflag, iomode)
     char *path;
     int oflag;
     int iomode;
{
  struct {
    int flags;
    int mode;
    char data[256];
  } openArgs;
  int fd;

  strcpy(openArgs.data,path);
  openArgs.flags = oflag;
  openArgs.mode = iomode;
  traceEvent(gopenBeginID, &openArgs, sizeof(openArgs));
  fd = gopen(path, oflag, iomode);
  traceEvent(gopenEndID, (char *) &fd, sizeof(fd));
  return(fd);
}
*/

int traceGOPEN(path, oflag, iomode,  mode)
     char *path;
     int oflag;
     int iomode;
     mode_t mode;
{
  struct {
    int flags;
    int mode;
    char data[256];
  } openArgs;
  int fd;

  strcpy(openArgs.data,path);
  openArgs.flags = oflag;
  openArgs.mode = iomode;
  traceEvent(gopenBeginID, &openArgs, sizeof(openArgs));
  fd = gopen(path, oflag, iomode, mode);
  traceEvent(gopenEndID, (char *) &fd, sizeof(fd));
  return(fd);
}

#ifdef FORTRAN_LIB
tracegopen_(unit, path , iomode)
     int *unit;
     char *path;
     int *iomode;
{
  
  struct {
    int flags;
    int mode;
    char data[256];
  } openArgs;

  strcpy(openArgs.data,path);
  openArgs.flags = *iomode;
  openArgs.mode = *iomode;
  traceEvent(gopenBeginID, &openArgs, sizeof(openArgs));
  gopen_(unit,path,iomode);
  traceEvent(gopenEndID, (char *) &unit, sizeof(unit));
}
#endif


#ifdef __PARAGON__
void traceIOWAIT(id)
     long id;
{
  int ret;
  int ioID = id;
  ret = -1;

  traceEvent(iowaitBeginID, (char *) &ioID, sizeof(int));
  iowait(id);
  traceEvent(iowaitEndID, (char *) &ret, sizeof(ret));
}

void traceiowait_(id)
     int *id;
{
  traceIOWAIT(*id);
}

#else
long traceIOWAIT(id)
     long id;
{
  int ret;
  int ioID = id;

  traceEvent(iowaitBeginID, (char *) &ioID, sizeof(int));
  ret = iowait(id);
  traceEvent(iowaitEndID, (char *) &ret, sizeof(ret));
  return(ret);
}

#ifdef FORTRAN_LIB
long traceiowait_(id)
     int *id;
{
  return(traceIOWAIT(*id));
}
#endif

#endif


long traceIODONE(id)
     long id;
{
  int ret;
  int ioID = id;
  traceEvent(iodoneBeginID, (char *) &ioID, sizeof(int));
  ret = iodone(id);
  traceEvent(iodoneEndID, (char *) &ret, sizeof(int));  
  return(ret);
}

#ifdef FORTRAN_LIB

long traceiodone_(id)
     int *id;
{
  return(traceIODONE(*id));
}

#endif
/* the "is" routines below provide a portable way of checking the
"nature" of system dependent event ids, which sometimes can be treated
alike. A new event id should be added here and in similar calls in all
system dependent files, not in the portable part of the library */


/* is this eventID a close begin? */

int isCloseBeginEvent(eventID) 
     int eventID;
{
  return((eventID == closeBeginID) ||
	 (eventID == fcloseBeginID));
}

int isCloseEndEvent(eventID) 
     int eventID;
{
  return((eventID == closeEndID) ||
	 (eventID == fcloseEndID));
}
int isOpenBeginEvent(eventID)
     int eventID;
{
  return((eventID == openBeginID) ||
	 (eventID == gopenBeginID) ||
	 (eventID == fopenBeginID));
}

int isOpenEndEvent(eventID)
     int eventID;
{
  return((eventID == openEndID) ||
	 (eventID == gopenEndID) ||
	 (eventID == fopenEndID));
}

int isOpenCloseEvent(eventID)
     int eventID;
{
  return(isOpenBeginEvent(eventID) || isOpenEndEvent(eventID) ||
	 isCloseBeginEvent(eventID) || isCloseEndEvent(eventID));
}


int isReadEndEvent(eventID) 
     int eventID;
{
  return (eventID == readEndID   ||
	  eventID == creadEndID  ||
	  eventID == creadvEndID ||
	  eventID == freadEndID );
}

int isReadBeginEvent(eventID) 
     int eventID;
{
  return (eventID == readBeginID   ||
	  eventID == creadBeginID  ||
	  eventID == creadvBeginID ||
	  eventID == freadBeginID );
}


int isWriteEndEvent(eventID) 
     int eventID;
{
  return (eventID == writeEndID   ||
	  eventID == cwriteEndID  ||
	  eventID == cwritevEndID ||
	  eventID == fwriteEndID );
}

int isWriteBeginEvent(eventID) 
     int eventID;
{
  return (eventID == writeBeginID   ||
	  eventID == cwriteBeginID  ||
	  eventID == cwritevBeginID ||
	  eventID == fwriteBeginID );
}


int isSeekEndEvent(eventID)
     int eventID;
{
  return(eventID == fseekEndID ||
	 eventID == lseekEndID);
}

int isSeekBeginEvent(eventID)
     int eventID;
{
  return(eventID == fseekBeginID ||
	 eventID == lseekBeginID);
}


doFileSummariesSystemDependent()
{
  extern TR_RECORD *blockSummaryEventRecordFunction();

  setEventRecordFunction(creadBeginID, blockSummaryEventRecordFunction);  
  setEventRecordFunction(creadEndID,   blockSummaryEventRecordFunction);  
  setEventRecordFunction(creadvBeginID,   blockSummaryEventRecordFunction);  
  setEventRecordFunction(creadvEndID,   blockSummaryEventRecordFunction);  

  setEventRecordFunction(cwriteBeginID, blockSummaryEventRecordFunction);  
  setEventRecordFunction(cwriteEndID,   blockSummaryEventRecordFunction);  
  setEventRecordFunction(cwritevBeginID,   blockSummaryEventRecordFunction);  
  setEventRecordFunction(cwritevEndID,   blockSummaryEventRecordFunction);  
}



doTimeSummariesSystemDependent()
{
  extern TR_RECORD *timeStatisticsEventRecordFunction();

  setEventRecordFunction(creadBeginID, timeStatisticsEventRecordFunction);  
  setEventRecordFunction(creadEndID,   timeStatisticsEventRecordFunction);  
  setEventRecordFunction(creadvBeginID,   timeStatisticsEventRecordFunction);  
  setEventRecordFunction(creadvEndID,   timeStatisticsEventRecordFunction);  

  setEventRecordFunction(cwriteBeginID, timeStatisticsEventRecordFunction);  
  setEventRecordFunction(cwriteEndID,   timeStatisticsEventRecordFunction);  
  setEventRecordFunction(cwritevBeginID,   timeStatisticsEventRecordFunction); 
  setEventRecordFunction(cwritevEndID,   timeStatisticsEventRecordFunction);  
}

doBlockSummariesSystemDependent()
{
  extern TR_RECORD *blockSummaryEventRecordFunction();

  setEventRecordFunction(creadBeginID, blockSummaryEventRecordFunction);  
  setEventRecordFunction(creadEndID,   blockSummaryEventRecordFunction);  
  setEventRecordFunction(creadvBeginID,   blockSummaryEventRecordFunction);  
  setEventRecordFunction(creadvEndID,   blockSummaryEventRecordFunction);  

  setEventRecordFunction(cwriteBeginID, blockSummaryEventRecordFunction);  
  setEventRecordFunction(cwriteEndID,   blockSummaryEventRecordFunction);  
  setEventRecordFunction(cwritevBeginID,   blockSummaryEventRecordFunction); 
  setEventRecordFunction(cwritevEndID,   blockSummaryEventRecordFunction);  
}
