/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *          (R)
 * The Pablo    Performance Analysis Environment software is NOT in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1991-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * PABLO is a registered trademark of
 * The Board of Trustees of the University of Illinois
 * registered in the U.S. Patent and Trademark Office.
 *
 * Author:  Roger J. Noe (noe@cs.uiuc.edu)
 * Contributing Author:  Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004,
 * by a grant from the Digital Equipment Corporation External Research
 * Program, and by a collaborative research agreement with the Intel
 * Supercomputer Systems Division.
 *
 */

/*
 * iPSCtrace.c:
 *	This file contains interface functions for tracing the specific
 *	message-passing routines found on Intel iPSC systems.  It is
 *	intended to be used with the Pablo instrumentation library and
 *	the extensions to that library found in the file MesgTrace.c.
 *	Interface functions for Fortran precede their C counterparts.
 */

#undef MESGTRACE
#include "iPSCtrace.h"


/*
 * iPSCtraceInit:
 *	This function is the iPSC tracing interface function for
 *	initialization of the Pablo instrumentation library message-
 *	passing extensions.  Inputs are arrays of the event IDs defined
 *	above.  This function must be invoked before calling any of the
 *	iPSC-specific message-passing trace interface functions below.
 */

ipsctraceinit_()
{
	iPSCtraceInit();
}

iPSCtraceInit()
{
	static int	beginIDs[ 5 ] = {
			csendBeginID, crecvBeginID,
			isendBeginID, irecvBeginID,
			mesgBeginID
	};

	static int	endIDs[ 5 ] = {
			csendEndID, crecvEndID,
			isendEndID, irecvEndID,
			mesgEndID
	};

	initMesgTrace( beginIDs, endIDs );
}


/*
 * traceCSEND:
 *	Intel iPSC interface function for tracing calls to csend.
 */

long 
tracecsend_( messageType, dataPointer, messageSize, destinationNode, PID )

long	*messageType;
char	*dataPointer;
long	*messageSize;
long	*destinationNode;
long	*PID;
{
	return traceCSEND( *messageType, dataPointer, *messageSize,
			   *destinationNode, *PID );
}

long
traceCSEND( messageType, dataPointer, messageSize, destinationNode, PID )

long	messageType;
char	*dataPointer;
long	messageSize;
long	destinationNode;
long	PID;
{
	long	status;
	struct {
		long	type;
		long	length;
		long	node;
		long	pid;
	} csendArgs;

	csendArgs.type = messageType;
	csendArgs.length = messageSize;
	csendArgs.node = destinationNode;
	csendArgs.pid = PID;

	traceEvent( csendBeginID, (char *) &csendArgs, sizeof csendArgs );

	status = csend( messageType, dataPointer, messageSize,
			destinationNode, PID );

	traceEvent( csendEndID, (char *) 0, 0 );

	return( status );
}


/*
 * traceCRECV:
 *	Intel iPSC interface function for tracing calls to crecv.
 */

long
tracecrecv_( typeSelector, dataPointer, messageSize )

long	*typeSelector;
char	*dataPointer;
long	*messageSize;
{
	return traceCRECV( *typeSelector, dataPointer, *messageSize );
}

long
traceCRECV( typeSelector, dataPointer, messageSize )

long	typeSelector;
char	*dataPointer;
long	messageSize;
{
	long	status;
	struct {
		long	type;
		long	length;
	} crecvArgs;
	long	sourceNode, infonode();

	crecvArgs.type = typeSelector;
	crecvArgs.length = messageSize;

	traceEvent( crecvBeginID, (char *) &crecvArgs, sizeof crecvArgs );

	status = crecv( typeSelector, dataPointer, messageSize );

	sourceNode = infonode();

	traceEvent( crecvEndID, (char *) &sourceNode, sizeof sourceNode );

	return( status );
}


/*
 * traceISEND:
 *	Intel iPSC interface function for tracing calls to isend.
 */

long
traceisend_( messageType, dataPointer, messageSize, destinationNode, PID )

long	*messageType;
char	*dataPointer;
long	*messageSize;
long	*destinationNode;
long	*PID;
{
	return traceISEND( *messageType, dataPointer, *messageSize,
			   *destinationNode, *PID );
}

long
traceISEND( messageType, dataPointer, messageSize, destinationNode, PID )

long	messageType;
char	*dataPointer;
long	messageSize;
long	destinationNode;
long	PID;
{
	long	status;
	struct {
		long	type;
		long	length;
		long	node;
		long	pid;
	} isendArgs;

	isendArgs.type = messageType;
	isendArgs.length = messageSize;
	isendArgs.node = destinationNode;
	isendArgs.pid = PID;

	traceEvent( isendBeginID, (char *) &isendArgs, sizeof isendArgs );

	status = isend( messageType, dataPointer, messageSize,
			destinationNode, PID );

	traceEvent( isendEndID, (char *) 0, 0 );

	return( status );
}


/*
 * traceIRECV:
 *	Intel iPSC interface function for tracing calls to irecv.
 */

long
traceirecv_( typeSelector, dataPointer, messageSize )

long	*typeSelector;
char	*dataPointer;
long	*messageSize;
{
	return traceIRECV( *typeSelector, dataPointer, *messageSize );
}

long
traceIRECV( typeSelector, dataPointer, messageSize )

long	typeSelector;
char	*dataPointer;
long	messageSize;
{
	long	status;
	struct {
		long	type;
		long	length;
	} irecvArgs;

	irecvArgs.type = typeSelector;
	irecvArgs.length = messageSize;

	traceEvent( irecvBeginID, (char *) &irecvArgs, sizeof irecvArgs );

	status = irecv( typeSelector, dataPointer, messageSize );

	traceEvent( irecvEndID, (char *) 0, 0 );

	return( status );
}


/*
 * traceGCOL:
 *	Intel iPSC interface function for tracing calls to gcol.
 */

long
tracegcol_( x, xlen, y, ylen, ncnt )

char	*x;
long	*xlen;
char	*y;
long	*ylen;
long	*ncnt;
{
	return traceGCOL( x, *xlen, y, *ylen, ncnt );
}

long
traceGCOL( x, xlen, y, ylen, ncnt )

char	*x;
long	xlen;
char	*y;
long	ylen;
long	*ncnt;
{
 	long	status;
 	long	gcolType = PABLO_IPSC_GCOL;

	traceEvent( mesgBeginID, (char *) &gcolType, sizeof gcolType );

	status = gcol( x, xlen, y, ylen, ncnt );

	traceEvent( mesgEndID, (char *) &gcolType, sizeof gcolType );

	return status;
}


/*
 * traceGCOLX:
 *	Intel iPSC interface function for tracing calls to gcolx.
 */

long
tracegcolx_( x, xlens, y )

char	*x;
long	*xlens;
char	*y;
{
	return traceGCOLX( x, *xlens, y );
}

long
traceGCOLX( x, xlens, y )

char	*x;
long	xlens;
char	*y;
{
	long	status;
	long	gcolxType = PABLO_IPSC_GCOLX;

	traceEvent( mesgBeginID, (char *) &gcolxType, sizeof gcolxType );

	status = gcolx( x, xlens, y );

	traceEvent( mesgEndID, (char *) &gcolxType, sizeof gcolxType );

	return status;
}


/*
 * traceGDHIGH:
 *	Intel iPSC interface function for tracing calls to gdhigh.
 */

long
tracegdhigh_( x, n, work )

double	*x;
long	*n;
double	*work;
{
	return traceGDHIGH( x, *n, work );
}

long
traceGDHIGH( x, n, work )

double	*x;
long	n;
double	*work;
{
	long	status;
	long	gdhighType = PABLO_IPSC_GDHIGH;

	traceEvent( mesgBeginID, (char *) &gdhighType, sizeof gdhighType );

	status = gdhigh( x, n, work );

	traceEvent( mesgEndID, (char *) &gdhighType, sizeof gdhighType );

	return status;
}


/*
 * traceGDLOW:
 *	Intel iPSC interface function for tracing calls to gdlow.
 */

long
tracegdlow_( x, n, work )

double	*x;
long	*n;
double	*work;
{
	return traceGDLOW( x, *n, work );
}

long
traceGDLOW( x, n, work )

double	*x;
long	n;
double	*work;
{
	long	status;
	long	gdlowType = PABLO_IPSC_GDLOW;

	traceEvent( mesgBeginID, (char *) &gdlowType, sizeof gdlowType );

	status = gdlow( x, n, work );

	traceEvent( mesgEndID, (char *) &gdlowType, sizeof gdlowType );

	return status;
}


/*
 * traceGDPROD:
 *	Intel iPSC interface function for tracing calls to gdprod.
 */

long
tracegdprod_( x, n, work )

double	*x;
long	*n;
double	*work;
{
	return traceGDPROD( x, *n, work );
}

long
traceGDPROD( x, n, work )

double	*x;
long	n;
double	*work;
{
	long	status;
	long	gdprodType = PABLO_IPSC_GDPROD;

	traceEvent( mesgBeginID, (char *) &gdprodType, sizeof gdprodType );

	status = gdprod( x, n, work );

	traceEvent( mesgEndID, (char *) &gdprodType, sizeof gdprodType );

	return status;
}


/*
 * traceGDSUM:
 *	Intel iPSC interface function for tracing calls to gdsum.
 */

long
tracegdsum_( x, n, work )

double	*x;
long	*n;
double	*work;
{
	return traceGDSUM( x, *n, work );
}

long
traceGDSUM( x, n, work )

double	*x;
long	n;
double	*work;
{
	long	status;
	long	gdsumType = PABLO_IPSC_GDSUM;

	traceEvent( mesgBeginID, (char *) &gdsumType, sizeof gdsumType );

	status = gdsum( x, n, work );

	traceEvent( mesgEndID, (char *) &gdsumType, sizeof gdsumType );

	return status;
}


/*
 * traceGIAND:
 *	Intel iPSC interface function for tracing calls to giand.
 */

long
tracegiand_( x, n, work )

long	*x;
long	*n;
long	*work;
{
	return traceGIAND( x, *n, work );
}

long
traceGIAND( x, n, work )

long	*x;
long	n;
long	*work;
{
	long	status;
	long	giandType = PABLO_IPSC_GIAND;

	traceEvent( mesgBeginID, (char *) &giandType, sizeof giandType );

	status = giand( x, n, work );

	traceEvent( mesgEndID, (char *) &giandType, sizeof giandType );

	return status;
}


/*
 * traceGIHIGH:
 *	Intel iPSC interface function for tracing calls to gihigh.
 */

long
tracegihigh_( x, n, work )

long	*x;
long	*n;
long	*work;
{
	return traceGIHIGH( x, *n, work );
}

long
traceGIHIGH( x, n, work )

long	*x;
long	n;
long	*work;
{
	long	status;
	long	gihighType = PABLO_IPSC_GIHIGH;

	traceEvent( mesgBeginID, (char *) &gihighType, sizeof gihighType );

	status = gihigh( x, n, work );

	traceEvent( mesgEndID, (char *) &gihighType, sizeof gihighType );

	return status;
}


/*
 * traceGILOW:
 *	Intel iPSC interface function for tracing calls to gilow.
 */

long
tracegilow_( x, n, work )

long	*x;
long	*n;
long	*work;
{
	return traceGILOW( x, *n, work );
}

long
traceGILOW( x, n, work )

long	*x;
long	n;
long	*work;
{
	long	status;
	long	gilowType = PABLO_IPSC_GILOW;

	traceEvent( mesgBeginID, (char *) &gilowType, sizeof gilowType );

	status = gilow( x, n, work );

	traceEvent( mesgEndID, (char *) &gilowType, sizeof gilowType );

	return status;
}


/*
 * traceGIOR:
 *	Intel iPSC interface function for tracing calls to gior.
 */

long
tracegior_( x, n, work )

long	*x;
long	*n;
long	*work;
{
	return traceGIOR( x, *n, work );
}

long
traceGIOR( x, n, work )

long	*x;
long	n;
long	*work;
{
	long	status;
	long	giorType = PABLO_IPSC_GIOR;

	traceEvent( mesgBeginID, (char *) &giorType, sizeof giorType );

	status = gior( x, n, work );

	traceEvent( mesgEndID, (char *) &giorType, sizeof giorType );

	return status;
}


/*
 * traceGIPROD:
 *	Intel iPSC interface function for tracing calls to giprod.
 */

long
tracegiprod_( x, n, work )

long	*x;
long	*n;
long	*work;
{
	return traceGIPROD( x, *n, work );
}

long
traceGIPROD( x, n, work )

long	*x;
long	n;
long	*work;
{
	long	status;
	long	giprodType = PABLO_IPSC_GIPROD;

	traceEvent( mesgBeginID, (char *) &giprodType, sizeof giprodType );

	status = giprod( x, n, work );

	traceEvent( mesgEndID, (char *) &giprodType, sizeof giprodType );

	return status;
}


/*
 * traceGISUM:
 *	Intel iPSC interface function for tracing calls to gisum.
 */

long
tracegisum_( x, n, work )

long	*x;
long	*n;
long	*work;
{
	return traceGISUM( x, *n, work );
}

long
traceGISUM( x, n, work )

long	*x;
long	n;
long	*work;
{
	long	status;
	long	gisumType = PABLO_IPSC_GISUM;

	traceEvent( mesgBeginID, (char *) &gisumType, sizeof gisumType );

	status = gisum( x, n, work );

	traceEvent( mesgEndID, (char *) &gisumType, sizeof gisumType );

	return status;
}


#ifndef __PARAGON__

/*
 * traceGIXOR:
 *	Intel iPSC interface function for tracing calls to gixor.
 */

long
tracegixor_( x, n, work )

long	*x;
long	*n;
long	*work;
{
	return traceGIXOR( x, *n, work );
}

long
traceGIXOR( x, n, work )

long	*x;
long	n;
long	*work;
{
	long	status;
	long	gixorType = PABLO_IPSC_GIXOR;

	traceEvent( mesgBeginID, (char *) &gixorType, sizeof gixorType );

	status = gixor( x, n, work );

	traceEvent( mesgEndID, (char *) &gixorType, sizeof gixorType );

	return status;
}

#endif /* __PARAGON__ */


/*
 * traceGLAND:
 *	Intel iPSC interface function for tracing calls to gland.
 */

long
tracegland_( x, n, work )

long	*x;
long	*n;
long	*work;
{
	return traceGLAND( x, *n, work );
}

long
traceGLAND( x, n, work )

long	*x;
long	n;
long	*work;
{
	long	status;
	long	glandType = PABLO_IPSC_GLAND;

	traceEvent( mesgBeginID, (char *) &glandType, sizeof glandType );

	status = gland( x, n, work );

	traceEvent( mesgEndID, (char *) &glandType, sizeof glandType );

	return status;
}


/*
 * traceGLOR:
 *	Intel iPSC interface function for tracing calls to glor.
 */

long
traceglor_( x, n, work )

long	*x;
long	*n;
long	*work;
{
	return traceGLOR( x, *n, work );
}

long
traceGLOR( x, n, work )

long	*x;
long	n;
long	*work;
{
	long	status;
	long	glorType = PABLO_IPSC_GLOR;

	traceEvent( mesgBeginID, (char *) &glorType, sizeof glorType );

	status = glor( x, n, work );

	traceEvent( mesgEndID, (char *) &glorType, sizeof glorType );

	return status;
}


#ifndef __PARAGON__

/*
 * traceGLXOR:
 *	Intel iPSC interface function for tracing calls to glxor.
 */

long
traceglxor_( x, n, work )

long	*x;
long	*n;
long	*work;
{
	return traceGLXOR( x, *n, work );
}

long
traceGLXOR( x, n, work )

long	*x;
long	n;
long	*work;
{
	long	status;
	long	glxorType = PABLO_IPSC_GLXOR;

	traceEvent( mesgBeginID, (char *) &glxorType, sizeof glxorType );

	status = glxor( x, n, work );

	traceEvent( mesgEndID, (char *) &glxorType, sizeof glxorType );

	return status;
}

#endif /* __PARAGON__ */


/*
 * traceGOPF:
 *	Intel iPSC interface function for tracing calls to gopf.
 */

long
tracegopf_( x, xlen, work, f )

char	*x;
long	*xlen;
char	*work;
long	(*f)();
{
	return traceGOPF( x, *xlen, work, f );
}

long
traceGOPF( x, xlen, work, f )

char	*x;
long	xlen;
char	*work;
long	(*f)();
{
	long	status;
	long	gopfType = PABLO_IPSC_GOPF;

	traceEvent( mesgBeginID, (char *) &gopfType, sizeof gopfType );

	status = gopf( x, xlen, work, f );

	traceEvent( mesgEndID, (char *) &gopfType, sizeof gopfType );

	return status;
}


/*
 * traceGSENDX:
 *	Intel iPSC interface function for tracing calls to gsendx.
 */

long
tracegsendx_( type, x, xlen, nodenums, nlen )

long	*type;
char	*x;
long	*xlen;
long	*nodenums;
long	*nlen;
{
	return traceGSENDX( *type, x, *xlen, nodenums, *nlen );
}

long
traceGSENDX( type, x, xlen, nodenums, nlen )

long	type;
char	*x;
long	xlen;
long	*nodenums;
long	nlen;
{
	long	status;
	long	gsendxType = PABLO_IPSC_GSENDX;

	traceEvent( mesgBeginID, (char *) &gsendxType, sizeof gsendxType );

	status = gsendx( type, x, xlen, nodenums, nlen );

	traceEvent( mesgEndID, (char *) &gsendxType, sizeof gsendxType );

	return status;
}


/*
 * traceGSHIGH:
 *	Intel iPSC interface function for tracing calls to gshigh.
 */

long
tracegshigh_( x, n, work )

float	*x;
long	*n;
float	*work;
{
	return traceGSHIGH( x, *n, work );
}

long
traceGSHIGH( x, n, work )

float	*x;
long	n;
float	*work;
{
	long	status;
	long	gshighType = PABLO_IPSC_GSHIGH;

	traceEvent( mesgBeginID, (char *) &gshighType, sizeof gshighType );

	status = gshigh( x, n, work );

	traceEvent( mesgEndID, (char *) &gshighType, sizeof gshighType );

	return status;
}


/*
 * traceGSLOW:
 *	Intel iPSC interface function for tracing calls to gslow.
 */

long
tracegslow_( x, n, work )

float	*x;
long	*n;
float	*work;
{
	return traceGSLOW( x, *n, work );
}

long
traceGSLOW( x, n, work )

float	*x;
long	n;
float	*work;
{
	long	status;
	long	gslowType = PABLO_IPSC_GSLOW;

	traceEvent( mesgBeginID, (char *) &gslowType, sizeof gslowType );

	status = gslow( x, n, work );

	traceEvent( mesgEndID, (char *) &gslowType, sizeof gslowType );

	return status;
}


/*
 * traceGSPROD:
 *	Intel iPSC interface function for tracing calls to gsprod.
 */

long
tracegsprod_( x, n, work )

float	*x;
long	*n;
float	*work;
{
	return traceGSPROD( x, *n, work );
}

long
traceGSPROD( x, n, work )

float	*x;
long	n;
float	*work;
{
	long	status;
	long	gsprodType = PABLO_IPSC_GSPROD;

	traceEvent( mesgBeginID, (char *) &gsprodType, sizeof gsprodType );

	status = gsprod( x, n, work );

	traceEvent( mesgEndID, (char *) &gsprodType, sizeof gsprodType );

	return status;
}


/*
 * traceGSSUM:
 *	Intel iPSC interface function for tracing calls to gssum.
 */

long
tracegssum_( x, n, work )

float	*x;
long	*n;
float	*work;
{
	return traceGSSUM( x, *n, work );
}

long
traceGSSUM( x, n, work )

float	*x;
long	n;
float	*work;
{
	long	status;
	long	gssumType = PABLO_IPSC_GSSUM;

	traceEvent( mesgBeginID, (char *) &gssumType, sizeof gssumType );

	status = gssum( x, n, work );

	traceEvent( mesgEndID, (char *) &gssumType, sizeof gssumType );

	return status;
}
