/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *          (R)
 * The Pablo    Performance Analysis Environment software is NOT in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1991-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * PABLO is a registered trademark of
 * The Board of Trustees of the University of Illinois
 * registered in the U.S. Patent and Trademark Office.
 *
 * Author:  Roger J. Noe (noe@cs.uiuc.edu)
 * Contributing Author:  Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004,
 * by a grant from the Digital Equipment Corporation External Research
 * Program, and by a collaborative research agreement with the Intel
 * Supercomputer Systems Division.
 *
 */

/*
 *  SystemDepend.c:
 *	This file contains all the system and machine dependent portions
 *	of the portable Pablo software performance instrumentation library.
 *	This particular version was developed for the:
 *
 *		Intel iPSC/2 and iPSC/860 hypercubes and Paragon.
 *
 *	All the functions that appear below are REQUIRED by the portable
 *	Pablo trace library.  Their names, return types, and formal parameter
 *	types cannot be changed without concurrent changes to the portable
 *	portions of the trace library.
 *
 *	Most parts of the system dependent library are straightforward,
 *	and the necessary changes for porting to other systems should be
 *	obvious.
 */

#include "Assert.h"
#include "SystemDepend.h"
#include "TraceParam.h"
#include "Trace.h"

char	*traceFileName = (char *) NULL;	/* local trace file name	    */
char	defaultTraceFileName[ 128 ];	/* default trace file name	    */
FILE	*traceFile = (FILE *) NULL;	/* local trace file pointer	    */

unsigned long
	serverAddress = INADDR_ANY;	/* inet address for socket output   */
int	dgramPort = -1;			/* port for datagram output	    */
int	streamPort = -1;		/* port for stream output	    */
int	dgramSocket = -1;		/* socket for trace record output   */
int	streamSocket = -1;		/* socket for record descriptors    */

#ifdef DEBUG

char	*debugFileName = (char *) NULL;	/* debug output file name	    */
char	defaultDebugFileName[ 128 ];	/* default debug file name	    */
FILE	*debugFile = (FILE *) NULL;	/* debug output file pointer	    */

#endif /* DEBUG */

#ifdef __PARAGON__
struct rpm				/* RPM device address, for	    */
	*RPMaddress =			/* accessing the global clock	    */
		(struct rpm *) NULL;
#endif /* __PARAGON__ */


#ifndef TRsetTraceFile

/*
 *	TRsetTraceFile:
 *	   This function is called by newTraceFile before library
 *	   initialization ONLY (i.e. before TRinit called).  It should
 *	   record the name of the trace file requested by the user
 *	   program, and then open that file at initialization time.
 */

void
TRsetTraceFile( fileName )

char	*fileName;
{
	/* Has this function been called previously?  If so, discard	    */
	/* previously passed file name.					    */

	if (( traceFileName != (char *) NULL )
	   && ( traceFileName != defaultTraceFileName ))
		free( traceFileName );

	/* Allocate storage for copy of file name and make copy.	    */

	traceFileName = malloc( 1 + strlen( fileName ) );

	if ( traceFileName == (char *) NULL )
		TRfailure(
			"cannot allocate storage for trace file name" );

	strcpy( traceFileName, fileName );

	/* Un-do socket output mode, if previously set.			    */

	serverAddress = INADDR_ANY;
	dgramPort = streamPort = -1;
}

#endif /* TRsetTraceFile */


#ifndef TRsetDebugFile

#ifdef DEBUG

/*
 *	TRsetDebugFile:
 *	   This function is called by newDebugFile before library
 *	   initialization ONLY (i.e. before TRinit called).  It should
 *	   record the name of the debugging file requested by the user
 *	   program, and then open that file at initialization time.
 */

void
TRsetDebugFile( fileName )

char	*fileName;
{
	/* Has this function been called previously?  If so, discard	    */
	/* previously passed file name.					    */

	if (( debugFileName != (char *) NULL )
	   && ( debugFileName != defaultDebugFileName ))
		free( debugFileName );

	/* Allocate storage for copy of file name and make copy.	    */

	debugFileName = malloc( 1 + strlen( fileName ) );

	if ( debugFileName == (char *) NULL )
		TRfailure(
			"cannot allocate storage for debug file name" );

	strcpy( debugFileName, fileName );
}

#endif /* DEBUG */

#endif /* TRsetDebugFile */


#ifndef TRsetSocketAddress

/*
 *	TRsetSocketAddress:
 *	   This function is called by newSocketAddress before library
 *	   initialization ONLY (i.e. before TRinit called).  It merely
 *	   records the socket address and port(s) specified by the user
 *	   program, performing the socket set-up at initialization time.
 */

void
TRsetSocketAddress( socketAddress, udpPort, tcpPort )

unsigned long	socketAddress;
int		udpPort;
int		tcpPort;
{
	serverAddress = socketAddress;
	dgramPort = udpPort;
	streamPort = tcpPort;

	/* Un-do file output mode, if previously set.			    */

	if (( traceFileName != (char *) NULL )
	   && ( traceFileName != defaultTraceFileName )) {

		free( traceFileName );
		traceFileName = (char *) NULL;
	}
}

#endif /* TRsetSocketAddress */


#ifndef TRinit

/*
 *	TRinit:
 *	   This function performs the system-dependent portion of trace
 *	   library initialization.  It is invoked by the portable trace
 *	   library code, exactly once, when the first trace library
 *	   interface function is executed.  This implementation does
 *	   trace output initialization, initial synchronization, and
 *	   initializes the mechanism for synchronous trace dumping.
 */

void
TRinit()
{
	static char		nullMessage = '\0';
	struct sockaddr_in	server;

	if ( serverAddress != INADDR_ANY ) {

	    /* initialize output socket					    */

	    dgramSocket = socket( AF_INET, SOCK_DGRAM, 0 );

	    if ( dgramSocket < 0 )
		TRfailure( "cannot create datagram socket for output" );

	    server.sin_family = AF_INET;
	    server.sin_port = htons( dgramPort );
	    server.sin_addr.s_addr = serverAddress;

	    /* "connect" the datagram socket to simplify coding		    */

	    if ( connect( dgramSocket, (struct sockaddr *) &server,
			  sizeof server ) < 0 )
		TRfailure( "cannot connect datagram socket" );

	    if ( streamPort > 0 ) {

		/* initialize stream socket				    */

		streamSocket = socket( AF_INET, SOCK_STREAM, 0 );

		if ( streamSocket < 0 )
		    TRfailure( "cannot create stream socket for output" );

		server.sin_family = AF_INET;
		server.sin_port = htons( streamPort );
		server.sin_addr.s_addr = serverAddress;

		if ( connect( streamSocket, (struct sockaddr *) &server,
			      sizeof server ) < 0 )
		    TRfailure( "cannot connect stream socket" );
	    }

	} else {

	    /* initialize local trace output file			    */

	    if ( traceFileName == (char *) NULL ) {
		/* Use default trace file name if not changed by user	    */

		sprintf( defaultTraceFileName, "Pablo%.4d.bin", mynode() );

		traceFileName = defaultTraceFileName;
	    }

	    traceFile = fopen( traceFileName, "w" );

	    if ( traceFile == (FILE *) NULL ) {
		fprintf( stderr,
		    "Pablo:\tWarning: node %d cannot open trace file %s\n",
			mynode(), traceFileName );

		traceFile = stdout;		/* last resort		    */

		if ( traceFileName != defaultTraceFileName )
			free( traceFileName );

		strcpy( defaultTraceFileName, "standard output" );

		traceFileName = defaultTraceFileName;
	    }
	}

#ifdef DEBUG
	/* initialize local debugging output file			    */

	if ( debugFileName == (char *) NULL ) {
		/* Use default debug file name if not changed by user	    */

		sprintf( defaultDebugFileName, "Pablo%.4d.dbug", mynode() );

		debugFileName = defaultDebugFileName;
	}

	debugFile = fopen( debugFileName, "w" );

	if ( debugFile == (FILE *) NULL ) {
		fprintf( stderr,
		    "Pablo:\tWarning: node %d cannot open debug file %s\n",
			mynode(), debugFileName );

		debugFile = stderr;		/* last resort		    */

		if ( debugFileName != defaultDebugFileName )
			free( debugFileName );

		strcpy( defaultDebugFileName, "standard error" );

		debugFileName = defaultDebugFileName;
	}
#endif	/* DEBUG */

	if ( traceLibraryMode == MODE_SYNC ) {

#ifndef __PARAGON__
	    synchronizeClocks();	/* synchronize the clocks	    */
#endif /* __PARAGON__ */

					/* post the message handler	    */
	    hrecv( GLOBALDUMP, &nullMessage, sizeof nullMessage,
		   synchronizeDump );
	}

	return;
}

#endif /* TRinit */


#ifndef TRcleanup

/*
 *	TRcleanup:
 *	   This function performs the system-dependent portion of trace
 *	   library termination.  It is invoked by the portable library
 *	   function endTracing, BEFORE the portable library data structures
 *	   are themselves cleaned up.  This implementation does the final
 *	   flushing of the buffer, synchronization, and closing of trace
 *	   (and possibly debugging) file(s).
 */

void
TRcleanup()
{
	if ( traceLibraryMode == MODE_SYNC ) {
				/* dump the remainder of the buffer	    */
	    synchronizeDump( -1, 0, mynode(), mypid() );

	} else {
					/* asynchronous buffer dumping	    */
	    toggleClock();
	    dumpBuffer();
	    toggleClock();
	}

	if ( dgramSocket >= 0 ) {

	    close( dgramSocket );	/* close output socket		    */

	    dgramSocket = -1;
	    dgramPort = -1;

	    serverAddress = INADDR_ANY;

	    if ( streamSocket >= 0 ) {

		close( streamSocket );

		streamSocket = -1;
		streamPort = -1;
	    }

	} else {

	    fflush( traceFile );

	    if ( traceFile != stdout )
		fclose( traceFile );		/* close trace file	    */

	    traceFile = (FILE *) NULL;

	    if ( traceFileName != defaultTraceFileName )
		free( traceFileName );

	    traceFileName = (char *) NULL;
	}

#ifdef DEBUG
	fflush( debugFile );

	if ( debugFile != stderr )
		fclose( debugFile );		/* close debug file	    */

	debugFile = (FILE *) NULL;

	if ( debugFileName != defaultDebugFileName )
		free( debugFileName );

	debugFileName = (char *) NULL;
#endif /* DEBUG */
}

#endif /* TRcleanup */


#ifndef TRgetClock

/*
 *	TRgetClock:
 *	   Retrieves the value of the system clock.  When using a clock
 *	   with coarse granularity, many trace events may appear simultaneous
 *	   even though other code was executed between calls to the 
 *	   instrumentation library.  There really isn't much one can do to
 *	   fix such a problem (other than harrassing vendors to provide
 *	   decent clocks at the user level).
 *
 *	   Likewise, two successive calls to some Unix clocks may yield
 *	   values that seem to differ even though the time between the
 *	   calls is less than the clock resolution.  In fact, the system
 *	   is simply incrementing the previous clock value to give the
 *	   illusion that the clock is better than it is.
 *
 *	   For the iPSC/860, TRgetClock is simply hwclock().  The Paragon
 *	   accesses the global clock.  The iPSC/2 uses mclock() and watches
 *	   for wrap-arounds.
 */

void
TRgetClock( currentTime )

TR_CLOCK	*currentTime;
{
#ifdef __PARAGON__
	static int	globalClockInitialized = FALSE;
	double		globalClock;
	esize_t		*ep;

	if ( globalClockInitialized == FALSE ) {
		InitGlobalClock();
		globalClockInitialized = TRUE;
	}

	if ( RPMaddress != (struct rpm *) NULL ) {
		globalClock = RPMaddress->rpm_time;
		ep = (esize_t *) &globalClock;
		currentTime->slow = ep->slow;
		currentTime->shigh = ep->shigh;
	} else {
		/*
		 * For some reason we are unable to access the global
		 * clock.  Resort to hwclock instead.
		 */

		hwclock((esize_t *) currentTime);
	}
#else /* __PARAGON__ */
	static TR_CLOCK	lastclock = { 0, 0 };

	currentTime->slow = mclock();

	if ( currentTime->slow < lastclock.slow )
		lastclock.shigh++;			/* wrap around	    */

	lastclock.slow = currentTime->slow;

	currentTime->shigh = lastclock.shigh;
#endif /* __PARAGON__ */

	return;
}

#endif /* TRgetClock */


#ifndef TRclockDifference

/*
 *	TRclockDifference:
 *	   Computes the difference between two clock values.
 */

CLOCK
TRclockDifference( endTime, startTime )

TR_CLOCK	endTime;
TR_CLOCK	startTime;
{
	CLOCK	timeDiff;

	timeDiff.clkHigh = endTime.shigh - startTime.shigh;

	if ( endTime.slow >= startTime.slow ) {

		timeDiff.clkLow = endTime.slow - startTime.slow;

	} else {

		timeDiff.clkLow = 1 + ~( startTime.slow - endTime.slow );
		timeDiff.clkHigh--;

	}

	return timeDiff;
}

#endif /* TRclockDifference */


#ifndef TRincrementClock

/*
 *	TRincrementClock:
 *	   Increments a clock value by a specified amount.  Note that
 *	   this does *not* change the actual time, merely the value of
 *	   the first argument.  This routine is perhaps better called
 *	   "timeAddition."  If you're using a Unix time structure, this
 *	   routine may be considerably more complicated.
 */

void
TRincrementClock( clockValue, clockIncrement )

TR_CLOCK	*clockValue;
CLOCK		clockIncrement;
{
	long		carryBit = 0;
	unsigned long	sumLow;

	sumLow = clockValue->slow + clockIncrement.clkLow;

	if (( sumLow < clockValue->slow ) ||
	    ( sumLow < clockIncrement.clkLow ))
		carryBit = 1;

	clockValue->slow = sumLow;
	clockValue->shigh += clockIncrement.clkHigh + carryBit;

	return;
}

#endif /* TRincrementClock */


#ifndef TRgetBuffer

/*
 *	TRgetBuffer:
 *	   Allocates a buffer of the specified size in bytes. 
 *	   Depending on the frequency of buffer allocation and
 *	   deallocation and the efficiency of your system's memory
 *	   allocator, you may want to maintain your own free memory
 *	   pool and allocate from it.
 */

char *
TRgetBuffer( bufferSize )

unsigned int	bufferSize;
{
	return malloc( bufferSize );
}

#endif /* TRgetBuffer */


#ifndef TRfreeBuffer

/*
 *	TRfreeBuffer:
 *	   Deallocates a buffer.  See the note above concerning
 *	   memory management.
 */

void
TRfreeBuffer( bufferPointer )

char	*bufferPointer;
{
	if ( bufferPointer != (char *) 0 )
		free( bufferPointer );
}

#endif /* TRfreeBuffer */


#ifndef TRfailure

/*
 *	TRfailure:
 *	   This routine is invoked on an error elsewhere in the
 *	   code.  At this point, we've concluded that the error
 *	   is not recoverable, so we print an error message and die.
 */

void
TRfailure( errorMessage )

char	*errorMessage;
{
	(void) fprintf( stderr, "Pablo:\tFatal error:\t%s\n",
			errorMessage );

	exit( 1 );
}

#endif /* TRfailure */


#ifndef TRflush

/*
 *	TRflush:
 *	   This routine is invoked when a portion of the trace buffer
 *	   has filled and needs to be emptied in preparation for the
 *	   arrival of additional trace records.
 *
 *	   Depending on the software environment, one may want to
 *	   analyze the data in some way, dump it secondary storage,
 *	   or send it elsewhere via some network.  This simple
 *	   interface merely writes the data to a file or socket.
 */

void
TRflush( bufferStart, bufferEnd )

char	*bufferStart;
char	*bufferEnd;
{
	if ( dgramSocket >= 0 ) {		/* socket output	    */

	    if ( libraryInitLevel == INIT_FULL )

		(void) write( dgramSocket, bufferStart,
			      bufferEnd - bufferStart );

	    else if ( streamSocket >= 0 )

		(void) write( streamSocket, bufferStart,
			      bufferEnd - bufferStart );

	    /* else discard SDDF header or record descriptor		    */

	} else					/* file output		    */

	    (void) fwrite( bufferStart, sizeof(char),
			   bufferEnd - bufferStart, traceFile );
}

#endif /* TRflush */


#ifndef TRdump

/*
 *	TRdump:
 *	   This routine is invoked when the local trace buffer has
 *	   filled.  At this point, a good implementation should force
 *	   not only this buffer to be dumped but also force other
 *	   processors to dump their buffers as well.  By forcing all
 *	   processors to collectively synchronize and dump their
 *	   instrumentation buffers, we minimize the perturbation across
 *	   the application's threads of control.
 *
 *	   For the Intel iPSC/2, iPSC/860 and Paragon, we force this global
 *	   synchronization via a call to a message handler.
 */

void
TRdump()
{
	if ( traceLibraryMode == MODE_SYNC ) {

	    synchronizeDump( LOCALDUMP, 0, mynode(), mypid() );

	} else {
					/* asynchronous buffer dump	    */
	    toggleClock();
	    dumpBuffer();
	    toggleClock();
	}

	return;
}

#endif /* TRdump */


/*
 *	synchronizeDump:
 *	   This routine is the global synchronization for dumping
 *	   trace buffers.  To prevent skew of individual threads of
 *	   control, we force all processors to wait when the trace
 *	   buffer of any one processor fills.
 *
 *	   The messageType of LOCALDUMP results from execution of TRdump,
 *	   that is when the local trace buffer has filled.  This sends
 *	   the GLOBALDUMP message to other nodes.  A messageType of
 *	   GLOBALDUMP results only from an hrecv trap, when the GLOBALDUMP
 *	   message is received from another node.  When invoked as the
 *	   final dump from TRcleanup (from cleanupLibrary), the application
 *	   has called endTracing and the messageType is neither GLOBALDUMP
 *	   nor LOCALDUMP.
 */

synchronizeDump( messageType, count, node, pid )

long	messageType;
long	count;
long	node;
long	pid;
{
	long		oldState;
	static char	dumpMessage;
	char		unused = '\0';

	oldState = masktrap( 1 );	/* disable reentrant calls	    */

	toggleClock();			/* stop the tracing clock	    */

	/* Advise the other nodes that they must dump their buffers	    */

	if ( messageType == LOCALDUMP )
		csend( GLOBALDUMP, &unused, sizeof unused, -1, mypid() );

	gsync();			/* synchronize nodes		    */

	dumpBuffer();			/* dump the data at this node	    */

	gsync();			/* resynchronize		    */

	toggleClock();			/* restart the clock		    */

#ifndef __PARAGON__
	synchronizeClocks();		/* synchronize the clocks	    */
#endif /* __PARAGON__ */

					/* Reenable the handler		    */
	if ( messageType != LOCALDUMP )
		hrecv( GLOBALDUMP, &dumpMessage, sizeof dumpMessage,
			( messageType == GLOBALDUMP ) ? synchronizeDump
							: falseDump );

	masktrap( oldState );		/* restore the masktrap state	    */
}


/*
 *	falseDump:
 *	   This routine is a stub, used to catch GLOBALDUMP synchronous
 *	   dump messages after the executing node has performed trace
 *	   library clean-up.  It is invoked only via hrecv.  Its purpose
 *	   is to prevent trace library deadlock in the event a node
 *	   completes its portion of execution under trace library control
 *	   well before other nodes have.
 */

falseDump( messageType, count, node, pid )

long	messageType;
long	count;
long	node;
long	pid;
{
	long		oldState;
	static char	dumpMessage;

	oldState = masktrap( 1 );	/* disable reentrant calls	    */

	gsync();			/* synchronize nodes		    */

	/* There is no data for this node to dump, wait for others	    */

	gsync();			/* resynchronize		    */

#ifndef __PARAGON__
	synchronizeClocks();		/* synchronize the clocks	    */
#endif /* __PARAGON__ */

					/* Reenable the handler		    */
	hrecv( GLOBALDUMP, &dumpMessage, sizeof dumpMessage, falseDump );

	masktrap( oldState );		/* restore the masktrap state	    */
}


#ifndef __PARAGON__

/*
 *	synchronizeClocks:
 *	   This routine does elementary clock synchronization for the Intel
 *	   hypercubes.  The algorithm is simple: node 0 becomes the "master"
 *	   of the cube and sends a message to each other node requesting
 *	   that it send its current clock value (the one used in portable
 *	   portions of the trace library) back in another message.  By noting
 *	   the clock value on node 0 immediately before sending the request
 *	   and immediately after receiving the response, the offsets between
 *	   the clocks on each node in comparison to node 0 may be obtained.
 *	   Node 0 then determines which clock is "latest" in the cube and
 *	   computes how much each clock must be adjusted (by moving the epoch
 *	   backward, never forward) in order to be synchronized with the
 *	   latest clock.  Node 0 sends these adjustment messages to all
 *	   the other nodes.
 *
 *	   This routine is called from TRinit() and also immediately before
 *	   each time the trace buffers are dumped.  Since the trace library
 *	   does not enforce a resynchronization interval, the amount of
 *	   actual drift is proportional to the product of the skew between
 *	   different nodes' clocks and the actual length of time between
 *	   successive dumps.  The length of time between dumps is related
 *	   to the trace buffer size and rate of trace data (volume) generation.
 */

synchronizeClocks()
{
	static CLOCK	*offsets = (CLOCK *) 0;
	TR_CLOCK	before, after;
	CLOCK		clockValue, delta, maxOffset;
	int		nodeNumber;

	gsync();			/* initial barrier synchronization  */

	if ( mynode() == 0 ) {

	    /* Node 0 is the "master" node for clock synchronization        */

	    if ( offsets == (CLOCK *) 0 ) {
		/* Allocate a vector for holding clock offsets		    */

		offsets = (CLOCK *) TRgetBuffer( numnodes() * sizeof(CLOCK) );

		if ( offsets == (CLOCK *) 0 )
		    TRfailure( "unable to allocate clock sync vector" );

		/* offsets is NEVER deallocated; this is not a bug!	    */
	    }

	    /* Compute each slave node's clock offset and find the maximum  */

	    maxOffset = zeroClock;

	    for ( nodeNumber = 1; nodeNumber < numnodes(); nodeNumber++ ) {

	        clockValue = noSuchClock;

		TRgetClock( &before );

		csend( CLOCKVALUE, (char *) &clockValue, sizeof clockValue,
		       nodeNumber, mypid() );

		crecv( CLOCKVALUE, (char *) &clockValue, sizeof clockValue );

		TRgetClock( &after );

		/* Start with the difference in clock values		    */

		clockValue = clockSubtract( clockValue,
					TRclockDifference( before, epoch ) );

		/* Compute the round-trip message duration		    */

		delta = TRclockDifference( after, before );

		/* and halve it to estimate the one-way message time	    */

		delta.clkLow >>= 1;

		if (( delta.clkHigh & 01 ) != 0 )
			delta.clkLow |= 0x80000000;

		delta.clkHigh /= 2;

		/* Correct the offset for message propagation time	    */

		offsets[ nodeNumber ] = clockSubtract( clockValue, delta );

		if ( clockCompare( offsets[ nodeNumber ], maxOffset ) > 0 )

			maxOffset = offsets[ nodeNumber ];
	    }

	    /* Adjust node zero's clock					    */

	    clockValue = clockSubtract( zeroClock, maxOffset );

	    TRincrementClock( &epoch, clockValue );

	    /* And now all of the slave nodes' clocks			    */

	    for ( nodeNumber = 1; nodeNumber < numnodes(); nodeNumber++ ) {

		clockValue = clockSubtract( offsets[ nodeNumber ],
					    maxOffset );

		csend( CLOCKSYNC, (char *) &clockValue, sizeof clockValue,
		       nodeNumber, mypid() );
	    }

	} else {

	    /* All nodes except node 0, i.e. the "slave" nodes		    */

	    crecv( CLOCKVALUE, (char *) &clockValue, sizeof clockValue );

	    /* This is precisely a getClock but without the function call   */

	    TRgetClock( &before );

	    clockValue = TRclockDifference( before, epoch );

	    csend( CLOCKVALUE, (char *) &clockValue, sizeof clockValue, 0,
		   mypid() );

	    crecv( CLOCKSYNC, (char *) &clockValue, sizeof clockValue );

	    TRincrementClock( &epoch, clockValue );
	}
}

#endif /* __PARAGON__ */


#ifndef TRbcopy

/*
 *	TRbcopy:
 *	   This routine copies a group of bytes from one location to
 *	   another.  As the name suggests, if your system has a bcopy
 *	   routine, this is the place to use it.
 */

void
TRbcopy( bufferFrom, bufferTo, bufferLength )

char	*bufferFrom;
char	*bufferTo;
int	bufferLength;
{
	(void) memcpy( bufferTo, bufferFrom, bufferLength );
}

#endif /* TRbcopy */


#ifndef TRlock

/*
 *	TRlock:
 *	   This routine provides rudimentary locking for updates to
 *	   instrumentation data structures.  Although the need for this
 *	   may not seem obvious when only one application thread is
 *	   executing on each processor, it is important when the possibility
 *	   exists to asynchronously dump trace buffers.  In this case,
 *	   a trace buffer fill on another processor can interrupt this thread
 *	   to dump its buffer as well.  It is important that buffer pointers
 *	   and other trace library data structures not be modified during
 *	   this process.
 */

TR_LOCK
TRlock()
{
	return masktrap( 1 );
}

#endif /* TRlock */


#ifndef TRunlock

/*
 *	TRunlock:
 *	   This is the companion routine to TRlock.
 */

void
TRunlock( restoreValue )

TR_LOCK	restoreValue;
{
	masktrap( restoreValue );
}

#endif /* TRunlock */


#ifndef TRgetNode

/*
 *	TRgetNode:
 *	   Return the local node identifier.
 */

TRgetNode()
{
	return mynode();
}

#endif /* TRgetNode */


#ifdef __PARAGON__

/*
 *	InitGlobalClock:
 *	   The Paragon makes available a global clock through its RPM
 *	   boards.  It's difficult to get to, but once it's all set-up
 *	   it's easy and fast to access.  This is the one-time set-up
 *	   that needs to be done.  See TRgetClock for how the Paragon
 *	   global clock is accessed.
 */

# include <mach.h>
# include <mach/norma_special_ports.h>
# include <mach/vm_prot.h>
# include <mach/boolean.h>
# include <device/device.h>

InitGlobalClock()
{
    mach_port_t		serverPort, devicePort, pagerPort;
    vm_address_t	vmAddress;

    /* No point in continuing if there is no RPM board.			    */

    if ((( * (int *) (RPM_BASE_VADDR + 0x1000)) & 0x1 ) == 0 )
	return;

    /* Get the device server port for this node.			    */

    if ( norma_get_device_port( task_by_pid(-1), _myphysnode(), &serverPort )
	 != KERN_SUCCESS )
	return;

    /* Open the RPM device and get its device port ID.			    */

    if ( device_open( serverPort, (D_READ|D_WRITE), "rpm0", &devicePort )
	 != KERN_SUCCESS )
	return;

    /* Map it and get get a pager port.					    */

    if ( device_map( devicePort, (VM_PROT_READ|VM_PROT_WRITE), 0, 0x1000,
		     &pagerPort, 0 ) != KERN_SUCCESS )
	return;

    /* Map it into this task's address space.				    */

    if ( vm_map( mach_task_self(), &vmAddress, 0x1000, 0, TRUE,
                 pagerPort, 0, FALSE, (VM_PROT_READ|VM_PROT_WRITE),
                 (VM_PROT_READ|VM_PROT_WRITE), VM_INHERIT_NONE )
	 != KERN_SUCCESS )
        return;

    RPMaddress = (struct rpm *) vmAddress;
}

#endif /* __PARAGON__ */
