/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Contributing Author: Tom Birkett  (birkett@cs.uiuc.edu) 
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 */
/*
 * RecordDossier.C - methods of the class used to cache structure information
 *		      and data within the system
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Descriptors/RCS/RecordDossier.C,v 1.20 1994/02/25 04:31:50 aydt Exp $
 */

#include "RecordDossier.h"

#include "Assert.h"
#include "Defines.h"
#include "StructureDescriptor.h"
#include "StructureDescriptorIterator.h"

/*
 * The constructor with a single argument to be used as the name.  This
 * builds a "dummy" dossier with tag=0 and no fields.
 */
RecordDossier::RecordDossier( const CString& dossierName )
{
	_setClassName( MY_CLASS );
	setName( dossierName );

	tag = 0;
	fieldCount = 0;
}

/*
 * The constructor used to create a copy of a RecordDossier from another
 * Dossier instance.  
 */
RecordDossier::RecordDossier( const RecordDossier& originalDossier )
{
        _setClassName( MY_CLASS );

	setName( originalDossier.name );
	attributes = originalDossier.attributes;
	tag = originalDossier.tag;

	fieldCount = originalDossier.fieldCount;

	if ( fieldCount == 0 ) {
	     field = NULL;
	} else {
	     field = new DossierField[fieldCount];
	}

	int i;
	for ( i = 0; i < fieldCount; i++ ) {
	     field[i].descriptor = new FieldDescriptor( 
				      *(originalDossier.field[i].descriptor) );
	     field[i].value = new Value( *(originalDossier.field[i].value) );
	}
}


/* 
 * The constructor usually called by RecordDictionary.createEntry().  It 
 * creates a template with the correct number of DossierField instances.
 * No actual record data is contained in the RecordDossier instance at 
 * this point.
 */
RecordDossier::RecordDossier( int sysTag,
			      const StructureDescriptor& structureDescr )
{
	_setClassName( MY_CLASS );
	setName( structureDescr.getName() );

	attributes = structureDescr.getAttributes();
	tag = sysTag;

	fieldCount = structureDescr.entryCount();
	if ( fieldCount == 0 ) {
	    field = NULL;
	} else {
	    field = new DossierField[ fieldCount ];
	}

	StructureDescriptorIterator* iterator = 
		new StructureDescriptorIterator( structureDescr );
	int i;
	for ( i = 0; i < fieldCount; i++ ) {
	    field[i].descriptor = new FieldDescriptor( iterator->next() );
	    Assert ( field[i].descriptor->getName() != CString::NOMATCH );

	    field[i].value = new Value( field[i].descriptor->getTraits() );
	}
	delete iterator;
}

/* 
 * The destructor frees space used by the DossierField instances.
 */
RecordDossier::~RecordDossier()
{
	/*
	 * Free the space for fields
	 */
	int i;
	for ( i = 0; i < fieldCount; i++ ) {
	    delete field[i].descriptor;
	    delete field[i].value;
	}
	if ( fieldCount > 0 ) {
	    delete [] field;
	}
}

int
RecordDossier::bitsToObj( const char *dataPtr )
{
	int i;
	int currentOffset = 0; 

	for ( i = 0; i < fieldCount; i++ ){
	    currentOffset += field[i].value->bitsToObj( dataPtr+currentOffset );
	} 

	return currentOffset;
}

int
RecordDossier::bytesNeeded() const
{
	int i;
	int bytesRequired = 0;
	for ( i = 0; i < fieldCount; i++ ) {
	    bytesRequired += field[i].value->bytesNeeded();
	}
	return bytesRequired;
}

int
RecordDossier::cbitsToObj( const char *dataPtr, 
			   const DataCharacteristics *cnvInfo )
{
	int i;
	int currentOffset = 0; 

	for ( i = 0; i < fieldCount; i++ ){
	    currentOffset += field[i].value->cbitsToObj( dataPtr+currentOffset,
							 cnvInfo );
	} 

	return currentOffset;
}

int
RecordDossier::cbytesNeeded( const DataCharacteristics *cnvInfo ) const
{
	int i;
	int bytesRequired = 0;
	for ( i = 0; i < fieldCount; i++ ) {
	    bytesRequired += field[i].value->cbytesNeeded( cnvInfo );
	}
	return bytesRequired;
}

Array *
RecordDossier::getArrayP( int fieldID ) const
{
	if ( fieldID < fieldCount ) {
	    return( (Array *)( *field[ fieldID ].value)  );
	} else {
   	    return NULL;
	}
}

Array *
RecordDossier::getArrayP( const CString& fieldName ) const
{
	int i;
	for ( i = 0; i < fieldCount; i++ ) {
	    if ( field[i].descriptor->getName() == fieldName ) {
		return( (Array *)( *field[i].value ) );
	    }
	}
	return NULL;
}

FieldDescriptor *
RecordDossier::getField( const CString& fieldName ) const
{
	int i;
	for ( i = 0; i < fieldCount; i++ ) {
	    if ( field[i].descriptor->getName() == fieldName ) {
		return field[i].descriptor;
	    }
	}
	return NULL;
}

FieldDescriptor *
RecordDossier::getField( int fieldID ) const
{
	if ( fieldID < fieldCount ) {
	    return field[ fieldID ].descriptor;
	} else {
	    return NULL;
	}
}

int
RecordDossier::getFieldID( const CString& fieldName ) const
{
	int i;
	for ( i = 0; i < fieldCount; i++ ) {
	    if ( field[i].descriptor->getName() == fieldName ) {
		return i;
	    }
	}
	return -1;
}

Value&
RecordDossier::getValue( const CString& fieldName ) const
{
	int i;
	for ( i = 0; i < fieldCount; i++ ) {
	    if ( field[i].descriptor->getName() == fieldName ) {
		return( *(field[i].value) );
	    }
	}
	return( Value::NOVALUE );
}

Value&
RecordDossier::getValue( int fieldID ) const
{
	if ( fieldID < fieldCount ) {
	     return( *(field[ fieldID ].value) ) ;
	} else {
	     return( Value::NOVALUE );
	}
}

Value *
RecordDossier::getValueP( const CString& fieldName ) const
{
	int i;
	for ( i = 0; i < fieldCount; i++ ) {
	    if ( field[i].descriptor->getName() == fieldName ) {
		return field[i].value;
	    }
	}
	return( NULL );
}

Value *
RecordDossier::getValueP( int fieldID ) const
{
	if ( fieldID < fieldCount ) {
	    return field[ fieldID ].value;
	} else {
	    return NULL;
	}
}

int
RecordDossier::objToBits( char *const bufPtr, int bufLen ) const
{
	if ( bytesNeeded() > bufLen ) {
 	     return -1;
	}
	
	int i;
	int bytesUsed = 0;
	for ( i = 0; i < fieldCount; i++  ) {
	    bytesUsed += field[i].value->objToBits( bufPtr+bytesUsed,
						    bufLen-bytesUsed );
	} 
	return bytesUsed;
}

int
RecordDossier::objToCbits( char *const bufPtr, int bufLen,
			   const DataCharacteristics *cnvInfo ) const
{
	if ( cbytesNeeded( cnvInfo ) > bufLen ) {
 	     return -1;
	}
	
	int i;
	int bytesUsed = 0;
	for ( i = 0; i < fieldCount; i++  ) {
	    bytesUsed += field[i].value->objToCbits( bufPtr+bytesUsed,
						     bufLen-bytesUsed, 
						     cnvInfo );
	} 
	return bytesUsed;
}

Boolean_
RecordDossier::setValue( const CString& fieldName, const Value& fieldValue ) 
{
	int i;
	for ( i = 0; i < fieldCount; i++ ) {
	    if ( field[i].descriptor->getName() == fieldName ) {
	        return ( field[i].value->setQuantity( fieldValue ) );
	    }
	}
	warning( "\n\tsetValue() failed - no field %s in record", 
						(const char *)fieldName);
	return FAILURE_;
}

Boolean_
RecordDossier::setValue( int fieldID, const Value& fieldValue ) 
{
	if ( fieldID < fieldCount ) {
	    return ( field[fieldID].value->setQuantity( fieldValue ) );
	} else {
	    warning( "\n\tsetValue() failed - no field %d in record", fieldID );
	    return FAILURE_;
	}
}

/*
 * The assignment operator frees "variable" space associated with this 
 * RecordDossier instance and allocates new space to accomodate a copy
 * of the rhs recordDossier.  A duplicate copy is made - this method makes
 * a copy of the data - not just a copy of the pointers.
 */
RecordDossier& 
RecordDossier::operator=( const RecordDossier& recordDossier )
{
	/*
	 * First free space currently in use for fields & data
	 */
	int i;
	for ( i = 0; i < fieldCount; i++ ) {
	    delete field[i].descriptor;
	    delete field[i].value;
	}

	if ( fieldCount > 0 ) {
	    delete [] field;
	}

	/*
	 * Next, copy values
	 */
	setName( recordDossier.name );
	attributes = recordDossier.attributes;
	tag = recordDossier.tag;

	fieldCount = recordDossier.fieldCount;

	if ( fieldCount == 0 ) {
	     field = NULL;
	} else {
	     field = new DossierField[fieldCount];
	}

	for ( i = 0; i < fieldCount; i++ ) {
	     field[i].descriptor = new FieldDescriptor( 
					 *(recordDossier.field[i].descriptor)) ;
	     field[i].value = new Value( *(recordDossier.field[i].value) );
	}

	return *this;
}

void 		/* virtual */
RecordDossier::printOn( ostream& os ) const
{
	os << "<RecordDossier begin>\n";
	os << "name: " ;
	name.printOn( os );
	os << NL << "tag: " << tag << NL;
	attributes.printOn( os );
	os << "# of fields: " << fieldCount << NL;

	int i;
	for ( i = 0; i < fieldCount; i++ ) {
	    os << *(field[i].descriptor);
	    os << *(field[i].value);
	}
	os << "<RecordDossier end> \n";
}

/*
 * Initialize the static data 
 */
const char *const RecordDossier::MY_CLASS = "RecordDossier";

/*
 * Definition of the static data if not done in a central place
 */
#ifdef CONSTRUCT_NONLOCAL_STATICS
  	RecordDossier RecordDossier::NODOSSIER( CString::NOMATCH );
#endif
