/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Pincipal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * ConversionPipeReader.C - methods that deal with the Binary Pipes connected
 * to input files whose data characteristics do not match those of the native
 * machine.
 */
#include <stdlib.h>

#include "ConversionPipeReader.h"

#include "Assert.h"
#include "Attributes.h"
#include "DataCharacteristics.h"
#include "RecordDossier.h"
#include "StructureDescriptor.h"
#include "StreamPipe.h"
#include "SystemErrors.h"

#include "GetPut.h"

ConversionPipeReader::ConversionPipeReader( StreamPipe *initPipe ) 
		 :PipeReader( initPipe )
{
	_setClassName( MY_CLASS );
	currentHeader.type = PIPE_EMPTY;

	if ( initPipe->getInputType() != CONVERT_BINARY_SDDF ) {
	    error( "Can't connect to data stream %s",
		    initPipe->getObjectName() );
	    pipe = NULL;
	    bufferSize = 0;
	    packetBuffer = NULL;
	} else {
	    dataInfo = initPipe->getInputCharacteristics();
	    Assert( dataInfo != NULL );
	    PACKET_HEADER_SIZE = PACKET_HEADER_FIELD_CNT * dataInfo->intSize();
	    bufferSize = PACKET_HEADER_SIZE;
	    packetBuffer = (char *) malloc( bufferSize );
	    if ( packetBuffer == NULL ) {
		abort( "Can't malloc space for bufferSize %d: %s",
			bufferSize,  errorString() );
	    }
        }
}

ConversionPipeReader::~ConversionPipeReader()
{
	if ( packetBuffer != NULL ) {
	    free( packetBuffer );
	}
}

Boolean_
ConversionPipeReader::_readPacket()
{
	/*
	 * Look for a packet header on the pipe and if one is there load it
	 * into the first bytes of packetBuffer.  If we can't read the full 
	 * header we assume that the pipe is empty and return FAILURE_.
	 */
	int n = pipe->peek( packetBuffer, PACKET_HEADER_SIZE );
	if ( n != PACKET_HEADER_SIZE ) {
	    return( FAILURE_ );
	}

	/* 
	 * We found a packet header so move it from packetBuffer into 
	 * currentHeader and then skip past the header information to
	 * load the rest of the packet into packetBuffer,  allocating 
	 * more space in packetBuffer if necessary.  We are guaranteed 
	 * that partial packets will not be written to the pipe.
	 */
	int type;
	char *bufPtr = packetBuffer;
	dataInfo->functionGet( &bufPtr, &currentHeader.length ); 
	dataInfo->functionGet( &bufPtr, &type );	
	currentHeader.type = (PacketType) type;
	dataInfo->functionGet( &bufPtr, &currentHeader.tag );

	if ( currentHeader.length > bufferSize ) {
	    bufferSize = currentHeader.length;
	    packetBuffer = (char *) realloc( packetBuffer, bufferSize );
            if ( packetBuffer == NULL ) {
                abort( "Can't realloc space for packet length %d: %s",
                        bufferSize,  errorString() );
            }
	}

	pipe->getSkippingPeek( packetBuffer + PACKET_HEADER_SIZE, 
			       currentHeader.length - PACKET_HEADER_SIZE );

	/* 
	 * If this is a new Pipe Attribute packet, then update pipeAttributes
	 */
	if ( currentHeader.type == PKT_ATTRIBUTE ) {
	    int by = pipeAttributes.cbitsToObj( packetBuffer+PACKET_HEADER_SIZE,					        dataInfo );
		
			
  	    Assert( by == ( currentHeader.length - PACKET_HEADER_SIZE ) );
	}

	return TRUE_;
}

/*
 * Public methods
 */

Boolean_ 		/* virtual */
ConversionPipeReader::getData( RecordDossier& recordDossier) 
{
	if ( currentHeader.type != PKT_DATA ) {
	    return( FAILURE_ );
	}

	int by = recordDossier.cbitsToObj( packetBuffer+PACKET_HEADER_SIZE,
					   dataInfo );
	if ( by == currentHeader.length - PACKET_HEADER_SIZE ) {
	    return( SUCCESS_ );
	} else {
	    return( FAILURE_ );
	}
}

Boolean_ 		/* virtual */
ConversionPipeReader::getDescriptor( StructureDescriptor& structDescr ) 
{
	if ( currentHeader.type != PKT_DESCRIPTOR ) {
	    return( FAILURE_ );
	}

	int by = structDescr.cbitsToObj( packetBuffer+PACKET_HEADER_SIZE,
					 dataInfo );
	if ( by == currentHeader.length - PACKET_HEADER_SIZE ) {
	    return( SUCCESS_ );
	} else {
	    return( FAILURE_ );
	}
}

PacketHeader		/* virtual */
ConversionPipeReader::getPacketHeader( )
{
	do {

 	    if ( _readPacket() == FALSE_ ) {  
		return( noHeader );
	    }
		
	    switch ( currentHeader.type ) {

		case PKT_ATTRIBUTE:
	  	case PKT_DESCRIPTOR:
		case PKT_DATA:
		case PKT_COMMAND:
		    return ( currentHeader );
		    //break;

		case PKT_UNKNOWN:
		case PIPE_EMPTY:
		default:
		    warning( "Found bad packet type = %d", currentHeader.type );
		    break;
	    }
	} while ( TRUE_ );
}
	

/*
 * Initialize the static data structure
 */
const char *const ConversionPipeReader::MY_CLASS = "ConversionPipeReader";
PacketHeader ConversionPipeReader::noHeader = { 0, PIPE_EMPTY, 0 };
