/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Modified by Celso L. Mendes to handle AIMS instead of PICL traces - 05/93
 * Modified by Ruth A. Aydt 10/93 to 
 *	    start SDDF tags at 1; 
 *	    use dossier reference instead of Record Dictionary lookup each time;
 *	    better error checking in case of invalid tag;
 *	    do Writer->putData outside of switch;
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * ConvertAIMS.C Program to read an AIMS trace file and
 *	convert it to an Ascii or Binary SDDF file.
 *
 */
#include <stream.h>
#include <string.h>
#include <time.h>

#include "InitializeStatic.C"

#include "PipeReader.h"
#include "AsciiPipeWriter.h"
#include "BinaryPipeWriter.h"
#include "ConversionPipeWriter.h"
#include "InputFileStreamPipe.h"
#include "OutputFileStreamPipe.h"

#include "Attributes.h"
#include "PacketHeader.h"
#include "RecordDictionary.h"
#include "RecordDossier.h"
#include "StructureDescriptor.h"

void processRecordFormats( PipeReader *InPipe, PipeWriter *OutPipe, int diag );
void processTraceFile( FILE *inFile, PipeWriter *OutPipe, int diag );

#define MAXAIMSTYPE	26		/* valid AIMS types are 0 to 25; these 
					 * are mapped to SDDF tags 1 to 26    */

RecordDictionary	*RecordDict;	/* our global record dictionary       */


/******************************************************************************
 *  main() accepts user input, opens files, writes general configuration      *
 *  information to the output file, calls routines to do the hard work        *
 *  closes files, and exits.				                      *
 *									      *
 *  Note that we query the user and send informational and general diagnostic *
 *  messages to stdout.  We use stderr for reporting messages generated as a  *
 *  result of selecting "detailed diagnostics".  This allows the user to      *
 *  redirect stderr (the verbose diagnostics) to a file while still seeing    *
 *  questions and general information messages on stdout.                     *
 ******************************************************************************/

main()
{
	RecordDict = new RecordDictionary();	// Construct global dictionary

	enum Status { INVALID, VALID };
	Status 	fileStatus;			// Status of file open        
	char    BUF[512];			// Buffer for keyboard input 

	/*******************************************************************
	 *    Get the name of the Ascii SDDF file containing the
	 *    descriptions of the SDDF record formats we will be 
	 *    generating.  Open the file and attach a Reader to it.
	 *******************************************************************/

	InputFileStreamPipe  	*FormatFile;	     // Pipe connected to the
						     // input format file.
	PipeReader		*FormatPipeReader;   // Reads SDDF packets 

	do {
	    cout << "SDDF file containing record formats?  ";
	    cin >> BUF;

	    /* 
	     * Create an input file stream pipe instance connected to
	     * the named file on disk.  If file open is successful, 
	     * connect an SDDF reader to the pipe.
	     */
	    FormatFile = new InputFileStreamPipe( BUF );
	    if ( FormatFile->successfulOpen() ) {
	    	fileStatus = VALID;
	        FormatPipeReader = FormatFile->createPipeReader();
	    } else {
	    	cout << "ERROR: unable to open specified file." << endl;
	    	fileStatus = INVALID;
	    	delete FormatFile;
	    }
	} while ( fileStatus == INVALID );

	/*******************************************************************
	 *    Get the name of the AIMS trace file. Open the file.
	 *******************************************************************/

	FILE *traceFile;		// AIMS trace file to be converted
	char  traceFileName[256];	// Name of trace file

  	do {
      	    cout << "AIMS trace file for input? ";
	    cin >> BUF;
	    traceFile = fopen( BUF, "r" );
	    if ( traceFile == NULL ) {
	   	cout << "Error opening AIMS trace file for input." << endl;
	    } else {
	    	strcpy( traceFileName, BUF );
	    }
	} while ( traceFile == NULL );


	/*******************************************************************
	 *    Ask if the output file should be Ascii, Binary, or Converted 
	 *    Binary format.
	 * *****************************************************************/

	enum FileType	{ UNDEFINED, ASCII, BINARY, CONVERT };
	FileType	fileType = UNDEFINED;

	while ( fileType == UNDEFINED ) {
	    cout << "Write SDDF file in Ascii, Binary, or Converted "
	         << "(reverse byte order binary)\n format (A, B, or C) ? ";
	    cin >> BUF;
	    if ( BUF[0] == 'a' || BUF[0] == 'A' ) {
	    	fileType = ASCII;
	    } else if ( BUF[0] == 'b' || BUF[0] == 'B' ) {
	    	fileType = BINARY;
	    } else if ( BUF[0] == 'c' || BUF[0] == 'C' ) {
	    	fileType = CONVERT;
	    } else {
	    	cout << "ERROR: invalid format type." << endl;
	    }
	}

	/******************************************************************
	 *    Get the name of the output file. Open or Create it and 
	 *    attach a Writer to it.
	 ******************************************************************/

	OutputFileStreamPipe	*OutFile;	// Pipe connected to output file
	PipeWriter   		*PipeWriter;	// Writes SDDF packets to pipe 
	
	do {
	    cout << "Please enter name of the output SDDF file: ";
	    cin >> BUF;

	    /* 
	     * Create an output file stream pipe instance connected to
	     * the named file on disk.  If file open is successful, 
	     * connect the appropriate (Ascii or Binary)  SDDF writer 
	     * to the pipe.
	     */
	    OutFile = new OutputFileStreamPipe( BUF );
	    if ( OutFile->successfulOpen() ) {
	    	fileStatus = VALID;
	    	if ( fileType == ASCII ) {
	            PipeWriter = new AsciiPipeWriter( OutFile );
	    	} else if ( fileType == BINARY ) {
	    	    PipeWriter = new BinaryPipeWriter( OutFile );
	    	} else {
	    	    PipeWriter = new ConversionPipeWriter( OutFile );
	    	}
	    } else {
	    	cout << "ERROR: unable to open specified file." << endl;
	    	fileStatus = INVALID;
	    	delete OutFile;
	    }
	} while ( fileStatus == INVALID );

	/******************************************************************
	 *    Ask if verbose diagnostic messages are desired.
	 ******************************************************************/

        int diagOutput;

        cout << "Do you want detailed diagnostics sent to stderr (Y or N) ? ";
        cin >> BUF;
        if ( BUF[0] == 'y' || BUF[0] == 'Y' ) {
            diagOutput = 1;
        } else {
            diagOutput = 0;
        }

	/******************************************************************
	 *    Here's where we start the real work.  
	 *
	 *    We begin by creating an Attributes instance and populating it
	 *    with information about the AIMS file name and the date of the 
	 *    conversion.  We use the Writer to send this packet to the 
	 *    output file.
	 *
	 *    Next, we call processRecordFormats() to load our dictionary
	 *    with the record descriptor information from the SDDF file
	 *    We also send that information to the output file using the
	 *    Writer.
	 *
	 *    Finally, we call processTraceFile() to read records from the
	 *    original AIMS trace file. We lookup each record tag in our 
	 *    dictionary, parse the record, and use the Writer to send the
	 *    corresponding SDDF packet to the output file.
	 *
	 ******************************************************************/

	time_t now = time( 0 );
	char *date = ctime( &now );
	char *nl = strchr( date, '\n');
	if ( nl ) *nl = 0;

	Attributes	attributes;
	attributes.insert( "Trace from AIMS data file", traceFileName );
	attributes.insert( "File converted", date );
	PipeWriter->putAttributes( attributes );

	cout << "\nStarting to process format file." << endl;
	processRecordFormats( FormatPipeReader, PipeWriter, diagOutput );
	delete FormatPipeReader;		// Finished with Format File
	
	cout << "Starting to process AIMS trace file." << endl;
	processTraceFile( traceFile, PipeWriter, diagOutput );

	delete OutFile; 			// Closes the file

}

/******************************************************************************
 *  processRecordFormats() reads SDDF packets from the Reader InPipe, loads   *
 *  the Descriptor information into our RecordDictionary, and rewrites the    *
 *  corresponding SDDF packets to the Writer OutPipe.  We expect these        *
 *  packets to contain record format information (Descriptor packets), but    *
 *  in fact they may be any of the four SDDF packet types.  		      *	
 ******************************************************************************/

void
processRecordFormats( PipeReader *InPipe, PipeWriter *OutPipe, int diag )
{
	Attributes		*Ap;
	StructureDescriptor	*SDp;
	PacketHeader PH = InPipe->getPacketHeader();

        while( PH.type != PIPE_EMPTY ) {

            switch( PH.type ) {
              case PKT_ATTRIBUTE:
            	  Ap = new Attributes();
                  InPipe->getAttributes( *Ap );
                  OutPipe->putAttributes( *Ap );
                  if ( diag ) cerr << "\n" << *Ap;
                  delete Ap;
                  break;

              case PKT_DESCRIPTOR:
                  SDp = new StructureDescriptor();
                  InPipe->getDescriptor( *SDp );
                  OutPipe->putDescriptor( *SDp, PH.tag );
                  RecordDict->insert( PH.tag, *SDp );
                  if ( diag ) cerr << "\n" << *SDp;
                  delete SDp;
                  break;

              case PKT_DATA:
                  InPipe->getData( RecordDict->fetch( PH.tag ) );
                  OutPipe->putData( RecordDict->fetch( PH.tag ) );
                  if ( diag ) cerr << "\n" <<  ( RecordDict->fetch( PH.tag ) );
                  break;

              case PKT_COMMAND:
                  OutPipe->putCommand( PH.tag );
                  if ( diag ) cerr << "\nCommand tag = " << PH.tag << "\n";
                  break;
            }
            PH = InPipe->getPacketHeader();

        }
}

/******************************************************************************
 *  processTraceFile() reads lines from the AIMS file, parses the data,       *
 *  and writes the corresponding packet to the SDDF output file.              *
 ******************************************************************************/

void
processTraceFile( FILE *inFile, PipeWriter *OutPipe, int diag )
{
	/* 
	 * Keep some count of records and types processed....  
	 */
	int recordCount = 0;
	int typeCount[ MAXAIMSTYPE+1 ];	
	for ( int i = 0; i <= MAXAIMSTYPE; typeCount[i++] = 0 );

	/* 
	 * Read the trace records 
	 */
	char	Buffer[2048];	// Buffer to hold record
	char   *Bufp;		// Buffer pointer
	int	value[11];	// Individual values in record
	int	valueCnt;	// Number of values found
	int	sddfTag;	// The SDDF tag = AIMS type + 1
	int	clock;		// Value of merged second & microsecond entries
	char	string[128];	// Buffer to hold string
	int     errorFlag;	// 0 = no error; -1 = bad type; 1 = bad fields

	cout << "Currently processing record... \n";

	while ( ( Bufp = fgets( Buffer, 2048, inFile ) ) != NULL ) {

	    errorFlag = 0;
	    valueCnt = sscanf( Bufp, "%d %d %d %d %d %d %d %d %d %d %d",
	                       &value[0], &value[1], &value[2], &value[3], 
	    	               &value[4], &value[5], &value[6], &value[7], 
	    	               &value[8], &value[9], &value[10]);

	    /* 
	     * Update our counters and check that values were found 
	     */
	    if ( ( ++recordCount % 100 ) == 0 ) {
	        cout << recordCount << "... " << flush;
	    }

	    if (  ( valueCnt == 0 ) || ( valueCnt == EOF ) ) {
	        cout << "\nError: Unable to get values for record number " 
	             << recordCount << endl;
	        continue;
	    }

	    /* For all records: 
	     * + Record type is field 0. SDDF tags begin at 1, so we increment 
	     *   the AIMS type to get the corresponding SDDF tag.  
	     * + Fields 1 and 2 have values for seconds and microseconds. We 
	     *   merge those into a single clock value.
	     * + Field 3 contains the node number.
	     *
	     * We use the SDDF tag to look up the record in our Record
	     * Dictionary.  The dictionary contains the field descriptors
	     * and types for each record.  It also contains space to hold
	     * the field values which we can access via the RecordDossier
	     * reference.  We parse the data we have read in from 
	     * our AIMS trace file and use it to populate the field
	     * values for our particular record.
	     */

	    sddfTag = value[0] + 1;
	
	    if ( ( 0 < sddfTag ) && ( sddfTag <= MAXAIMSTYPE ) ) {
	        ++typeCount[ sddfTag ];		  // expected type
            } else {
                ++typeCount[ 0 ];                 // unexpected type
	        cout << "\nWARNING: Record number " << recordCount 
	             << " had unknown AIMS type " << value[0]
	             << " (SDDF tag " << sddfTag << ") and was skipped. \n";
		continue;
            }

	    RecordDossier& dossier = RecordDict->fetch( sddfTag );
	    if ( ! dossier.notNull() ) {
		// We really don't expect to get here if our input SDDF file
		// and this program are correct!
	        cout << "\nWARNING: Record number " << recordCount 
	             << " with SDDF tag " << sddfTag 
	             << " is not in the Record Dictionary and was skipped. \n";
		continue;
	    }

	    if ( valueCnt >= 4 ) {
	        clock = value[1]*1000000 + value[2];

	        dossier.setValue( "Clock - microseconds", clock );
	        dossier.setValue( "Node", value[3] );
	    } 

	    switch( sddfTag ) {

		/* format S */
	    	case 1:				// Trace_Begin
	    	case 2:				// Trace_End
	    	case 22:			// Mon_End
	      	  if ( valueCnt != 4 ) {
	 	      errorFlag = 1;
	          }
	        break;

		/* format C */
	    	case 3:				// Proc_Begin
	    	case 4:				// Proc_End
	    	case 5:				// Block_Begin
	    	case 6:				// Block_End
	    	case 7:				// Marker
	    	case 8:				// Global_Begin
	    	case 9:				// Global_End
	    	case 21:			// Mon_Begin
	          if ( valueCnt != 6 ) {
	 	      errorFlag = 1;
	          } else {
	              dossier.setValue( "File identifier", value[4] );
	      	      dossier.setValue( "Object identifier", value[5] );
		  }
	          break;

		/* format M */
	    	case 10:			// Sync_Send
	    	case 11:			// Sync_Send_Blk
	    	case 13:			// Async_Send
	    	case 16:			// Sync_Recv
	    	case 18:			// Sync_Recv_Unblk
	    	case 20:			// Async_Recv_Unblk
	      	  if ( valueCnt != 10 ) {
	 	      errorFlag = 1;
	      	  } else {
	      	      dossier.setValue( "Other node", value[4] );
	      	      dossier.setValue( "Type of message", value[5] );
	      	      dossier.setValue( "Size of message", value[6] );
	      	      dossier.setValue( "File identifier", value[7] );
	      	      dossier.setValue( "Object identifier", value[8] );
	      	      dossier.setValue( "Message identifier", value[9] );
		  }
	          break;

		/* format SM */
	    	case 12:			// Sync_Send_Unblk
	    	case 14:			// Async_Send_Blk
	    	case 15:			// Async_Send_Unblk
	    	case 17:			// Sync_Recv_Blk
	    	case 19:			// Async_Recv_Blk
	      	  if ( valueCnt != 7 ) {
	 	      errorFlag = 1;
	          } else {
	      	      dossier.setValue( "File identifier", value[4] );
	      	      dossier.setValue( "Object identifier", value[5] );
	      	      dossier.setValue( "Message identifier", value[6] );
	          }
	          break;

	       /* format F */
	       case 23:				// Flush_Begin
	       case 24:				// Flush_End
	         if ( valueCnt != 11 ) {
	 	     errorFlag = 1;
	         } else {
	      	     clock = value[4]*1000000 + value[5];
	             dossier.setValue( "Accumulated flush time - microseconds", 
		   		       clock );
	      	     clock = value[6]*1000000 + value[7];
	      	     dossier.setValue( "Flush time - microseconds", clock );
	      	     dossier.setValue( "Count", value[8] );
	      	     dossier.setValue( "Bytes", value[9] );
	      	     dossier.setValue( "Total bytes", value[10] );
	         }
	         break;

	       /* format T */
	       case 25:				// Define_Grid
	       case 26:				// Define_Grid_Node
	         if ( valueCnt != 6 ) {
	  	     errorFlag = 1;
	         } else {
	             dossier.setValue( "Row", value[4] );
	             dossier.setValue( "Column", value[5] );
		 }
	         break;
	    
	       default:				// Unknown
	         errorFlag = -1;
	         break;

	   }					// end of switch on sddfTag

	   /* 
	    * Here we check for errors in the input and print messages
	    * if we found any.  If none occured, then we call the Writer
	    * method "putData" with the dossier we just populated.  The
	    * Writer sends the data, in SDDF format, to the output file.
	    */

	    if ( errorFlag == -1 ) {
		// We don't expect to get here because of checks before
		// the switch() but just in case...
	        cout << "\nWARNING: Record number " << recordCount 
	             << " had unknown AIMS type " << value[0]
	             << " (corresponding to SDDF tag " << sddfTag 
	             << ") and was skipped. \n";
	    } else if ( errorFlag == 1 ) {
	        cout << "WARNING: Record number " << recordCount << " shown "
	             << "had incorrect number of entries and was skipped. \n"
	             << Buffer 
		     << endl;
	    } else {
	        OutPipe->putData( dossier );
	      	if ( diag ) {
		   cerr << "\n***\n" << dossier;
		}
	    }

	}					// end of while()

	/*
  	 * Print the total count information.  Also write totals for each
	 * category as attributes to the SDDF file.
	 */
	Attributes	attributes;

	sprintf( Buffer, "Total Records Processed: " );
	sprintf( string, "%d", recordCount );

	cout << "\n\n" << Buffer << string << endl;
	attributes.insert( Buffer, string );

	for ( int t = 1; t <= MAXAIMSTYPE ; t++ ) {
	    sprintf( Buffer, "Records of type %d: ", t );
	    sprintf( string, "%d ", typeCount[t] );

	    cout << Buffer << string << endl;
	    attributes.insert( Buffer, string );
	}

	sprintf( Buffer, "Records of invalid type : " );
	sprintf( string, "%d ", typeCount[0] );

	cout << Buffer << string << endl;
	attributes.insert( Buffer, string );

	OutPipe->putAttributes( attributes );
}

