/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * ConvertPICL.C Program to read a compact format PICL trace file and
 *	convert it to an Ascii or Binary SDDF file.
 *
 */
#include <stream.h>
#include <string.h>
#include <time.h>

#include "InitializeStatic.C"

#include "AsciiPipeReader.h"
#include "AsciiPipeWriter.h"
#include "BinaryPipeWriter.h"
#include "InputFileStreamPipe.h"
#include "OutputFileStreamPipe.h"

#include "Attributes.h"
#include "PacketHeader.h"
#include "RecordDictionary.h"
#include "StructureDescriptor.h"

void processRecordFormats( PipeReader *InPipe, PipeWriter *OutPipe, int diag );
void processTraceFile( FILE *inFile, PipeWriter *OutPipe, int diag );

#define MAXPICLTYPE	21		/* valid PICL types are 1 to 21 */

RecordDictionary	*RecordDict;	/* our global record dictionary */


/******************************************************************************
 *  main() accepts user input, opens files, writes general configuration      *
 *  information to the output file, calls routines to do the hard work        *
 *  closes files, and exits.				                      *
 *									      *
 *  Note that we query the user and send informational and general diagnostic *
 *  messages to stdout.  We use stderr for reporting messages generated as a  *
 *  result of selecting "detailed diagnostics".  This allows the user to      *
 *  redirect stderr (the verbose diagnostics) to a file while still seeing    *
 *  questions and general information messages on stdout.                     *
 ******************************************************************************/

main()
{
	RecordDict = new RecordDictionary();	// Our global dictionary

	enum Status { INVALID, VALID };
	Status 	fileStatus;			// Status of file open        
	char    BUF[512];			// Buffer for keyboard input 

	/*******************************************************************
	 *    Get the name of the Ascii SDDF file containing the
	 *    descriptions of the SDDF record formats we will be 
	 *    generating.  Open the file and attach a Reader to it.
	 *******************************************************************/

	InputFileStreamPipe  	*FormatFile;	     // Pipe connected to the
						     // input format file.
	PipeReader		*FormatPipeReader;   // Reads SDDF packets 

	do {
		cout << "Ascii SDDF file containing record formats to output? ";
		cin >> BUF;

		/* 
		 * Create an input file stream pipe instance connected to
		 * the named file on disk.  If file open is successful, 
		 * connect an Ascii SDDF reader to the pipe.
		 */
		FormatFile = new InputFileStreamPipe( BUF );
		if ( FormatFile->successfulOpen() ) {
			fileStatus = VALID;
		        FormatPipeReader = new AsciiPipeReader( FormatFile );
		} else {
			cout << "ERROR: unable to open specified file.\n";
			fileStatus = INVALID;
			delete FormatFile;
		}
	} while ( fileStatus == INVALID );

	/*******************************************************************
	 *    Get the name of the PICL trace file. Open the file.
	 *******************************************************************/

	FILE *traceFile;		// PICL trace file to be converted
	char  traceFileName[256];	// Name of trace file

  	do {
      		cout << "PICL trace file (compact format) for input? ";
	  	cin >> BUF;
		traceFile = fopen( BUF, "r" );
	  	if ( traceFile == NULL ) {
			cout << "Error opening PICL trace file for input.\n" ;
		} else {
			strcpy( traceFileName, BUF );
		}
	} while ( traceFile == NULL );


	/*******************************************************************
	 *    Ask if the output file should be Ascii or Binary format.
	 * *****************************************************************/

	enum FileType	{ UNDEFINED, ASCII, BINARY };
	FileType	fileType = UNDEFINED;

	while ( fileType == UNDEFINED ) {
	     	cout << "Do you want Ascii or Binary SDDF format (A or B) ? ";
		cin >> BUF;
		if ( BUF[0] == 'a' || BUF[0] == 'A' ) {
			fileType = ASCII;
		} else if ( BUF[0] == 'b' || BUF[0] == 'B' ) {
			fileType = BINARY;
		} else {
			cout << "ERROR: invalid format type.\n";
		}
	}

	/******************************************************************
	 *    Get the name of the output file. Open or Create it and 
	 *    attach a Writer to it.
	 ******************************************************************/

	OutputFileStreamPipe	*OutFile;	// Pipe connected to output file
	PipeWriter   		*PipeWriter;	// Writes SDDF packets to pipe 
	
	do {
		cout << "Please enter name of the output SDDF file: ";
		cin >> BUF;

		/* 
		 * Create an output file stream pipe instance connected to
		 * the named file on disk.  If file open is successful, 
		 * connect the appropriate (Ascii or Binary)  SDDF writer 
		 * to the pipe.
		 */
		OutFile = new OutputFileStreamPipe( BUF );
		if ( OutFile->successfulOpen() ) {
			fileStatus = VALID;
			if ( fileType == ASCII ) {
		        	PipeWriter = new AsciiPipeWriter( OutFile );
			} else {
				PipeWriter = new BinaryPipeWriter( OutFile );
			}
		} else {
			cout << "ERROR: unable to open specified file.\n";
			fileStatus = INVALID;
			delete OutFile;
		}
	} while ( fileStatus == INVALID );

	/******************************************************************
	 *    Ask if verbose diagnostic messages are desired.
	 ******************************************************************/

        int diagOutput;

        cout << "Do you want detailed diagnostics sent to stderr (Y or N) ? ";
        cin >> BUF;
        if ( BUF[0] == 'y' || BUF[0] == 'Y' ) {
            	diagOutput = 1;
        } else {
            	diagOutput = 0;
        }

	/******************************************************************
	 *    Here's where we start the real work.  
	 *
	 *    We begin by creating an Attributes instance and populating it
	 *    with information about the PICL file name and the date of the 
	 *    conversion.  We use the Writer to send this packet to the 
	 *    output file.
	 *
	 *    Next, we call processRecordFormats() to load our dictionary
	 *    with the record descriptor information from the Ascii SDDF
	 *    file.  We also send that information to the output file using
	 *    the Writer.
	 *
	 *    Finally, we call processTraceFile() to read records from the
	 *    original PICL trace file. We lookup each record tag in our 
	 *    dictionary, parse the record, and use the Writer to send the
	 *    corresponding SDDF packet to the output file.
	 ******************************************************************/

	time_t now = time( 0 );
	char *date = ctime( &now );
	char *nl = strchr( date, '\n');
	if ( nl ) *nl = 0;

	Attributes	attributes;
	attributes.insert( "Trace from PICL data file", traceFileName );
	attributes.insert( "File converted", date );
	PipeWriter->putAttributes( attributes );

	cout << "\nStarting to process format file.\n";
	processRecordFormats( FormatPipeReader, PipeWriter, diagOutput );
	delete FormatPipeReader;		// Finished with Format File
	
	cout << "Starting to process PICL trace file.\n";
	processTraceFile( traceFile, PipeWriter, diagOutput );

	delete PipeWriter;
	delete FormatFile;
	delete OutFile; 			// Closes the file

}

/******************************************************************************
 *  processRecordFormats() reads SDDF packets from the Reader InPipe, loads   *
 *  the Descriptor information into our RecordDictionary, and rewrites the    *
 *  corresponding SDDF packets to the Writer OutPipe.  We expect these        *
 *  packets to contain record format information (Descriptor packets), but    *
 *  in fact they may be any of the four SDDF packet types.  		      *	
 ******************************************************************************/

void
processRecordFormats( PipeReader *InPipe, PipeWriter *OutPipe, int diag )
{
	Attributes		*Ap;
	StructureDescriptor	*SDp;
	PacketHeader PH = InPipe->getPacketHeader();

        while( PH.type != PIPE_EMPTY ) {

            switch( PH.type ) {
              case PKT_ATTRIBUTE:
            	  Ap = new Attributes();
                  InPipe->getAttributes( *Ap );
                  OutPipe->putAttributes( *Ap );
                  if ( diag ) cerr << "\n" << *Ap;
                  delete Ap;
                  break;

              case PKT_DESCRIPTOR:
                  SDp = new StructureDescriptor();
                  InPipe->getDescriptor( *SDp );
                  OutPipe->putDescriptor( *SDp, PH.tag );
                  RecordDict->insert( PH.tag, *SDp );
                  if ( diag ) cerr << "\n" << *SDp;
                  delete SDp;
                  break;

              case PKT_DATA:
                  InPipe->getData( RecordDict->fetch( PH.tag ) );
                  OutPipe->putData( RecordDict->fetch( PH.tag ) );
                  if ( diag ) cerr <<"\n"<< ( RecordDict->fetch( PH.tag ) );
                  break;

              case PKT_COMMAND:
                  OutPipe->putCommand( PH.tag );
                  if ( diag ) cerr << "\nCommand tag = " << PH.tag << "\n";
                  break;
            }
            PH = InPipe->getPacketHeader();

        }
}

/******************************************************************************
 *  processTraceFile() reads lines from the compact format PICL file,         *
 *  parses the data, and write the corresponding packet to the SDDF           *
 *  output file.							      *
 ******************************************************************************/

void
processTraceFile( FILE *inFile, PipeWriter *OutPipe, int diag )
{
	/* 
	 * Keep some count of records and types processed....  
	 */
	int recordCount = 0;
	int typeCount[ MAXPICLTYPE+1 ];	
	for ( int i = 0; i <= MAXPICLTYPE ; typeCount[i++] = 0 );

	/* 
	 * Read the trace records 
	 */
	char	Buffer[2048];	// Buffer to hold record
	char   *Bufp;		// Buffer pointer
	int	value[10];	// Individual values in record
	int	valueCnt;	// Number of values found
	int	clock;		// Value of merged second & microsecond entries
	char	string[128];	// Buffer to hold string


	int errorFlag = 0;
	cout << "Currently processing record... \n";

	while ( ( Bufp = fgets( Buffer, 2048, inFile ) ) != NULL ) {

	    valueCnt = sscanf( Bufp, "%d %d %d %d %d %d %d %d %d %d",
		               &value[0], &value[1], &value[2], &value[3], 
			       &value[4], &value[5], &value[6], &value[7], 
			       &value[8], &value[9] );

	    /* 
	     * Update our counters and check that values were found 
	     */
	    if ( ( ++recordCount % 100 ) == 0 ) {
		cout << recordCount << "... ";
		cout.flush();
	    }

	    if (  ( valueCnt == 0 ) || ( valueCnt == EOF ) ) {
		cout << "\nError: Unable to get values for record number " 
		     << recordCount << "\n";
		continue;
	    }

	    if ( ( 0 < value[0] ) && ( value[0] <= MAXPICLTYPE ) ) {
		++typeCount[ value[0] ];	// expected type
	    } else {
		++typeCount[ 0 ];		// unexpected type
	    }

	    /*
	     * All the records should have values for seconds and microseconds.
	     * Merge those into a single clock value.
	     */
	    if ( valueCnt >= 3 ) {
		clock = value[1]*1000000 + value[2];
	    }

	    /*	
	     * Process the different record types.  Value[0], the first
	     * integer in each record, identifies the type of the PICL 
	     * trace record.  We use that tag to look up the record in
	     * our Record Dictionary.  The dictionary contains the field
	     * descriptors and types for each record.  It also contains
	     * space to hold the field values.  We parse the data we have
	     * read in from our PICL trace file, and use it to populate
	     * the field values for this type of record in the Record
	     * Dictionary.  We then call the Writer method "putData" with
	     * the Record Dictionary entry we have just populated, and the
	     * Writer sends the data, in SDDF format, to the output file.
	     */
	    switch( value[0] ) {
	    case 1:				// Trace Start
	      if ( valueCnt != 7 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 1, "Clock - microseconds", clock );
		  RecordDict->setValue( 1, "Node", value[3] );
		  RecordDict->setValue( 1, "Event", value[4] );
		  RecordDict->setValue( 1, "Computation Stats", value[5] );
		  RecordDict->setValue( 1, "Communication Stats", value[6] );
		  OutPipe->putData( RecordDict->fetch( 1 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 1 ) );
	      }
	      break;

	    case 2:				// Open
	      if ( valueCnt != 4 && valueCnt != 5 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 2, "Clock - microseconds", clock );
		  RecordDict->setValue( 2, "Node", value[3] );
		  if ( value[3] == -32768 ) {	// Open from host 
		      if ( valueCnt != 5 ) {
			  errorFlag = 1;
		      } else {
			  RecordDict->setValue( 2, 
					"Processors Allocated", value[4] );
	              }	
		  } else {			// Open from node
		      if ( valueCnt != 4 ) {
		 	  errorFlag = 1;
		      } else {
			  RecordDict->setValue( 2, "Processors Allocated", 0 );
		      }
		  }
		  if ( errorFlag == 0 ) {
		      OutPipe->putData( RecordDict->fetch( 2 ) );
		      if ( diag ) {
			  cerr << "\n****\n" << ( RecordDict->fetch( 2 ) );
		      }
		  }
	      }
	      break;

	    case 3:				// Load
	      if ( valueCnt != 5 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 3, "Clock - microseconds", clock );
		  RecordDict->setValue( 3, "Node", value[3] );
		  RecordDict->setValue( 3, "Loading Node", value[4] );
		  OutPipe->putData( RecordDict->fetch( 3 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 3 ) );
	      }
	      break;

	    case 4:				// Send
	      if ( valueCnt != 7 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 4, "Clock - microseconds", clock );
		  RecordDict->setValue( 4, "Sending Node", value[3] );
		  RecordDict->setValue( 4, "Receiving Node", value[4] );
		  RecordDict->setValue( 4, "Message Type", value[5] );
		  RecordDict->setValue( 4, "Message Length", value[6] );
		  OutPipe->putData( RecordDict->fetch( 4 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 4 ) );
	      }
	      break;

	    case 6:				// Receive
	      if ( valueCnt != 7 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 6, "Clock - microseconds", clock );
		  RecordDict->setValue( 6, "Receiving Node", value[3] );
		  RecordDict->setValue( 6, "Sending Node", value[4] );
		  RecordDict->setValue( 6, "Message Type", value[5] );
		  RecordDict->setValue( 6, "Message Length", value[6] );
		  OutPipe->putData( RecordDict->fetch( 6 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 6 ) );
	      }
	      break;

	    case 7:				// Receive-Blocking
	      if ( valueCnt != 5 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 7, "Clock - microseconds", clock );
		  RecordDict->setValue( 7, "Receiving Node", value[3] );
		  RecordDict->setValue( 7, "Message Type", value[4] );
		  OutPipe->putData( RecordDict->fetch( 7 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 7 ) );
	      }
	      break;

	    case 8:				// Receive-Waking
	      if ( valueCnt != 7 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 8, "Clock - microseconds", clock );
		  RecordDict->setValue( 8, "Receiving Node", value[3] );
		  RecordDict->setValue( 8, "Sending Node", value[4] );
		  RecordDict->setValue( 8, "Message Type", value[5] );
		  RecordDict->setValue( 8, "Message Length", value[6] );
		  OutPipe->putData( RecordDict->fetch( 8 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 8 ) );
	      }
	      break;

	    case 9:				// Message
	      if ( valueCnt != 4 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 9, "Clock - microseconds", clock );
		  RecordDict->setValue( 9, "Node", value[3] );
		  OutPipe->putData( RecordDict->fetch( 9 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 9 ) );
	      }
	      break;

	    case 10:				// Sync
	      if ( valueCnt != 4 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 10, "Clock - microseconds", clock );
		  RecordDict->setValue( 10, "Node", value[3] );
		  OutPipe->putData( RecordDict->fetch( 10 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 10 ) );
	      }
	      break;

	    case 11:				// Computation Stats
	      if ( valueCnt != 6 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 11, "Clock - microseconds", clock );
		  RecordDict->setValue( 11, "Node", value[3] );
		  RecordDict->setValue( 11, "Idle - microseconds", 
					    value[4]*1000000 + value[5] );
		  OutPipe->putData( RecordDict->fetch( 11 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 11 ) );
	      }
	      break;

	    case 12:				// Communication Stats
	      if ( valueCnt != 9 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 12, "Clock - microseconds", clock );
		  RecordDict->setValue( 12, "Node", value[3] );
		  RecordDict->setValue( 12, "Messages Received", value[4] );
		  RecordDict->setValue( 12, "Bytes Received", value[5] );
		  RecordDict->setValue( 12, "Messages Sent", value[6] );
		  RecordDict->setValue( 12, "Bytes Sent", value[7] );
		  RecordDict->setValue( 12, "Probes of Message Queue",
								    value[8] );
		  OutPipe->putData( RecordDict->fetch( 12 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 12 ) );
	      }
	      break;

	    case 13:				// Close
	      if ( valueCnt != 4 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 13, "Clock - microseconds", clock );
		  RecordDict->setValue( 13, "Node", value[3] );
		  OutPipe->putData( RecordDict->fetch( 13 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 13 ) );
	      }
	      break;

	    case 14:				// Trace Level
	      if ( valueCnt != 7 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 14, "Clock - microseconds", clock );
		  RecordDict->setValue( 14, "Node", value[3] );
		  RecordDict->setValue( 14, "Event", value[4] );
		  RecordDict->setValue( 14, "Computation Stats", value[5] );
		  RecordDict->setValue( 14, "Communication Stats", value[6] );
		  OutPipe->putData( RecordDict->fetch( 14 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 14 ) );
	      }
	      break;

	    case 15:				// Trace Mark
	      if ( valueCnt != 5 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 15, "Clock - microseconds", clock );
		  RecordDict->setValue( 15, "Node", value[3] );
		  RecordDict->setValue( 15, "Type", value[4] );
		  OutPipe->putData( RecordDict->fetch( 15 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 15 ) );
	      }
	      break;

	    case 16:				// Trace Message
	      /* This contain integers and string so rescan */
	      valueCnt = sscanf( Bufp, "%d %d %d %d %[ -~]", &value[0], 
				 &value[1], &value[2], &value[3], string );
	      if ( valueCnt != 5 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 16, "Clock - microseconds", clock );
		  RecordDict->setValue( 16, "Node", value[3] );

		  Array *arrayP = RecordDict->getArrayP( 16, "Message" );
		  int slength = strlen( string );
		  arrayP->setDimSizes( &slength );
		  arrayP->setCellString( string );

		  OutPipe->putData( RecordDict->fetch( 16 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 16 ) );
	      }
	      break;

	    case 17:				// Trace Stop
	      /* This contain integers and string so rescan */
	      valueCnt = sscanf( Bufp, "%d %d %d %d %[ -~]", &value[0], 
				 &value[1], &value[2], &value[3], string );
	      if ( valueCnt != 5 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 17, "Clock - microseconds", clock );
		  RecordDict->setValue( 17, "Node", value[3] );

		  Array *arrayP = RecordDict->getArrayP( 17, "Description" );
		  int slength = strlen( string );
		  arrayP->setDimSizes( &slength );
		  arrayP->setCellString( string );

		  OutPipe->putData( RecordDict->fetch( 17 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 17 ) );
	      }
	      break;

	    case 18: 				// Trace Flush
	      if ( valueCnt != 6 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 18, "Clock - microseconds", clock );
		  RecordDict->setValue( 18, "Node", value[3] );
		  RecordDict->setValue( 18, "Finish Clock - microseconds", 
					    value[4] * 1000000 + value[5] );
		  OutPipe->putData( RecordDict->fetch( 18 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 18 ) );
	      }
	      break;

	    case 19: 				// Trace Exit
	      if ( valueCnt != 5 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 19, "Clock - microseconds", clock );
		  RecordDict->setValue( 19, "Node", value[3] );
		  RecordDict->setValue( 19, 
				    "Bytes of trace data generated", value[4] );
		  OutPipe->putData( RecordDict->fetch( 19 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 19 ) );
	      }
	      break;

	    case 20: 				// Block Begin
	      if ( valueCnt != 7 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 20, "Clock - microseconds", clock );
		  RecordDict->setValue( 20, "Node", value[3] );
		  RecordDict->setValue( 20, "Block Type", value[4] );
		  RecordDict->setValue( 20, "Location Type", value[5] );
		  RecordDict->setValue( 20, "Parameter Type", value[6] );
		  OutPipe->putData( RecordDict->fetch( 20 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 20 ) );
	      }
	      break;

	    case 21: 				// Block End
	      if ( valueCnt != 7 ) {
	 	  errorFlag = 1;
	      } else {
		  RecordDict->setValue( 21, "Clock - microseconds", clock );
		  RecordDict->setValue( 21, "Node", value[3] );
		  RecordDict->setValue( 21, "Block Type", value[4] );
		  RecordDict->setValue( 21, "Location Type", value[5] );
		  RecordDict->setValue( 21, "Parameter Type", value[6] );
		  OutPipe->putData( RecordDict->fetch( 21 ) );
		  if ( diag ) cerr << "\n****\n" << ( RecordDict->fetch( 21 ) );
	      }
	      break;
	    
	    default:				// Unknown
	      errorFlag = -1;
	      break;

	    }

	    if ( errorFlag == -1 ) {
		cout << "\nWARNING: Record number " << recordCount 
		     << " had unknown type " << value[0] 
		     << " and was skipped.\n";
	    } else if ( errorFlag == 1 ) {
		cout << "WARNING: Record number " << recordCount << " shown "
		     << "had incorrect number of entries and was skipped.\n"
		     << Buffer << "\n";
	    }
	    errorFlag = 0;
	}

	/*
  	 * Print the total count information.  Also write totals for each
	 * category as attributes to the SDDF file.
	 */
	Attributes	attributes;

	sprintf( Buffer, "Total Records Processed: " );
	sprintf( string, "%d", recordCount );

	cout << "\n\n" << Buffer << string << "\n";
	attributes.insert( Buffer, string );

	for ( int t = 0; t <= MAXPICLTYPE ; t++ ) {
	    sprintf( Buffer, "Records of type %d: ", t );
	    sprintf( string, "%d ", typeCount[t] );

	    cout << Buffer << string << "\n";
	    attributes.insert( Buffer, string );
	}

	OutPipe->putAttributes( attributes );
}

