/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * FindSddfError.C  Program to help narrow down the location of an error
 *                  in an SDDF file.
 *
 */
#include <stdlib.h>
#include <stream.h>

#include "InitializeStatic.C"

#include "PipeReader.h"
#include "AsciiPipeWriter.h"
#include "InputFileStreamPipe.h"
#include "OutputFileStreamPipe.h"

#include "Attributes.h"
#include "PacketHeader.h"
#include "RecordDictionary.h"
#include "StructureDescriptor.h"

void DEBUG( int, int );

main()
{
        enum FileType     { UNDEFINED, ASCII, BINARY, CONVERT };
	enum Status       { INVALID, VALID };

	char BUF[512];		/* buffer for input from stdin */

	/**** DEBUG ****/
	cerr << "For debugging, you can enter the subroutine 'DEBUG'\n";
	cerr << "after a given number of packets have been processed.\n";
        cerr << "Number of packets before calling DEBUG (-1 = don't call): ";
	cin >> BUF;
	int stopAfter = atoi(BUF);
	/**** DEBUG ****/

 	/****************************************************************
         *    Get the name of the input file, and open the file
         ****************************************************************/
	Status 		     inputFileStatus;
	InputFileStreamPipe  *In;
	PipeReader     	     *InPipe;

	do {
  	    cerr << "Please enter name of the input SSDF file: ";
	    cin >> BUF;

	    In = new InputFileStreamPipe( BUF );
	    if ( In->successfulOpen() ) {
		inputFileStatus = VALID;
		InPipe = In->createPipeReader();
		if ( InPipe == NULL ) {
		    cerr << "ERROR: Couldn't attach pipe to input file\n";
		    inputFileStatus = INVALID;
		    delete In;
		}
	    } else {
		cerr << "ERROR: Problem with input file\n";
		inputFileStatus = INVALID;
		delete In;
	    }
	} while ( inputFileStatus == INVALID );

	cerr << "\n";
	switch( In->getInputType() ) {
	    case ASCII_SDDF:
		cerr << "File is in SDDF ASCII format";
		break;
	    case BINARY_SDDF:
		cerr << "File is in SDDF Binary format";
		break;
	    case CONVERT_BINARY_SDDF:
		cerr << "File is in SDDF Converted (reverse byte order) format";
		break;
	    default:
		cerr << "File is not in SDDF format";
		break;
	}
	cerr << "\n";

 	/****************************************************************
         *    Get the name of the output file, and open the file
         ****************************************************************/
	Status 		     outputFileStatus;
	OutputFileStreamPipe *Out;
	PipeWriter           *OutPipe;

	do {
  	    cerr << "Name of ASCII SDDF file for output of valid packets: ";
	    cin >> BUF;

	    Out = new OutputFileStreamPipe( BUF );
	    if ( Out->successfulOpen() ) {
		outputFileStatus = VALID;
		OutPipe = new AsciiPipeWriter( Out );
	    } else {
		outputFileStatus = INVALID;
		delete Out;
	    }
	} while ( outputFileStatus == INVALID );

 	/****************************************************************
         *    Process the packets in the files.  Read from the input 
	 *    pipe and write to the output pipe.
         ****************************************************************/
	Attributes *Ap;
	StructureDescriptor *SDp;
	RecordDictionary RD;

	/**** DEBUG ****/
	int pktNumber = 1;
	off_t inOffset = InPipe->getOffset();
	off_t outOffset = OutPipe->getOffset();

	cerr << form( "\nBefore reading Packet %d at offset %d\n", 
		       pktNumber, inOffset );
	/**** DEBUG ****/

	PacketHeader PH = InPipe->getPacketHeader();

	while( PH.type != PIPE_EMPTY ) {
	    /**** DEBUG ****/
	    inOffset = InPipe->getOffset();
	    cerr << form( "  Type %d; Tag %d; Length %d; Next Offset %d\n",
	   	             PH.type, PH.tag, PH.length, inOffset );
	    cerr.flush();
	    /**** DEBUG ****/

	    switch( PH.type ) {
	      case PKT_ATTRIBUTE:
		  Ap = new Attributes();
		  InPipe->getAttributes( *Ap );
		  OutPipe->putAttributes( *Ap );
		  delete Ap;
		  break;
	      case PKT_DESCRIPTOR:
		  SDp = new StructureDescriptor();
		  InPipe->getDescriptor( *SDp );
		  OutPipe->putDescriptor( *SDp, PH.tag );
		  RD.insert( PH.tag, *SDp );
		  delete SDp;
		  break;
	      case PKT_DATA:
		  {				
		      RecordDossier& dossier = RD.fetch( PH.tag );
		      InPipe->getData( dossier );
		      OutPipe->putData( dossier );
		  }
		  break;
	      case PKT_COMMAND:
		  OutPipe->putCommand( PH.tag );
		  break;
	      }

	      /*** DEBUG ****/
	      outOffset = OutPipe->getOffset();	   // forces flush of output

	      if ( ( stopAfter != -1 ) && ( pktNumber >= stopAfter ) ) {
		  DEBUG( pktNumber, (int)outOffset );	
	      }
		
	      pktNumber++;
	      cerr << form( "Before reading Packet %d at offset %d\n", 
			     pktNumber, inOffset );
	      /**** DEBUG ****/
	      
	      PH = InPipe->getPacketHeader();

	}
	cerr << form( "No more packets\n" );

	delete InPipe;
	delete In;
	delete OutPipe;
	delete Out;
}

void
DEBUG( int pktNo, int outOffset )
{
	cerr << form( "DEBUG: Wrote packet %d; output file has %d bytes\n",
		       pktNo, outOffset );
}

