/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Tara M. Madhyastha (tara@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */

#include <stream.h>
#include <stdlib.h>
#include <string.h>
#include <values.h>

#ifndef __GNUG__
#include "AwesimeFiles/SampleStatistic.h"
#include "AwesimeFiles/SampleHistogram.h"
#else
#include <SmplHist.h>
#endif



#ifdef __GNUG__
extern "C" {
#include <search.h>
ENTRY *hsearch();
int hcreate();
void hdestroy();
};
#else
extern "C" {
#include <search.h>
}
ENTRY *hsearch(ENTRY item, ACTION action);
int hcreate(unsigned nel);

#endif

#define MINCHAR	0x00
#define MAXCHAR 0x7f

#ifndef MININT
#define MININT ( 1 << ( BITS(int) - 1 ) )
#endif

#ifndef MAXINT
#define MAXINT ~MININT
#endif

#include "InitializeStatic.C"

#include "PipeReader.h"
#include "InputFileStreamPipe.h"

#include "Attributes.h"
#include "PacketHeader.h"
#include "RecordDictionary.h"
#include "RecordDictionaryPIterator.h"
#include "StructureDescriptor.h"
#include "StructureDescriptorIterator.h"

/* info stored in hash table other than the key */
struct info {
  float readTime;
  int   readBytes;
  int   readCount;
  float writeTime;
  int writeBytes;
  int writeCount;
  float seekTime;
  int   seekBytes;
  int  seekCount;
  float overhead;
  float lifetime;
};

int FileLifetimeSummaries=0;


#define NUM_FILES   100


#define STAT_TAG 10000


extern void buildStatRecord( StructureDescriptor *inSDp, 
			     StructureDescriptor *statSDp );

extern void updateStatRecord( RecordDossier& inDossier, 
			      RecordDossier& statDossier );

extern void printStatRecords( RecordDictionary& RDict );

extern void updateIOStats(RecordDossier& inDossier);

extern void printIOStats();

extern void printValue( const Value& value );


/* hashing stuff */
char string_space[NUM_FILES*256];
struct info info_space[NUM_FILES];

char *str_ptr = string_space;
char *init_str_ptr = string_space;
struct info *info_ptr = info_space;

SampleHistogram LifetimeHist(0, 1000., 100.);

SampleHistogram readTimeStats(0, 1., .1), 
                writeTimeStats(0, 1., .1),
		seekTimeStats(0, 1., .1), 
                readByteStats(0, 10000, 1000), 
		writeByteStats(0, 10000, 1000), 
		seekByteStats(0, 10000, 1000), 
		ioWaitTimeStats(0, 1., .1); 

SampleHistogram ioDistanceStats(0, 1, .1);
                
double minTimestamp = MAXDOUBLE;
double  maxTimestamp = 0.0;

int fileStatsFlag = 0;

void usage(char *s)
{
  cerr << form( "Usage: %s [-f]  filename\n", s );
  exit (-1);
}

main( int argc, char **argv )
{
 	/****************************************************************
         *    Get the name of the input file, and open the file
         ****************************************************************/
	enum Status       { INVALID, VALID };

	char BUF[512];		
	Status 		     inputFileStatus;
	InputFileStreamPipe  *In;
	PipeReader     	     *InPipe;


	(void) hcreate(NUM_FILES);
	memset((char *)info_space, 0, sizeof(struct info) * NUM_FILES);

        if ((argc != 2) && (argc != 3)) {
	  usage(argv[0]);
	}

	if ( argc == 3 ) {
	  if (strcmp(argv[1], "-f") == 0) {
	    fileStatsFlag = 1;
	    strcpy(BUF, argv[2]);
	  } else {
	    usage(argv[0]);
	  }
	}
	
	if ( argc == 2 ) {
	  strcpy( BUF, argv[1] );
	}

	In = new InputFileStreamPipe( BUF );
	if ( In->successfulOpen() ) {
	  inputFileStatus = VALID;
	  InPipe = In->createPipeReader();
	  if ( InPipe == NULL ) {
	    cerr << "ERROR: Couldn't attach pipe to input file\n";
	    inputFileStatus = INVALID;
	    delete In;
	  }
	} else {
	  cerr << "ERROR: Problem with input file\n\n";
	  inputFileStatus = INVALID;
	  delete In;
	}
	

 	/****************************************************************
         *    Process the packets in the files.  Read from the input 
	 *    pipe and update the profile information.
         ****************************************************************/
	RecordDictionary     RecDict;

	int pktCount = 0;
	int attrCount = 0; 
	int descrCount = 0;
	int dataCount = 0;
	int cmdCount = 0;
	int duplicateCount = 0;

	StructureDescriptor *origSDp;
	StructureDescriptor *statSDp;
	CString		     statName;

	cerr << "Currently processing packet... \n";

	PacketHeader PH = InPipe->getPacketHeader();

	while( PH.type != PIPE_EMPTY ) {
	    if ( ( ++pktCount % 1000 ) == 0 ) {
	        cerr << pktCount << "... ";
		cerr.flush();
	    }

	    switch( PH.type ) {

	      case PKT_ATTRIBUTE:
		  attrCount++;
		  break;

	      case PKT_DESCRIPTOR:
		  descrCount++;
		  origSDp = new StructureDescriptor();
		  InPipe->getDescriptor( *origSDp );

		  if ( RecDict.insert( PH.tag, *origSDp ) == SUCCESS_ ){
		      statName = origSDp->getName() + ":";
		      statSDp = new StructureDescriptor( statName );
		      buildStatRecord( origSDp, statSDp );
		      if ( ! RecDict.insert( PH.tag+STAT_TAG, *statSDp ) ){
			  cerr << "ERROR: Duplicate Stat tag seen ";
			  cerr << PH.tag+STAT_TAG << "\n";
			  exit( -1 );
		      }
		      delete statSDp;
		  } else {
		      duplicateCount++;
		  }

		  delete origSDp;
		  break;

	      case PKT_DATA:
		  {
		  dataCount++;
	          RecordDossier& origDossier = RecDict.fetch( PH.tag );
		  InPipe->getData( origDossier );
		  updateIOStats(origDossier);
		  }
		  break;

	      case PKT_COMMAND:
		  cmdCount++;
		  break;

	    }
	    PH = InPipe->getPacketHeader();
	}

	printf( "\nThere are %d packets in the file %s.\n", 
		 	    attrCount+descrCount+dataCount+cmdCount, BUF );

	printf( "%d Descriptor; %d Data; %d Attribute; %d Command.\n",
		 	    descrCount, dataCount, attrCount, cmdCount );

     printIOStats();

	if ( duplicateCount != 0 ) {
	    printf( "%d of the Descriptor packets had duplicate tags.\n", 
			    duplicateCount );
	}

/*	printStatRecords(); */

	delete InPipe;
	delete In;
}

void
updateIOStats(RecordDossier& inDossier)
{
  ENTRY item, *found_item;
  struct info *ioinfo;
  Array *inArrayP;
  int inCellCount,i;
  char name[256], *s;
  Value newValue;
  double ts;
  static double lastIOTime=0;
  
/* Every field had better have a timestamp, double, called Seconds */
  if (lastIOTime == 0) {
    lastIOTime = inDossier.getValue("Seconds");
  }

  if (inDossier.getName() == "Read") {
    ts = inDossier.getValue("Seconds");
    if (ts > maxTimestamp) { maxTimestamp = ts; }
    if (ts < minTimestamp) { minTimestamp = ts; }
    
    readByteStats += inDossier.getValue("Number Bytes");
    readTimeStats += inDossier.getValue("Duration");
    ioDistanceStats += (ts - lastIOTime);
    lastIOTime = ts;
  }

  if (inDossier.getName() == "Write") {
    ts = inDossier.getValue("Seconds");
    if (ts > maxTimestamp) { maxTimestamp = ts; }
    if (ts < minTimestamp) { minTimestamp = ts; }
    
    writeByteStats += inDossier.getValue("Number Bytes");
    writeTimeStats += inDossier.getValue("Duration");

    ioDistanceStats += (ts - lastIOTime);
    lastIOTime = ts;
  }

  if (inDossier.getName() == "Seek") {
    ts = inDossier.getValue("Seconds");
    if (ts > maxTimestamp) { maxTimestamp = ts; }
    if (ts < minTimestamp) { minTimestamp = ts; }
    
    seekTimeStats += inDossier.getValue("Duration");
    ioDistanceStats += (ts - lastIOTime);
    lastIOTime = ts;
  }


  if (inDossier.getName() == "IO Wait End") { 
    ts = inDossier.getValue("Seconds");
    if (ts > maxTimestamp) { maxTimestamp = ts; }
    if (ts < minTimestamp) { minTimestamp = ts; }

    ioWaitTimeStats += inDossier.getValue("Total IO Time");
  }
  

  if (inDossier.getName() == "File Lifetime Summary") {
    if (FileLifetimeSummaries == 0) {FileLifetimeSummaries = 1; }
    ts = inDossier.getValue("Seconds");
    if (ts > maxTimestamp) { maxTimestamp = ts; }
    if (ts < minTimestamp) { minTimestamp = ts; }

    inArrayP = inDossier.getArrayP("File Name");
    inCellCount = inArrayP->getCellCount();
    s = name;
    for(i=0; i < inCellCount; i++) {
      inArrayP->getTheCellValue(newValue,i);
      name[i]= (char) newValue;
    }
    name[inCellCount]= '\0';

    item.key = 	name;
    

/*    if (strlen(item.key) > 0) {
    fprintf(stderr, "file name is %s\n", item.key);
  } */
    if ((found_item = hsearch(item, FIND)) != NULL) {
			       /* item is in table */
//      fprintf(stderr, "found entry for %s\n", item.key);
      ioinfo = (struct info *) found_item->data;
      ioinfo->readTime += (float) inDossier.getValue("Read Time");
      ioinfo->readBytes += (int) inDossier.getValue("Read Number Bytes");
      ioinfo->readCount += (int) inDossier.getValue("Read Frequency");
      ioinfo->writeTime += (float)inDossier.getValue("Write Time");
      ioinfo->writeBytes += (int) inDossier.getValue("Write Number Bytes");
      ioinfo->writeCount += (int) inDossier.getValue("Write Frequency");
      ioinfo->seekTime += (float) inDossier.getValue("Seek Time");
      ioinfo->seekBytes += (int) inDossier.getValue("Seek Number Bytes");
      ioinfo->seekCount += (int) inDossier.getValue("Seek Frequency");
      ioinfo->overhead += (float) inDossier.getValue("Overhead");
      ioinfo->lifetime += (float) inDossier.getValue("File Lifetime");
    } else {
//      fprintf(stderr, "making entry for %s\n", item.key);
      /* insert item into hash table */
      ioinfo = (struct info *) info_ptr;

      ioinfo->readTime += (float) inDossier.getValue("Read Time");
      ioinfo->readBytes += (int) inDossier.getValue("Read Number Bytes");
      ioinfo->readCount += (int) inDossier.getValue("Read Frequency");
      ioinfo->writeTime += (float)inDossier.getValue("Write Time");
      ioinfo->writeBytes += (int) inDossier.getValue("Write Number Bytes");
      ioinfo->writeCount += (int) inDossier.getValue("Write Frequency");
      ioinfo->seekTime += (float) inDossier.getValue("Seek Time");
      ioinfo->seekBytes += (int) inDossier.getValue("Seek Number Bytes");
      ioinfo->seekCount += (int) inDossier.getValue("Seek Frequency");
      ioinfo->overhead += (float) inDossier.getValue("Overhead");
      ioinfo->lifetime += (float) inDossier.getValue("File Lifetime");
      strcpy(str_ptr, item.key);

      item.data = (char *)info_ptr;
      str_ptr += strlen(str_ptr) + 1;
      info_ptr++;
      (void) hsearch(item, ENTER);
    }

  }

}

void
printIOStats()
{
  int i = 0;
  char *s;
  struct info summary;
  s = init_str_ptr;
  memset((char *)&summary, 0, sizeof(struct info));
  SampleStatistic lifetimeStats;
  
  extern void doLifetimes();

  extern void doLevel0stats();



  if (FileLifetimeSummaries) {
    doLifetimes();
  } else {

  doLevel0stats();
}

}

void 
doLevel0stats()
{
  printf("Basic Instrumentation Summary\n");
  printf("-----------------------------\n");

  printf("Total Number Reads:          %d\n", readByteStats.samples());
  printf("Total Bytes Read:            %d\n", (int) readByteStats.mean() * readByteStats.samples());
  printf("Mean Read Size:              %3.3f bytes\n", readByteStats.mean());
  printf("Read Size -- Std Deviation:  %3.3f\n", readByteStats.stdDev());
  printf("Read Size -- Variance:       %3.3f\n", readByteStats.var());
  printf("Mean Read Time:              %3.3f\n", readTimeStats.mean());
  printf("Read Time -- Std Deviation:  %3.3f\n", readTimeStats.stdDev());
  printf("Read Time -- Variance:       %3.3f\n", readTimeStats.var());  
  

  printf("Total Number Writes:         %d\n", writeByteStats.samples());
  printf("Total Bytes Written:         %3.3f\n", writeByteStats.mean() * writeByteStats.samples());
  printf("Mean Write Size:             %d bytes\n", (int) writeByteStats.mean());
  printf("Write Size -- Std Deviation: %3.3f\n", writeByteStats.stdDev());
  printf("Write Size -- Variance:      %3.3f\n", writeByteStats.var());
  printf("Mean Write Time:             %3.3f\n", writeTimeStats.mean());
  printf("Write Time -- Std Deviation: %3.3f\n", writeTimeStats.stdDev());
  printf("Write Time -- Variance:      %3.3f\n", writeTimeStats.var());  

  printf("Total Number Seeks:          %d\n", seekTimeStats.samples());
  printf("Mean Seek Time:              %3.3f\n", seekTimeStats.mean());
  printf("Seek Time -- Std Deviation   %3.3f\n", seekTimeStats.stdDev());
  printf("Seek Time -- Variance        %3.3f\n", seekTimeStats.var());

  if (readTimeStats.mean() > 0) {
    printf("Read Throughput:             %3.2f bytes/sec\n", (readByteStats.mean())/(readTimeStats.mean()));
  }

  if (writeTimeStats.mean() > 0) {
    printf("Write Throughput:             %3.2f bytes/sec\n", (writeByteStats.mean())/(writeTimeStats.mean()));
  }
  printf("\n");
  printf("Mean time between I/Os:       %f sec\n", ioDistanceStats.mean());
  printf("Histogram of time between I/O requests (secs)\n");
  ioDistanceStats.printBuckets(cout);

  printf("\n");

  if (readByteStats.samples() > 0) {
    printf("Histogram of Read Times(0 -- 1 sec)\n");
    readTimeStats.printBuckets(cout);
    printf("Histogram of Read Sizes(0 -- 10000 bytes)\n");
    readByteStats.printBuckets(cout);
  }

  if (writeByteStats.samples() > 0) {
    printf("Histogram of Write Times(0 -- 1 sec)\n");
    writeTimeStats.printBuckets(cout);
    printf("Histogram of Write Sizes(0 -- 10000 bytes)\n");
    writeByteStats.printBuckets(cout);
  }


}


void
doLifetimes()
{
  int i = 0;
  char *s;
  struct info summary;
  s = init_str_ptr;
  memset((char *)&summary, 0, sizeof(struct info));
  SampleHistogram lifetimeStats(0, 1000, 100);

  if (fileStatsFlag == 1) {
    printf("Filename\t\trTime\trCount\trBytes\twTime\twCount\twBytes\tsTime\tsCount\tsBytes\tohead\tlife\n");
  }
  while(strlen(s) > 0) {
    
    if (fileStatsFlag == 1) {
      printf("%16s\t%3.2f\t%d\t%d\t%3.2f\t%d\t%d\t%3.2f\t%d\t%d\t%3.2f\t%3.2f\n",
	     s,
	     info_space[i].readTime,
	     info_space[i].readCount,
	     info_space[i].readBytes,
	     info_space[i].writeTime,
	     info_space[i].writeCount,
	     info_space[i].writeBytes,
	     info_space[i].seekTime,
	     info_space[i].seekCount,
	     info_space[i].seekBytes,
	     info_space[i].overhead,
	     info_space[i].lifetime);
    }

      lifetimeStats+= info_space[i].lifetime;
      

      summary.readTime +=  info_space[i].readTime;
      summary.readCount += info_space[i].readCount;
      summary.readBytes += info_space[i].readBytes;
      summary.writeTime += info_space[i].writeTime;
      summary.writeCount += info_space[i].writeCount;
      summary.writeBytes += info_space[i].writeBytes;
      summary.seekTime += info_space[i].seekTime;
      summary.seekCount += info_space[i].seekCount;
      summary.seekBytes += info_space[i].seekBytes;
      summary.overhead += info_space[i].overhead;
      summary.lifetime += info_space[i].lifetime;


      s+=strlen(s) +1;
      i++;
    }
  printf("Overall:  Total Read Time:      %3.2f\n", summary.readTime);
  printf("          Number Reads:         %d\n", summary.readCount);
  printf("          Number Bytes Read:    %d\n",summary.readBytes);
  printf("          Total Write Time:     %3.2f\n", summary.writeTime);
  printf("          Number Writes:        %d\n", summary.writeCount);
  printf("          Number Bytes Written: %d\n", summary.writeBytes);
  printf("          Total Seek Time:      %3.2f\n", summary.seekTime);
  printf("          Number Seeks:         %d\n", summary.seekCount);
  printf("          Number Bytes Seeked:  %d\n", summary.seekBytes);
  printf("          Open/Close Overhead:  %3.2f\n", summary.overhead);
  printf("          Sum of Lifetimes:     %3.2f\n\n", summary.lifetime); 


  printf("Mean Read Size:              %3.3f bytes\n", (float) summary.readBytes/summary.readCount);
  printf("Mean Write Size:             %3.3f bytes\n", (float) summary.writeBytes/summary.writeCount);
  printf("Mean Seek Size:              %3.3f bytes\n", (float) summary.seekBytes/summary.seekCount);
  printf("Mean File Lifetime:          %3.3f sec\n", lifetimeStats.mean());
  printf("Variance -- File Lifetime:   %3.3f sec\n", lifetimeStats.var());
  printf("Std. Dev -- File Lifetime:   %3.3f sec\n", lifetimeStats.stdDev());
  printf("%% Time Reading:              %3.5f\n", 100. * summary.readTime/(maxTimestamp - minTimestamp));
  printf("%% Time Writing:              %3.5f\n", 100. * summary.writeTime/(maxTimestamp - minTimestamp));
  printf("%% Time Seeking:              %3.5f\n", 100. * summary.seekTime/(maxTimestamp - minTimestamp));
  printf("%% Open/Close overhead:       %3.5f\n", 100. * summary.overhead/(maxTimestamp - minTimestamp));
  printf("Frequency of Reads:          %3.5f/sec\n", summary.readCount/(summary.lifetime));
  printf("Frequency of Writes:         %3.5f/sec\n", summary.writeCount/(summary.lifetime));
  printf("Frequency of Seeks:          %3.5f/sec\n", summary.seekCount/(summary.lifetime));
  printf("Distribution of Requests:    %3.2f%% read  %3.2f%% write %3.2f%% seek\n", 
   (float) 100. * summary.readCount/(summary.readCount+summary.writeCount+summary.seekCount),
   (float) 100. * summary.writeCount/(summary.readCount+summary.writeCount+summary.seekCount),
   (float) 100. * summary.seekCount/(summary.readCount+summary.writeCount+summary.seekCount));
  printf("Read throughput:              %3.2f bytes/sec\n",
	 (summary.readBytes/summary.readTime));

  printf("Write throughput:             %3.2f bytes/sec\n",
	 (summary.writeBytes/summary.writeTime));

  printf("Lifetime Histogram:\n");
  lifetimeStats.printBuckets(cout);

}

void
buildStatRecord( StructureDescriptor *inSDp, StructureDescriptor *statSDp )
{
	static Attributes attr;
	static CString    fieldName;
	FieldDescriptor  *fieldP;

	/* 
	 * The first field of a statistics record is always a count of the 
	 * data records of this type in the file. 
	 */
	fieldP = new FieldDescriptor( "_Count Of Records_", attr, INTEGER, 0 );
	statSDp->insert( *fieldP );
	delete fieldP;

	/*
	 * Next, we iterate over the fields in the incoming descriptor and
	 * create corresponding fields in the statistics record to hold
	 * the Minimum and Maximum values.  For non-scalar fields, we also
	 * create a field in the statistics record to hold the Minimum and
	 * Maximum sizes for each of the dimensions.
	 */

	StructureDescriptorIterator iterator( *inSDp ); 
	FieldDescriptor inField = iterator.first();

	while( inField.notNull() ) {
	    fieldP = new FieldDescriptor( inField );	
	    fieldP->setDimension( 1 );
	    statSDp->insert( *fieldP );
	    delete fieldP;

	    if ( inField.getDimension() > 0 ) {
		fieldName = inField.getName() + " Dimension Sizes";
		fieldP = new FieldDescriptor( fieldName, attr, INTEGER, 1 );
	        statSDp->insert( *fieldP );
		delete fieldP;
	    }

	    inField = iterator.next();
	}
}

void
updateStatRecord( RecordDossier& inDossier, RecordDossier& statDossier )
{
	// These are static just so they do not have to be recreated each time
	static Value one( 1 );
	static int   fieldCount;
	static Value minValue;
	static Value maxValue;
	static Value newValue;

	FieldDescriptor *inFieldP; 
	Array           *inArrayP;
	int 		inDimension;
	int		inCellCount;
	const int	*inDimSizes;
	Array           *statArrayP;
	int		dimSize;

	int		a;		// index for cells in input array
	int		c;		// index for cells in stat array
	int		d;		// input for cells in dimension array
	int	        i;		// index for fields in input dossier
	int		s;		// index for fields in stat dossier

	/* 
	 * First, update our count of records seen for this type.
	 * Set flag if this is the first time we've seen a record of this type.
	 */
	Value *valueP = statDossier.getValueP( 0 );
	*valueP = *valueP + one;
	Boolean_ firstOne = ( *valueP == one );

	/* 
	 * Next, we cycle through each field in the input record and update
	 * our statistic Minimum and Maximum values if appropriate.  
	 */
	s = 1;

	fieldCount = inDossier.entryCount();
        for ( i = 0; i < fieldCount; i++ ) {

	    inFieldP = inDossier.getField( i );
	    statArrayP = statDossier.getArrayP( s++ );
	    inDimension = inFieldP->getDimension();

	    if ( inDimension > 0 ) {
		inArrayP = inDossier.getArrayP( i );
		inCellCount = inArrayP->getCellCount();
		inDimSizes = inArrayP->getDimSizes();
	    }

	    if ( firstOne ) {
		/*
		 * The first time we see a record of a given type, we set
		 * the dimension size for our statistics Min/Max field
		 * and initialize both the minimum and maximum values to
		 * the input field value.
		 */
		dimSize = 2;			// Always 2 for Min and Max 
		statArrayP->setDimSizes( &dimSize );

		if ( inDimension == 0 ) {	
		    /* 
		     * For scalars, set Minimum and Maximum values
		     */
		    statArrayP->setTheCellValue( 0, inDossier.getValue( i ) );
		    statArrayP->setTheCellValue( 1, inDossier.getValue( i ) ); 
		} else {			// An array value
		    /*
		     * For arrays, find Minimum and Maximum values over
		     * all elements of the array.
		     */
		    if ( inCellCount != 0 ) {
		        inArrayP->getTheCellValue( minValue, 0 );
		        maxValue = minValue;
		    } else {
        		switch ( inArrayP->getType() ) {
            		    case CHARACTER:
				minValue = (char) MAXCHAR;
				maxValue = (char) MINCHAR;
                		break;
            		    case INTEGER:
				minValue = (int) MAXINT;
				maxValue = (int) MININT;
                		break;
            	 	    case FLOAT:
				minValue = (float) MAXFLOAT;
				maxValue = (float) MINFLOAT;
                		break;
            		    case DOUBLE:
				minValue = (double) MAXDOUBLE;
				maxValue = (double) MINDOUBLE;
                		break;
            		    case UNDEFINED:
            		    default:
				minValue = Value::NOVALUE;
				maxValue = Value::NOVALUE;
				break;
			}
		    }

	            for ( a = 1; a < inCellCount; a++ ) {
		 	inArrayP->getTheCellValue( newValue, a );
		 	if ( newValue < minValue ) {
		     	    minValue = newValue;
			} 
			if ( newValue > maxValue ) {
			    maxValue = newValue;
			}
		    }
		    statArrayP->setTheCellValue( 0, minValue );
		    statArrayP->setTheCellValue( 1, maxValue ); 

		    /*
		     * Now, we also need to set the dimensions for our
		     * DimSizes statistics field and initialize the Minimum
		     * and Maximum values there.  (We never let dimSize = 2
		     * so we can distinguish these "special fields"
		     * for array dimensions from the normal statistics
		     * min/max fields.)
		     */
		    statArrayP = statDossier.getArrayP( s++ );
		    dimSize = (inDimension * 2);
		    if ( dimSize == 2 ) {
			dimSize = 3;
		    }
		    statArrayP->setDimSizes( &dimSize );

		    c = 0;
		    for ( d = 0; d < inDimension; d++ ) {
			statArrayP->setTheCellValue( c++, inDimSizes[d] ) ;
			statArrayP->setTheCellValue( c++, inDimSizes[d] ) ;
		    }
		}

	    } else {			

		if ( inDimension == 0 ) {
		    /*
		     * For scalars, adjust Minimum and Maximum values as
		     * needed.
		     */
		    newValue = inDossier.getValue( i );
		    if ( newValue < statArrayP->getTheCellValue( 0 ) ) {
			statArrayP->setTheCellValue( 0, newValue );
		    }
		    if ( newValue > statArrayP->getTheCellValue( 1 ) ) {
			statArrayP->setTheCellValue( 1, newValue );
		    }

		} else {
		    /*
		     * For arrays, compare each cell value to the
		     * statistic record Minimum and Maximum. 
		     */
		    statArrayP->getTheCellValue( minValue, 0 );
		    statArrayP->getTheCellValue( maxValue, 1 );

	            for ( a = 0; a < inCellCount; a++ ) {
		 	inArrayP->getTheCellValue( newValue, a );
		 	if ( newValue < minValue ) {
		     	    minValue = newValue;
			} 
			if ( newValue > maxValue ) {
			    maxValue = newValue;
			}
		    }
		    statArrayP->setTheCellValue( 0, minValue );
		    statArrayP->setTheCellValue( 1, maxValue ); 

		    /* 
		     * Then, update the DimSizes Minimum and Maximum Values.
		     */
		    statArrayP = statDossier.getArrayP( s++ );

		    c = 0;
		    for ( d = 0; d < inDimension; d++ ) {
			if ( inDimSizes[d] < 
				      (int)statArrayP->getTheCellValue( c ) ) {
			    statArrayP->setTheCellValue( c, inDimSizes[d] ) ;
			}
			c++;
			if ( inDimSizes[d] > 
				      (int)statArrayP->getTheCellValue( c ) ) {
			    statArrayP->setTheCellValue( c, inDimSizes[d] ) ;
			}
			c++;
		    }
		}
	    }
	}
}

void
printStatRecords( RecordDictionary& RDict )
{
	RecordDossier *dossier;
	Array	      *arrayP;
	Value	       min;
	Value	       max;
	int	       count;
	int	       fieldCount;
	const int     *dimSizes;
	int	       dimensions;
	int	       i, j, k;

	printf( "\nInformation by Record Type");
	printf( "\n--------------------------");

	RecordDictionaryPIterator iterator( RDict );
	dossier = iterator.first();

	while ( dossier != NULL ) {

	    if ( dossier->getFieldID( "_Count Of Records_" ) == 0 ) {
		/* 
		 * This is a statistics Record!
		 */
		count = dossier->getValue( 0 );
		printf( "\n%s        (%d Data Records)\n",
			 (const char *)dossier->getName(), count );

		fieldCount = dossier->entryCount();
		for ( i = 1; i < fieldCount; i++ ) {
		    /*
		     * We know that all the fields except the first (which
		     * we already processed) are vectors.  
		     */
		    arrayP = dossier->getArrayP( i );
		    dimSizes = arrayP->getDimSizes();

		    if ( *dimSizes == 2 ) {
			/*
			 * This is a Minimum-Maximum field descriptor
			 */
		        printf( "    Field: %s", 
				 (const char*)dossier->getField(i)->getName() );
		        if ( count > 0 ) {
		            min = arrayP->getTheCellValue( 0 );
		            max = arrayP->getTheCellValue( 1 );
			    if ( min > max ) {
				printf( "\n      No Values seen." );
			    } else {
		                printf( "\n      Minimum: " );
		                printValue( min );
		                printf( "\n      Maximum: " );
		                printValue( max );
			    }
		        }
		        printf( "\n" );

		    } else {
			/* 
			 * This is an Array dimension sizes field descriptor
			 */
		        if ( count > 0 ) {
			    dimensions = (*dimSizes)/2;
			    k = 0;
			    for ( j = 0; j < dimensions; j++ ) {
			        printf("      Dimension %d:", j );
			        printf("\tMinimum Size %d  \tMaximum Size %d\n",
				          (int)arrayP->getTheCellValue( k++ ),
				          (int)arrayP->getTheCellValue( k++ ) );

			    }		// for ( j = ... )
		        }		// if ( count > 0 )
		    }			// if ( *dimSizes == 2 )
		} 			// for ( i = ... )
	    } 				// if ( dossier->getFieldID

	    dossier = iterator.next();
	}

}

void
printValue( const Value& value ) 
{
        switch( value.getTraits().getType() ) {
          case CHARACTER:
		printf( "%c", (char)value );
                break;
          case INTEGER:
                printf( "%d", (int)value );
                break;
          case FLOAT:
                printf( "%.7g", (float)value );
                break;
          case DOUBLE:
                printf( "%.16g", (double)value );
                break;
          case UNDEFINED:
          default:
                break;
        }
}

#ifndef __GNUG__
#include "AwesimeFiles/SampleStatistic.C"
#include "AwesimeFiles/SampleHistogram.C"
#endif
