/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Contributing Author: Roger J. Noe (noe@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * MergeTwoFiles.C: Program to merge two binary SDDF files based on
 *		    specified scalar field value.
 *
 */

#include <stdlib.h>
#include <stream.h>
#include <string.h>
#include <sys/limits.h>

#include "InitializeStatic.C"

#include "PipeReader.h"
#include "BinaryPipeWriter.h"
#include "InputFileStreamPipe.h"
#include "OutputFileStreamPipe.h"

#include "Attributes.h"
#include "PacketHeader.h"
#include "RecordDictionary.h"
#include "StructureDescriptor.h"

int processPackets( PipeReader *InPipe, PipeWriter *OutPipe );

/* 
 * Global record dictionary
 */
RecordDictionary RD;

main( int argc, char **argv )
{
	enum Status { INVALID, VALID };

	char BUF[512];		/* buffer for input from stdin */

 	/****************************************************************
         *    Parse command-line options, if any
         ****************************************************************/
	char	*keyField = (char *) 0;		/* key field name	*/
	char	*outputFile = (char *) 0;	/* output file name	*/

	Boolean_ arg_error = FALSE_;
	argv++;		// skip over program name
	argc--;		

	while ( ( argc > 0 ) && ( *argv[0] == '-' ) ) {
            if ( strcmp(*argv, "-k") == 0 ) {
                if ( --argc == 0 ) {
	            cerr << "-k needs an argument" << NL;
		    arg_error = TRUE_;
		    break;
                } else {
                    argv++;
                    keyField = *argv;
		}

            } else if ( strcmp(*argv, "-o") == 0 ) {
                if ( --argc == 0 ) {
	            cerr << "-o needs an argument" << NL;
		    arg_error = TRUE_;
		    break;
                } else {
                    argv++;
                    outputFile = *argv;
		}

            } else {
                cerr << "Invalid argument: " << *argv << NL;
		arg_error = TRUE_;
		break;
	    }

	    argc--;
            argv++;
        }

	if ( arg_error ) {
	    cerr << "Usage: MergeTwoFiles [-k key-field] [-o output-file] ";
	    cerr << "[input-file [input-file]]\n";
  	    exit(1);
	}

 	/****************************************************************
         *    Get the name of the input files, and open the files
         ****************************************************************/
	Status 		     inputFileStatus;
	InputFileStreamPipe  *In1, *In2;
	PipeReader     	     *In1Pipe, *In2Pipe;

	do {
	    if ( argc > 0 ) {
		In1 = new InputFileStreamPipe( *argv );
		++argv, --argc;
	    } else {
		cerr << "Please enter name of the first input SDDF file: ";
		cin >> BUF;

		In1 = new InputFileStreamPipe( BUF );
	    }

	    if ( In1->successfulOpen() ) {
		inputFileStatus = VALID;
		In1Pipe = In1->createPipeReader();
                if ( In1Pipe == NULL ) {
                    cerr << "ERROR: Couldn't attach pipe to input file\n";
                    inputFileStatus = INVALID;
                    delete In1;
                }
	    } else {
		cerr << "ERROR: Unable to open file for input\n";
		inputFileStatus = INVALID;
		delete In1;
	    }
	} while ( inputFileStatus == INVALID );

	do {
	    if ( argc > 0 ) {
		In2 = new InputFileStreamPipe( *argv );
		++argv, --argc;
	    } else {
		cerr << "Please enter name of the second input SDDF file: ";
		cin >> BUF;

		In2 = new InputFileStreamPipe( BUF );
	    }

	    if ( In2->successfulOpen() ) {
		inputFileStatus = VALID;
		In2Pipe = In2->createPipeReader();
                if ( In2Pipe == NULL ) {
                    cerr << "ERROR: Couldn't attach pipe to input file\n";
                    inputFileStatus = INVALID;
                    delete In2;
                }
	    } else {
		cerr << "ERROR: Unable to open file for input\n";
		inputFileStatus = INVALID;
		delete In2;
	    }
	} while ( inputFileStatus == INVALID );

 	/****************************************************************
         *    Get the name of the output file, and open the file
         ****************************************************************/
	Status 		     outputFileStatus;
	OutputFileStreamPipe *Out;
	PipeWriter           *OutPipe;

	do {
	    if ( outputFile != (char *) 0 ) {
		Out = new OutputFileStreamPipe( outputFile );
	    } else {
		cerr << "Please enter name of the output SDDF file: ";
		cin >> BUF;

		Out = new OutputFileStreamPipe( BUF );
	    }

	    if ( Out->successfulOpen() ) {
		outputFileStatus = VALID;
		OutPipe = new BinaryPipeWriter( Out );
	    } else {
		outputFileStatus = INVALID;
		delete Out;
		outputFile = (char *) 0;
	    }
	} while ( outputFileStatus == INVALID );

 	/****************************************************************
         *    Get the name of the key field for the merge
         ****************************************************************/

	if ( keyField == (char *) 0 ) {
	    cerr << "Please enter name of the merge key field: ";
	    cin >> BUF;
	    keyField = BUF;
	}

 	/****************************************************************
	 * BUF may not be reused for the remainder of the program
         ****************************************************************/

 	/****************************************************************
         *    Process the packets in the files.  
	 *    For the data packets, look for the one with the 
	 *      lowest key field value and write to the output pipe.
	 *    A tag =  0 means no unprocessed data record,
	 *	    >  0 means valid data record 
	 *          = -1 means file is empty
         ****************************************************************/
	
	int tag1 = 0;
	int tag2 = 0;
	RecordDossier	Rdoss1( "Dossier1" );
	RecordDossier	Rdoss2( "Dossier2" );
	Value		*scalar1P, *scalar2P;
	int		whichPut;

	do {
	    while ( tag1 == 0 ) {
		tag1 = processPackets( In1Pipe, OutPipe );
		if ( tag1 > 0 ) {
		    Rdoss1 = RD.fetch( tag1 );
		    scalar1P = Rdoss1.getValueP( keyField );
		    if ( scalar1P == (Value *) NULL ) {
			OutPipe->putData( Rdoss1 );
			tag1 = 0;
		    }
		}
	    }

	    while ( tag2 == 0 ) {
		tag2 = processPackets( In2Pipe, OutPipe );
		if ( tag2 > 0 ) {
		    Rdoss2 = RD.fetch( tag2 );
		    scalar2P = Rdoss2.getValueP( keyField );
		    if ( scalar2P == (Value *) NULL ) {
			OutPipe->putData( Rdoss2 );
			tag2 = 0;
		    }
		}
	    }

	    whichPut = 0;

	    if (( tag1 > 0 ) && ( tag2 < 0 ))
		whichPut = 1;
	    else if (( tag1 < 0 ) && ( tag2 > 0 ))
		whichPut = 2;
	    else if (( tag1 > 0 ) && ( tag2 > 0 )) {
		if ( *scalar1P <= *scalar2P )
		    whichPut = 1;
		else
		    whichPut = 2;
	    }

	    if ( whichPut == 1 ) {
		OutPipe->putData( Rdoss1 );
		tag1 = 0;
	    } else if ( whichPut == 2 ) {
		OutPipe->putData( Rdoss2 );
		tag2 = 0;
	    }

	} while ( ( tag1 >= 0 ) || ( tag2 >= 0 ) );

	delete In1;
	delete In2;
	delete Out;
}


/*****************************************************************************
 *
 *	Process packets from the input pipe.
 *      If packet is of type Attributes or Command, write it to the output pipe
 *	   and process next packet.
 *	If packet of type Descriptor, insert it into the global Record
 *	   Dictionary.  If it isn't a duplicate, write it to the output pipe.
 *	   Process next packet
 *	If packet of type Data, then return it's tag.
 *      If the pipe is empty, return -1.
 *
 *****************************************************************************/

int
processPackets( PipeReader *InPipe, PipeWriter *OutPipe )
{
	PacketHeader PH;
	Attributes *Ap;
	StructureDescriptor *SDp;

	do { 

	    PH = InPipe->getPacketHeader();

	    switch( PH.type ) {

	      case PKT_ATTRIBUTE:
		  Ap = new Attributes();
		  InPipe->getAttributes( *Ap );
		  OutPipe->putAttributes( *Ap );
		  delete Ap;
		  break;

	      case PKT_COMMAND:
		  OutPipe->putCommand( PH.tag );
		  break;

	      case PKT_DESCRIPTOR:
		  SDp = new StructureDescriptor();
		  InPipe->getDescriptor( *SDp );
		  if ( RD.insert( PH.tag, *SDp ) ) {
		      OutPipe->putDescriptor( *SDp, PH.tag );
		  }
		  delete SDp;
		  break;

	      case PKT_DATA:
		  InPipe->getData( RD.fetch( PH.tag ) );
		  break;
	    }

	} while( PH.type != PIPE_EMPTY  && PH.type != PKT_DATA ) ;

	if ( PH.type == PIPE_EMPTY ) {
	    return -1;
	} else {
	    return PH.tag;
	}

}

