/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Dave Kohr (drk@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * SummaryInfo.h Information about fields in summary file records.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/SDDFlibrary/SamplePrograms/SDDFStatistics/RCS/SummaryInfo.h,v 1.1 1994/04/06 22:26:27 aydt Exp $
 */

#ifndef SummaryInfo_h
#define SummaryInfo_h


#include "AsciiPipeWriter.h"
#include "CString.h"
#include "FieldDescriptor.h"
#include "OutputFileStreamPipe.h"
#include "RecordDictionary.h"
#include "RecordDossier.h"
#include "Value.h"


#include "SDDFStats_Misc.h"


class TagHistogram;


/**********************************************************************
The class PerFileSummary contains all the data in a summary of an input
SDDF file.  It has a complex usage convention which I have to document
someday.
**********************************************************************/

class PerFileSummary {
	int curBin, curRecordNum;	// Histogram stuff.
	double recordsPerBin;
	TagHistogram *histogram;
	void addTag(int tag);
	OutputFileStreamPipe *outFile;	// Output file stuff.
	AsciiPipeWriter *outWriter;
					// Input record descriptions.
	RecordDictionary inRecDict;
					// Output summary record data.
	RecordDictionary outRecDict;
	RecordDossier *perFileRecord;	// Per-file summary data.
public:
					// Info about the output record.
	static const char *structureName();
	static const char *structureDescription();
	enum Fields {
		NAME,
		SIZE,
		DESCRIPTOR_COUNT,
		DATA_COUNT,
		ATTRIBUTE_COUNT,
		COMMAND_COUNT,
		DUPLICATE_DESCRIPTOR_COUNT,
		DOMINANT_TYPE
	};
	struct FieldDesc {
		const char *description;
		const char *name;
		MachineDataType dataType;
		int dimensions;
	};
	static const FieldDesc fieldDesc[];
				// Handy C++ trick: This enum value may
				// be used as an integer constant.
	enum {
		numFields = 8
	};
	static int tag();

	const CString inFileName;	// Per-file statistics.
	int size;
	int descriptorCount, dataCount, attributeCount, commandCount,
	    dupDescCount;
	int num_bins;
	int *dominantType;

					// Look but don't touch....
	const RecordDictionary &getInRecDict(), &getOutRecDict();

					// Read packets on various passes.
	void readAttributeFirstPass(const Attributes &attr);
	void readCommandFirstPass();
	void readDataFirstPass(RecordDossier &inDossier, int inputTag);
	void readDescriptorFirstPass(const StructureDescriptor &inDesc,
				     int inputTag);
	void readAttributeSecondPass(const Attributes &attr);
	void readCommandSecondPass();
	void readDataSecondPass(RecordDossier &inDossier, int inputTag);
	void readDescriptorSecondPass(const StructureDescriptor &inDesc,
				      int inputTag);

	void endFirstPass();		// Demarcate phases of processing.
	void endSecondPass();
					// Doesn't use output file name
					// right now....
	void writeSummaryFile(const char *);

					// Will need to be able to change
					// num_bins dynamically someday.
					// Should not need output file name
					// before writeSummaryFile() is
					// called....
	PerFileSummary(const char *_inFileName, const char *_outFileName,
		       int _num_bins = 10);
	~PerFileSummary();
};

/*
PerFileSummary accessor methods.
*/

inline const RecordDictionary &
PerFileSummary::getInRecDict()
{
	return inRecDict;
}

inline const RecordDictionary &
PerFileSummary::getOutRecDict()
{
	return outRecDict;
}

/*
PerFileSummary SDDF packet-handling methods.  The ones which are inlined
are trivial and can potentially be called quite often.
*/

inline void
PerFileSummary::readAttributeFirstPass(const Attributes &)
{
	attributeCount++;
}

inline void
PerFileSummary::readCommandFirstPass()
{
	commandCount++;
}

inline void
PerFileSummary::readAttributeSecondPass(const Attributes &)
{
}

inline void
PerFileSummary::readCommandSecondPass()
{
}

inline void
PerFileSummary::readDescriptorSecondPass(const StructureDescriptor &, int)
{
}


/**********************************************************************
The class PerRecordSummary exists solely to encapsulate all the information
about the format, initialization, and computation of the output per-record
summary records, and make information about the format (only) available to
other modules which need it.  As only PerFileSummary objects should be
initializing and computing summary records, only this class has access to
all of PerRecordSummary's member functions.  Because some of this
information has the same names as in the PerFieldSummary, (e.g.,
FixedFields, VarFields, FieldDesc), encapsulating it in a class also helps
prevent name-space collisions.  This class does not contain any data on a
per-object basis (all data and functions are static), and it is not
intended that objects of this class be created (hence the private
constructor).
**********************************************************************/

class PerRecordSummary {
	friend class PerFileSummary;

	PerRecordSummary() { fatalError("PerRecordSummary::"
		"PerRecordSummary() called; should never happen!"); }
public:
	enum FixedFields {
		RECORD_COUNT,
		RECORD_TAG
	};
	enum {
		firstFixedField = 0,
		lastFixedField = 1,
		numFixedFields = lastFixedField - firstFixedField + 1
	};
	enum VarFields {
		MIN = RECORD_TAG + 1,
		MAX,
		VALUES_COUNT,
		MEAN,
		STDDEV,
				// Plain UNDERFLOW and OVERFLOW would be
				// better, but conflict with something in
				// <math.h>.
		UNDER_FLOW,
		OVER_FLOW,
		BIN_COUNT,
		HISTOGRAM,
		MIN_DIMENSION,
		MAX_DIMENSION,
		MEAN_DIMENSION,
		STDDEV_DIMENSION
	};
	enum {
		firstVarField = MIN,
		lastVarField = STDDEV_DIMENSION,
		numVarFields = lastVarField - firstVarField + 1
	};
	struct FieldDesc {
		const char *description;
		const char *name;
		MachineDataType dataType;
		int dimensions;
	};
	static const FieldDesc fieldDesc[];
	static CString fieldName(const CString &inFieldName,
				 VarFields field);
	static int tag(int inputTag);
	static int numHistogramBins();
private:
	static void
	defineSummaryRecord(const StructureDescriptor &inDesc,
			    int inputTag, RecordDictionary &outRecDict,
			    PipeWriter &outWriter);
	static Value initialValue(FixedFields outputField, int inputTag);
	static Value initialValue(VarFields outputField,
				  const FieldDescriptor &inputField);
	static void readDataFirstPass(const RecordDossier &inDossier,
				      RecordDictionary &outRecDict);
	static void
	updateFieldFirstPass(const Value &inValue, const Array *arrayP,
			     RecordDossier &outDossier, int outField,
			     VarFields varField, Boolean_ isFirstVal,
			     Boolean_ isFirstRec);
	static void endFirstPass(RecordDictionary &outRecDict);
	static void readDataSecondPass(const RecordDossier &inDossier,
				       RecordDictionary &outRecDict);
	static void endSecondPass(RecordDictionary &outRecDict);
};


// These have to appear after the full PerRecordSummary class declaration.

inline void
PerFileSummary::readDataFirstPass(RecordDossier &inDossier, int)
{
	PerRecordSummary::readDataFirstPass(inDossier, outRecDict);
	dataCount++;
}

inline void
PerFileSummary::readDataSecondPass(RecordDossier &inDossier, int inputTag)
{
	PerRecordSummary::readDataSecondPass(inDossier, outRecDict);
	addTag(inputTag);
}


#endif	// #ifndef SummaryInfo_h
