/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * SDDFconverter.C Program to convert a SDDF data file in one format to
 *		    another, usually different, format.
 *
 */
#include <stream.h>

#include "InitializeStatic.C"

#include "PipeReader.h"
#include "AsciiPipeWriter.h"
#include "BinaryPipeWriter.h"
#include "ConversionPipeWriter.h"
#include "InputFileStreamPipe.h"
#include "OutputFileStreamPipe.h"

#include "Attributes.h"
#include "PacketHeader.h"
#include "RecordDictionary.h"
#include "StructureDescriptor.h"

main()
{

        enum FileType     { UNDEFINED, ASCII, BINARY, CONVERT };
	enum Status       { INVALID, VALID };
	enum UserResponse { NOT_REQUESTED, REQUESTED };

	char BUF[512];		/* buffer for input from stdin */

 	/****************************************************************
         *    Get the name of the input file, and open the file
         ****************************************************************/
	Status 		     inputFileStatus;
	InputFileStreamPipe  *In;
	PipeReader     	     *InPipe;

	do {
  	    cerr << "Please enter name of the input SSDF file: ";
	    cin >> BUF;

	    In = new InputFileStreamPipe( BUF );
	    if ( In->successfulOpen() ) {
		inputFileStatus = VALID;
		InPipe = In->createPipeReader();
		if ( InPipe == NULL ) {
		    cerr << "ERROR: Couldn't attach pipe to input file\n";
		    inputFileStatus = INVALID;
		    delete In;
		}
	    } else {
		cerr << "ERROR: Problem with input file\n";
		inputFileStatus = INVALID;
		delete In;
	    }
	} while ( inputFileStatus == INVALID );

	cerr << "\n";
	switch( In->getInputType() ) {
	    case ASCII_SDDF:
		cerr << "File is in SDDF ASCII format";
		break;
	    case BINARY_SDDF:
		cerr << "File is in SDDF Binary format";
		break;
	    case CONVERT_BINARY_SDDF:
		cerr << "File is in SDDF Converted (reverse byte order) format";
		break;
	    default:
		cerr << "File is not in SDDF format";
		break;
	}
	cerr << "\n";

 	/****************************************************************
         *    Ask if output file is ascii, binary, or conversion format
         ****************************************************************/
	FileType	outputFileType = UNDEFINED;

        while ( outputFileType == UNDEFINED ) {
            cerr << "\nOutput in Ascii, Binary or Converted "
		 << "(reverse byte order) format [A, B, or C]: ";
            cin >> BUF;

            if ( BUF[0] == 'a' || BUF[0] == 'A' ) {
                outputFileType = ASCII;
            } else if ( BUF[0] == 'b' || BUF[0] == 'B' ) {
                outputFileType = BINARY;
            } else if ( BUF[0] == 'c' || BUF[0] == 'C' ) {
                outputFileType = CONVERT;
            } else {
                cerr << "ERROR: invalid format type.\n";
            }
        }

 	/****************************************************************
         *    Get the name of the output file, and open the file
         ****************************************************************/
	Status 		     outputFileStatus;
	OutputFileStreamPipe *Out;
	PipeWriter           *OutPipe;

	do {
  	    cerr << "Please enter name of the output SDDF file: ";
	    cin >> BUF;

	    Out = new OutputFileStreamPipe( BUF );
	    if ( Out->successfulOpen() ) {
		outputFileStatus = VALID;
		if ( outputFileType == ASCII ) { 
		    OutPipe = new AsciiPipeWriter( Out );
		} else if ( outputFileType == BINARY ) { 
		    OutPipe = new BinaryPipeWriter( Out );
		} else {
		    OutPipe = new ConversionPipeWriter( Out );
		}
	    } else {
		outputFileStatus = INVALID;
		delete Out;
	    }
	} while ( outputFileStatus == INVALID );

 	/****************************************************************
         *    Are diagnostic messages requested?
         ****************************************************************/
	UserResponse	diagOutput;

	cerr << "\nDo you want diagnostic messages printed [Y or N]: ";
	cin >> BUF;

	if ( BUF[0] == 'y' || BUF[0] == 'Y' ) {
	    diagOutput = REQUESTED;
	} else {
	    diagOutput = NOT_REQUESTED;
	}

 	/****************************************************************
         *    Process the packets in the files.  Read from the input 
	 *    pipe and write to the output pipe.
         ****************************************************************/
	Attributes *Ap;
	StructureDescriptor *SDp;
	RecordDictionary RD;

	PacketHeader PH = InPipe->getPacketHeader();

	while( PH.type != PIPE_EMPTY ) {

	    switch( PH.type ) {
	      case PKT_ATTRIBUTE:
		  Ap = new Attributes();
		  InPipe->getAttributes( *Ap );
		  OutPipe->putAttributes( *Ap );
		  if (diagOutput) {
		      cout << "\n*****\n" << *Ap;
		  }
		  delete Ap;
		  break;
	      case PKT_DESCRIPTOR:
		  SDp = new StructureDescriptor();
		  InPipe->getDescriptor( *SDp );
		  OutPipe->putDescriptor( *SDp, PH.tag );
		  RD.insert( PH.tag, *SDp );
		  if (diagOutput) {
		      cout << "\n*****\n" << *SDp;
		  }
		  delete SDp;
		  break;
	      case PKT_DATA:
		  {				
		      RecordDossier& dossier = RD.fetch( PH.tag );
		      InPipe->getData( dossier );
		      OutPipe->putData( dossier );
		      if (diagOutput) {
		          cout << "\n*****\n" << dossier;
		      }
		  }
		  break;
	      case PKT_COMMAND:
		  OutPipe->putCommand( PH.tag );
		  if (diagOutput) {
		      cout << "\n*****COMMAND " << PH.tag << "\n";
		  }
		  break;
	      }
	      PH = InPipe->getPacketHeader();
	}

	delete InPipe;
	delete In;
	delete OutPipe;
	delete Out;
}

