/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author:  Roger J. Noe (noe@cs.uiuc.edu)
 * Contributing Author:  Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004, and
 * by a grant from the Digital Equipment Corporation External Research
 * Program.
 *
 */

/*
 * SDDFmerge.c
 *	Driver program to merge an arbitrary number of SDDF files.  Uses
 *	sequential binary tree merge algorithm, invoking pairwise merging
 *	program for each pair of files.  Output is on standard output stream
 *	if no output file name specified.  Default merge key field name is
 *	"Seconds".  Default pairwise merging program is "MergeTwoFiles".
 */

#include <sys/types.h>
#include <sys/wait.h>
#include <errno.h>
#include <stdio.h>

#ifndef NULL
# define NULL	0
#endif

#ifndef BUFSIZ
# define BUFSIZ	1024
#endif

extern int	optind;
extern char	*optarg;
extern char	*malloc();
void	copyOut();

char	usage[] =
"Usage:  %s [-k key-field] [-m merge-program] [-o output-file] filename ...\n";

#define KEYFIELD	"Seconds"	/* Name of field used as merge key  */
#define MERGER		"MergeTwoFiles"	/* Name of the pairwise merger	    */

/* Exit status values - returned by main()				    */

#define EXIT_OK		0		/* Successful termination	    */
#define ERR_USAGE	1		/* Invocation error		    */
#define ERR_ALLOC	2		/* Memory allocation error	    */
#define ERR_EXEC	3		/* Cannot execute merge program	    */
#define ERR_FORK	4		/* Cannot fork child process	    */
#define ERR_OPEN	5		/* Cannot open output file	    */

/*
 * Structure for member of argument list.  If it's a temporary file, this
 * will be stored in the temp array member and name will point to it.
 * If not a temporary file, name will point elsewhere, e.g. into argv[].
 */

typedef struct {

	char	temp[ L_tmpnam ];	/* temporary file name		    */

	char	*name;			/* pointer to argument name	    */

} ARG;

char	*myName;			/* The name of this program	    */
char	*keyField = KEYFIELD;		/* The merging key field name	    */
char	*mergeProgram = MERGER;		/* The pair merge program used	    */
char	*outputFile = (char *) NULL;	/* The output file name		    */


main( argc, argv )

int	argc;
char	**argv;
{
	ARG	*inputArgs,	*outputArgs;
	ARG	*inputPointer,	*outputPointer;
	int	numInputs,	numOutputs;
	int	inputIndex,	outputIndex;
	int	option;
	int	pid;

	/* Record program name and process options.			    */

	myName = *argv;

	while (( option = getopt( argc, argv, "k:m:o:" )) != EOF ) {

		switch ( option ) {

		case 'k':
			keyField = optarg;
			break;

		case 'm':
			mergeProgram = optarg;
			break;

		case 'o':
			outputFile = optarg;
			break;

		case '?':
		default:
			fprintf( stderr, usage, myName );
			return ERR_USAGE;
		}
	}

	/* Skip past program name and options.				    */

	argc -= optind;
	argv += optind;

	/* Check for incorrect invocation.				    */

	if ( argc <= 0 ) {

		fprintf( stderr, usage, myName );

		return ERR_USAGE;
	}

	/* Create the first list of arguments from the remaining	    */
	/* original argv elements.					    */

	numInputs = argc;

	inputArgs = (ARG *) malloc( numInputs * sizeof( ARG ) );

	if ( inputArgs == (ARG *) NULL ) {

		fprintf( stderr, "%s: Cannot allocate storage\n", myName );

		return ERR_ALLOC;
	}

	for ( inputIndex = 0, inputPointer = & inputArgs[ 0 ];
	      inputIndex < numInputs;
	      inputIndex++, inputPointer++ ) {

		inputPointer->name = *argv++;
	}

	/* The following loop repeats for each level of the binary tree,    */
	/* the number of arguments halving at every iteration.		    */

	while ( numInputs > 1 ) {

		/* Create the list of output files.			    */

		numOutputs = ( numInputs + 1 ) >> 1;

		outputArgs = (ARG *) malloc( numOutputs * sizeof( ARG ) );

		if ( outputArgs == (ARG *) NULL ) {

			fprintf( stderr, "%s: Cannot allocate storage\n",
					myName );

			return ERR_ALLOC;
		}

		/* Now take successive pairs of input files and create	    */
		/* a temporary output file to hold their merged contents.   */

		for ( inputIndex = 1, inputPointer = &inputArgs[ 1 ],
			outputIndex = 0, outputPointer = &outputArgs[ 0 ];

		      inputIndex < numInputs;

		      inputIndex += 2, inputPointer += 2,
			outputIndex++, outputPointer++ ) {


			/* Create the temporary output file name.	    */

			tmpnam( outputPointer->temp );

			outputPointer->name = outputPointer->temp;

			/* Fork off a child which will execute the	    */
			/* pairwise merging program.			    */

			if (( pid = fork() ) == 0 ) {

				/* Child process:			    */

				execlp( mergeProgram, mergeProgram,
					"-k", keyField,
					"-o", outputPointer->name,
					(inputPointer - 1)->name,
					inputPointer->name,
					(char *) NULL );

				fprintf( stderr, "%s: Cannot exec %s\n",
						myName, mergeProgram );

				return ERR_EXEC;
			}

			if ( pid < 0 ) {

				fprintf( stderr, "%s: Cannot fork\n",
						myName );

				return ERR_FORK;
			}

			/* Block until child process has exited.	    */

			while (( wait( (int *) NULL ) >= 0 ) ||
			       ( errno != ECHILD ))

				/* REPEAT */;

			/* If the merge program could be counted on to	    */
			/* return exit status reliably, the status could    */
			/* be saved and examined, improving error	    */
			/* checking and handling.			    */
		}

		/* If there were an odd number of inputs at this level,	    */
		/* then the unmatched input becomes an output.		    */

		if ( inputIndex == numInputs ) {

			inputPointer = &inputArgs[ numInputs - 1 ];

			/* If the odd input was a temporary file, copy	    */
			/* its name to the output argument list.	    */

			if ( inputPointer->name == inputPointer->temp ) {

				strcpy( outputPointer->temp,
					inputPointer->temp );

				outputPointer->name = outputPointer->temp;

				inputPointer->name = (char *) NULL;

			} else {

				outputPointer->name = inputPointer->name;
			}
		}

		/* Remove all temporary input files.			    */

		for ( inputIndex = 0, inputPointer = &inputArgs[ 0 ];
		      inputIndex < numInputs;
		      inputIndex++, inputPointer++ ) {

			if (( inputPointer->name == inputPointer->temp ) &&
			    ( unlink( inputPointer->name ) < 0 )) {

				fprintf( stderr, "%s: Cannot unlink %s\n",
						myName, inputPointer->name );

				/* Not a fatal error.			    */
			}
		}

		/* Free the allocated inputs list.			    */

		free( (char *) inputArgs );

		/* Outputs list becomes inputs list.			    */

		numInputs = numOutputs;

		inputArgs = outputArgs;

		/* Repeat loop for next level of tree merge.		    */
	}

	/* All of the original input files have now been merged into	    */
	/* the single file in the input list.  Copy it to the output.	    */

	if ( outputFile != (char *) NULL &&
	     freopen( outputFile, "w", stdout ) != stdout ) {

		fprintf( stderr,
			 "%s: Cannot open %s for output\n",
			 myName, outputFile );

		return ERR_OPEN;
	}

	inputPointer = &inputArgs[ 0 ];

	copyOut( inputPointer->name );

	/* If a temporary file is left, remove it.			    */

	if (( inputPointer->name == inputPointer->temp ) &&
	    ( unlink( inputPointer->name ) < 0 )) {

		fprintf( stderr, "%s: Cannot unlink %s\n",
				myName, inputPointer->name );

		/* Not a fatal error.					    */
	}

	/* Successful termination.					    */

	return EXIT_OK;
}


/*
 * copyOut:
 *	Open the file specified by argument and copy it to standard output.
 */

void copyOut( fileName )

char	*fileName;
{
	char	buffer[ BUFSIZ ];
	size_t	numBytes;

	if ( freopen( fileName, "r", stdin ) != stdin ) {

		fprintf( stderr, "%s: Cannot open %s for input\n",
				myName, fileName );

		return;
	}

	while (( numBytes = fread( buffer, sizeof(char), BUFSIZ, stdin )) > 0 )

		fwrite( buffer, sizeof(char), numBytes, stdout );

	return;
}
