/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *      Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * Sample.C: Program to write file in ascii, binary, or byte-swapped 
 * 	     binary SDDF format.
 *
 */
#include <stream.h>

#include "InitializeStatic.C"

#include "AsciiPipeWriter.h"
#include "BinaryPipeWriter.h"
#include "ConversionPipeWriter.h"
#include "OutputFileStreamPipe.h"

#include "Attributes.h"
#include "RecordDossier.h"
#include "StructureDescriptor.h"

main()
{
        char BUF[512];          /* buffer to keyboard input  */

        /****************************************************************
         *    First, ask if the output file should be ascii or binary
         *    format.
         ****************************************************************/
        enum FileType   { UNDEFINED, ASCII, BINARY, CONVERT };
        FileType        fileType = UNDEFINED;

        while ( fileType == UNDEFINED ) {
	    cout << "\nAscii, Binary, or Converted (reverse byte order) "
		 << "format [A, B, or C]: ";

            cin >> BUF;
            if ( BUF[0] == 'a' || BUF[0] == 'A' ) {
                fileType = ASCII;
            } else if ( BUF[0] == 'b' || BUF[0] == 'B' ) {
                fileType = BINARY;
            } else if ( BUF[0] == 'c' || BUF[0] == 'C' ) {
                fileType = CONVERT;
            } else {
                cerr << "ERROR: invalid format type.\n";
            }
        }

        /****************************************************************
         *    Next, ask for name of output file.  If error in opening,
         *    continue asking until something valid is entered.
         ****************************************************************/
        enum Status { INVALID, VALID };
        Status                  fileStatus;     /* status of pipe open        */
        OutputFileStreamPipe    *OutFile;       /* pipe that writes output to 
                                                 * a disk file                */
        PipeWriter              *PipeWriter;    /* write SDDF to pipe         */
        
        do {
            cout << "Please enter name of data file for output: ";
            cin >> BUF;

            /* 
             * Create an output file stream pipe instance connected to
             * the named file on disk.  If file open is successful, 
             * connect a SDDF writer of the chosen type to the pipe.
             */
            OutFile = new OutputFileStreamPipe( BUF );
            if ( OutFile->successfulOpen() ) {
                fileStatus = VALID;
                if ( fileType == ASCII ) {
                    PipeWriter = new AsciiPipeWriter( OutFile );
                } else if ( fileType == BINARY ) {
                    PipeWriter = new BinaryPipeWriter( OutFile );
                } else {
                    PipeWriter = new ConversionPipeWriter( OutFile );
                }
            } else {
                cerr << "ERROR: unable to open specified file.\n";
                fileStatus = INVALID;
                delete OutFile;
            }
        } while ( fileStatus == INVALID );

        /****************************************************************
         *    Here we'll start writing the SDDF packets.
         ****************************************************************/

        Attributes              attributes;
        StructureDescriptor     *structureP;
        FieldDescriptor         *fieldP;
        int                     tag;
        RecordDossier           *messageRecord;
        RecordDossier           *contextSwitchRecord;
        RecordDossier           *procCallRecord;

        /****************************************************************
         * First, a stream attribute packet
         ****************************************************************/
        attributes.clearEntries();      /* don't really need this here since 
                                         * attributes hasn't been used before
                                         * so it doesn't have any entries yet */
        attributes.insert( "creation date", "Nov. 1, 1990" );
        attributes.insert( "machine", "Intel iPSC/2" );
        PipeWriter->putAttributes( attributes );

        /****************************************************************
         * Now, some record descriptor packets to describe the types of 
         * data records we expect to see later.
         ****************************************************************/
        tag = 1;                        /* we need unique tags for each type  */

        /* 
         * First we build up a "message send" structure descriptor.  We 
         * set up attributes to associate with this structure descriptor 
         * and then create an instance of it.
         */
        attributes.clearEntries();
        attributes.insert( "event", "message sent to one or more processors" );
        structureP = new StructureDescriptor( "message send", attributes );

            /*
             * Field descriptors for this structure descriptor.  We set up
             * each field descriptor with it's own (optional) attributes, 
             * name, data type, and dimension information and then insert 
             * the completed field template into the  structure descriptor 
             * template.  Delete each field descriptor instance when we are
             * finished with it.
             */
            attributes.clearEntries();  
            attributes.insert( "From", "Processor Sending Message" );
            fieldP = new FieldDescriptor( "fromPE", attributes, INTEGER, 0 );
            structureP->insert( *fieldP );
            delete fieldP;

            attributes.clearEntries();  
            attributes.insert( "To", "Processor(s) Receiving Message" );
            fieldP = new FieldDescriptor( "toPE", attributes, INTEGER, 1 );
            structureP->insert( *fieldP );
            delete fieldP;

            attributes.clearEntries();  
            attributes.insert( "Size", "Message length in bytes" );
            fieldP = new FieldDescriptor( "msglength", attributes, INTEGER, 0 );
            structureP->insert( *fieldP );
            delete fieldP;

            attributes.clearEntries();  
            attributes.insert( "Type", "Character denoting message type" );
            fieldP = new FieldDescriptor( "msgType", attributes, CHARACTER, 0 );
            structureP->insert( *fieldP );
            delete fieldP;

            attributes.clearEntries();  
            attributes.insert( "Time", "Timestamp" );
            attributes.insert( "Units", "msec" );
            fieldP = new FieldDescriptor( "time stamp", attributes, DOUBLE, 0 );
            structureP->insert( *fieldP );
            delete fieldP;

        /*
         * Now that we have built up a complete structure descriptor template
         * for the "message send", we use it to write the record descriptor 
         * packet.  For later use, we also create an instance of a 
         * RecordDossier which  contains the information in the structure 
         * descriptor template, the tag we associated with the record 
         * descriptor packet we just wrote,  and which will have space to 
         * hold data for this type of record.
         */
        PipeWriter->putDescriptor( *structureP, tag );

        messageRecord = new RecordDossier( tag, *structureP );
        delete structureP;

        /*
         * Now we build up a "context switch" descriptor.
         */
        tag++; 

        attributes.clearEntries();      
        structureP = new StructureDescriptor( "context switch", attributes );

            /*
             * Field descriptors 
             */
            attributes.clearEntries();  
            attributes.insert( "description", "clock" );
            attributes.insert( "unit", "msec" );
            fieldP = new FieldDescriptor( "time stamp", attributes, DOUBLE, 0 );
            structureP->insert( *fieldP );
            delete fieldP;

            attributes.clearEntries();  
            fieldP = new FieldDescriptor( "processor_number", attributes, 
                                           INTEGER, 0 );
            structureP->insert( *fieldP );
            delete fieldP;

            fieldP = new FieldDescriptor( "proc_name", attributes, 
                                           CHARACTER, 1 );
            structureP->insert( *fieldP );
            delete fieldP;

        PipeWriter->putDescriptor( *structureP, tag );
        contextSwitchRecord = new RecordDossier( tag, *structureP );
        delete structureP;

        /*
         * Next a "procedure call summary" descriptor. Force it to
         * have tag = 9.
         */
        tag++; 

        attributes.clearEntries();      
        structureP = new StructureDescriptor( "procedure call summary", 
                                              attributes );

            /*
             * Field descriptors 
             */
            fieldP = new FieldDescriptor( "data file", attributes, 
                                          CHARACTER, 1 );
            structureP->insert( *fieldP );
            delete fieldP;

            fieldP = new FieldDescriptor( "procedures", attributes, 
                                          CHARACTER, 2 );
            structureP->insert( *fieldP );
            delete fieldP;

            attributes.insert( "dimension 0", "procedure" );
            attributes.insert( "dimension 1", "processor" );
            attributes.insert( "cell value", "count" );
            fieldP = new FieldDescriptor( "count of calls", attributes, 
                                          INTEGER, 2 );
            structureP->insert( *fieldP );
            delete fieldP;

        PipeWriter->putDescriptor( *structureP, tag );
        procCallRecord = new RecordDossier( tag, *structureP );
        delete structureP;

        /****************************************************************
         * A test of the command packet -- currently these packets pass
         * through Pablo with no effect.
         ****************************************************************/
        PipeWriter->putCommand( 17 );

        /****************************************************************
         * And, another stream attribute
         ****************************************************************/
        attributes.clearEntries();
        attributes.insert( "information", "end of record descriptors" );
        PipeWriter->putAttributes( attributes );

        /****************************************************************
         * Now some record data packets.  
         * First, set up pointers to the fields that are arrays and we'll
         * reuse those when setting dimensions sizes and array cell values
         * throughout.
	 *
	 * NOTE:  We put {}'s around the code sending each data packet as
	 * some compilers (Paragon) choked with too many temporaries and
	 * stack overflow when the {}'s were left out.
         ****************************************************************/

        Array *message_toPE = messageRecord->getArrayP( "toPE" );
        Array *context_switch_proc_name = 
                                contextSwitchRecord->getArrayP( "proc_name" );
        Array *proc_call_data_file = 
                                procCallRecord->getArrayP( "data file" );
        Array *proc_call_procedures = procCallRecord->getArrayP( "procedures" );
        Array *proc_call_count = procCallRecord->getArrayP( "count of calls" );

        int *dimSizes1 = new int[1];	
        int *dimSizes2 = new int[2];

        /* 
         * A Context Switch Data Packet 
         */
	{
        contextSwitchRecord->setValue( "time stamp", 100.15 );
        contextSwitchRecord->setValue( "processor_number", 2 );

        dimSizes1[0] = 30;
        context_switch_proc_name->setDimSizes( dimSizes1 );
        context_switch_proc_name->setCellString( "run1" );

        PipeWriter->putData( *contextSwitchRecord );
	}

        /* 
         * A Message Data Packet 
         */
	{
        messageRecord->setValue( "fromPE", 0 );

        dimSizes1[0] = 4;
        message_toPE->setDimSizes( dimSizes1 );
        message_toPE->setCellValue( 1, 0 );
        message_toPE->setCellValue( 3, 1 );
        message_toPE->setCellValue( 5, 2 );
        message_toPE->setCellValue( 7, 3 );

        messageRecord->setValue( "msglength", 512 );
        messageRecord->setValue( "msgType", 'B' );
        messageRecord->setValue( "time stamp", 100.1 );

        PipeWriter->putData( *messageRecord );
	}

        /* 
         * A Message Data Packet 
         */
	{
        messageRecord->setValue( "fromPE", 7 );

        dimSizes1[0] = 1;
        message_toPE->setDimSizes( dimSizes1 );
        message_toPE->setCellValue( 1, 0 );

        messageRecord->setValue( "msglength", 1012 );
        messageRecord->setValue( "msgType", 'S' );
        messageRecord->setValue( "time stamp", 102.15 );

        PipeWriter->putData( *messageRecord );
	}

        /* 
         * A Context Switch Data Packet.  Notice that we aren't resizing
         * our proc_name array.... we'll use the value from above (30)
         */
	{
        contextSwitchRecord->setValue( "time stamp", 108 );
        contextSwitchRecord->setValue( "processor_number", 4 );

        context_switch_proc_name->setCellString( "printf" );

        PipeWriter->putData( *contextSwitchRecord );
	}

        /* 
         * A Message Data Packet 
         */
	{
        messageRecord->setValue( "fromPE", 3 );

        dimSizes1[0] = 7;
        message_toPE->setDimSizes( dimSizes1 );
        message_toPE->setCellValue( 0, 0 );
        message_toPE->setCellValue( 1, 1 );
        message_toPE->setCellValue( 2, 2 );
        message_toPE->setCellValue( 4, 3 );
        message_toPE->setCellValue( 5, 4 );
        message_toPE->setCellValue( 6, 5 );
        message_toPE->setCellValue( 7, 6 );

        messageRecord->setValue( "msglength", 102 );
        messageRecord->setValue( "msgType", 'B' );
        messageRecord->setValue( "time stamp", 109 );

        PipeWriter->putData( *messageRecord );	
	}

        /* 
         * A Context Switch Data Packet.  Notice that we aren't resizing
         * our proc_name array.... we'll use the value from above (30)
         */
	{
        contextSwitchRecord->setValue( "time stamp", 320 );
        contextSwitchRecord->setValue( "processor_number", 5 );

        context_switch_proc_name->setCellString( "scanf" );

        PipeWriter->putData( *contextSwitchRecord );	
	}

        /* 
         * A Context Switch Data Packet. Note here we resize proc_name. 
         */
	{
        contextSwitchRecord->setValue( "time stamp", 412.77 );
        contextSwitchRecord->setValue( "processor_number", 2 );

        dimSizes1[0] = 15;
        context_switch_proc_name->setDimSizes( dimSizes1 );
        context_switch_proc_name->setCellString( "printf" );

        PipeWriter->putData( *contextSwitchRecord );
	}

        /*
         * A Procedure Call Summary Packet
         */
	{
        dimSizes1[0] = 30;
        proc_call_data_file->setDimSizes( dimSizes1 );
        proc_call_data_file->setCellString( "/mnt/data/testdata-1" );

        dimSizes2[0] = 3;
        dimSizes2[1] = 20;
        proc_call_procedures->setDimSizes( dimSizes2 );
        proc_call_procedures->setCellString( "compare()", 0 );
        proc_call_procedures->setCellString( "swap()", 1 );
        proc_call_procedures->setCellString( "error()", 2 );

        dimSizes2[0] = 3;
        dimSizes2[1] = 4;
        proc_call_count->setDimSizes( dimSizes2 );
        proc_call_count->setCellValue( 15, 0, 0 );
        proc_call_count->setCellValue( 10, 0, 1 );
        proc_call_count->setCellValue(  2, 0, 2 );
        proc_call_count->setCellValue( 20, 0, 3 );
        proc_call_count->setCellValue( 10, 1, 0 );
        proc_call_count->setCellValue(  7, 1, 1 );
        proc_call_count->setCellValue(  2, 1, 2 );
        proc_call_count->setCellValue(  5, 1, 3 );
        proc_call_count->setCellValue(  8, 2, 0 );
        proc_call_count->setCellValue(  3, 2, 1 );
        proc_call_count->setCellValue( 17, 2, 2 );
        proc_call_count->setCellValue(  2, 2, 3 );

        PipeWriter->putData( *procCallRecord );
	}

        /* 
         * Finally, a stream attribute to end the file
         */
	{
        attributes.clearEntries();
        attributes.insert( "information", "end of data" );
        PipeWriter->putAttributes( attributes );
	}

	/*
	 * Call destructors to tidy up memory and close output file.
	 */
	delete messageRecord;
	delete contextSwitchRecord;
	delete procCallRecord;
	delete dimSizes1;
	delete dimSizes2;
	delete PipeWriter;
        delete OutFile; 
}
