/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * StructureDictionary.cc - Methods for the StructureDictionary class.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Dictionaries/RCS/StructureDictionary.C,v 1.15 1994/02/25 04:31:54 aydt Exp $
 */

#include "Assert.h"
#include "Defines.h"
#include "StructureDictionary.h"

#include "AsciiPipeWriter.h"
#include "OutputFileStreamPipe.h"
#include "AsciiPipeReader.h"
#include "InputFileStreamPipe.h"
#include "PacketHeader.h"

StructureDictionary::StructureDictionary()
{
	headEntry = NULL;
	tailEntry = NULL;

	_setClassName( MY_CLASS );

	uniqueSystemTag = 1;
}

StructureDictionary::~StructureDictionary()
{
	StructureDictEntry* currentEntry = headEntry;
	StructureDictEntry* followingEntry;

	while ( currentEntry != NULL ) {
		followingEntry = currentEntry->nextEntry;
		delete currentEntry;
		currentEntry = followingEntry;
	}
}

Boolean_
StructureDictionary::contains( const int tag ) const
{
	StructureDictEntry *currentEntry = headEntry;

	while ( currentEntry != NULL ) {
	   if ( tag == currentEntry->systemTag )
	      return TRUE_;
	   else
	      currentEntry = currentEntry->nextEntry;
	}

	return FALSE_;
}

const StructureDescriptor& 
StructureDictionary::fetch( const int tag ) const
{
	StructureDictEntry *currentEntry = headEntry;

	while ( currentEntry != NULL ) {
	   if ( tag == currentEntry->systemTag )
	      return( currentEntry->structure );
	   else
	      currentEntry = currentEntry->nextEntry;
	}

	return noStructure;			// Not in list
}

int
StructureDictionary::insert( const StructureDescriptor& structDescr )
{
	StructureDictEntry *newEntry = new StructureDictEntry;

	newEntry->systemTag = uniqueSystemTag++;
	newEntry->structure = structDescr;

	if ( headEntry == NULL ) {
	   headEntry = newEntry;
	   tailEntry = newEntry;
	} else {
	   tailEntry->nextEntry = newEntry;
	   tailEntry = newEntry;
	}
	return newEntry->systemTag;
}

Boolean_
StructureDictionary::loadFromFile( const CString &filename )
{
	Assert( headEntry == NULL );		// Dictionary should be empty!
	Assert( uniqueSystemTag == 1 );

	InputFileStreamPipe *in = 
			       new InputFileStreamPipe( filename.getValue() );

	if ( in->successfulOpen() ) {

	    int LargestTag = 0;
	    StructureDictEntry *newEntry;

	    AsciiPipeReader *reader = new AsciiPipeReader( in );
	    PacketHeader PH = reader->getPacketHeader();

	    while( PH.type != PIPE_EMPTY ) {

		if ( PH.type == PKT_DESCRIPTOR ) {    
	    	    StructureDescriptor SD;
		   
		    if ( reader->getDescriptor( SD ) ) {	
		       if ( contains( PH.tag ) ) {
			  error( "Duplicate tag while loading StructDict: %d\n",
				  PH.tag );
		       } else {
			  newEntry = new StructureDictEntry();
			  newEntry->systemTag = PH.tag;
			  newEntry->structure = SD;

			  if ( headEntry == NULL ) {
			     headEntry = newEntry;
			     tailEntry = newEntry;
			  } else {
			     tailEntry->nextEntry = newEntry;
		             tailEntry = newEntry;
			  }
			  if ( PH.tag > LargestTag ) {
		             LargestTag	= PH.tag;
			  }
		       }	// if contains( PH.tag )
		    }		// if reader->getDescriptor( SD )
		}		// if PH.type == PKT_DESCRIPTOR

		PH = reader->getPacketHeader();
	    }

	    uniqueSystemTag = LargestTag + 1;
	    delete reader;
	    delete in;

	    if ( uniqueSystemTag == 1 ) {
		return FAILURE_;		// no entries - an error
	    } else {
	        return SUCCESS_;
	    }

	} else {

	    delete in;
	    return FAILURE_;

	}
}

void
StructureDictionary::reset()
{
	StructureDictEntry* currentEntry = headEntry;
	StructureDictEntry* followingEntry;

	while ( currentEntry != NULL ) {
		followingEntry = currentEntry->nextEntry;
		delete currentEntry;
		currentEntry = followingEntry;
	}

	headEntry = NULL;
	tailEntry = NULL;
	uniqueSystemTag = 1;
}

Boolean_
StructureDictionary::saveToFile( const CString& filename ) const
{
	OutputFileStreamPipe *out = 
				new OutputFileStreamPipe( filename.getValue() );

	if ( out->successfulOpen() ) {
	    AsciiPipeWriter *writer = new AsciiPipeWriter( out );

	    StructureDictEntry *dictEntry = headEntry;

	    while( dictEntry != NULL ) {
		writer->putDescriptor( dictEntry->structure, 
				       dictEntry->systemTag );
		dictEntry = dictEntry->nextEntry;
	    }

	    delete writer;
	    delete out;
	    return SUCCESS_;
	} else {
	    error( "Unable to open file: %s \n", filename.getValue() );
	    delete out;
	    return FAILURE_;
	}
		
}

void 		/* virtual */
StructureDictionary::printOn( ostream& os ) const
{
	os << "<StructureDictionary begin> \n";

	StructureDictEntry *currentEntry = headEntry;
	while ( currentEntry != NULL ) {
	   os << "System Tag: " << currentEntry->systemTag << "\n";
	   os << currentEntry->structure;
	   currentEntry = currentEntry->nextEntry;
	}

	os << "<StructureDictionary end> \n";
}

/*
 * 	Initialize the static data.   Only executed once.
 */
const char *const StructureDictionary::MY_CLASS = "StructureDictionary";
int StructureDictionary::uniqueSystemTag = -1;

/*
 * Definition of the static data if not done in a central place
 */
#ifdef CONSTRUCT_NONLOCAL_STATICS
       StructureDescriptor StructureDictionary::noStructure( CString::NOMATCH );
#endif
