/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * TagMappingDictionary.cc - Methods for the TagMappingDictionary class.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Dictionaries/RCS/TagMappingDictionary.C,v 1.16 1994/02/25 04:31:57 aydt Exp $
 */

#include <stdio.h>

#include "Defines.h"
#include "TagMappingDictionary.h"
#include "SystemErrors.h"

TagMappingDictionary::TagMappingDictionary()
{
	headEntry = NULL;
	tailEntry = NULL;

	_setClassName( MY_CLASS );
}

TagMappingDictionary::~TagMappingDictionary()
{
	TagMappingDictEntry *currentEntry = headEntry;
	TagMappingDictEntry *followingEntry;

	while ( currentEntry != NULL ) {
		followingEntry = currentEntry->nextEntry;
		delete currentEntry;
		currentEntry = followingEntry;
	}
}

void
TagMappingDictionary::clearAllEntries()
{
	TagMappingDictEntry *currentEntry = headEntry;
	TagMappingDictEntry *followingEntry;

	while ( currentEntry != NULL ) {
		followingEntry = currentEntry->nextEntry;
		delete currentEntry;
		currentEntry = followingEntry;
	}
	headEntry = NULL;
	tailEntry = NULL;
}

void
TagMappingDictionary::clearSrcEntries( int srcId )
{
	/* 
	 * This is called to remove all entries with the given sourceId from
	 * the TagMappingDictionary.  We do this by following the links
	 * through the current dictionary entries, and reinserting those
	 * that are from different sources in the new dictionary chain we
	 * build and deleteing entries from the old chain as we go.
	 */
	TagMappingDictEntry *oldEntry = headEntry;	// old dictionary
	TagMappingDictEntry *nextOldEntry;

	headEntry = NULL;				// reset
	tailEntry = NULL;				// reset

    	while ( oldEntry != NULL ) {
	    if ( oldEntry->sourceId != srcId ) {	// move to new
		insert( oldEntry->sourceId, oldEntry->sourceTag, 	
			oldEntry->systemTag );
	    }
            nextOldEntry = oldEntry->nextEntry;
	    delete oldEntry;			       // delete the old
	    oldEntry = nextOldEntry;
        }
}


int
TagMappingDictionary::fetch( const int srcId, const int srcTag ) const
{
	TagMappingDictEntry *currentEntry = headEntry;

	while ( currentEntry != NULL ) {
	   if ( ( srcId == currentEntry->sourceId ) &&
		( srcTag == currentEntry->sourceTag ) )
	      return currentEntry->systemTag;
	   else
	      currentEntry = currentEntry->nextEntry;
	}

	return FAILURE_;			// Not in list
}

Boolean_
TagMappingDictionary::insert( const int srcId, 
			      const int srcTag, 
			      const int sysTag )
{
	if ( fetch( srcId, srcTag ) ) {		// no duplicate srcId/srcTags!
	    return FAILURE_;	
	}

	TagMappingDictEntry *newEntry = new TagMappingDictEntry;
	newEntry->sourceId = srcId;
	newEntry->sourceTag = srcTag;
	newEntry->systemTag = sysTag;

	if ( headEntry == NULL ) {
	   headEntry = newEntry;
	   tailEntry = newEntry;
	} else {
	   tailEntry->nextEntry = newEntry;
	   tailEntry = newEntry;
	}
	return SUCCESS_;
}

Boolean_
TagMappingDictionary::loadFromFile( const CString& filename )
{
        clearAllEntries();                        // clear current entries 

        FILE *fp = fopen( filename.getValue(), "r" );

        if ( fp == NULL ) {
            error( "Unable to open %s: %s\n", filename.getValue(),
							errorString() );
            return FAILURE_;
        } else {
            int srcId, srcTag, sysTag;

            while ( fscanf( fp, "%d %d %d\n", &srcId, &srcTag, &sysTag ) 
                                                                    != EOF ) {
                insert( srcId, srcTag, sysTag );
            }

            fclose( fp );
            return SUCCESS_;
        }
}

Boolean_ 
TagMappingDictionary::saveToFile( const CString & filename ) const
{
	FILE *fp = fopen( filename.getValue(), "w" );

	if ( fp == NULL ) {
		error( "Unable to open %s: %s\n", filename.getValue(), 
						 errorString() );
		return FAILURE_;
	} else {
		TagMappingDictEntry *dictEntry = headEntry;

		while ( dictEntry != NULL ) {
			fprintf(fp, "%d %d %d\n", dictEntry->sourceId,
						  dictEntry->sourceTag,
						  dictEntry->systemTag );
			dictEntry = dictEntry->nextEntry;
		}
		fclose( fp );
		return SUCCESS_;
	}
}

void 
TagMappingDictionary::printOn( ostream& os ) const
{
	os << "<TagMappingDictionary begin> \n";

	TagMappingDictEntry *currentEntry = headEntry;
	while ( currentEntry != NULL ) {
	   os << "Source Id: " << currentEntry->sourceId << "\t"; 
	   os << "Source Tag: " << currentEntry->sourceTag << "\t"; 
	   os << "System Tag: " << currentEntry->systemTag << "\n"; 
	   currentEntry = currentEntry->nextEntry;
	}

	os << "<TagMappingDictionary end> \n";
}

/*
 * 	Initialize the static data.   Only executed once.
 */
const char *const TagMappingDictionary::MY_CLASS = "TagMappingDictionary";

