/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Authors: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *	    Daniel A. Reed (reed@cs.uiuc.edu)
 *          Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * BarGraphFU.cc - A polymorophic functional unit for a dyanamic bargraph.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/BarGraphFU.C,v 1.36 1994/03/15 16:41:23 aydt Exp $
 */

#include "BarGraphFU.h"
#include "BargraphFormWrapper.h"	

#include "FUParams.h"
#include "InputPort.h"
#include "ParamConfig.h"
#include "ParamDisplay.h"
#include "SystemErrors.h"

#define Normalize(x) ( ( ((double) x) - minValue ) / (maxValue - minValue) )
#define Round(x)     ( (int)(x + .5) )

BarGraphFU::BarGraphFU()
{
	_setClassName( MY_CLASS );
	inputPort = NULL;
	dialog = NULL;
	barGraphForm = NULL;
}

BarGraphFU::~BarGraphFU()
{
	delete inputPort;
	delete barGraphForm;
	delete dialog;
}

void 
BarGraphFU::configure()
{
#ifndef XtNminValue
#define	XtNminValue "minValue"
#define	XtNmaxValue "maxValue"
#endif

#ifndef XtNstickyHold
#define XtNstickyHold "stickyHold"
#endif
	/*
	 * The expected minimum and maximum data values are selected by
	 * the user, and these are used to scale the actual value seen
	 * by the BarGraphFU into the range of integer values accepted
	 * by the Bargraph widgets.  We never change the widget's minimum
	 * and maximum values.
	 */
	double 	    fuMin, fuMax;
	int	    fuHold;
	const char *labelBottom, *labelLeft;

	if ( ! isConfigured ) {
	    // ----- Input dimension for scalar/vector/array support ------
            dimension = inputPort->getTraits().getDimension();

	    int wMin, wMax;

            // ---- Obtain the Bargraph attributes from the Form child ---- 
	    barGraphForm->getBargraphAttr( XtNminValue, (XtArgVal) &wMin );
	    barGraphForm->getBargraphAttr( XtNmaxValue, (XtArgVal) &wMax );
	    barGraphForm->getBargraphAttr( XtNstickyHold, (XtArgVal) &fuHold );
 
	    fuMin = wMin;
	    fuMax = wMax;
	    valueOffset = wMin;
	    valueRange = wMax - wMin;
	    isConfigured = TRUE_;
	} else {
	    fuMin = minValue;
	    fuMax = maxValue;
	    fuHold = stickyHold;
	}

	// ----- Configuration Phase ------ 
        barGraphForm->getBargraphLabel( L_BOTTOM_HORIZ, &labelBottom );
        barGraphForm->getBargraphLabel( L_LEFT_VERT, &labelLeft );

	FUParams params;
	params.addTextParam( "Historical marker interval", 
				          BaseFUParamEntry::Integer, fuHold );
 	params.addTextParam( "Minimum value", BaseFUParamEntry::Real, fuMin );
 	params.addTextParam( "Maximum value", BaseFUParamEntry::Real, fuMax );
        params.addTextParam( "Vertical axis label", 
					  BaseFUParamEntry::Str, labelLeft );
        params.addTextParam( "Horizontal axis label", 
					  BaseFUParamEntry::Str, labelBottom );

        CString configPanelName = "Bargraph FU Config: " + getName();
        ParamConfig pc( Pablo::TopLevel(), params, configPanelName ); 
	pc.run();

	BaseFUParamEntry& stickyHoldEntry = 
				params.getEntry( "Historical marker interval" );
 	BaseFUParamEntry& minEntry = params.getEntry( "Minimum value" );
        BaseFUParamEntry& maxEntry = params.getEntry( "Maximum value" );
        BaseFUParamEntry& vertLabelEntry =
		                params.getEntry( "Vertical axis label" );
        BaseFUParamEntry& horizLabelEntry = 
		                params.getEntry( "Horizontal axis label" );
	
	if ( stickyHoldEntry.valueIsValid() ) {
	    fuHold = stickyHoldEntry.getValue().getInteger();
	} else {
	    warning( "Bargraph historical marker was not valid.\n" );
	}
	if ( minEntry.valueIsValid() ) {
            fuMin = minEntry.getValue().getReal();
        } else {
            warning( "Bargraph minimum value was not valid.\n" );
	}
        if ( maxEntry.valueIsValid() ) {
            fuMax = maxEntry.getValue().getReal();
        } else {
            warning( "Bargraph maximum value was not valid.\n" );
	}

	if ( fuMin >= fuMax ) {
	    warning( "Minimum >= Maximum.  Resetting Max to Min+1.\n");
	    fuMax = fuMin + 1;
	}

	// ----- Set Data to Configured Values for FU and Widget ------ 
	stickyHold = fuHold;
	minValue = fuMin;
	maxValue = fuMax;

	barGraphForm->setBargraphLabel( fuMin, fuMax );
        barGraphForm->setBargraphLabel( L_BOTTOM_HORIZ, 
			      horizLabelEntry.getValue().getString() );
	barGraphForm->setBargraphLabel( L_LEFT_VERT, 
	 		      vertLabelEntry.getValue().getString() );
        barGraphForm->setBargraphAttr( XtNstickyHold, fuHold );
}

FunctionalUnit *
BarGraphFU::copy()
{
	BarGraphFU *copy = new BarGraphFU();
	return copy;
}

void 
BarGraphFU::fuCallback( int iRow, int iCol )
{
	if ( inputValue.isUndefined() ) {
	    return;
	}

	FUParams params;
        if ( dimension == 0 ) {
            params.addDisplayParam( "Current Bargraph Value", inputValue );
        } else {
	    Value cValue;
	    Array *aInputData = (Array *)inputValue;
            const int *dimSizes = aInputData->getDimSizes();

            if ( dimension == 1 ) {
                if ( iCol < dimSizes[0] ) {
                   cValue = aInputData->getCellValue( iCol ); 
                   params.addDisplayParam( "Current Bargraph Value", cValue );
                } else {
                   params.addDisplayParam( "Current Bargraph Value", 
					    noDataMSG );
                }        
	        params.addDisplayParam( "Bargraph Column", iCol );
            } else if (dimension == 2) {
                if ( ( iRow < dimSizes[0] ) && ( iCol < dimSizes[1] ) ) {
                   cValue = aInputData->getCellValue( iRow, iCol );
                   params.addDisplayParam( "Current Bargraph Value", cValue );
                } else {
                   params.addDisplayParam( "Current Bargraph Value", 
					    noDataMSG );
                }
                params.addDisplayParam( "Bargraph Row", iRow );
                params.addDisplayParam( "Bargraph Column", iCol );
            }
        }

        CString callbackTitle = "Callback: " + getName();
	ParamDisplay *pd = new ParamDisplay( Pablo::TopLevel(), params, 
					     callbackTitle );
}

void 
BarGraphFU::init()
{
	/*
	 * Start by clearing up any 'leftovers' that will be around if 
	 * init() isn't being called for the first time.  The goal is to
	 * start with a 'virgin' FU.
	 */
	if ( inputPort != NULL ) {
	    delete inputPort;
	}
	if ( barGraphForm != NULL ) {
	    delete barGraphForm;
	}
	if ( dialog != NULL ) {
	    delete dialog;
	}

	/*
	 * Here's the real code to initialize the FU
	 */
	isConfigured = FALSE_;
	inputValue = Value::NOVALUE;

	inputPort = new InputPort( "Input" );
	inputPort->addTraits( INTEGER, 0 );
	inputPort->addTraits( INTEGER, 1 );
	inputPort->addTraits( INTEGER, 2 );
	inputPort->addTraits( FLOAT, 0 );
	inputPort->addTraits( FLOAT, 1 );
	inputPort->addTraits( FLOAT, 2 );
	inputPort->addTraits( DOUBLE, 0 );
	inputPort->addTraits( DOUBLE, 1 );
        inputPort->addTraits( DOUBLE, 2 );
	_addInputPort( inputPort );

	dialog = _getTitledFormDialog( "FUDialog", getName() );
	barGraphForm = new BargraphFormWrapper( dialog, this, NullArgs, 
								"Contents" );
	dialog->manage();
        XtVaSetValues( dialog->getWidget(), XmNdefaultPosition, False, NULL );

}

Boolean_		/* virtual */
BarGraphFU::loadConfigFromFile( const CString& fileName )
{
        FILE *fp = fopen( fileName.getValue(), "r" );
        if ( fp == NULL ) {
            warning( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
        } else {
	    int x, y, width, height;
            int labelSize;
	    char buf[LABEL_MAX_LEN];

	    // ---- Comment line and numeric paramaters ---- 
	    fscanf( fp,"%*[^\n]\n" ); 
            if ( fscanf( fp, "%lf %lf %d %d %d %d %d %d\n",
                           &minValue, &maxValue, &stickyHold,
			   &x, &y, &width, &height, &dimension ) != 8 ) {
                warning( "Unable to read configuration information from %s\n",
                          fileName.getValue() );
	    } else {
		// ---- Comment Line; Horizontal Label Size and String ---- 
		labelSize = 0;
		fscanf( fp, "%*[^\n]\n" );
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
	        if ( labelSize == 0 ) {
                   strcpy( buf, "\0");
                }

                barGraphForm->setBargraphLabel( L_BOTTOM_HORIZ, buf );

		// ---- Comment Line; Vertical Label Size and String ---- 
		labelSize = 0;
                fscanf( fp, "\n%*[^\n]\n" );
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
                if ( labelSize == 0 ) {
		   strcpy( buf, "\0" );
                }
                barGraphForm->setBargraphLabel( L_LEFT_VERT, buf );

		/* 
		 * Recalculate the valueOffset and valueRange based on the 
		 * widget's min and max values. Use loaded parameters to set 
		 * other widget values.
		 */
     		int wMin, wMax;
                barGraphForm->getBargraphAttr( XtNminValue, (XtArgVal) &wMin );
                barGraphForm->getBargraphAttr( XtNmaxValue, (XtArgVal) &wMax );
                valueOffset = wMin;
                valueRange = wMax - wMin;

		barGraphForm->setBargraphLabel( minValue, maxValue );
		barGraphForm->setBargraphAttr( XtNstickyHold, stickyHold );

		barGraphForm->setPerfWidgetPosition( x, y, width, height );
                isConfigured = TRUE_;
            } 
            fclose( fp );
        }

	if ( ! isConfigured ) {
	    configure();
	}
        return isConfigured;
}

Boolean_ 
BarGraphFU::ready()
{
	return TRUE_;
}

void			/* virtual */
BarGraphFU::restart()
{
        /* 
	 * First, clear the values for the existing bargraph displays
	 * Use the current "Input Value" to get the number of rows and 
	 * columns.
	 */
        switch ( dimension ) {
          case SCALAR:
	    barGraphForm->setDisplayValues( valueOffset );
            break;

          case VECTOR: 
          case ARRAY_2D:
	    {
	    // ---- Get the number of elements in the vector or array.
            Array *aInputData = (Array *) inputValue;
            const int *aDimSizes = aInputData->getDimSizes();
            int nRows, nCols;

	    if ( dimension == VECTOR ) {
          	nRows = 1;
		nCols = aDimSizes[0];
	    } else if ( dimension == ARRAY_2D) {
           	nRows = aDimSizes[0]; 
		nCols = aDimSizes[1];
	    }

	    // ---- Fill the vector/array structure with "zero" values
	    int nValues = nRows * nCols;
            int *vals = new int[ nValues ];

            for ( int i = 0; i < nValues; i++ ) { 
               vals[i] = valueOffset;
            }

            barGraphForm->setDisplayValues( nCols, nRows, vals );

            delete[] vals;
	    }
            break;
        }

	inputValue = Value::NOVALUE;
}

void 
BarGraphFU::run( Boolean_& /* errorFlag */ )
{
	Assert( inputPort->valueAvailable() );
	inputValue = inputPort->getValue();
       
	/* 
	 * We interactively vary the Bargraph layout depending upon
	 * the incoming data size; dimension is fixed.
	 */
        switch ( dimension ) {
          case SCALAR:
	    {
            double inputValueScaled = warnings.fuDataRangeWarningCheck( this,
				       (double)inputValue, minValue, maxValue );
	    int val = Round( Normalize( inputValueScaled ) * valueRange );
	    val += valueOffset;    
	    barGraphForm->setDisplayValues( val );
	    }
            break;

          case VECTOR: 
          case ARRAY_2D:
	    {
            Array *aInputData = (Array *) inputValue;

	    // ---- Get the number of elements in the vector or array.
            const int *aDimSizes = aInputData->getDimSizes();
            int nRows, nCols;

	    if ( dimension == VECTOR ) {
          	nRows = 1;
		nCols = aDimSizes[0];
	    } else if ( dimension == ARRAY_2D) {
           	nRows = aDimSizes[0]; 
		nCols = aDimSizes[1];
	    }

	    // ---- Fill the vector/array structure with input data 
            double tmpValueScaled;
	    int nValues = nRows * nCols;
            int *vals = new int[ nValues ];

            for ( int i = 0; i < nValues; i++ ) { 
               tmpValueScaled = warnings.fuDataRangeWarningCheck( this,
                       			 (double)aInputData->getTheCellValue(i),
					 minValue, maxValue );
               vals[i] = Round( Normalize( tmpValueScaled ) * valueRange );
               vals[i] += valueOffset;
            }

            barGraphForm->setDisplayValues( nCols, nRows, vals );

            delete[] vals;
	    }
            break;
        }
}

Boolean_		/* virtual */
BarGraphFU::saveConfigToFile( const CString& fileName ) const
{
        Boolean_ result;

        FILE *fp = fopen( fileName.getValue(), "w" );
        if ( fp == NULL ) {
            error( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
            result = FAILURE_;
        } else {
            int x, y, width, height;
	    const char *label;

	    barGraphForm->getPerfWidgetPosition( &x, &y, &width, &height );
	    fprintf( fp, "# Min Max History X Y Width Height Dimension\n" );
	    fprintf( fp, "%lf %lf %d %d %d %d %d %d\n",
			 minValue, maxValue, stickyHold, x, y, 
			 width, height, dimension );

            barGraphForm->getBargraphLabel( L_BOTTOM_HORIZ, &label );
            fprintf( fp, "# Horizontal Label\n" );
            fprintf( fp, "%d %s\n", strlen( label), label );

            barGraphForm->getBargraphLabel( L_LEFT_VERT, &label );
            fprintf( fp, "# Vertical Label\n" );
            fprintf( fp, "%d %s\n", strlen( label ), label );

	    fclose( fp );
            result = SUCCESS_;
        }

        return result;
}


/*
 *      Initialize the static data.   Only executed once.
 */
const char *const BarGraphFU::MY_CLASS = "BarGraphFU";
