/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * BinaryMathFU.cc - A polymorophic functional unit for binary math
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/BinaryMathFU.C,v 1.19 1994/03/15 16:41:28 aydt Exp $
 */
 
#include <string.h>

#include "BinaryMathFU.h"
#include "Addition.h"
#include "Subtraction.h"
#include "Multiplication.h"
#include "Division.h"
#include "Minimum.h"
#include "Maximum.h"
#include "Average.h"
#include "DblDivision.h"
#include "DblAverage.h"

#include "FUParams.h"
#include "ParamConfig.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "SystemErrors.h"

#define Max(a, b) ( (a) > (b) ? (a) : (b) )

BinaryMathFU::BinaryMathFU()
{
	_setClassName( MY_CLASS );

	leftOperand = NULL;
	rightOperand = NULL;
	result = NULL;
	myOperator = NULL;
	myOperatorNumber = -1;
	isReady = FALSE_;
}

BinaryMathFU::~BinaryMathFU()
{
	delete leftOperand;
	delete rightOperand;
	delete result;
	delete myOperator;
}

void 
BinaryMathFU::_setOperator( const int opId )
{
	if ( myOperator != NULL ) {
  	    delete myOperator;
	}

  	myOperatorNumber = opId;

  	switch ( opId ) {
     	   case ADDITION:
              myOperator = new Addition;
              break;

     	   case SUBTRACTION:
              myOperator = new Subtraction;
              break;

     	   case MULTIPLICATION:
              myOperator = new Multiplication;
              break;

     	   case DIVISION:
              myOperator = new Division;
              break;

     	   case MINIMUM:
              myOperator = new Minimum;
              break;

     	   case MAXIMUM:
              myOperator = new Maximum;
              break;

     	   case AVERAGE:
              myOperator = new Average;
              break;

	   case DBL_DIVISION:
	      myOperator = new DblDivision;
              break;

	   case DBL_AVERAGE:
	      myOperator = new DblAverage;
	      break;

     	   default:
              warning( 
		   "\nOperator %d does not correspond to any known operation",
	           opId );
              myOperatorNumber = -1;
              break;
  	}

  	if ( myOperatorNumber < 0 ) {
      	   isReady = FALSE_;
  	} else {
           isReady = TRUE_;
  	}
}

void                            /* virtual */
BinaryMathFU::configureOperation()
{
  	int opId = -1;
  	FUParams params;
  	params.addRadioButtonsParam( "Binary Operation",
			    BaseFUParamEntry::Integer, 0, BinaryOpNameList );
	ParamConfig pc ( Pablo::TopLevel(), params, getName() );
  	BaseFUParamEntry& opEntry = params.getEntry( "Binary Operation" );

	if ( myOperatorNumber != -1 ) {
	    opEntry.setValue( myOperatorNumber );
	}
  	pc.run();

  	if ( opEntry.valueIsValid() ) {
     	   opId = opEntry.getValue().getInteger();
  	} else {
     	   warning( "Operation selection was not valid." );
	   opId = 0;
  	}

  	_setOperator( opId );
}

FunctionalUnit *		/* virtual */
BinaryMathFU::copy()
{
	BinaryMathFU *copy = new BinaryMathFU();
	return copy;
}

void 				/* virtual */
BinaryMathFU::init()
{
	/*
	 * BinaryOpNameList is a static that is only initialized once 
	 */
	if ( BinaryOpNameList.isEmpty() ) {
	    BinaryOpNameList.addElement( "Addition" );
	    BinaryOpNameList.addElement( "Subtraction" );
	    BinaryOpNameList.addElement( "Multiplication" );
	    BinaryOpNameList.addElement( "Division" );
	    BinaryOpNameList.addElement( "Minimum" );
	    BinaryOpNameList.addElement( "Maximum" );
	    BinaryOpNameList.addElement( "Average" );
	    BinaryOpNameList.addElement( "Double-precision_Division" );
	    BinaryOpNameList.addElement( "Double-precision_Average" );
	}

	/*
	 * Continue by clearing up any 'leftovers' that will be around if
	 * init() isn't being called for the first time.  The goal is
	 * to start with a 'virgin' FU.
	 */
	if ( leftOperand != NULL ) {
	    delete leftOperand;
	}
	if ( rightOperand != NULL ) {
	    delete rightOperand;
	}
	if ( result != NULL ) {
	    delete result;
	}
	if ( myOperator != NULL ) {
	    delete myOperator;
	    myOperator = NULL;
	    myOperatorNumber = -1;
	    isReady = FALSE_;
	}

	/*
	 * Here's the real code to initialize the FU
	 */
	leftOperand = new InputPort( "LeftOperand" );
	leftOperand->addTraits( INTEGER, -1 );
	leftOperand->addTraits( FLOAT, -1 );
	leftOperand->addTraits( DOUBLE, -1 );
	_addInputPort( leftOperand );

	rightOperand = new InputPort( "RightOperand" );
	rightOperand->addTraits( INTEGER, -1 );
	rightOperand->addTraits( FLOAT, -1 );
	rightOperand->addTraits( DOUBLE, -1 );
	_addInputPort( rightOperand );

	result = new OutputPort( "Result" );
	_addOutputPort( result );

}

Boolean_ 			/* virtual */
BinaryMathFU::inputTraitsValid()
{
	DataTraits leftT = leftOperand->getTraits();
	DataTraits rightT = rightOperand->getTraits();

	/* 
	 * If either of the operands is a scalar, we are able to compute
	 * the result.  If both are arrays, then we make sure they have
	 * the same dimensions.
	 */

	Boolean_ valid;
	if ( leftT.isScalar() || rightT.isScalar() )  {
	    valid = TRUE_;
	} else {				// Both Arrays
	    if ( leftT.getDimension() == rightT.getDimension() ) {
		valid = TRUE_;
	    } else {
		valid = FALSE_;
	    }
	}

	if ( valid ) {
	    /*
	     * If we are doing double precision division or average, then 
	     * always output double precision.  Otherwise,  we create a 
	     * fake value whose type matches the left operand and then 
	     * call the getResultType method on it with the type of the 
	     * right operand to determine the type of the result.
	     */
	    if ( ( myOperatorNumber == DBL_DIVISION ) || 
		 ( myOperatorNumber == DBL_AVERAGE ) ) {
	        DataTraits resultT( DOUBLE, 
			   Max( leftT.getDimension(), rightT.getDimension() ) );
	        result->setTraits( resultT );
	    } else {
	        Value v( leftT.getType(), 0 );	
	        DataTraits resultT( v.getResultType( rightT.getType() ), 
			   Max( leftT.getDimension(), rightT.getDimension() ) );
	        result->setTraits( resultT );
	    }
	}
	
	return valid;
}

Boolean_                        /* virtual */
BinaryMathFU::loadConfigFromFile( const CString& fileName )
{
  	char opName[256];
  	int op;

  	FILE *fp = fopen( fileName.getValue(), "r" );
  	if ( fp == NULL ) {
    	   warning( "\nUnable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
  	} else {
	   fscanf( fp, "%*[^\n]\n" );  // Comment line
	    
    	   if ( fscanf( fp, "%s", opName ) == 1 ) {
              if ( ( op = BinaryOpNameList.lookup( opName ) ) < 0 ) {
                  warning( "\n%s is not a valid operation name", opName );
              } else {
		 _setOperator( op );         // sets isReady flag if successful
      	      }
    	   }
    	   fclose(fp);
  	}

  	if ( isReady == FALSE_ ) { 	// one last chance to configure
	   configureOperation();
           configure();
  	}

  	return( isReady );
}

Boolean_ 			/* virtual */
BinaryMathFU::ready()
{
	return isReady;
}

void 				/* virtual */
BinaryMathFU::run( Boolean_& errorFlag )
{
	Assert( leftOperand->valueAvailable() );
	Assert( rightOperand->valueAvailable() );

        myOperator->getBinaryResult( *( leftOperand->getValueP() ),
				     *( rightOperand->getValueP() ), 
				     resultV );

	if ( resultV.isDefined() ) {
	    result->setValue( resultV );
	    _writeOutput();
	} else {
	    errorFlag = TRUE_;
	}
}

Boolean_ 
BinaryMathFU::saveConfigToFile( const CString& fileName ) const
{
  	Boolean_ retCode = FAILURE_;

  	FILE *fp = fopen( fileName.getValue(), "w" );

  	if ( fp == NULL ) {
     	   error( "\nUnable to open %s: %s\n", fileName.getValue(),
	  		errorString() );
  	} else {
	   fprintf( fp, "# OperationName\n" );
    	   fprintf( fp, "%s\n",
		    BinaryOpNameList.getElement(myOperatorNumber).getValue() );

    	   retCode = SUCCESS_;
    	   fclose( fp );
  	}

  	return retCode;
}
    
/*
 * 	Initialize the static data.   Only executed once.
 */
const char *const BinaryMathFU::MY_CLASS = "BinaryMathFU";
CStringObjList BinaryMathFU::BinaryOpNameList;
