/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Contributing Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * SynthesizeCoordinates.cc - A functional unit for populating a 
 * 2-dimensional array with coordinates of (a, b, ... z) values.
 * Suitable as input to the Scatter3D and XYGraph functional units.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/ClusterFU.C,v 1.6 1994/04/11 15:34:22 aydt Exp $
 */

#include <stdlib.h>

#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/PushB.h>
#include <Xm/Separator.h>
#include <Xm/Text.h>

#include "ClusterFU.h"
#include "WidgetBase.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "SystemErrors.h"

#define BaseWidget       ( Pablo::TopLevel()->getWidget() )


ClusterFU::ClusterFU() 
{
        _setClassName( MY_CLASS );
}


ClusterFU::~ClusterFU()
{
}


Widget
ClusterFU::_setupClusterConfigWindow()
{
        static ClusterConfigData clusterConfigData;

	clusterConfigData.dialog = XmCreateFormDialog( BaseWidget,
					      "ClusterConfigDialogContainer",
					      NULL, 0 );
	Widget dialogActionArea = XtVaCreateManagedWidget("ConfigActionArea",
						  xmFormWidgetClass,
						  clusterConfigData.dialog,
						      NULL );

	/* Data dimension */
        Widget dialogDataDimensionArea =
	     multivariateWrapper.addTextFieldToDialog( dialogActionArea,
				                       NULL, 
					      "ConfigDialogDataDimension",
				       &(clusterConfigData.totalDimension) );
        char string[STRING_MAX_LEN];
        sprintf( string, "%d", dimension );
        XmTextSetString( clusterConfigData.totalDimension, string );

        /* Clustering frequency */
	Widget dialogClusteringFrequencyArea =
	    multivariateWrapper.addTextFieldToDialog( dialogActionArea,
						      dialogDataDimensionArea,
					    "ConfigDialogClusteringFrequency",
		                  &(clusterConfigData.clusteringFrequency) );  
	sprintf( string, "%d", clusteringFrequency );
	XmTextSetString( clusterConfigData.clusteringFrequency, string );

        /* Control area */
	Widget paneSeparator = XtVaCreateManagedWidget(
	                            "DialogPaneSeparator",
				     xmSeparatorWidgetClass,
				     clusterConfigData.dialog,
				     XmNtopWidget, dialogActionArea,
					 NULL );
	Widget dialogControlArea = XtVaCreateManagedWidget( 
					      "DialogControlArea",
					       xmFormWidgetClass,
				               clusterConfigData.dialog,
					       XmNtopWidget, paneSeparator,
					       XmNfractionBase, 1,
						   NULL );
	Widget dialogOKButton = XtVaCreateManagedWidget(
					       "DialogOKButton",
					        xmPushButtonWidgetClass,
						dialogControlArea,
						XmNleftPosition, 0,
						XmNrightPosition, 1,
						   NULL );
        addCallback( dialogOKButton, XmNactivateCallback, &Callback::callback1,
		       this, &clusterConfigData );	
			    
        return( clusterConfigData.dialog );
}


// This is the done-with-config-window callback
void ClusterFU::callback1( Widget /* callbackWidget */,
		           XtPointer ptr1, XtPointer /* ptr2 */ )
{
        ClusterConfigData *data = (ClusterConfigData *)ptr1;
	XtUnmanageChild( data->dialog );

        // Extract data dimension
        char *dimensionText = XmTextGetString( data->totalDimension );
	sscanf( dimensionText, "%d", &dimension );
	XtFree( dimensionText );


	// Extract clustering frequency
	char *frequencyText = XmTextGetString( data->clusteringFrequency );
	sscanf( frequencyText, "%d", &clusteringFrequency );
	XtFree( frequencyText );
					      
	// Define initial datasize 
	if ( savedValues == NULL ) {
	   savedValues = ( double * ) malloc( clusteringFrequency*dimension * 
					      sizeof(double) );
	} else {
	   savedValues = ( double * ) realloc( savedValues,
		            clusteringFrequency*dimension * sizeof(double) );
	}
	// Temp hack
        cluster.setNumberOfClusters( 4 ); 
}


void
ClusterFU::configure()
{
        static Widget configureShell;
	
	if ( configureShell == NULL ) {
           configureShell = _setupClusterConfigWindow();
	}
	XtManageChild( configureShell );
}


FunctionalUnit *      /* virtual */
ClusterFU::copy()
{
        ClusterFU *copy = new ClusterFU();
	return copy;
}


void
ClusterFU::init()
{
        // Input port
        inputPort = new InputPort( "Input" );
        inputPort->addTraits( DOUBLE, 2 );
	_addInputPort( inputPort );

	// Output port
	outputPort = new OutputPort( "Clustering Output" );
	_addOutputPort( outputPort );
}


Boolean_
ClusterFU::inputTraitsValid()
{
        DataTraits portTraits( DOUBLE, 2 );
	outputPort->setTraits( portTraits );

	return( TRUE_ );
}


Boolean_             /* virtual */
ClusterFU::loadConfigFromFile( const CString& fileName )
{
        FILE *fp = fopen( fileName.getValue(), "r" );
	
	if ( fp == NULL ) {
           warning(  "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
	   configure();
	} else {
	   
	    // ---- Comment line and numeric paramaters ---- 
	    fscanf( fp,"%*[^\n]\n" );
	    if ( fscanf( fp, "%d %d\n", 
			 &dimension, &clusteringFrequency ) != 2 ) {
	       warning( "Unable to read configuration information from %s\n",
		         fileName.getValue() );
	    } else {
	      // Allocate space for saved data 
	      if ( savedValues == NULL ) {
	         savedValues = (double *) malloc( 
			  clusteringFrequency * dimension * sizeof(double) );
	      } else {
		 savedValues = (double *) realloc( savedValues, 
			   clusteringFrequency * dimension * sizeof(double) );
	      }
	      isConfigured = TRUE_;

    	      // Temp hack
              cluster.setNumberOfClusters( 4 ); 
	   }
	}
	fclose( fp );
	if ( ! isConfigured ) {
           configure();
	} 
	
	return( isConfigured );
}
       

Boolean_
ClusterFU::ready()
{
        return TRUE_;
}


void
ClusterFU::run( Boolean_& /* errorFlag */ )
{
        Assert( inputPort->valueAvailable() );

	inputValue = inputPort->getValue();  
	Array *a = (Array *)inputValue;
	int *dimSizes = (int *)a->getDimSizes();
	int overallPointCount = dimSizes[0];

	int i;
	for (i=0; i<dimension; i++) {
	    savedValues[ pointCount*dimension + i ] = 
	                           a->getCellValue(overallPointCount-1, i); 
	}

	if ( ++pointCount % clusteringFrequency == 0 ) {
	    cluster.loadDataset( savedValues, dimension, clusteringFrequency );
	    cluster.cluster();
	    
	    int numberOfClusters = cluster.getNumberOfClusters();
	    int *repList = (int *)malloc( numberOfClusters * sizeof(int) );
	    cluster.getClusterReps( repList );

	    Array *outputValueP = new Array( DOUBLE, 2 );
	    dimSizes[0] = numberOfClusters;
	    dimSizes[1] = dimension;
	    outputValueP->setDimSizes( dimSizes );
	    
	    int i, j;
	    for ( i=0; i < numberOfClusters; i++) {
	        for (j=0; j < dimension; j++) {
		    outputValueP->setCellValue(
			   savedValues[ repList[i]*dimension + j ], i, j );
		}
	    }
	    outputPort->setValue( (Value)outputValueP );
	    _writeOutput();
            pointCount = 0;

            delete outputValueP;
	    free( repList );
	}
}


Boolean_              /* virtual */
ClusterFU::saveConfigToFile( const CString& fileName ) const
{
        Boolean_ result;
	
        FILE *fp = fopen( fileName.getValue(), "w" );
	if ( fp == NULL ) {
	   error( "Unable to open %s: %s\n", fileName.getValue(),
		   errorString() );
	   result = FAILURE_;
	} else {
	   fprintf( fp, "# Dimension ClusterFrequency\n" );
	   fprintf( fp, "%d %d\n", dimension, clusteringFrequency );
           result = SUCCESS_;
	}
	fclose( fp );
	return( result );
}


/*
 *      Initialize the static data.   Only executed once.
 */
const char *const ClusterFU::MY_CLASS = "ClusterFU";
