/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Authors: Philip C. Roth (proth@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/***************************************************
 *
 * NumberLine.C
 *
 * method implementation for NumberLine class.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/NumberLine.C,v 1.13 1994/04/08 13:05:42 aydt Exp $
 *
 ***************************************************/
#include <strstream.h> 
#include <stdlib.h>
#include <math.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/DrawingA.h>
#include <Xm/ScrollBar.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/Separator.h>
#include <Xm/Label.h>
#include <Xm/ToggleBG.h>

#include "NumberLine.h"
#include "Selection.h"

#ifdef ultrix
extern "C" {
/* for some reason, rint isn't in <math.h> on ULTRIX V4.2 (Rev. 96) */
double rint( double );
}
#endif

/*
 * Definition of useful macros
 */
#define	MIN(x,y) ( (x) < (y) ? (x) : (y) )
#define	MAX(x,y) ( (x) > (y) ? (x) : (y) )


/*
 * #defines of constants used for array dimensions
 * g++ allows 'const int' for this, but CC doesn't
 */
#define	MAX_DIGITS	10


/*
 * initialization of static constant class data
 */
const int NumberLine::MARGIN = 30;
const int NumberLine::TICK = 10;
const int NumberLine::LNWID = 9;
const int NumberLine::ARROW = 10;
const int NumberLine::ARROW_GRAVITY = 2*ARROW;
const char* const NumberLine::NL_CHARSET = "charset1";


/*
 * NumberLine()
 *
 * Class constructor:
 * Sets up a numberline on range [lo,hi], with no selection.
 * Does some X intitialization for drawing on the NumberLine DrawingArea widget.
 */
NumberLine::NumberLine( Widget par, int lo, int hi )
{
	XmFontContext	myContext;
	XmStringCharSet chset;



	/*
	 * verify that the numberline range is ok
	 * if it isn't, force it to be a valid range
	 */
	if( lo > hi ) {
		int temp = hi;
		hi = lo;
		lo = temp;
	}

	/*
	 * Initialize private data dealing with the visual range
	 */
	minValue = minVisValue = minFocValue = lo;
	maxValue = maxVisValue = maxFocValue = hi;
	interval = _goodInterval( minValue, maxValue );


	/* 
	 * set flags
	 */
	pointFlag = True;	// point mode at first
	selectFlag = True;	// select mode at first
	focFlag = False;
	rangeEndSet = False;

	/* 
	 * set the initial selection to be empty
	 */
	currSList = NULL;

	/* create Widget tree implementing the NumberLine */
	hierarchy = _makeNumberLineWidgets( par );
	
	
	/* get Drawing Area resources of interest */
	XtVaGetValues( infoWidgets[DA], 
			XtNheight, &height,
			XtNwidth, &width,
			NULL );
	XtVaGetValues( getHierarchy(),
			XmNlabelFontList, &nlFontList,
			NULL );

	/* get the desired font out of the font list */
	/* do not change or free this font */
	XmFontListInitFontContext( &myContext, nlFontList );
	XmFontListGetNextFont( myContext, &chset, &nlFont ); 
	XmFontListFreeFontContext( myContext );

}



NumberLine::~NumberLine( void )
{

	delete currSList;	
	XtDestroyWidget( hierarchy );
}



/*
 * _clearAllSelections()
 *
 * Clears the current selections from the NumberLine, so that nothing
 * is selected.
 */
void
NumberLine::_clearAllSelections( void )
{
	delete currSList;
	currSList = NULL;

	/* force a redraw of the numberline */
	XClearArea( dpy, wndw, 0, 0, 0, 0, True );
}




/*
 * _clearPoint()
 *
 * Clear the point over which user released the mouse button.
 */
void
NumberLine::_clearPoint( void )
{


	_drawCircle( ptrVal, True );		// erase point with label


	/* remove the selection */
	currSList = currSList->removePoint( ptrVal );

	XtRemoveEventHandler( infoWidgets[DA], ButtonMotionMask, FALSE,
			(XtEventHandler)dragPointEH, this );


	/* force window redraw */
	XClearArea( dpy, wndw, 0, 0, 0, 0, True );
}




/*
 * _clearRange()
 *
 * Clear range user just selected from the selection list.
 */
void
NumberLine::_clearRange( void )
{
	int lo = MIN( ptrVal, rangeEnd );
	int hi = MAX( ptrVal, rangeEnd );

	/* remove newly selected range to list */
	currSList = currSList->removeRange( lo, hi );

	/* stop tracking range */
	XtRemoveEventHandler( infoWidgets[DA], ButtonMotionMask, FALSE,
			(XtEventHandler)dragRangeEH, this );

	/* force redraw */
	XClearArea( dpy, wndw, 0, 0, 0, 0, True );
}




/*
 * _dragFocusEnd()
 *
 * drags the focus end around the NumberLine.
 */

void
NumberLine::_dragFocusEnd( XEvent *event )
{
	int *iptr;		// ptr to focus end to be updated
	Boolean redraw = False;	// flags whether automatic redraw is done
	int val;		// old logical value


	val = _logicalVal( event->xbutton.x );

	if( (val < minValue) || (val > maxValue) )
		return;

	_drawArrow( ptrVal, focDir );		// erase existing arrow
	ptrVal = val; 				// update tracking value

	/* determine which end we are updating */
	iptr = ( focDir == LEFT ? &maxFocValue : &minFocValue );

	/* make sure we don't scroll past end of the NumberLine */
	if( ptrVal > maxVisValue ) {
		redraw = True;	

		if( maxVisValue != maxValue ) {
			/* scroll one interval right */
			*iptr = ( (ptrVal > maxValue) ||
				  	(maxVisValue + interval > maxValue)
				  ? maxValue 
				  : maxVisValue + interval );
			ptrVal = *iptr;

			_handleOneUp();
		}
		else {
			*iptr = maxValue;	// don"t scroll past max end
			ptrVal = maxValue;
		}
	}
	else if( ptrVal < minVisValue ) {
		redraw = True;

		if( minVisValue != minValue ) {
			/* scroll one interval right */
			*iptr = ( (ptrVal < minValue) ||
				  	(minVisValue - interval < minValue)
				  ? minValue 
				  : minVisValue - interval );

			ptrVal = *iptr;
			_handleOneDown();
		}
		else {
			*iptr = minValue;	// don"t scroll past min end
			ptrVal = minValue;
		}
	}
	else {
		*iptr = ptrVal;			// track the drag
	}

	if( !redraw ) {
		_drawArrow( ptrVal, focDir );
	}


	/* handle the case when the arrows passed each other */
	if( minFocValue > maxFocValue ) {
		if( !redraw ) {
			/* erase existing arrows */
			_drawArrow( minFocValue, RIGHT );
			_drawArrow( maxFocValue, LEFT );
	
			/* draw arrows with correct direction */
			_drawArrow( minFocValue, LEFT );
			_drawArrow( maxFocValue, RIGHT );
		}

		/* swap ends */
		{
			int tmp = minFocValue;
			minFocValue = maxFocValue;
			maxFocValue = tmp;
		}

		/* indicate switch in direction of arrow being dragged */
		focDir = ( focDir == LEFT ? RIGHT : LEFT );
	}
}




/*
 * _dragPoint()
 *
 * Handles the drag of the selected point.
 */
void
NumberLine::_dragPoint( XEvent *event )
{
	int val;
	Boolean redraw = False;


	val = _logicalVal( event->xbutton.x );

	if( (val < minValue) || (val > maxValue) )
		return;

	_drawCircle( ptrVal, True );		// erase existing circle
	ptrVal = val;				// update tracking value

	/* scroll window if necessary */
	if( ptrVal > maxVisValue ) {
		redraw = True;

		if( maxVisValue != maxValue ) {
			/* scroll right one interval if possible */
			ptrVal = ( (ptrVal > maxValue) ||
					(maxVisValue + interval > maxValue )
				? maxValue
				: maxVisValue + interval );
			_handleOneUp();
		}
		else {
			ptrVal = maxValue;
		}
	}
	else if( ptrVal < minVisValue ) {
		redraw = True;

		if( minVisValue != minValue ) {
			/* scroll left one interval if possible */
			ptrVal = ( (ptrVal < minValue) ||
					(minVisValue - interval < minValue )
				? minValue
				: minVisValue - interval );
			_handleOneDown();
		}
		else {
			ptrVal = minValue;
		}
	}

	/* redraw selection if there is not automatic redraw */
	if( !redraw ) {
		_drawCircle( ptrVal, True );
	}
}



/*
 * _dragRange()
 *
 * rubber-bands a range selection, tracking user mouse movement.
 */
void
NumberLine::_dragRange( XEvent *event )
{
	int val;
	Boolean redraw = False;


	val = _logicalVal( event->xbutton.x );

	if( (val < minValue) || (val > maxValue) )
		return;

	/* erase existing range */
	if( ptrVal < rangeEnd ) {
		_drawRange( ptrVal, rangeEnd );
	}
	else {
		_drawRange( rangeEnd, ptrVal );
	}
	ptrVal = val;				// update tracking value

	/* scroll window if necessary */
	if( ptrVal > maxVisValue ) {
		redraw = True;

		if( maxVisValue != maxValue ) {
			/* scroll right one interval if possible */
			ptrVal = ( (ptrVal > maxValue) ||
					(maxVisValue + interval > maxValue )
				? maxValue
				: maxVisValue + interval );
			_handleOneUp();
		}
		else {
			ptrVal = maxValue;
		}
	}
	else if( ptrVal < minVisValue ) {
		redraw = True;

		if( minVisValue != minValue ) {
			/* scroll left one interval if possible */
			ptrVal = ( (ptrVal < minValue) ||
					(minVisValue - interval < minValue )
				? minValue
				: minVisValue - interval );
			_handleOneDown();
		}
		else {
			ptrVal = minValue;
		}
	}

	/* redraw selection if there is not automatic redraw */
	if( !redraw ) {
		if( ptrVal < rangeEnd ) {
			_drawRange( ptrVal, rangeEnd );
		}
		else {
			_drawRange( rangeEnd, ptrVal );
		}
	}
}



/*
 * _draw()
 *
 * Draws the visual part of the numberline on its DrawingArea, including
 * the line, tick marks, end arrows if needed, and labels.
 */
void
NumberLine::_draw( void )
{
	char	valstr[MAX_DIGITS];	// string for label
	int	slen;			// length of label string
	Dimension	swidth;		// width of label string
	int	sheight;		// height of a string in font nlFont
	int	x, xint;		// x value and x interval for ticks
	int	val;			// logical value on numberline


	yloc = (2*height) / 3;		// y-coordinate of horizontal line


	/* compute the visual interval */
	xint = maxVisValue == minVisValue ? 1 : 
				(int)ceil( (width - 2*MARGIN) * interval) / 
				(maxVisValue - minVisValue );
	
	/* 
	 * draw ticks and labels 
	 */
	sheight = nlFont->ascent + nlFont->descent;

	/* draw first n-1 ticks */
	for( val = minVisValue, x = MARGIN; 
	     val < maxVisValue; 
	     val += interval, x = _physicalVal( val ) ) 
	{
		/* draw the ticks */
		XDrawLine( dpy, wndw, gc, x, yloc - TICK, x, yloc + TICK );
		
		/* create the label */
		ostrstream ostr( valstr, MAX_DIGITS );
		ostr << val << '\0';
		slen = strlen( valstr );
		swidth = XTextWidth( nlFont, valstr, slen );

		/* draw the label */
		XDrawString( dpy, wndw, gc, 
			x - (swidth/2), yloc + TICK + sheight,
			valstr, slen );
	}

	/* draw last tick */
	/* this centers the tick if there is only one number on the line */
	x = ( minVisValue != maxVisValue ? width - MARGIN : width / 2 );
	XDrawLine( dpy, wndw, gc, x, yloc - TICK, x, yloc + TICK );

	/* create the label */
	ostrstream ostr( valstr, MAX_DIGITS );
	ostr << maxVisValue << '\0';
	slen = strlen( valstr );
	swidth = XTextWidth( nlFont, valstr, slen );

	/* draw the label */
	XDrawString( dpy, wndw, gc, 
		x - (swidth/2), yloc + TICK + sheight,
		valstr, slen );



	if( minVisValue != maxVisValue )
	{

		/* draw horizontal line */
		XDrawLine( dpy, wndw, gc, MARGIN, yloc, x, yloc );
	
		/* draw end arrows if needed */
		if( minVisValue != minValue ) 
		{
			XDrawLine( dpy, wndw, gc, MARGIN, yloc,
					MARGIN - 2*TICK, yloc );
			XDrawLine( dpy, wndw, gc, MARGIN - 2*TICK, yloc,
					MARGIN - TICK, yloc - TICK );
			XDrawLine( dpy, wndw, gc, MARGIN - 2*TICK, yloc,
					MARGIN - TICK, yloc + TICK );
		}
		
		if( maxVisValue != maxValue )
		{
			XDrawLine( dpy, wndw, gc, x, yloc,
					x + 2*TICK, yloc );
			XDrawLine( dpy, wndw, gc, x + 2*TICK, yloc,
					x + TICK, yloc - TICK );
			XDrawLine( dpy, wndw, gc, x + 2*TICK, yloc,
					x + TICK, yloc + TICK );
		}

		/* draw Focus arrows if necessary */
		if( minFocValue >= minVisValue ) {
			_drawArrow( minFocValue, RIGHT );
		}
	
		if( maxFocValue <= maxVisValue ) {
			_drawArrow( maxFocValue, LEFT );
		}
					
	}


	_drawSelection();

}




/*
 * _drawArrow()
 *
 * draws the arrow indicating an end of the focus range, along
 * with its associated label.
 */
void
NumberLine::_drawArrow( int val, int dir )
{
	char valstr[MAX_DIGITS];	// the label string
	int  slen;
	Dimension swidth;
	Position  x;
	XPoint	points[3];
	ostrstream ostr( valstr, MAX_DIGITS );	// string output stream


	/* switch to XOR drawing mode */
	XSetFunction( dpy, gc, GXxor );

	
	/* find the position corresponding to logical value val */
	x = _physicalVal( val );

	/* draw value label */
	ostr << val << '\0';
	slen = strlen( valstr );
	swidth = XTextWidth( nlFont, valstr, slen );

	/* draw the label */
	XDrawString( dpy, wndw, gc, 
		x - (swidth/2), yloc - 3*TICK - ARROW - 4,
		valstr, slen );

	/* draw arrow */
	if( dir == LEFT ) {
		points[0].x = x; 	points[0].y = yloc - 3*TICK;
		points[1].x = 0;	points[1].y = -ARROW;
		points[2].x = -ARROW;	points[2].y = ARROW/2;
	}
	else {
		points[0].x = x; 	points[0].y = yloc - 3*TICK;
		points[1].x = 0;	points[1].y = -ARROW;
		points[2].x = ARROW;	points[2].y = ARROW/2;
	}
	XFillPolygon( dpy, wndw, gc,
		points, 3,
		Convex,
		CoordModePrevious );


	/* restore gc function */
	XSetFunction( dpy, gc, GXcopy );

}



/*
 * _drawCircle()
 *
 * Draws a circle of diameter LNWID centered on the line at the
 * given NumberLine value.
 * The value label is drawn if drawLabel is True.
 */
void 
NumberLine::_drawCircle( int val, Boolean drawLabel )
{
	char valstr[MAX_DIGITS];
	int slen;
	Dimension swidth;
	ostrstream ostr( valstr, MAX_DIGITS );	// string output stream


	/* switch gc functions */
	XSetFunction( dpy, gc, GXxor );

	if( (val >= minVisValue) && (val <= maxVisValue) ) {

		/* find the (x,y) coordinates of the circle */
		Position x = _physicalVal( val );

		XFillArc( dpy, wndw, gc,
				x - LNWID/2 - 1, yloc - LNWID/2,
				LNWID, LNWID, 0, 64*360 );
	
		/* draw label if desired */
		if ( drawLabel ) {
			ostr << val << '\0';
			slen = strlen( valstr );
			swidth = XTextWidth( nlFont, valstr, slen );
		
			/* draw the label */
			XDrawString( dpy, wndw, gc, 
				x - (swidth/2), yloc - TICK - 4,
				valstr, slen );
		}

	}

	/* restore gc function */
	XSetFunction( dpy, gc, GXcopy );

}


/*
 * _drawRange()
 *
 * Highlights the given range on the numberline
 * It assumes lo < hi.
 */
void
NumberLine::_drawRange( int lo, int hi )
{
	/* switch gc functions and line attributes */
	XSetFunction( dpy, gc, GXxor );
	XSetLineAttributes( dpy, gc, LNWID, LineSolid, CapRound, JoinMiter );

	/* this should do some clipping to the visible range */
	if( lo < minVisValue )
		lo = minVisValue;
	if( hi > maxVisValue )
		hi = maxVisValue;

	XDrawLine( dpy, wndw, gc, 
			_physicalVal( lo ), yloc, _physicalVal( hi ), yloc );

	/* restore gc functions and line attributes */
	XSetFunction( dpy, gc, GXcopy );
	XSetLineAttributes( dpy, gc, 0, LineSolid, CapRound, JoinMiter );
}




/*
 * _drawSelection()
 *
 * Highlights the current selection on the visible portion of the NumberLine.
 * Assumes the selection is minimal - there are no distinct Selection
 * objects with intersecting ranges.
 */
void
NumberLine::_drawSelection( void )
{
	Selection* curr;
	int	low, high;		/* curr's selection range */



	/* step through all selection ranges and highlight the selection */
	curr = currSList;
	while( curr ) {
		curr->getRange( low, high );

		/* highlight current selection */
		if( low == high ) 
			_drawCircle( low, False );
		else
			_drawRange( low, high );

		curr = (Selection*)curr->getNextInList();
	}


}




/*
 * _expand2x()
 *
 * Expands the visible NumberLine range by a factor of 2.
 */
void
NumberLine::_expand2x( void )
{
	int range = maxVisValue - minVisValue;


	switch( range ) {
		case 0:	
			if( maxValue == minValue ) {
				return;		// nothing to expand to
			}

		case 1:	
			if( maxVisValue != maxValue ) {
				maxVisValue++;
			}
			else {
				minVisValue--;
			}
			break;

		default:
			minVisValue = MAX( minVisValue-( range / 2 ),minValue );
			maxVisValue = MIN( maxVisValue+( range / 2 ),maxValue );
	}



	/* compute new NumberLine interval between ticks */
	interval = _goodInterval( minVisValue, maxVisValue );

	/* reset slider values */
	XtVaSetValues( infoWidgets[SBAR],
		XmNvalue,		minVisValue,
		XmNsliderSize,		maxVisValue - minVisValue,
		XmNincrement,		interval,
		XmNpageIncrement,	maxVisValue - minVisValue,
		NULL );

	/* force a redraw of the numberline */
	XClearArea( dpy, wndw, 0, 0, 0, 0, True );
}


/*
 * _expandCompletely()
 *
 * Expands the visible NumberLine range to the whole numberline.
 */
void
NumberLine::_expandCompletely( void )
{

	minVisValue = minValue;
	maxVisValue = maxValue;

	/* check for a line with only one element */
	if( maxValue == minValue ) {
		return;
	}
	
	/* compute new NumberLine interval between ticks */
	interval = _goodInterval( minVisValue, maxVisValue );

	/* reset slider values */
	XtVaSetValues( infoWidgets[SBAR],
		XmNvalue,		minVisValue,
		XmNsliderSize,		maxVisValue - minVisValue,
		XmNincrement,		interval,
		XmNpageIncrement,	maxVisValue - minVisValue,
		NULL );

	/* force a redraw of the numberline */
	XClearArea( dpy, wndw, 0, 0, 0, 0, True );
}



/*
 * _goodInterval()
 *
 * We want approximately 8 intervals in the range from lo to hi, but want
 * them to be readable as well.  _goodInterval computes the integers such
 * that this constraint is satisfied.
 */
/* static */
int
NumberLine::_goodInterval( int lo, int hi )
{
	int range = hi - lo + 1;
	
	if( range >= 8 )
		return (int)ceil( range / 8 );
	else if( range ) {
		return 1;
	}
	else {
		return 0;
	}
}




/*
 * _handleButtonPress()
 *
 * handles button press events
 */
void
NumberLine::_handleButtonPress( XEvent *be )
{
	/* handle clicks depending on which button */
	switch( be->xbutton.button  ) {

		case Button1:
			if( _selectedNumberLinePoint( be ) ) {

				/* are we dealing with points or ranges? */
				if( pointFlag ) {
					_startDraggingPoint();
				}
				else {
					_startDraggingRange();
				}
			}

			/* check for focus arrow dragging */
			else if( _selectedFocArrow( be ) ) {
				_startFocusEnd();
				focFlag = True;
			}
			break;

		case Button2:
			break;

		case Button3:
			break;
	}

}



/*
 * _handleButtonRelease()
 *
 */
void
NumberLine::_handleButtonRelease( XEvent *be )
{

	/* handle clicks depending on which button */
	switch( be->xbutton.button  ) {

		case Button1:

			/* check if user was draggin forcus arrow */
			if( focFlag ) {
				/* user had selected focus arrow */
				_setFocusEnd();
				focFlag = False;
			}

			/* was user manipulating a point? */
			else if( pointFlag ) {

				if( selectFlag ) {
					_setSelectPoint();
				}
				else {
					_clearPoint();
				}
			}

			/* user was manipulating a range */
			else if( rangeEndSet ) {
				if( selectFlag ) {
					_setSelectRange();
				}
				else {
					_clearRange();
				}
			}

			break;

		case Button2:
			break;

		case Button3:
			break;
	}

}



/* 
 * _handleExpose()
 *
 * handles X  Expose events 
 */
void
NumberLine::_handleExpose( XmDrawingAreaCallbackStruct *cb )
{
	XRectangle cliprect;
	

	/* only redraw inside the clip region */
	Region cliprgn = XCreateRegion();
	XtAddExposureToRegion( cb->event, cliprgn );
	XSetRegion( dpy, gc, cliprgn );

	/* update the display */
	_draw();

	XDestroyRegion( cliprgn );

	/* reset clip region to the whole window */
	cliprect.x = 0;
	cliprect.y = 0;
	cliprect.width = width;
	cliprect.height = height;
	XSetClipRectangles( dpy, gc, 0, 0, &cliprect, 1, Unsorted );

}


/*
 * _handleMouse()
 *
 * handles events generated by pressing, dragging, and releasing mouse
 * button.
 */
void
NumberLine::_handleMouse( XmDrawingAreaCallbackStruct *cb )
{
	XEvent *be = cb->event;


	ptrVal = _logicalVal( be->xbutton.x );	// which ptrValue on the Line?

	/* is this a buttonPress or Release? */
	if( be->xany.type == ButtonPress ) {
		_handleButtonPress( be );
	}

	else if( be->xany.type == ButtonRelease ) {
		_handleButtonRelease( be );
	}
}




/*
 * _handleOneDown()
 *
 * Reset visible range down by one interval.
 */
void
NumberLine::_handleOneDown( void )
{

	/* make sure we don't try to go too far up the numberline */
	if( minVisValue - interval < minValue )  {
		minVisValue = minValue;
		maxVisValue = minValue + interval;
	}
	else {
		minVisValue -= interval;
		maxVisValue -= interval;
	}

	/* reset scrollbar value  */
	XtVaSetValues( infoWidgets[SBAR],
		XmNvalue,	minVisValue,
		NULL );

	/* force a redraw of the numberline */
	XClearArea( dpy, wndw, 0, 0, 0, 0, True );
}



/*
 * _handleOneUp()
 *
 * Reset visible range up by one interval.
 */
void
NumberLine::_handleOneUp( void )
{
	/* make sure we don't try to go too far up the numberline */
	if( maxVisValue + interval > maxValue ) {
		maxVisValue = maxValue;
		minVisValue = maxValue - interval;
	}
	else {
		minVisValue += interval;
		maxVisValue += interval;
	}

	/* reset scrollbar value */
	XtVaSetValues( infoWidgets[SBAR],
		XmNvalue, 	minVisValue,
		NULL );

	/* force a redraw of the numberline */
	XClearArea( dpy, wndw, 0, 0, 0, 0, True );

}




/*
 * _handlePageDown()
 *
 * Reset visible range down by one screenfull.
 */
void
NumberLine::_handlePageDown( void )
{
	int range = maxVisValue - minVisValue;

	/* make sure we don't try to go too far up the numberline */
	if( minVisValue - range < minValue )  {
		minVisValue = minValue;
		maxVisValue = minValue + range;
	}
	else {
		maxVisValue = minVisValue;
		minVisValue -= range;
	}

	/* reset scrollbar value */
	XtVaSetValues( infoWidgets[SBAR],
		XmNvalue,	minVisValue,
		NULL );

	/* force a redraw of the numberline */
	XClearArea( dpy, wndw, 0, 0, 0, 0, True );
}



/*
 * _handlePageUp()
 *
 * Resets the visible numberline range by one screenfull.
 */
void
NumberLine::_handlePageUp( void )
{
	int range = maxVisValue - minVisValue;

	/* make sure we don't try to go too far up the numberline */
	if( maxVisValue + range > maxValue ) {
		maxVisValue = maxValue;
		minVisValue = maxValue - range;
	}
	else {
		minVisValue = maxVisValue;
		maxVisValue += range;
	}

	/* reset scrollbar value */
	XtVaSetValues( infoWidgets[SBAR],
		XmNvalue,	minVisValue,
		NULL );

	/* force a redraw of the numberline */
	XClearArea( dpy, wndw, 0, 0, 0, 0, True );
}



/*
 * _handleResize()
 *
 * handles X resize events
 */
void
NumberLine::_handleResize( void )
{


	/* clear what used to be visible:
		the use of width and height instead of 0, 0 forces the
		corresponding exposure event to indicate the rectangle
		of the pre-resized window, as opposed to the new window.
		This avoids multiple redraws of the same parts of the
		NumberLine, which would cause erasure due to the use of
		GXxor drawing mode.	*/
	if( XtIsRealized( infoWidgets[DA] ) ) 
		XClearArea( dpy, wndw, 0, 0, width, height, True );

	/* get new size of Drawing Area */
	XtVaGetValues( infoWidgets[DA],
			XmNwidth,	&width,
			XmNheight,	&height,
			NULL );

}



/*
 * _handleToBottom()
 *
 * Handles moving ScrollBar thumb to bottom of its range
 */
void
NumberLine::_handleToBottom( void )
{
	int range = maxVisValue - minVisValue;

	/* move the visual range to the top of the NumberLine range */
	minVisValue = minValue;
	maxVisValue = minValue + range;

	/* set new scrollbar value */
	XtVaSetValues( infoWidgets[SBAR], 
			XmNvalue,	minVisValue,
			NULL );

	/* force a redraw of the numberline */
	XClearArea( dpy, wndw, 0, 0, 0, 0, True );
}



/*
 * _handleToTop()
 *
 * Handles moving ScrollBar thumb to top of its range
 */
void
NumberLine::_handleToTop( void )
{
	int range = maxVisValue - minVisValue;

	/* move the visual range to the top of the NumberLine range */
	maxVisValue = maxValue;
	minVisValue = maxValue - range;

	/* set new scrollbar value */
	XtVaSetValues( infoWidgets[SBAR],
				XmNvalue,	minVisValue,
				NULL );

	/* force a redraw of the numberline */
	XClearArea( dpy, wndw, 0, 0, 0, 0, True );
}




/*
 * _handleValueChanged()
 *
 * Callback which handles scrollbar drags and value changes.
 * Simply resets the minimum and maximum visible values and redraws the
 * scrollbar.
 */
void
NumberLine::_handleValueChanged( XmScrollBarCallbackStruct *cd )
{
	int range = maxVisValue - minVisValue;	// save the initial visual range

	/* reset the visual range of the NumberLine */
	minVisValue = cd->value;
	maxVisValue = minVisValue + range;	// keep same width of range

	/* redraw NumberLine */
	XClearArea( dpy, wndw, 0, 0, 0, 0, True );
}



/*
 * _logicalVal()
 *
 * computes the logical value most closely associated with the
 * visual x-coordinate it is given as an argument.
 */
int
NumberLine::_logicalVal( Position x )
{
	int range = maxVisValue - minVisValue;

#ifdef sparc
	return ( range ? minVisValue + 
		nint( range * (float)( x - MARGIN )/( width - 2*MARGIN ) )
		  	: minVisValue );
#else
	return ( range ? minVisValue + 
		(int)rint( range * (float)( x - MARGIN )/( width - 2*MARGIN ) )
		  	: minVisValue );
#endif sparc
}




/*
 * _makeFocusArea()
 *
 * creates the widget tree for the NumberLine focus management area.
 */
Widget
NumberLine::_makeFocusArea( Widget par )
{
	Widget focArea = XtVaCreateWidget( "nlFocusForm",
				xmFormWidgetClass, par,
				NULL );

	Widget focLabel = XtVaCreateManagedWidget( "nlFocusLabel",
				xmLabelWidgetClass, focArea,
				NULL );

	Widget toArrow = XtVaCreateManagedWidget( "nlChangeToArrow",
				xmPushButtonWidgetClass, focArea, 
				NULL );
						
	Widget expandC = XtVaCreateManagedWidget( "nlExpandCompletely",
				xmPushButtonWidgetClass, focArea,
				NULL );

	Widget expand2 = XtVaCreateManagedWidget( "nlExpand2",
				xmPushButtonWidgetClass, focArea, 
				NULL );
						
	Widget zoom2 = XtVaCreateManagedWidget( "nlZoom2",
				xmPushButtonWidgetClass, focArea, 
				NULL );
						
	/* specify spatial relationships between widgets */
	XtVaSetValues( toArrow,
			XmNtopWidget,	focLabel,
			NULL );

	XtVaSetValues( expandC,
			XmNtopWidget,	toArrow,
			NULL );

	XtVaSetValues( expand2,
			XmNtopWidget,	expandC,
			NULL );

	XtVaSetValues( zoom2,
			XmNtopWidget,	expandC,
			XmNleftWidget,	expand2,
			NULL );
	
	XtManageChild( focArea );


	/* set callbacks for focus widgets */
	XtAddCallback( toArrow, XmNactivateCallback,
				(XtCallbackProc)refocusCB, this );
	XtAddCallback( expandC, XmNactivateCallback,
				(XtCallbackProc)expandCompleteCB, this );
	XtAddCallback( expand2, XmNactivateCallback,
				(XtCallbackProc)expand2xCB, this );
	XtAddCallback( zoom2, XmNactivateCallback,
				(XtCallbackProc)zoom2xCB, this );

	return focArea;
}





/*
 * _makeNumberLineWidgets()
 *
 * creates the NumberLine widget hierarchy
 */
Widget
NumberLine::_makeNumberLineWidgets( Widget par )
{
	

	Widget form = XtVaCreateWidget( "nlForm",
				xmFormWidgetClass, par, 
				NULL );

	Widget da = XtVaCreateWidget( "nlDA",
				xmDrawingAreaWidgetClass, form, 
				NULL );
	infoWidgets[DA] = da;
	
	/* create scrollbar */
	/* be careful to make sure its minimum value and maximum value */
	/* aren't the same if the NumberLine is to be a point */
	int tmpval = ( minValue == maxValue ? minValue + 1 : maxValue );
	Widget sbar = XtVaCreateManagedWidget( "nlScroll",
				xmScrollBarWidgetClass, form, 
				XmNtopWidget,	da,
				XmNminimum,	minValue,
				XmNmaximum,	tmpval,
				XmNsliderSize, 	tmpval - minValue,
				XmNvalue,	minValue,
				XmNincrement,	interval,
				XmNpageIncrement,	tmpval - minValue,
				XmNorientation,	XmHORIZONTAL,
				NULL );
	infoWidgets[SBAR] = sbar;


	Widget sep1 = XtVaCreateManagedWidget( "nlInteractionSep",
				xmSeparatorWidgetClass, form,
				XmNtopAttachment,	XmATTACH_WIDGET,
				XmNtopWidget,		sbar,
				XmNleftAttachment,	XmATTACH_POSITION,
				XmNleftPosition,	50,
				XmNbottomAttachment,	XmATTACH_FORM,
				XmNbottomOffset,	5,
				XmNorientation,		XmVERTICAL,
				NULL );

	Widget selArea = _makeSelectionArea( form );
	Widget focArea = _makeFocusArea( form );

	/* 
	 * set spatial relationships between these interaction areas
	 * and the actual NumberLine
	 */
	XtVaSetValues( selArea,
			XmNtopWidget,	sbar,
			XmNrightWidget,	sep1,
			NULL );
	XtVaSetValues( focArea,
			XmNtopWidget,	sbar,
			XmNleftWidget,	sep1,
			NULL );

	XtManageChild( da );
	XtManageChild( form );

	/* 
	 * set callbacks
	 */

	/* for the drawing area */
	XtAddCallback( da, XmNexposeCallback, (XtCallbackProc)exposeCB, this );
	XtAddCallback( da, XmNresizeCallback, (XtCallbackProc)resizeCB, this );
	XtAddCallback( da, XmNinputCallback, 
					(XtCallbackProc)handleMouseCB, this );

	/* for the scrollbar */
	XtAddCallback( sbar, XmNincrementCallback, 
				(XtCallbackProc)oneUpCB, this );
	XtAddCallback( sbar, XmNdecrementCallback,
				(XtCallbackProc)oneDownCB, this );
	XtAddCallback( sbar, XmNpageIncrementCallback,
				(XtCallbackProc)pageUpCB, this );
	XtAddCallback( sbar, XmNpageDecrementCallback,
				(XtCallbackProc)pageDownCB, this );
	XtAddCallback( sbar, XmNtoTopCallback, 
				(XtCallbackProc)toTopCB, this );
	XtAddCallback( sbar, XmNtoBottomCallback, 
				(XtCallbackProc)toBottomCB, this );
	XtAddCallback( sbar, XmNvalueChangedCallback, 
				(XtCallbackProc)valueChangedCB, this );
	XtAddCallback( sbar, XmNdragCallback, 
				(XtCallbackProc)valueChangedCB, this );
				

	return form;
}



/*
 * _makeSelectionArea()
 * 
 * creates the widget tree for the NumberLine selection management area.
 */
Widget
NumberLine::_makeSelectionArea( Widget par )
{
	Widget selArea = XtVaCreateWidget( "nlSelectionForm",
				xmFormWidgetClass, par,
				NULL );


	Widget selLabel = XtVaCreateManagedWidget( "nlSelLabel",
				xmLabelWidgetClass, selArea,
				NULL );

	Widget frame = XtVaCreateWidget( "extentFrame",
				xmFrameWidgetClass, selArea,
				XmNtopWidget,	selLabel,
				NULL );

	Widget rbox = XmCreateRadioBox( frame, "nlSelectionExtent", NULL, 0 );

	/* add toggle buttons to rbox */
	Widget point = XtVaCreateManagedWidget( "nlSelPoint",
				xmToggleButtonGadgetClass, rbox,
				NULL );

	Widget range = XtVaCreateManagedWidget( "nlSelRange",
				xmToggleButtonGadgetClass, rbox,
				NULL );

	XtManageChild( rbox );
	XtManageChild( frame );

	frame = XtVaCreateWidget( "modeFrame",
				xmFrameWidgetClass, selArea,
				XmNtopWidget,	selLabel,
				NULL );

	rbox = XmCreateRadioBox( frame, "nlSelectionMode", NULL, 0 );

	/* add toggle buttons to rbox */
	Widget select = XtVaCreateManagedWidget( "nlSelSelect",
				xmToggleButtonGadgetClass, rbox,
				NULL );

	Widget deselect = XtVaCreateManagedWidget( "nlSelDeselect",
				xmToggleButtonGadgetClass, rbox,
				NULL );

	XtManageChild( rbox );
	XtManageChild( frame );


	Widget clrAll = XtVaCreateManagedWidget( "nlClearAll",
				xmPushButtonWidgetClass, selArea,
				XmNtopWidget,	frame,
				NULL );

	XtManageChild( selArea );

	/* set callbacks for these widgets */
	XtAddCallback( clrAll, XmNactivateCallback,
				(XtCallbackProc)clearAllCB, this );

	XtAddCallback( select, XmNvalueChangedCallback,
			(XtCallbackProc)selectModeCB, this );
	XtAddCallback( point, XmNvalueChangedCallback,
			(XtCallbackProc)pointModeCB, this );


	return selArea;
}



/*
 * _physicalVal()
 *
 * computes the physical value most closely associated with the
 * logical value of its argument.
 */
Position
NumberLine::_physicalVal( int val )
{
	int range = maxVisValue - minVisValue;

#ifdef sparc
	return ( range ? MARGIN + nint( (width - 2*MARGIN) * 
			(float)( val - minVisValue)/ range )
			: width / 2 );
#else
	return ( range ? MARGIN + (int)rint( (width - 2*MARGIN) * 
			(float)( val - minVisValue)/ range )
			: width / 2 );
#endif sparc
}



/*
 * _refocus()
 *
 * Sets the visual range to that defined by the focus range.
 * Called when user has refocused on a portion of the NumberLine.
 */
void
NumberLine::_refocus( void )
{
	minVisValue = minFocValue;
	maxVisValue = maxFocValue;

	/* compute new NumberLine interval between ticks */
	interval = _goodInterval( minVisValue, maxVisValue );

	/* reset slider values */
	/* check for 1 element numberlines */
	int tmpval = ( minVisValue == maxVisValue ? minVisValue + 1 
						  : maxVisValue );
	XtVaSetValues( infoWidgets[SBAR],
		XmNvalue,		minVisValue,
		XmNsliderSize,		tmpval - minVisValue,
		XmNincrement,		interval,
		XmNpageIncrement,	tmpval - minVisValue,
		NULL );

	/* force a redraw of the numberline */
	XClearArea( dpy, wndw, 0, 0, 0, 0, True );

}



/*
 * _selectedFocArrow()
 *
 * returns True if the button press selected a focus arrow, 
 * False otherwise.
 * As side effect, sets ptrVal to the focus end value which was
 * selected.
 */
Boolean
NumberLine::_selectedFocArrow( XEvent *be )
{
	Boolean result = True;
	Position	minx, maxx;


	/* check if there are focus arrows to select */
	if( minVisValue == maxVisValue )
		result = False;

	/* check if it was not a numberline click */
	if( be->xbutton.y > yloc - 2*TICK )
		result = False;

	/* check that it was close to a focus arrow */
	minx = _physicalVal( minFocValue );
	maxx = _physicalVal( maxFocValue );

	if( ( abs( be->xbutton.x - minx ) > ARROW_GRAVITY ) && 
		( abs( be->xbutton.x - maxx ) > ARROW_GRAVITY ) ) 
		result = False;

	/* apply "gravity" if one of the arrows was nearly selected */
	if( abs( be->xbutton.x - minx ) <= ARROW_GRAVITY )
		ptrVal = minFocValue;
	else if( abs( be->xbutton.x - maxx ) <= ARROW_GRAVITY )
		ptrVal = maxFocValue;

	return result;
}


/*
 * _selectedNumberLinePoint()
 *
 * returns True if user selected a point "on" the numberline, and
 * False otherwise.
 * Should check if the click was on an existing selection or not,
 * and return that selection.
 */
Boolean
NumberLine::_selectedNumberLinePoint( XEvent *be )
{


	if( abs( be->xbutton.y - yloc ) > TICK )
		return False;

	if( minVisValue != maxVisValue ) {
		if( ( be->xbutton.x > width - MARGIN ) ||
		    ( be->xbutton.x < MARGIN ) )
			return False;
	}
	else {
		/* there is only one point on the numberline */
		/* see if we're close */
		if( abs( be->xbutton.x - (Position)(width/2) ) > TICK )
			return False;
	}

	return True;
}




/*
 * _setFocusEnd()
 *
 * Ends drag of current focus range ends.
 */
void
NumberLine::_setFocusEnd()
{
	/* make sure that user didn't try to make focus range empty */
	if( minFocValue == maxFocValue ) {
		minFocValue = oldMinFocValue;
		maxFocValue = oldMaxFocValue;

		/* force a redraw of the numberline, to correct the mistake */
		XClearArea( dpy, wndw, 0, 0, 0, 0, True );
	}

	/* remove the event handler for tracking the pointer */
	XtRemoveEventHandler( infoWidgets[DA], ButtonMotionMask, FALSE,
			(XtEventHandler)dragFocusEndEH, this );

}



/*
 * _setSelectPoint()
 *
 * Selects the point over which user released the mouse button.
 */
void
NumberLine::_setSelectPoint( void )
{
	Selection *sel;


	_drawCircle( ptrVal, True ); 		// erase point with label 
	_drawCircle( ptrVal, False ); 		// redraw without 


	/* add this new selection to the existing list */
	sel = new Selection( ptrVal );
	currSList = (Selection *)sel->insertYourself( currSList );

	/* don't need to track the drag anymore */
	XtRemoveEventHandler( infoWidgets[DA], ButtonMotionMask, FALSE,
			(XtEventHandler)dragPointEH, this );

	/* force redraw of the NumberLine */
	XClearArea( dpy, wndw, 0, 0, 0, 0, True );
}



/*
 * _setSelectRange()
 *
 * Adds range user selected to selection list.
 */
void
NumberLine::_setSelectRange( void )
{

	Selection *sel;


	/* add the newly selected range to list */
	sel = new Selection( MIN( ptrVal, rangeEnd ), MAX( ptrVal, rangeEnd ) );
	currSList = (Selection *)sel->insertYourself( currSList );

	/* stop tracking range */
	XtRemoveEventHandler( infoWidgets[DA], ButtonMotionMask, FALSE,
			(XtEventHandler)dragRangeEH, this );

	/* force redraw */
	XClearArea( dpy, wndw, 0, 0, 0, 0, True );
}





/*
 * _startDraggingPoint()
 *
 * Initiates a drag on a NumberLine point.
 */
void
NumberLine::_startDraggingPoint( void )
{
	/* draw initial arrow */
	_drawCircle( ptrVal, True );

	/* add an event handler to track the drag */
	XtAddEventHandler( infoWidgets[DA], ButtonMotionMask, FALSE,
				(XtEventHandler)dragPointEH, this );
}


/*
 * _startDraggingRange()
 *
 * begins a rubber-band type drag of a range
 */
void
NumberLine::_startDraggingRange( void )
{

	/* set initial end of the range */
	rangeEnd = ptrVal;
	rangeEndSet = True;

	/* draw initial "range" */
	_drawRange( ptrVal, ptrVal );

	/*add event handler to track drag */
	XtAddEventHandler( infoWidgets[DA], ButtonMotionMask, FALSE,
				(XtEventHandler)dragRangeEH, this );
}




/*
 * _startFocusEnd()
 *
 * start tracking a drag of the end of the focus range.
 */
void
NumberLine::_startFocusEnd()
{
	/* save starting values of focus range */
	oldMinFocValue = minFocValue;
	oldMaxFocValue = maxFocValue;

	/* which end are we dragging? */
	focDir = ( ptrVal == minFocValue ? RIGHT : LEFT );

	/* add event handler to track the pointer */
	XtAddEventHandler( infoWidgets[DA], ButtonMotionMask, FALSE, 
			(XtEventHandler)dragFocusEndEH, this );

}





/*
 * _zoom2x()
 *
 * Zoom the visible NumberLine range by a factor of 2.
 */
void
NumberLine::_zoom2x( void )
{
	int range = maxVisValue - minVisValue;


	switch( range ) {
		case 0:
		case 1: 	return;		// do nothing for these cases

		case 2:		
		case 3:		maxVisValue--;
				break;

		default:	minVisValue += ( range / 4 );
				maxVisValue -= ( range / 4 );
				break;
	}
	
	/* compute new NumberLine interval between ticks */
	interval = _goodInterval( minVisValue, maxVisValue );

	/* reset slider values */
	XtVaSetValues( infoWidgets[SBAR],
		XmNvalue,		minVisValue,
		XmNsliderSize,		maxVisValue - minVisValue,
		XmNincrement,		interval,
		XmNpageIncrement,	maxVisValue - minVisValue,
		NULL );

	/* force a redraw of the numberline */
	XClearArea( dpy, wndw, 0, 0, 0, 0, True );
}




/* static */
void
NumberLine::clearAllCB( Widget /* w */, NumberLine *nl, XtPointer /* cd */ )
{
	nl->_clearAllSelections();
}




/*
 * createGC()
 *
 * Creates the graphics contexts used by the NumberLine.
 * call this only after the widget hierarchy has been realized, so
 * that infoWidgets[DA] already has a window to get.
 */
void
NumberLine::createGC( void )
{
	XGCValues gcv;		// for creating DrawingArea GC



	/* initialize Window and Display members */
	dpy = XtDisplay( infoWidgets[DA] );
	wndw = XtWindow( infoWidgets[DA] );

	/* create graphics context for drawing in the DA */
	XtVaGetValues( infoWidgets[DA],
			XmNforeground, &gcv.foreground,
			XmNbackground, &gcv.background,
			NULL );

	gcv.function = GXcopy;
	gcv.font = nlFont->fid;
	gcv.cap_style = CapRound;
	gc = XCreateGC( dpy, wndw,
			GCFont | GCFunction | GCCapStyle |
				GCForeground | GCBackground,
			&gcv );


}



/*
 * setSelection
 *
 * sets the NumberLine selection to a copy of the indicated list.
 */
void
NumberLine::setSelection( Selection* orig )
{
	Selection* dup = NULL;

	/* deallocate current list if we have one */
	delete currSList;

	/* duplicate the original list */
	if( orig ) {
		dup = orig->copy();
	}
	currSList = dup;
}



/*
 * getSelection
 *
 * returns a copy of the NumberLine selection.
 */
Selection*
NumberLine::getSelection( void )
{
	return ( currSList ? currSList->copy() : NULL );
}



/* static */
void
NumberLine::dragFocusEndEH( Widget /* w */, NumberLine *nl, XEvent *event )
{
	nl->_dragFocusEnd( event );
}



/* static */
void
NumberLine::dragPointEH( Widget /* w */, NumberLine *nl, XEvent *event )
{
	nl->_dragPoint( event );
}



/* static */
void
NumberLine::dragRangeEH( Widget /* w */, NumberLine *nl, XEvent *event )
{
	nl->_dragRange( event );
}



/* static */
void
NumberLine::expand2xCB( Widget /* w */, NumberLine *nl, XtPointer /* cd */ )
{
	nl->_expand2x();
}



/* static */
void
NumberLine::expandCompleteCB( Widget /* w */, NumberLine *nl, 
							XtPointer /* cd */ )
{
	nl->_expandCompletely();
}


/* static */
void
NumberLine::exposeCB( Widget /* w */, NumberLine *nl, 
					XmDrawingAreaCallbackStruct *cb )
{
	nl->_handleExpose( cb );
}



/* static */
void
NumberLine::handleMouseCB( Widget /* w */, NumberLine *nl,
					XmDrawingAreaCallbackStruct *cb )
{
	nl->_handleMouse( cb );
}



/* static */
void
NumberLine::oneDownCB( Widget /* w */, NumberLine *nl, XtPointer /* cd */ )
{
	nl->_handleOneDown();
}



/* static */
void
NumberLine::oneUpCB( Widget /* w */, NumberLine *nl, XtPointer /* cd */ )
{
	nl->_handleOneUp();
}




/* static */
void
NumberLine::pageDownCB( Widget /* w */, NumberLine *nl, XtPointer /* cd */ )
{
	nl->_handlePageDown();
}


/* static */
void
NumberLine::pageUpCB( Widget /* w */, NumberLine *nl, XtPointer /* cd */ )
{
	nl->_handlePageUp();
}



/* static */
void
NumberLine::pointModeCB( Widget /* w */, NumberLine *nl, XtPointer /* cd */ )
{
	nl->_switchPointMode();
}



/* static */
void
NumberLine::refocusCB( Widget /* w */, NumberLine *nl, XtPointer /* cd */ )
{
	nl->_refocus();
}



/* static */
void
NumberLine::resizeCB( Widget /* w */, NumberLine *nl, XtPointer /* cd */ )
{
	nl->_handleResize();
}



/* static */
void
NumberLine::selectModeCB( Widget /* w */, NumberLine *nl, XtPointer /* cd */ )
{
	nl->_switchSelectMode();
}



/* static */
void
NumberLine::toBottomCB( Widget /* w */, NumberLine *nl, XtPointer /* cd */ )
{
	nl->_handleToBottom();
}



/* static */
void
NumberLine::toTopCB( Widget /* w */, NumberLine *nl, XtPointer /* cd */ )
{
	nl->_handleToTop();
}


/* static */
void
NumberLine::valueChangedCB( Widget /* w */, NumberLine *nl, 
				XmScrollBarCallbackStruct *cd )
{
	nl->_handleValueChanged( cd );
}


/* static */
void
NumberLine::zoom2xCB( Widget /* w */, NumberLine *nl, XtPointer /* cd */ )
{
	nl->_zoom2x();
}






