/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Author: Keith A. Shields (shields@cs.uiuc.edu)
 *		        Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * PiechartFU.cc - A polymorophic functional unit for a dynamic pie chart
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/PiechartFU.C,v 1.32 1994/03/15 16:41:49 aydt Exp $
 */
#include "PiechartFU.h"
#include "PiechartFormWrapper.h"

#include "FUParams.h"
#include "ParamConfig.h"
#include "ParamDisplay.h"
#include "InputPort.h"
#include "SystemErrors.h"

#define Normalize(x)	( (((double) x) - minValue) / (maxValue - minValue) )
#define Round(x)	((int)(x + .5))

PiechartFU::PiechartFU()
{
        _setClassName( MY_CLASS );
	inputPort = NULL;
	dialog = NULL;
	piechartForm = NULL;
}

PiechartFU::~PiechartFU()
{
	delete inputPort;
	delete piechartForm;
	delete dialog;
}

void 			
PiechartFU::configure()		/* virtual */
{

#ifndef XtNminValue
#define XtNminValue "minValue"
#define XtNmaxValue "maxValue"
#endif

#ifndef XtNentries
#define XtNentries "numEntries"
#endif

        /*
         * The expected minimum and maximum data values are selected by
         * the user, and these are used to scale the actual value seen
         * by the Piechart FU into the range of integer values accepted
         * by the Piechart widgets.  We never change the widget's minimum
         * and maximum values.
         */

	double fuMin, fuMax;
        int  fuEntries;
	const char *labelBottom, *labelLeft;

	if ( ! isConfigured ) {
	    // Obtain the Piechart attributes from the Form child
            int wMin, wMax;
	    piechartForm->getPiechartAttr( XtNminValue, (XtArgVal) &wMin );
            piechartForm->getPiechartAttr( XtNmaxValue, (XtArgVal) &wMax );
	    piechartForm->getPiechartAttr( XtNentries, (XtArgVal) &fuEntries );

	    fuMin = wMin;
	    fuMax = wMax;
	    valueOffset = wMin;
	    valueRange = wMax - wMin;
	    isConfigured = TRUE_;
	} else {
            fuMin = minValue;
            fuMax = maxValue;
            fuEntries = numEntries;
	}

	// --- Configuration Phase
        piechartForm->getPiechartLabel( L_BOTTOM_HORIZ, &labelBottom );
        piechartForm->getPiechartLabel( L_LEFT_VERT, &labelLeft );

	FUParams params;
	params.addTextParam( "Number of pie chart entries", 
			     BaseFUParamEntry::Integer, fuEntries );
 	params.addTextParam( "Minimum value", BaseFUParamEntry::Real, fuMin );
        params.addTextParam( "Maximum value", BaseFUParamEntry::Real, fuMax );
        params.addTextParam( "Vertical axis label",
			     BaseFUParamEntry::Str, labelLeft);
        params.addTextParam( "Horizontal axis label",
			     BaseFUParamEntry::Str, labelBottom );

	CString configPanelName = "Piechart FU Config: " + getName();
	ParamConfig pc( Pablo::TopLevel(), params, configPanelName );
	pc.run();

	BaseFUParamEntry& sliceEntry =
		params.getEntry( "Number of pie chart entries" );
 	BaseFUParamEntry& minEntry = params.getEntry( "Minimum value" );
        BaseFUParamEntry& maxEntry = params.getEntry( "Maximum value" );
        BaseFUParamEntry &vertLabelEntry =
		              params.getEntry( "Vertical axis label" );
        BaseFUParamEntry &horizLabelEntry = 
		              params.getEntry( "Horizontal axis label" );

	if ( sliceEntry.valueIsValid() ) {
	    fuEntries = sliceEntry.getValue().getInteger();
	} else {
	    warning( "Number of entries was not valid\n" );
	}
	if ( minEntry.valueIsValid() ) {
            fuMin = minEntry.getValue().getReal();
        } else {
            warning( "Minimum value was not valid\n" );
	}
        if ( maxEntry.valueIsValid() ) {
            fuMax = maxEntry.getValue().getReal();
        } else {
            warning( "Maximum value was not valid\n" );
	}

	if ( fuMin >= fuMax ) {
	    warning( "Minimum >= Maximum.  Resetting Maximum to Minimum + 1" );
	    fuMax = fuMin + 1;
	}

	// ------ Set Data to Configured Values for FU and Widget ------
        numEntries = fuEntries;
	minValue = fuMin;
	maxValue = fuMax;

        piechartForm->setPiechartLabel( fuMin, fuMax );
	piechartForm->setPiechartLabel( L_LEFT_VERT, 
	 		      vertLabelEntry.getValue().getString() );
        piechartForm->setPiechartLabel( L_BOTTOM_HORIZ,
	                     horizLabelEntry.getValue().getString() );
	piechartForm->setPiechartAttr( XtNentries, numEntries );
        piechartForm->setPerfWidgetColors();
}

FunctionalUnit *	/* virtual */
PiechartFU::copy()
{
	PiechartFU *copy = new PiechartFU();
	return copy;
}

void 
PiechartFU::fuCallback( Widget /* callbackWidget */, int pieSelected, 
			float piePercentage )
{
	if ( inputValue.isUndefined() ) {
	    return;
	}

        FUParams params;

	Value callbackDataValue;
	Array* arrayInputValue = (Array *)inputValue;
	const int *dimSizes = arrayInputValue->getDimSizes();

	if ( pieSelected < dimSizes[0] ) {
	    arrayInputValue->getCellValue( callbackDataValue, pieSelected );
 	    params.addDisplayParam( "Piechart Value", callbackDataValue );
        } else {
	    params.addDisplayParam( "Piechart Value", noDataMSG );
        }
        params.addDisplayParam( "Pie Fraction", piePercentage );
        params.addDisplayParam( "Pie Vector Index", pieSelected );

	CString callbackTitle = "Callback: " + getName();
	ParamDisplay *pd = new ParamDisplay( Pablo::TopLevel(), params,
				      	     callbackTitle );
}

void			/* virtual */ 
PiechartFU::init()
{
	/* Start by clearing up any 'leftovers' that will be around if
	 * init() isn't being called for the first time.  The goal is
	 * to start with a 'virgin' FU.
	 */
	if ( inputPort != NULL ) {
	    delete inputPort;
	}
	if ( dialog != NULL ) {
	    delete dialog;
	}
	if ( piechartForm != NULL ) {
	    delete piechartForm;
	}

	inputPort = new InputPort( "Input" );
	inputPort->addTraits( INTEGER, 1 );
	inputPort->addTraits( FLOAT, 1 );
	inputPort->addTraits( DOUBLE, 1 );
	_addInputPort( inputPort );

	dialog = _getTitledFormDialog( "FUDialog", getName() );
	piechartForm= new PiechartFormWrapper( dialog, this, NullArgs, 
							     "Contents");
	dialog->manage();
	XtVaSetValues( dialog->getWidget(), XmNdefaultPosition, False, NULL );

	isConfigured = FALSE_;
	inputValue = Value::NOVALUE;
}

Boolean_                	/* virtual */
PiechartFU::loadConfigFromFile( const CString& fileName )
{
        FILE *fp = fopen( fileName.getValue(), "r" );

        if ( fp == NULL ) {
            warning( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
        } else {
            int x, y, width, height;
            int labelSize;
	    char buf[LABEL_MAX_LEN];

            fscanf(fp, "%*[^\n]\n");

	    if ( fscanf( fp, "%lf %lf %d %d %d %d %d\n",
                           &minValue, &maxValue, &numEntries, 
                           &x, &y, &width, &height ) == 7 ) {
				
		dialog->unmanage();

                // ----- Comment Line; Horizontal Label Size and String
                labelSize = 0;
                fscanf( fp, "%*[^\n]\n" );
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
                if ( labelSize == 0 ) {
                    strcpy( buf, "\0" );
                }
                piechartForm->setPiechartLabel( L_BOTTOM_HORIZ, buf );

                // ----- Comment Line; Vertical Label Size and String
                labelSize = 0;
                fscanf( fp, "\n%*[^\n]\n" );
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
                if ( labelSize == 0 ) {
                    strcpy( buf, "\0" );
                }
                piechartForm->setPiechartLabel( L_LEFT_VERT, buf );

		int wMin, wMax;
		piechartForm->getPiechartAttr( XtNminValue, (XtArgVal) &wMin );
		piechartForm->getPiechartAttr( XtNmaxValue, (XtArgVal) &wMax );

		valueOffset = wMin;
		valueRange = wMax - wMin;

		// Set values in the widget
	        piechartForm->setPiechartAttr( XtNentries, numEntries);
		piechartForm->setPiechartLabel( minValue, maxValue );
        	piechartForm->setPerfWidgetColors();

		dialog->manage();
                piechartForm->setPerfWidgetPosition( x, y, width, height );

                isConfigured = TRUE_;
            } else {
                warning( "Unable to read configuration information from %s\n",
                          fileName.getValue() );
            }
            fclose( fp );
        }

	if ( ! isConfigured ) {
	    configure();
	}
        return isConfigured;
}

Boolean_ 
PiechartFU::ready()
{
	return isConfigured;
}

void 			/* virtual */
PiechartFU::run(  Boolean_& errorFlag )
{
	Assert(inputPort->valueAvailable() );

       /* data values are scaled to fit the range of values allowed
           by the piechart widget - that range is determined by the
           widget's max/min values in the resource file. A data value
           out of range is trapped, an error message is produced, and the
           value is replaced by the maximum or minimum allowed value
           (depending on which way the value is out of range) - this
           is done since the widget would report out of range data in
           terms of scaled data and widget boundaries, which may not 
           correspond to user configured boundaries */

	inputValue = inputPort->getValue();
        if (inputValue.getTraits().getDimension() != 1) {
		warning("Piechart needs a vector as input");
		errorFlag = TRUE_;
		return;
	}
	Array *aInputData = (Array *) inputValue;
	const int *aDimSizes = aInputData->getDimSizes();    

        // Set number of piechart entries
	int nx = aDimSizes[0];
        // Crete vector and fill with scaled data values
	int *piechartValues = new int [nx];
        int x, val;
        double nextval;

	for (x = 0; x < nx; x++) {
            nextval = (double) aInputData->getCellValue( x );
            double nextvalScaled = warnings.fuDataRangeWarningCheck( this,
			                    nextval, minValue, maxValue );
            val = Round( Normalize( nextvalScaled ) * valueRange );
	    val += valueOffset;   // In case widget range does not start at 0
	    piechartValues[x] = val;
	}
	piechartForm->setDisplayValues( nx, piechartValues);
	delete[] piechartValues;
}

Boolean_		/* virtual */
PiechartFU::saveConfigToFile( const CString& fileName ) const
{
        Boolean_ result;
        FILE *fp = fopen( fileName.getValue(), "w" );

        if ( fp == NULL ) {
            error( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
            result = FAILURE_;
        } else {
            int   x, y, width, height;
	    const char *label;

            piechartForm->getPerfWidgetPosition( &x, &y, &width, &height );

            fprintf( fp, "# Min Max Slices X Y Width Height\n" );
	    fprintf( fp, "%lf %lf %d %d %d %d %d\n",
                       minValue, maxValue, numEntries, x, y, width, height );

            piechartForm->getPiechartLabel( L_BOTTOM_HORIZ, &label );
            fprintf( fp, "# Horizontal Label\n" );
	    fprintf( fp, "%d %s\n", strlen(label), label );

            piechartForm->getPiechartLabel( L_LEFT_VERT, &label );
            fprintf( fp, "# Vertical Label\n" );
            fprintf( fp, "%d %s\n", strlen(label), label );

            fclose( fp );
            result = SUCCESS_;
        }
        return result;
}

/*
 *      Initialize the static data.   Only executed once.
 */
const char *const PiechartFU::MY_CLASS = "PiechartFU";

