/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Author: Keith A. Shields (shields@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * PolarPlotFU.cc - A polymorophic functional unit for a dynamic polar plot
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/PolarPlotFU.C,v 1.22 1994/03/15 16:41:53 aydt Exp $
 */

#include "PolarPlotFU.h"
#include "PolarWrapper.h"

#include "FUParams.h"
#include "ParamConfig.h"
#include "ParamDisplay.h"
#include "InputPort.h"
#include "SystemErrors.h"

#define Round(x)  (x >= 0.0 ? (int)(x + .5) : (int)(x - .5))

PolarPlotFU::PolarPlotFU()
{
	_setClassName( MY_CLASS );
	input = NULL;
	dialog = NULL;
	polar = NULL;
}

PolarPlotFU::~PolarPlotFU()
{
	delete input;
	delete polar;
	delete dialog;
}

void 			/* virtual */
PolarPlotFU::configure()
{
#ifndef XtNmagnitude
#define XtNmagnitude "magnitude"
#endif
	int fuMagnitude;
	const char *labelBottom, *labelLeft;

        if ( ! isConfigured ) {
            polar->getPolarAttr( XtNmagnitude, (XtArgVal) &fuMagnitude );
	    isConfigured = TRUE_;
	} else {
	    fuMagnitude = magnitude;
	}
	// --- CONFIGURATION PARAMETERS --------------------------------------
        polar->getPolarLabel( L_BOTTOM_HORIZ, &labelBottom );
        polar->getPolarLabel( L_LEFT_VERT, &labelLeft );

	FUParams params;
 	params.addTextParam( "Maximum magnitude", BaseFUParamEntry::Integer,
                              fuMagnitude );
	params.addTextParam( "Vertical axis label",
			     BaseFUParamEntry::Str, labelLeft);
        params.addTextParam( "Horizontal axis label",
			     BaseFUParamEntry::Str, labelBottom );

	CString configPanelName = "Polar Plot Config: " + getName() ;
	ParamConfig pc( Pablo::TopLevel(), params, configPanelName );
	pc.run();

	BaseFUParamEntry &magnitudeEntry = 
				params.getEntry( "Maximum magnitude" );
        BaseFUParamEntry &vertLabelEntry =
		              params.getEntry( "Vertical axis label" );
        BaseFUParamEntry &horizLabelEntry = 
		              params.getEntry( "Horizontal axis label" );

	// === PARAMETER ERROR-CHECKING =======================================
	if ( magnitudeEntry.valueIsValid() ) {
            fuMagnitude = magnitudeEntry.getValue().getInteger();
        } else {
            warning( "Magnitude not valid.\n" );
	}

	magnitude = fuMagnitude;
	polar->setPolarLabel( 0.0, (double)magnitude );
        polar->setPolarLabel( L_BOTTOM_HORIZ, 
			      horizLabelEntry.getValue().getString() );
	polar->setPolarLabel( L_LEFT_VERT, 
	 		      vertLabelEntry.getValue().getString() );
	polar->setPolarAttr( XtNmagnitude, magnitude );
}


FunctionalUnit *	/* virtual */
PolarPlotFU::copy()
{
	PolarPlotFU *copy = new PolarPlotFU();
	return copy;
}

void
PolarPlotFU::fuCallback( int nearestXValue, int nearestYValue )
{
         if ( inputValue.isUndefined() ) {
            return;
	 }
	 FUParams    params;
	 params.addDisplayParam( "Current Polar X Data Index", nearestXValue );
	 params.addDisplayParam( "Current Polar Y Data Index", nearestYValue );

	 CString callbackTitle = "Callback: " + getName();
	 ParamDisplay *pd = new ParamDisplay( Pablo::TopLevel(), params,
					      callbackTitle );

}


void 			/* virtual */
PolarPlotFU::init()	
{
	if ( input != NULL ) {
	    delete input;
	}
	if ( polar != NULL ) {
	    delete polar;
	}
	if ( dialog != NULL ) {
	    delete dialog;
	}

	input = new InputPort( "Input" );
	input->addTraits( INTEGER, 2 );
	input->addTraits( FLOAT, 2 );
	input->addTraits( DOUBLE, 2 );
	_addInputPort( input );

	dialog = _getTitledFormDialog( "FUDialog", getName() );
	polar = new PolarWrapper( dialog, this, NullArgs, "Contents" );
	dialog->manage();
	XtVaSetValues( dialog->getWidget(), XmNdefaultPosition, False, NULL );

	isConfigured = FALSE_;
}


 
Boolean_ 		/* virtual */
PolarPlotFU::loadConfigFromFile( const CString& fileName )
{
        FILE *fp = fopen( fileName.getValue(), "r" );

        if ( fp == NULL ) {
            warning( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
        } else {
	    int x, y, width, height;
	    int labelSize;
	    char buf[LABEL_MAX_LEN];

	    fscanf( fp, "%*[^\n]\n" );

	    if ( fscanf( fp, "%d %d %d %d %d\n", 
			   &magnitude, &x, &y, &width, &height ) != 5 ) {
            	warning( "Unable to read configuration information from %s\n", 
			  fileName.getValue() );
            } else {
		dialog->unmanage();
		// ----- Comment Line; Horizontal Label Size and String
		labelSize = 0;
                fscanf( fp, "%*[^\n]\n" );
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
		if ( labelSize == 0 ) {
		    strcpy( buf, "\0" );
		}
                polar->setPolarLabel( L_BOTTOM_HORIZ, buf );

		// ----- Comment Line; Vertical Label Size and String
		labelSize = 0;
                fscanf( fp, "\n%*[^\n]\n" );
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
		if ( labelSize == 0 ) {
		    strcpy( buf, "\0" );
		}
                polar->setPolarLabel( L_LEFT_VERT, buf );

		polar->setPolarLabel( 0.0, (double)magnitude );
        	polar->setPolarAttr( XtNmagnitude, magnitude);

		dialog->manage();
                polar->setPerfWidgetPosition( x, y, width, height );
			
		isConfigured = TRUE_;
	    }
	    fclose( fp );
        }

	if ( ! isConfigured ) {
	    configure();
	}
        return isConfigured;
}

Boolean_ 
PolarPlotFU::ready()
{
	return isConfigured;
}

void 			/* virtual */
PolarPlotFU::run(  Boolean_&  errorFlag )
{
	Assert( input->valueAvailable() );
	inputValue = input->getValue();

	const Value *valueP = input->getValueP();

	if ( inputValue.getTraits().isScalar() ) {
	    warning( "PolarPlot needs an array as input" );
	    errorFlag = TRUE_;
  	    return;
	} else if ( inputValue.getTraits().getDimension() != 2 ) {
	    warning("PolarPlot needs a 2-d array as input");
	    errorFlag = TRUE_;
	    return;
	}

	const Array *a = (Array *)inputValue;
	const int *dimSizes = a->getDimSizes();

	if ( dimSizes[1] != 2) {
	    warning( 
		  "PolarPlot only accepts points specified with (X,Y) pairs." );
	    errorFlag = TRUE_;
	    return;
	}

	int nPairs = dimSizes[0];
	int *xptr = new int [ nPairs ];
	int *yptr = new int [ nPairs ];

	for ( int p = 0; p < nPairs; p++ ) {
	   double xin = a->getCellValue(p,0);
	   double yin = a->getCellValue(p,1);
	   xptr[p] = Round( xin );
	   yptr[p] = Round( yin );
       	}

	polar->setDisplayValues( nPairs, xptr, yptr);

	delete[] xptr;
	delete[] yptr;	
}

Boolean_		/* virtual */
PolarPlotFU::saveConfigToFile( const CString& fileName ) const
{
	Boolean_ result;
        FILE *fp = fopen( fileName.getValue(), "w" );

        if ( fp == NULL ) {
            error( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
	    result = FAILURE_;
        } else {
	    int x, y, width, height;
	    const char *label;

	    polar->getPerfWidgetPosition( &x, &y, &width, &height );

	    fprintf( fp, "# Magnitude, X, Y, Width, Height\n" );
	    fprintf( fp, "%d %d %d %d %d\n", 
			  magnitude, x, y, width, height );

            polar->getPolarLabel( L_BOTTOM_HORIZ, &label );
            fprintf( fp, "# Horizontal Label\n" );
	    fprintf( fp, "%d %s\n", strlen( label ), label );

            polar->getPolarLabel( L_LEFT_VERT, &label );
            fprintf( fp, "# Vertical Label\n" );
	    fprintf( fp, "%d %s\n", strlen( label ), label );

	    fclose( fp );
	    result = SUCCESS_;
        }
        return result;
}

/*
 *      Initialize the static data.   Only executed once.
 */
const char *const PolarPlotFU::MY_CLASS = "PolarPlotFU";

