/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Authors: Keith A. Shields (shields@cs.uiuc.edu)
 *                       Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *      Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * ProfileFU.C - A functional unit to compute profiles from procedure
 *		 exit trace data that includes procedure lifetimes.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/ProfileFU.C,v 1.11 1994/03/15 19:48:24 aydt Exp $
 */

#include <string.h>
#include <Xm/Text.h>

#include "ProfileFU.h"

#include "FUParams.h"
#include "ParamConfig.h"
#include "ParamDisplay.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "PiechartFormWrapper.h"
#include "FormWidget.h"
#include "FrameWidget.h"
#include "LabelWidget.h"
#include "RowColumnWidget.h"
#include "ScrolledWindowWidget.h"
#include "TextWidget.h"
#include "SystemErrors.h"

extern "C" {
#include "Piechart.h"
}

ProfileFU::ProfileFU()
{
	_setClassName( MY_CLASS );
	isConfigured = FALSE_;

        eventID = NULL;
        PENumber = NULL;
        elapsedTime = NULL;
        currentTime = NULL;

	outEventIDs = NULL;
        outPENumber = NULL;
        outCurrentTime = NULL;
        callCount = NULL;
        percentCount = NULL;
        totalTime = NULL;
        percentTime = NULL;

	PEInfoList = NULL;
	PEsToDisplay = NULL;

	dialog = NULL;
	scrollBox = NULL;
	rowColumn = NULL;
	pieCountForm = NULL;
	pieTimeForm = NULL;
 	profileText = NULL;
        profileForm = NULL;
	profileLabel = NULL;
	profileFrame = NULL;

        outputOption = -1;
        ProcessorOption = -1;
        displayFrequency = 1;
        PEInfoListSize = 0;
        NumPEsToDisplay = 0;
        MaxPESeenSoFar = -1;
}

ProfileFU::~ProfileFU()
{
	delete eventID;
	delete PENumber;
	delete elapsedTime;
	delete currentTime;

	delete outEventIDs;
	delete outPENumber;
	delete outCurrentTime;
	delete callCount;
	delete percentCount;
   	delete totalTime;
	delete percentTime;

	delete PEInfoList;
	delete PEsToDisplay;

	for (int i = 0; i < NumPEsToDisplay; i++) {
	   delete pieTimeForm[i];
	   delete pieCountForm[i];
	   delete profileText[i];
	   delete profileForm[i];
	   delete profileLabel[i];
	   delete profileFrame[i];
	}

	delete pieTimeForm;
	delete pieCountForm;
	delete profileText;
	delete profileForm;
	delete profileLabel;
	delete profileFrame;

	delete rowColumn;
	delete scrollBox;
	delete dialog;
}

void				/* virtual */
ProfileFU::configure()
{
	int		PEOption;

	isConfigured = TRUE_;

	FUParams params;

        params.addTextParam( "Display/Output Frequency",
             			BaseFUParamEntry::Integer, 1 );

        params.addRadioButtonsParam( "Select Processors",
                                     BaseFUParamEntry::Integer, 0,
                                     PEOptionList );

	ParamConfig pc( Pablo::TopLevel(), params, getName() );

	BaseFUParamEntry& processorEntry =
                params.getEntry( "Select Processors" );

        BaseFUParamEntry& displayEntry=
                params.getEntry( "Display/Output Frequency" );

        if ( ProcessorOption != -1 ) {
           processorEntry.setValue( ProcessorOption );
        }

	if ( displayFrequency != -1 ) {
	   displayEntry.setValue ( displayFrequency );
	}

	pc.run();

        if ( processorEntry.valueIsValid() ) {
           PEOption = processorEntry.getValue().getInteger();
        } else {
           warning( "Invalid processor option in Profile" );
	   PEOption = ONE_PE;
        }

	ProcessorOption = PEOption;

        if ( displayEntry.valueIsValid() ) {
           displayFrequency = displayEntry.getValue().getInteger();
        } else {
           warning( "Invalid display frequency option in Profile" );
        }

	FUParams PEparams;
	PEparams.addTextParam( "Processor Number",
			      BaseFUParamEntry::Integer, 0 );
	   
	ParamConfig pcPE( Pablo::TopLevel(), PEparams, getName() );
	BaseFUParamEntry& PEEntry = PEparams.getEntry( "Processor Number" );

	if ( ProcessorOption  == ALL_PEs ) {

	  if (PEsToDisplay != NULL) {
	    delete PEsToDisplay;
	    PEsToDisplay = NULL;
	  }

	  NumPEsToDisplay = PEInfoListSize;
	  if (PEInfoListSize > 0) {
	    PEsToDisplay = new int[PEInfoListSize];
	    for (int i=0; i< PEInfoListSize; i++) {
	      PEsToDisplay[i] = i;
	    }
	  }
	}

	if ( ProcessorOption == ONE_PE ) {
	   NumPEsToDisplay = 1;
	   PEsToDisplay = new int[1];

	   pcPE.run();
	   
           if ( PEEntry.valueIsValid() ) {
	      PEsToDisplay[0] = PEEntry.getValue().getInteger();
           } else {
              warning( "Invalid processor number in Profile" );
           }
	 }

	if ( ProcessorOption == SELECTED_PEs ) {
	  FUParams PECntparams;

	  PECntparams.addTextParam( "Number of Processors",
				   BaseFUParamEntry::Integer, NumPEsToDisplay );

	  ParamConfig pcPECnt( Pablo::TopLevel(), PECntparams, getName() );

	  BaseFUParamEntry& PECntEntry =
	    PECntparams.getEntry( "Number of Processors" );

	  pcPECnt.run();

	  if ( PECntEntry.valueIsValid() ) {
	    NumPEsToDisplay = PECntEntry.getValue().getInteger();
	  } else {
	    warning( "Invalid number of processors in Profile" );
	    NumPEsToDisplay = 1;
	  }

	  PEsToDisplay = new int[NumPEsToDisplay];

	  for (int i = 0; i < NumPEsToDisplay; i++) { 
	    PEEntry.setValue( BaseFUParamEntry::Integer, i );
	  
	    pcPE.run();

	    if ( PEEntry.valueIsValid() ) {
	      PEsToDisplay[i] = PEEntry.getValue().getInteger();
	    } else {
	      warning( "Invalid processor number in Profile" );
	    }
	  }
	}

	if (dialog == NULL) {
	  _createDialog();
	}

	if ( ( outputOption == DISPLAY ) && ( NumPEsToDisplay > 0 ) ) {
	  configureDisplay();
	   
	  if (PEInfoListSize > 0) {
	    for (int i = 0; i < NumPEsToDisplay; i++) {
	      _updateDisplay( PEsToDisplay[i] );
	    }
	  }
	}
}


void
ProfileFU::configureDisplay()
{
	int	i;
	char	dialogLabel[1024];
	CString tmpString;
	double	minValue;
	double	maxValue;
	int     numChildren;

	if (rowColumn != NULL) {

	  XtVaGetValues( rowColumn->getWidget(),
			 XmNnumChildren, &numChildren,
			 NULL );

	  for (i=0; i<numChildren; i++) {
	    delete profileLabel[i];
	    delete pieCountForm[i];
	    delete pieTimeForm[i];
	    delete profileText[i];
	    delete profileForm[i];
	    delete profileFrame[i];
	  }

	  delete rowColumn;
	  rowColumn = NULL;
	}

	rowColumn = new RowColumnWidget( scrollBox, NullArgs, "rowColumn" );
	
	XtVaSetValues( rowColumn->getWidget(),
		       XmNpacking, XmPACK_COLUMN,
  		       XmNorientation, XmHORIZONTAL,
		       NULL );
	
	profileFrame = new FrameWidget* [NumPEsToDisplay];
	profileLabel = new LabelWidget* [NumPEsToDisplay];
	pieCountForm = new PiechartFormWrapper* [NumPEsToDisplay];
	pieTimeForm = new PiechartFormWrapper* [NumPEsToDisplay];
	profileText = new TextWidget* [NumPEsToDisplay];
	profileForm = new FormWidget* [NumPEsToDisplay]; 

	XArgs textArgs;
	textArgs.addArg( XmNeditMode, XmMULTI_LINE_EDIT );
	textArgs.addArg( XmNrows, 6 );

	for (i = 0; i < NumPEsToDisplay; i++) {

	  profileFrame[i] = new FrameWidget( rowColumn, NullArgs, "PEFrame");
	  profileForm[i] = new FormWidget( profileFrame[i], NullArgs, 
					  "PEForm" );
	  
	  sprintf( dialogLabel, "Processor %d", PEsToDisplay[i] );
	  tmpString = dialogLabel;

	  profileLabel[i] = new LabelWidget( profileForm[i], NullArgs,
					    "profileLabel", tmpString );

	  XtVaSetValues( profileLabel[i]->getWidget(),
			XmNtopAttachment, XmATTACH_FORM,
			XmNleftAttachment, XmATTACH_FORM,
			XmNrightAttachment, XmATTACH_FORM,
			NULL );

	  profileLabel[i]->manage();

	  profileText[i] = new TextWidget( profileForm[i], textArgs,
					  "profileText", TRUE_, TRUE_ );

	  /* 
	   * We don't need the horizontal scroll bar that is provided with
	   * the ScrolledTextWidget because our line length is fixed.  So,
	   * unmanage it.
	   */
	  Widget hsb;
	  XtVaGetValues( XtParent( profileText[i]->getWidget() ),
			 XmNhorizontalScrollBar, &hsb,
			 NULL );
          XtUnmanageChild( hsb );

	  XtVaSetValues( XtParent( profileText[i]->getWidget() ),
			XmNtopAttachment, XmATTACH_WIDGET,
			XmNtopWidget, profileLabel[i]->getWidget(),
			XmNleftAttachment, XmATTACH_FORM,
			XmNrightAttachment, XmATTACH_FORM,
			NULL );

	  headerString[0] = '\0';
	  strcat( headerString, "  Event     % Count        Count      % Time          Time\n" );
	  XmTextSetString( profileText[i]->getWidget(), headerString );
	  headerSize = strlen( headerString );
	  
	  profileText[i]->manage();

	  pieCountForm[i] = new PiechartFormWrapper( profileForm[i],
						     (PiechartFU *)this,
						     NullArgs, 
						     "CountContents" );
	  
	  XtVaSetValues( pieCountForm[i]->getWidget(),
			XmNtopAttachment, XmATTACH_WIDGET,
			XmNtopWidget, XtParent(profileText[i]->getWidget()),
			XmNtopOffset, 10,
			XmNleftAttachment, XmATTACH_FORM,
			NULL );

//	  pieCountForm[i]->returnFormOnCallback(TRUE_);
	  pieCountForm[i]->manage();

	  pieTimeForm[i] = new PiechartFormWrapper( profileForm[i],
						   (PiechartFU *)this,
						   NullArgs,
						   "TimeContents");

	  XtVaSetValues( pieTimeForm[i]->getWidget(),
			XmNtopAttachment, XmATTACH_WIDGET,
			XmNtopWidget, XtParent(profileText[i]->getWidget()),
			XmNtopOffset, 10,
			XmNleftAttachment, XmATTACH_WIDGET,
			XmNleftWidget, pieCountForm[i]->getWidget(), 
			NULL );

//	  pieTimeForm[i]->returnFormOnCallback(TRUE_);
	  pieTimeForm[i]->manage();

	  minValue = 0.0;
	  maxValue = 100.0;

	  pieCountForm[i]->setPiechartLabel( minValue, maxValue );
	  pieTimeForm[i]->setPiechartLabel( minValue, maxValue );
	
	  pieCountForm[i]->setPiechartLabel(
			       L_BOTTOM_HORIZ, "Call Distribution" );

	  pieTimeForm[i]->setPiechartLabel(
         		       L_BOTTOM_HORIZ, "Time Distribution" );

	  pieCountForm[i]->setPerfWidgetColors();
	  pieTimeForm[i]->setPerfWidgetColors( );
	  profileForm[i]->manage();
	  profileFrame[i]->manage();

	}
	rowColumn->manage();
}


void
ProfileFU::configureOperation()
{
	/*
	 * We only want to allow user to select if this is the
	 * initial configuration - else just display the choice!
	 */
	FUParams params;
	Boolean_ readOnly = FALSE_;

	if ( outputOption == -1 ) {
 	    params.addRadioButtonsParam( "Display/Output Option",
				     	  BaseFUParamEntry::Integer, 0,
				     	  OutputOptionList );
	} else {
	    params.addDisplayParam( "Display/Output Choice", 
			OutputOptionList.getElement( outputOption ) );
	    readOnly = TRUE_;
	  }    
        BaseFUParamEntry& outputEntry =
	  params.getEntry( "Display/Output Option" );

	ParamConfig pc( Pablo::TopLevel(), params, getName(), readOnly );
	pc.run();

	if ( outputOption == -1 ) {
	   if ( outputEntry.valueIsValid() ) {
	      outputOption = outputEntry.getValue().getInteger();
	   } else {
	      warning( "Invalid display/output option in Profile" );
	      outputOption = DISPLAY;
	   }
	 }

	if ( outputOption == OUTPUT ) {
           outCurrentTime = new OutputPort( "Current Clock" );
	   outPENumber = new OutputPort( "PE Number" );
	   outEventIDs = new OutputPort( "Event Identifiers" );

           callCount = new OutputPort( "Number of Calls" );
           percentCount = new OutputPort( "Percent Calls" );
           totalTime = new OutputPort( "Total Time" );
           percentTime = new OutputPort( "Percent Time" );

	   _addOutputPort( outPENumber );
	   _addOutputPort( outCurrentTime );
	   _addOutputPort( outEventIDs );
	   _addOutputPort( callCount );
	   _addOutputPort( percentCount );
	   _addOutputPort( totalTime );
	   _addOutputPort( percentTime );
	}

}


FunctionalUnit *		/* virtual */
ProfileFU::copy()
{
	ProfileFU *copy = new ProfileFU();
	return copy;
}


void 
ProfileFU::fuCallback( Widget callbackWidget,
		       int pieSelected, float piePercentage ) 
{
        int selectedPE;
	Boolean countSelected = FALSE_;
	char buf[256];
	Widget pieForm = XtParent(callbackWidget);

	for (int i=0; i<NumPEsToDisplay; i++) {
	  if (pieForm == pieCountForm[i]->getWidget()) {
	    selectedPE = PEsToDisplay[i];
	    countSelected = TRUE_;
	  }

	  if (pieForm == pieTimeForm[i]->getWidget()) {
	    selectedPE = PEsToDisplay[i];
	    countSelected = FALSE_;
	  }
	}

	char *text, *currentPtr, lineOfText[256];
	int  lineEID, numRead, totalCharsRead, event;

	int displayIdx = _lookupDisplayIdx(selectedPE);

	text = XmTextGetString(profileText[displayIdx]->getWidget());

	if (strlen(text) > headerSize) {
	  event = PEInfoList[selectedPE].eventInfoList[pieSelected].EID;
	  currentPtr = text + headerSize; // skip header line
	  totalCharsRead = headerSize+1;

	  do {
	    sscanf(currentPtr, "%[^\n]\n", lineOfText);
	    numRead = strlen(lineOfText);
	    totalCharsRead += numRead+1;
	    lineOfText[numRead+1] = '\0';
	    sscanf(lineOfText, "%d", &lineEID);
	    currentPtr += numRead+1;

	  } while (lineEID != event);

	  XtFree(text);
	  int startHiLightPos = totalCharsRead-strlen(lineOfText);
	  int endHiLightPos = totalCharsRead-1;

	  XmTextSetHighlight(profileText[displayIdx]->getWidget(), 
			     startHiLightPos,
			     endHiLightPos,
			     XmHIGHLIGHT_SELECTED);

	  FUParams params;

	  if (countSelected) {
	    sprintf(buf, "Count Callback for Processor %d", selectedPE);
	  } else {
	    sprintf(buf, "Time Callback for Processor %d", selectedPE);
	  }

	  params.addDisplayParam("Event", event );
	  params.addDisplayParam("Percentage", piePercentage*100.0 );

	  CString callbackTitle = buf;
	  ParamDisplay *pd = new ParamDisplay( Pablo::TopLevel(), params,
					       callbackTitle );


	  XmTextSetHighlight(profileText[displayIdx]->getWidget(), 
			     startHiLightPos,
			     endHiLightPos,
			     XmHIGHLIGHT_NORMAL);
	}
}


void 				/* virtual */
ProfileFU::init()
{
	/*
	 * Start by clearing up any leftovers that will be around if 
	 * init() isn't being called for the first time.  The goal is to
	 * start with a 'virgin' FU.
	 */

	if ( eventID != NULL ) {
	    delete eventID;
	}
	if ( PENumber != NULL ) {
	   delete PENumber;
	}
        if ( elapsedTime != NULL ) {
           delete elapsedTime;
        }
	if ( currentTime != NULL ) {
	    delete currentTime;
	}
	if ( outEventIDs != NULL ) {
	   delete outEventIDs;
	}
        if ( outPENumber != NULL ) {
           delete PENumber;
        }
        if ( outCurrentTime != NULL ) {
            delete currentTime;
        }
	if ( callCount != NULL ) {
	    delete callCount;
	}
        if ( percentCount != NULL ) {
            delete percentCount;
        }
	if ( totalTime != NULL ) {
	    delete totalTime;
	}
	if ( percentTime != NULL ) {
	    delete percentTime;
	}

	if ( NumPEsToDisplay > 0 ) {
	   for (int i = 0; i < NumPEsToDisplay; i++) {
	      delete profileLabel[i];
	      delete pieTimeForm[i];
	      delete pieCountForm[i];
	      delete profileText[i];
	      delete profileForm[i];
	      delete profileFrame[i];
	   }

	   delete pieTimeForm;
	   delete pieCountForm;
	   delete profileForm;
	   delete profileLabel;
	   delete profileText;
	   delete profileFrame;
	}

	if ( scrollBox != NULL ) {
	   delete scrollBox;
	}

	if ( dialog != NULL ) {
	   delete dialog;
	}

	eventID = new InputPort( "Event ID" );
	eventID->addTraits( INTEGER, 0 );

        PENumber = new InputPort( "Processor Number" );
        PENumber->addTraits( INTEGER, 0 );

        elapsedTime = new InputPort( "Procedure Lifetime" );
        elapsedTime->addTraits( INTEGER, 0 );
        elapsedTime->addTraits( FLOAT, 0 );
        elapsedTime->addTraits( DOUBLE, 0 );

        currentTime = new InputPort( "Current Time" );
        currentTime->addTraits( INTEGER, 0 );
        currentTime->addTraits( FLOAT, 0 );
        currentTime->addTraits( DOUBLE, 0 );

	_addInputPort( PENumber );
	_addInputPort( eventID );
	_addInputPort( elapsedTime );
	_addInputPort( currentTime );

	OutputOptionList.reset();
	OutputOptionList.addElement( "Display" );
	OutputOptionList.addElement( "Output" );

	PEOptionList.reset();
	PEOptionList.addElement( "All Processors" );
	PEOptionList.addElement( "One Processor" );
	PEOptionList.addElement( "Selected Processors" );

	isConfigured = FALSE_;

	outputOption = -1;
	ProcessorOption = -1;
	MaxPESeenSoFar = -1;
	displayFrequency = 1;
	outputCount = 0;
}


Boolean_			/* virtual */
ProfileFU::inputTraitsValid()
{
	/* 
	 * All the output traits of the profile's output ports are
	 * known except for that of time.  It is set to the same as
	 * the traits of the input time.
	 */

        DataTraits dataT = currentTime->getTraits();

        DataTraits      resultTraits( dataT );
	DataTraits	scalarIntegerTraits( INTEGER, 0 );
        DataTraits 	vectorIntegerTraits( INTEGER, 1 );
	DataTraits	vectorDoubleTraits( DOUBLE, 1 );

	if ( outputOption == OUTPUT ) {
	   outCurrentTime->setTraits( resultTraits );

	   outEventIDs->setTraits( vectorIntegerTraits );
           totalTime->setTraits( vectorDoubleTraits );
	   outPENumber->setTraits( scalarIntegerTraits );
           callCount->setTraits( vectorIntegerTraits );
           percentCount->setTraits( vectorDoubleTraits );
           percentTime->setTraits( vectorDoubleTraits );
	}

	return TRUE_;
}
	
Boolean_			/* virtual */
ProfileFU::loadConfigFromFile( const CString& fileName )
{
	int	outOpt;
	int	PEOpt;
	int	freq;
	int	sz;
	int	tmp;
	char	aHeaderLine[1024];

	Boolean_	isReady = FALSE_;

        FILE *fp = fopen( fileName.getValue(), "r" );
        if ( fp == NULL ) {
            warning( "\nUnable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
        } else {
            fscanf(fp, "%[^\n]\n", aHeaderLine );
            if ( fscanf( fp, "%d %d %d %d\n", &outOpt, &PEOpt, &freq, &sz ) 
			 == 4 ) {
		outputOption = outOpt;
		ProcessorOption = PEOpt;
		displayFrequency = freq;
		if ( ProcessorOption != ALL_PEs ) {
		    NumPEsToDisplay = sz;
		} else {
		    NumPEsToDisplay = 0;
		    Assert( PEInfoListSize == 0 );
		}

		if ( dialog == NULL ) {
		  _createDialog();
		}

		if ( ( outputOption == DISPLAY ) && ( NumPEsToDisplay > 0 ) ) {
		  PEsToDisplay = new int[NumPEsToDisplay];
		  for (int i = 0; i < NumPEsToDisplay; i++) {
		      fscanf( fp, "%d", &tmp );
		      PEsToDisplay[i] = tmp;
		  }
		  configureDisplay();
	        }

	        isReady = TRUE_;
            } else {
                warning( "\nUnable to read configuration information from %s\n",
                          fileName.getValue() );
	         isReady = FALSE_;
            }
            fclose( fp );
        }

	if ( isReady == FALSE_ ) {	    // last chance to configure!
	    isConfigured = FALSE_;
	    configure();		    // sets isReady if successful
	    isReady = TRUE_;
	}

	return ( isReady );
}

Boolean_ 			/* virtual */
ProfileFU::ready()
{
	return ( isConfigured );
}


void 				/* virtual */
ProfileFU::run(  Boolean_& /* errorFlag */ )
{
	Boolean_ createOutput;
	int	i;

	Assert( eventID->valueAvailable() );
	Assert( PENumber->valueAvailable() );
	Assert( elapsedTime->valueAvailable() );
	Assert( currentTime->valueAvailable() );

	int PE = PENumber->getValue();
	int EID = eventID->getValue();
	double ET = (double) elapsedTime->getValue();
	double Now = (double) currentTime->getValue();

	Boolean alreadyDisplayingPE = _displayPE(PE);

	_processPE( PE );
	_processEvent( PE, EID );

	PEInfoList[PE].PESummaryInfo.count++;
	PEInfoList[PE].PESummaryInfo.totalTime += ET;

	int evtIdx = _eventLookup( PE, EID );

	PEInfoList[PE].eventInfoList[evtIdx].count++;
	PEInfoList[PE].eventInfoList[evtIdx].totalTime += ET;


	if (( ProcessorOption == ALL_PEs ) && ( outputOption == DISPLAY )) {
	  if (! alreadyDisplayingPE) {
	    configureDisplay();
	  }

	  for (i = 0; i < NumPEsToDisplay; i++) {
	    _updateDisplay( PEsToDisplay[i] );
	  }
	}

	createOutput = _displayPE(PE);
	outputCount++;

	if ( displayFrequency == 1 ) {
	   createOutput = CnvToBoolean_(TRUE_ & createOutput);
	} else {
	   if ( outputCount % displayFrequency != 0 ) {
	      createOutput = FALSE_;
	   }
	}

	if ( createOutput == TRUE_ ) {
           for (i = 0; i < PEInfoList[PE].PESummaryInfo.EIDCount; i++) {
              PEInfoList[PE].eventInfoList[i].countPercent = 
		(double) PEInfoList[PE].eventInfoList[i].count
		  / (double) PEInfoList[PE].PESummaryInfo.count;

              PEInfoList[PE].eventInfoList[i].timePercent = 
		(double) PEInfoList[PE].eventInfoList[i].totalTime
		  / (double) PEInfoList[PE].PESummaryInfo.totalTime;
	    }

	   if ( outputOption == OUTPUT ) {
              outPENumber->setValue( PENumber->getValue() );
              outCurrentTime->setValue( currentTime->getValue() );

	      int dimSizes[1];
	      dimSizes[0] = PEInfoList[PE].PESummaryInfo.EIDCount;

	      Value dataVector ( DOUBLE, 1 );
              Array *newVectorP = (Array *) dataVector;
              Value dataIntVector ( DOUBLE, 1 );
              Array *newIntVectorP = (Array *) dataIntVector;
	      Value EIDVector ( INTEGER, 1 );
	      Array *newEIDP = (Array *) EIDVector;

	      Value tmpIntegerValue( INTEGER, 0 );
	      Value tmpDoubleValue( DOUBLE, 0 );

	      newVectorP->setDimSizes( dimSizes );
	      newIntVectorP->setDimSizes( dimSizes );
	      newEIDP->setDimSizes( dimSizes );

              for (i = 0; i < PEInfoList[PE].PESummaryInfo.EIDCount; i++) {
                 tmpIntegerValue = PEInfoList[PE].eventInfoList[i].EID;
                 newEIDP->setCellValue( tmpIntegerValue, i );
              }
              outEventIDs->setValue( EIDVector );

	      for (i = 0; i < PEInfoList[PE].PESummaryInfo.EIDCount; i++) {
	         tmpIntegerValue = PEInfoList[PE].eventInfoList[i].count; 
	         newIntVectorP->setCellValue( tmpIntegerValue, i );
	      }
	      callCount->setValue( dataIntVector );

              for (i = 0; i < PEInfoList[PE].PESummaryInfo.EIDCount; i++) {
                 tmpDoubleValue = PEInfoList[PE].eventInfoList[i].countPercent;
                 newVectorP->setCellValue( tmpDoubleValue, i );
              }
              percentCount->setValue( dataVector );

              for (i = 0; i < PEInfoList[PE].PESummaryInfo.EIDCount; i++) {
                 tmpDoubleValue = PEInfoList[PE].eventInfoList[i].totalTime;
                 newVectorP->setCellValue( tmpDoubleValue, i );
              }
              totalTime->setValue( dataVector );

              for (i = 0; i < PEInfoList[PE].PESummaryInfo.EIDCount; i++) {
                 tmpDoubleValue = PEInfoList[PE].eventInfoList[i].timePercent;
                 newVectorP->setCellValue( tmpDoubleValue, i );
              }
              percentTime->setValue( dataVector );

              _writeOutput();

	   } else if ( outputOption == DISPLAY ) {
	      _updateDisplay( PE );
	   }
	}

}

Boolean_			/* virtual */
ProfileFU::saveConfigToFile( const CString& fileName ) const
{
	Boolean_	result;

        FILE *fp = fopen( fileName.getValue(), "w" );

        if ( fp == NULL ) {
            error( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
            result = FAILURE_;
        } else {
            fprintf( fp, "# OutputOpt, PEOpt, Display Frequency, Num PEs TO Display\n" );
            fprintf( fp, "%d %d %d %d\n", outputOption, ProcessorOption,
		     displayFrequency, NumPEsToDisplay );

	    int tmp;
	    if ( NumPEsToDisplay > 0 ) {
	       for (int i = 0; i < NumPEsToDisplay; i++) {
		  tmp = PEsToDisplay[i];
		  fprintf( fp, "%d\n", tmp );
	       }
	    }

            fclose( fp );
            result = SUCCESS_;
        }
        return result;
}

void
ProfileFU::_createDialog()
{
          dialog = _getTitledFormDialog( "profileDialog", getName() );

	  XArgs xArgs;

	  xArgs.addArg( XmNscrollBarDisplayPolicy, XmAS_NEEDED );
	  xArgs.addArg( XmNscrollingPolicy, XmAUTOMATIC );
	  xArgs.addArg( XmNleftAttachment, XmATTACH_FORM );
	  xArgs.addArg( XmNrightAttachment, XmATTACH_FORM );
	  xArgs.addArg( XmNbottomAttachment, XmATTACH_FORM );

	  scrollBox = new ScrolledWindowWidget( dialog, xArgs, "Contents" );
	  scrollBox->manage();
	  dialog->manage();
}

Boolean_
ProfileFU::_displayPE( const int PE )
{
        for (int i=0; i< NumPEsToDisplay; i++) {
	  if (PEsToDisplay[i] == PE) {
	    return TRUE_;
	  }
	}

	return FALSE_;
}


int
ProfileFU::_eventLookup( const int PE, const int EID ) 
{

        for ( int i=0; i < PEInfoList[PE].PESummaryInfo.EIDCount; i++) {
	  if ( PEInfoList[PE].eventInfoList[i].EID == EID ) {
	    return i;
	  }
	}

	return -1;
} 


void
ProfileFU::_expandPEList( const int newPEListSize )
{
        int i;
	int PE = newPEListSize-1;
        PEInfoStruct *newPEInfoList = new PEInfoStruct[newPEListSize];

	for (i=0; i<newPEListSize; i++) {
	  newPEInfoList[i].encounteredYet = FALSE_;
	  newPEInfoList[i].PESummaryInfo.EIDCount = 0;
	  newPEInfoList[i].PESummaryInfo.count = 0;
	  newPEInfoList[i].PESummaryInfo.totalTime = 0;
	  newPEInfoList[i].eventInfoList = NULL;
	}

	for (i=0; i<PEInfoListSize; i++) {
	  newPEInfoList[i].encounteredYet = PEInfoList[i].encounteredYet;
	  newPEInfoList[i].PESummaryInfo.EIDCount = PEInfoList[i].PESummaryInfo.EIDCount;
	  newPEInfoList[i].PESummaryInfo.count = PEInfoList[i].PESummaryInfo.count;
	  newPEInfoList[i].PESummaryInfo.totalTime = PEInfoList[i].PESummaryInfo.totalTime;
	  newPEInfoList[i].eventInfoList = PEInfoList[i].eventInfoList;
	}
	 
	newPEInfoList[PE].encounteredYet = TRUE_;

	newPEInfoList[PE].PESummaryInfo.EIDCount = 0;
	newPEInfoList[PE].PESummaryInfo.count = 0;
	newPEInfoList[PE].PESummaryInfo.totalTime = 0;

	if ( PEInfoList != NULL ) {
	  delete PEInfoList;
	  PEInfoList = NULL;
	}

	PEInfoListSize = newPEListSize;
	PEInfoList = newPEInfoList;
}


int
ProfileFU::_lookupDisplayIdx( const int PE )
{
        for ( int i=0; i< NumPEsToDisplay; i++) {
	  if (PEsToDisplay[i] == PE) {
	    return i;
	  }
	}
	return -1;
}


void
ProfileFU::_processEvent( const int PE, const int EID )
{
        if ( _eventLookup( PE, EID ) == -1 ) {

	  int eventsSoFar = PEInfoList[PE].PESummaryInfo.EIDCount;
	  profileStruct *newEventInfoList = new profileStruct[eventsSoFar+1];

	  for ( int i=0; i<eventsSoFar; i++ ) {
	    newEventInfoList[i].count = PEInfoList[PE].eventInfoList[i].count;
	    newEventInfoList[i].totalTime = PEInfoList[PE].eventInfoList[i].totalTime;
	    newEventInfoList[i].countPercent = PEInfoList[PE].eventInfoList[i].countPercent;
	    newEventInfoList[i].timePercent = PEInfoList[PE].eventInfoList[i].timePercent;
	    newEventInfoList[i].EID = PEInfoList[PE].eventInfoList[i].EID;
	  }

	  newEventInfoList[eventsSoFar].count = 0;
	  newEventInfoList[eventsSoFar].totalTime = 0;
	  newEventInfoList[eventsSoFar].countPercent = 0;
	  newEventInfoList[eventsSoFar].timePercent = 0;
	  newEventInfoList[eventsSoFar].EID = EID;

	  if ( PEInfoList[PE].eventInfoList != NULL ) {
	    delete PEInfoList[PE].eventInfoList;
	    PEInfoList[PE].eventInfoList = NULL;
	  }

	  PEInfoList[PE].eventInfoList = newEventInfoList;
	  PEInfoList[PE].PESummaryInfo.EIDCount++;
	}
}


void
ProfileFU::_processPE( const int PE )
{
        int i;
	Boolean_ AlreadyIn = FALSE_;

        if ( PE > MaxPESeenSoFar ) {
	  _expandPEList(PE+1);
	  MaxPESeenSoFar = PE;
	}


	if (ProcessorOption == ALL_PEs) {
	  for (i=0; i<NumPEsToDisplay; i++) {
	    if (PEsToDisplay[i] == PE) {
	      AlreadyIn = TRUE_;
	    }
	  }
	
	  if (!AlreadyIn) {
	    int *newDisplayList = new int[NumPEsToDisplay+1];
	    for (i=0; i< NumPEsToDisplay; i++) {
	      newDisplayList[i] = PEsToDisplay[i];
	    }
	    newDisplayList[NumPEsToDisplay] = PE;
	    NumPEsToDisplay++;

	    if (PEsToDisplay != NULL) {
	      delete PEsToDisplay;
	      PEsToDisplay = NULL;
	    }

	    PEsToDisplay = newDisplayList;
	  }
	}

	PEInfoList[PE].encounteredYet = TRUE_;
}


void
ProfileFU::_updateDisplay( const int PE )
{
	int	i;
	int	*cntValues;
       	int    	*timeValues;
       	char   	eventBuf[1024];
       	int    	textLoc;
       	int    	newTextCnt;

	int numEntries = PEInfoList[PE].PESummaryInfo.EIDCount;
	
	int displayIdx = _lookupDisplayIdx(PE);

        pieCountForm[displayIdx]->setPiechartAttr( XtNentries, numEntries );
        pieTimeForm[displayIdx]->setPiechartAttr( XtNentries, numEntries );

	cntValues = new int[numEntries];
	timeValues = new int[numEntries];

	newTextCnt = headerSize;

        XmTextSetString( profileText[displayIdx]->getWidget(), headerString );

	for (i = 0; i < numEntries; i++) {
	   cntValues[i] = (int)(PEInfoList[PE].eventInfoList[i].countPercent *
				100.0 + 0.5);
	   timeValues[i] = (int) (PEInfoList[PE].eventInfoList[i].timePercent *
				  100.0 + 0.5);

	   sprintf( eventBuf, "%8d\t%5.2f\t%8d\t%5.2f\t%10.2f\n",
		    PEInfoList[PE].eventInfoList[i].EID,
		    PEInfoList[PE].eventInfoList[i].countPercent * 100.0,
                    PEInfoList[PE].eventInfoList[i].count,
		    PEInfoList[PE].eventInfoList[i].timePercent * 100.0,
		    PEInfoList[PE].eventInfoList[i].totalTime ); 

	   newTextCnt += strlen(eventBuf);
	   textLoc = headerSize + i * strlen(eventBuf);

           XmTextInsert( profileText[displayIdx]->getWidget(),
		         textLoc, eventBuf );

           XmTextPosition pos;

           pos = XmTextGetLastPosition(profileText[displayIdx]->getWidget() );
           XmTextShowPosition(profileText[displayIdx]->getWidget(), pos );

	}

	textCnt = newTextCnt;
 	XmTextShowPosition( profileText[displayIdx]->getWidget(), 0 );

	pieCountForm[displayIdx]->setDisplayValues( numEntries, cntValues );
	pieTimeForm[displayIdx]->setDisplayValues( numEntries, timeValues );

	delete cntValues;
	delete timeValues;
}

/*
 * 	Initialize the static data.   Only executed once.
 */
const char *const ProfileFU::MY_CLASS = "ProfileFU";

