/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Bradley Schwartz (schwartz@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */

#ifndef __GNUG__
#include <malloc.h>
#else
#include <stdlib.h>
#endif

#include <sys/time.h>
#include <sys/resource.h>

#include <Xm/List.h>
#include <Xm/Text.h>

#define Square(x)        ( (x) * (x) )
#include "ScatterPlotMatrixFU.h"
#include "ScatterPlotMatrixFormWrapper.h"

#include "ParamDisplay.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "PabloMainInterface.h"
#include "SystemErrors.h"

#define Round(x)         ( x >= 0.0 ? (int)(x + .5) : (int)(x - .5) )
#define Norm(i, p)       ( ( ((double)p) - fuMinValues.getElement(i))/ \
               (fuMaxValues.getElement(i) - fuMinValues.getElement(i) ) )
#define BaseWidget       ( Pablo::TopLevel()->getWidget() )
#define AppContext       ( Pablo::TopLevel()->getAppContext() )

#define REALLOC_INTERVAL 10



ScatterPlotMatrixFU::ScatterPlotMatrixFU()
{
       FU_CLASS = "ScatterPlotMatrix";
       portOption = ARRAY_ELEMENT;
       connectOption = -1;

       cluster = new SquareErrorClusterVariable();
}


ScatterPlotMatrixFU::~ScatterPlotMatrixFU()
{
}


Widget
ScatterPlotMatrixFU::_setupSPMPerCategoryDialog( int dimension )
{
       static SPMConfigPerDimDataStruct data;

       Widget container =  _setupPerCategoryDialog( BaseWidget, 
			   	         (ConfigPerDimDataStruct *)&data,
				         "Config" );
       /* Initialize the areas */
       char label[STRING_MAX_LEN];
       int i;
       for (i=0; i<dimension; i++) {
	  /* Category label */
  	  XmString currentLabelString = XmStringCreateSimple( 
				  scatterPlotMatrixForm->getCategoryLabel(i) );
	  XmListAddItemUnselected( data.scrolledLists[0],
				   currentLabelString, i+1 );

	  /* Min value */
	  if ( fuMinValues.count() <= i ) {	
             fuMinValues.setElement(i, 0.0);
          }
	  sprintf(label, "%lf", fuMinValues.getElement(i) ); 
	  currentLabelString = XmStringCreateSimple( label ); 
	  XmListAddItemUnselected( data.scrolledLists[1],
				   currentLabelString, i+1 );


	  /* Max value */
	  if ( fuMaxValues.count() <= i ) {
	     fuMaxValues.setElement(i, 0.0);
          }
 	  sprintf(label, "%lf", fuMaxValues.getElement(i) ); 
	  currentLabelString = XmStringCreateSimple( label ); 
	  XmListAddItemUnselected( data.scrolledLists[2],
				   currentLabelString, i+1 );
	  XmStringFree( currentLabelString );
       }

       return( container );
}



Widget
ScatterPlotMatrixFU::_setupSPMConfigWindow()
{
       static SPMConfigDataStruct spmConfigDataStruct;

       spmConfigDataStruct.dialog = XmCreateFormDialog( BaseWidget,
						 "SPMConfigDialogContainer",
	       				         NULL, 0 );

       Widget dialogActionArea = XtVaCreateManagedWidget("ConfigActionArea",
						  xmFormWidgetClass,
						  spmConfigDataStruct.dialog,
						      NULL );
       /* Multivariate widget option */
       Widget dialogWidgetOptionArea =
	     multivariateWrapper.addRadioToggleListToDialog(
					        dialogActionArea,
					        NULL,
						"ConfigDialogWidgetOption",
						2, 0,
			       &(spmConfigDataStruct.widgetOptionRadioBox) );

       /* Data dimension */
       Widget dialogDimensionOptionsArea =
	     multivariateWrapper.addTextFieldToDialog( dialogActionArea,
				                       dialogWidgetOptionArea,
					      "ConfigDialogDataDimension",
				    &(spmConfigDataStruct.totalDimension) );
       char dimensionString[STRING_MAX_LEN];
       sprintf(dimensionString, "%d", dimension );
       XmTextFieldSetString( spmConfigDataStruct.totalDimension, 
			     dimensionString );

       /* Dataset name */
       Widget datasetNameArea =
	     multivariateWrapper.addTextFieldToDialog( dialogActionArea,
				                 dialogDimensionOptionsArea,
					         "ConfigDialogDatasetName",
				    &(spmConfigDataStruct.datasetNameWidget) );
       XmTextFieldSetString( spmConfigDataStruct.datasetNameWidget,
		             scatterPlotMatrixForm->getPointSetName() );

       /* Update interval */
       Widget updateIntervalArea =
	     multivariateWrapper.addTextFieldToDialog( dialogActionArea,
						       datasetNameArea,
					       "ConfigDialogUpdateInterval",
			      &(spmConfigDataStruct.updateIntervalWidget) );

       char updateIntervalString[STRING_MAX_LEN];
       sprintf( updateIntervalString, "%d", updateInterval );
       XmTextFieldSetString( spmConfigDataStruct.updateIntervalWidget,
		             updateIntervalString );

       /* Data input option */
       Widget dialogDataInputOptionArea =
	     multivariateWrapper.addRadioToggleListToDialog(
					        dialogActionArea,
					        updateIntervalArea,
						"ConfigDialogDataInputOption",
						2, 1,
			       &(spmConfigDataStruct.inputOptionRadioBox) );
  
       /* Additional window options */
       Widget dialogAdditionalOptionsArea =
	              multivariateWrapper.addToggleToDialog(
					      dialogActionArea,
					      dialogDataInputOptionArea,
					     "ConfigDialogAdditionalOptions",
			       &(spmConfigDataStruct.categoryNameToggle ) );

       Widget paneSeparator = XtVaCreateManagedWidget(
	                            "DialogPaneSeparator",
				     xmSeparatorWidgetClass,
				     spmConfigDataStruct.dialog,
				     XmNtopWidget, dialogActionArea,
					 NULL );
       Widget dialogControlArea = XtVaCreateManagedWidget( 
					        "DialogControlArea",
						xmFormWidgetClass,
				                spmConfigDataStruct.dialog,
					        XmNtopWidget, paneSeparator,
						XmNfractionBase, 1,
						   NULL );
       Widget dialogOKButton = XtVaCreateManagedWidget(
					       "DialogOKButton",
					        xmPushButtonWidgetClass,
						dialogControlArea,
        					XmNleftPosition, 0,
						XmNrightPosition, 1,
						   NULL );
       addCallback( dialogOKButton, XmNactivateCallback, &Callback::callback1,
		       this, &spmConfigDataStruct );

       // Fix the size of the control button area
       Dimension controlHeight, marginHeight;
       XtVaGetValues( dialogOKButton,
		      XmNheight, &controlHeight,
		         NULL );
       XtVaGetValues( dialogControlArea,
		      XmNmarginHeight, &marginHeight,
		         NULL );
       XtVaSetValues( paneSeparator,
		      XmNbottomOffset, (int)(controlHeight + marginHeight),
		         NULL );

       return(spmConfigDataStruct.dialog);
}




// This is the callback for OK in the main configuration window
void 
ScatterPlotMatrixFU::callback1( Widget /* callbackWidget */, 
			        XtPointer ptr1, XtPointer /* ptr2 */ )
{
        SPMConfigDataStruct *spmConfigDataStruct = (SPMConfigDataStruct *)ptr1;
        XtUnmanageChild( spmConfigDataStruct->dialog );
        scatterPlotMatrixForm->realize();

	/* Extract widget type */
	int numWidgetTypes;
	WidgetList childList;
	XtVaGetValues( spmConfigDataStruct->inputOptionRadioBox,
		       XmNchildren, &childList,
		       XmNnumChildren, &numWidgetTypes,
		           NULL );
	int i;
	for (i=0; i<numWidgetTypes; i++) {
            if ( XmToggleButtonGetState( childList[i] ) ) {
	       widgetType = (SPMWidgetTypeEntry) i;
	       break;
	    }
        }
        scatterPlotMatrixForm->setType( 0 );

	/* Extract data dimension */
        char *dimensionText = XmTextGetString(
				        spmConfigDataStruct->totalDimension ); 
	sscanf( dimensionText, "%d", &dimension );
	XtFree( dimensionText );

	scatterPlotMatrixForm->setDimension( dimension );
	if ( savedValues != NULL ) {
           free( savedValues );
        }
        savedValues = (double *)XtMalloc( REALLOC_INTERVAL * dimension *
					  sizeof(double) ); 

        scatterPlotMatrixForm->setPerfWidgetColors(); 
	
	/* Extract data run mode */
	int numDataRunChoices;
	XtVaGetValues( spmConfigDataStruct->inputOptionRadioBox,
		       XmNchildren, &childList,
		       XmNnumChildren, &numDataRunChoices,
		           NULL );
	for (i=0; i<numDataRunChoices; i++) {
            if ( XmToggleButtonGetState( childList[i] ) ) {
               portOption = i;
	       break;
	    }
        }

	/* Extract other options */
	static Widget categoryNameDialog = NULL;
	if (XmToggleButtonGetState(spmConfigDataStruct->categoryNameToggle)) {
           if ( categoryNameDialog == NULL ) {
              categoryNameDialog = _setupSPMPerCategoryDialog( dimension );
	   }
	   XtManageChild( categoryNameDialog );
	}

	/* Update fu min/max information */
        int dimMinValue, dimMaxValue;
	for (i=0; i<dimension; i++) {
	    scatterPlotMatrixForm->getSPMAttr( i, XtNminValue,
					       (XtArgVal)&dimMinValue );
	    scatterPlotMatrixForm->getSPMAttr( i, XtNmaxValue, 
					       (XtArgVal)&dimMaxValue );
            widgetMinValues.setElement( i, dimMinValue );
            widgetRanges.setElement( i, dimMaxValue-dimMinValue );
        }

	/* Update the dataset name */
	char *datasetName = XmTextFieldGetString( 
				     spmConfigDataStruct->datasetNameWidget );
	scatterPlotMatrixForm->setPointSetName( datasetName );

	/* Update the update interval */
	char *updateIntervalString = XmTextFieldGetString(
			           spmConfigDataStruct->updateIntervalWidget );
	sscanf( updateIntervalString, "%d", &updateInterval );
}	
	


// This callback is for end-of-run processing
void 
ScatterPlotMatrixFU::callback5( Widget /* callbackWidget */,
			        XtPointer /* ptr1 */, XtPointer /* ptr2 */ )
{
       _completePP();
       scatterPlotMatrixForm->setPointCount( pointCount );

       int i,j;

       // Draw data
       int *scaledDataPoints = new int[ (pointCount+1) * dimension ];
       for (i=0; i<pointCount; i++) {
           for (j=0; j<dimension; j++) {
        	scaledDataPoints[i*dimension+j] = 
		                 Round( Norm(j,savedValues[i*dimension+j]) *
					      widgetRanges.getElement(j) ) + 
					      widgetMinValues.getElement(j);
	   }
       }     
       int pointsPerLine[1];
       pointsPerLine[0] = pointCount;
       scatterPlotMatrixForm->setSPMValues(scaledDataPoints, 1, pointCount,
					   pointsPerLine);
       delete scaledDataPoints;
}



void
ScatterPlotMatrixFU::configure()
{
        static Widget configureShell;

	if ( configureShell == NULL ) {
           configureShell = _setupSPMConfigWindow();
	}
	XtManageChild( configureShell );
}



FunctionalUnit *	/* virtual */
ScatterPlotMatrixFU::copy()
{
	ScatterPlotMatrixFU *copy = new ScatterPlotMatrixFU();
	return copy;
}



// This callback handles display of univariate widget callback data
void
ScatterPlotMatrixFU::fuCallback1( Widget callbackWidget,
				  Widget callbackMainWindow,
				  Callback1DTextWidgetSet textWidgetSet,
				  int cbDim )
{
         if ( inputValue.isUndefined() ) {
	    return;
	 }

	 // Get indices into FU value array for points of interest
	 int smallestIndex, largestIndex, medianIndex, Q1index, Q3index;
	 XtVaGetValues( callbackWidget,
			XtNsmallestIndex, &smallestIndex,
			XtNlargestIndex, &largestIndex,
			XtNmedianIndex, &medianIndex,
			XtNQ1Index, &Q1index,
			XtNQ3Index, &Q3index,
			   NULL );
         char minimumText[STRING_MAX_LEN],
	      smallestText[STRING_MAX_LEN],
	      largestText[STRING_MAX_LEN],
	      medianText[STRING_MAX_LEN],
	      Q1Text[STRING_MAX_LEN],
	      Q3Text[STRING_MAX_LEN],
	      maximumText[STRING_MAX_LEN];

	 Array *a = (Array *)inputValue;

         sprintf(minimumText, "%lf", fuMinValues.getElement(cbDim) );
	 sprintf(smallestText,"%lf",
		              (double)a->getCellValue(smallestIndex, cbDim) );
	 sprintf(largestText, "%lf", 
		              (double)a->getCellValue(largestIndex, cbDim) );
	 sprintf(medianText, "%lf",
		              (double)a->getCellValue(medianIndex ,cbDim) );
	 sprintf(Q1Text, "%lf", (double)a->getCellValue(Q1index, cbDim) );
	 sprintf(Q3Text, "%lf", (double)a->getCellValue(Q3index, cbDim) );
	 sprintf(maximumText, "%lf", (double)fuMaxValues.getElement(cbDim) );

	 XmString minimumString = XmStringCreateSimple( minimumText );
	 XmString smallestString = XmStringCreateSimple( smallestText );
	 XmString largestString = XmStringCreateSimple( largestText );
	 XmString medianString = XmStringCreateSimple( medianText );
	 XmString Q1String = XmStringCreateSimple( Q1Text );
	 XmString Q3String = XmStringCreateSimple( Q3Text );
	 XmString maximumString = XmStringCreateSimple( maximumText );

	 XtVaSetValues( textWidgetSet.minimumTextWidget,
			XmNlabelString, minimumString,
			    NULL );
	 XtVaSetValues( textWidgetSet.smallestTextWidget,
		        XmNlabelString, smallestString,
			    NULL );
	 XtVaSetValues( textWidgetSet.largestTextWidget,
		        XmNlabelString, largestString,
		            NULL );
	 XtVaSetValues( textWidgetSet.medianTextWidget,
			XmNlabelString, medianString,
			    NULL );
	 XtVaSetValues( textWidgetSet.Q1valueTextWidget,
			XmNlabelString, Q1String,
			    NULL );
	 XtVaSetValues( textWidgetSet.Q3valueTextWidget,
		        XmNlabelString, Q3String,
			    NULL );
	 XtVaSetValues( textWidgetSet.maximumTextWidget,
		        XmNlabelString, maximumString,
		            NULL );
         XmStringFree( minimumString );
	 XmStringFree( smallestString );
	 XmStringFree( largestString );
	 XmStringFree( medianString );
	 XmStringFree( Q1String );
	 XmStringFree( Q3String );
	 XmStringFree( maximumString );
	 XtPopup( callbackMainWindow, XtGrabNone );
}


void
ScatterPlotMatrixFU::fuCallback2( int initialIndex, int clusterPoints )
{
	 cluster->setNumberOfClusters(3);
	 cluster->loadDataset( &(savedValues[initialIndex*dimension]),
			       dimension, clusterPoints );
         cluster->cluster();
         int numberOfClusters = cluster->getNumberOfClusters();
         int *clustering = cluster->getClusters();
	 int *cntlist = cluster->getClusterCnts();
	 int *repList = cluster->getClusterReps();


	 /* Cluster drawing stuff */
         double *dataPoint = new double[dimension];
	 int *scaledDataPoints = new int[(pointCount+1) * dimension ];
	 int i, j;
         for (i=0; i<pointCount; i++) {
	     for (j=0; j<dimension; j++) {
	         dataPoint[j] = warnings.fuDataRangeWarningCheck( this,
				             savedValues[i*dimension+j],
				             fuMinValues.getElement(j),
					     fuMaxValues.getElement(j));
	         scaledDataPoints[i*dimension+j] = 
			               Round( Norm(j,dataPoint[j]) *
					      widgetRanges.getElement(j) ) + 
					      widgetMinValues.getElement(j);
             }
         }
	 scatterPlotMatrixForm->setSPMValues( scaledDataPoints,
				              clustering,
					      numberOfClusters, pointCount );


/* Output attribute stuff  */
/*
	 Array *outputArrayP = (Array *)outputValue;
	 int newDimSize[2];
	 newDimSize[0] = numberOfClusters;
	 newDimSize[1] = dimension;
	 outputArrayP->setDimSizes( newDimSize );

	 int i, j;
	 for (i=0; i<numberOfClusters; i++) {
cout << "Rep " << i << " is " << repList[i] << " \n";
             for (j=0; j<dimension; j++) {
cout << "Cluster " << i << " dim " << j << " = " << savedValues[repList[i]*dimension+j] << " \n";
	         outputArrayP->setCellValue( 
                               savedValues[repList[i]*dimension+j],i,j );
             }
cout << "A rep set\n";
         }

         outputPort->setValue( (Value)outputArrayP );
cout << "Value set\n";
         _writeOutput();
cout << "output written\n";
*/


	for (i=0; i<numberOfClusters; i++) {
cout << "number in cluster " << i << " is " << cntlist[i] << " \n";
        }

// fprintf(stderr, "User clustering time was %lf\n", userTime );
//	free( clusterValues );
}


/*
void
ScatterPlotMatrixFU::fuCallback2( SPMPreprocessOptions *data )
{
	 preprocessMask = 0;
  	 int i;
	 for (i=0; i<data->numberOptions; i++) {
             if (XmToggleButtonGetState(data->toggleOptions[i])) {
                 preprocessMask |= pow(2,i);
	     }
	 }
}
*/



void
ScatterPlotMatrixFU::fuSubclassProcessPerCategoryCallback()
{
       int i;
       for (i=0; i<dimension; i++) {
	    if ( XmStringEmpty( (XmString)fuCategoryLabels.getElement(i) ) ) {
	       fuCategoryLabels.setElement(i, XmStringCreateSimple(" "));
            }
	    scatterPlotMatrixForm->setCategoryLabel(
		              (XmString)fuCategoryLabels.getElement(i), i );
       }
}


void 
ScatterPlotMatrixFU::init()
{
        // Input port
	inputPort = new InputPort( "Input" );
        inputPort->addTraits( DOUBLE,2 );
	_addInputPort( inputPort );

	// Output port
	outputPort = new OutputPort( "Cluster Representatives" );
	_addOutputPort( outputPort );
	outputValue = new Array( DOUBLE, 2 );

	// Display wrapper
	scatterPlotMatrixForm = new ScatterPlotMatrixFormWrapper( 
			   	                      BaseWidget, this );

	// Tell the main interface to signal us when run is complete
	Widget signalWidget = XtVaCreateWidget( "SPMSignalWidget",
		                       xmPushButtonWidgetClass,
				       Pablo::TopLevel()->getWidget(),
				         NULL );
	Pablo::MainInterface()->addSignalWidget( signalWidget );
	addCallback( signalWidget, XmNactivateCallback,
		     &Callback::callback5, this );
}


Boolean_
ScatterPlotMatrixFU::inputTraitsValid()
{
        DataTraits portTraits( DOUBLE, 2 );
	outputPort->setTraits( portTraits );

	return( TRUE_ );
}


Boolean_		/* virtual */
ScatterPlotMatrixFU::loadConfigFromFile( const CString& fileName )
{
        FILE *fp = fopen( fileName.getValue(), "r" );

        if ( fp == NULL ) {
            warning( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
            configure();
        } else {
	    int x, y, width, height;
	    char buf[STRING_MAX_LEN];

	    // ----- Comment line and numeric parameters -----
            fscanf(fp, "%[^\n]\n", buf );
	    
	    if ( fscanf( fp, "%d %d %d %d %d %d %d\n",
                      &x, &y, &width, &height, &dimension, &widgetType,
		      &portOption) != 7) {
		warning( "Unable to read configuration information from %s\n",
			  fileName.getValue() );
	    } else {
		scatterPlotMatrixForm->setSPMAttr( x,y, width, height );
	      	scatterPlotMatrixForm->realize();
		scatterPlotMatrixForm->setType( widgetType );
	        scatterPlotMatrixForm->setDimension( dimension );

		// Allocate initial space for data 
		if ( savedValues != NULL ) {
		   free( savedValues );
		}
		savedValues = (double *)malloc(REALLOC_INTERVAL*dimension *
					       sizeof(double) ); 

                scatterPlotMatrixForm->setPerfWidgetColors(); 
		
		// Set per-dim widget ranges
		int dimMinValue, dimMaxValue;
		int i;
		for (i=0; i<dimension; i++) {
		    scatterPlotMatrixForm->getSPMAttr( i,
						    XtNminValue, 
						    (XtArgVal) &dimMinValue );
		    scatterPlotMatrixForm->getSPMAttr( i,
						    XtNmaxValue, 
						    (XtArgVal) &dimMaxValue );
                    widgetMinValues.setElement( i, dimMinValue );
                    widgetRanges.setElement( i, dimMaxValue-dimMinValue );
                }
                isConfigured = TRUE_;

		// ----- Comment Line; Point set name
		fscanf( fp, "%[^\n]\n", buf );
		int labelSize;
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
		if ( labelSize == 0 ) {
                   strcpy( buf, "\0" );
		}
		fscanf( fp, "\n", buf );
		scatterPlotMatrixForm->setPointSetName( buf );

		// ----- Comment Line; Dimension categories
		fscanf( fp, "%[^\n]\n", buf );
		for (i=0; i<dimension; i++) {
                    fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
		    if ( labelSize == 0 ) {
                       strcpy( buf, "\0" );
		    }
                    scatterPlotMatrixForm->setCategoryLabel( buf, i );
		    fscanf( fp, "\n", buf );
	        }
		
		// ----- Comment Line; Min/Max Info
		fscanf( fp, "%[^\n]\n", buf );
		double currentMinValue, currentMaxValue;
		for (i=0; i<dimension; i++) {
                    fscanf(fp, "%lf %lf\n",&currentMinValue,&currentMaxValue );
		    fuMinValues.setElement(i, currentMinValue );
		    fuMaxValues.setElement(i, currentMaxValue );
		}

		// ----- Comment Line; Child Widget Info
		scatterPlotMatrixForm->setConnectOption( connectOption );
            } 
            fclose( fp );
        }

	if ( portOption == -1 ) {
	    configureOperation();
	}

	if ( ! isConfigured ) {
	    configure();
	}
	return isConfigured;
}



Boolean_
ScatterPlotMatrixFU::ready()
{
	return isConfigured;
}


void
ScatterPlotMatrixFU::run(  Boolean_& /* errorFlag */ ) 
{
        Assert( inputPort->valueAvailable() );
	pointCount++;

        inputValue = inputPort->getValue();
	Array *a = (Array *)inputValue;
	int *dimSizes = (int *)a->getDimSizes();
	int nPoints = dimSizes[0];	

        double *dataPoint = new double[dimension];
	double *pDataPoint;
	int i, j;
	int *scaledDataPoints = new int[(nPoints+1) * dimension ];
        int pointsPerLine[1];

	switch ( portOption ) {
	    case FULL_ARRAY:
	       for (i=0; i<nPoints; i++) {
		   for (j=0; j<dimension; j++) {
		       dataPoint[j] = a->getCellValue(i,j); 
		       dataPoint[j] = warnings.fuDataRangeWarningCheck( this,
				             dataPoint[j],
				             fuMinValues.getElement(j),
					     fuMaxValues.getElement(j));
		       scaledDataPoints[i*dimension+j] = 
			               Round( Norm(j,dataPoint[j]) *
					      widgetRanges.getElement(j) ) + 
					      widgetMinValues.getElement(j);
                   }
               }     
	       pointsPerLine[0] = nPoints;
               scatterPlotMatrixForm->setSPMValues(scaledDataPoints, 1,
						   nPoints, pointsPerLine );
	       // savedValues = a;
	       break;

	   case ARRAY_ELEMENT:
	       // Do preprocessing if user has requested it
     	       if ( preprocessMask ) {
	          if ( ! ppData.isInitialized ) {
		     pDataPoint = _initializePPData(a);
	             ppData.isInitialized = True;
                  } else {             
                     pDataPoint = _addPPData( a, nPoints );
		  }

	       // Otherwise do normal processing
	       } else {		  
	          int *scaledDataPoint = new int[dimension];

	          for (j=0; j<dimension; j++) {
                     dataPoint[j] = a->getCellValue(nPoints-1,j);
		     dataPoint[j] = warnings.fuDataRangeWarningCheck( this,
		                                              dataPoint[j],
		        fuMinValues.getElement(j), fuMaxValues.getElement(j));

		     // Keep data within existing range
		     // Map point to value within the integer range
		     scaledDataPoint[ j ] =  Round( Norm(j,dataPoint[j]) *  
		                                 widgetRanges.getElement(j) ) +
			                   widgetMinValues.getElement(j);
	          }
		  if ( pointCount % updateInterval == 0 ) {
	              scatterPlotMatrixForm->setSPMValues( scaledDataPoint );
		  }
		  pDataPoint = (double *)dataPoint;
	          delete scaledDataPoint;
	       }

	       // Save the complete history
	       if ( (pointCount % REALLOC_INTERVAL) == 0 ) {
                  savedValues = (double *) realloc( (double *)savedValues, 
         	     (pointCount+REALLOC_INTERVAL)*dimension*sizeof(double) );
	       }	
	       for (i=0; i<dimension; i++) {
		  savedValues[ (pointCount-1)*dimension + i ] = pDataPoint[i];
	       }

	       // Period clustering
	       int clusteringInterval;
	       clusteringInterval =
	                       scatterPlotMatrixForm->getClusteringInterval();
	       if ( scatterPlotMatrixForm->isClusteringEnabled() ) {  
                  if ((pointCount % clusteringInterval) == 0 ) {
                      fuCallback2( pointCount-clusteringInterval, 
				   clusteringInterval);
                  }  
               }
	}
	delete dataPoint;
        delete scaledDataPoints;
}


Boolean_		/* virtual */
ScatterPlotMatrixFU::saveConfigToFile( const CString &fileName ) const
{
	Boolean_ result;
        FILE *fp = fopen( fileName.getValue(), "w" );

        if ( fp == NULL ) {
            error( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
            result = FAILURE_;
	} else {
            int x, y, width, height;
	    scatterPlotMatrixForm->getPerfWidgetPosition( x, y,
							  width, height);

            fprintf( fp, "# X Y Wdth Hght DataDim WidgetType PortOpt\n" );
            fprintf( fp, "%d %d %d %d %d %d %d\n", 
			 x, y, width, height, dimension, widgetType,
		         portOption );

	    fprintf( fp, "# Dataset Name\n" );
	    fprintf( fp, "%d %s\n",
		         strlen(scatterPlotMatrixForm->getPointSetName() ),
		         scatterPlotMatrixForm->getPointSetName() );

	    fprintf( fp, "# Dimension Categories\n");
	    int i;
	    for (i=0; i<dimension; i++) {
                fprintf(fp," %d %s\n",
			strlen(scatterPlotMatrixForm->getCategoryLabel(i)),
			scatterPlotMatrixForm->getCategoryLabel(i) );
	    }
	    fprintf( fp, "# Min/Max Values\n");
	    for (i=0; i<dimension; i++) {
	        fprintf(fp, "%lf %lf\n",
		    fuMinValues.getElement(i), fuMaxValues.getElement(i));
	    }
            fclose( fp );
            result = SUCCESS_;
        }
        return result;
}



// Global class definition
const char *const ScatterPlotMatrixFU::MY_CLASS = "ScatterPlotMatrixFU";

