/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Authors: Philip C. Roth (proth@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/***************************************************************
 *
 * Selection.C
 * 
 * Phil Roth
 * Implementation of methods for class Selection.
 *
 ***************************************************************/
#include <fstream.h>
#include <stdio.h>
#include <strstream.h>
#include "Selection.h"


/*
 * Some simple, useful macro definitions
 */
#define MIN(x,y) ( (x) < (y) ? (x) : (y) )
#define MAX(x,y) ( (x) > (y) ? (x) : (y) )




Selection::Selection( int val )
{
	lo = hi = val;
}



Selection::Selection( int low, int high )
{
	lo = low;
	hi = high;
}




Selection *
Selection::_makeCanonical( Selection *list )
{
	Selection *curr = list;		// ptr through list
	Selection *temp;	// result of union of selections


	while( curr ) {

	   while( curr->next ) {
		if( curr->hi > ((Selection *)curr->next)->lo - 2 ) {

   		   /* the two should be merged */
		   /* find their union */
		   temp = curr->unionYourself( (Selection *) curr->next );

		   /* replace curr, curr->next with their union */
		   curr->lo = temp->lo;
		   curr->hi = temp->hi;

		   if( temp->next ) {
			((Selection *)curr->next)->lo = 
				((Selection *)temp->next)->lo;
			((Selection *)curr->next)->hi = 
				((Selection *)temp->next)->hi;
  		   }
		   else {
		     list = (Selection *)curr->next->removeYourself( list );
		   }

		   /* destroy the union list */
		   delete temp;
		}
		else {
			break;
		}
	    }
	    curr = (Selection *)curr->next;	// advance to next segment
	}

	return list;
}




/* virtual */
Boolean_
Selection::_insertHere( ListNode *curr )
{
	if( lo > ((Selection *)curr)->lo ) {
		return FALSE_;
	}
	else {
		return TRUE_;
	}
}

Selection *
Selection::copy( void )
{
      	Selection* curr = this;
      	Selection* newlist = NULL;
      	Selection* newlistend = NULL;


      	/* duplicate the selections in the list */
      	while( curr ) {
              	/* duplicate curr */
              	Selection* temp = new Selection( curr->lo, curr->hi );
              	if( !temp ) {
                       	cerr << "Class selection: out of memory.\n";
                      	return NULL;            /* couldn't allocate memory */
              	}

              	/* add it to the end of the newlist */
              	if( newlistend ) {
                      	newlistend->setNextInList( temp );
              	}
               	else {
               	        newlist = temp;
               	}
               	newlistend = temp;

               	curr = (Selection *)curr->next;
       }

       return newlist;
}


void
Selection::getRange( int &low, int &high )
{
	low = lo;
	high = hi;
}



void
Selection::getPoint( int &val )
{
	val = lo;
}




/* virtual */
ListNode *
Selection::insertYourself( ListNode *list )
{
	ListNode *temp = ListNode::insertYourself( list );

	return _makeCanonical( (Selection *)temp );
}




/*
 * intersectYourself()
 *
 * Returns the intersection of "this" and s.
 * The intersection is always a single selection, or NULL.
 * The test for intersection, in particular the last conjunction,
 * assumes that s->lo <= s->hi. 
 */
Selection *
Selection::intersectYourself( Selection *s )
{
	Selection *temp = NULL;

	/* check if ranges overlap, or one is contained entirely within *
	 * the other */
	if( ((lo >= s->lo) && (lo <= s->hi) ) ||
	    ((hi >= s->lo) && (hi <= s->hi) ) ||
	    ((s->lo > lo ) && (s->hi < hi)  )   ) {

		temp = new Selection( MAX( lo, s->lo ),
						MIN( hi, s->hi ) );
	}

	return temp;
}



Boolean_ 
Selection::isPoint( void )
{
	return (lo == hi ?  TRUE_ : FALSE_ );
}




/* static */
void
Selection::printList( ostream& s, Selection *list )
{
	Selection *curr = list;

	while( curr ) {
		curr->printOn( s );
		curr = (Selection *)curr->next;
	}
	s << "[]\n";
}




/* static */
Selection *
Selection::readSelectionList( FILE *fp )
{
 	Selection *list = NULL;
 	Selection *last_entry = NULL;
 	Selection *selection;
	int low, high;


	while( fscanf( fp, "[%d%d]\n", &low, &high ) == 2 ) {
		selection = new Selection( low, high );

		/* link selection to end of the Selection list */
 		if( list == NULL ) {
 		    list = selection;
 		} 
		else {
 		    last_entry->next = selection;
		}
		last_entry = selection;
	}

	return list;
}



Selection *
Selection::removePoint( int val )
{
	Selection *list = this;
	Selection *curr = list;


	/* find selection which contains point */
	while( curr ) {
		if( (val >= curr->lo) && ( val <= curr->hi) ) {

			if( curr->isPoint() ) {
			    return (Selection *)curr->removeYourself( list );
			}

			/* it is a range - split the range if necsesary */
			else if( val == curr->lo ) {
				/* save effort and just modify */
				/* current selection range */
				curr->lo = val + 1;
			}

			else if( val == curr->hi ) {
				/* again, just modify current range */
				curr->hi = val - 1;
			}
			else {
				/* split range into two ranges */
				Selection *temp = 
					new Selection( val + 1, curr-> hi );
				curr->hi = val - 1;

				list = (Selection *)temp->insertYourself( list );
			}
			break;
		}

		else {
			/* advance pointer */
			curr = (Selection *)curr->next;
		}
	}

	/* val wasn't in the list, do nothing */
	return list;
}


Selection *
Selection::removeRange( int low, int high )
{
	Selection *list = this;
	Selection *curr = list;
	Selection *range = new Selection( low, high );	// the range to remove
	Selection *temp, *tmp2;
	

	/* find all intersecting selection ranges */
	while( curr ) {

		/* find the possible intersection between curr and range */
		temp = range->intersectYourself( curr );

		/* check if they intersect */
		if( temp ) {

			/* check for entire containment of curr in range */
			if( ( curr->lo >= range->lo ) && 
				( curr->hi <= range->hi ) ) {

				tmp2 = (Selection *)curr->next;
				list = (Selection *)curr->removeYourself( list );

				curr = tmp2;	// continue intersecting
			}

			/* check for containment of range in curr */
			else if( (curr->lo < range->lo) &&
				 (curr->hi > range->hi) ) {

				/* split curr around range */
				tmp2 = new Selection( temp->hi + 1, curr->hi );
				curr->hi = temp->lo - 1;

				list = (Selection *)tmp2->insertYourself( list );

				/* advance the pointers */
				curr = (Selection *)curr->next;
			}

			/* check for overlap */
			else if( range->lo <= curr->lo ) {
				curr->lo = temp->hi + 1;
			}

			else {
				curr->hi = range->lo - 1;
			}
		}

		/* they don't intersect, so look at next selection range */
		else {
			curr = (Selection *)curr->next;	// advance pointer
		}

		delete temp;
	}

	delete range;
	return list;		// [lo, high] wasn"t in list, do nothing 
}







void
Selection::setPoint( int val )
{
	lo = hi = val;
}



void
Selection::setRange( int low, int high )
{
	lo = low;
	hi = high;
}



/*
 * unionYourself()
 *
 * Returns union of "this" and s.
 * This could be a single selection if they overlap, or a list
 * of two selections if not.
 * No objects are deleted or modified.
 * Copies of "this" and s are created.
 */
Selection *
Selection::unionYourself( Selection *s )
{
	Selection *temp = intersectYourself( s );
	Selection *temp1;


	/* check if they overlap, or are within one unit */
	if( temp || ( s->lo - hi == 1 ) || ( s->hi - lo == 1 )) {

		/* they are close: create new selection of union */
		temp1 = new Selection( MIN( lo, s->lo ), 
							MAX( hi, s->hi ) );
	}

	else {
		/* they don't overlap */
		temp1 = new Selection( lo, hi );
		Selection *temp2 = new Selection( s->lo, s->hi );

		temp1->next = temp2;
	}

	delete temp;		// delete the temp object from intersection

	return temp1;
}



void
Selection::printOn( ostream& s )
{
	s << '[' << lo << ' ' << hi << "]\n";
}



