/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Tara Madhyastha (tara@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * SoundFunctionalUnit.cc: Implements SoundFunctionalUnit.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/SoundFunctionalUnit.C,v 1.28 1994/03/31 03:19:38 aydt Exp $
 *	
 */
#include <unistd.h>
#include <string.h>

#include "SoundFunctionalUnit.h"
#include "SWMultiParams.h"
#include "FileSelectionDialog.h"
#include "ParamConfig.h"
#include "InputPort.h"
#include "SystemErrors.h"

DEVICETYPE _mySoundDevice;

const char *const SoundFunctionalUnit::MY_CLASS = "SoundFunctionalUnit";

SoundFunctionalUnit::SoundFunctionalUnit()
{
  _setClassName( MY_CLASS );
  isReady = FALSE_;
  blockingFlag = 1;
}

SoundFunctionalUnit::~SoundFunctionalUnit()
{
delete input;
}


double *SoundFunctionalUnit::arrayToDoubleVector(Array *a, int &nx)
{
// If given an array, the array must look something like this:
// 3 v3
// 2 v2
// 1 v1
// 0 v0
//    0
// This is 2-dimensional(?)
//
// See comments at init() about what a hack this is.

  int i;
  double *dVec;

  const int* dimSizes = a->getDimSizes();

  if ( a->getDimension() == 1) {
    nx = dimSizes[0];
    dVec = new double[nx];
    for( i = 0; i < nx; i++) {
      dVec[i] = (double) a->getCellValue(i);
    }
  } else {
    nx = dimSizes[1];
    dVec = new double[nx];
    for( i = 0; i < nx; i++) {
      dVec[i] = (double) a->getCellValue(0, i);
    }    
  }
  return dVec;
}
  

char *SoundFunctionalUnit::getWidgetControlFile(char *title, char *basename)
{
  /*
   * Create the file selection dialog and set the title
   */
  FileSelectionDialog *FSD = new FileSelectionDialog();
  FSD->setXmStringResource( XmNdialogTitle, title );

  /*
   * Set the mask and restrict to display only files
   */
  CString wcfdir;
  getDefaultWCFdir( wcfdir );
  wcfdir = wcfdir + form( "/*%s.wcf", basename );

  FSD->setXmStringResource( XmNdirMask, (char *)wcfdir.getValue() );
  FSD->setResource( XmNfileTypeMask, (XtArgVal) XmFILE_REGULAR );
  FSD->setButtonSensitive( XmDIALOG_CANCEL_BUTTON, FALSE_ );


  /*
   * Run the dialog.  We only allow "OKAY"
   */
  char *wcfFile;
  if ( FSD->run() ) {                 // Okay button
    char *buffer = strdup( FSD->getFileName().getValue() );
    wcfFile = buffer;
  } 
  delete FSD; 
  return wcfFile;
}

char *SoundFunctionalUnit::getWCFFileName()
{
  CString wcfdir;
  char buf[256];
  char *token, *s;
  FUParams params;
  int deviceId = 0;

  getDefaultWCFdir(wcfdir);
  FILE *fp = fopen(form("%s/.descriptions", (const char *) wcfdir), "r");
  
  if ( fp == NULL ) {
    error( "Unable to open %s/.descriptions: %s is not a valid WCF directory\n",
	  wcfdir.getValue(), wcfdir.getValue() );
    return NULL;
  } 
  CStringObjList wcfDeviceList, wcfDescriptionList;

  while(fgets(buf, 256, fp) != NULL) {
    token = strtok(buf, " \t"); // first token in string (subdirectory name)
    s = strtok(NULL, "\n");     // the rest of the string
    if (access(form("%s/%s/%s.wcf",
		    wcfdir.getValue(), token, getClassName()), R_OK) == 0) {
		      wcfDeviceList.addElement( token );
		      wcfDescriptionList.addElement( s );
		    }
  }
  // now wcfDeviceList and wcfDescriptionList contain the "directories"
  // and descriptions of sound devices which support this 
  // SoundFunctionalUnit class. If there is more than one possibility,
  // prompt for a valid device. If there are no possibilities, this is 
  // (obviously) an error.

  if (wcfDeviceList.count() == 0) {
    error( "No sound devices support this sound functional unit (%s) in %s",
	  getClassName(), wcfdir.getValue());
    return NULL;
  } else {
    if (wcfDeviceList.count() > 1) {

      params.addRadioButtonsParam("Sound Device",
				  BaseFUParamEntry::Integer,
				  0,
				  wcfDescriptionList);
  
      ParamConfig pc(Pablo::TopLevel(), params, "Sound Device Selection");
      pc.run();
  
      BaseFUParamEntry &deviceEntry = params.getEntry("Sound Device");

      if( deviceEntry.valueIsValid() ) {
	deviceId = deviceEntry.getValue().getInteger();
      } else {
	warning( "Sound device selection was not valid." );
      }
    }
  }
  return(strdup(form("%s/%s/%s.wcf", wcfdir.getValue(),
			      wcfDeviceList[deviceId].getValue(), getClassName())));
}
    

void SoundFunctionalUnit::getDefaultWCFdir(CString &wcfdir)
{
  char *dir;

  if ((dir = getenv("WCF_DIR")) == NULL) {
    dir = ".";
  }
  wcfdir = CString(dir);
}
  

void SoundFunctionalUnit::printOn(ostream &os) const
{
	os << getClassName();
}

Boolean_ SoundFunctionalUnit::ready()
{
	return isReady;
}

void SoundFunctionalUnit::run(  Boolean_& errorFlag )
{
  double *dvec;
  int nx;

  Assert( input -> valueAvailable() );

  const Value &inValue = input -> getValue();
	
  if (!isReady) {
    warning("SoundFunctionalUnit not ready to run");
    errorFlag = TRUE_;
    return;
  }

  if (inValue.getTraits().getDimension() > 0) {
    dvec = arrayToDoubleVector((Array *) inValue, nx);
    mySonicWidget->execute(dvec, nx);
    delete dvec;
  } else {
    mySonicWidget->execute((double) inValue);
  }
}



void SoundFunctionalUnit::init()
{

  input = new InputPort( "Input" );
  input -> addTraits( INTEGER );
  input -> addTraits( FLOAT );
  input -> addTraits( DOUBLE );

  input -> addTraits( INTEGER, 1 );
  input -> addTraits( FLOAT, 1 );
  input -> addTraits( DOUBLE, 1 );

// It is really a Hack to permit arrays into these input ports. This
// is done because vectors are used in the sound stuff to denote 
// "coordinates of a point". At the current time it is hard to group together
//  a variable number of arguments and stuff them into a vector. It is
//  even more impossible to conceive of variable numbers of input ports.
//  So we are using PopulateXYPairs for purposes it was not intended for,
//  i.e., interpreting it as a vector. Forgive me.

  input -> addTraits( INTEGER, 2 );
  input -> addTraits( FLOAT, 2 );
  input -> addTraits( DOUBLE, 2 );
  
  _addInputPort( input );
}

void SoundFunctionalUnit::configure()
{

  CString foo;
  char name[256];
  foo = getName();
  strcpy(name, foo);

  if ( isReady == FALSE_ ) {
      /*
       * We ask for the Widget Control File and verify as much as possible
       * that it is valid.  If it isn't we ask again.  If it is, then we
       * go on to create the widget.  The createWidget() call duplicates
       * some of our checks on the WCF file, but if it fails there is no
       * recovery.  If it fails, it should be for some reason other than
       * our inability to access the WCF file and we abort.  Before things
       * just continued on their merry way, but then the Save Configuration
       * would coredump because there wasn't a valid configuration to save -
       * that's worse...		(RA 2/94)
       */
      Boolean_ wcfOK;
      struct stat statbuf;
      char *wcf;

      do {
	  wcfOK = FALSE_;
          wcf = getWidgetControlFile("Sound Widget Control File");

          if ( ! widgetReader.looksValid( wcf ) ) {           
	      info( "Selected file %s doesn't end in .wcf", wcf );
	  } else {
              if ( stat((char *)wcf, &statbuf) != 0 ) {
                  info ( "Can't open file %s: %s", wcf, errorString() );
              } else {
                  if ( (statbuf.st_mode & S_IFDIR) ) {       
                      info( "Selected file %s is a directory.", wcf );
                  } else {                                  
  	              FILE *fp = fopen(wcf, "r");
                      if ( fp == NULL ) {
                          info( "Can't open file %s: %s", wcf, errorString() );
                      } else {
                          wcfOK = TRUE_;
 		          fclose( fp );	      // createWidget opens it for good!
	             }
	          }
	      }
          }
      } while ( wcfOK == FALSE_ );
	            
      if ((mySonicWidget = widgetReader.createWidget(wcf, name)) == NULL) {
          abort( "Cannot initialize Sonic Widget." );
      } else {
          // turn on blocking so we will be synchronized with Pablo displays
          mySonicWidget->setBlocking(blockingFlag);
          isReady = TRUE_;
      }

  }

  SWMultiParams *swp = new SWMultiParams(Pablo::TopLevel()->getWidget(), 
			  getClassName(),
			  "Pablo", mySonicWidget);
  swp->run();
  delete swp;

}

	
Boolean_           /* virtual */
SoundFunctionalUnit::loadConfigFromFile ( const CString& fileName) 
{
  CString foo = getName();
  char name[256];
  
  strcpy(name, foo);

  CString wcfFileName = (CString& ) fileName + ".wcf";

  Boolean_ status = FAILURE_;
  
  if (!isReady) {
    if ((mySonicWidget = 
	      widgetReader.createWidget((char *)wcfFileName.getValue(), 
						   name)) == NULL) {
      error( "Cannot initialize Sonic Widget" );
      return status;
    }
    mySonicWidget->setBlocking(blockingFlag);
    isReady = TRUE_;
    return SUCCESS_;
  }
  return(status);
}

Boolean_           /* virtual */
SoundFunctionalUnit::saveConfigToFile( const CString& fileName) const
{
  CString wcfFileName = (CString& )fileName + ".wcf";
  filebuf wcffb;
  
  if( wcffb.open( wcfFileName.getValue(), output ) == 0 ) {
    error( "Unable to open %s: %s\n", wcfFileName.getValue(), errorString() );
    return FAILURE_;
  }
  ostream wcfout(&wcffb);
  mySonicWidget->writeWCF(wcfout);
  return SUCCESS_;

}


FunctionalUnit *SoundFunctionalUnit::copy()
{
	SoundFunctionalUnit *copy = new SoundFunctionalUnit();

	return copy;
}
