/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Contributing Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * SynthesizeArray.cc - A functional unit for synthesizing a two-dimensional
 * 			data array
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/SynthesizeArray.C,v 1.19 1994/03/15 16:43:19 aydt Exp $
 */

#include "SynthesizeArray.h"
#include "Addition.h"
#include "Subtraction.h"
#include "Multiplication.h"
#include "Division.h"
#include "Minimum.h"
#include "Maximum.h"
#include "BinaryCount.h"

#include "FUParams.h"
#include "ParamConfig.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "SystemErrors.h"

SynthesizeArray::SynthesizeArray()
{
	_setClassName( MY_CLASS );

	index0 = NULL;
	index1 = NULL;
	cellValue = NULL;
	outputValue = NULL;
	myOperator = NULL;
}

SynthesizeArray::~SynthesizeArray()
{
	delete index0;
	delete index1;
	delete cellValue;
	delete outputValue;
	delete myOperator;
}

void			
SynthesizeArray::_resizeArray( int newIndex0, int newIndex1 )
{
	int newDimSize[2];

	if ( newIndex0 >= dimension0 ) {
	    newDimSize[0] = newIndex0 + 1;
	} else {
	    newDimSize[0] = dimension0;
	}

	if ( newIndex1 >= dimension1 ) {
            newDimSize[1] = newIndex1 + 1;
        } else {
            newDimSize[1] = dimension1;
        }

	Array* newArrayP = new Array( DOUBLE, 2 );
	newArrayP->setDimSizes( newDimSize );

	Array *arrayP = (Array *)array;
        int x, y;

	for ( x = 0; x < newDimSize[0]; x++ ) {
            for ( y = 0; y < newDimSize[1]; y++ ) {
		if ( ( x < dimension0 ) && ( y < dimension1 ) ) {
                   newArrayP->setCellValue( arrayP->getCellValue(x, y), x, y );
		} else {
                   newArrayP->setCellValue( initialValue, x, y );
		}
            }
        }

	array = newArrayP;
	dimension0 = newDimSize[0];
	dimension1 = newDimSize[1];

	delete newArrayP;
}

void 
SynthesizeArray::_setOperator( const int opId )
{
	if ( myOperator != NULL ) {
	    delete myOperator;
	}
  	myOperatorNumber = opId;

  	switch ( opId ) {
     	   case ADDITION:
              myOperator = new Addition;
              break;

     	   case SUBTRACTION:
              myOperator = new Subtraction;
              break;

     	   case MULTIPLICATION:
              myOperator = new Multiplication;
              break;

     	   case DIVISION:
              myOperator = new Division;
              break;

     	   case MINIMUM:
              myOperator = new Minimum;
              break;

     	   case MAXIMUM:
              myOperator = new Maximum;
              break;

	   case COUNT:
	      myOperator = new BinaryCount;
	      break;

     	   default:
              warning( 
		"\nOperator id %d does not correspond to any known operation",
	        opId );
              myOperatorNumber = -1;
              break;
  	}

  	if ( myOperatorNumber < 0 ) {
      	   isReady = FALSE_;
  	} else {
           isReady = TRUE_;
  	}
}

void				/* virtual */
SynthesizeArray::configure()
{
	double initVal;

	if ( isConfigured ) {
	    initVal = (double)initialValue;
	} else {
	    dimension0 = 8;
	    dimension0Max = 64;
	    dimension1 = 8;
	    dimension1Max = 64;
	    initVal = 0.0;
	    isConfigured = TRUE_;
	}

	FUParams params;

   	/*
         * I use a zero-entry button list to position a label which tells
         * us what was selected in configureOperation() - don't want to
         * allow changes at this point.  ICK! (RAA)
         */
        static char buf[128];
        if ( actionOption == REPLACE ) {
            sprintf( buf, "Action is %s",
                 (const char *)SynthesisActionList.getElement( actionOption ) );
        } else {
            sprintf( buf, "Action is %s; Operation is %s",
               (const char *)SynthesisActionList.getElement(actionOption),
               (const char *)SynthesisOpNameList.getElement(myOperatorNumber) );
        }
	CStringObjList nullList(0);
	params.addRadioButtonsParam( buf, BaseFUParamEntry::Integer, 0, 
				     nullList );
	params.addTextParam( "Initial Array Size (Dimension 0)",
			      BaseFUParamEntry::Integer, dimension0 );
	params.addTextParam( "Maximum Array Size (Dimension 0)", 
			      BaseFUParamEntry::Integer, dimension0Max );
	params.addTextParam( "Initial Array Size (Dimension 1)",
                              BaseFUParamEntry::Integer, dimension1 );
        params.addTextParam( "Maximum Array Size (Dimension 1)",
                              BaseFUParamEntry::Integer, dimension1Max );
  	params.addTextParam( "Initial Cell Value",
                              BaseFUParamEntry::Real, initVal  );

	ParamConfig pc( Pablo::TopLevel(), params, getName() );

	BaseFUParamEntry& sz0Entry = params.getEntry( 
					   "Initial Array Size (Dimension 0)" );
	BaseFUParamEntry& sz0maxEntry = params.getEntry( 
					   "Maximum Array Size (Dimension 0)" );
 	BaseFUParamEntry& sz1Entry = params.getEntry( 
					   "Initial Array Size (Dimension 1)" );
        BaseFUParamEntry& sz1maxEntry = params.getEntry( 
					   "Maximum Array Size (Dimension 1)" );
        BaseFUParamEntry& initialEntry = params.getEntry( 
					   "Initial Cell Value" );
	
	pc.run();

	if ( sz0Entry.valueIsValid() ) {
	    dimension0 = sz0Entry.getValue().getInteger();
	} else {
	    warning( "\nInvalid array size (dimension 0). Defaulting to 8.");
	    dimension0 = 8;
	}

  	if ( sz1Entry.valueIsValid() ) {
            dimension1 = sz1Entry.getValue().getInteger();
        } else {
            warning( "\nInvalid array size (dimension 1). Defaulting to 8.");
	    dimension1 = 8;
        }

	if ( sz0maxEntry.valueIsValid() ) {
	    dimension0Max = sz0maxEntry.getValue().getInteger();
	} else {
	    warning( "\nInvalid maximum size (dimension 0). Defaulting to 64.");
	    dimension0Max = 64;
	}

 	if ( sz1maxEntry.valueIsValid() ) {
            dimension1Max = sz1maxEntry.getValue().getInteger();
        } else {
	    warning( "\nInvalid maximum size (dimension 1). Defaulting to 64.");
	    dimension1Max = 64;
        }

	if ( initialEntry.valueIsValid() ) {
            initVal = initialEntry.getValue().getReal();
	    initialValue = initVal;
        } else {
            warning( "\nInvalid initial cell value - defaulting to 0.");
	    initialValue = (double)0.0;
        }
		
	if ( dimension0 > dimension0Max ) {
	    dimension0 = dimension0Max;
	}
	if ( dimension1 > dimension1Max ) {
            dimension1 = dimension1Max;
        }

	int newDimSize[2];
	newDimSize[0] = dimension0;
	newDimSize[1] = dimension1;
	Array *outputArrayP = (Array *)array;
	outputArrayP->setDimSizes( newDimSize );

	int i;
	for ( i = 0; i < outputArrayP->getCellCount(); i++ ) {
	   outputArrayP->setTheCellValue( i, initialValue );
	}

}

void                            /* virtual */
SynthesizeArray::configureOperation()
{
        int opId = -1;
        FUParams params;

        params.addRadioButtonsParam( "Action", BaseFUParamEntry::Integer, 0,
                                     SynthesisActionList );
        params.addRadioButtonsParam( "Operation (Used only for Update Action)",
                                     BaseFUParamEntry::Integer, 0,
                                     SynthesisOpNameList );
        ParamConfig pc( Pablo::TopLevel(), params, getName() );

        BaseFUParamEntry& actionEntry = params.getEntry( "Action");
        BaseFUParamEntry& typeEntry = params.getEntry(
                                     "Operation (Used only for Update Action)");

        if ( actionOption != -1 ) {
            actionEntry.setValue( actionOption );
        }
        if ( myOperatorNumber != -1 ) {
            typeEntry.setValue( myOperatorNumber );
        }
        pc.run();

        if ( actionEntry.valueIsValid() ) {
            actionOption = actionEntry.getValue().getInteger();
        } else {
            warning( "\nInvalid action - defaulting to Replace." );
            actionOption = REPLACE;
        }

        if ( typeEntry.valueIsValid() ) {
            opId = typeEntry.getValue().getInteger();
        } else {
            if ( actionOption == UPDATE ) {
                warning( "\nInvalid operation - defaulting to Addition." );
            }
            opId = ADDITION;
        }
        _setOperator( opId );

}

FunctionalUnit *		/* virtual */
SynthesizeArray::copy()
{
	SynthesizeArray *copy = new SynthesizeArray();
	return copy;
}

void 				/* virtual */
SynthesizeArray::init()
{
        /*
         * The SynthesisActionList and SynthesisOpNameList are static so
         * we only set them the first time an instance of this class
         * is created
         */
        if ( SynthesisActionList.isEmpty() ) {
            SynthesisActionList.addElement( "Replace" );
            SynthesisActionList.addElement( "Update" );
        }

        if ( SynthesisOpNameList.isEmpty() ) {
            SynthesisOpNameList.addElement( "Addition" );
            SynthesisOpNameList.addElement( "Subtraction" );
            SynthesisOpNameList.addElement( "Multiplication" );
            SynthesisOpNameList.addElement( "Division" );
            SynthesisOpNameList.addElement( "Minimum" );
            SynthesisOpNameList.addElement( "Maximum" );
            SynthesisOpNameList.addElement( "Count" );
        }

        /*
         * Continue by clearing up any 'leftovers' that will be around
         * if init() isn't being called for the first time.  The goal is
         * to start with a 'virgin' FU.
         */
        if ( index0 != NULL ) {
            delete index0;
        }
        if ( index1 != NULL ) {
            delete index1;
        }
        if ( cellValue != NULL ) {
            delete cellValue;
        }
        if ( outputValue != NULL ) {
            delete outputValue;
        }
        if ( myOperator != NULL ) {
            delete myOperator;
            myOperator = NULL;
        }

        /*
         * Here's the real code to initialize the FU
         */
        myOperatorNumber = -1;
        actionOption = -1;
	isReady = FALSE_;
	isConfigured = FALSE_;

	index0 = new InputPort( "Index 0" );
	index0->addTraits( INTEGER, 0 );
	index0->addTraits( FLOAT, 0 );
	index0->addTraits( DOUBLE, 0 );
	_addInputPort( index0 );

	index1 = new InputPort( "Index 1" );
	index1->addTraits( INTEGER, 0 );
	index1->addTraits( FLOAT, 0 );
	index1->addTraits( DOUBLE, 0 );
	_addInputPort( index1 );

	cellValue = new InputPort( "Cell Value" );
	cellValue->addTraits( INTEGER, 0 );
	cellValue->addTraits( FLOAT, 0 );
	cellValue->addTraits( DOUBLE, 0 );
	_addInputPort( cellValue );

	outputValue = new OutputPort( "Array" );
	_addOutputPort( outputValue );

	array = new Array( DOUBLE, 2 );

}

Boolean_			/* virtual */
SynthesizeArray::inputTraitsValid()
{
	/* 
	 * SynthesizeArray will always output an array with
	 * Traits=( DOUBLE, 2 ) (i.e., a array) and our input ports
	 * have types that are restrictive enough to insure any
	 * bindings are valid without further verification.
	 */

	DataTraits arrayTraits( DOUBLE, 2 );
	outputValue->setTraits( arrayTraits );

	return TRUE_;
}
	
Boolean_			/* virtual */
SynthesizeArray::loadConfigFromFile( const CString& fileName )
{
        FILE *fp = fopen( fileName.getValue(), "r" );

        if ( fp == NULL ) {
            warning( "\nUnable to open %s:\n%s", fileName.getValue(),
                                                  errorString() );
        } else {
	    char     opName[256];
	    char     actionName[256];
	    int      op;
  	    double   initVal;

	    fscanf( fp, "%*[^\n]\n" );      // Comment line
            if ( fscanf( fp, "%d %d %d %d %lf %s %s\n",
			&dimension0, &dimension0Max, &dimension1, 
			&dimension1Max, &initVal, opName, actionName ) == 7 ) {

		initialValue = initVal;
	        int newDimSize[2];
	        newDimSize[0] = dimension0;
	        newDimSize[1] = dimension1;
	        Array *outputArrayP = (Array *) array; 
	        outputArrayP->setDimSizes( newDimSize );
	        for ( int i = 0; i < outputArrayP->getCellCount(); i++ ) {
	            outputArrayP->setTheCellValue(i, initialValue );
	        }
		isConfigured = TRUE_;

                if ( ( op = SynthesisOpNameList.lookup( opName ) ) < 0) {
                    warning( "Restored operation name (%s) is not valid",
                             opName );
                } else {
                    _setOperator( op );           // Sets isReady
                }

                if ( ( actionOption = SynthesisActionList.lookup( actionName ) )
                                            < 0 ) {
                    warning( "Restored action name (%s) is not valid",
                             actionName );
                    isReady = FALSE_;
                }

            } else {
                warning( "\nUnable to read configuration information from\n %s",
                          fileName.getValue() );
            }
            fclose( fp );
        }

        if ( isReady == FALSE_ ) {              // last chance to configure!
            configureOperation();
        }
        if ( isConfigured == FALSE ) {
            configure();
        }

        return ( isReady );
}

Boolean_ 			/* virtual */
SynthesizeArray::ready()
{
	return ( isReady );
}

void 				/* virtual */
SynthesizeArray::run(  Boolean_& errorFlag )
{
	Assert( index0->valueAvailable() );
	Assert( index1->valueAvailable() );
	Assert( cellValue->valueAvailable() );
	
	int idx0 = (int) index0->getValue();
	int idx1 = (int) index1->getValue();

	if ( ( 0 <= idx0 ) && ( idx0 < dimension0Max ) && 
	     ( 0 <= idx1 ) && ( idx1 < dimension1Max ) ) {

	   if ( ( idx0 >= dimension0 ) || ( idx1 >= dimension1 ) ) {
	      _resizeArray( idx0, idx1 );
	   }

  	   Array *outputArrayP = (Array *)array;

	   if ( actionOption == REPLACE ) {
	      outputArrayP->setCellValue( cellValue->getValue(), idx0, idx1 );
	      outputValue->setValue( array );
	      _writeOutput();
	   } else {
              myOperator->getBinaryResult(
				outputArrayP->getCellValue( idx0, idx1 ),
				cellValue->getValue(), resultV );
	      if ( resultV.isDefined() ) {
	          outputArrayP->setCellValue( resultV, idx0, idx1 );
	          outputValue->setValue( array );
	          _writeOutput();
	      } else {
		  errorFlag = TRUE_;
	      }
	   }

	} else {
	    warning( "Index out of range ( %d, %d )\n", idx0, idx1 );
	    errorFlag = TRUE_;
	}
}

Boolean_			/* virtual */
SynthesizeArray::saveConfigToFile( const CString& fileName ) const
{
        Boolean_ retCode;

        FILE *fp = fopen( fileName.getValue(), "w" );

        if ( fp == NULL ) {
            error( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
            retCode = FAILURE_;
        } else {
            fprintf( fp, 
		"# Dim0 Dim0Max Dim1 Dim1Max initialValue Operation Action\n" );
            fprintf( fp, "%d %d %d %d %lf %s %s\n", 
			 dimension0, dimension0Max,
			 dimension1, dimension1Max, (double)initialValue,
		(const char *)SynthesisOpNameList.getElement(myOperatorNumber),
		(const char *)SynthesisActionList.getElement(actionOption) );
            fclose( fp );
            retCode = SUCCESS_;
        }
        return retCode;
}

/*
 * 	Initialize the static data.   Only executed once.
 */
const char *const SynthesizeArray::MY_CLASS = "SynthesizeArray";
CStringObjList SynthesizeArray::SynthesisOpNameList;
CStringObjList SynthesizeArray::SynthesisActionList;


