/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * SynthesizeVector.cc - A functional unit for synthesizing data vector
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/SynthesizeVector.C,v 1.18 1994/03/15 16:45:49 aydt Exp $
 */

#include <string.h>

#include "SynthesizeVector.h"
#include "Addition.h"
#include "Subtraction.h"
#include "Multiplication.h"
#include "Division.h"
#include "Minimum.h"
#include "Maximum.h"
#include "BinaryCount.h"

#include "FUParams.h"
#include "ParamConfig.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "SystemErrors.h"

SynthesizeVector::SynthesizeVector()
{
	_setClassName( MY_CLASS );

	index = NULL;
	cellValue = NULL;
	outputValue = NULL;
	myOperator = NULL;
}

SynthesizeVector::~SynthesizeVector()
{
	delete index;
	delete cellValue;
	delete outputValue;
	delete myOperator;
}

void			
SynthesizeVector::_resizeVector( int newIndex )
{
	Assert( newIndex >= dimension );	// size should only grow

	int i;

	Array* newVectorP = new Array( DOUBLE, 1 );
	int newDimSize = newIndex + 1;
	newVectorP->setDimSizes( &newDimSize );

	Array *vectorP = (Array *)vector;

	for ( i = 0; i < dimension; i++ ) {
	    newVectorP->setTheCellValue( i, vectorP->getTheCellValue(i) );
	}
	for ( i = dimension; i < newDimSize; i++ ) {
	    newVectorP->setTheCellValue( i, initialValue );
	}

	vector = newVectorP;
	dimension = newDimSize;

	delete newVectorP;
}

void 
SynthesizeVector::_setOperator( const int opId )
{
	if ( myOperator != NULL ) {
  	    delete myOperator;
	}
  	myOperatorNumber = opId;

  	switch ( opId ) {
     	   case ADDITION:
              myOperator = new Addition;
              break;

     	   case SUBTRACTION:
              myOperator = new Subtraction;
              break;

     	   case MULTIPLICATION:
              myOperator = new Multiplication;
              break;

     	   case DIVISION:
              myOperator = new Division;
              break;

     	   case MINIMUM:
              myOperator = new Minimum;
              break;

     	   case MAXIMUM:
              myOperator = new Maximum;
              break;

	   case COUNT:
	      myOperator = new BinaryCount;
	      break;

     	   default:
              warning( 
		"\nOperator id %d does not correspond to any known operation",
	        opId );
              myOperatorNumber = -1;
              break;
  	}

  	if ( myOperatorNumber < 0 ) {
      	   isReady = FALSE_;
  	} else {
           isReady = TRUE_;
  	}
}

void				/* virtual */
SynthesizeVector::configure()
{
	double initVal;

	if ( isConfigured ) {
	    initVal = (double)initialValue ;
	} else {
	    dimension = 8;
	    dimensionMax = 64;
	    initVal = (double)0.0;
	    isConfigured = TRUE_;
	}

	FUParams params;

	/* 
	 * I use a zero-entry button list to position a label which tells
	 * us what was selected in configureOperation() - don't want to 
	 * allow changes at this point.  ICK! (RAA) 
	 */
	static char buf[128];
	if ( actionOption == REPLACE ) {
	    sprintf( buf, "Action is %s", 
	       (const char *)SynthesisActionList.getElement( actionOption ) );
	} else {
	    sprintf( buf, "Action is %s; Operation is %s",
	       (const char *)SynthesisActionList.getElement(actionOption),
	       (const char *)SynthesisOpNameList.getElement(myOperatorNumber) );
	}

	CStringObjList nullList(0);
 	params.addRadioButtonsParam( buf, BaseFUParamEntry::Integer, 0, 
				     nullList );
	params.addTextParam( "Initial Vector Size",
			      BaseFUParamEntry::Integer, dimension );
	params.addTextParam( "Maximum Vector Size", 
			      BaseFUParamEntry::Integer, dimensionMax );
  	params.addTextParam( "Initial Cell Value",
                              BaseFUParamEntry::Real, initVal );

	ParamConfig pc( Pablo::TopLevel(), params, getName() );

	BaseFUParamEntry& szEntry = params.getEntry( "Initial Vector Size" );
	BaseFUParamEntry& szmaxEntry = params.getEntry( "Maximum Vector Size" );
	BaseFUParamEntry& initialEntry = params.getEntry( "Initial Cell Value");

	pc.run();

	if ( szEntry.valueIsValid() ) {
	    dimension = szEntry.getValue().getInteger();
	} else {
	    warning( "\nInvalid vector size - defaulting to 8." );
	    dimension = 8;
	}

	if ( szmaxEntry.valueIsValid() ) {
	    dimensionMax = szmaxEntry.getValue().getInteger();
	} else {
	    warning( "\nInvalid maximum vector size - defaulting to 64." );
	    dimensionMax = 64;
	}

	if ( initialEntry.valueIsValid() ) {
            initVal = initialEntry.getValue().getReal();
	    initialValue = initVal;
        } else {
            warning( "\nInvalid initial cell value - defaulting to 0." );
	    initialValue = (double)0.0;
        }
		
	if ( dimension > dimensionMax ) {
	    dimension = dimensionMax;
	}

	Array *outputVectorP = (Array *)vector;
	outputVectorP->setDimSizes( &dimension );

	for ( int i = 0; i < dimension; i++ ) {
	   outputVectorP->setTheCellValue( i, initialValue );
	}

}

void				/* virtual */
SynthesizeVector::configureOperation()
{
	int opId = -1;
	FUParams params;

 	params.addRadioButtonsParam( "Action", BaseFUParamEntry::Integer, 0,
				     SynthesisActionList );
	params.addRadioButtonsParam( "Operation (Used only for Update Action)",
				     BaseFUParamEntry::Integer, 0,
				     SynthesisOpNameList );
	ParamConfig pc( Pablo::TopLevel(), params, getName() );

 	BaseFUParamEntry& actionEntry = params.getEntry( "Action");
 	BaseFUParamEntry& typeEntry = params.getEntry( 
				     "Operation (Used only for Update Action)");

        if ( actionOption != -1 ) {
            actionEntry.setValue( actionOption );
        }
        if ( myOperatorNumber != -1 ) {
            typeEntry.setValue( myOperatorNumber );
        }
	pc.run();

	if ( actionEntry.valueIsValid() ) {
	    actionOption = actionEntry.getValue().getInteger();
	} else {
	    warning( "\nInvalid action - defaulting to Replace." );
	    actionOption = REPLACE;
	}

        if ( typeEntry.valueIsValid() ) {
            opId = typeEntry.getValue().getInteger();
        } else {
	    if ( actionOption == UPDATE ) {
                warning( "\nInvalid operation - defaulting to Addition." );
	    }
	    opId = ADDITION;
        }
	_setOperator( opId );

}

FunctionalUnit *		/* virtual */
SynthesizeVector::copy()
{
	SynthesizeVector *copy = new SynthesizeVector();
	return copy;
}

void 				/* virtual */
SynthesizeVector::init()
{
	/*
	 * The SynthesisActionList and SynthesisOpNameList are static so
	 * we only set them the first time an instance of this class
	 * is created 
	 */
	if ( SynthesisActionList.isEmpty() ) {
	    SynthesisActionList.addElement( "Replace" );
	    SynthesisActionList.addElement( "Update" );
	}

	if ( SynthesisOpNameList.isEmpty() ) {
	    SynthesisOpNameList.addElement( "Addition" );
	    SynthesisOpNameList.addElement( "Subtraction" );
	    SynthesisOpNameList.addElement( "Multiplication" );
	    SynthesisOpNameList.addElement( "Division" );
	    SynthesisOpNameList.addElement( "Minimum" );
	    SynthesisOpNameList.addElement( "Maximum" );
	    SynthesisOpNameList.addElement( "Count" );
	}

	/* 
	 * Continue by clearing up any 'leftovers' that will be around
	 * if init() isn't being called for the first time.  The goal is
	 * to start with a 'virgin' FU.
	 */
	if ( index != NULL ) {
	    delete index;
	}
	if ( cellValue != NULL ) {
	    delete cellValue;
	}
	if ( outputValue != NULL ) {
	    delete outputValue;
	}
	if ( myOperator != NULL ) {
	    delete myOperator;
	    myOperator = NULL;
	}

	/*
	 * Here's the real code to initialize the FU
	 */
	myOperatorNumber = -1;
	actionOption = -1;
	isReady = FALSE_;
	isConfigured = FALSE_;

	index = new InputPort( "Index" );
	index->addTraits( INTEGER, 0 );
	index->addTraits( FLOAT, 0 );
	index->addTraits( DOUBLE, 0 );
	_addInputPort( index );

	cellValue = new InputPort( "Cell Value" );
	cellValue->addTraits( INTEGER, 0 );
	cellValue->addTraits( FLOAT, 0 );
	cellValue->addTraits( DOUBLE, 0 );
	_addInputPort( cellValue );

	outputValue = new OutputPort( "Vector" );
	_addOutputPort( outputValue );

	vector = new Array( DOUBLE, 1 );
}


Boolean_			/* virtual */
SynthesizeVector::inputTraitsValid()
{
	/* 
	 * SynthesizeVector will always output an array with
	 * Traits=( DOUBLE, 1 ) (i.e., a vector) and our input ports
	 * have types that are restrictive enough to insure any
	 * bindings are valid without further verification.
	 */

	DataTraits arrayTraits( DOUBLE, 1 );
	outputValue->setTraits( arrayTraits );

	return TRUE_;
}
	
Boolean_			/* virtual */
SynthesizeVector::loadConfigFromFile( const CString& fileName )
{
        FILE *fp = fopen( fileName.getValue(), "r" );

        if ( fp == NULL ) {
            warning( "\nUnable to open %s:\n%s", fileName.getValue(),
                                                  errorString() );
        } else {
	    char	opName[256];
	    char	actionName[256];
	    int		op;
	    double	initVal;

	    fscanf( fp, "%*[^\n]\n" );	   // Comment line

            if ( fscanf( fp, "%d %d %lf %s %s\n",
			 &dimension, &dimensionMax, &initVal,
			 opName, actionName ) == 5 ) {

		initialValue = initVal;
                Array *outputVectorP = (Array *)vector;
	        outputVectorP->setDimSizes( &dimension );
	        for ( int i = 0; i < dimension; i++ ) {
   	            outputVectorP->setTheCellValue( i, initialValue );
 	        }
		isConfigured = TRUE_;

                if ( ( op = SynthesisOpNameList.lookup( opName ) ) < 0) {
                    warning( "Restored operation name (%s) is not valid", 
			     opName );
                } else {
                    _setOperator( op );		  // Sets isReady
                }

	        if ( ( actionOption = SynthesisActionList.lookup( actionName ) )
					    < 0 ) {
		    warning( "Restored action name (%s) is not valid", 
		 	     actionName );
		    isReady = FALSE_;
		}

            } else {
                warning( "\nUnable to read configuration information from\n%s",
                          fileName.getValue() );
            }
            fclose( fp );
        }

	if ( isReady == FALSE_ ) {		// last chance to configure!
	    configureOperation();	
	}
	if ( isConfigured == FALSE ) {
	    configure();		
	}

	return ( isReady );
}

Boolean_ 			/* virtual */
SynthesizeVector::ready()
{
	return ( isReady );
}

void 				/* virtual */
SynthesizeVector::run( Boolean_& errorFlag )
{
	Assert( index->valueAvailable() );
	Assert( cellValue->valueAvailable() );
	
	int idx = (int) index->getValue();

	if ( ( 0 <= idx ) && ( idx < dimensionMax ) ) {
	   if ( idx >= dimension ) {
	      _resizeVector( idx );
	   }

  	   Array *outputVectorP = (Array *)vector;

	   if ( actionOption == REPLACE ) {
	      outputVectorP->setCellValue( cellValue->getValue(), idx );
	      outputValue->setValue( vector );
	      _writeOutput();
	   } else {
              myOperator->getBinaryResult( outputVectorP->getCellValue( idx ),
				cellValue->getValue(), resultV );
	      if ( resultV.isDefined() ) {
	          outputVectorP->setCellValue( resultV, idx );
	          outputValue->setValue( vector );
	          _writeOutput();
	      } else {
		  errorFlag = TRUE_;
	      }
	   }

	} else {
	   warning( "Index out of range ( %d )\n", idx );
	   errorFlag = TRUE_;
	}
}

Boolean_			/* virtual */
SynthesizeVector::saveConfigToFile( const CString& fileName ) const
{
        Boolean_ retCode;

        FILE *fp = fopen( fileName.getValue(), "w" );

        if ( fp == NULL ) {
            error( "\nUnable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
            retCode = FAILURE_;
        } else {
            fprintf( fp, "# Dim DimMax initialValue Operation Action\n" );
            fprintf( fp, "%d %d %lf %s %s\n", 
	      dimension, dimensionMax, (double)initialValue,
	      (const char *) SynthesisOpNameList.getElement( myOperatorNumber ),
	      (const char *) SynthesisActionList.getElement( actionOption ) );

            fclose( fp );
            retCode = SUCCESS_;
        }
        return retCode;
}

/*
 * 	Initialize the static data.   Only executed once.
 */
const char *const SynthesizeVector::MY_CLASS = "SynthesizeVector";
CStringObjList SynthesizeVector::SynthesisOpNameList;
CStringObjList SynthesizeVector::SynthesisActionList;
