/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Authors: Philip C. Roth (proth@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/**************************************************************
 * UtilDisplay.C
 * 
 * Class implementation of base class for Utilization displays
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/UtilDisplay.C,v 1.14 1994/02/25 04:33:18 aydt Exp $
 *
 **************************************************************/
#include <string.h>
#include <strstream.h>

#include <Xm/Xm.h>
#include <Xm/DrawingA.h>
#include <Xm/Form.h>
#include <Xm/ScrolledW.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>
#include <Xm/PushB.h>
#include <Xm/Label.h>
#include <Xm/Text.h>
#include <Xm/PanedW.h>

#include "PabloMainInterface.h"

#include "LabelUtilities.h"
#include "Selection.h"
#include "UtilDisplay.h"
#include "UtilDisplayManager.h"
#include "UtilFU.h"



/*
 * #defines for sizes of arrays
 * g++ allows 'const int' to do this, but CC doesn't
 */
#define MAX_LABEL_SIZE	16



/* 
 * static data 
 */
const char *UtilDisplay::averageLabelString = "avg";
const char *UtilDisplay::minimumLabelString = "min";
const char *UtilDisplay::maximumLabelString = "max";

const Dimension UtilDisplay::defSizeWidthLimit = 400;
const Dimension UtilDisplay::defSizeHeightLimit = 400;
const int UtilDisplay::notDisplayed = -1;

int UtilDisplay::waitingSelDisplays = 0;   // count of selection displays up






UtilDisplay::UtilDisplay( char *name, UtilFU *ufu, Widget par )
{
	int i;


	/*
	 * initialize other object data
	 */
	parent = par;
	fuParent = ufu;

	myName = NULL;
	setName( name );

	currSel = NULL;			// no initial selection
	numIndividuals = 0;
	map = NULL;
	individualUtils = NULL;

	for( i = 0; i < numAggregates; i++ ) {
		aggregateUtils[i] = NULL;
	}

	dialog = _makeDisplay( par );
	setDialogTitle( name );

	/* set default labels */
	horiLabel = vertLabel = NULL;

	minWidth = minHeight = 0;
}


UtilDisplay::~UtilDisplay( void )
{

	if( currSel ) {			// delete Selection list
		delete currSel;
	}

	cleanupPerfWidgets();

	delete[] myName;
}




/*
 * _insertHere()
 *
 * Indicates where to insert 'this' in a given list of UtilDisplays
 */
/* virtual */
Boolean_
UtilDisplay::_insertHere( ListNode* /* node */)
{
	/* the list is unordered, so the front of the list is fine */
	return TRUE_;
}



/*
 * _handleSelectCB()
 *
 * Displays a Dialog reporting the utilization information about the
 * selected performance widget
 */
void
UtilDisplay::_handleSelectCB( UtilFUData *data, int val )
{
	char buf[MAX_LABEL_SIZE];
	ostrstream ostr( buf, MAX_LABEL_SIZE );


	/* stop the main interface from executing */
	SignalStop();			// indicate one more will stop
	Pablo::MainInterface()->runStop();


	/* put up a dialog */
	Widget dlog = XmCreateFormDialog( dialog, "callbackDialog", NULL, 0 );
	data->dialog = dlog;

	Widget pane = XtVaCreateWidget( "pane",
			xmPanedWindowWidgetClass, dlog,
			XmNsashHeight,	1,
			XmNsashWidth,	1,
			NULL );

	/*
	 * create control area
	 */
	Widget contform = XtVaCreateWidget( "contform",
			xmFormWidgetClass, pane,
			NULL );


	Widget uval = XtVaCreateManagedWidget( "utilval",
			xmLabelWidgetClass, contform,
			NULL );

	Widget utext  = XtVaCreateManagedWidget( "utiltext",
			xmTextWidgetClass, contform,
			XmNtopWidget,	uval,
			NULL );
	XmTextSetEditable( utext, False );
	ostr << val << "%" << '\0';
	XmTextSetString( utext, buf );

	XtManageChild( contform );


	/*
	 * create action area
	 */
	Widget actform = XtVaCreateWidget( "actform",
			xmFormWidgetClass, pane,
			XmNfractionBase,	4,
			NULL );

	Widget okBut = XtVaCreateManagedWidget( "okButton",
			xmPushButtonWidgetClass, actform,
			XmNshowAsDefault,	True,
			XmNdefaultButtonShadowThickness,	1,
			NULL );
	/* 
	 * add a callback for the OK button to check whether to restart
	 */
	XtAddCallback( okBut, XmNactivateCallback,
			(XtCallbackProc)selectDoneCB, data );

	Widget helpBut = XtVaCreateManagedWidget( "helpButton",
			xmPushButtonWidgetClass, actform,
			XmNshowAsDefault,	False,
			XmNdefaultButtonShadowThickness,	1,
			XmNsensitive,	False,
			NULL );

	XtManageChild( actform );
		

        /* fix Action area at its initial height */
	Dimension height;
	XtManageChild( actform );
	XtVaGetValues( okBut,
		XmNheight,      &height,
		NULL );
	XtVaSetValues( actform,
		XmNpaneMaximum, height,
		XmNpaneMinimum, height,
		NULL );

	XtManageChild( pane );
	XtManageChild( dlog );
}


/* virtual */
Widget
UtilDisplay::_makeDisplay( Widget par )
{
	Widget hsbar, vsbar;


	Widget dialog = XmCreateFormDialog( par, "displayDialog", NULL, 0 );

	/*
	 * Create a DrawingArea widget so that resize events are reported
	 */
	Widget da = XtVaCreateWidget( "da",
			xmDrawingAreaWidgetClass, dialog,
			XmNtopAttachment,	XmATTACH_FORM,
			XmNleftAttachment,	XmATTACH_FORM,
			XmNrightAttachment,	XmATTACH_FORM,
			XmNbottomAttachment,	XmATTACH_FORM,
			XmNmarginHeight,	0,
			XmNmarginWidth,		0,
			NULL );

	/*
	 * Create a ScrolledWindow widget to handle display of performance
	 * widgets
	 */
	Widget sw = XtVaCreateWidget( "scrolledWindow",
			xmScrolledWindowWidgetClass, da,
			NULL );
	XtVaGetValues( sw,
			XmNverticalScrollBar,	&vsbar,
			XmNhorizontalScrollBar,	&hsbar,
			NULL );
	XtVaGetValues( vsbar,
			XtNwidth,	&sbarWidth,
			NULL );
	XtVaGetValues( hsbar,
			XtNheight,	&sbarHeight,
			NULL );
		

	workWindow = XtVaCreateWidget( "displayForm",
			xmFormWidgetClass, sw,
			NULL );

	/* form labels */
	vlabel = XtVaCreateManagedWidget( "vlabel",
			xmLabelWidgetClass,	workWindow,
			XmNlabelString,		XmStringCreateLtoR( "",
						XmSTRING_DEFAULT_CHARSET ),
			NULL );

	hlabel = XtVaCreateManagedWidget( "hlabel",
			xmLabelWidgetClass,	workWindow,
			XmNleftWidget,		vlabel,
			XmNlabelString,		XmStringCreateLtoR( "",
						XmSTRING_DEFAULT_CHARSET ),
			NULL );

	rc = XtVaCreateManagedWidget( "displayRowCol",
			xmRowColumnWidgetClass,	workWindow,
			XmNleftWidget,		vlabel,
			XmNbottomWidget,	hlabel,
			XmNtopAttachment,	XmATTACH_FORM,
			XmNrightAttachment,	XmATTACH_FORM,
			NULL );

	/*
	 * set the ScrolledWindow's workWindow
	 */
	XtVaSetValues( sw,
			XmNworkWindow,	workWindow,
			NULL );


	/*
	 * add callback for resize events to the DrawingArea
	 */
	UtilDisplayData* data = new UtilDisplayData;
	data->disp = this;
	data->w = sw;
	XtAddCallback( da, XmNresizeCallback, (XtCallbackProc)resizeCB, data );

	XtManageChild( workWindow );
	XtManageChild( sw );
	XtManageChild( da );


	return dialog;
}




/*
 * _makeMap()
 *
 * Creates an associative mapping between the individual PEs selected
 * and their position in the arrays.
 */
void
UtilDisplay::_makeMap( Selection *sel, int& size )
{
	int hi, lo;
	int val, i;


	size = 0;

	/* check that there is a selection */
	if( !sel ) {
		map = NULL;
		return;
	}

	/* determine the selection size */
	Selection* curr = sel;
	while( curr ) {
		curr->getRange( lo, hi );
		size += ( hi - lo + 1 );
		curr = (Selection*)curr->getNextInList();
	}

	/* get space for the map array */
	delete[] map;
	map = new int[size];

	/* form the mapping */
	curr = sel;
	curr->getRange( lo, hi );
	for( i = 0, val = lo; i < size; i++, val++ ) {

		/* store the map */
		map[i] = val;

		if( val == hi ) {
			curr = (Selection*)curr->getNextInList();
			if( curr ) {

				/* it will be incremented by loop control */
				curr->getRange( lo, hi );
				val = lo - 1;
			}
		}
	}
}



/*
 * _indexIntoMap()
 *
 * Given a PE number, returns its index into the map and other arrays
 */
int
UtilDisplay::_indexIntoMap( int proc )
{
	int j;
	 
	for( j = 0; j < numIndividuals; j++ ) {
		 if( map[j] == proc ) {
			 return j;
		 }
	}
	return notDisplayed;
}



/*
 * cleanupPerfWidgets()
 *
 * Deallocates memory used by the performance widgets for this
 * UtilDisplay
 */
void
UtilDisplay::cleanupPerfWidgets( void )
{
	int i;


	switch( getDisplayedType() ) {
		case IndividualsShown:
			for( i = 0; i < numIndividuals; i++ ) {
				XtUnmanageChild( XtParent( individualUtils[i]));
				XtDestroyWidget( XtParent( individualUtils[i]));
			}
			XtFree( (char*)individualUtils );

			delete[] map;
			break;

		case AggregateShown:
			for( i = 0; i < numAggregates; i++ ) {
				if( aggregateUtils[i] ) {
					XtDestroyWidget( 
						XtParent( aggregateUtils[i]));
				}
			}
			break;
	}
}



/*
 * getDisplayByName()
 *
 * Given the name of a display, find it ( if it exists ) in the
 * list headed by 'this'
 */
UtilDisplay *
UtilDisplay::getDisplayByName( char *name )
{
        UtilDisplay *curr;
	char *dispName;

	curr = this;
	dispName = getName();

	/* find the display named name, if it exists in the list */
	while( curr && strcmp( dispName, name ) ) {
		delete[] dispName;

		curr = (UtilDisplay *)curr->next;	// advance to next

		if( curr ) {
			dispName = curr->getName();
		}
	}
	delete[] dispName;

	return curr;
}



/*
 * (get|set)HoriLabel()
 *
 * Manipulate the label string that this UtilDisplay uses for its
 * horizontal label
 */
char *
UtilDisplay::getHoriLabel( void ) const
{
	char *labl = NULL;


	if( horiLabel ) {
		labl = new char[ strlen( horiLabel ) + 1 ];
		strcpy( labl, horiLabel );
	}

	return labl;
}

void
UtilDisplay::setHoriLabel( char *labl )
{
	XmString xmstr;


	delete[] horiLabel;

	horiLabel = new char[ strlen( labl ) + 1 ];
	strcpy( horiLabel, labl );

	if( labl ) {
		xmstr = XmStringCreateLtoR( labl, XmSTRING_DEFAULT_CHARSET );
	}
	else {
		xmstr = XmStringCreateLtoR( "", XmSTRING_DEFAULT_CHARSET );
	}

	XtVaSetValues( hlabel,
		XmNlabelString,	xmstr,
		NULL );
}





/*
 * (get|set)Name()
 *
 * Manipulate the character string that this UtilDisplay uses for its name
 */
char *
UtilDisplay::getName( void ) const
{
	char *name;

	name = new char[ strlen( myName ) + 1 ];
	strcpy( name, myName );

	return name;
}

void
UtilDisplay::setName( char *name )
{
	delete[] myName;
	myName = new char[ strlen(name) + 1 ];

	strcpy( myName, name );
}




/*
 * (get|set)VertLabel()
 *
 * Manipulate the character string that this UtilDisplay uses as
 * its vertical label
 */
void
UtilDisplay::setVertLabel( char *labl )
{
	LabelUtilities lu;
	XmString xmstr;


	delete[] vertLabel;

	vertLabel = new char[ strlen(labl) + 1 ];
	strcpy( vertLabel, labl );

	if( labl ) {
		xmstr = lu.verticalizeLabel( labl );
	}
	else {
		xmstr = XmStringCreateLtoR( "", XmSTRING_DEFAULT_CHARSET );
	}

	XtVaSetValues( vlabel,
		XmNlabelString,	xmstr,
		NULL );
}

char *
UtilDisplay::getVertLabel( void ) const
{
	char *labl = NULL;

	if( vertLabel ) {
		labl = new char[ strlen( vertLabel ) + 1 ];
		strcpy( labl, vertLabel );
	}

	return labl;
}



/*
 * initialResizeDialog()
 *
 * Resizes the Utilization display window if needed/possible.
 * Setting the size according to the current size of the RowColumn does
 * not work correctly; it gives too much space in the RowColumn's major
 * dimension.  To correct this, the major dimension is sized according
 * to the function arguments.
 */ 
void
UtilDisplay::initialResizeDialog( void )
{
	Dimension width = minWidth + sbarWidth + 20;
	width = ( width > defSizeWidthLimit ? defSizeWidthLimit : width );

	Dimension height = minHeight + sbarHeight + 20;
	height = ( height > defSizeHeightLimit ? defSizeHeightLimit : height );

	XtVaSetValues( dialog,
		XmNwidth,	width,
		XmNheight,	height,
		NULL );
}



/*
 * selectCB()
 *
 * The method used as a callback function for the XtNselect callback of
 * the performance widgets used by this UtilDisplay.
 */
/* static */
void
UtilDisplay::selectCB( Widget /* w */, UtilDisplay *disp, XtPointer cd )
{
	int value = (int) cd;

	UtilFUData *data = new UtilFUData;

	data->ufu = disp->getFUParent();
	data->dialog = (Widget)NULL;

	disp->_handleSelectCB( data, value );
}


/*
 * selectDoneCB
 *
 * Callback which is executed when a selection is completed ( user presses
 * the ok button ).  The callback restarts execution if no other 
 * selection dialogs exist.
 */
/* static */
void
UtilDisplay::selectDoneCB( Widget w, UtilFUData* data, XtPointer /*cd*/ )
{

	int remaining = WaitStop();		// indicate that one less
						// display is up
	Boolean_ started = data->ufu->StartedRunning();

	/* pop down the dialog */
	UtilFU::cancelCB( w, data, NULL );

	/* if there are none left, restart the main interface */
	if( !remaining && started ) {
		Pablo::MainInterface()->runRun();
	}

}




/*
 * uniqueDisplayName()
 *
 * Returns TRUE_ if the given name is unique in the list of UtilDisplays
 * which starts at 'this,' and FALSE_ otherwise.
 */
Boolean_
UtilDisplay::uniqueDisplayName( char *name )
{

	/* find a display with the same name, if it exists */
	UtilDisplay *match = getDisplayByName( name );
	return ( match ? FALSE_ : TRUE_ );
}







/*
 * resizeCB
 *
 * Method used as callback for DrawingArea resize events.
 */
/* static */
void
UtilDisplay::resizeCB( Widget w, UtilDisplayData* data, XtPointer /* cb */  )
{
	Dimension width, height;	// new dimensions of dialog


	/*
	 * Resize the ScrolledWindow widget to fit its new dialog
	 */
	XtVaGetValues( w, 
		XtNwidth,	&width,
		XtNheight,	&height,
		NULL );
	XtVaSetValues( data->w,
		XtNwidth,	width,
		XtNheight,	height,
		NULL );

	/*
	 * Resize the data showing through the ScrolledWindow
	 */
	data->disp->_handleResize( data->w );
}



/*
 * setDialogTitle
 *
 * Sets the dialog's title to <name>:<type>
 */
void
UtilDisplay::setDialogTitle( const char* name )
{
	char* title = new char[ strlen( name ) + 1 ];
	strcpy( title, name );

	XtVaSetValues( XtParent( getDialog() ),
		XmNtitle,	(XtArgVal)title,
		NULL );
}



/* virtual */
void
UtilDisplay::printOn( ostream& s ) const
{
	char *str;
	Position x, y;
	Dimension width, height;

	XtVaGetValues( getDialog(),
		XmNx,		&x,
		XmNy,		&y,
		XmNwidth,	&width,
		XmNheight,	&height,
		NULL );

	/* dump name */
	s << "# name \n";
	str = getName();
	s << strlen( str ) << ' ' << str << '\n';
	delete[] str;

	/* dump type */
	s << "# type\n";
	s << strlen( getType() ) << ' ';
	s << getType() << '\n';

	/* dump screen chax */
	s << "# x y width height\n";
	s << (int)x << ' ' << (int)y << ' ' <<
	     (int)width << ' ' << (int)height << '\n';

	/* dump utilizations shown */
	s << "# utilizations shown\n";
	s << getDisplayedType() << '\n';

	/* dump labels */
	s << "# horizontal label\n";
	str = getHoriLabel();
	if( str ) {
		s << strlen( str ) << ' ' << str << '\n';
	}
	else {
		s << "0\n";
	}
	delete[] str;

	s << "# vertical label\n";
	str = getVertLabel();
	if( str ) {
		s << strlen( str ) << ' ' << str << '\n';
	}
	else {
		s << "0\n";
	}
	delete[] str;

	/* dump current selection of PEs */
	Selection::printList( s, getSelection() );
	s << '\n';
}

