/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Authors: Philip C. Roth (proth@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*******************************************************************
 *
 * UtilFU.C
 *
 * Definition of class methods for Pablo Utilization Functional Unit class.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/UtilFU.C,v 1.16 1994/03/15 16:42:11 aydt Exp $
 *
 *******************************************************************/
#include <stream.h>
#include <fstream.h>
#include <strstream.h>
#include <stdlib.h>
#include <stdio.h>
#include <Xm/Xm.h>
#include <Xm/DialogS.h>
#include <Xm/DrawingA.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/List.h>
#include <Xm/MessageB.h>
#include <Xm/PanedW.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/Separator.h>
#include <Xm/ScrolledW.h>
#include <Xm/Text.h>
#include <Xm/ToggleBG.h>
#include "Defines.h"
#include "FUParams.h"
#include "InputPort.h"
#include "PabloHelpSystem.h"
#include "ParamConfig.h"
#include "SystemErrors.h"


#include "EventNode.h"
#include "NumberLine.h"
#include "Selection.h"
#include "UtilFU.h"
#include "UtilDisplay.h"
#include "UtilDisplayManager.h"





/* 
 * static data
 */
const char *const UtilFU::MY_CLASS = "UtilFU";
CStringObjList UtilFU::initialStatusList;

const int UtilFU::max_name_len = 80;
DefaultUtilFUData UtilFU::defaultData;
Boolean_ UtilFU::startedRunning = FALSE_;


/*
 * X Resource List
 * for parameters which are initialized from an X resource file
 */
#define	XmNutilNumProcessors		"utilNumProcessors"
#define	XmCUtilNumProcessors		"UtilNumProcessors"

#define	XmNutilWindowSize		"utilWindowSize"
#define	XmCUtilWindowSize		"UtilWindowSize"

#define	XmNutilWindowIncr		"utilWindowIncr"
#define	XmCUtilWindowIncr		"UtilWindowIncr"

#define	XmNutilInitialStateIsBusy	"utilInitialStateIsBusy"
#define	XmCUtilInitialStateIsBusy	"UtilInitialStateIsBusy"

#define	XmNutilHorizontalLabel		"utilHorizontalLabel"
#define	XmCUtilHorizontalLabel		"UtilHorizontalLabel"

#define	XmNutilVerticalLabel		"utilVerticalLabel"
#define	XmCUtilVerticalLabel		"UtilVerticalLabel"


static XtResource resources[] = {
	{
		XmNutilNumProcessors,
		XmCUtilNumProcessors,
		XmRInt,
		sizeof( int ),
		XtOffset( DefaultUtilFUDataPtr, numProcessors ),
		XmRImmediate,
		(XtPointer)32,
	},
	{
		XmNutilWindowSize,
		XmCUtilWindowSize,
		XmRFloat,
		sizeof( float ),
		XtOffset( DefaultUtilFUDataPtr, windowSize ),
		XmRImmediate,
		(XtPointer)0,
	},
	{
		XmNutilWindowIncr,
		XmCUtilWindowIncr,
		XmRFloat,
		sizeof( float ),
		XtOffset( DefaultUtilFUDataPtr, windowIncr ),
		XmRImmediate,
		(XtPointer)0,
	},
	{
		XmNutilInitialStateIsBusy,
		XmCUtilInitialStateIsBusy,
		XmRBoolean,
		sizeof( Boolean ),
		XtOffset( DefaultUtilFUDataPtr, initialStateIsBusy ),
		XmRImmediate,
		(XtPointer)True,
	},
	{
		XmNutilHorizontalLabel,
		XmCUtilHorizontalLabel,
		XmRString,
		sizeof( char* ),
		XtOffset( DefaultUtilFUDataPtr, horiLabel ),
		XmRImmediate,
		(XtPointer)NULL,
	},
	{
		XmNutilVerticalLabel,
		XmCUtilVerticalLabel,
		XmRString,
		sizeof( char* ),
		XtOffset( DefaultUtilFUDataPtr, vertLabel ),
		XmRImmediate,
		(XtPointer)NULL,
	},
};




/*
 * UtilFU()
 *
 * class constructor : initializes essential instance data right away
 */
UtilFU::UtilFU( void )			
{


	/*
	 * initialize other object data
	 */
	_setClassName( MY_CLASS );

	dialog = NULL;
	typeDialog = NULL;
	modDialog = NULL;
	numberline = NULL;
	delDialog = NULL;

	brTime = NULL;
	brProcNum = NULL;

	dispManager = NULL;
	numDisplays = 0;
	displays = NULL;

	/* initialize sliding window data */
	pwinds = NULL;
	evts = lastEvt = NULL;
	leadEdgeTS = 0;			// timestamp of leading edge of window
	leadInterval = NULL;		// ptr to interval containing leading
					// edge of the window
}



/*
 * ~UtilFU()
 *
 * class destructor : deletes storage allocated by a UtilFU instance
 */
UtilFU::~UtilFU()
{
	// delete memory allocated for widgets used
	if( dialog ) {
		XtDestroyWidget( dialog );
		dialog = NULL;
	}

	if( typeDialog ) {
		XtDestroyWidget( typeDialog );
		typeDialog = NULL;
	}

	if( modDialog ) {
		XtDestroyWidget( modDialog );
		modDialog = NULL;
	}

	if( delDialog ) {
		XtDestroyWidget( delDialog );
		delDialog = NULL;
	}

	delete numberline;		// destroy our numberline
	delete displays;		// destroy our display list
	delete dispManager;

	delete brTime;
	delete brProcNum;

	delete[] pwinds;
	delete evts;
}




/*
 * _addDisplay()
 *
 * Finishes the addition of a new UtilDisplay to this UtilFU's list
 * of displays
 */
void
UtilFU::_addDisplay( char *name, UtilsDisplayedType dispType )
{
	XmString *xmTypeName;
	char 	*typeName;


	/* extract type from the dialog's list widget */
	XtVaGetValues( infoWidgets[disp_type], 
				XmNselectedItems,   &xmTypeName,
				NULL );
	XmStringGetLtoR( xmTypeName[0], XmSTRING_DEFAULT_CHARSET, &typeName );

	/* make a new UtilDisplay with the given type and name */
	UtilDisplay *pudisp = dispManager->makeNewDisplay( name, 
			dispManager->nameToType( typeName ), dispType );
	XtFree( typeName );
	
	/* add it to our list of existing displays */
	displays = (UtilDisplay *)pudisp->insertYourself( displays );
	numDisplays++;

	_updateDisplayList();		// update main dialog list

}



/*
 * _advanceWindow()
 *
 * Advances the sliding window used to compute utilization averages.
 */
void
UtilFU::_advanceWindow( void )
{
	int i;


	/* 
	 * add in any idle time from the newly added window interval 
	 */
	double account = windowIncrement;	// amount of time to account for
						// in sliding the window
	while( account ) {
		BusyStatus* states = leadInterval->getStates();

		/* compute the amount of time remaining in current interval */
		double intRange = leadInterval->getEndTS() - leadEdgeTS;
		if( intRange < 0 ) {	// is current interval complete?
			return;		// no: this should never happen
		}

		/* find min( what's remaining in the interval, account ) */
		/* so we don't go too far forward */
		double range = ( intRange < account ? intRange : account );

		/* check for idle PEs in this range */
		for( i = 0; i < numProcessors; i++ ) {
			if( states[i] == idleState ) {
				pwinds[i].duration += range;
			}
		}

		/* we have accounted for 'range' units of the new interval */
		account -= range;
		leadEdgeTS += range;		// advance the window

		/* move to new interval if we used up this interval */
		if( range == intRange ) {
			leadInterval =(EventNode*)leadInterval->getNextInList();
		}
	}


	/* 
	 * remove any idle time from the newly removed window interval 
	 */

	/* check that there is a full window before removing anything */
	if( leadEdgeTS - evts->getBeginTS() > windowSize ) {
		
		/* compute timestamp of trailing edge of window */
		double trailEdgeTS = leadEdgeTS - windowSize;

		/* compute the interval to remove from the window to */
		/* bring the window down to the correct size */
		account = trailEdgeTS - evts->getBeginTS();

		while( account > 0 ) {
			BusyStatus* states = evts->getStates();

			/* compute amount remaining in first interval */
			double intRange = evts->getRange();
			if( intRange < 0 ) {	// is current interval complete?
				return;		// this shouldn"t happen
			}

			/* find minimum of intRange and account */
			/* so we don't go too far forward */
			double range = ( intRange < account ? intRange 
							    : account );

			/* check for idle PEs in this range */
			for( i = 0; i < numProcessors; i++ ) {
				if( states[i] == idleState ) {
					pwinds[i].duration -= range;
				}
			}

			/* have accounted for 'range' units of old interval */
			account -= range;

			/* move to new interval if we used up this interval */
			if( range == intRange ) {
				EventNode* temp = evts;
				evts = (EventNode*)temp->removeYourself( evts );

				delete temp;	// deallocate the interval
			}

		}

		/* clip beginning of oldest interval to */
		/* beginning of sliding window */
		evts->setBeginTS( trailEdgeTS );
	}

}



/*
 * _deleteDisplay()
 *
 * Finishes the removal of a UtilDisplay from this UtilFU's list of
 * existing displays.
 */
void
UtilFU::_deleteDisplay( UtilFUData *data )
{
	UtilDisplay *goner;


	/* find which display is currently selected */
	if( (goner = _getCurrentDisplay() ) == NULL ) {
		cerr << "No current selection - but should be one\n";	
		exit( 1 );
	}	

	/* remove the display */
	displays = (UtilDisplay *)goner->removeYourself( displays );
	delete goner;
	numDisplays--;
	
	/* update the display list to reflect the change */
	_updateDisplayList();

	/* pop down the "Delete Dialog" */
	_popdownCleanup( data );
}




/*
 * _getCurrentDisplay()
 *
 * Returns the UtilDisplay which is currently selected in the UtilFU
 * display list
 */
UtilDisplay *
UtilFU::_getCurrentDisplay( void )
{
	UtilDisplay *result = NULL;
	XmString *xmname;

	char *name;



	if( numDisplays ) {
		/* get name of selected display */
		XtVaGetValues( infoWidgets[disp_list],
					XmNselectedItems, &xmname,
					NULL );
		XmStringGetLtoR( xmname[0], XmSTRING_DEFAULT_CHARSET, &name );
	
		result = displays->getDisplayByName( name );
		XtFree( name );
	}

	return result;
}




/*
 * _handleAggregateToggle()
 *
 * reset the scrolled list of display types to a list of 
 * display types which make sense for the aggregates
 */
void
UtilFU::_handleAggregateToggle( void )
{
	int	numTypes;	// number of types in the type list
	XmString *xmtypes;
	int	i;



	XtUnmanageChild( infoWidgets[disp_type] );

	/*
	 * Individual utilization displays are allowed on any type of
	 * utilization display.
	 */
	const char** typeList = dispManager->getAggregateTypeList( numTypes );

	/* convert type list to XmStrings */
	xmtypes = (XmString *)XtMalloc( numTypes * sizeof( XmString ) );
	for( i = 0; i < numTypes; i++ ) {
		xmtypes[i] = XmStringCreateLtoR( (char*)typeList[i], 
						XmSTRING_DEFAULT_CHARSET );
	}

	XtVaSetValues( infoWidgets[disp_type],
			XmNitems,           xmtypes,
			XmNitemCount,       numTypes,
			XmNselectedItems,   xmtypes,
			XmNselectedItemCount,       1,
			XmNvisibleItemCount,	5,
			NULL );

	/* 
	 * set default name for the display 
	 */
	char	buf[max_name_len];
	ostrstream ostr( buf, max_name_len );
	ostr << typeList[0] << "-" << numDisplays+1 << '\0';
	XmTextSetString( infoWidgets[disp_name], buf );


	XtManageChild( infoWidgets[disp_type] );
}



/*
 * _handleBrowse()
 *
 * Handles the XmNbrowseCallback for the List widget which displays
 * the currently available types of displays.
 */
void
UtilFU::_handleBrowse( XmListCallbackStruct *cb )
{
	XmStringComponentType comp_type;
	char* txtbuf;
	XmStringCharSet chset;
	XmStringDirection dir;


	char buf[max_name_len];			// default name"s number
	ostrstream ostr( buf, max_name_len );


	/* get type name from latest selection */
	XmStringContext myContext;
	XmStringInitContext( &myContext, cb->item );
	do {
		comp_type = XmStringGetNextComponent( myContext,
				&txtbuf, &chset, &dir, NULL, NULL, NULL );
	} while( comp_type != XmSTRING_COMPONENT_TEXT );
	XmStringFreeContext( myContext );

	/* set default name  to the last item selected, plus a number */
	ostr << txtbuf << "-" << numDisplays+1 << '\0';
	XmTextSetString( infoWidgets[disp_name], buf );
}



/*
 * _handleIndividualToggle()
 *
 * reset the scrolled list of display types to a list of 
 * all possible types
 */
void
UtilFU::_handleIndividualToggle( void )
{
	int	numTypes;	// number of types in the type list
	XmString* xmtypes;
	int	i;



	XtUnmanageChild( infoWidgets[disp_type] );

	/*
	 * Individual utilization displays are allowed on any type of
	 * utilization display.
	 */
	const char** typeList = dispManager->getTypeList( numTypes );

	/* convert type list to XmStrings */
	xmtypes = (XmString *)XtMalloc( numTypes * sizeof( XmString ) );
	for( i = 0; i < numTypes; i++ ) {
		xmtypes[i] = XmStringCreateLtoR( (char*)typeList[i], 
						XmSTRING_DEFAULT_CHARSET );
	}

	XtVaSetValues( infoWidgets[disp_type],
			XmNitems,           xmtypes,
			XmNitemCount,       numTypes,
			XmNselectedItems,   xmtypes,
			XmNselectedItemCount,       1,
			XmNvisibleItemCount,	5,
			NULL );

	/* 
	 * set default name for the display 
	 */
	char	buf[max_name_len];
	ostrstream ostr( buf, max_name_len );
	ostr << typeList[0] << "-" << numDisplays+1 << '\0';
	XmTextSetString( infoWidgets[disp_name], buf );


	XtManageChild( infoWidgets[disp_type] );
}



/* 
 * _handleNewDisplay
 *
 * based on users name and type preferences, handles creation of a new
 * PU display.
 * This is the member function callback for pressing the "New" button in the 
 * main dialog, called from the static function callback1.
 */
void
UtilFU::_handleNewDisplay( UtilFUData *data )
{
	
	/* make the dialog asking for type of utilization to display */
	typeDialog = _makeTypeDialog( data );
	data->dialog = typeDialog;

	/* grab the pointer for answering the dialog */
	XtManageChild( typeDialog );
	XtAddGrab( typeDialog, FALSE, FALSE );
}



/*
 * _handleSpecifyRest()
 * Handles the completion of the specification of the attributes of a
 * newly created or newly modified UtilDisplay
 */
void
UtilFU::_handleSpecifyRest( UtilFUData *data )
{
	Boolean set;			// current value of display type toggle
	


	/* extract information from the "type" dialog */
	XtVaGetValues( infoWidgets[indv_slct],
			XmNset,		&set,
			NULL );
	char *name = XmTextGetString( infoWidgets[disp_name] );

	/* create a new data struct */
	UtilFUData* new_data = new UtilFUData;
	new_data->dialog = (Widget)NULL;
	new_data->ufu = data->ufu;


	/* add a new UtilDisplay to the list if name is unique and non-null */
	if( !name || !strcmp( name, "" ) ) {
		_infoDialog( new_data, "A display must have a non-NULL name.\n\
				Please choose a name for the display." );
	}
	else if( displays && !(displays->uniqueDisplayName( name )) ){
		/* give warning about giving unique name */
		_infoDialog( new_data, "Display names must be unique.\n\
				Please choose a unique display name." );
	}
	else {
		UtilsDisplayedType dispType = ( set ? IndividualsShown
						    : AggregateShown );

		/* 
		 * make a new UtilDisplay instance and then cleanup things
		 */
		_addDisplay( name, dispType );
		_popdownCleanup( data ); 	// pop down the "type" dialog
		_handleModifyDisplay( new_data );// complete the specification
	}
}





/*
 * _handleModifyDisplay()
 *
 * Handles the displaying of the "Modify Dialog" for modifying 
 * UtilDisplay objects
 */
void
UtilFU::_handleModifyDisplay( UtilFUData *data )
{

	data->dialog = modDialog = _makeModDialog( data );

	XtManageChild( modDialog );
	XtAddGrab( modDialog, FALSE, FALSE );

	/* set up graphics context for drawing in numberline */
	if( numberline ){
		numberline->createGC();
	}
}




/*
 * _handleDeleteDisplay()
 *
 * Handles the displaying of the "Delete Dialog" for deleting
 * existing UtilDisplay objects
 */
void
UtilFU::_handleDeleteDisplay( UtilFUData *data )
{
	data->dialog = delDialog = _makeDelDialog( data );

	XtManageChild( delDialog );
	XtAddGrab( delDialog, FALSE, FALSE );
}




/*
 * _infoDialog()
 *
 * Displays a Motif information dialog which displays the message
 * contained in the parameter str.
 */
void
UtilFU::_infoDialog( UtilFUData *data,  char *str )
{
	Widget dlog;
	Widget childButton;
	XmString	xstr;
	Arg	wargs[1];
	int	n;


	/* create message */
	xstr = XmStringCreateLtoR( str, XmSTRING_DEFAULT_CHARSET );
	
	n = 0;
	XtSetArg( wargs[0], XmNmessageString, xstr );	n++;
	dlog = XmCreateInformationDialog( dialog, "infoDialog", 
				wargs, n );

	/* remove cancel button */
	childButton = XmMessageBoxGetChild( dlog, XmDIALOG_CANCEL_BUTTON );
	XtUnmanageChild( childButton );

	/* de-sensitize Help button for now */
	childButton = XmMessageBoxGetChild( dlog, XmDIALOG_HELP_BUTTON );
	XtVaSetValues( childButton,
		XmNsensitive,	False,
		NULL );

	/* add callback to the OK button */
	childButton = XmMessageBoxGetChild( dlog, XmDIALOG_OK_BUTTON );
	XtAddCallback( childButton, XmNactivateCallback,
				(XtCallbackProc)cancelCB, data );

	XtManageChild( dlog );
	XtAddGrab( dlog, FALSE, FALSE );

}



/*
 * _makeDelDialog()
 *
 * This method creates the actual X/Motif widget hierarchy for the
 * Delete dialog, returning the top widget in the hierarchy
 */
Widget
UtilFU::_makeDelDialog( UtilFUData *data )
{
	Widget	delDialog;
	Widget	childButton;


	// check that there are displays to delete
	if( numDisplays ) {
		delDialog = XmCreateWarningDialog( dialog, "delDialog",
					NULL, 0 );
		
		// add callback to the OK button to do the actual delete
		childButton = XmMessageBoxGetChild( delDialog, 
					XmDIALOG_OK_BUTTON );
		XtAddCallback( childButton, XmNactivateCallback,
					(XtCallbackProc)deleteDisplayCB, data );
		XtVaSetValues( delDialog,
			XmNdefaultButton,	childButton,
			NULL );


		// set "Do Nothing" callback for Cancel button
		childButton = XmMessageBoxGetChild( delDialog, 
					XmDIALOG_CANCEL_BUTTON );
		XtAddCallback( childButton, XmNactivateCallback,
					(XtCallbackProc)cancelCB, data );

		childButton = XmMessageBoxGetChild( delDialog, 
				XmDIALOG_HELP_BUTTON );
		XtAddCallback( childButton, XmNactivateCallback,
				(XtCallbackProc)helpCB, "UtilizationDelete" );
		
	}
	else {
		delDialog = XmCreateInformationDialog( dialog, "delInfo", 
				NULL, 0 );
		childButton = XmMessageBoxGetChild( delDialog, 
				XmDIALOG_CANCEL_BUTTON );
		XtUnmanageChild( childButton );

		/*
		 * de-sensitize Help button - none is available
		 */
		childButton = XmMessageBoxGetChild( delDialog, 
				XmDIALOG_HELP_BUTTON );
		XtVaSetValues( childButton,
			XmNsensitive,	False,
			NULL );
		
		// set callbacks on this dialog"s buttons to do nothing
		childButton = XmMessageBoxGetChild( delDialog, 
				XmDIALOG_OK_BUTTON );
		XtAddCallback( childButton, XmNactivateCallback,
					(XtCallbackProc)cancelCB, data ); 
		XtVaSetValues( delDialog,
			XmNdefaultButton,	childButton,
			NULL );
	}

	return delDialog;
}




/*
 * _makeMainDialog()
 *
 * This method creates the X/Motif widget hierarchy used by the main
 * UtilFU dialog.  The FormDialog widget is returned.
 */
Widget
UtilFU::_makeMainDialog( void )
{
	Dimension height;			// for fixing action height


	Widget pdialog = XmCreateFormDialog( Pablo::TopLevel()->getWidget(), 
					"PUDialog", NULL, 0 );

	Widget pane = XtVaCreateWidget( "pane",
				xmPanedWindowWidgetClass, pdialog,
				XmNsashHeight,	1,
				XmNsashWidth,	1,
				NULL );

	/* 
	 * create Control Area 
	 */
	Widget contform = XtVaCreateWidget( "contform",
				xmFormWidgetClass, pane,
				NULL );

	Widget label = XtVaCreateManagedWidget( "displayLabel",
				xmLabelWidgetClass, contform,
				NULL );

	Widget list = XmCreateScrolledList( contform, "displayList",
				NULL, 0 );
	infoWidgets[disp_list] = list;

	XtVaSetValues( XtParent( list ),
			XmNtopAttachment,	XmATTACH_WIDGET,
			XmNtopWidget,		label,
			XmNtopOffset,		5,
			XmNleftAttachment,	XmATTACH_FORM,
			XmNleftOffset,		5,
			XmNrightAttachment,	XmATTACH_FORM,
			XmNrightOffset,		5,
			XmNbottomAttachment,	XmATTACH_FORM,
			NULL );

	XtVaSetValues( list,
			XmNitems,		NULL,
			XmNitemCount,		0,
			XmNvisibleItemCount,	4, 
			NULL );

	XtManageChild( list );
	XtManageChild( contform );

	/* 
	 * create Action Area 
	 */
	Widget actform = XtVaCreateWidget( "actform",
				xmFormWidgetClass, pane,
				XmNfractionBase,	5,
				NULL );

	Widget newBut = XtVaCreateManagedWidget( "newButton",
				xmPushButtonWidgetClass, actform,
				XmNshowAsDefault,	True,
				XmNdefaultButtonShadowThickness,	1,
				NULL );
	XtAddCallback( newBut, XmNactivateCallback,
				(XtCallbackProc)newCB, this );
	XtVaSetValues( pdialog,
		XmNdefaultButton,	newBut,
		NULL );

	Widget modBut = XtVaCreateManagedWidget( "modButton",
				xmPushButtonWidgetClass, actform,
				XmNshowAsDefault,	False,
				XmNdefaultButtonShadowThickness,	1,
				NULL );
	XtAddCallback( modBut, XmNactivateCallback,
				(XtCallbackProc)modifyCB, this );

	Widget delBut = XtVaCreateManagedWidget( "delButton",
				xmPushButtonWidgetClass, actform,
				XmNshowAsDefault,	False,
				XmNdefaultButtonShadowThickness,	1,
				NULL );
	XtAddCallback( delBut, XmNactivateCallback,
				(XtCallbackProc)deleteCB, this );

	Widget helpBut = XtVaCreateManagedWidget( "helpButton",
				xmPushButtonWidgetClass, actform,
				XmNshowAsDefault,	False,
				XmNdefaultButtonShadowThickness,	1,
				NULL );
	XtAddCallback( helpBut, XmNactivateCallback,
				(XtCallbackProc)helpCB, "UtilizationDialog" );


	/* fix Action area at its initial height */
	XtManageChild( actform );
	XtVaGetValues( newBut, 
		XmNheight,	&height,
		NULL );
	XtVaSetValues( actform,
		XmNpaneMaximum,	height,
		XmNpaneMinimum,	height,
		NULL );
	XtManageChild( pane );

	return pdialog;
}



/*
 * _makeModDialog()
 *
 * This method creates the X/Motif widget hierarchy to implement the
 * Modify Dialog.  The FormDialog at the top of the hierarchy is returned.
 */
Widget
UtilFU::_makeModDialog( UtilFUData *data )
{
	Widget modDlog;
	Widget childButton;
	Dimension height;
	UtilDisplay *curr;		// currently selected display


	/* give warning if there are no existing displays */
	if( numDisplays ) {

		/* retrieve the current display */
		curr = _getCurrentDisplay();


		modDlog = XmCreateFormDialog( dialog, "modDialog",
						NULL, 0 );
		
		Widget pane = XtVaCreateWidget( "pane",
				xmPanedWindowWidgetClass, modDlog,
				XmNsashHeight,	1,
				XmNsashWidth,	1,
				NULL );

		/* 
		 * create Control Area 
		 */
		Widget contform = XtVaCreateWidget( "contform",
				xmFormWidgetClass, pane,
				NULL );

		Widget nlabel = XtVaCreateManagedWidget( "modNameLabel",
				xmLabelWidgetClass, contform,
				NULL );

		Widget ntext = XtVaCreateManagedWidget( "modNameText",
				xmTextWidgetClass, contform,
				XmNeditMode,	XmSINGLE_LINE_EDIT,
				XmNtopWidget,	nlabel,
				NULL );
		infoWidgets[disp_mnam] = ntext;
		XmTextSetString( ntext, curr->getName() );


		Widget hlabel = XtVaCreateManagedWidget( "modHoriLabel",
				xmLabelWidgetClass, contform,
				XmNtopWidget, ntext,
				NULL );

		Widget htext = XtVaCreateManagedWidget( "modHoriText",
				xmTextWidgetClass, contform,
				XmNeditMode,	XmSINGLE_LINE_EDIT,
				XmNtopWidget,	hlabel,
				NULL );
		infoWidgets[hori_labl] = htext;


		Widget vlabel = XtVaCreateManagedWidget( "modVertLabel",
				xmLabelWidgetClass, contform,
				XmNtopWidget,	htext,
				NULL );

		Widget vtext = XtVaCreateManagedWidget( "modVertText",
				xmTextWidgetClass, contform,
				XmNeditMode,	XmSINGLE_LINE_EDIT,
				XmNtopWidget,	vlabel,
				NULL );
		infoWidgets[vert_labl] = vtext;


		/*
		 * set default display labels
		 */
		char* hlabstr = curr->getHoriLabel();
		char* vlabstr = curr->getVertLabel();
		if( !hlabstr ) {
			hlabstr = defaultData.horiLabel;
		}
		if( !vlabstr ) {
			vlabstr = defaultData.vertLabel;
		}
		XmTextSetString( htext, hlabstr );
		XmTextSetString( vtext, vlabstr );


		if( curr->getDisplayedType() == IndividualsShown ) {

			/* create a numberline for this dialog */
			numberline = new NumberLine( contform, 0, 
							numProcessors - 1 );
			XtVaSetValues( numberline->getHierarchy(),
					XmNtopWidget,	vtext,
					NULL );

			/* recall which were the selected PEs */
			numberline->setSelection( curr->getSelection() );

		}
		else {
			/* attach vertical text label to the form's bottom */
			XtVaSetValues( vtext,
				XmNbottomAttachment,	XmATTACH_FORM,
				XmNbottomOffset,	5,
				NULL );
		}

		XtManageChild( contform );


		/*
		 * create Action Area
		 */

		Widget actform = XtVaCreateWidget( "actform",
				xmFormWidgetClass,	pane,
				XmNfractionBase,	4,
				NULL );

		Widget okBut = XtVaCreateManagedWidget( "okButton",
				xmPushButtonWidgetClass, actform,
				XmNshowAsDefault,	True,
				XmNdefaultButtonShadowThickness,	1,
				NULL );
		XtAddCallback( okBut, XmNactivateCallback, 
			(XtCallbackProc)modifyDisplayCB, data );
		XtVaSetValues( modDlog,
			XmNdefaultButton, okBut,
			NULL );

		Widget cancelBut = XtVaCreateManagedWidget( "cancelButton",
				xmPushButtonWidgetClass, actform,
				XmNshowAsDefault,	False,
				XmNdefaultButtonShadowThickness,	1,
				NULL );
		XtAddCallback( cancelBut, XmNactivateCallback, 
			(XtCallbackProc)cancelCB, data );

		Widget helpBut = XtVaCreateManagedWidget( "helpButton",
				xmPushButtonWidgetClass, actform,
				XmNshowAsDefault,	False,
				XmNdefaultButtonShadowThickness,	1,
				NULL );
		XtAddCallback( helpBut, XmNactivateCallback, 
			(XtCallbackProc)helpCB, "UtilizationModify" );



		/* fix Action area at its initial height */
		XtManageChild( actform );
		XtVaGetValues( okBut, 
			XmNheight,	&height,
			NULL );
		XtVaSetValues( actform,
			XmNpaneMaximum,	height,
			XmNpaneMinimum,	height,
			NULL );

		XtManageChild( pane );
	}

	else {
		modDlog = XmCreateInformationDialog( dialog, "modInfo", 
				NULL, 0 );
		childButton = XmMessageBoxGetChild( modDlog, 
				XmDIALOG_CANCEL_BUTTON );
		XtUnmanageChild( childButton );

		/* 
		 * de-sensitize Help button  - none is available
		 */
		childButton = XmMessageBoxGetChild( modDlog,
				XmDIALOG_HELP_BUTTON );
		XtVaSetValues( childButton,
				XmNsensitive,	False,
				NULL );

		// set callbacks on this dialog"s buttons to do nothing
		childButton = XmMessageBoxGetChild( modDlog, 
				XmDIALOG_OK_BUTTON );
		XtAddCallback( childButton, XmNactivateCallback,
					(XtCallbackProc)cancelCB, data ); 
		XtVaSetValues( modDlog,
			XmNdefaultButton,	childButton,
			NULL );
	}

	return modDlog;
}




/*
 * _makeTypeDialog()
 *
 * This method makes the X/Motif widget hierarchy for implementing the
 * Type Dialog ( also called the New Dialog ).  The FormDialog at the
 * top of the hierarchy is returned.
 */
Widget
UtilFU::_makeTypeDialog( UtilFUData *data )
{
	Dimension height;		// for fixing height of the dialog
	int	i;


	/* create form dialog for the dialog */
	Widget dlog = XmCreateFormDialog( dialog, "typeDialog", NULL, 0 );

	/* use PanedWindow to separate Action and Control areas */
	Widget pane = XtVaCreateWidget( "pane",
			xmPanedWindowWidgetClass, dlog,
			XmNsashHeight,	1,
			XmNsashWidth,	1,
			NULL );

	/*
	 * create Control Area
	 */
	Widget contform = XtVaCreateWidget( "contform",
			xmFormWidgetClass, pane,
			NULL );


	Widget namelabel = XtVaCreateManagedWidget( "nameLabel",
			xmLabelWidgetClass, contform,
			NULL );
	Widget nametext = XtVaCreateManagedWidget( "nameText",
			xmTextWidgetClass, contform,
			XmNeditMode,	XmSINGLE_LINE_EDIT,
			XmNtopWidget,	namelabel,
			NULL );
	infoWidgets[disp_name] = nametext;


	Widget frame = XtVaCreateManagedWidget( "frame",
			xmFrameWidgetClass, contform,
			XmNtopWidget,	nametext,
			NULL );

	Widget boxform = XtVaCreateWidget( "boxform",
			xmFormWidgetClass, frame,
			NULL );

	Widget typelabel = XtVaCreateManagedWidget( "typeLabel",
			xmLabelWidgetClass, boxform,
			NULL );

	Widget rbox = XmCreateRadioBox( boxform, "typeRadio", NULL, 0 );
	XtVaSetValues( rbox,
			XmNtopWidget,	typelabel,
			NULL );

	Widget indiv = XtVaCreateManagedWidget( "individualToggle",
			xmToggleButtonGadgetClass, rbox,
			XmNset,		True,
			NULL );
	Widget aggr = XtVaCreateManagedWidget( "aggregateToggle",
			xmToggleButtonGadgetClass, rbox, 
			XmNset,		False,
			NULL );
	infoWidgets[indv_slct] = indiv;
	infoWidgets[aggr_slct] = aggr;
	XtAddCallback( indiv, XmNvalueChangedCallback, 
			(XtCallbackProc)handleIndividualToggleCB, data ); 
	XtAddCallback( aggr, XmNvalueChangedCallback, 
			(XtCallbackProc)handleAggregateToggleCB, data ); 

	Widget frame2 = XtVaCreateManagedWidget( "listframe",
			xmFrameWidgetClass, contform,
			XmNtopWidget,	frame,
			NULL );
	Widget listform = XtVaCreateWidget( "listform",
			xmFormWidgetClass, frame2,
			NULL );

	Widget dlabel = XtVaCreateManagedWidget( "dispLabel",
			xmLabelWidgetClass, listform,
			NULL );

		

	/*
	 * create the scrolled list of display types
	 * The default display selection ( individual PEs ) allows for
	 * any type of Utilization display
	 */
	int numTypes;
	const char** typeList = dispManager->getTypeList( numTypes );
	XmString* xmtypes = (XmString *)XtMalloc( numTypes*sizeof( XmString ) );
	for( i = 0; i < numTypes; i++ ) {
		xmtypes[i] = XmStringCreateLtoR( (char*)typeList[i], 
						XmSTRING_DEFAULT_CHARSET );
	}
	Widget list = XmCreateScrolledList( listform, "typeList", NULL, 0 );
	infoWidgets[disp_type] = list;
	XtAddCallback( list, XmNbrowseSelectionCallback,
			(XtCallbackProc)typeBrowseCB, this );

	XtVaSetValues( XtParent( list ),
			XmNtopAttachment,	XmATTACH_WIDGET,
			XmNtopWidget,		dlabel,
			XmNtopOffset,		5,
			XmNleftAttachment,	XmATTACH_FORM,
			XmNleftOffset,		5,
			XmNrightAttachment,	XmATTACH_FORM,
			XmNrightOffset,		5,
			XmNbottomAttachment,	XmATTACH_FORM,
			XmNbottomOffset,	5,
			XmNheight,		100,
			XmNwidth,		200,
			NULL );

	XtVaSetValues( list,
			XmNitems,           xmtypes,
			XmNitemCount,       numTypes,
			XmNselectedItems,   xmtypes,
			XmNselectedItemCount,       1,
			XmNvisibleItemCount,	5,
			NULL );
	XtManageChild( list );

	/* 
	 * set default name for the display 
	 */
	char	buf[max_name_len];
	ostrstream ostr( buf, max_name_len );
	ostr << typeList[0] << "-" << numDisplays+1 << '\0';
	XmTextSetString( nametext, buf );


	XtManageChild( rbox );
	XtManageChild( boxform );
	XtManageChild( listform );
	XtManageChild( contform );


	/* 
	 * create Action area 
	 */
	Widget actform = XtVaCreateWidget( "actform",
			xmFormWidgetClass, pane,
			XmNfractionBase,	4,
			NULL );

	Widget okBut = XtVaCreateManagedWidget( "okButton",
			xmPushButtonWidgetClass, actform,
			XmNshowAsDefault,	True,
			XmNdefaultButtonShadowThickness,	1,
			NULL );
	XtAddCallback( okBut, XmNactivateCallback, 
			(XtCallbackProc)specifyRestCB, data ); 
	XtVaSetValues( dlog,
			XmNdefaultButton,	okBut,
			NULL );


	Widget cancelBut = XtVaCreateManagedWidget( "cancelButton",
			xmPushButtonWidgetClass, actform,
			XmNshowAsDefault,	False,
			XmNdefaultButtonShadowThickness,	1,
			NULL );
	XtAddCallback( cancelBut, XmNactivateCallback, 
			(XtCallbackProc)cancelCB, data );

	Widget helpBut = XtVaCreateManagedWidget( "helpButton",
			xmPushButtonWidgetClass, actform,
			XmNshowAsDefault,	False,
			XmNdefaultButtonShadowThickness,	1,
			NULL );
	XtAddCallback( helpBut, XmNactivateCallback, 
			(XtCallbackProc)helpCB, "UtilizationNew" );

	/* fix Action area at its initial height */
	XtManageChild( actform );
	XtVaGetValues( okBut, 
		XmNheight,	&height,
		NULL );
	XtVaSetValues( actform,
		XmNpaneMaximum,	height,
		XmNpaneMinimum,	height,
		NULL );

	XtManageChild( pane );

	return dlog;
}




/*
 * _modifyDisplay()
 *
 * Commits the user's modifications to an existing UtilDisplay.
 */
void
UtilFU::_modifyDisplay( UtilFUData *data )
{
	UtilDisplay *curr = _getCurrentDisplay();	// current display
	char *currname = curr->getName();	// current display"s name
	char *name, *hori, *vert;
	UtilFUData *new_data = new UtilFUData;


	new_data->dialog = (Widget)NULL;
	new_data->ufu = data->ufu;


	/* retrieve information from Modify dialog widgets */
	name = XmTextGetString( infoWidgets[disp_mnam] );
	hori = XmTextGetString( infoWidgets[hori_labl] );
	vert = XmTextGetString( infoWidgets[vert_labl] );

	/* check that name is unique and non-NULL */
	if( !name || !strcmp( name, "" ) ) {
		_infoDialog( new_data, "A display must have a non-NULL name.\n\
				Please choose a name for the display." );
	}
	else if( !strcmp( name, currname) || 
				displays->uniqueDisplayName( name ) ) {


		/* modify display to reflect the changes */
		curr->setName( name );
		curr->setDialogTitle( name ); 
		curr->setHoriLabel( hori );
		curr->setVertLabel( vert );

		/* 
		 * set the display's utilization selections 
		 * This only applies if individual PEs are selected
		 */
		curr->cleanupPerfWidgets();
		if( curr->getDisplayedType() == IndividualsShown ) {
			/* deallocate old displays selection */
			delete curr->getSelection();
			curr->setSelection( numberline->getSelection() );
		}
		else {
			/* no individual selection is being displayed */
			curr->setSelection( NULL );
		}
		curr->updatePerfWidgets();

		_updateDisplayList();
		_popdownCleanup( data );
	}
	else {
		_infoDialog( new_data, "Display names must be unique.\n\
				Please choose a unique display name." );
	}


	XtFree( name );
	XtFree( hori );
	XtFree( vert );
	XtFree( currname );
}






/*
 * _popdownCleanup()
 *
 * Pops down the dialog indicated by its argument, and then frees the
 * memory allocated to its argument struct.
 */
void
UtilFU::_popdownCleanup( UtilFUData *data )
{



	if( data->dialog ) {


		/* pop down the dialog */
		XtUnmanageChild( data->dialog );


		/*
		 * Reset the UtilFU's dialog variables
		 */
		if( data->dialog == dialog ) {
			dialog = NULL;
		}
		else if( data->dialog == typeDialog ) {
			typeDialog = NULL;
		}
		else if( data->dialog == modDialog ) {
			/* if dialog was the modify dialog, */
			/* should delete the numberline too */
			delete data->ufu->numberline;
			data->ufu->numberline = NULL;

			modDialog = NULL;
		}
		else if( data->dialog == delDialog ) {
			delDialog = NULL;
		}

		/* deallocate the dialog hierarchy */
		XtDestroyWidget( data->dialog );
	}

	delete data;
}





/*
 * _updateDisplayList
 *
 * updates the list of existing displays on the main dialog box
 * from the existing display list
 */
void
UtilFU::_updateDisplayList( void )
{
	XmString *xmOldList,
		 *xmNewList = NULL;
	XmString xnamestr, xtempstr;	// name XmStrings
	int	xmOldCount;
	UtilDisplay *dispPtr;
	char *namestr;			// character strings with name and type
	int i;
	
	

	/* free old list */
	XtVaGetValues( infoWidgets[disp_list], 
			XmNitems,   &xmOldList,
			XmNitemCount, &xmOldCount,
			NULL );
	

	/* make new list */
	if( numDisplays ) {
		xmNewList = (XmString *)XtMalloc( sizeof(XmString)*numDisplays);
		dispPtr = displays;
		for( i = 0; i < numDisplays; i++ ) {


			/* create C strings with name and type */
			namestr = dispPtr->getName();

			xnamestr = XmStringCreate( namestr,
					XmSTRING_DEFAULT_CHARSET );
			xtempstr = XmStringCreate( ":",
					XmSTRING_DEFAULT_CHARSET );


			xmNewList[i] = XmStringConcat( xnamestr, xtempstr );
			const char *typestr = dispPtr->getType();
			xmNewList[i] = XmStringConcat( xmNewList[i],
				XmStringCreateLtoR( (char *)typestr, 
				XmSTRING_DEFAULT_CHARSET ) );

			/* free unneeded XmString storage */
			XmStringFree( xnamestr );
			XmStringFree( xtempstr );
			XtFree( namestr ); 

			dispPtr = (UtilDisplay*)dispPtr->getNextInList();
		}
	}

	/* set XmList items to new list */
	XtVaSetValues( infoWidgets[disp_list], 
			XmNitems,		xmNewList,
			XmNitemCount,		numDisplays,
			NULL );

	if( numDisplays ) {
		XtVaSetValues( infoWidgets[disp_list], 
			XmNselectedItems,   &xmNewList[0],
			XmNselectedItemCount,       1,
			NULL );
	}
	else {
		XtVaSetValues( infoWidgets[disp_list],
			XmNselectedItems,   NULL,
			XmNselectedItemCount,       0,
			NULL );
	}

}




/*
 * configure()
 *
 * Handles configuration of UtilFU object proper.
 * Configures the number of PEs to track,
 * the size of the sliding window,
 * the size of the window increment,
 * and the initial state ( busy or idle ) of the PEs.
 *
 * Also, creates and displays the Motif dialog used for the 
 * main dialog of this FU.
 */
/* virtual */
void
UtilFU::configure( void )
{
	int previousNumProcessors;

	int numprocs;
	double swindowsize; 
	double swindincr;
	int busy;
	int i;

	/* initialize data for results from FUParams */
	if ( isConfigured ) {
	    previousNumProcessors = numProcessors;
	    numprocs = numProcessors;
	    swindowsize = windowSize;
	    swindincr = windowIncrement;
	    if ( defaultData.initialStateIsBusy ) {
		busy = ( initialState == busyState ? 0 : 1 );
  	    } 
	    else {
		busy = ( initialState == idleState ? 0 : 1 );
	    }
	} 
	else {
	    /* this is the first configure, so use defaults */
	    previousNumProcessors = 0;
	    numprocs = defaultData.numProcessors;	
	    swindowsize = defaultData.windowSize;
	    swindincr = defaultData.windowIncr;
	    busy = 0;

	    brTimeDim = brTime->getTraits().getDimension();
	    brProcNumDim = brProcNum->getTraits().getDimension();
	    
	    isConfigured = TRUE_;
	}

	FUParams params;
	params.addTextParam( "PE Count", 
				BaseFUParamEntry::Integer, numprocs );
	params.addTextParam( "Sliding Window Length",
				BaseFUParamEntry::Real, swindowsize );
	params.addTextParam( "Window Increment",
				BaseFUParamEntry::Real, swindincr );
	params.addRadioButtonsParam( "Initial State of PEs",
				BaseFUParamEntry::Integer,
				busy, initialStatusList );

	CString title = "Utilization FU Config: " + getName();

	Boolean_ numprocsOK;
	Boolean_ swindowsizeOK;
	Boolean_ swindowincrOK;
	Boolean_ initStateOK;

	/* Show the configuration panel until valid data entered */
	ParamConfig dlog( Pablo::TopLevel(), params, title );
	do {
	    dlog.run();
		
	    BaseFUParamEntry& numProcsEntry = 
				params.getEntry( "PE Count" );
	    BaseFUParamEntry& windowSizeEntry =
				params.getEntry( "Sliding Window Length" );
	    BaseFUParamEntry& windowIncrEntry =
				params.getEntry( "Window Increment" );
	    BaseFUParamEntry& initStateEntry = 
				params.getEntry( "Initial State of PEs" );

	    /* check validity of the number */
	    numprocsOK = FALSE_;
	    if ( numProcsEntry.valueIsValid() ) {
		numprocs = numProcsEntry.getValue().getInteger();
		if ( numprocs < 1 ) {
		    warning( "PE Count must be at least one." );
		} 
		else {
		    numprocsOK = TRUE_;
		}
	    } 
	    else {
		warning( "PE Count integer was not a valid integer." );
	    }

	    swindowsizeOK = FALSE_;
	    if ( windowSizeEntry.valueIsValid() ) {
		swindowsize = windowSizeEntry.getValue().getReal();
		if ( swindowsize <= 0 ) {
		    warning( "Sliding Window Length must be positive.");
		} 
		else {
		    swindowsizeOK = TRUE_;
		}
	    } 
	    else {
		warning( "Sliding Window Length was not a valid real." );
 	    }

	    swindowincrOK = FALSE_;
	    if ( windowIncrEntry.valueIsValid() ) {
		swindincr = windowIncrEntry.getValue().getReal();
		if ( swindincr <= 0  ) {
		    warning( "Window Increment must be positive." );
		} 
		else {
	 	    swindowincrOK = TRUE_;
		}
	    } 
	    else {
		warning( "Window Increment was not a valid real." );
    	    }

	    initStateOK = FALSE_;
	    if ( initStateEntry.valueIsValid() ) {
		i = initStateEntry.getValue().getInteger();
		if ( defaultData.initialStateIsBusy ) {
		    initialState = ( i == 0 ? busyState : idleState );
		} 
		else {
		    initialState = ( i == 0 ? idleState : busyState );
		}
		initStateOK = TRUE_;
	    } 
	    else {
		warning( "Initial State entry was not valid." );
	    }

	} while(!(numprocsOK && swindowsizeOK && swindowincrOK && initStateOK));

	numProcessors = numprocs;
	windowSize = swindowsize;
	windowIncrement = swindincr;

	/* if number of processors has changed adjust info */
	if ( previousNumProcessors != numProcessors ) {
	    if ( previousNumProcessors == 0 ) {
		/* first time, no data to preserve */
		Assert( pwinds == NULL );
		pwinds = new procData[ numProcessors ];
		evts = new EventNode( NULL, 0, numProcessors, initialState );
		lastEvt = evts;
	    } 
	    else {
		/* reconfigure, preserve as much info as possible */
		int minNumProcessors = 
			      ( previousNumProcessors < numProcessors ) ?
			        previousNumProcessors : numProcessors;

	        /* here we deal with the processor window data */
		Assert( pwinds != NULL );
		procData *previousPwinds = pwinds;
		pwinds = new procData[ numProcessors ];

		for ( i = 0; i < minNumProcessors; i++ ) {
		    pwinds[i].duration = previousPwinds[i].duration;
		    pwinds[i].util = previousPwinds[i].util;
		}
		delete[] previousPwinds;

		/* 
		 * Here we deal with our list of events for each processor. 
	     	 * Copy as many states from the old intervals as possible
	         * PEs which weren't known about before are set to
	         * the initialState, since it is impossible to know which
	         * state they are actually in during this interval.
	         * Therefore, there is no guarantee that the utilizations
	         * reported for the new PEs are correct or not.
	         */
	        EventNode *eNode = evts;
	        while( eNode ) {

		    BusyStatus *oldStates = eNode->getStates();

		    /* allocate space for new array of PE states */
		    BusyStatus *newStates = new BusyStatus[ numProcessors ];

		    /* preserve as many states as possible */
		    for ( i = 0; i < minNumProcessors; i++ ) {
			newStates[i] = oldStates[i];
		    }

		    /* fill out the rest of the states */
		    for ( i = minNumProcessors; i < numProcessors; i++ ) {
			newStates[i] = initialState;
		    }

		    /* free the old states and save the new array of states */
		    delete[] oldStates;
		    eNode->setStates( newStates );
	
		    /* advance to the next interval */
		    eNode = (EventNode*)eNode->getNextInList();
		}
	    }
	}

	/* 
	 * make X/Motif dialog used by this FU 
	 * unless there is already an existing display
	 */
	if ( dialog == NULL ) {
	    dialog = _makeMainDialog();
	    XtManageChild( dialog );

	    dispManager = new UtilDisplayManager( dialog, this );
	}
}






/*
 * copy()
 *
 * creates a copy of the existing UtilFU.
 * The copy is not exact, in that the existing utilization displays are
 * not duplicated.
 */
/* virtual */
FunctionalUnit *
UtilFU::copy( void )
{
	UtilFU *ufu = new UtilFU();
	return ufu;
}




/*
 * init()
 *
 * Performs initialization of the module data.
 * Called when the user add's a module from the "Add Module" dialog of Pablo.
 */
/* virtual */
void
UtilFU::init( void )
{
	/* clean up any existing objects and dialogs */
	if( dialog ) {
		XtDestroyWidget( dialog );
		dialog = NULL;
	}
	delete brTime;
	delete brProcNum;

	isConfigured = FALSE_;

	/* 
	 * configure the input ports 
	 */
	brTimeValue = Value::NOVALUE;
	brProcNumValue = Value::NOVALUE;

	brTime = new InputPort( "Seconds Timestamp" );
	brProcNum = new InputPort( "PE Number" );

	brTime->addTraits( INTEGER, 0 );
	brTime->addTraits( FLOAT, 0 );
	brTime->addTraits( DOUBLE, 0 );
	_addInputPort( brTime );

	brProcNum->addTraits( INTEGER, 0 );
	_addInputPort( brProcNum );


	/*
	 * get X resources for configuration for this FU
	 */
	XtVaGetApplicationResources( Pablo::TopLevel()->getWidget(),
		&defaultData,
		resources,
		XtNumber( resources ),
		NULL );

	/*
	 * Initialize the initialStatusList list of states
	 * Since it is static, only initialize once for whole class
	 */
	if( initialStatusList.isEmpty() ) {
		if( defaultData.initialStateIsBusy ) {
			initialStatusList.addElement( "Busy" );
			initialStatusList.addElement( "Idle" );
		}
		else {
			initialStatusList.addElement( "Idle" );
			initialStatusList.addElement( "Busy" );
		}
	}

}




/*
 * loadConfigFromFile()
 *
 * Loads a previously saved UtilFU configuration from the given stream.
 */
/* virtual */
Boolean_
UtilFU::loadConfigFromFile( const CString& fname ) 
{


	FILE *fp = fopen( fname.getValue(), "r" );

	if( fp == NULL ) {
		warning( "Unable to open %s: %s\n", fname.getValue(),
						errorString() );
	}
	else {
		int ix, iy, iwid, ihei;
		Position x, y;
		Dimension width, height;


		/* skip two comment lines, then get numeric parameters */
		fscanf( fp, "%*[^\n]\n" );
		fscanf( fp, "%*[^\n]\n" );

		if( fscanf( fp, "%d %d %d %d %d %d %lf %lf %d", 
				&ix, &iy, &iwid, &ihei,
				&numProcessors, 
				&numDisplays, &windowSize, &windowIncrement,
				&initialState ) != 9 ) {
			warning( "Unable to read configuration information \
				from %s\n", fname.getValue() );
		}
		else {
			fscanf( fp, "\n" );		// start on next line

			/* pop down the dialog if it exists */
			if( dialog ) {
				XtUnmanageChild( dialog );
			}
			else {
				/* otherwise, make it */
				dialog = _makeMainDialog();
				delete dispManager;
				dispManager = new UtilDisplayManager( dialog, 
								this );
			}

			/* read in rest of the configuration */
			displays = dispManager->readDisplayListFromFile( fp, 
						numDisplays );
			fclose( fp );

			/* set up enough space for the sliding windows */
			delete[] pwinds;
			pwinds = new procData[numProcessors];

			/* begin first interval in the sliding window */
			evts = lastEvt = new EventNode( NULL, 0, 
						numProcessors, initialState );
			leadInterval = NULL;

			_updateDisplayList();
			isConfigured = TRUE_;
			XtManageChild( dialog );

			/* reset correct screen chax */
			x = ix;
			y = iy;
			width = iwid;
			height = ihei;
			XtVaSetValues( dialog,
				XmNx,		x,
				XmNy,		y,
				XmNwidth,	width,
				XmNheight,	height,
				NULL );
		}
	}

	if( !isConfigured ) {
		configure();
	}
	return isConfigured;
}




/* virtual */
Boolean_
UtilFU::ready( void )
{
	return TRUE_;
}



/*
 * run()
 *
 * take data from the input ports, compute the utilization
 * figures, and pass them to the displays.
 * 30 July 1993: PCR changed timestamp to be double instead of int.
 */
/* virtual */
void
UtilFU::run( Boolean_& /* errorFlag */ )
{
	double ts; 			// timestamp of current event
	double windSizeSeen;		// amount of sliding window which has
					// been seen so far 
	int procnum;			// PE number of reciever
	int i;


	/* indicate we've started running somewhere */
	startedRunning = TRUE_;

	/* 
	 * retrieve an event 
	 */
	Assert( brTime->valueAvailable() );
	Assert( brProcNum->valueAvailable() );
	brTimeValue = brTime->getValue();
	brProcNumValue = brProcNum->getValue();
	ts = brTimeValue.getValueAs( DOUBLE );
	procnum = brProcNumValue.getValueAs( INTEGER );


	/*
	 * sanity check: make sure that the PE number is valid
	 */
	if( procnum > numProcessors - 1 ) {
		return;			// ignore the event
	}


	/* finish current state interval  */
	lastEvt->setEndTS( ts );
	if( !leadInterval ) {
		leadInterval = lastEvt;
	}

	/* place event at the end of the event list */
	EventNode* temp = new EventNode( lastEvt, procnum, numProcessors );
	temp->insertAfter( lastEvt );
	lastEvt = temp;


	/*
	 * keep sliding the window while we have enough room to slide
	 */
	while( lastEvt->getBeginTS() - leadEdgeTS >= windowIncrement ) {

		/* advance the sliding window */
		_advanceWindow();

		/* compute amount of window which has been seen so far */
		/* so that more accurate utilization values can be reported */
		/* for large or incomplete window sizes */
		/* assumes timestamps start at 0.0 */
		windSizeSeen = ( (leadEdgeTS > windowSize) ? windowSize :
							   leadEdgeTS );

		/* 
		 * compute all utilizations 
		 */
		float avgutil = 0.0;		// variables for aggregate
		float minutil = 1.0;		// utilizations
		float maxutil = 0.0;


		for( i = 0; i < numProcessors; i++ ) {
			float util =((float)windSizeSeen-pwinds[i].duration) /
								windSizeSeen;

			/* check boundaries of utilization values */
			if( util > 1 ) {
				pwinds[i].util = 1;
			}
			else if ( util < 0 ) {
				pwinds[i].util = 0;
			}
			else {
				pwinds[i].util = util;
			}

			// update aggregate statistics
			avgutil += util;
			if( util <  minutil ) {
				minutil = util;
			}
			if( util > maxutil ) {
				maxutil = util;
			}
		}
		avgutil /= numProcessors;

		/* 
		 * broadcast the new utilizations 
		 */
		UtilDisplay* curr;
		for( curr = displays; curr; 
				curr = (UtilDisplay*)curr->getNextInList() ) {

			if( curr->getDisplayedType() == AggregateShown ) {
				curr->changeValue( Average, avgutil );
				curr->changeValue( Minima, minutil );
				curr->changeValue( Maxima, maxutil );
			}
			else {
				for( i = 0; i < numProcessors; i++ ) {
					curr->changeValue( i, pwinds[i].util );
				}
			}
		}
	}
}





/*
 * saveConfigToFile()
 *
 * Saves the configuration of the existing UtilFU, as well as all its
 * UtilDisplay objects, to the given stream.  They may be reread later
 * using the loadConfigFromFile() method.
 */
/* virtual */
Boolean_
UtilFU::saveConfigToFile( const CString& fname ) const
{
	Boolean_ result;


	/* attempt to open the desired file */
	ofstream ofs( fname.getValue() );

	if( !ofs ) {
		error( "Unable to open %s: %s\n", fname.getValue(),
						errorString() );
		result = FAILURE_;
	}
	else {
		Position x, y;
		Dimension width, height;

		/* get screen parameters for the module */
		XtVaGetValues( dialog,
			XmNx,		&x,
			XmNy,		&y,
			XmNwidth,	&width,
			XmNheight,	&height,
			NULL );


		ofs << "# X Y width height\n";
		ofs << "# numProcessors numDisplays";
		ofs << " windowSize windowIncrement initialState\n";
		ofs << (int)x << ' ' << (int)y << ' '
		    << (int)width << ' ' << (int)height << ' ';
		ofs << numProcessors << ' ' << numDisplays << ' ';

		ofs.precision( 12 );		// output with more precision
		ofs << windowSize << ' ' << windowIncrement << ' ';
		ofs.precision( 6 );		// reset the default

		ofs << initialState << '\n';

		/* save the state of this object */
		dispManager->saveDisplayListToFile( ofs, displays );

		ofs.close();
		result = SUCCESS_;
	}

	return result;
}




/*--------------------------------------------------------------*/
/*                    static member functions			*/
/*                used as Motif widget callbacks		*/
/*--------------------------------------------------------------*/



/*
 * cancelCB()
 *
 * XmNactivateCallback for all "Cancel" buttons in UtilFU dialogs.
 */
/* static */
void
UtilFU::cancelCB( Widget /* w */, UtilFUData *data, XtPointer /* callData */ )
{
	/* pop down the dialog and cleanup */
	data->ufu->_popdownCleanup( data );
}



/*
 * deleteCB()
 *
 * XmNactivateCallback for the "Delete" button of the main UtilFU dialog.
 */
/* static */
void
UtilFU::deleteCB( Widget /* w */, UtilFU *ufu, XtPointer /* callData */ )
{
	UtilFUData *data = new UtilFUData;


	/* pass along the UtilFU object to the method 
	/* which actually does the delete */
	data->dialog = (Widget)NULL;
	data->ufu = ufu;
	
	ufu->_handleDeleteDisplay( data );
}




/*
 * deleteDisplayCB()
 *
 * XmNactivateCallback for the "OK" button of the Delete dialog.
 * Calls the method which actually performs the deletion.
 */
/* static */
void
UtilFU::deleteDisplayCB( Widget /* w */, UtilFUData *data, 
						XtPointer /* callData */ )
{
	data->ufu->_deleteDisplay( data );
}



/*
 * handleAggregateToggleCB()
 * 
 * XmNvalueChangedCallback for the "Aggregate" toggle button in the
 * New Dialog.  If the toggle was changed to the "set" state, the
 * list of display types is updated.
 */
/* static */
void
UtilFU::handleAggregateToggleCB( Widget w, UtilFUData *data, 
						XtPointer /* cd  */)
{
	Boolean set;


	/* determine if toggle button is now set */
	XtVaGetValues( w,
		XmNset,	&set,
		NULL );

	if( set ) {
		/* change the list of display types to show only */
		/* those suitable for aggregate data */
		data->ufu->_handleAggregateToggle();
	}
}



/*
 * handleIndividualToggleCB()
 * 
 * XmNvalueChangedCallback for the "Individual" toggle button in the
 * New Dialog.  If the toggle was changed to the "set" state, the
 * list of display types is updated.
 */
/* static */
void
UtilFU::handleIndividualToggleCB( Widget w, UtilFUData *data, 
						XtPointer /* cd */ )
{
	Boolean set;


	/* determine if toggle button is now set */
	XtVaGetValues( w,
		XmNset,	&set,
		NULL );

	if( set ) {
		/* change the list of display types to show */
		/* those suitable for individual utilization displays */
		data->ufu->_handleIndividualToggle();
	}
}



/*
 * helpCB()
 *
 * XmNactivateCallback for "Help" buttons in all dialogs.
 * Invokes the Pablo help system to give help on its parameter topic.
 */
/* static */
void
UtilFU::helpCB( Widget /* w */, char *topic, XtPointer /* cd */)
{
	Pablo::HelpSystem()->giveHelpOn( topic );
}



/*
 * newCB()
 *
 * XmNactivateCallback for "New" button in the Main UtilFU dialog.
 * Invokes another method which will handle specification of the 
 * attributes of the new display.
 */
/* static */
void
UtilFU::newCB( Widget /* w */, UtilFU *ufu, XtPointer /* cd */ )
{
	UtilFUData *data = new UtilFUData;


	data->dialog = (Widget)NULL;
	data->ufu = ufu;

	ufu->_handleNewDisplay( data );
}



/*
 * modifyCB()
 *
 * XmNactivateCallback for "Modify" button in the Main UtilFU dialog.
 * Invokes another method which will handle the presentation of the 
 * Modify dialog.
 */
/* static */
void
UtilFU::modifyCB( Widget /* w */, UtilFU *ufu, XtPointer /* callData */ )
{
	UtilFUData *data = new UtilFUData;


	data->dialog = (Widget)NULL;
	data->ufu = ufu;

	ufu->_handleModifyDisplay( data );
}



/*
 * modifyDisplayCB()
 * 
 * XmNactivateCallback for "OK" button in the Modify dialog.
 * Commits the modifications to the current UtilDisplay object.
 */
/* static */
void
UtilFU::modifyDisplayCB( Widget /* w */, UtilFUData *data, 
				XtPointer /* callData */ )
{

	data->ufu->_modifyDisplay( data );
}



/*
 * specifyRestCB()
 * 
 * XmNactivateCallback for the "OK" button of the New/Type dialog.
 * Invokes a method which pops down the New/Type dialog, and presents
 * the Modify dialog.
 */
/* static */
void
UtilFU::specifyRestCB( Widget /* w */, UtilFUData *data, 
						XtPointer /* callData */)
{
	data->ufu->_handleSpecifyRest( data );
}



/*
 * typeBrowseCB()
 * 
 * XmNbrowseSelectCallback for the list of types of UtilDisplays presented
 * by the New/Type dialog.
 */
/* static */
void
UtilFU::typeBrowseCB( Widget /* w */, UtilFU *ufu, XmListCallbackStruct *cb )
{
	ufu->_handleBrowse( cb );
}




/*
 * procData constructor
 * Initializes a procData struct with reasonable data
 */
procData::procData( void )
{
	duration = 0;
	util = 0;
}



