/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Author: Robert Olson (olson@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * InputPipeSocket.h: An InputPipeSocket is created by an FUWrapper
 * for each Input Pipe connected to a module.   
 *
 * $Header $
 *	
 */
#ifndef InputPipeSocket_h
#define InputPipeSocket_h

#include "Assert.h"
#include "CString.h"
#include "Obj.h"
#include "PacketHeader.h"
#include "RecordDossier.h"
#include "ActionTable.h"

class BinaryPipeReader;
class FUWrapper;
class InputPort;
class ModuleId;
class RecordDictionary;
class StreamPipe;

class InputPipeSocket : public Obj {
/*
 * An InputPipeSocket is created by the FUWrapper for each pipe coming into
 * a module. The InputPipeSocket maintains a dictionary of the record 
 * descriptors seen on the pipe, an Action Table for each of the records, 
 * and Binding Tables for the records which are field-extracted mapping
 * input fields to input ports or output dossier fields
 */

private: 
	static const char *const MY_CLASS;
	static PacketHeader      noHeader;	        

	char 		 *_objectName;		// A combination of _className 
						// and the name of our module.

	StreamPipe 	 *pipe;			// Pipe we are associated with 
	FUWrapper 	 *wrapper;		// Our FUwrapper
	ModuleId	 *source;		// Module id of pipe source

	int		 pipeIndex;		// Our relative pipe number

	BinaryPipeReader *reader;		// We construct this pipe reader
	RecordDictionary *dictionary;		// Our local dictionary
	ActionTable 	 actionTable;		// Actions for each record type

	int 		 inputPortsCnt;         // Number of Input ports we fill
	InputPort*	 *inputPortsBound;	// Pointers to the input ports

	int		 outputFieldsCnt;	// Number of output Flds we fill
	int	         *outputFieldsBound;	// Indices of those output flds


	int		 activeTag;		// Tag of active data record
	PacketHeader     currentHeader;		// Current header from pipe
	Boolean_	 dataPending;	        // TRUE_ when currentHeader
						// contains unprocessed data
						// packet.

	Boolean_	newRecordType;		// TRUE_ when new type of
						// record seen since configure()

	/* Method _doFieldExtraction:		This method is called from
	*					processData when the current 
	*					packet is a data record with 
	*					ActionType Extract.  It sets
	*					the Dictionary Values for the
	*					record from the packet; updates
	*					activeTag; calls setValueP for
	*					the inputPorts; calls setValue
	*					for the outputPorts.          */
	void _doFieldExtraction();

	/* Method _readDescriptor:		Read a descriptor packet from
	*					the input pipe and enter it
	*					into our dictionary.  If it's
	*					a new entry, set the action 
	*					type to discard and raise the 
	*					newRecordType flag.           */
	void _readDescriptor();

public:
	/* Method InputPipeSocket:		The constructor.              */
	InputPipeSocket( StreamPipe *inPipe, FUWrapper *w, ModuleId *src );

	/* Method ~InputPipeSocket:		The destructor.               */
	~InputPipeSocket();

	/* Method bindFieldToInputPort:		Adjust the ActionTable entry
	*					for record 'tag' to bind
	*					indicated field to indicated
	*					input port.                   */
	void bindFieldToInputPort( int tag, int fieldId, 
					    const CString& inputPortName );

	/* Method bindFieldToOutputField:	Adjust the ActionTable entry
	*					for record 'tag' to bind
	*					indicated field to indicated
	*					output field.                 */
	void bindFieldToOutputField( int tag, int fieldId, 
					      const CString& outputFieldName );

	/* Method bindingToInputActive:		Check to see if the specified
	*					binding for activeTag is 
	*					valid. Called from InputPort
	*					after reconfiguration.        */
	Boolean_ bindingToInputActive( InputPort *inPort, int fieldNum ); 

	/* Method configure:			Load dictionary with any new 
	*					descriptors. 		      */
	void configure( Boolean_ singleModule );

	/* Method entriesInDictionary:		Returns the number of entries
	*					in the RecordDictionary 
	*					associated with the socket.   */
	int entriesInDictionary( ) const;

	/* Method getAction:			Get action associated with 
	*					recordTag. ActionInvalid is
	*					returned if no Action on file */
	ActionType getAction( int recordTag );

	/* Method getActiveRecord:		Returns the RecordDossier that 
	*					is active in the dictionary.
	*					called by InputPort.          */
	const RecordDossier& getActiveRecord() const;

	/* Method getDictionary:		Returns reference to record
	*					dictionary for socket.        */
	RecordDictionary& getDictionary() const;

	/* Method getFieldToInputPort:		Returns the field number bound
	*					to the input port at the 
	*					specified index in the given
	*					record.                       */
	int getFieldToInputPort( int recordTag, int inputPortIdx );

	/* Method getFieldToOutputField:	Returns the field number bound
	*					to the output field at the 
	*					specified index in the given
	*					record.                       */
	int getFieldToOutputField( int recordTag, int outputFieldIdx );

	/* Method getInputPort:			Returns the true input port 
	*					index of the nth input port
	*					bound to this socket.         */
	int getInputPort( int inputPortBound ) const;

	/* Method getOutputField:		Returns the true field index of
	*					the nth output field bound to 
	*					this socket.                  */
	int getOutputField( int outputFieldBound ) const; 

	/* Method getPipeIndex:			Returns our pipe index.       */
	int getPipeIndex() const;
	
	/* Method getModuleId:			Returns pointer to associated
	*					ModuleID    		      */
	ModuleId * getModuleId() const;

	/* Method init:				Initializes the state of the
	*					socket. Called by FUWrapper's
	*					init() method. If pipeIndex is 
	*					-1, then init() has not been
	*					called before and pipeIndex is 
	*					set to thePipeIndex; objectName
	*					is set; pipeReader created.  
	*					Otherwise, expect pipeIndex 
	*					and thePipeIndex to match. 
	*					Always clears Dictionary and
	*					resets actions and bindings.  */
	void init( int thePipeIndex );

	/* Method inputPortsFilled:		Returns the number of input
	*					ports bound to this socket.   */
	int inputPortsFilled() const;

	/* Method loadConfigurationFromDir: 	Reload saved configuration
	*					information.                  */
	Boolean_ loadConfigurationFromDir( const CString& dir, int moduleIdx );

	/* Method outputFieldsFilled:		Returns the number of output
	*					fields bound to this socket.  */
	int outputFieldsFilled() const;

	/* Method processData:			Processes packets from the
	*					input pipe based on Actions
	*					and Bindings. Passes thru all
	*					Attribute, Command and Descr
	*					packets. Discards and copies
	*					thru Data Packets w/ those
	*					Action types. 'Stops' when a
	*					Data Packet with type Field
	*					Extract is seen.  Processes
	*					that packet by reading it into
	*					the dictionary and setting
	*					input ports and output fields
	*					per Bindings. Returns TRUE_ if
	*					extract data packet was found,
	*					else returns FALSE_. Called by
	*					FUWrapper::ready().           */
	Boolean_ processData();

	/* Method resetConfiguration:		Resets the configuration info
	*					relating to bindings & actions
	*					-- called by InterfaceClass   */
	void resetConfiguration();

	/* Method saveConfigurationToDir:	Save configuration info.      */
	Boolean_ saveConfigurationToDir( const CString& dir, 
					 int moduleIdx ) const;

	/* Method setActionType:		Set the Action type for the
	*					specified tag.                */
	void setActionType( int recordTag, ActionType action );

	/* Method setInputPorts:		Sets the number of input ports
	*					bound to this socket and the
	*					pointers to them.             */
	void setInputPorts( int count, int *inputPortIdxs );

	/* Method setOutputFields:		Sets the number of output fields
	*					bound to this socket and the
	*					pointers to them.             */
	void setOutputFields( int count, int *outputFieldIdxs );

	/* Method getObjectName:		Overloads Obj::getObjectName to
	*					return the full name including
	*					the class and module names.   */
	virtual const char * getObjectName() const;

	/* Method printOn: 			Stream output function        */
	virtual void printOn( ostream& os = cout ) const ;
};

inline RecordDictionary& 
InputPipeSocket::getDictionary() const 
{
	return *dictionary;
}

inline ModuleId *
InputPipeSocket::getModuleId() const 
{
	return source;
}

inline int
InputPipeSocket::getOutputField( int outputFieldBound ) const
{
	Assert( outputFieldBound < outputFieldsCnt );
	return( outputFieldsBound[ outputFieldBound ] );
}

inline int
InputPipeSocket::getPipeIndex() const 
{
	return pipeIndex;
}

inline int
InputPipeSocket::inputPortsFilled() const 
{
	return inputPortsCnt;
}

inline int
InputPipeSocket::outputFieldsFilled() const 
{
	return outputFieldsCnt;
}

inline ostream& operator<<( ostream& os, InputPipeSocket& obj )
{
	obj.printOn( os );
	return os;
}

#endif
