/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Philip C. Roth (proth@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * UtilDisplay.h: The base class for all utilization displays
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Includes/RCS/UtilDisplay.h,v 1.14 1994/02/25 04:31:49 aydt Exp $
 */

#ifndef UtilDisplay_h
#define	UtilDisplay_h

#include <strstream.h>
#include <X11/Intrinsic.h>
#include "Defines.h"
#include "ListNode.h"

class Selection;

 // define some enumerated types for use as constants

enum UtilsDisplayedType { AggregateShown, IndividualsShown };

const int numAggregates = 3;
enum AggregateUtils { Average, Minima, Maxima };

// struct for passing information to callback functions

class UtilDisplay;

typedef struct {
	UtilDisplay*	disp;
	Widget		w;
} UtilDisplayData;

class UtilFU;
struct UtilFUData;

class UtilDisplay : public ListNode {
/*
 * UtilDisplay is the base class for all utilization functional unit
 * displays.
*/

	friend class UtilDisplayManager;

private:
	static const Dimension defSizeWidthLimit;	// limits on size of
	static const Dimension defSizeHeightLimit;	// resized dialogs

	static String rightArgString;	// Resource name for right attachment
	static String bottomArgString;	// Resource name for bottom attachment

	 static int waitingSelDisplays;  // count of selection callback displays
					 // to pop down before restarting
					 // Pablo execution


	char	*myName;		// this object"s name
	char	*horiLabel;		// The horizontal and vertical labels
	char	*vertLabel;

	int	numIndividuals;		// count of individual PEs being tracked
	Selection *currSel;		// current list of individual PEs

	UtilsDisplayedType utilsDisplayed;	// indicates which type of
						// utilizations are displayed


	UtilFU	*fuParent;		// Utilization FU which controls this

	
	// X/Motif variables

	Widget	dialog;			// the display Dialog widget
	Widget	workWindow;		// the widget subtree used as workWindow
	Widget	rc;			// the display"s RowColumn widget
	Widget	hlabel;			// Horizontal, vertical Label widgets
	Widget	vlabel;

	Dimension minWidth,		// dimensions of minimum dialog size
		  minHeight;		// without scroll bars
	Dimension sbarWidth, sbarHeight;	// height/width of scrolled
						// window ScrollBars
	Dimension perfLabelDim;		// dimension of performance setup label

	/* Method _insertHere:		Indicates position of
	 *				"this" in a given List.
	 *				The method should return TRUE_ if
	 *				this is to be placed before its
	 *				argument in a list, FALSE_ otherwise.
	 *				The resulting list is returned.	*/
	virtual Boolean_ _insertHere( ListNode* );

	/* Method _handleResize:	Manages a special display
	 *				dialog window resizing behavior	*/
	virtual void _handleResize( Widget ) = 0;


	/* Method _handleSelectCB:	Actual callback for handling
	 *				the XtNselect callback which is
	 *				supported by the Pablo performance
	 *				widgets				*/
	void _handleSelectCB( UtilFUData* data, int val );


	/* Method _makeDisplay:		Creates the Motif widget hierarchy
	 *				which is common to all UtilFU
	 *				performance displays		*/
	virtual	Widget _makeDisplay( Widget par );


        /* Method SignalStop:           Indicates that yet one more 
         *                              selection display is being 
         *                              displayed                       */
        static void SignalStop( void );


        /* Method WaitStop:             Indicates that one more selection 
         *                              display is being popped down.   */
        static int WaitStop();

	
protected:
	Widget	parent;			// parent of the PU display

	Widget	aggregateUtils[numAggregates];	// widgets for aggregates
	Widget	*individualUtils;		// dynamic array of widgets 
						// for individual PEs

	int	*map;			// mapping between PE numbers with
					// location in the arrays

	/* Method _makeMap:		Creates the mapping between PE
	 *				numbers and the location of its
	 *				associated performance widget in the
	 *				arrays.  The Selection tells which
	 *				PEs to map, and the size argument
	 *				indicates how many PEs are in the
	 *				map				*/
	void _makeMap( Selection *sel, int& size );


	/* Method _indexIntoMap:	Performs an associative lookup into
	 *				the map, returning the index 
	 *				associated with argument proc in the
	 *				arrays				*/
	int _indexIntoMap( int proc );


public:
	static	const char *averageLabelString;	// labels for aggregate
	static	const char *minimumLabelString;	// statistic"s performance
	static	const char *maximumLabelString;	// widgets

	static const int notDisplayed;	// result code indicating utilization
					// is not being displayed

	/* Method UtilDisplay:		The class constructor.
	 *				The display created will have the
	 *				given name, belong to the indicated
	 *				Utilization FU, and have X parent
	 *				par				*/
	UtilDisplay( char *name, UtilFU *ufu,  Widget par );

	/* Method ~UtilDisplay:		The class destructor		*/
	~UtilDisplay( void );

	/* Method changeValue:		Updates the performance widget
	 *				for the	given PE to show the new
	 *				value of its utilization, indicated
	 *				by the util argument		*/
	virtual void changeValue( int proc, float util ) = 0;

	/* Method cleanupPerfWidgets:	Ceallocates any storage
	 *				used by performance widgets and
	 *				the arrays used to hold them	*/
	void cleanupPerfWidgets( void );

	/* Method createPerfSetup:	Creates a performance
	 *				widget setup for display in this
	 *				UtilDisplay.  Subclasses must
	 *				override this method.		*/
	virtual Widget createPerfSetup(  const char* label ) = 0;

	/* Method resizeCB:		Callback for handling
	 *				the resizing the underlying
	 *				performance widgets when their
	 *				dialog parent is resized	*/
	static void resizeCB( Widget, UtilDisplayData*, XtPointer );

	/* Method getNaturalDimensions:	Fetches the dimension of the
	 *				performance widget setup that
	 *				constrains the size when multiple
	 *				performance widgets are shown in
	 *				the UtilDisplay			*/
        void  getNaturalDimensions( Dimension& w, Dimension& h ) const;

        /* Method setNaturalDimensions:     Sets the dimension of the
         *                              performance widget setup that
         *                              constrains the size when multiple
         *                              performance widgets are shown in
         *                              the UtilDisplay                 */
	void setNaturalDimensions( Dimension w, Dimension h );

	/* Method setDialog:		Sets the X/Motif dialog used
	 *				to display the performance widgets
	 *				for this UtilDisplay		*/
	void setDialog( Widget d );

        /* Method getDialog:            Gets the X/Motif dialog used
         *                              to display the performance widgets
         *                              for this UtilDisplay            */
	Widget getDialog( void ) const;

	/* Method getDisplayByName:	Lookup the display
	 *				with the given name in the 
	 *				list of displays that starts
	 *				at "this"			*/
	UtilDisplay * getDisplayByName( char *name );

	/* Method getDisplayedType:	Gets the type of utilizations
	 *				displayed ( i.e., aggregate or
	 *				individual			*/
        UtilsDisplayedType getDisplayedType( void ) const;

        /* Method setDisplayedType:     Sets the type of utilizations
         *                              displayed ( i.e., aggregate or
         *                              individual                      */
	void setDisplayedType( UtilsDisplayedType t );

	/* Method getFUParent:		Returns the Utilization FU that
	 *				is the parent of this
	 *				UtilDisplay			*/
	UtilFU * getFUParent( void ) const;

	/* Method getHoriLabel:		gets the horizontal label
	 *				for this UtilDisplay		*/
        char * getHoriLabel( void ) const;

        /* Method setHoriLabel:         sets the horizontal label
         *                              for this UtilDisplay            */
	void setHoriLabel( char *labl );

	/* Method getHoriLabelWidget:	Retrieves the horizontal label
	 *				widget for this display		*/
	Widget getHoriLabelWidget( void ) const;

	/* Method getName:		Gets the name of this UtilDisplay */
        char * getName( void ) const;

        /* Method setName:              Sets the name of this UtilDisplay */
	void setName( char *name );

	/* Method setNumIndividuals:	Sets the number of individual
	 *				processors currently displayed	*/
	void setNumIndividuals( int n );

        /* Method getNumIndividuals:    Gets the number of individual
         *                              processors currently displayed  */
	int getNumIndividuals( void ) const;

	/* Method getPerfLabelDim:	Gets the size of the labels
	 *				used in the minor dimension
	 *				for each performance widget setup */
	Dimension getPerfLabelDim( void ) const;

        /* Method setPerfLabelDim:      Sets the size of the labels
         *                              used in the minor dimension
         *                              for each performance widget setup */
	void setPerfLabelDim( Dimension d );

	/* Method getRowCol:		Returns the RowColumn widget used
	 *				to contain the performance widgets
	 *				for this display		*/
	Widget getRowCol( void ) const;

	/* Method getSelection:		Gets the current selection of
	 *				individual processors		*/
        Selection * getSelection( void ) const;

        /* Method setSelection:         Sets the current selection of
         *                              individual processors           */
	void setSelection( Selection* );

	/* Method getType:		Returns the display type of this as
	 *				a character string		*/
	virtual const char *	const getType( void ) const = 0;

	/* Method getVertLabel:		Gets the vertical label
	 *				for this UtilDisplay		*/
        char * getVertLabel( void ) const;

     	/* Method setVertLabel:         Sets the vertical label
         *                              for this UtilDisplay            */
	void setVertLabel( char *labl );

	/* Method getVertLabelWidget:	Retrieves the vertical label widget
	 *				for this display		*/
	Widget getVertLabelWidget( void ) const;

	/* Method getWorkWindow:	Gets the X/Motif widget used
	 *				to display the performance widgets
	 *				for this UtilDisplay		*/
	Widget getWorkWindow( void ) const;

	/* Method initialResizeDialog:	Resizes the X/Motif Dialog
	 *				that displays the performance
	 *				widgets for this UtilDisplay	*/
	void initialResizeDialog( void );

	/* Method selectCB:		Callback or the performance
	 *				widget's XtNselect callback	*/
	static void selectCB( Widget, UtilDisplay *, XtPointer );

	/* Method selectDoneCB:		Callback indicating when to
	 *				stop handling X events and
	 *				return control to Pablo		*/
	static void selectDoneCB( Widget, UtilFUData *, XtPointer );

        /* Method setDialogTitle:       Sets the window manager dialog's
         *                              title to <name>:<type>          */
        void setDialogTitle( const char* name );

	/* Method uniqueDisplayName:	Verifies that the
	 *				given name is unique in the
	 *				list of displays which begin
	 *				at "this"			*/
	Boolean_ uniqueDisplayName( char *name );

	/* Method updatePerfWidgets:	Updates the set of performance 
	 *				widgets shown in the display	*/
	virtual void updatePerfWidgets( void ) = 0;

	/* Method printOn:		Prints a representation of
	 *				this UtilDisplay on the given
	 *				stream				*/
	virtual void printOn( ostream& s ) const;
};

inline void
UtilDisplay::setDialog( Widget d )
{ 
	dialog = d;
}

inline Widget
UtilDisplay::getDialog( void ) const
{
	return dialog;
}

inline void
UtilDisplay::setDisplayedType( UtilsDisplayedType t ) 
{ 
	utilsDisplayed = t; 
}

inline UtilsDisplayedType 
UtilDisplay::getDisplayedType( void ) const
{ 
	return utilsDisplayed;
}

inline UtilFU *
UtilDisplay::getFUParent( void ) const
{ 
	return fuParent;
}

inline Widget
UtilDisplay::getHoriLabelWidget( void ) const
{
	return hlabel;
}

inline void
UtilDisplay::getNaturalDimensions( Dimension& w, Dimension& h ) const
{
	w = minWidth;
	h = minHeight;
}

inline void
UtilDisplay::setNaturalDimensions( Dimension w, Dimension h )
{
	minWidth = w;
	minHeight = h;
}

inline void
UtilDisplay::setNumIndividuals( int n )
{
	numIndividuals = n;
}

inline int
UtilDisplay::getNumIndividuals( void ) const
{ 
	return numIndividuals;
}

inline Dimension
UtilDisplay::getPerfLabelDim( void ) const
{
	return perfLabelDim;
}

inline void
UtilDisplay::setPerfLabelDim( Dimension d )
{
	perfLabelDim = d;
}

inline Widget
UtilDisplay::getRowCol( void ) const
{
	return rc;
}

inline void
UtilDisplay::setSelection( Selection *list )
{
	currSel = list;
}

inline Selection*
UtilDisplay::getSelection( void ) const
{
	return currSel;
}

inline Widget
UtilDisplay::getVertLabelWidget( void ) const
{
	return vlabel;
}

inline Widget
UtilDisplay::getWorkWindow( void ) const
{
	return workWindow;
}

inline void				/* static */
UtilDisplay::SignalStop( void )
{
        waitingSelDisplays++;
}


inline int				/* static */
UtilDisplay::WaitStop()
{
        return --waitingSelDisplays;
}

#endif UtilDisplay_h
