/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Robert Olson (olson@cs.uiuc.edu)
 * Contributing Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Autor: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * AddModuleDialog.C:   This module implements a custom dialog for
 *		        creating instances of analysis modules on the
 *			primary Pablo graph palette.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Interface/RCS/AddModuleDialog.C,v 1.29 1994/03/01 00:10:28 aydt Exp $
 *	
 */

#include <Xm/Text.h>

#include "AddModuleDialog.h"

#include "PabloHelpSystem.h"
#include "PabloMainInterface.h"
#include "WidgetBase.h"

static char *FUsToManuallyInsert[] = {
	"FileInput", "FileOutput"
};

static int nFUsToManuallyInsert = 
		sizeof(FUsToManuallyInsert) / sizeof(FUsToManuallyInsert[0]);

AddModuleDialog::AddModuleDialog( PabloMainInterface *createdBy )
{
	XmString	moduleMessage;
	XmString	fuMessage;
	int		i;

	_setClassName( MY_CLASS );

        if ( pabloMainInterface != NULL ) {
            abort( "Can't have more than one Add Module Dialog instance.\n" );
        } else {
            pabloMainInterface = createdBy;
        }

	isManaged = FALSE_;

	uniqueNumDict = new GeneralDictionary();

	addDialog = XtVaCreatePopupShell( "addDialog",
               		topLevelShellWidgetClass,
			Pablo::TopLevel()->getWidget(),
	       	        XtNtitle, "Pablo Module Creation",
			NULL );

	dialogPane = XtVaCreateWidget( "dialogPane",
			xmPanedWindowWidgetClass, addDialog,
               	        XmNsashWidth, 1, 
			XmNsashHeight, 1,
			NULL );

	controlArea = XtVaCreateWidget( "addControlArea",
			xmFormWidgetClass, dialogPane,
			XmNfractionBase, 7,
			NULL );

     	actionArea = XtVaCreateWidget( "addActionArea",
                        xmFormWidgetClass, dialogPane,
                        NULL );

	//  The action area contains three buttons "Add", "Dismiss",
	//  and "Help".  Of these, the "Add" button is the most important;
	//  it instantiates an instance of the selected analysis module.

   	addButton = XtVaCreateManagedWidget( "Add", 
                     	xmPushButtonWidgetClass, actionArea,
		     	XmNtopAttachment, XmATTACH_FORM,
                     	XmNrightAttachment, XmATTACH_POSITION,
		     	XmNdefaultButtonShadowThickness, 1,
                     	NULL );

        dismissButton = XtVaCreateManagedWidget( "Dismiss",
                     	xmPushButtonWidgetClass, actionArea,
                     	XmNtopAttachment, XmATTACH_FORM,
                     	XmNleftAttachment, XmATTACH_POSITION,
		     	XmNrightAttachment, XmATTACH_POSITION,
                     	XmNdefaultButtonShadowThickness, 1,
                     	NULL );

   	helpButton = XtVaCreateManagedWidget( "Help", 
                     	xmPushButtonWidgetClass, actionArea,
		     	XmNtopAttachment, XmATTACH_FORM,
                     	XmNrightAttachment, XmATTACH_POSITION,
                     	XmNdefaultButtonShadowThickness, 1,
                     	NULL );

	//  The primary dialog consists of a scrollbox of labeled scrollboxes.
	//  The individual scrollboxes contain analysis modules of a particular
	//  type.  The outer scrollbox exists to ensure that future addition
	//  of more analysis module classes will not cause the dialog"s
	//  size to increase.

	fuScrollBox = XtVaCreateWidget( "fuScrollBox",
			xmScrolledWindowWidgetClass, controlArea, 
			XmNscrollingPolicy, XmAUTOMATIC,
			XmNscrollBarDisplayPolicy, XmSTATIC,
		        XmNtopAttachment, XmATTACH_FORM,
		        XmNleftAttachment, XmATTACH_FORM,
			XmNrightAttachment, XmATTACH_FORM,
			NULL );

	fuRowColumn = XtVaCreateManagedWidget ( "fuRowColumn",
                      	xmRowColumnWidgetClass, fuScrollBox,
		      	XmNpacking, XmPACK_COLUMN,
		      	XmNorientation, XmHORIZONTAL,
                        NULL );

	categoryCount = getFUTypeCount();

	//  For each of the analysis module categories, we create a
	//  labeled scrollbox containing the names of the modules in
	//  the category.

	for ( i = 0; i < categoryCount; i++ ) {
	   fuBox[i] = XtVaCreateManagedWidget( "fuBox",
		   	xmRowColumnWidgetClass, fuRowColumn,
			XmNorientation, XmVERTICAL,
			XmNnumColumns, 1,
			NULL );

           fuMessage =  XmStringCreateLtoR( getFUClassName(i),
                                             XmSTRING_DEFAULT_CHARSET );

           fuLabel = XtVaCreateManagedWidget( "fuLabel",
                        xmLabelWidgetClass, fuBox[i],
                        XmNalignment, XmALIGNMENT_CENTER,
                        XmNresizable, FALSE,
           		XmNlabelString, (XtArgVal) fuMessage,
                        NULL );

           XmStringFree( fuMessage );

	   fuList[i] = XmCreateScrolledList( fuBox[i], "fuList", NULL, 0 );

           XtVaSetValues( fuList[i],
                          XmNselectionPolicy, XmSINGLE_SELECT,
                          NULL );

	   XtManageChild( fuList[i] );

           XtAddCallback( fuList[i], XmNsingleSelectionCallback,
                      (XtCallbackProc)AddModuleDialog::_staticDefaultCallback,
                      (XtPointer) this );
	}

	_initFromFUM();

	//  In addition to the scroll boxes, we display the default name
	//  of the selected analysis module and allow the user to edit that
	//  name if desired.

	nameBox = XtVaCreateManagedWidget( "nameBox",
			xmRowColumnWidgetClass, controlArea,
			XmNorientation, XmVERTICAL,
			XmNnumColumns, 1,
			XmNtopAttachment, XmATTACH_WIDGET,
			XmNtopWidget, fuScrollBox,
			XmNleftAttachment, XmATTACH_FORM,
			XmNbottomAttachment, XmATTACH_FORM,
			NULL );

        moduleMessage =  XmStringCreateLtoR( "Module name: ",
                                          XmSTRING_DEFAULT_CHARSET );

 	nameLabel = XtVaCreateManagedWidget( "nameLabel",
                        xmLabelWidgetClass, nameBox,
                        XmNalignment, XmALIGNMENT_BEGINNING,
                        XmNresizable, FALSE,
        		XmNlabelString, (XtArgVal) moduleMessage,
                        NULL );

	XmStringFree( moduleMessage );

	moduleText = XtVaCreateManagedWidget( "moduleText",
			xmTextFieldWidgetClass, nameBox,
			NULL );

	//  Finally, we display a pixmap representing the icon for the
	//  module most recently selected by the user.

        iconBox = XtVaCreateManagedWidget( "iconBox",
                        xmRowColumnWidgetClass, controlArea,
                        XmNorientation, XmVERTICAL,
                        XmNnumColumns, 1,
                        XmNtopAttachment, XmATTACH_WIDGET,
                        XmNtopWidget, fuScrollBox,
                        XmNrightAttachment, XmATTACH_FORM,
                        XmNbottomAttachment, XmATTACH_FORM,
                        NULL );

        moduleMessage =  XmStringCreateLtoR( "Module icon: ",
                                          XmSTRING_DEFAULT_CHARSET );

        iconLabel = XtVaCreateManagedWidget( "iconLabel",
                        xmLabelWidgetClass, iconBox,
                        XmNalignment, XmALIGNMENT_BEGINNING,
                        XmNresizable, FALSE,
        		XmNlabelString, (XtArgVal) moduleMessage, 
                        NULL );

	XmStringFree( moduleMessage );

        pixmapFrame = XtVaCreateManagedWidget( "pixmapFrame",
                        xmFrameWidgetClass, iconBox,
                        NULL );

        fuGraphic = XtVaCreateWidget( "fuPixmap",
                        xmLabelWidgetClass, pixmapFrame,
	  		XmNalignment, XmALIGNMENT_CENTER,
                        XmNlabelType, XmPIXMAP,
                        NULL );

	//  Motif action areas are never supposed to change size.
	//  We fix the size of the pane containing the buttons.

   	Dimension h; 
   	XtVaGetValues( dismissButton, XmNheight, &h, NULL );

   	XtVaSetValues( actionArea,
		       XmNpaneMaximum, h,
		       XmNpaneMinimum, h,
		       NULL );

	XtManageChild( dialogPane );
	XtManageChild( controlArea );
	XtManageChild( actionArea );
	XtManageChild( fuScrollBox );
	XtManageChild( fuGraphic );

	int height, width;
	char *bits = getFUIconInfo( "GeneralFUIcon", &height, &width );
        _createPixmap( height, width, bits );

        XtAddCallback( dismissButton, XmNactivateCallback,
		      (XtCallbackProc)AddModuleDialog::_staticDismissCallback,
		      (XtPointer) this );
        XtAddCallback( addButton, XmNactivateCallback,
		      (XtCallbackProc)AddModuleDialog::_staticAddCallback,
		      (XtPointer) this );
        XtAddCallback( helpButton, XmNactivateCallback,
		      (XtCallbackProc)AddModuleDialog::_staticHelpCallback,
		      (XtPointer) this );

}

AddModuleDialog::~AddModuleDialog()
{
	XtDestroyWidget( dialogPane );
	XtDestroyWidget( controlArea );
	XtDestroyWidget( actionArea );
	XtDestroyWidget( addDialog );

	delete uniqueNumDict;
	pabloMainInterface = NULL;
}

void
AddModuleDialog::_addCallback()
{
        /*
         *  This callback is invoked when the user presses the "Add"
         *  button.  
	 *  + Tell the main interface to create an instance of this module
	 *    with the module name.
	 *  + Increment the default number for the next instance of the module
	 *    with same type.
	 *  + Update the Add Module display with the new Name/Number
	 */

        CString fuName = XmTextGetString( moduleText );
        pabloMainInterface->moduleAddRequest( selectedFU, fuName );

        int FUNum = _getUniqueNumFor( selectedFU );
        FUNum++;
        uniqueNumDict->insertKeyValue( selectedFU, (void*) FUNum );

        char    tmpBuf[100];
        sprintf( tmpBuf, "%s-%d", selectedFU.getValue(), FUNum );
        XmTextSetString( moduleText, tmpBuf );
}

void
AddModuleDialog::_addFUToList( const char *name, FUClassType type )
{
        Widget list;

        if (( type < 0 ) || ( type > NFUClassTypes )) {
           error("Unknown type %d in _addFUToList", type);
        } else {
           list = fuList[type];         // find the appropriate scrollbox
           XmString xmstr = XmStringCreateLtoR( (char *) name,
                                                XmSTRING_DEFAULT_CHARSET );
           XmListAddItemUnselected( list, xmstr, 0 );
           XmStringFree( xmstr );
        }
}

void
AddModuleDialog::_defaultCallback( Widget w,
                        	   XtPointer /* clientData */, 
				   XmListCallbackStruct *ret )
{
        char    tmpBuf[100];
        int     i;

        if ( ret->item == NULL ) {
           return;
        }

        //  To avoid confusion, we deselect all items in all scrollboxes
        //  except the one most recently chosen by the user

        for ( i = 0; i < categoryCount; i++ ) {
           if ( fuList[i] != w ) {
              XmListDeselectAllItems( fuList[i] );
           }
        }

        CString item = getCStringFromMotifString( ret->item );

        selectedFU = item;

        //  Having determined the instance, create a default name for it
        //  and display the module"s icon

        int FUNum = _getUniqueNumFor( selectedFU );

        sprintf( tmpBuf, "%s-%d", item.getValue(), FUNum );

        XmTextSetString( moduleText, tmpBuf );

	int height, width;
	char *bits = getFUIconInfo( item.getValue(), &height, &width );
        _createPixmap( height, width, bits );
}

void
AddModuleDialog::_dismissCallback()
{
        /*  
 	 * This callback is invoked when the user presses the "Dismiss"
         *  button.  To avoid confusion, we deselect all items before
         *  closing the dialog.  
	 */

        int     i;
        for ( i = 0; i < categoryCount; i++ ) {
            XmListDeselectAllItems( fuList[i] );
        }
        XtUnmanageChild( addDialog );   // popdown the dialog
	isManaged = FALSE_;
}

void
AddModuleDialog::_createPixmap( int pixmapHeight, int pixmapWidth,
                                     char *pixmapBits )
{
        Pixmap  fuPixmap;

        //  We create a pixmap representing the icon for the currently
        //  selected analysis module.  This icon will later be used on
        //  the analysis graph display palette.

        XtVaGetValues( fuGraphic,
                       XmNforeground, &pabloForeground,
                       XmNbackground, &pabloBackground,
                       NULL );

        fuPixmap = XCreatePixmapFromBitmapData( XtDisplay(controlArea),
                                XtScreen(controlArea)->root,
                                pixmapBits, pixmapWidth, pixmapHeight,
				BlackPixel(XtDisplay(controlArea),
					DefaultScreen(XtDisplay(controlArea))),
  				WhitePixel(XtDisplay(controlArea),
					DefaultScreen(XtDisplay(controlArea))),
                                DefaultDepth( XtDisplay(controlArea),
                                     	DefaultScreen(XtDisplay(controlArea))));

        if ( fuPixmap == XmUNSPECIFIED_PIXMAP ) {
           warning("Unable to load Pablo pixmap\n");
        } else {
	   XtVaSetValues( fuGraphic,
                          XmNlabelPixmap, (XtArgVal) fuPixmap,
                          NULL );
	}
}

int 
AddModuleDialog::_getUniqueNumFor( const CString &FU )
{
        int FUNum = (int) uniqueNumDict->lookupKey( FU );

        if ( FUNum == 0 ) {
           FUNum = 1;
           uniqueNumDict->insertKeyValue( FU, (void*) FUNum );
        }

        return FUNum;
}

void
AddModuleDialog::_helpCallback()
{
        //  If the user has selected a particular analysis module,
        //  we provide help on that module.  Otherwise, we provide
        //  help on the module addition process.

        if ( selectedFU != "" ) {
           Pablo::HelpSystem()->giveHelpOn( selectedFU.getValue() );
        } else {
           Pablo::HelpSystem()->giveHelpOn( "AddAModule" );
        }
}

void
AddModuleDialog::_initFromFUM()
{
        CStringObjList  *listOfFUs = Pablo::FUM()->getFUList();
        FUClassType     type;
        int             i;

 	// Clear the list in each category.
        for ( i = 0; i < categoryCount; i++ ) {
            XtVaSetValues( fuList[i], XmNitemCount, 0, XmNitems, NULL, NULL );
        }

        // Fetch each analysis module and add it to the appropriate
        // scroll box
        for ( i = 0; i < listOfFUs->count(); i++ ) {
	    const char* fu = listOfFUs->getElement(i).getValue();
            type = getFUClassification( fu );
            _addFUToList( fu, type );
        }

        // Manually insert the file input and output modules
        for (i  = 0; i < nFUsToManuallyInsert; i++ ) {
            type = getFUClassification( FUsToManuallyInsert[i] );
            _addFUToList( FUsToManuallyInsert[i], type );
        }
}

void
AddModuleDialog::_staticAddCallback( Widget /* w */,
                                     XtPointer clientData,
                                     XtPointer /* callData */ )
{
        AddModuleDialog *obj = (AddModuleDialog *) clientData;
        obj->_addCallback();
}

void
AddModuleDialog::_staticDefaultCallback( Widget w,
                                    XtPointer clientData,
                                    XmListCallbackStruct *cbs )
{
        AddModuleDialog *obj = (AddModuleDialog *) clientData;
        obj->_defaultCallback( w, clientData, cbs );
}

void
AddModuleDialog::_staticDismissCallback( Widget /* w */,
                                    	 XtPointer clientData,
                                    	 XtPointer /* callData */ )
{
        AddModuleDialog *obj = (AddModuleDialog *) clientData;
        obj->_dismissCallback();
}

void
AddModuleDialog::_staticHelpCallback( Widget /* w */,
                                      XtPointer clientData,
                                      XtPointer /* callData */ )
{
        AddModuleDialog *obj = (AddModuleDialog *) clientData;
        obj->_helpCallback();
}

void
AddModuleDialog::raiseDialog()
{
	if ( isManaged ) {
            XRaiseWindow( XtDisplay(addDialog), XtWindow(addDialog) );
	} else {
            XtManageChild( addDialog );     // popup the dialog
	    isManaged = TRUE_;
	}

	/*
	 *  Force selection of FileInput each time the Add Module Dialog
	 *  is selected.
	 */
	XmString firstSelection = XmStringCreateLtoR( "FileInput",
                                          XmSTRING_DEFAULT_CHARSET );
	XmListSelectItem( fuList[FU_File], firstSelection, True );
	XmStringFree( firstSelection );
}

void 
AddModuleDialog::resetUniqueNums()
{
	delete uniqueNumDict;
	uniqueNumDict = new GeneralDictionary();

	// update the current display with a new string;
        char    tmpBuf[100];
	int FUNum = _getUniqueNumFor( selectedFU );
	sprintf( tmpBuf, "%s-%d", selectedFU.getValue(), FUNum );
	XmTextSetString( moduleText, tmpBuf );
}
		
void
AddModuleDialog::printOn( ostream &os ) const 
{
	os << "AddModuleDialog\n";
}

/*
 * Initialize the static data
 */
const char *const AddModuleDialog::MY_CLASS = "AddModuleDialog";

PabloMainInterface *AddModuleDialog::pabloMainInterface = NULL;
